# +=========================================================================+
# || CipUX::RBAC::Client                                                   ||
# ||                                                                       ||
# || Example client for command line interface for CipUX RBAC class.       ||
# ||                                                                       ||
# ||                                                                       ||
# ||                                                                       ||
# ||                                                                       ||
# || (C) Copyright 2008 by Christian Kuelker. All rights reserved!         ||
# ||                                                                       ||
# || License: GPLv2 or later                                               ||
# ||                                                                       ||
# +=========================================================================+
#
# ID:       $Id$
# Revision: $Revision$
# Head URL: $HeadURL$
# Date:     $Date$
# Source:   $Source$

package CipUX::RBAC::Client;

use 5.008001;
use strict;
use warnings;
use Carp;
use Class::Std;
use Data::Dumper;
use Getopt::Long;
use Pod::Usage;
use List::MoreUtils qw(any);
use Log::Log4perl qw(get_logger :levels);
use English qw( -no_match_vars);
use Readonly;
use base qw(CipUX::RBAC);

{

    delete @ENV{qw(PATH IFS CDPATH ENV BASH_ENV)};    # Make %ENV safer
    use version; our $VERSION = qv('3.4.0');
    use re 'taint';    # Keep data captured by parens tainted

    # +=======================================================================+
    # || CONSTANTS                                                           ||
    # +=======================================================================+
    Readonly::Scalar my $SCRIPT       => 'CipUX::RBAC::Client';
    Readonly::Scalar my $EMPTY_STRING => q{};
    Readonly::Scalar my $CR           => "\n";
    Readonly::Scalar my $LINEWIDTH    => 76;
    Readonly::Scalar my $FATAL_EXIT   => 256;
    Readonly::Scalar my $L4P_CONF     => ( -e '/etc/cipux/log4perl.conf' )
      ? '/etc/cipux/log4perl.conf'
      : '/etc/cipux-cat-web/log4perl.conf';
    Readonly::Scalar my $L            => ( q{=} x $LINEWIDTH ) . $CR;
    Readonly::Scalar my $DISTRIBUTION => 'debian';

    # +=======================================================================+
    # || GLOBALS                                                             ||
    # +=======================================================================+

    # normally we do not need GLOBALS.

    # +=======================================================================+
    # || CLI API                                                             ||
    # +=======================================================================+
    sub run {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        # dummy option, since we do only have cipux_rbac_client as the only one
        my $name =
          exists $arg_r->{name}
          ? $self->l( $arg_r->{name} )
          : $self->perr('name');

        # if needed there can be a different configuration file. This is mostly
        # for writhing Perl tests
        my $cfg_file =
          exists $arg_r->{cfg_file}
          ? $self->l( $arg_r->{cfg_file} )
          : undef;

        # +------------------------------------------------------------------+
        # | getopt
        my $opt_hr = {};
        Getopt::Long::Configure('bundling');

        my $msg = $L . 'Problems parsing command line!' . "\n" . $L;

        GetOptions(
            $opt_hr,     'access|a', 'cfg|c=s',      'write_dot|d=s',
            'debug|D',   'from|f=s', 'help|h|?',     'list|l',
            'man',       'pretty|p', 'read_xml|r=s', 'rpc',
            'scope|s=s', 'task',     'to|t=s',       'version|V',
            'verbose',   'write_xml|w=s'
        ) or pod2usage( -msg => $msg );

        # set output to debug, if we are in debug mode
        my $debug = $self->_set_debug( { debug => $opt_hr->{debug} } );
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        # (1) parse some CLI option an get action list
        my $action_ar = $self->_parsing_cli( { opt_hr => $opt_hr } );
        $logger->debug( 'rbac client action from CLI: ',
            { filter => \&Dumper, value => $action_ar } );

        # (2) do the remaining aux opt
        my $pretty = ( exists $opt_hr->{pretty} ) ? 1 : 0;
        $logger->debug( 'pretty: ', $pretty );
        my $verbose = ( exists $opt_hr->{verbose} ) ? 1 : 0;
        $logger->debug( 'verbose: ', $verbose );

        # +------------------------------------------------------------------+
        # | main

        # for nearly all we need the rbac_scope
        $logger->debug('CALL rbac_config()');

        my ( $dummy1_hr, $rbac_scope_hr, $dummy2_hr ) =
          $self->rbac_config( { cfg_file => $cfg_file } );
        $msg = "EXCEPTION: no distribution part in rbac_scope_hr!\n";
        $rbac_scope_hr =
          exists $rbac_scope_hr->{$DISTRIBUTION}
          ? $self->h( $rbac_scope_hr->{$DISTRIBUTION} )
          : croak $msg;

        # +------------------------------------------------------------------+
        # | RBAC list                                                        |
        # +------------------------------------------------------------------+
        # display rbac scope list and exit
        if ( any { /list/smx } @{$action_ar} ) {
            $logger->debug('CALL _list_rbac()');
            my $page_ar = $self->_list_rbac(
                { pretty => $pretty, rbac_scope_hr => $rbac_scope_hr } );
            foreach my $line ( @{$page_ar} ) {
                print $line or croak "Can not print _list_rbac row!\n";
            }
            $logger->debug('END 1');
            exit 0;
        }

        # for the rest we need a real scope
        my $scope = $self->_check_scope( { scope => $opt_hr->{scope} } );
        $logger->debug( 'scope: ', $scope );

        # +------------------------------------------------------------------+
        # | read or construct graph                                          |
        # +------------------------------------------------------------------+
        # We do the graph stuff now, because others like "access" need it
        my $graph;
        if ( any { /read_xml/smx } @{$action_ar} ) {

            $logger->debug( 'ACTION: ', 'read_xml' );
            my $file = $self->l( $opt_hr->{read_xml} );
            $logger->debug('CALL read_xml_file()');
            $graph = $self->read_xml_file( { file => $file, graph => $graph } );

        }
        else {

            $logger->debug( 'ACTION: ', 'construct graph' );
            $logger->debug( 'configfile: ',
                { filter => \&Dumper, value => $cfg_file } );
            $logger->debug( 'scope: ', $scope );
            $logger->debug('CALL construct_directed_graph()');

            # TODO: check parameter
            my $method =
                exists $opt_hr->{rpc}  ? 'rpc'
              : exists $opt_hr->{file} ? 'file'
              : exists $opt_hr->{task} ? 'task'
              :                          'task';
            $logger->debug("method $method is used to get access info");

            $graph = $self->construct_directed_graph(
                {
                    rbac_scope_hr => $rbac_scope_hr,
                    scope         => $scope,
                    method        => $method,
                }
            );
            $logger->debug( 'graph: ',
                { filter => \&Dumper, value => $graph } );

        }

        # +-------------------------------------------------------------------+
        # | check access test                                                 |
        # +-------------------------------------------------------------------+
        if ( any { /access/smx } @{$action_ar} ) {

            $logger->debug('ACTION: access');
            $logger->debug('CALL _check_access()');
            my $access = $self->_check_access(
                {
                    graph   => $graph,
                    from    => $opt_hr->{from},
                    to      => $opt_hr->{to},
                    pretty  => $pretty,
                    verbose => $verbose,
                }
            );
            $logger->debug( 'access from _check_access(): ', $access );
            $logger->debug( 'verbose: ',                     $verbose );

        }

        # +-----------------------------------------------------------------+
        # | write dot file                                                  |
        # +-----------------------------------------------------------------+
        if ( any { /write_dot/smx } @{$action_ar} ) {

            $logger->debug('ACTION: write_dot');
            my $file = $self->l( $opt_hr->{write_dot} );
            $logger->debug('CALL _write_dot_file()');
            $self->write_dot_file( { file => $file, graph => $graph } );

        }

        # +--------------------------------------------------------------------+
        # | write xml file                                                     |
        # +--------------------------------------------------------------------+
        if ( any { /write_xml/smx } @{$action_ar} ) {

            $logger->debug('ACTION: write_xml');
            my $file = $self->l( $opt_hr->{write_xml} );
            $logger->debug('CALL _write_xml_file()');
            $self->write_xml_file( { file => $file, graph => $graph } );

        }

        $logger->debug('END 2');

        # +------------------------------------------------------------------+
        # | API
        return;
    }

    # +=======================================================================+
    # || _parsing_cli                                                        ||
    # +=======================================================================+
    sub _parsing_cli {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $opt_hr =
          exists $arg_r->{opt_hr}
          ? $self->h( $arg_r->{opt_hr} )
          : $self->perr('opt_hr');

        # +------------------------------------------------------------------+
        # | main
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        my @action = ();

        # +-------------------------------------------------------------------+
        # | action with exit                                                  |
        # +-------------------------------------------------------------------+

        # display help page
        if ( exists $opt_hr->{help} ) {
            pod2usage( -verbose => 1 );
        }

        # display version and exit
        if ( exists $opt_hr->{version} ) {
            print "$SCRIPT $VERSION\n"
              or croak 'Can not print version to STDOUT!';
            exit 0;
        }

        # +-------------------------------------------------------------------+
        # | dispatch action                                                   |
        # +-------------------------------------------------------------------+

        # so if we do not exit, it makes sense to collect all actions we have
        # to perform

        # ordering matters
        foreach my $opt ('list') {
            if ( exists $opt_hr->{$opt} ) {
                push @action, $opt;
            }
        }

        # add action "access" if all data is there
        if (    exists $opt_hr->{access}
            and exists $opt_hr->{from}
            and exists $opt_hr->{to} )
        {
            push @action, 'access';
        }

        # the remaining read and write stuff
        foreach my $opt (qw(read_xml write_xml write_dot)) {
            if ( exists $opt_hr->{$opt} ) {
                push @action, $opt;
            }
        }

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return \@action;

    }

    # +=======================================================================+
    # || _set_debug                                                          ||
    # +=======================================================================+
    sub _set_debug {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $debug =
          ( exists $arg_r->{debug} and defined $arg_r->{debug} ) ? 1 : 0;

        # +------------------------------------------------------------------+
        # | main
        if ( $debug eq $EMPTY_STRING ) {
            $debug = 1;
        }

        if ($debug) {
            Log::Log4perl::init_once($L4P_CONF);
        }

        return $debug;
    }

    # +=======================================================================+
    # || _check_scope                                                        ||
    # +=======================================================================+
    sub _check_scope {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $scope =
          exists $arg_r->{scope} ? $self->l( $arg_r->{scope} ) : $EMPTY_STRING;

        # +------------------------------------------------------------------+
        # | main
        my $logger = get_logger(__PACKAGE__);

        $logger->debug( 'scope: ', $scope );

        # complain if we do not have a scope
        if ( $scope eq $EMPTY_STRING ) {
            my $msg = $L . 'EXCEPTION: ';
            $msg .= 'Missing scope! Please use -s or --scope.';
            $msg .= "\n" . $L;

            pod2usage( -msg => $msg, -verbose => 0 );
        }

        $logger->debug('non empty scope');
        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return $scope;

    }

    # +=======================================================================+
    # || _check_access                                                       ||
    # +=======================================================================+
    sub _check_access {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $graph =
          exists $arg_r->{graph} ? $arg_r->{graph} : $self->perr('graph');

        my $from =
          $arg_r->{from} ? $self->l( $arg_r->{from} ) : $self->perr('from');

        my $to = $arg_r->{to} ? $self->l( $arg_r->{to} ) : $self->perr('to');

        my $pretty =
            $arg_r->{pretty}
          ? $self->l( $arg_r->{pretty} )
          : 0;

        my $verbose =
            $arg_r->{verbose}
          ? $self->l( $arg_r->{verbose} )
          : 0;

        # +------------------------------------------------------------------+
        # | main
        my $logger = get_logger(__PACKAGE__);
        my $r      = $self->access(
            {
                graph       => $graph,
                source      => $from,
                destination => $to
            }
        );

        # If pretty, we just make an output
        if ( $pretty or $verbose ) {
            if ( defined $r and $r ) {
                print "YES\n"
                  or croak 'Can not print answer YES to STDOUT!';
            }
            else {
                print "NO\n" or croak 'Can not print answer NO to STDOUT!';
            }
        }

        # If not, we construct an exit value, so this can be used elsewhere
        else {

            $logger->debug('END 1');
            exit $FATAL_EXIT if not defined $r;
            exit 0           if $r;
            exit 1           if not $r;
        }

        # If we did not exit, we will return
        $logger->debug('END 2');

        # +------------------------------------------------------------------+
        # | API
        return $r;

    }

    # +=======================================================================+
    # || _list_rbac                                                          ||
    # +=======================================================================+

    sub _list_rbac {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $pretty =
          exists $arg_r->{pretty}
          ? $self->l( $arg_r->{pretty} )
          : $self->perr('pretty');

        # a hash ref to the configuration
        my $rbac_scope_hr =
          exists $arg_r->{rbac_scope_hr}
          ? $self->h( $arg_r->{rbac_scope_hr} )
          : $self->perr('rbac_scope_hr');

        # +------------------------------------------------------------------+
        # | main
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');
        $logger->debug( '-> pretty:        ', $pretty );
        $logger->debug( '-> rbac_scope_hr: ', $rbac_scope_hr );
        my @page = ();

        my $list_ar = $self->list_rbac( { rbac_scope_hr => $rbac_scope_hr } );
        $logger->debug( 'got list_ar: ',
            { filter => \&Dumper, value => $list_ar } );

        # mostly this stuff is for pretty print
        my $max_col = 0;
        my $width   = 0;

        if ($pretty) {

            # calculate maximum columns
            foreach my $line ( @{$list_ar} ) {
                if ( $max_col < length $line ) {
                    $max_col = length $line;
                }
            }
            $logger->debug( 'max_col: ', $max_col );

            $width = 2 + $max_col;
            $logger->debug( 'width: ', $width );
            push @page, q{+} . q{-} x $width . "+\n";
            push @page, sprintf( '| %-' . $max_col . "s |\n", 'rbac' );
            push @page, q{+} . q{=} x $width . "+\n";
        }

        foreach my $line ( @{$list_ar} ) {
            chomp $line;
            if ($pretty) {
                push @page, sprintf( '| %-' . $max_col . "s |\n", $line );
            }
            else {
                push @page, "$line\n";
            }
        }

        if ($pretty) {
            push @page, q{+} . q{-} x $width . "+\n";
        }

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return \@page;

    }
}

1;

__END__

=head1 NAME

CipUX::RBAC::Client - create static role based access control


=head1 VERSION

This document describes CipUX::RBAC::Client version 3.4.0


=head1 SYNOPSIS

    use CipUX::RBAC::Client;

    my $client = CipUX::RBAC::Client->new();

    $client->run( { name => 'cipux_rbac_client' } );


=head1 DESCRIPTION

Retrieved role base access data from CipUX and can store them in a file.
It is also able to convert or output the file to a different format.


=head1 INTERFACE 

TODO

=head2 run

TODO

=head1 DIAGNOSTICS

TODO

=for author to fill in:
    List every single error and warning message that the module can
    generate (even the ones that will "never happen"), with a full
    explanation of each problem, one or more likely causes, and any
    suggested remedies.

=over

=item C<< Error message here, perhaps with %s placeholders >>

[Description of error here]

=item C<< Another error message here >>

[Description of error here]

[Et cetera, et cetera]

=back


=head1 CONFIGURATION AND ENVIRONMENT

=for author to fill in:
    A full explanation of any configuration system(s) used by the
    module, including the names and locations of any configuration
    files, and the meaning of any environment variables or properties
    that can be set. These descriptions must also include details of any
    configuration language used.
  
CipUX::RBAC::Client requires no configuration files or environment variables.


=head1 DEPENDENCIES

=for author to fill in:
    A list of all the other modules that this module relies upon,
    including any restrictions on versions, and an indication whether
    the module is part of the standard Perl distribution, part of the
    module's distribution, or must be installed separately. ]

None.


=head1 INCOMPATIBILITIES

=for author to fill in:
    A list of any modules that this module cannot be used in conjunction
    with. This may be due to name conflicts in the interface, or
    competition for system or program resources, or due to internal
    limitations of Perl (for example, many modules that use source code
    filters are mutually incompatible).

None reported.


=head1 BUGS AND LIMITATIONS

=for author to fill in:
    A list of known problems with the module, together with some
    indication Whether they are likely to be fixed in an upcoming
    release. Also a list of restrictions on the features the module
    does provide: data types that cannot be handled, performance issues
    and the circumstances in which they may arise, practical
    limitations on the size of data sets, special cases that are not
    (yet) handled, etc.

No bugs have been reported.



=head1 AUTHOR

Christian Kuelker  C<< <christian.kuelker@cipworx.org> >>


=head1 LICENCE AND COPYRIGHT

Copyright (c) 2008, Christian Kuelker C<< <christian.kuelker@cipworx.org> >>. All rights reserved.

This module is free software; you can redistribute it and/or
modify it under the same terms as Perl itself. See L<perlartistic>.


=head1 DISCLAIMER OF WARRANTY

BECAUSE THIS SOFTWARE IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY
FOR THE SOFTWARE, TO THE EXTENT PERMITTED BY APPLICABLE LAW. EXCEPT WHEN
OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES
PROVIDE THE SOFTWARE "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE SOFTWARE IS WITH
YOU. SHOULD THE SOFTWARE PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL
NECESSARY SERVICING, REPAIR, OR CORRECTION.

IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING
WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR
REDISTRIBUTE THE SOFTWARE AS PERMITTED BY THE ABOVE LICENCE, BE
LIABLE TO YOU FOR DAMAGES, INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL,
OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OR INABILITY TO USE
THE SOFTWARE (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING
RENDERED INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A
FAILURE OF THE SOFTWARE TO OPERATE WITH ANY OTHER SOFTWARE), EVEN IF
SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF
SUCH DAMAGES.
