# +==========================================================================+
# || CipUX::RBAC                                                            ||
# ||                                                                        ||
# || CipUX RBAC Role Based Access Control                                   ||
# ||                                                                        ||
# || (C) Copyright 2008 by Christian Kuelker. All rights reserved!          ||
# ||                                                                        ||
# || License: GPL version 2 or later                                        ||
# ||                                                                        ||
# +==========================================================================+
#
# ID:       $Id$
# Revision: $Revision$
# Head URL: $HeadURL$
# Date:     $Date$
# Source:   $Source$

package CipUX::RBAC;

use 5.008001;
use strict;
use warnings;
use Carp;
use Class::Std;
use Data::Dumper;
use English qw( -no_match_vars);
use Frontier::Client;
use Graph;
use Graph::Directed;
use Graph::Reader::XML;
use Graph::Writer::XML;
use Graph::Writer::Dot;
use Log::Log4perl qw(:easy);
use Readonly;
use Scalar::Util qw(looks_like_number);    # core since Perl 5.8
use base qw(CipUX);

{                                          # BEGIN CLASS

    use version; our $VERSION = qv('3.4.0');
    use re 'taint';    # Keep data captured by parens tainted
    delete @ENV{qw(PATH IFS CDPATH ENV BASH_ENV)};    # Make %ENV safe

    # +======================================================================+
    # || Constants                                                          ||
    # +======================================================================+
    Readonly::Scalar my $EMPTY_STRING => q{};
    Readonly::Scalar my $SCRIPT       => 'CipUX::RBAC';
    Readonly::Scalar my $CR           => "\n";
    Readonly::Scalar my $LINEWIDTH    => 76;
    Readonly::Array my @CFG_FILE      => qw( ~/.cipux/cipux-rbac.conf
      /etc/cipux/cipux-rbac.conf
      /usr/share/cipux-rbac/etc/cipux-rbac.cfg);
    Readonly::Scalar my $L => ( q{=} x $LINEWIDTH ) . $CR;

    # +======================================================================+
    # || GLOBALS                                                            ||
    # +======================================================================+

    # finite directed or indirected graph of n vertices
    my $adjacency_matrix = {};

    #my %rbac_config_of = ATTR( set => 'rbac_config_of' );
    #my $rbac_scope_hr = $self->rbac_config();

    # +======================================================================+
    # || open module features                                               ||
    # +======================================================================+

    #sub BUILD {

    #    $rbac_scope_hr = rbac_config({});

    #  }

    # +======================================================================+
    # || rbac_config                                                        ||
    # +======================================================================+
    sub rbac_config {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        # "cfg_file" param for alternative file, this is mostly for:
        # (1) if there will be an option for CLI client (not implemented)
        # (2) for Perl tests
        my $cfg_file_ar = [];
        if ( exists $arg_r->{cfg_file} and defined $arg_r->{cfg_file} ) {
            push @{$cfg_file_ar}, $self->l( $arg_r->{cfg_file} );
        }

        # +------------------------------------------------------------------+
        # | prepare
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');
        foreach my $file (@CFG_FILE) {
            push @{$cfg_file_ar}, $self->homedir($file);
        }

        $logger->debug( 'cfg_file_ar:',
            { filter => \&Dumper, value => $cfg_file_ar } );

        # +------------------------------------------------------------------+
        # | main
        # TODO: make cfg location flexible!
        my ( $cfg_rbac0_hr, $cfg_rbac1_hr, $cfg_rbac2_hr ) =
          $self->source( { cfg => '/etc/cipux/cipux-rbac.conf' } );

        # +------------------------------------------------------------------+
        # | check results
        my $msg = 'Variable not defined in configuration file. ';
        $msg .= 'Please provide a valid configuration file.';
        if ( not defined $cfg_rbac0_hr, ) {
            $self->exc( { msg => $msg, value => 'config_hr0 (cfg_rbac0_hr)' } );
        }
        if ( not defined $cfg_rbac1_hr, ) {
            $self->exc( { msg => $msg, value => 'config_hr1 (cfg_rbac1_hr)' } );
        }
        if ( not defined $cfg_rbac2_hr, ) {
            $self->exc( { msg => $msg, value => 'config_hr2 (cfg_rbac2_hr)' } );
        }

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        #  $rbac_cfg_hr, $rbac_scope_hr, $rbac_explain_hr
        return ( $cfg_rbac0_hr, $cfg_rbac1_hr, $cfg_rbac2_hr );

    }

    # +=======================================================================+
    # || construct_directed_graph                                            ||
    # +=======================================================================+
    #
    # LEVEL
    # 0    1      2      3 ...
    # |    |      |      |
    # :    :      :      :
    # Task::cipux_task_list_teacher_accounts | task         \
    #  |                                     |              | ou=Task
    #  `-- cipuxMemberPid                    | attribute    |
    #      = teacher.cgi                     | program name / \
    #         |                              |                | ou=CAT
    #         `-- cipuxMemberRid             | attribute      |
    #             = teacher                  | role name    \ /
    #                |                       |              |
    #                `-- memberUid           | attribute    | ou=Group
    #                    = chkuelker         | user name    /
    # :    :      :      :
    # |    |      |      |
    # 0    1      2      3 ...
    #
    # This will construct a tree like as HASH reference:
    #
    #   $acl_hr->{cipux_task_list_teacher_accounts}
    #          ->{teacher.cgi}
    #          ->{teacher}
    #          ->{chkuelker}
    sub construct_directed_graph {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        # This modiefies $adjacency_matrix

        my $scope =
          exists $arg_r->{scope}
          ? $self->l( $arg_r->{scope} )
          : $self->perr('scope');

        # rbac_scope_hr is a hash reference from the 2nd hash reference of
        # cipux-rbac.conf without dimension of distribution (debian)
        my $rbac_scope_hr =
          exists $arg_r->{rbac_scope_hr}
          ? $self->h( $arg_r->{rbac_scope_hr} )
          : $self->perr('rbac_scope_hr');

        # method = rpc|task|file  (file not implement jet)
        my $method =
          exists $arg_r->{method}
          ? $self->l( $arg_r->{method} )
          : $self->perr('method');

        # +------------------------------------------------------------------+
        # | prepare
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');
        $logger->debug( '-> scope:         ', $scope );
        $logger->debug( '-> rbac_scope_hr->{scope}: ',
            { filter => \&Dumper, value => $rbac_scope_hr->{$scope} } );

        # collect all levels
        my @level = sort keys %{ $rbac_scope_hr->{$scope} };
        $logger->debug(
            'all level from rbac_scope_hr: ',
            { filter => \&Dumper, value => \@level }
        );

        # go for every level 0..n
        foreach my $level (@level) {
            $logger->debug( 'level:  ', $level );

            # fetch a list for starting
            my $start_ar = $self->get_object_list(
                {

                    mode          => 'list',
                    rbac_scope_hr => $rbac_scope_hr,
                    level         => $level,
                    scope         => $scope,
                    method        => $method,
                    object        => undef,         # not needed for "list" mode
                }
            );
            $logger->debug( "start_ar for level $level: ",
                { filter => \&Dumper, value => $start_ar } );

            # traverse over that starting list
            foreach my $object ( @{$start_ar} ) {

                $logger->debug( 'get related objects from: ', $object );
                $logger->debug( 'level:  ',                   $level );

                $self->get_related_objects(
                    {

                        mode          => 'member',
                        rbac_scope_hr => $rbac_scope_hr,
                        level         => $level,
                        scope         => $scope,
                        method        => $method,
                        object        => $object,

                    }
                );
            }
        }
        my $g = Graph::Directed->new;

        $logger->debug( 'adjacency_matrix: ',
            { filter => \&Dumper, value => $adjacency_matrix } );

        foreach my $row ( keys %{$adjacency_matrix} ) {
            $g->add_vertex($row);
            foreach my $column ( keys %{ $adjacency_matrix->{$row} } ) {
                $g->add_vertex($column);
                $g->add_edge( $row, $column );
            }
        }

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return $g;

    }

    # +=======================================================================+
    # || get_object_list                                                     ||
    # +=======================================================================+
    sub get_object_list {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        # rbac configuration
        my $rbac_scope_hr =
          exists $arg_r->{rbac_scope_hr}
          ? $self->h( $arg_r->{rbac_scope_hr} )
          : $self->perr('rbac_scope_hr');

        # cat|task
        my $scope =
          exists $arg_r->{scope}
          ? $self->l( $arg_r->{scope} )
          : $self->perr('scope');

        # 0..n
        my $level =
          exists $arg_r->{level}
          ? $self->li( $arg_r->{level} )
          : $self->perr('level');

        # sub get_object_list mode: list|member
        my $mode =
          exists $arg_r->{mode}
          ? $self->l( $arg_r->{mode} )
          : $self->perr('mode');

        # object is needed for "member" mode, object might be "undef" for "list"
        # mode:
        my $object =
          exists $arg_r->{object}
          ? $self->l( $arg_r->{object} )
          : $self->perr('object');

        my $method =
          exists $arg_r->{method}
          ? $self->l( $arg_r->{method} )
          : $self->perr('method');

        # +------------------------------------------------------------------+
        # | prepare
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');
        $logger->debug("> mode:   $mode");
        $logger->debug("> level:  $level");
        $logger->debug("> scope:  $scope");

        #$logger->debug("> object: $object");
        $logger->debug("> method: $method");

        # cfg_file, scope, level, dist
        my $msg = "EXCEPTION: Malformed configuraton\n";

        my $task =
          exists $rbac_scope_hr->{$scope}->{$level}->{$mode}
          ? $self->l( $rbac_scope_hr->{$scope}->{$level}->{$mode} )
          : croak $msg;

        # execute CipUX::Task or RPC command and @return
        my @return    = ();
        my $return_hr = {};
        if ( $method eq 'task' ) {
            $logger->debug('using method: task');
            $return_hr =
              $self->query_via_task( { task => $task, object => $object } );
        }
        elsif ( $method eq 'file' ) {
            $logger->debug('using method: file');

            # TODO
            croak 'Not implemented';
        }

        $msg = "EXCEPTION: Malformed configuration!\n";

        my $attribute =
          exists $rbac_scope_hr->{$scope}->{$level}->{ $mode . '_attr' }
          ? $self->l( $rbac_scope_hr->{$scope}->{$level}->{ $mode . '_attr' } )
          : croak $msg;

        $logger->debug( 'got attribute from configuration: ', $attribute );

        foreach my $element ( sort keys %{$return_hr} ) {
            $logger->debug( 'task returns element: ', $element );

            my $value = $return_hr->{$element}->{$attribute};

            if ( defined $value ) {
                $logger->debug( 'value: ',
                    { filter => \&Dumper, value => $value } );

                push @return, @{$value};
            }
            else {

                $logger->debug('value: undef');

            }

            # return () if not defined $list;
        }
        $logger->debug( 'return ARRAY contains: ',
            { filter => \&Dumper, value => \@return } );
        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return \@return;
    }

    # +=======================================================================+
    # || get_related_objects                                                 ||
    # +=======================================================================+
    sub get_related_objects {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        # This modiefies $adjacency_matrix

        # rbac configuration
        my $rbac_scope_hr =
          exists $arg_r->{rbac_scope_hr}
          ? $self->h( $arg_r->{rbac_scope_hr} )
          : $self->perr('rbac_scope_hr');

        # cat|task
        my $scope =
          exists $arg_r->{scope}
          ? $self->l( $arg_r->{scope} )
          : $self->perr('scope');

        # 0..n
        my $level =
          exists $arg_r->{level}
          ? $self->li( $arg_r->{level} )
          : $self->perr('level');

        # sub get_object_list mode: list|member
        my $mode =
          exists $arg_r->{mode}
          ? $self->l( $arg_r->{mode} )
          : $self->perr('mode');

        # object is needed for "member" mode, object might be "undef" for "list"
        # mode:
        my $object =
          exists $arg_r->{object}
          ? $self->l( $arg_r->{object} )
          : $self->perr('object');

        my $method =
          exists $arg_r->{method}
          ? $self->l( $arg_r->{method} )
          : $self->perr('method');

        # +------------------------------------------------------------------+
        # | prepare
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');
        if ( ref $rbac_scope_hr eq 'HASH' ) {
            $logger->debug('> rbac_scope_hr: HASH');
        }
        $logger->debug("> scope:         $scope");
        $logger->debug("> level:         $level");
        $logger->debug("> mode:          $mode");
        $logger->debug("> object:        $object");

        # object has access to itself ( object->object )
        my $related_object = $object;
        $logger->debug("$level SET $object -> $related_object = 1");
        $adjacency_matrix->{$object}->{$related_object} = 1;

        my $related_object_ar = $self->get_object_list(
            {
                mode          => 'member',
                rbac_scope_hr => $rbac_scope_hr,
                level         => $level,
                scope         => $scope,
                method        => $method,
                object        => $related_object,
            }
        );
        foreach my $related_object ( @{$related_object_ar} ) {
            $logger->debug("$level SET $object -> $related_object = 1");
            $adjacency_matrix->{$object}->{$related_object} = 1;
        }
        $logger->debug( 'adjancency: ',
            { filter => \&Dumper, value => $adjacency_matrix } );

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return;

    }

    sub cat_modules_by_user {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $graph =
          exists $arg_r->{graph}
          ? $arg_r->{graph}
          : $self->perr('graph');

        my $user =
          exists $arg_r->{user}
          ? $self->l( $arg_r->{user} )
          : $self->perr('user');

        # +------------------------------------------------------------------+
        # | prepare
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');
        $logger->debug("query cat module for user [$user]");

        #$logger->debug('using graph', { filter => \&Dumper, value => $graph} );

        my %opt = ();
        $opt{first_root} = $user;
        $logger->debug( 'first user: ', $user );
        $opt{next_root} = undef;

        # revert direction of arrows
        my $t = $graph->transpose_graph();

        # clip only to a given user and reduce arrows
        my $sptg = $t->SPT_Dijkstra(%opt);

        # return only leaf elements
        my @return = $sptg->sink_vertices;
        $logger->debug( 'return valid modules',
            { filter => \&Dumper, value => \@return } );

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return \@return;

    }

    sub users_by_cat_module {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $graph =
          exists $arg_r->{graph}
          ? $arg_r->{graph}
          : $self->perr('graph');

        my $user =
          exists $arg_r->{module}
          ? $self->l( $arg_r->{module} )
          : $self->perr('module');

        # +------------------------------------------------------------------+
        # | prepare
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');
        $logger->debug( '-> user: ', $user );
        $logger->debug('-> graph: (not printed)');

        my %opt = ();
        $opt{first_root} = $user;
        $opt{next_root}  = undef;

        # clip only to a given user and reduce arrows
        my $sptg = $graph->SPT_Dijkstra(%opt);

        # return only leaf elements
        my @return = $sptg->sink_vertices;
        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return \@return;

    }

    # +=======================================================================+
    # || get_acl                                                             ||
    # +=======================================================================+

    sub get_acl {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $scope =
          exists $arg_r->{scope}
          ? $self->l( $arg_r->{scope} )
          : $self->perr('scope');

        # +------------------------------------------------------------------+
        # | prepare
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');
        $logger->debug( '-> scope: ', $scope );

        my $acl_hr = {};

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return $acl_hr;
    }

    # +=======================================================================+
    # || set_acl                                                             ||
    # +=======================================================================+

    sub set_acl {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $scope =
          exists $arg_r->{scope}
          ? $self->l( $arg_r->{scope} )
          : $self->perr('scope');

        my $acl_hr =
          exists $arg_r->{acl_hr}
          ? $self->lh( $arg_r->{acl_hr} )
          : $self->perr('acl_hr');

        # +------------------------------------------------------------------+
        # | prepare
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('SUB set_acl');

        #my $acl_hr = {};

        $logger->debug('SUB set_acl');

        # +------------------------------------------------------------------+
        # | API
        return $acl_hr;
    }

    # +=======================================================================+
    # || read_xml_file                                                      ||
    # +=======================================================================+

    sub read_xml_file {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $file =
          exists $arg_r->{file}
          ? $self->l( $arg_r->{file} )
          : $self->perr('file');

        # +------------------------------------------------------------------+
        # | prepare
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        my $reader = Graph::Reader::XML->new();
        my $graph  = $reader->read_graph($file);

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return $graph;

    }

    # +=======================================================================+
    # || write_xml_file                                                      ||
    # +=======================================================================+

    sub write_xml_file {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $file =
          exists $arg_r->{file}
          ? $self->l( $arg_r->{file} )
          : $self->perr('file');

        my $graph =
          exists $arg_r->{graph}
          ? $arg_r->{graph}
          : $self->perr('graph');

        # +------------------------------------------------------------------+
        # | prepare
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');
        $logger->debug( '-> file: ', $file );
        $logger->debug('-> graph: (not printed)');

        my $writer = Graph::Writer::XML->new();
        $writer->write_graph( $graph, $file );

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return 1;

    }

    # +=======================================================================+
    # || write_dot_file                                                      ||
    # +=======================================================================+

    sub write_dot_file {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $file =
          exists $arg_r->{file}
          ? $self->l( $arg_r->{file} )
          : $self->perr('file');

        my $graph =
          exists $arg_r->{graph}
          ? $arg_r->{graph}
          : $self->perr('graph');

        # +------------------------------------------------------------------+
        # | prepare
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');
        $logger->debug( '-> file: ', $file );
        $logger->debug('-> graph: (not printed)');

        my $writer = Graph::Writer::Dot->new();
        $writer->write_graph( $graph, $file );

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return 1;

    }

    # +=======================================================================+
    # || access                                                              ||
    # +=======================================================================+

    sub access {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $graph =
          ( exists $arg_r->{graph} and defined $arg_r->{graph} )
          ? $arg_r->{graph}
          : $self->perr('graph');

        my $source =
          exists $arg_r->{source}
          ? $self->l( $arg_r->{source} )
          : $self->perr('source');

        my $destination =
          exists $arg_r->{destination}
          ? $self->l( $arg_r->{destination} )
          : $self->perr('destination');

        # +------------------------------------------------------------------+
        # | prepare
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');
        $logger->debug( '-> source:      ', $source );
        $logger->debug( '-> destination: ', $destination );
        $logger->debug( '-> graph:       ', '(not printed)' );

        my $tcg = $graph->TransitiveClosure_Floyd_Warshall;

        if ( $tcg->is_reachable( $destination, $source ) ) {
            $logger->debug('access TRUE (1)');
            $logger->debug('END');
            return 1;
        }
        else {
            $logger->debug('access FALSE (0)');
            $logger->debug('END');
            return 0;
        }

        $logger->debug('access UNKNOWN (undef)');
        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return;
    }

    # +======================================================================+
    # || list_rbac                                                          ||
    # +======================================================================+

    sub list_rbac {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $rbac_scope_hr =
          exists $arg_r->{rbac_scope_hr}
          ? $self->h( $arg_r->{rbac_scope_hr} )
          : $self->perr('rbac_scope_hr');

        # +------------------------------------------------------------------+
        # | prepare
        my $logger = get_logger(__PACKAGE__);

        $logger->debug('BEGIN');

        my @return;

        my $cfg_hr = $rbac_scope_hr;
        $logger->debug( 'cfg_hr: ', { filter => \&Dumper, value => $cfg_hr } );

        foreach my $scope ( sort keys %{$cfg_hr} ) {
            $logger->debug( 'scope: ', $scope );
            push @return, $scope;
        }

        $logger->debug('END');

        # +------------------------------------------------------------------+
        # | API
        return \@return;

    }

    sub query_via_task {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $task =
          exists $arg_r->{task}
          ? $self->l( $arg_r->{task} )
          : $self->perr('task');

        my $object =
          exists $arg_r->{object}
          ? $self->l( $arg_r->{object} )
          : $self->perr('object');

        # +------------------------------------------------------------------+
        # | main
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        my $return_hr = {};

        eval { require CipUX::Task; };
        if ($EVAL_ERROR) {
            my $msg = 'Can not use CipUX::Task for RBAC access';
            $msg .= 'information, CipUX::Task not installed!';
            $logger->debug($msg);
            croak $msg;
        }
        else {
            $logger->debug('CipUX::Task installed, using task command!');

            # prepare the task library
            my $cipux = CipUX::Task->new();

            # everything is packed we will start the journey
            my %mattrvalue = ();
            if ( defined $object ) {
                $logger->debug("exec task: $task -o $object");
            }
            else {
                $logger->debug("exec task: $task");
            }

            $return_hr = $cipux->task(

                {
                    script  => $SCRIPT,
                    task    => $task,
                    mode    => 'cmd',
                    object  => $object,
                    attr_hr => \%mattrvalue,

                }
            );

        }
        $logger->debug( 'TASK return_hr:',
            { filter => \&Dumper, value => $return_hr } );

        # +------------------------------------------------------------------+
        # | API
        return $return_hr->{taskres_r};
    }

}    # END INSIDE-OUT CLASS

1;

__END__

=pod

=head1 NAME

CipUX::RBAC - RBAC class for CipUX

=head1 VERSION

3.4.0

=head1 SYNOPSIS

  use CipUX::RBAC;

=head1 DESCRIPTION

Provides the functions Role Based Access Control.

=head1 ABSTRACT

The CipUX RBAC class provides services to CipUX XML-RPC (CipUX::RPC) 
server and the CipUX Administration Tool CAT (CipUX::CAT::Web).

=head1 CONFIGURATION AND ENVIRONMENT

TODO

=head1 DEPENDENCIES

Carp
Scalar::Util
Pod::Usage
Data::Dumper
CipUX
CipUX::Task
Graph
Graph::Directed
Graph::Reader::XML
Graph::Writer::XML
Graph::Writer::Dot

=head1 SUBROUTINES/METHODS

The following functions will be exported by CipUX::RBAC.


=head2 rbac_config

TODO

=head2 access

Test the role based access via transitive closure graph.

=head2 construct_directed_graph

creates a direced graph.

=head2 get_acl

TODO

=head2 get_acl_for_cat_module

TODO

=head2 get_acl_for_task

TODO

=head2 get_object_list

TODO

=head2 get_related_objects

TODO

=head2 get_task

TODO

=head2 read_xml_file

Reads graph from an XML file.

=head2 retreive_api

TODO

=head2 retreive_cfg

TODO

=head2 retreive_explain

TODO

=head2 set_acl

TODO

=head2 write_dot_file

Writes graph to a dot file. This can be view with dotty or some other tool.

=head2 write_xml_file

Writes graph to an XML file.

=head2 list_rbac

List CipUX RBAC scope class commands.

B<Syntax:>

 $cipux->list_rbac({ });

=head2 cat_modules_by_user

TODO

=head2 users_by_cat_module

TODO

=head2 query_via_rpc

TODO

=head2 query_via_task

TODO

=head2 xmlrpc

TODO

=head1 DIAGNOSTICS

TODO

=head1 INCOMPATIBILITIES

Not known.

=head1 BUGS AND LIMITATIONS

Not known.

=head1 SEE ALSO

See the CipUX webpage and the manual at L<http://www.cipux.org>

See the mailing list L<http://www.cipux.org/mailman/listinfo/>

=head1 AUTHOR

Christian Kuelker  E<lt>christian.kuelker@cipworx.orgE<gt>

=head1 LICENSE AND COPYRIGHT

Copyright (C) 2008 by Christian Kuelker

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2, or (at
your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA

=cut

