#ifndef H_CDW_WIDGETS
#define H_CDW_WIDGETS

#include <stddef.h> /* size_t */

#include "main.h"
#include "cdw_colors.h"
#include "cdw_ncurses.h"



/* widget IDs, useful e.g. for selecting constructors
   when building form with many widgets */
#define CDW_WIDGET_LABEL     1
#define CDW_WIDGET_CHECKBOX  2
#define CDW_WIDGET_INPUT     3
#define CDW_WIDGET_DROPDOWN  4
#define CDW_WIDGET_BUTTON    5
#define CDW_WIDGET_TEXT      6 /* like label, but larger, and with special requirements */


/* some widgets have table of return keys - keys on which
   a driver returns control to caller; 20 seems to be reasonable
   limit at this time; there is no guard, widgets have to maintain
   n_return_keys counter */
#define N_RETURN_KEYS_MAX 20


/* a dialog can be either dialog with input field, or dialog with
   buttons; in future it may be possible that the two will be
   possible at the same time */
enum {
	CDW_DIALOG_BUTTONS = 1,
	CDW_DIALOG_INPUT = 2
};




/* number and type of buttons in dialog with buttons */
enum {
	CDW_BUTTONS_OK = 0,
	CDW_BUTTONS_OK_CANCEL = 3,
	CDW_BUTTONS_YES_NO_CANCEL = 4,
};



typedef struct {

	const char *window_title;    /* title of input window and error dialog windows */
	const char *prompt_message;  /* message in input window */
	char *error_message;   /* error message window displayed at the end in case of failure */

	int attempts_max;      /* how many times should the widget ask for correct (i.e. without unsafe chars) string? */

	int input_type;        /* CDW_NCURSES_INPUT_{NONE | HIDDEN | INTEGER | NUMERIC } */

	int chars_max;      /* maximal allowed length of entered string, zero for no limit */
	char **buffer;         /* pointer to data buffer, the buffer must be provided by code using safe dialog */

} cdw_safe_input_data_t;


struct cdw_form_type;
typedef struct cdw_form_type cdw_form_t;
typedef int (* cdw_form_widget_function_t)(cdw_form_t *, void *);

typedef struct {
	WINDOW *parent;
	WINDOW *strip;

	cdw_form_widget_function_t on_click;

	int begin_x;
	int begin_y;

	char *label;
	size_t label_len;

	cdw_colors_t colors;
} CDW_BUTTON;



typedef struct {
	WINDOW *parent;
	int parent_x;
	int parent_y;

	int width;

	/* labels for old-style dropdowns, new-style dropdowns use 'cdw_dropdown_item_t **items' */
	const char **labels;

	MENU *menu;
	ITEM **menu_items;

	/* the difference between n_items_max and n_items is this:
	   when you initially create "new style" dropdown you may not know
	   exact number of items in the dropdown (which may depend on some
	   conditions), but you are sure that the number will be no larger
	   than some value. This maximal value is n_items_max, and the
	   actual number of items added to dropdown is n_items.
	   n_items_max must be specified when you call cdw_dropdown_new(),
	   and value of n_items is known when you add all necessary dropdown
	   items */
	int n_items_max;
	int n_items;

	int current_item_ind;

	cdw_colors_t colors;

	bool visible;

	WINDOW *menu_window;
	WINDOW *menu_window_sub;
	int begin_x;
	int begin_y;

	cdw_id_label_t **items;

} CDW_DROPDOWN;


/* align of text in textarea widget */
enum {
	CDW_ALIGN_LEFT,
	CDW_ALIGN_RIGHT,
	CDW_ALIGN_CENTER
};




typedef struct {
	WINDOW *parent;

	WINDOW *win;
	int n_cols;  /* width - number of columns of input field */
	/* no height field, it is always equal 1 */
	cdw_colors_t colors;

	int begin_y;
	int begin_x;

	int chars_max;  /* maximum number of chars that user can enter into input line */
	int input_type;

	FIELD *field[2];
	FORM *form;

	/* a hack: should input line driver save data from field buffer to data buffer? */
	bool save_data_on_tab;
} CDW_INPUT_LINE;



CDW_INPUT_LINE *cdw_input_line_new(WINDOW *parent, int begin_y, int begin_x, int n_cols, const char *initial_string, int input_type, int chars_max);
void cdw_input_line_delete(CDW_INPUT_LINE **input_line);
// int cdw_input_line_driver(CDW_INPUT_LINE *input_line, char **buffer);
int cdw_input_line_safe_driver(CDW_INPUT_LINE *input_line, char **buffer, int attempts_max);
void cdw_input_line_refresh_with_string(CDW_INPUT_LINE *input_line, const char *string);

// cdw_rv_t cdw_one_line_form(WINDOW *window, WINDOW *subwindow, char **buffer, int input_type, size_t limit);
cdw_rv_t cdw_buttons_dialog(const char *title, const char *message, int buttons_type, cdw_colors_t colors);
cdw_rv_t cdw_safe_input_dialog(cdw_safe_input_data_t *data);


void cdw_put_ckbox(WINDOW *win, int y, int x);

CDW_BUTTON *cdw_button_new(WINDOW *parent, int begin_y, int begin_x, const char *label, cdw_colors_t colors);
void cdw_button_delete(CDW_BUTTON **button);
void cdw_button_free(CDW_BUTTON *button);
void cdw_button_focus(CDW_BUTTON *button);
void cdw_button_unfocus(CDW_BUTTON *button);

int cdw_textarea_print_message(WINDOW *window, const char *message, int align);


CDW_DROPDOWN *cdw_dropdown_new(WINDOW *parent, int begin_y, int begin_x, int width, int n_items_max, cdw_colors_t colors);
CDW_DROPDOWN *cdw_dropdown_maker_wrapper(WINDOW *window, int begin_y, int begin_x, int width, int n_items, cdw_id_clabel_t *items);
cdw_rv_t      cdw_dropdown_add_item(CDW_DROPDOWN *dropdown, cdw_id_t id, const char *label);
cdw_rv_t      cdw_dropdown_finalize(CDW_DROPDOWN *dropdown);
cdw_rv_t      cdw_dropdown_delete(CDW_DROPDOWN **dropdown);
void          cdw_dropdown_free(CDW_DROPDOWN *dropdown);
cdw_rv_t      cdw_dropdown_driver(CDW_DROPDOWN *dropdown);
void          cdw_dropdown_display_current_item(CDW_DROPDOWN *dropdown);

void cdw_dropdown_focus(CDW_DROPDOWN *dropdown);
void cdw_dropdown_unfocus(CDW_DROPDOWN *dropdown);
void cdw_dropdown_make_visible(CDW_DROPDOWN *dropdown);
void cdw_dropdown_make_invisible(CDW_DROPDOWN *dropdown);

cdw_rv_t    cdw_dropdown_set_current_item_by_id(CDW_DROPDOWN *dropdown, cdw_id_t id);
cdw_rv_t    cdw_dropdown_set_current_item_by_ind(CDW_DROPDOWN *dropdown, int ind);
int         cdw_dropdown_get_current_item_ind(CDW_DROPDOWN *dropdown);
cdw_id_t    cdw_dropdown_get_current_item_id(CDW_DROPDOWN *dropdown);
const char *cdw_dropdown_get_current_item_label(CDW_DROPDOWN *dropdown);
const char *cdw_dropdown_get_label_by_ind(CDW_DROPDOWN *dropdown, int ind);
const char *cdw_dropdown_get_label_by_id(CDW_DROPDOWN *dropdown, cdw_id_t id);



typedef struct {
	WINDOW *parent;
	int begin_y;
	int begin_x;
	bool checked;
	bool visible;

	cdw_form_widget_function_t on_toggle;
} CDW_CHECKBOX;


CDW_CHECKBOX *cdw_checkbox_new(WINDOW *parent, int begin_y, int begin_x, bool checked);
void cdw_checkbox_delete(CDW_CHECKBOX **checkbox);
void cdw_checkbox_free(CDW_CHECKBOX *checkbox);
bool cdw_checkbox_toggle(CDW_CHECKBOX *checkbox);
bool cdw_checkbox_get_state(CDW_CHECKBOX *checkbox);
bool cdw_checkbox_set_state(CDW_CHECKBOX *checkbox, bool checked);
void cdw_checkbox_set_visibility(CDW_CHECKBOX *checkbox, bool visible);
void cdw_checkbox_draw(CDW_CHECKBOX *checkbox);


#endif /* ifndef H_CDW_WIDGETS */
