/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2011 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <string.h>

#include "cdw_task.h"
#include "gettext.h"
#include "cdw_string.h"
#include "cdw_widgets.h"
#include "cdw_debug.h"
#include "cdw_mkisofs_options.h"
#include "cdw_iso9660.h"


static cdw_task_t *local_task = (cdw_task_t *) NULL;



static FIELD *page_mkisofs_fields[CDW_MKISOFS_OPTIONS_N_FIELDS + 1]; /* +1 for last field == NULL */

static CDW_DROPDOWN *cdw_mkisofs_options_make_rock_ridge_dropdown(WINDOW *window, int begin_y, int begin_x, int width);
static CDW_DROPDOWN *cdw_mkisofs_options_make_iso_level_dropdown(WINDOW *window, int begin_y, int begin_x, int width);

static cdw_form_dropdown_maker_t dropdown_makers[] = {
	cdw_mkisofs_options_make_rock_ridge_dropdown,
	cdw_mkisofs_options_make_iso_level_dropdown,
};


enum {
	f_mkisofs_joliet_l,
	f_mkisofs_joliet_cb,
	f_mkisofs_rock_ridge_l,
	f_mkisofs_rock_ridge_dd,
	f_mkisofs_follow_symlinks_l,
	f_mkisofs_follow_symlinks_cb,
	f_mkisofs_iso_level_l,
	f_mkisofs_iso_level_dd,
	f_mkisofs_joliet_long_l,
	f_mkisofs_joliet_long_cb,
	f_mkisofs_pad_l,
	f_mkisofs_pad_cb,
	f_mkisofs_root_l,
	f_mkisofs_root_i,
	f_mkisofs_other_l,
	f_mkisofs_other_i };



cdw_rv_t cdw_mkisofs_options_form(cdw_form_t *cdw_form, void *cdw_task, int first_col, int second_col, int width_wide, int width_narrow)
{
	cdw_task_t *task = (cdw_task_t *) cdw_task;
	local_task = task;
	cdw_form->fields = page_mkisofs_fields;

	int rr_width = 15;
	cdw_form_descr_t descr[] = {
		/*     type      begin_y   begin_x      n_cols    n_lines   field enum                    data1                                          data2 */

		{ CDW_WIDGET_LABEL,    1,  first_col,   width_narrow,   1,  f_mkisofs_joliet_l,           _("Joliet information"),                           0 },
		{ CDW_WIDGET_CHECKBOX, 1,  second_col,  0,              0,  f_mkisofs_joliet_cb,          (void *) NULL,  task->create_image.joliet_information ? 1 : 0 },

		{ CDW_WIDGET_LABEL,    2,  first_col,   width_narrow,   1,  f_mkisofs_rock_ridge_l,       _("Rock Ridge information"),                       0 },
		{ CDW_WIDGET_DROPDOWN, 2,  second_col,  rr_width,       1,  f_mkisofs_rock_ridge_dd,      dropdown_makers,                                   0 },

		{ CDW_WIDGET_LABEL,    3,  first_col,   width_narrow,   1,  f_mkisofs_follow_symlinks_l,  _("Follow symbolic links"),                        0 },
		{ CDW_WIDGET_CHECKBOX, 3,  second_col,  0,              0,  f_mkisofs_follow_symlinks_cb, (void *) NULL,     task->create_image.follow_symlinks ? 1 : 0 },

		{ CDW_WIDGET_LABEL,    4,  first_col,   width_narrow,   1,  f_mkisofs_iso_level_l,        _("ISO level"),                                    0 },
		{ CDW_WIDGET_DROPDOWN, 4,  second_col,  rr_width,       1,  f_mkisofs_iso_level_dd,       dropdown_makers,                                   1 },

		{ CDW_WIDGET_LABEL,    5,  first_col,   width_narrow,   1,  f_mkisofs_joliet_long_l,      _("Long Joliet file names"),                       0 },
		{ CDW_WIDGET_CHECKBOX, 5,  second_col,  1,              1,  f_mkisofs_joliet_long_cb,     (void *) NULL, task->create_image.mkisofs.joliet_long ? 1 : 0 },

		{ CDW_WIDGET_LABEL,    6,  first_col,   width_narrow,   1,  f_mkisofs_pad_l,              _("Pad at the end"),                               0 },
		{ CDW_WIDGET_CHECKBOX, 6,  second_col,  0,              0,  f_mkisofs_pad_cb,             (void *) NULL,        task->create_image.pad ? 1 : 0 },

		{ CDW_WIDGET_LABEL,    8,  first_col,   width_wide,     1,  f_mkisofs_root_l,             _("Root directory:"),                              0 },
		{ CDW_WIDGET_INPUT,    9,  first_col,   width_wide - 3, 1,  f_mkisofs_root_i,             task->create_image.mkisofs.root_dir,               0 },

		{ CDW_WIDGET_LABEL,   10,  first_col,   width_wide,     1,  f_mkisofs_other_l,            _("Other mkisofs options:"),                       0 },
		{ CDW_WIDGET_INPUT,   11,  first_col,   width_wide - 3, 1,  f_mkisofs_other_i,            task->create_image.mkisofs.other_mkisofs_options,  0 },

		/* guard */
		{ -1,                  0,  0,           0,              0,  0,                            (void *) NULL,                                     0 }};

	cdw_rv_t crv = cdw_form_description_to_fields(descr, cdw_form);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_ERROR;
	} else {
		return CDW_OK;
	}
}





#define N_ISO_LEVEL_ITEMS_MAX 4 /* iso level = 1, 2, 3, 4; id == level */
static cdw_id_clabel_t iso_items[N_ISO_LEVEL_ITEMS_MAX] = {
	/* 2TRANS: do not translate */
	{ 1,       gettext_noop("1") },
	/* 2TRANS: do not translate */
	{ 2,       gettext_noop("2") },
	/* 2TRANS: do not translate */
	{ 3,       gettext_noop("3") },
	/* 2TRANS: do not translate */
	{ 4,       gettext_noop("4") } };


CDW_DROPDOWN *cdw_mkisofs_options_make_iso_level_dropdown(WINDOW *window, int begin_y, int begin_x, int width)
{
	cdw_id_t iso_level = local_task->create_image.mkisofs.iso_level;
	cdw_assert (iso_level > 0 && iso_level < 5,
		    "ERROR: invalid value of iso level: %lld\n", iso_level);

	CDW_DROPDOWN *dd = cdw_dropdown_maker_wrapper(window, begin_y, begin_x, width, N_ISO_LEVEL_ITEMS_MAX, iso_items);
	cdw_dropdown_set_current_item_by_id(dd, iso_level);
	return dd;
}






#define N_RR_ITEMS_MAX 3 /* none/useful/full */
static cdw_id_clabel_t rr_items[N_RR_ITEMS_MAX] = {
	/* 2TRANS: none = no Rock Ridge extensions */
	{ CDW_ISO9660_RR_NONE,         gettext_noop("None") },
	/* 2TRANS: useful = some values of attributes of file system are set to some more useful values */
	{ CDW_ISO9660_RR_USEFUL,       gettext_noop("Useful") },
	/* 2TRANS: full = full Rock Ridge extensions */
	{ CDW_ISO9660_RR_FULL,         gettext_noop("Full") } };


CDW_DROPDOWN *cdw_mkisofs_options_make_rock_ridge_dropdown(WINDOW *window, int begin_y, int begin_x, int width)
{
	cdw_id_t rock_ridge = local_task->create_image.rock_ridge;
	cdw_assert (rock_ridge >= CDW_ISO9660_RR_NONE && rock_ridge <= CDW_ISO9660_RR_FULL,
		    "ERROR: invalid value of Rock Ridge: %lld\n", rock_ridge);
	CDW_DROPDOWN *dd = cdw_dropdown_maker_wrapper(window, begin_y, begin_x, width, N_RR_ITEMS_MAX, rr_items);
	cdw_dropdown_set_current_item_by_id(dd, rock_ridge);

	return dd;
}





cdw_rv_t cdw_mkisofs_options_validate(cdw_form_t *cdw_form, int *fi)
{
	char *s = cdw_ncurses_get_field_string(*(cdw_form->fields + f_mkisofs_root_i));
	cdw_rv_t crv = cdw_string_security_parser(s, (char *) NULL);
	if (crv != CDW_OK) {
		*fi = f_mkisofs_root_i;
		return CDW_NO;
	}

	s = cdw_ncurses_get_field_string(*(cdw_form->fields + f_mkisofs_other_i));
	crv = cdw_string_security_parser(s, (char *) NULL);
	if (crv != CDW_OK) {
		*fi = f_mkisofs_other_i;
		return CDW_NO;
	}

	return CDW_OK;
}





cdw_rv_t cdw_mkisofs_options_save(cdw_form_t *cdw_form, cdw_task_t *task)
{
	task->create_image.joliet_information = cdw_form_get_checkbox_state(cdw_form, f_mkisofs_joliet_cb);


	CDW_DROPDOWN *dd = cdw_form_get_dropdown(cdw_form, f_mkisofs_rock_ridge_dd);
	cdw_id_t rr = cdw_dropdown_get_current_item_id(dd);
	cdw_assert (rr == CDW_ISO9660_RR_NONE
		    || rr == CDW_ISO9660_RR_USEFUL
		    || rr == CDW_ISO9660_RR_FULL, "ERROR: field stores invalid value of Rock Ridge: %lld\n", rr);
	task->create_image.rock_ridge = rr;


	task->create_image.follow_symlinks = cdw_form_get_checkbox_state(cdw_form, f_mkisofs_follow_symlinks_cb);


	dd = cdw_form_get_dropdown(cdw_form, f_mkisofs_iso_level_dd);
	int il = (int) cdw_dropdown_get_current_item_id(dd);
	cdw_assert (il >=1 && il <= 4,
		    "ERROR: field stores invalid value of ISO level: %d\n", il);
	task->create_image.mkisofs.iso_level = il;


	task->create_image.mkisofs.joliet_long = cdw_form_get_checkbox_state(cdw_form, f_mkisofs_joliet_long_cb);


	task->create_image.pad = cdw_form_get_checkbox_state(cdw_form, f_mkisofs_pad_cb);


	cdw_string_set(&(task->create_image.mkisofs.root_dir), cdw_ncurses_get_field_string(*(cdw_form->fields + f_mkisofs_root_i)));
	cdw_string_set(&(task->create_image.mkisofs.other_mkisofs_options), cdw_ncurses_get_field_string(*(cdw_form->fields + f_mkisofs_other_i)));

	return CDW_OK;
}


