/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2010 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#define _BSD_SOURCE /* PATH_MAX */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
// #include <dirent.h> /* PATH_MAX */

#include <ncursesw/ncurses.h>

#include "cdw_colors.h"
#include "cdw_string.h"
#include "cdw_fs.h"
#include "cdw_debug.h"



/**
   This file implements
    - reading color scheme (pairs of color attributes for fore- and
      background color) from cdw colors configuration file
   - passing those pairs to ncurses function init_pair() to initialize
      'curses pair' (pair in strict, ncurses-related sense, see
      'man init_pair' for reference)
    - a mechanism of referring to these 'curses pairs' by symbolic names
      in form of CDW_COLORS_MAIN or CDW_COLORS_MENU etc.

   Conventions used in this file:
   - 'color attribute' is variable with name of form CONF_COLOR_MAIN_FORE
     or CONF_COLOR_DIALOG_BACK - an int value describing color of foreground
     or background of some class of UI elements (1 is class of elements
     used in main window, 2 is class of elements used to build dialog
     windows, see list below)
   - 'pair' is used to call two color attributes describing fore- and
     background of the same class of UI elements. Values of pair of the
     attributes are used as 2nd and 3rd arguments to init_pair() function
     call. First argument of this function call is a 'ncurses pair' - pair
     in strict sense. Such pairs are referred to in code in other files by
     symbolic names such as CDW_COLORS_WARNING
   - 'line' is one line read from cdw color configuration file

   CDW uses following pairs of foreground-background colors (values of type
   cdw_colors_t):
   1 -> CDW_COLORS_MAIN - main window
   2 -> CDW_COLORS_DIALOG - dialog windows of various kinds - from simple message boxes to prefs window
   3 -> CDW_COLORS_TITLE -  application title in main window
   4 -> CDW_COLORS_MENU - main application menu (left side of main window)
   5 -> CDW_COLORS_PROGRESS - progress bar
   6 -> CDW_COLORS_INPUT - input fields
   7 -> CDW_COLORS_WARNING - warning - may be used in dialogs
   8 -> CDW_COLORS_SIZE_INFO - (unused, kept for compatibility)
   9 -> CDW_COLORS_DISABLED - disabled
   10 -> CDW_COLORS_UNUSED - unused
   11 -> CDW_COLORS_TOOLTIPS - tooltips at the bottom of main window
   12 -> CDW_COLORS_TEXTAREA - text areas, like window displaying license
   13 -> CDW_COLORS_ERROR - error - may be used in dialogs

   These values should be used when passing color argument to cdw widgets, e.g.
   cdw_dialogbox("Message",
                 "This is how you should use values of type cdw_colors_t",
		 DIALOG_OK, CDW_COLORS_DIALOG);

   When setting colors of bare ncurses widgets you should use COLORS_PAIR()
   macro, like this:
   wbkgd(my_window, COLOR_PAIR(CDW_COLORS_DIALOG));
*/


enum {
	CONF_EMPTY = -3,      /* empty line */
	CONF_INVALID = -2,    /* invalid entry */
	CONF_COMMENT = -1,    /* comment, line starting with '#' */
	CONF_COLOR_MAIN_FORE = 0,
	CONF_COLOR_MAIN_BG,
	CONF_COLOR_DIALOG_FORE,
	CONF_COLOR_DIALOG_BG,
	CONF_COLOR_TITLE_FORE,
	CONF_COLOR_TITLE_BG,
	CONF_COLOR_MENU_FORE,
	CONF_COLOR_MENU_BG,
	CONF_COLOR_PROGRESS_FORE,
	CONF_COLOR_PROGRESS_BG,
	CONF_COLOR_INPUT_FORE,
	CONF_COLOR_INPUT_BG,
	CONF_COLOR_WARNING_FORE,
	CONF_COLOR_WARNING_BG,
	CONF_COLOR_SIZE_INFO_FORE,
	CONF_COLOR_SIZE_INFO_BG,
	CONF_COLOR_DISABLED_FORE,
	CONF_COLOR_DISABLED_BG,
	CONF_COLOR_UNUSED_FORE,
	CONF_COLOR_UNUSED_BG,
	CONF_COLOR_TOOLTIPS_FORE,
	CONF_COLOR_TOOLTIPS_BG,
	CONF_COLOR_TEXTAREA_FORE,
 	CONF_COLOR_TEXTAREA_BG,
	CONF_COLOR_ERROR_FORE,
	CONF_COLOR_ERROR_BG
};



static int get_attribute_name(char *line);
static int get_attribute_value(const char *source);
static void read_colors_config_file(FILE *file);


/**
   \brief Set up defaut (hardwired) color schemes

   Use this function right after setting up basic curses to
   provide colors for some widgets. It is required to have some
   colors defined when fs module is not yest set up, because you
   might have to dialog or input widgets to configure fs module.
*/
void curses_colors_init_phase1(void)
{
	/* first set default values: foreground - background */
	init_pair(CDW_COLORS_MAIN, COLOR_WHITE, COLOR_BLACK);
	init_pair(CDW_COLORS_DIALOG, COLOR_WHITE, COLOR_BLUE);
	init_pair(CDW_COLORS_TITLE, COLOR_RED, COLOR_BLACK);
	init_pair(CDW_COLORS_MENU, COLOR_WHITE, COLOR_BLACK);
	init_pair(CDW_COLORS_PROGRESS, COLOR_YELLOW, COLOR_BLACK);
	init_pair(CDW_COLORS_INPUT, COLOR_WHITE, COLOR_BLACK);
	init_pair(CDW_COLORS_WARNING, COLOR_BLUE, COLOR_YELLOW);
	init_pair(CDW_COLORS_SIZE_INFO, COLOR_WHITE, COLOR_YELLOW);
	init_pair(CDW_COLORS_DISABLED, COLOR_BLACK, COLOR_WHITE);
	init_pair(CDW_COLORS_UNUSED, COLOR_WHITE, COLOR_RED);
	init_pair(CDW_COLORS_TOOLTIPS, COLOR_YELLOW, COLOR_BLACK);
	init_pair(CDW_COLORS_TEXTAREA, COLOR_WHITE, COLOR_BLACK);
	/* FIXME: black letters on red background aren't too readable,
	   fix this by using other color scheme for error dialogs */
	init_pair(CDW_COLORS_ERROR, COLOR_BLACK, COLOR_RED);

	return;
}





/**
   \brief Read custom colors from config file, use them to set up color schemes

   Call this function when fs module is set up.
*/
cdw_rv_t curses_colors_init_phase2(void)
{

	const char *home_dir_fullpath = cdw_fs_get_home_dir_fullpath();
	if (home_dir_fullpath == (char *) NULL) {
		return CDW_GEN_ERROR;
	}
	cdw_vdm ("home dir full path is \"%s\"\n", home_dir_fullpath);

	/* dir name is already ended with '/' */
	size_t len = strlen(home_dir_fullpath) + strlen(".cdw.colors");
	/* PATH_MAX includes ending null */
	if (len > PATH_MAX - 1) {
		return CDW_GEN_ERROR;
	}

	char *path = (char *) malloc(len + 1);
	if (path == (char *) NULL) {
		return CDW_MEM_ERROR;
	} else {
		/* snprintf() prints ending '\0' */
		snprintf(path, len + 1, "%s.cdw.colors", home_dir_fullpath);
		FILE *f = fopen(path, "r");
		free(path);
		path = (char *) NULL;

		if (f == (FILE *) NULL) {
			cdw_vdm ("can't open color file\n");
			return CDW_SYS_ERROR;
		} else {
			read_colors_config_file(f);
			fclose(f);
			cdw_vdm ("color file open and read properly\n");
			return CDW_OK;
		}
	}
}





/**
   Read cdw colors configuration file

   Read color scheme selected by user from given configuration file.
   Values read from file will be used as arguments of init_pair() calls.

   \param file - color configuration file reference, it has to be already open
*/
void read_colors_config_file(FILE *file)
{
	cdw_assert (file != (FILE *) NULL, "you forgot to open file\n");
	/* this function is called when there is some color config file,
	   let's change default values */

	short int colors[(CDW_COLORS_GUARD - 1) * 2]; /* there can't be more color attributes than (currently) 26 */
	char buffer[999];
	int fb = 0; /* indicates which value was read in: fore- or background; even values are for fore */
	int i = 0;

	short int fore = 0; /* temporary variable for storing foreground color */
	short int back = 0; /* temporary variable for storing background color */

	while (fgets(buffer, sizeof(buffer), file) != NULL) {
		char *line = cdw_string_ltrim(buffer);

		switch (i = get_attribute_name(line)) {
			case CONF_COMMENT:
			case CONF_EMPTY:
				break;
			default:
				/* code in 'default' branch checks if both
				   foreground and background colors were read
				   in properly, and if so, then places them
				   in color attributes table; if one or both
				   of values are invalid, they are discarded,
				   and default values are used (are not
				   overwritten) */

				if (fb % 2 == 0) { /* this is line with foreground */
					fore = (short int) get_attribute_value(line);
					cdw_sdm ("fb = %d, fore = %d, line = '%s'\n", fb, fore, line);
					fb++;
				} else { /* this is line with background */
					back = (short int) get_attribute_value(line);
					cdw_sdm ("fb = %d, back = %d, line = '%s'\n", fb, back, line);
					fb++;
				}

				/* if fb is even (e.g. after reading two
				   values: first 'fore', then 'back', then it
				   is time to check the two values and if they
				   are valid then save them in colors[] */

				if (fb % 2 == 0) {
					if (fore == CONF_INVALID || back == CONF_INVALID) {
						; /* just skip those two entries */
					} else {
						/* -1 because foreground was read in previous loop cycle */
						colors[i - 1] = fore;
						colors[i] = back;
					}
				}
				break;
		} /* switch */
	} /* while */

	/* CDW_COLORS_GUARD - 1 is maximal amount of color pairs */
	cdw_colors_t j = 0;
	for (j = CDW_COLORS_MAIN; j < CDW_COLORS_GUARD; j++) {
		/* init_pair(short pair, short fore, short back) */
		init_pair(j, colors[(j * 2) - 2], colors[(j * 2) - 1]);

	}

	return;
}





/**
   Check what variable is described by given line

   Check which one of all possible color scheme attributes is described by
   given line. The line should be read from cdw color configuration file.

   This function can recognize empty or invalid line (is not empty and is
   not a comment) and discard it (return some error value).

   Lines containing only white chars are treated as empty lines.
   Lines starting with comment char ('#') are also signalled.

   \param line - line from configuration file to be parsed

   \return CONF_COMMENT if given line has only comment (starts with comment char)
   \return CONF_EMPTY if given line is empty
   \return CONF_INVALID if line is invalid
   \return CONF_COLOR_* - value corresponding to particular color scheme attribute.
*/
int get_attribute_name(char *line)
{
	const char *tline = cdw_string_ltrim(line);

	if (tline[0] == '#') {
		return CONF_COMMENT;
	} else if ((strlen(tline) == 0) || tline[0] == '\n') {
		return CONF_EMPTY;
	} else {
		/* strings passed as second arguments of starts_with_ci()
		   are left unchanged for compatibility with existing
		   cdw color configuration files, although it would be
		   nice to have variable names in more readable form of
		   Conf_color_main_fore */
		if (cdw_string_starts_with_ci(tline, "Color1_fore")) return CONF_COLOR_MAIN_FORE;
		if (cdw_string_starts_with_ci(tline, "Color1_bg")) return CONF_COLOR_MAIN_BG;
		if (cdw_string_starts_with_ci(tline, "Color2_fore")) return CONF_COLOR_DIALOG_FORE;
		if (cdw_string_starts_with_ci(tline, "Color2_bg")) return CONF_COLOR_DIALOG_BG;
		if (cdw_string_starts_with_ci(tline, "Color3_fore")) return CONF_COLOR_TITLE_FORE;
		if (cdw_string_starts_with_ci(tline, "Color3_bg")) return CONF_COLOR_TITLE_BG;
		if (cdw_string_starts_with_ci(tline, "Color4_fore")) return CONF_COLOR_MENU_FORE;
		if (cdw_string_starts_with_ci(tline, "Color4_bg")) return CONF_COLOR_MENU_BG;
		if (cdw_string_starts_with_ci(tline, "Color5_fore")) return CONF_COLOR_PROGRESS_FORE;
		if (cdw_string_starts_with_ci(tline, "Color5_bg")) return CONF_COLOR_PROGRESS_BG;
		if (cdw_string_starts_with_ci(tline, "Color6_fore")) return CONF_COLOR_INPUT_FORE;
		if (cdw_string_starts_with_ci(tline, "Color6_bg")) return CONF_COLOR_INPUT_BG;
		if (cdw_string_starts_with_ci(tline, "Color7_fore")) return CONF_COLOR_WARNING_FORE;
		if (cdw_string_starts_with_ci(tline, "Color7_bg")) return CONF_COLOR_WARNING_BG;
		if (cdw_string_starts_with_ci(tline, "Color8_fore")) return CONF_COLOR_SIZE_INFO_FORE;
		if (cdw_string_starts_with_ci(tline, "Color8_bg")) return CONF_COLOR_SIZE_INFO_BG;
		if (cdw_string_starts_with_ci(tline, "Color9_fore")) return CONF_COLOR_DISABLED_FORE;
		if (cdw_string_starts_with_ci(tline, "Color9_bg")) return CONF_COLOR_DISABLED_BG;
		if (cdw_string_starts_with_ci(tline, "Color10_fore")) return CONF_COLOR_UNUSED_FORE;
		if (cdw_string_starts_with_ci(tline, "Color10_bg")) return CONF_COLOR_UNUSED_BG;
		if (cdw_string_starts_with_ci(tline, "Color11_fore")) return CONF_COLOR_TOOLTIPS_FORE;
		if (cdw_string_starts_with_ci(tline, "Color11_bg")) return CONF_COLOR_TOOLTIPS_BG;
		if (cdw_string_starts_with_ci(tline, "Color12_fore")) return CONF_COLOR_TEXTAREA_FORE;
		if (cdw_string_starts_with_ci(tline, "Color12_bg")) return CONF_COLOR_TEXTAREA_BG;
		if (cdw_string_starts_with_ci(tline, "Color13_fore")) return CONF_COLOR_ERROR_FORE;
		if (cdw_string_starts_with_ci(tline, "Color13_bg")) return CONF_COLOR_ERROR_BG;
	}

	return CONF_INVALID;
}





/**
   Extract color value from line read from color config file

   Extract name of color (e.g. BLACK, BOLD_YELLOW etc.) from given string
   that comes from configuration file.

   Sample argument is like "Color10_bg = BLACK"
   we have to extract right-hand value after '=' char.

   Spurious white chars are stripped by this function.

   \param line - line read from color config file

   \return curses color attribute corresponding to value extracted from line
   \return -1 if line does not have any color attribute value
*/
int get_attribute_value(const char *line)
{
	char *eq = strstr(line, "=");
	if (eq == (char *) NULL) {
		return -1;
	}

	const char *value = cdw_string_ltrim(eq + 1);

	if (cdw_string_starts_with_ci(value, "BLACK")) return COLOR_BLACK;
	if (cdw_string_starts_with_ci(value, "RED")) return COLOR_RED;
	if (cdw_string_starts_with_ci(value, "GREEN")) return COLOR_GREEN;
	if (cdw_string_starts_with_ci(value, "YELLOW")) return COLOR_YELLOW;
	if (cdw_string_starts_with_ci(value, "BLUE")) return COLOR_BLUE;
	if (cdw_string_starts_with_ci(value, "MAGENTA")) return COLOR_MAGENTA;
	if (cdw_string_starts_with_ci(value, "CYAN")) return COLOR_CYAN;
	if (cdw_string_starts_with_ci(value, "WHITE")) return COLOR_WHITE;

	if (cdw_string_starts_with_ci(value, "BOLD_BLACK")) return (COLOR_BLACK | A_BOLD);
	if (cdw_string_starts_with_ci(value, "BOLD_RED")) return (A_BOLD | COLOR_RED);
	if (cdw_string_starts_with_ci(value, "BOLD_GREEN")) return COLOR_GREEN | A_BOLD;
	if (cdw_string_starts_with_ci(value, "BOLD_YELLOW")) return COLOR_YELLOW | A_BOLD;
	if (cdw_string_starts_with_ci(value, "BOLD_BLUE")) return COLOR_BLUE | A_REVERSE;
	if (cdw_string_starts_with_ci(value, "BOLD_MAGENTA")) return COLOR_MAGENTA | A_BOLD;
	if (cdw_string_starts_with_ci(value, "BOLD_CYAN")) return COLOR_CYAN | A_BOLD;
	if (cdw_string_starts_with_ci(value, "BOLD_WHITE")) return COLOR_WHITE | A_BOLD;

	return -1;
}

