/*  audiodevs: Abstraction layer for audio hardware & samples
    Copyright (C) 2003-2004 Nemosoft Unv.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    For questions, remarks, patches, etc. for this program, the author can be
    reached at camstream@smcc.demon.nl.
*/

#include <qlayout.h>
#include <qpushbutton.h>

#include "AudioMixerControlLinux.h"

/**
  \class CAudioControlScrollWidget

  \brief Helper class that allows complex and large mixer widgets to be scrollable

*/
CAudioControlScrollWidget::CAudioControlScrollWidget(QWidget *parent, const char *name, WFlags f)
	: QScrollView(parent, name, f)
{
   m_pCanvas = new QWidget(viewport());
   m_pGrid = new QGridLayout(m_pCanvas, 1, 6); // will expand
   if (m_pGrid == 0)
   {
     qWarning("CAudioControlScrollWidget: Failed to create grid");
     return;
   }

   m_pGrid->setSpacing(5);
   m_pGrid->setRowStretch(0, 10);
   m_pGrid->setRowStretch(1, 30);
   m_pGrid->setRowStretch(2, 30);
   m_pGrid->setRowStretch(3, 15);
   m_pGrid->setRowStretch(4, 10);
   m_pGrid->setRowStretch(5, 10);
   addChild(m_pCanvas);
}

/* stretch canvas to new width */
void CAudioControlScrollWidget::viewportResizeEvent(QResizeEvent *e)
{
   int w, h;

   w = e->size().width();
   h = e->size().height();
   if (h < m_pGrid->minimumSize().height())
     h = m_pGrid->minimumSize().height();
   m_pCanvas->resize(w, h);
}

/** Call this function after all elements are added, to calculate the width */
void CAudioControlScrollWidget::SetSize()
{
   if (m_pCanvas == 0 || m_pGrid == 0)
     return;
   QSize min = m_pGrid->minimumSize();
   viewport()->setMinimumWidth(min.width());
   viewport()->setMinimumHeight(100);
   min = m_pGrid->sizeHint();
   viewport()->resize(min);
}


/**
  \class CAudioMixerElement
  \brief Audio mixer widget

  This class is a helper class that shows a control from the mixer (e.g. master
  volume, PCM, Microphone input, etc). It shows the name (always), slider (optiona),
  mute (optional) &  capture setting (also optional).
*/

/* Much of the code in this class is (educated) guesswork, since the mixer interface
   isn't documented at all.
*/


CAudioMixerElement::CAudioMixerElement(snd_mixer_t *mixer, snd_mixer_elem_t *elem, int row, CAudioControlScrollWidget *parent)
	: QObject(parent)
{
   QLabel *label = 0;
   int i, j;
   long Min, Max;

   m_ChannelMask = 0;
   m_ChannelCount = 0;
   m_SliderType = Unknown;

   m_PlaybackMono = false;
   m_PlaybackSwitch = false;
   m_PlaybackSwitchIsMute = false;
   m_PlaybackVolumeJoined = false;
   m_PlaybackSwitchJoined = false;
   for (i = 0; i <= SND_MIXER_SCHN_LAST; i++)
     m_pPlaybackSlider[i] = 0;
   m_pDropdownBox = 0;

   m_CaptureMono = false;
   m_CaptureSwitch = false;
   m_CaptureVolumeJoined = false;
   m_CaptureSwitchJoined = false;
   m_CaptureExclusive = false;
   m_CaptureGroup = -1;
   m_pCaptureSlider = 0;

   m_pPlaybackCheckBox = 0;

   m_pCaptureRadioButton = 0;
   m_pCaptureCheckBox = 0;

   m_pMixer = mixer;
   //m_pElement = e;

   m_pSID = (snd_mixer_selem_id_t *)malloc(snd_mixer_selem_id_sizeof()); // GVDGVDGVD
   if (m_pSID == 0)
   {
     qWarning("CAudioMixerElement: failed to allocate memory for m_pSID");
   }
   else
   {
     snd_mixer_selem_get_id(elem, m_pSID); // <-- what is this?
   }
   // First column: label
   m_Name = snd_mixer_selem_get_name(elem);
   qDebug("Mixer element '%s'", m_Name.ascii());
   label = new QLabel(m_Name, parent->m_pCanvas);
   parent->m_pGrid->addWidget(label, row, 0);

   // See which type of channel this is: enumeration, or capture, playback or both
   // Second column
   if (snd_mixer_selem_is_enumerated(elem) > 0)
   {
     char enum_name[128];

     m_SliderType = Enumeration;
     j = snd_mixer_selem_get_enum_items(elem);
//     qDebug("  Is enumerated. Items = %d", j);
     m_pDropdownBox = new QComboBox(parent->m_pCanvas);
     parent->m_pGrid->addWidget(m_pDropdownBox, row, 1);
     connect(m_pDropdownBox, SIGNAL(activated(int)), SLOT(ChangedEnumerationSelection(int)));

     for (i = 0; i < j; i++)
     {
        if (snd_mixer_selem_get_enum_item_name(elem, i, 128, enum_name) >= 0)
        {
          enum_name[127] = '\0';
          m_pDropdownBox->insertItem(enum_name);
        }
     }
   }
   else
   {
     /* PLAYBACK */
     m_PlaybackMono         = (snd_mixer_selem_is_playback_mono(elem) > 0);
     m_PlaybackSwitchJoined = (snd_mixer_selem_has_playback_switch_joined(elem) > 0);
     m_PlaybackVolumeJoined = (snd_mixer_selem_has_playback_volume_joined(elem) > 0);
qDebug("Has playback switch joined: %c | volume joined: %c", m_PlaybackSwitchJoined ? 'T' : 'F', m_PlaybackVolumeJoined ? 'T' : 'F');
     // Second column
     if (snd_mixer_selem_has_playback_volume(elem) > 0)
     {
       m_SliderType = Playback;
       QSlider *stemp = 0;
       QBoxLayout *slider_layout = 0;
       QWidget *my_parent = parent->m_pCanvas;

       snd_mixer_selem_get_playback_volume_range(elem, &Min, &Max);
       if (m_PlaybackVolumeJoined)
       {
         // Simple, single slider for all channels
         stemp = m_pPlaybackSlider[0] = new QSlider(Min, Max, (Max - Min + 1) / 8, 0, QSlider::Horizontal, parent->m_pCanvas);
	 if (stemp != 0)
	 {
           parent->m_pGrid->addWidget(stemp, row, 1);
           connect(stemp, SIGNAL(valueChanged(int)), SLOT(MovedPlaybackVolume(int)));
	 }
       }
       else
       {
///qDebug("creating slider_layout");
         my_parent = new QWidget(parent->m_pCanvas); // must create mini-widget
         slider_layout = new QVBoxLayout(my_parent); // which has a vertical layout
         if (my_parent != 0)
         {
           parent->m_pGrid->addWidget(my_parent, row, 1);
         }
       }

       for (i = 0; i <= SND_MIXER_SCHN_LAST; i++)
       {
         if (snd_mixer_selem_has_playback_channel(elem, (snd_mixer_selem_channel_id_t)i) > 0)
         {
	   if (!m_PlaybackVolumeJoined)
	   { // channels are NOT combined
	     stemp = m_pPlaybackSlider[m_ChannelCount] = new QSlider(Min, Max, (Max - Min + 1) / 8, 0, QSlider::Horizontal, my_parent);
             if (stemp != 0)
             {
               slider_layout->addWidget(stemp);
//               layout->addWidget(stemp, row, 1);
               connect(stemp, SIGNAL(valueChanged(int)), SLOT(MovedPlaybackVolume(int)));
             }
	   }
           m_Channels[m_ChannelCount++] = (snd_mixer_selem_channel_id_t)i;
           m_ChannelMask |= (1 << i);
         }
       }
     }

     /* CAPTURE */
     m_CaptureMono         = (snd_mixer_selem_is_capture_mono(elem) > 0);
     m_CaptureSwitchJoined = (snd_mixer_selem_has_capture_switch_joined(elem) > 0);
     m_CaptureVolumeJoined = (snd_mixer_selem_has_capture_volume_joined(elem) > 0);
     m_CaptureExclusive    = (snd_mixer_selem_has_capture_switch_exclusive(elem) > 0);

     // Capture volume slider will be third column */
     if (snd_mixer_selem_has_capture_volume(elem) > 0)
     {
       /* Some channels can be both playback and capture */
       if (m_SliderType == Playback)
       {
         m_SliderType = PlaybackCapture;
       }
       else
       {
         m_SliderType = Capture;
       }

       snd_mixer_selem_get_capture_volume_range(elem, &Min, &Max);
       m_pCaptureSlider = new QSlider(Min, Max, (Max - Min + 1) / 8, 0, QSlider::Horizontal, parent->m_pCanvas);
       parent->m_pGrid->addWidget(m_pCaptureSlider, row, 2);
       connect(m_pCaptureSlider, SIGNAL(valueChanged(int)), SLOT(MovedCaptureVolume(int)));

       /* Personally, I consider this channel method query somewhat cumbersome; a bitmask would have worked quite well too. */
       for (i = 0; i <= SND_MIXER_SCHN_LAST; i++)
       {
         if (snd_mixer_selem_has_capture_channel(elem, (snd_mixer_selem_channel_id_t)i) > 0)
         {
           m_Channels[m_ChannelCount++] = (snd_mixer_selem_channel_id_t)i;
           m_ChannelMask |= (1 << i);
         }
       }
     }
   } // not enumeration


   /* Fourth column: on/off switch or mute  */
   if (snd_mixer_selem_has_playback_switch(elem) > 0)
   {
     /* Note that these switches can be confusing... For Mute, when
        the option is 'on', it means the channel is off (muted). For
        a normal switch (for example 'Microphone boost') 'on' really
        means 'on'. However, this is confusing for the user, so
        we keep track of this is a mute option or not.
      */
     m_PlaybackSwitch = true;
     if (m_SliderType == Unknown)
     { // this seems to coincide with on/off switches, so don't put any text in here
       m_SliderType = Switch;
       m_pPlaybackCheckBox = new QCheckBox(parent->m_pCanvas);
     }
     else
     { // these are usually channels that can be muted
       m_PlaybackSwitchIsMute = true;
       m_pPlaybackCheckBox = new QCheckBox(tr("Mute"), parent->m_pCanvas);
     }
     parent->m_pGrid->addWidget(m_pPlaybackCheckBox, row, 3);
     connect(m_pPlaybackCheckBox, SIGNAL(toggled(bool)), SLOT(ToggledSwitch(bool)));
   }

   /* Fifth column: capture switch or group */
   if (snd_mixer_selem_has_capture_switch(elem) > 0)
   {
     m_CaptureSwitch    = true;
     if (m_CaptureExclusive)
     {
       QButtonGroup *group = 0;
       m_CaptureGroup = snd_mixer_selem_get_capture_group(elem);
       /// TODO: group radio buttons
       if (m_CaptureGroup < parent->m_CaptureGroups.size())
       {
         group = parent->m_CaptureGroups[m_CaptureGroup];
       }
       if (group == 0)
       {
         if (m_CaptureGroup >= parent->m_CaptureGroups.size())
         {
           if (!parent->m_CaptureGroups.resize(m_CaptureGroup + 1))
           {
             qWarning("Failed to resize capture radio button groups.");
           }
         }
qDebug("Creating buttongroup with id %d", m_CaptureGroup);
         group = new QButtonGroup(parent->m_pCanvas);
         if (group == 0)
         {
           qWarning("Failed to create capture radio button group.");
         }
         else
         {
           group->setExclusive(true);
           parent->m_CaptureGroups.insert(m_CaptureGroup, group);
         }
       }
       m_pCaptureRadioButton = new QRadioButton(tr("Capture"), parent->m_pCanvas);
       if (m_pCaptureRadioButton != 0)
       {
         parent->m_pGrid->addWidget(m_pCaptureRadioButton, row, 4);
         connect(m_pCaptureRadioButton, SIGNAL(stateChanged(int)), SLOT(ToggledCapture(int)));
         if (group != 0)
         {
           group->insert(m_pCaptureRadioButton);
         }
       }
     }
     else
     {
       m_pCaptureCheckBox = new QCheckBox(tr("Capture"), parent->m_pCanvas);
       parent->m_pGrid->addWidget(m_pCaptureCheckBox, row, 4);
       connect(m_pCaptureCheckBox, SIGNAL(stateChanged(int)), SLOT(ToggledCapture(int)));
     }
   }

   if (m_SliderType == Unknown)
   {
     qDebug("  Is unknown type of control");
   }

#if 0
        qDebug("Element %d has name %s", t++, snd_mixer_selem_get_name(pMixerElement));
        qDebug("  active: %s  playback_mono: %s capture_mono: %s",
               snd_mixer_selem_is_active(pMixerElement) ? "yes" : "no ",
               snd_mixer_selem_is_playback_mono(pMixerElement) ? "yes" : "no ",
               snd_mixer_selem_is_capture_mono(pMixerElement) ? "yes" : "no "
              );
        qDebug("  has_capture_front_left: %s   has_playback_front_left: %s",
               snd_mixer_selem_has_capture_channel(pMixerElement, SND_MIXER_SCHN_FRONT_LEFT) ? "yes": "no ",
               snd_mixer_selem_has_playback_channel(pMixerElement, SND_MIXER_SCHN_FRONT_LEFT) ? "yes": "no "
              );
        qDebug("  capture group: %d", snd_mixer_selem_get_capture_group(pMixerElement));
     if (m_PlaybackJoined)
       qDebug("  Has joined playback switch.");
     else
       qDebug("  Has playback switch(es).");
        if (snd_mixer_selem_get_playback_volume(pMixerElement, SND_MIXER_SCHN_FRONT_LEFT, &val) < 0)
          val = -1;
        qDebug("  volume front_left: %ld", val);
#endif
}


CAudioMixerElement::~CAudioMixerElement()
{
   free(m_pSID);
   m_pSID = 0;
}


// private slots

void CAudioMixerElement::MovedCaptureVolume(int volume)
{
   snd_mixer_elem_t *element = 0;
   if (m_pSID != 0)
   {
     element = snd_mixer_find_selem(m_pMixer, m_pSID);
     if (element != 0)
     {
       snd_mixer_selem_set_capture_volume_all(element, volume);
       emit UserClickedSomething();
     }
   }
}

void CAudioMixerElement::MovedPlaybackVolume(int volume)
{
   snd_mixer_elem_t *element = 0;
   if (m_pSID != 0)
   {
     element = snd_mixer_find_selem(m_pMixer, m_pSID);
     if (element != 0)
     {
       snd_mixer_selem_set_playback_volume_all(element, volume);
       emit UserClickedSomething();
     }
   }
}

void CAudioMixerElement::ChangedEnumerationSelection(int selection)
{
   emit UserClickedSomething();
}

/**
  \brief Called when either 'Mute' or other checkbox is modified
*/
void CAudioMixerElement::ToggledSwitch(bool mute)
{
   int Value;
   snd_mixer_elem_t *element = 0;

qDebug("ToggledSwitch(%s, %c)", m_Name.ascii(), mute ? 'T' : 'F');
   if (m_pSID != 0)
   {
     element = snd_mixer_find_selem(m_pMixer, m_pSID);
     if (element != 0)
     {
       // Note the reversed logic for Mute switches. This is only done
       // in the GUI, not in the configuration file.
       Value = (mute ^ m_PlaybackSwitchIsMute) ? 1 : 0;
       snd_mixer_selem_set_playback_switch_all(element, Value);
       emit UserClickedSomething();
     }
   }
}


void CAudioMixerElement::ToggledCapture(int state)
{
   int Value;
   snd_mixer_elem_t *element = 0;

qDebug("ToggledCapture(%s, %d)", m_Name.ascii(), state);
   if (state == 1) // "No change" tri-state.
     return;

   if (m_pSID != 0)
   {
     element = snd_mixer_find_selem(m_pMixer, m_pSID);
     if (element != 0)
     {
       Value = state ? 1 : 0;
       snd_mixer_selem_set_capture_switch_all(element, Value);
       emit UserClickedSomething();
     }
   }
}

// public

/**
   \brief Get current settings of element
   \param dom_doc An XML document object

   This function \b appends the current settings of this mixer element to
   the given XML node if it doesn't exist already.

   The XML-format looks something like this:
   <element name="Mic">
     <volume>23</volume>
     <mute>false</mute>
     <capture>true</capture>
   </element>
   The exact format differs per type of input.

   Note: this function returns the settings from the GUI element. For reasons
   I haven't been able to figure out yet, once the ALSA elements are initialized,
   I can't query the current values if they are changed outside the program
   (for example, by alsamixer). So we must hope that any changes we make
   in the GUI is properly set in the driver.

*/

void CAudioMixerElement::GetConfiguration(QDomNode &dom_node) const
{
   QDomNode walk;
   QDomElement elem, v;
   int i;

   walk = dom_node.firstChild();
   while (!walk.isNull())
   {
     if (walk.isElement())
     {
       elem = walk.toElement();
//qDebug("AME::GC name = %s",  elem.attribute("name").ascii());
       if (elem.nodeName() == "element" && elem.attribute("name") == m_Name)
       {
         // Remove node, we create a new one after this loop
         dom_node.removeChild(walk);
         break;
       }
     }
     walk = walk.nextSibling();
   }

   elem = dom_node.ownerDocument().createElement("element");
   elem.setAttribute("name", m_Name);
   dom_node.appendChild(elem);

   // Type
   switch (m_SliderType)
   {
     case Playback:
     case Capture:
     case PlaybackCapture:
       // Store volume & mute flag; capture is only for input channels
       if (m_PlaybackVolumeJoined) // only one slider
       {
         if (m_pPlaybackSlider[0] != 0)
         {
           v = dom_node.ownerDocument().createElement("playback_vol");
           v.appendChild(dom_node.ownerDocument().createTextNode(QString::number(m_pPlaybackSlider[0]->value())));
           elem.appendChild(v);
         }
       }
       else // not joined
       {
         for (i = 0; i < m_ChannelCount; i++)
         {
           if (m_pPlaybackSlider[i] != 0)
           {
             QDomElement v = dom_node.ownerDocument().createElement("playback_vol");
             v.setAttribute("channel", m_Channels[i]);
             v.appendChild(dom_node.ownerDocument().createTextNode(QString::number(m_pPlaybackSlider[i]->value())));
             elem.appendChild(v);
           }
         }
       }
       if (m_pCaptureSlider != 0)
       {
         QDomElement v = dom_node.ownerDocument().createElement("capture_vol");
         v.appendChild(dom_node.ownerDocument().createTextNode(QString::number(m_pCaptureSlider->value())));
         elem.appendChild(v);
       }
       if (m_pPlaybackCheckBox != 0)
       {
         QDomElement v = dom_node.ownerDocument().createElement("mute");
         v.appendChild(dom_node.ownerDocument().createTextNode(m_pPlaybackCheckBox->isChecked() ^ m_PlaybackSwitchIsMute ? "true" : "false"));
         elem.appendChild(v);
       }
       if (m_pCaptureRadioButton != 0 || m_pCaptureCheckBox != 0)
       {
         QDomElement v = dom_node.ownerDocument().createElement("capture");
         if (m_pCaptureRadioButton != 0)
           v.appendChild(dom_node.ownerDocument().createTextNode(m_pCaptureRadioButton->isChecked() ? "true" : "false"));
         if (m_pCaptureCheckBox != 0)
           v.appendChild(dom_node.ownerDocument().createTextNode(m_pCaptureCheckBox->isChecked() ? "true" : "false"));
         elem.appendChild(v);
       }
       break;

     case Switch:
       // On or off. Simple :)
       if (m_pPlaybackCheckBox != 0)
       {
         QDomElement v = dom_node.ownerDocument().createElement("switch");
         v.appendChild(dom_node.ownerDocument().createTextNode(m_pPlaybackCheckBox->isChecked() ? "true" : "false"));
         elem.appendChild(v);
       }
       break;

     case Enumeration:
       // Use text from enumeration
       if (m_pDropdownBox != 0)
       {
         QDomElement v = dom_node.ownerDocument().createElement("enum");
         v.appendChild(dom_node.ownerDocument().createTextNode(m_pDropdownBox->currentText()));
         elem.appendChild(v);
       }
       break;

     default:
       qWarning("Unknown slidertype (%d) for GetConfiguration! (%s)", m_SliderType, m_Name.ascii());
       break;
   }
}

/**
  \brief Set mixer elements according to data in XML node
*/
void CAudioMixerElement::SetConfiguration(const QDomNode &dom_node) const
{
   QDomElement v;
   int i, n;

   switch (m_SliderType)
   {
     case Playback:
     case Capture:
     case PlaybackCapture:
       // Retrieve capture volume, mute flag & capture flag when the GUI element is present and in the XML node
       if (m_PlaybackVolumeJoined)
       {
         if (m_pPlaybackSlider[0] != 0)
         {
           v = dom_node.namedItem("playback_vol").toElement();
           if (v.isElement())
           {
             int volume = v.text().toInt();
             m_pPlaybackSlider[0]->setValue(volume);
           }
         }
       }
       else // not joined; all channels set separately
       {
         QDomNodeList NList = dom_node.childNodes();
         for (n = 0; n < NList.count(); n++)
         {
            v = NList.item(n).toElement();
            // Okay, we now have an element; see if it's playback_vol, then get
            // the 'channel' attribute, and match that with one of our channels
            // It's convoluted, I know
///qDebug("SC: dom_node = %s, nodename = %s", dom_node.nodeName().ascii(), v.nodeName().ascii());
            if (v.isElement() && v.nodeName() == "playback_vol")
            {
              QString channel_attr = v.attribute("channel");
              int channel_num = 0;
              if (channel_attr.isNull())
                continue; // no attribute, next
              channel_num = channel_attr.toInt();
              for (i = 0; i < m_ChannelCount; i++)
              {
                if (m_Channels[i] == channel_num && m_pPlaybackSlider[i] != 0)
                {
///qDebug("Found slider, i = %d, channel_num = %d", i, channel_num);
                  m_pPlaybackSlider[i]->setValue(v.text().toInt());
                }
              }
            }
         }
       }
       if (m_pCaptureSlider != 0)
       {
         v = dom_node.namedItem("capture_vol").toElement();
         if (v.isElement())
         {
           int volume = v.text().toInt();
           m_pCaptureSlider->setValue(volume);
         }
       }
       if (m_pPlaybackCheckBox != 0)
       {
         v = dom_node.namedItem("mute").toElement();
         if (v.isElement())
         {
           bool mute_on = (v.text() == "true");
           m_pPlaybackCheckBox->setChecked(mute_on ^ m_PlaybackSwitchIsMute);
         }
       }
       if (m_SliderType == Capture)
       {
         v = dom_node.namedItem("capture").toElement();
         if (v.isElement())
         {
           bool capture_on = (v.text() == "true");
           if (m_pCaptureRadioButton != 0)
             m_pCaptureRadioButton->setChecked(capture_on);
           if (m_pCaptureCheckBox != 0)
             m_pCaptureCheckBox->setChecked(capture_on);
         }
       } // ..Capture
       break;

     case Switch:
       if (m_pPlaybackCheckBox != 0)
       {
         v = dom_node.namedItem("switch").toElement();
         if (v.isElement())
         {
           bool mute_on = (v.text() == "true");
           m_pPlaybackCheckBox->setChecked(mute_on);
         }
       }
       break;

     case Enumeration:
       if (m_pDropdownBox != 0)
       {
         dom_node.namedItem("enum").toElement();
         if (v.isElement())
         {
           QString SelText = v.text();
           int i;
           // walk options in combobox, find corresponding text
           for (i = 0; i < m_pDropdownBox->count(); i++)
           {
              if (m_pDropdownBox->text(i) == SelText)
              {
                m_pDropdownBox->setCurrentItem(i);
                break;
              }
           }
         }
       }
       break;

     default:
       qWarning("Unknown slidertype (%d) for SetConfiguration (%s)!", m_SliderType, m_Name.ascii());
       break;
   } // ..switch
}


int CAudioMixerElement::GetCaptureGroup() const
{
  return m_CaptureGroup;
}



// public slots

/**
  \brief Read settings from device, update widget(s)

  This function will update the widget, with the value queried from the hardware.
*/
void CAudioMixerElement::UpdateFromDevice()
{
   long Value;
   int Part, Sum, i;
   unsigned int u;
   snd_mixer_selem_channel_id_t Channel;
   snd_mixer_elem_t *element = 0;

   if (m_pSID == 0)
     return;
   element = snd_mixer_find_selem(m_pMixer, m_pSID);
   if (element == 0)
     return;
   Sum = 0;
   switch (m_SliderType)
   {
     case PlaybackCapture: // fallthrough
     case Playback:
       if (m_PlaybackVolumeJoined)
       {
         if (m_pPlaybackSlider[0] != 0)
         {
           if (snd_mixer_selem_get_playback_volume(element, m_Channels[0], &Value) == 0)
           {
             m_pPlaybackSlider[0]->setValue(Value);
           }
         }
       }
       else
       {
         for (i = 0; i < m_ChannelCount; i++)
         {
            if (snd_mixer_selem_get_playback_volume(element, m_Channels[i], &Value) == 0 &&
                m_pPlaybackSlider[i] != 0)
            {
              m_pPlaybackSlider[i]->setValue(Value);
            }
         }
       }
       if (m_SliderType == Playback)
         break;
     case Capture: // fallthrough
       for (i = 0; i < m_ChannelCount; i++)
       {
          Value = 0;
          if (snd_mixer_selem_get_capture_volume(element, m_Channels[i], &Value) == 0)
            Sum += Value;
       }
       if (m_ChannelCount > 0)
         Sum /= m_ChannelCount;
       m_pCaptureSlider->setValue(Sum);
       break;

     case Enumeration:
       // Weird; enumeration has a channel parameter, but no way to query the channels.
       // Therefore, assuming channel 0 (mono/left)
       Channel = SND_MIXER_SCHN_MONO;
       u = 0;
       if (snd_mixer_selem_get_enum_item(element, Channel, &u) == 0)
         m_pDropdownBox->setCurrentItem(u);
       break;

     default:
       break;
   }

   // Set playback/mute switch in GUI.
   if (m_PlaybackSwitch)
   {
     Sum = 0;
     for (i = 0; i < m_ChannelCount; i++)
     {
       if (snd_mixer_selem_get_playback_switch(element, m_Channels[i], &Part) == 0)
       {
         Sum += Part;
       }
     }
     // Reverse logic for mute (all channels off means mute is on)
     m_pPlaybackCheckBox->setChecked((Sum == 0) ^ m_PlaybackSwitchIsMute);
   }

   // Set recording switch GUI (checkbox/radio)
   if (m_CaptureSwitch)
   {
     Sum = 0;
     for (i = 0; i < m_ChannelCount; i++)
     {
       if (snd_mixer_selem_get_capture_switch(element, m_Channels[i], &Part) == 0)
       {
         Sum += Part;
       }
     }
     if (m_pCaptureRadioButton != 0)
       m_pCaptureRadioButton->setChecked(Sum != 0);
     if (m_pCaptureCheckBox != 0)
       m_pCaptureCheckBox->setChecked(Sum != 0);
   }
}


/**************************************************************************************************/

CAudioControlLinux::CAudioControlLinux(const char *device)
{
   snd_ctl_t *ctl_handle;
   snd_ctl_card_info_t *hw_info;
   int i, ret;

   qDebug(">> CAudioControlLinux::CAudioControlLinux(%s)", device);
   m_pMixer = 0;
   m_pMixerWidget = 0;
   m_Count = 0;
   m_pPollFDs = 0;

   m_DeviceName = device;
   ret = snd_ctl_open(&ctl_handle, device, SND_CTL_NONBLOCK);
   if (ret < 0)
   {
     qWarning("snd_ctl_open() failed '%s'", snd_strerror(ret));
     return;
   }
   snd_ctl_card_info_alloca(&hw_info);
   ret = snd_ctl_card_info(ctl_handle, hw_info);
   snd_ctl_close(ctl_handle);
//   free(hw_info); // don't free it, or it will go boom. Don't ask me why.

   /* open mixer device */
   ret = snd_mixer_open(&m_pMixer, 0);
   if (ret < 0)
     qWarning("snd_mixer_open() failed: %d", ret);
   else
   {
     ret = snd_mixer_attach(m_pMixer, device);
     if (ret < 0)
       qWarning("snd_mixer_attach() failed: %d", ret);
     ret = snd_mixer_selem_register(m_pMixer, NULL, NULL);
     if (ret < 0)
       qWarning("snd_mixer_selem_register() failed: %d", ret);
     ret = snd_mixer_load(m_pMixer);
     if (ret < 0)
       qWarning("snd_mixer_load() failed: %d", ret);

     m_Count = snd_mixer_get_count(m_pMixer);
   }

   // complete widget
   m_pMixerWidget = new QWidget;
   if (m_pMixerWidget == 0)
   {
     qWarning("CAudioControlLinux: Failed to create widget");
     return;
   }
   // vertical layout; top is (large) mixer widget, below are buttons
   QVBoxLayout *pVertLayout = new QVBoxLayout(m_pMixerWidget);
   if (pVertLayout == 0)
   {
     qWarning("CAudioControlLinux: Failed to create vert layout");
     return;
   }
   // top half scrolling part
   m_pScrollWidget = new CAudioControlScrollWidget(m_pMixerWidget);
   if (m_pScrollWidget == 0)
   {
     qWarning("CAudioControlLinux: Failed to create scroll widget");
     return;
   }
   // bottom part, button(s)
   QPushButton *pCloseButton = new QPushButton(tr("Close"), m_pMixerWidget);
   if (pCloseButton == 0)
   {
     qWarning("CAudioControlLinux: Failed to create pushbutton");
     return;
   }
   connect(pCloseButton, SIGNAL(clicked()), m_pMixerWidget, SLOT(close()));

   pVertLayout->addWidget(m_pScrollWidget);
   pVertLayout->addWidget(pCloseButton);

   if (m_Count > 0)
   {
     snd_mixer_elem_t *pMixerElement = 0;
     i = 0;

     pMixerElement = snd_mixer_first_elem(m_pMixer);
     while (pMixerElement != 0)
     {
        if (snd_mixer_selem_is_active(pMixerElement))
        {
          CAudioMixerElement *me;

          me = new CAudioMixerElement(m_pMixer, pMixerElement, i, m_pScrollWidget);
          if (me != 0)
          {
            connect(me, SIGNAL(UserClickedSomething()), SIGNAL(UpdateControls()));
            connect(this, SIGNAL(ForceUpdate()), me,  SLOT(UpdateFromDevice()));
            m_Elements.insert(me->GetName(), me);
          }
          i++;
        }
        else
        {
          qDebug("Element not active '%s'", snd_mixer_selem_get_name(pMixerElement));
        }
        pMixerElement = snd_mixer_elem_next(pMixerElement);
     }
   }

   /* Set up selector(s) for changes to mixer */
   m_PollFDCount = snd_mixer_poll_descriptors_count(m_pMixer);
   if (m_PollFDCount < 0)
   {
     qWarning("Could not determine poll descriptors for mixer!?!");
   }
   else
   {
     // I don't think there will ever be more than one descriptor, but better safe than sorry
     qDebug("Mixer has %d poll descriptors.", m_PollFDCount);
     m_Sockets.setAutoDelete(true);
     m_Sockets.resize(m_PollFDCount);
     m_pPollFDs = (struct pollfd *)malloc(sizeof(struct pollfd) * m_PollFDCount);
     if (m_pPollFDs == 0)
     {
       qWarning("Failed to allocate memory for poll descriptors.");
     }
     else
     {
       // Get poll file descriptors
       if (snd_mixer_poll_descriptors(m_pMixer, m_pPollFDs, m_PollFDCount) < 0)
       {
         qWarning("Failed to query poll descriptors from mixer.");
       }
       else
       {
         // Use only the 'fd' field to setup a QSocket
         for (i = 0; i < m_PollFDCount; i++)
         {
           QSocket *s;
qDebug("fd = %d, events = %x", m_pPollFDs[i].fd, m_pPollFDs[i].events);

           s = new QSocket(this, QString("mixer socket %1=%2)").arg(i).arg(m_pPollFDs[i].fd));
           if (s == 0)
           {
             qWarning("Failed to allocate QSocket for mixer.");
           }
           else
           {
             s->setSocket(m_pPollFDs[i].fd);
             m_Sockets.insert(i, s);
             // Link updates to the mixer socket to an update to our controls
             //connect(s, SIGNAL(readyRead()), SIGNAL(UpdateControls()));
             connect(s, SIGNAL(readyRead()), SLOT(SocketRead()));
           }
         }
       }
     }
   }
   m_pScrollWidget->SetSize();

   emit ForceUpdate();
   qDebug("<< CAudioControlLinux::CAudioControlLinux()");
}


CAudioControlLinux::~CAudioControlLinux()
{
   qDebug(">> CAudioControlLinux::~CAudioControlLinux()");
   if (m_pMixer != 0)
   {
     if (snd_mixer_detach(m_pMixer, m_DeviceName.ascii()) < 0)
       qWarning("snd_mixer_detach() failed?");
     if (snd_mixer_close(m_pMixer) < 0)
       qWarning("snd_mixer_close() failed?");
   }
   m_pMixer = 0;
   delete m_pMixerWidget;
   m_pMixerWidget = 0;
   free(m_pPollFDs);
   m_pPollFDs = 0;
   qDebug("<< CAudioControlLinux::~CAudioControlLinux()");
}

// private

// private slots

void CAudioControlLinux::SocketRead()
{
  unsigned short REvents = 0;

//  qDebug(">> CAudioControlLinux::SocketRead()");
  snd_mixer_poll_descriptors_revents(m_pMixer, m_pPollFDs, m_PollFDCount, &REvents);
//qDebug("REvents = 0x%02x", REvents);
  if (REvents & POLLIN)
  {
    snd_mixer_handle_events(m_pMixer);
    emit ForceUpdate();
  }
//  qDebug("<< CAudioControlLinux::SocketRead()");
}


// protected

// public

/**
  \brief Return current audio settings as XML node
  \param dom_node An existing XML node

  This function will return the current audio configuration (input levels,
  recording input, mute, etc.)

  You must create the node beforehand.

*/
void CAudioControlLinux::GetConfiguration(QDomNode &dom_node) const
{
   QDictIterator<CAudioMixerElement> it(m_Elements);

   if (dom_node.nodeName() != "mixer_settings")
   {
     qWarning("CAudioControlLinux::GetConfiguration: wrong node name (%s).", dom_node.nodeName().ascii());
     return;
   }

   for (; it.current(); ++it)
   {
     it.current()->GetConfiguration(dom_node);
   }
}

void CAudioControlLinux::SetConfiguration(const QDomNode &dom_node)
{
   QDomNode Child;

   qDebug(">> CAudioControlLinux::SetConfiguration");
   if (dom_node.nodeName() != "mixer_settings")
   {
     qWarning("CAudioControlLinux::SetConfiguration: wrong node name (%s).", dom_node.nodeName().ascii());
     return;
   }
   if (!dom_node.isElement())
   {
     qWarning("CAudioControlLinux::SetConfiguration: not a QDomElement.");
     return;
   }

   /* Traverse children of this XML node, fetch GUI element if present */
   Child = dom_node.firstChild();
   while (!Child.isNull())
   {
      QDomElement Elem;
      QString Name;
      CAudioMixerElement *Ame;

      if (Child.isElement() && Child.nodeName() == "element")
      {
        Elem = Child.toElement();
        Name = Elem.attribute("name");
        if (Name.isNull())
        {
          qWarning("CAudioControlLinux::SetConfiguration: node without name?");
        }
        else
        {
          Ame = m_Elements[Name];
          if (Ame != 0)
          {
            Ame->SetConfiguration(Elem);
          }
        }
      }
      Child = Child.nextSibling();
   }
///   emit ForceUpdate();
   qDebug("<< CAudioControlLinux::SetConfiguration");
}



void CAudioControlLinux::ShowControls()
{
   if (m_pMixerWidget != 0)
     m_pMixerWidget->show();
}


