package org.bouncycastle.crypto.test;

import org.bouncycastle.crypto.BlockCipher;
import org.bouncycastle.crypto.BufferedBlockCipher;
import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.CryptoException;
import org.bouncycastle.util.encoders.Hex;
import org.bouncycastle.util.test.SimpleTestResult;
import org.bouncycastle.util.test.Test;
import org.bouncycastle.util.test.TestResult;

/**
 * a basic test that takes a cipher, key parameter, and an input
 * and output string. This test wraps the engine in a buffered block
 * cipher with padding disabled.
 */
public class BlockCipherMonteCarloTest
    implements Test
{
    int                 id;
    int                 iterations;
    BlockCipher         engine;
    CipherParameters    param;
    byte[]              input;
    byte[]              output;

    public BlockCipherMonteCarloTest(
        int                 id,
        int                 iterations,
        BlockCipher         engine,
        CipherParameters    param,
        String              input,
        String              output)
    {
        this.id = id;
        this.iterations = iterations;
        this.engine = engine;
        this.param = param;
        this.input = Hex.decode(input);
        this.output = Hex.decode(output);
    }

    public String getName()
    {
        return engine.getAlgorithmName() + " Monte Carlo Test " + id;
    }

    public TestResult perform()
    {
        BufferedBlockCipher cipher = new BufferedBlockCipher(engine);

        cipher.init(true, param);

        byte[]  out = new byte[input.length];

        System.arraycopy(input, 0, out, 0, out.length);

        for (int i = 0; i != iterations; i++)
        {
            int len1 = cipher.processBytes(out, 0, out.length, out, 0);

            try
            {
                cipher.doFinal(out, len1);
            }
            catch (CryptoException e)
            {
                return new SimpleTestResult(false, 
                       getName() + ": failed - exception " + e.toString());
            }
        }

        if (!isEqualArray(out, output))
        {
            return new SimpleTestResult(false,
                    getName() + ": failed - " + "expected " + new String(Hex.encode(output), 0) + " got " + new String(Hex.encode(out), 0));
        }

        cipher.init(false, param);

        for (int i = 0; i != iterations; i++)
        {
            int len1 = cipher.processBytes(out, 0, out.length, out, 0);

            try
            {
                cipher.doFinal(out, len1);
            }
            catch (CryptoException e)
            {
                return new SimpleTestResult(false, 
                       getName() + ": failed reversal - exception " + e.toString());
            }
        }

        if (!isEqualArray(input, out))
        {
            return new SimpleTestResult(false, getName() + ": failed reversal");
        }

        return new SimpleTestResult(true, getName() + ": OKAY");
    }

    private boolean isEqualArray(
        byte[]  a,
        byte[]  b)
    {
        if (a.length != b.length)
        {
            return false;
        }

        for (int i = 0; i != a.length; i++)
        {
            if (a[i] != b[i])
            {
                return false;
            }
        }

        return true;
    }
}
