/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2005-2008  Marcel Holtmann <marcel@holtmann.org>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>

#include "baseband.h"
#include "lmp.h"

gchar *check_packet_size(gchar *text, guint dlen, guint size)
{
	gchar *temp = text;

	if (dlen == size)
		return text;

	text = g_strdup_printf("%s\n<span foreground=\"red\">"
				"Packet size is %d bytes</span>", temp, size);

	g_free(temp);

	return text;
}

gchar *create_hex(gchar *string, gpointer data, guint size)
{
	GString *text;
	guint i;

	if (data == NULL || size == 0)
		return string;

	text = g_string_new_len(NULL, size * 4);

	if (string) {
		g_string_append_printf(text, "%s\n", string);
		g_free(string);
	}

	g_string_append(text, "<span size=\"2048\">\n</span><small><tt>");
	for (i = 0; i < size; i++) {
		g_string_append_printf(text, "%02x ", ((guint8 *) data)[i]);
		if ((i + 1) % 16 == 0 && i != size - 1)
			g_string_append_c(text, '\n');
		else if ((i + 1) % 8 == 0)
			g_string_append_c(text, ' ');
	}
	g_string_append(text, "</tt></small>");

	return g_string_free(text, FALSE);
}

static gchar *status2str(guint8 status)
{
	switch (status) {
	case 0:
		return NULL;
	case 4:
		return "Header Error";
	case 5:
		return "Length Error";
	case 6:
		return "CRC Error";
	default:
		return "Unknown Error";
	}
}

static gchar *type2str(guint8 type)
{
	switch (type) {
	case 0x00:
		return "NULL";
	case 0x01:
		return "POLL";
	case 0x02:
		return "FHS";
	case 0x03:
		return "DM1";
	case 0x04:
		return "DH1";
	case 0x05:
		return "HV1";
	case 0x06:
		return "HV2";
	case 0x07:
		return "HV3";
	case 0x08:
		return "DV";
	case 0x09:
		return "AUX1";
	case 0x0a:
		return "DM3";
	case 0x0b:
		return "DH3";
	case 0x0c:
		return "EV4";
	case 0x0d:
		return "EV5";
	case 0x0e:
		return "DM5";
	case 0x0f:
		return "DH5";
	default:
		return "Unknown";
	}
}

static gchar *llid2str(guint8 llid)
{
	switch (llid) {
	case 0x00:
		return "Undefined";
	case 0x01:
		return "L2CAP Cont.";
	case 0x02:
		return "L2CAP Start";
	case 0x03:
		return "LMP Message";
	default:
		return "Unknown";
	}
}

gchar *decode_type2(gpointer data, guint size)
{
	gpointer ptr = data;
	guint8 offset, status, addr, type, llid, l2flow, channel;
	guint8 flow, arqn, seqn;
	guint8 len1, len2;
	guint16 plen;
	gchar *text = NULL;

	while (ptr - data < size) {
		gchar *linebreak, *error, *temp = text;

		offset = ((guint8 *) ptr)[0];
		status = (((guint8 *) ptr)[4] & 0xf0) >> 4;
		addr = ((guint8 *) ptr)[5] & 0x07;
		type = (((guint8 *) ptr)[5] & 0x78) >> 3;
		flow = (((guint8 *) ptr)[5] & 0x80) >> 7;
		arqn = ((guint8 *) ptr)[6] & 0x01;
		seqn = (((guint8 *) ptr)[6] & 0x02) >> 1;
		llid = (((guint8 *) ptr)[6] & 0x0c) >> 2;
		l2flow = (((guint8 *) ptr)[6] & 0x01) >> 4;
		len1 = ((guint8 *) ptr)[6] & 0xe0;
		len2 = ((guint8 *) ptr)[7] & 0x7f;
		plen = (len1 >> 5) | (len2 << 3);
		channel = ((guint8 *) ptr)[12];

		linebreak = (text == NULL) ? "" :
					"\n<span size=\"2048\">\n</span>";

		text = g_strdup_printf("%s%s%s (0x%x) LT_ADDR %d"
							"%s%s%s channel %d",
					temp ? temp : "", linebreak,
					type2str(type), type, addr,
					flow ? " FLOW" : "",
					arqn ? " ARQN" : "",
					seqn ? " SEQN" : "", channel);

		g_free(temp);

		if (llid != 0 || plen > 0) {
			temp = text;
			text = g_strdup_printf("%s\n%s (0x%x) flow %d plen %d",
						temp, llid2str(llid), llid,
								l2flow, plen);
			g_free(temp);
		}

		error = status2str(status);
		if (error != NULL) {
			temp = text;
			text = g_strdup_printf("%s\n"
					"<span foreground=\"red\">%s</span>",
						temp ? temp : "", error);
			g_free(temp);
		}

		if (llid == 3) {
			gchar *lmp = decode_lmp(ptr + offset, plen);
			if (lmp != NULL) {
				temp = text;
				text = g_strdup_printf("%s\n"
					"<span size=\"2048\">\n</span>%s",
								temp, lmp);
				g_free(temp);
				g_free(lmp);
			}
		} else
			text = create_hex(text, ptr + offset, plen);

		ptr += offset + plen;
	}

	return text;
}

gchar *decode_baseband(guint type, gpointer data, guint size)
{
	gchar *text;

	switch (type) {
	case 0x02:
		text = decode_type2(data, size);
		break;
	case 0xff:
		text = g_strndup(data, size);
		break;
	default:
		text = create_hex(NULL, data, size);
		break;
	}

	return text;
}
