# -*- coding: utf-8 -*-

# Bluemindo: A really simple but powerful audio player in Python/PyGTK.
# Copyright (C) 2007-2009  Erwan Briand

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation version 3 of the License.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from gettext import gettext as _
from threading import Thread
from os.path import join, isfile
from gobject import TYPE_STRING as gString, markup_escape_text, timeout_add
from gtk.gdk import (Pixbuf as gPixbuf, pixbuf_new_from_file, INTERP_BILINEAR,
                     threads_enter, threads_leave)
from gtk import (CellRendererPixbuf, CellRendererText, ListStore,
                 TreeViewColumn, TREE_VIEW_COLUMN_FIXED)

from modules.explorer.playlist import Playlist
from modules.explorer.musicdb import MusicDatabase
from modules.explorer import viewscommon
unescape = viewscommon.unescape
insensitive_sort = viewscommon.insensitive_sort

from common.functions import Functions

class NormalView(object):
    ref = None
    ref2 = None

    def __new__(cls, *args, **kws):
        # Singleton
        if cls.ref is None:
            cls.ref = super(NormalView, cls).__new__(cls, args, kws)
        return cls.ref

    def __init__(self, glade_file, config, userconf):
        self.playlist = Playlist(glade_file, config, userconf)
        self.glade_file = glade_file
        self.config = config
        self.userconf = userconf

        self.player_data = join(self.userconf.datadir, 'modules', 'player')
        self.functions = Functions()

        # Show widget
        self.glade_file.get_widget('hbox6').show()
        self.glade_file.get_widget('scrolledwindow4').show()
        self.glade_file.get_widget('scrolledwindow5').show()
        self.glade_file.get_widget('frame2').hide()

        if NormalView.ref2 is None: 
            NormalView.ref2 = 42 
            self.launch()

    def launch(self):
        render_pixbuf_ = CellRendererPixbuf()
        render_text_ = CellRendererText()

        # Album
        treestore_ = ListStore(gPixbuf, gString, gString)
        albumview = self.glade_file.get_widget('treeview4')
        albumview.set_model(treestore_)
        albumview.set_rules_hint(True)
        albumview.connect('row-activated', self.load_album)

        column1 = TreeViewColumn()
        column1.pack_start(render_pixbuf_, expand=False)
        column1.add_attribute(render_pixbuf_, 'pixbuf', 0)
        column1.pack_start(render_text_, expand=True)
        column1.add_attribute(render_text_, 'markup', 1)
        column1.set_min_width(250)
        column1.set_expand(True)
        column1.set_sizing(TREE_VIEW_COLUMN_FIXED)
        albumview.append_column(column1)

        column2 = TreeViewColumn()
        column2.pack_start(render_text_, expand=False)
        column2.add_attribute(render_text_, 'markup', 2)
        albumview.append_column(column2)

        # Artist
        treestore = ListStore(gString, gString)
        artistview = self.glade_file.get_widget('treeview3')
        artistview.set_model(treestore)
        artistview.set_rules_hint(True)
        artistview.connect('row-activated', self.load_artist)
        artselect = artistview.get_selection()
        artselect.connect('changed', self.browse_artist, treestore_, column1)

        column0_ = TreeViewColumn()
        column0_.pack_start(render_text_, expand=True)
        column0_.add_attribute(render_text_, 'markup', 0)
        column0_.set_min_width(280)
        column0_.set_expand(True)
        column0_.set_sizing(TREE_VIEW_COLUMN_FIXED)
        artistview.append_column(column0_)

        column1_ = TreeViewColumn()
        column1_.pack_start(render_text_, expand=False)
        column1_.add_attribute(render_text_, 'markup', 1)
        artistview.append_column(column1_)

        # Add artists
        self.vinfos = [treestore, treestore_, column1, column0_]
        self.reload_database()

    def reload_database(self, force_scan=False):
        pbar = self.glade_file.get_widget('progressbar1')
        pbar.set_text(_('Opening folder…'))
        pbar.show()

        thread = Thread(group=None, target=self.insert_artists,
                        name='artists', args=(pbar, self.vinfos[0],
                                              self.vinfos[1], self.vinfos[2],
                                              self.vinfos[3], force_scan))
        self.thread_name = thread

        thread.start()

    def timeout_progress(self, pbar):
        if self.thread_name.isAlive():
            pbar.pulse()
            return True
        else:
            pbar.hide()

    def insert_artists(self, pbar, treestore, treestore_, column1, column0,
                       force_scan):
        """This function is called at the start-up and add artists."""
        threads_enter()

        # Call database
        musicdb = MusicDatabase(self.config, pbar)

        if self.config['scan'] or force_scan:
            musicdb.scan()

        pbar.set_text(_('Loading music…'))
        self.database = musicdb.load()

        timeout_add(100, self.timeout_progress, pbar)
        self.songs_tree = viewscommon.get_representation(self.database,
                                                        self.config['artwork'])
        self.continue_to_pulse = False

        pbar.set_text(_('Loading artists…'))
        items = self.songs_tree.items()
        items.sort(insensitive_sort)
        items_sort = [key for key, value in items]

        treestore_.clear()
        treestore.clear()

        ni = -1
        for artist in items_sort:
            nb = 0
            for alb in self.songs_tree[artist]:
                nb = nb + len(self.songs_tree[artist][alb])

            treestore.append(('%s' % markup_escape_text(unicode(artist)),
                              '<b>%s</b>' % nb))
            ni = ni + nb

        lensongs = len(self.songs_tree)
        column0.set_title(_('%(art)d artists (%(sng)d songs)') %
                           {'art': lensongs, 'sng': ni})
        self.browse_artist(None, treestore_, column1)

        threads_leave()

    def browse_artist(self, selection, treestore_, column):
        """This function show albums by artist."""
        if selection:
            (mod, iter_) = selection.get_selected()
            if iter_:
               artist = unescape(mod.get_value(iter_, 0))

            try:
                treestore_.clear()
                items = self.songs_tree[artist].items()
                items.sort(insensitive_sort)
                items_sort = [key for key, value in items]
            except UnboundLocalError:
                return

            nb = 0
            for album_ in items_sort:
                nb = nb + len(self.songs_tree[artist][album_])

                _file = join(self.player_data, 'covers',
                             self.functions.get_hash(album_, artist))

                if isfile(_file):
                    icon = pixbuf_new_from_file(_file)
                else:
                    icon = pixbuf_new_from_file(join(self.functions.datadir,
                                                'image', 'logo_head_big.png'))

                icon = icon.scale_simple(32, 32, INTERP_BILINEAR)
                treestore_.append((icon,
                          '%s' % markup_escape_text(unicode(album_)),
                         ('<b>%s</b>' %
                          len(self.songs_tree[artist][album_]))
                ))
    
            column.set_title(_('%(alb)d albums (%(sng)d songs)') %
                             {'alb': len(self.songs_tree[artist]),
                              'sng': nb})

        else:
            albums = []

            i = 0
            n = -1
            for art in self.songs_tree:
                for alb in self.songs_tree[art]:
                    albums.append({'album': alb, 'artist': art})
                    i = i + 1
                    n = n + len(self.songs_tree[art][alb])

            albums.sort()

            column.set_title(_('%(alb)d albums (%(sng)d songs)') %
                              {'alb': i, 'sng': n})

            for alb in albums:
                album_ = alb['album']
                artist = alb['artist']

                _file = join(self.player_data, 'covers',
                             self.functions.get_hash(album_, artist))

                if isfile(_file):
                    icon = pixbuf_new_from_file(_file)
                else:
                    icon = pixbuf_new_from_file(join(self.functions.datadir,
                                                'image', 'logo_head_big.png'))

                icon = icon.scale_simple(32, 32, INTERP_BILINEAR)
                treestore_.append((icon,
                          '%s' % markup_escape_text(unicode(album_)),
                         ('<b>%s</b>' % len(self.songs_tree[artist][album_])
                )))

    def load_artist(self, tview, path, column):
        """This function adds artist's songs to playlist."""
        (mod, iter_) = tview.get_selection().get_selected()
        artist = unescape(mod.get_value(iter_, 0))
        songs = []

        for alb in self.songs_tree[artist]:
            for sg in self.songs_tree[artist][alb]:
                songs.append(sg)

        self.playlist.add_songs(songs)

    def load_album(self, tview, path, column):
        """This function adds album's songs to playlist."""
        try:
            tart = self.glade_file.get_widget('treeview3')
            (mod, iter_) = tart.get_selection().get_selected()
            artist = unescape(mod.get_value(iter_, 0))
            by_artist = True
        except TypeError:
            by_artist = False

        (mod, iter_) = tview.get_selection().get_selected()
        album = unescape(mod.get_value(iter_, 1))
        songs = []

        if by_artist:
            for sg in self.songs_tree[artist][album]:
                songs.append(sg)
        else:
            for art in self.songs_tree:
                for alb in self.songs_tree[art]:
                    if alb == album:
                        for sg in self.songs_tree[art][alb]:
                            songs.append(sg)

        if len(songs) > 0:
            self.playlist.add_songs(songs)