/* Copyright (c) 2004, Indiana University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  * Neither the name of Indiana University, Bloomington nor the names
 *    of its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */

package org.gmod.biomaj.ant.task.net;


import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.text.ParseException;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Vector;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Project;
import org.gmod.biomaj.ant.task.InputValidation;
import org.inria.biomaj.ant.logger.BiomajConsoleLogger;
import org.inria.biomaj.ant.task.BmajExtract;
import org.inria.biomaj.ant.task.BmajTask;
import org.inria.biomaj.internal.ant.task.net.RemoteFile;
import org.inria.biomaj.session.bank.Bank;
import org.inria.biomaj.session.bank.BiomajSQLQuerier;
import org.inria.biomaj.session.bank.FileDesc;
import org.inria.biomaj.session.bank.ProductionDirectory;
import org.inria.biomaj.singleton.BiomajLogger;
import org.inria.biomaj.utils.BiomajBuildException;
import org.inria.biomaj.utils.BiomajConst;
import org.inria.biomaj.utils.BiomajException;
import org.inria.biomaj.utils.BiomajUtils;

/**
 * This class takes an ftp listing generated by Ant's FTP task and it compares
 * the files in that listing to the same ones stored on local disk. It checks
 * dates between the two to see if the file needs to be downloaded or not. It is
 * particularly useful when you need to compare files that have been
 * uncompressed and/or whose filenames have changed This is the main java class
 * behind the Citrina package for mirroring biological databases.
 * 
 * Olivier Filangi<br>
 * This class has been modified to support multiple protocol.
 * 
 * @author Josh Goodman
 * 
 * @version Biomaj.0.9
 * 
 */
public class FileCheck extends BmajTask {

	/**
	 * remote files
	 * @uml.property  name="listingfile"
	 */
	private File listingfile;
	/**
	 * @uml.property  name="output"
	 */
	private File output;
	/**
	 * @uml.property  name="localFiles"
	 */
	private String localFiles;

	/**
	 * @uml.property  name="server"
	 */
	private String server;
	/**
	 * @uml.property  name="port"
	 */
	private int port;
	/**
	 * @uml.property  name="protocol"
	 */
	private String protocol;
	/**
	 * @uml.property  name="remoteDir"
	 */
	private String remoteDir;

	/**
	 * @uml.property  name="onlineDir"
	 */
	private File onlineDir;
	/**
	 * @uml.property  name="offlineDir"
	 */
	private File offlineDir;
	/**
	 * @uml.property  name="copyFileList"
	 */
	private File copyFileList;
	/**
	 * @uml.property  name="up2dateFile"
	 */
	private File up2dateFile;
	/**
	 * @uml.property  name="extractFileList"
	 */
	private File extractFileList;
	/**
	 * @uml.property  name="keepOfflineFileList"
	 */
	private File keepOfflineFileList;
	/**
	 * @uml.property  name="ftpListingFile"
	 */
	private BufferedReader ftpListingFile;
	/**
	 * @uml.property  name="ftpFilesNeeded"
	 */
	private BufferedWriter ftpFilesNeeded;

	/**
	 * @uml.property  name="keepOfflineFile"
	 */
	private BufferedWriter keepOfflineFile;
	/**
	 * @uml.property  name="extractFile"
	 */
	private BufferedWriter extractFile;
	/**
	 * @uml.property  name="copyFile"
	 */
	private BufferedWriter copyFile;
	/**
	 * @uml.property  name="up2dateFileWriter"
	 */
	private BufferedWriter up2dateFileWriter;

	/**
	 * @uml.property  name="countDownload"
	 */
	private int countDownload = 0;
	/**
	 * @uml.property  name="countExtract"
	 */
	private int countExtract  = 0;


	/**
	 * @uml.property  name="countKeepOffline"
	 */
	private int countKeepOffline = 0;
	/**
	 * @uml.property  name="countCopy"
	 */
	private int countCopyLocalOnline     = 0;

	/**
	 * @uml.property  name="countCopyLocalOffline"
	 */
	private int countCopyLocalOffline     = 0;

	/**
	 * @uml.property  name="filesUncompressedOnOffline"
	 */
	private boolean filesUncompressedOnOffline = false;

	/**
	 * @uml.property  name="sizeToDownload"
	 */
	private long sizeToDownload = 0;

	private Bank lastUpdateSession = null;
	
	private boolean fromScratch = false;

	@Override
	public void execute() {
		
		checkInput();			//Check the required attributes.
		setIO();				//Create the file IO objects.
		log("Finished initializing filecheck task.",Project.MSG_VERBOSE);

		boolean logFilesIsOk = Boolean.valueOf(getProject().getProperty(BiomajConst.logFilesProperty));


		//Initialize the line and count variables.
		String line = null;
		Vector<RemoteFile> listRF = new Vector<RemoteFile>();
		int compteur = 0;
		try {
			while ((line = ftpListingFile.readLine()) != null) {
				RemoteFile rf = new RemoteFile(line);
				listRF.add(rf);
				sizeToDownload += rf.getSize();
			}
			ftpListingFile.close();
			
			String name = getProject().getProperty(BiomajConst.dbNameProperty);
			//on cherche le dernier repertoire de prod creee
//			Vector<ProductionDirectory> lpd = BiomajQueryXmlStateFile.getAvailableDirectoriesProduction(name);
			
			List<ProductionDirectory> lpd = BiomajSQLQuerier.getAvailableProductionDirectories(name);
			
			if (lpd.size()<=0) {
				log("No production directories have been created !",Project.MSG_VERBOSE);
			} else {
				ProductionDirectory pd = lpd.get(lpd.size()-1);
				lastUpdateSession = new Bank();
				if (!BiomajSQLQuerier.getUpdateBank(name, pd.getSession(), lastUpdateSession, true))
					lastUpdateSession = null ;
				
			}
		}  catch (IOException ioe) {
			throw new BiomajBuildException(getProject(),ioe);
		} catch (BiomajException be) {
			throw new BuildException(be);
		} catch (ParseException pe) {
			throw new BiomajBuildException(getProject(),pe);
		}

		//Loop over the ftp listing file line by line.
		for (RemoteFile rFile : listRF) {
			
			float a = ((float)++compteur/ (float)listRF.size())*100;
			log(BiomajConsoleLogger.NOT_KEEP_LINE_ON_CONSOLE+"["+Integer.toString((int)a)+"%]",Project.MSG_INFO);
			//log(Integer.toString(compteur),Project.MSG_INFO);
			long remoteDate = rFile.getDate().getTime();
			long remoteSize  = rFile.getSize();
			String fileName = rFile.getAbsolutePath();
			
			if (!fromScratch) {

				//Correction de la methode getLocalNameFromStateFile : celle ci prend en compte si le fichier a ete extracte ou non!
				/*
				 * TODO:
				 * On peut optimiser l algo ci dessous:
				 * On a plus besoin de savoir si le fichier a ete etrait ou non puisque ca depend de no.extract!
				 * Est ce necessaire de faire une modif de cet algo puisqu il est destine a etre complement modifie? (Pris en compte seulement du statefile)
				 */
	
				//On test les archives seulement si les logs on ete generer.....
	
				if (logFilesIsOk && lastUpdateSession != null) {
					/*
					 * Cas ou on a une archive de type tar
					 * 1 archive correspond a plusieurs fichiers en prod
					 */
					Vector<FileDesc> lFd = lastUpdateSession.getGeneratedFiles(getProject(), rFile);
//					Vector<FileDesc> lFd = null;
					if (lFd != null && lFd.size() > 0) {
						log("Archives "+rFile.getAbsolutePath()+" with "+Integer.toString(lFd.size())+" files",Project.MSG_VERBOSE);		
						for (int aF=0;aF<lFd.size();aF++)
							log("   -> "+lFd.get(aF).getName()+" timestamp="+Long.toString(lFd.get(aF).getTime())+" size="+lFd.get(aF).getSize(),Project.MSG_VERBOSE);	
						
						int count = 0;
						Vector<String> toCopy = new Vector<String>();
						for (FileDesc fd : lFd) {
							log(rFile.getName()+" contains "+fd.getLocation(),Project.MSG_DEBUG);	
//							File inOnline = new File(onlineDir.getAbsolutePath()+"/"+fd.getLocation().replace(offlineDir.getAbsolutePath(), ""));
							File inOnline = new File(fd.getLocation());
							if (inOnline.exists()) {
								log(fd.getLocation()+" find in production!",Project.MSG_VERBOSE);
								try {
									toCopy.add(inOnline.getCanonicalPath());
								} catch (IOException e) {
									log(e, Project.MSG_ERR);
								}
								count++;
							} else {
								File inOffline = new File(offlineDir.getAbsolutePath() + " / " + fd.getLocation().replace(onlineDir.getAbsolutePath(), ""));
								if (inOffline.exists()) {
									log(fd.getLocation()+" find in offline!",Project.MSG_VERBOSE);
									filesUncompressedOnOffline = true;
									addFileToKeepOfflineList(inOffline.getAbsolutePath());
									count++;
								} else {
									log("file ["+fd.getLocation()+"] has been deleted, BioMAJ download "+rFile.getAbsolutePath(),Project.MSG_WARN);
									count = 0;
									break;
								}
							}
						}
						/*
						 * Si tous les fichiers existent, on ajoute en copy/link et on passe au remote file suivant!
						 */
						if (count == lFd.size()) {
							for (String path : toCopy) {
								addFileToCopyList(path);
							}
							continue;
						}
					} else {
						log("None file ["+rFile.getAbsolutePath()+"] have been stored in the last session ",Project.MSG_VERBOSE);
					}
				}
				/*
				 * Cas ou on a 1 archive correspondant a 1 fichier en prod
				 */
				String fileNameUncompressed = getLocalNameFromStateFile(rFile,logFilesIsOk);
				File fLocalUncompressedInOffline = new File(offlineDir,fileNameUncompressed);
				/*
				 * Fichier decompresse qui existe dans offline
				 */
				if (fLocalUncompressedInOffline.exists()) {
					FileDesc fd = null;//getFileDescFromFileCompressed(fileName);
					if (fd == null)
					{
						//log(fLocalUncompressedInOffline.getName()+" not handle by the statefile",Project.MSG_INFO);
						if (remoteDate<=fLocalUncompressedInOffline.lastModified()) {
							log(fLocalUncompressedInOffline.getAbsolutePath() +" find in offline directory",Project.MSG_VERBOSE);
							//!!! ON doit laisser une infos comme quoi il y a une nouvelle version a produire
							filesUncompressedOnOffline = true;
							addLocalOfflineFile(fLocalUncompressedInOffline.getAbsolutePath());
							addFileToKeepOfflineList(fLocalUncompressedInOffline.getAbsolutePath());
							countCopyLocalOffline++;
							continue;
						}
					} else {
						if ((remoteDate==fd.getTime())&&(remoteSize == fd.getSize()))
						{
							log(fLocalUncompressedInOffline.getAbsolutePath() +" find in offline directory",Project.MSG_VERBOSE);
							addLocalOfflineFile(fLocalUncompressedInOffline.getAbsolutePath());
							addFileToKeepOfflineList(fLocalUncompressedInOffline.getAbsolutePath());
							filesUncompressedOnOffline = true;
							countCopyLocalOffline++;
							continue;
						}
					}
				} 
				
				// le fichier existe en offline mais ne correspond pas au fichier distant : on doit l effacer
				// pour que les links du online vers le offline possible fonctionne par la suite
				fLocalUncompressedInOffline.delete();
	
				File fLocalUncompressedInOnline = new File(onlineDir,fileNameUncompressed);
				/*
				 * Fichier decompresse qui se trouve dans online
				 */
				if (fLocalUncompressedInOnline.exists()) {
					FileDesc fd = null;//getFileDescFromFileCompressed(fileName);
					if (fd == null)
					{
						//log(fLocalUncompressedInOnline.getName()+" not handle by the statefile",Project.MSG_INFO);
						long t = fLocalUncompressedInOnline.lastModified();
						if (remoteDate<=t) {
							log(fLocalUncompressedInOnline.getAbsolutePath() +" find in online directory",Project.MSG_VERBOSE);
							try {
								addFileToCopyList(fLocalUncompressedInOnline.getCanonicalPath());
							} catch (IOException e) {
								log(e, Project.MSG_ERR);
							}
							//si le meme fichier existe dans le offline, il faut l effacer
							if (fLocalUncompressedInOffline.exists())
							{
								log("delete:"+fLocalUncompressedInOffline.getAbsolutePath());
								fLocalUncompressedInOffline.delete();
							}
							continue;
						} else {
							log(rFile.getAbsolutePath()+": Local date ["+BiomajUtils.dateToString(new Date(fLocalUncompressedInOnline.lastModified()), Locale.US)+
									"] Remote date ["+BiomajUtils.dateToString(new Date(remoteDate), Locale.US)+"]",Project.MSG_VERBOSE);
						}
					} else {
						if ((remoteDate==fd.getTime())&&(remoteSize == fd.getSize())) {
							try {
								addFileToCopyList(fLocalUncompressedInOnline.getCanonicalPath());
							} catch (IOException e) {
								log(e, Project.MSG_ERR);
							}
						}
	//					si le meme fichier existe dans le offline, il faut l effacer
						if (fLocalUncompressedInOffline.exists())
						{
							log("delete:"+fLocalUncompressedInOffline.getAbsolutePath());
							fLocalUncompressedInOffline.delete();
						}
						continue;
					}
				}
	
				File fLocalcompressedInOffline = new File(offlineDir,fileName);
				/*
				 * Fichier compresse qui existe dans offline
				 */
				if (fLocalcompressedInOffline.exists()) {
					FileDesc fd = null;//  getFileDescFromFileCompressed(fileName);
					if (fd == null)
					{
						//log(fileName+" not handle by the statefile",Project.MSG_INFO);
						if (remoteSize==fLocalcompressedInOffline.length()&&(remoteDate<=fLocalcompressedInOffline.lastModified())) {
							log(fLocalcompressedInOffline.getAbsolutePath() +" find in offline directory",Project.MSG_VERBOSE);
							addFileToExtractList(fileName);
							addLocalOfflineFile(fLocalcompressedInOffline.getAbsolutePath());
							addFileToKeepOfflineList(fLocalcompressedInOffline.getAbsolutePath());
							countCopyLocalOffline++;
							continue;
						}
					} else {
						if ((remoteDate==fd.getTime())&&(fLocalcompressedInOffline.length()==fd.getSize())
								&&(remoteSize == fd.getSize()))
						{
							log(fLocalcompressedInOffline.getAbsolutePath() +" find in offline directory",Project.MSG_VERBOSE);
							addFileToExtractList(fileName);
							addLocalOfflineFile(fLocalcompressedInOffline.getAbsolutePath());
							addFileToKeepOfflineList(fLocalcompressedInOffline.getAbsolutePath());
							countCopyLocalOffline++;
							continue;
						}
					}
				}
	
	//			YOANN		Prise en compte des fichiers compresses dans le online		
				/*
				 * Fichier compresse qui se trouve dans online
				 */
	//			YOANN Prise en compte des fichiers compresses dans le online			
				File fLocalcompressedInOnline = new File(onlineDir,fileName);
	//			FIN YOANN	
				if (fLocalcompressedInOnline.exists()) {
					FileDesc fd = null;//getFileDescFromFileCompressed(fileName);
					if (fd == null)
					{
						//log(fLocalcompressedInOnline.getName()+" not handle by the statefile",Project.MSG_INFO);
						long t = fLocalcompressedInOnline.lastModified();
						if (remoteDate<=t) {
							log(fLocalcompressedInOnline.getAbsolutePath() +" find in online directory",Project.MSG_VERBOSE);
							try {
								addFileToCopyList(fLocalcompressedInOnline.getCanonicalPath());
							} catch (IOException e) {
								log(e, Project.MSG_ERR);
							}
							//si le meme fichier existe dans le offline, il faut l effacer
							if (fLocalcompressedInOffline.exists())
							{
								log("delete:"+fLocalcompressedInOffline.getAbsolutePath());
								fLocalcompressedInOffline.delete();
							}
							continue;
						}
					} else {
						if (remoteDate==fd.getTime() && remoteSize == fd.getSize()) {
							try {
								addFileToCopyList(fLocalcompressedInOnline.getCanonicalPath());
							} catch (IOException e) {
								log(e, Project.MSG_ERR);
							}
						}
	//					si le meme fichier existe dans le offline, il faut l effacer
						if (fLocalcompressedInOffline.exists())
						{
							log("delete:"+fLocalcompressedInOffline.getAbsolutePath());
							fLocalcompressedInOffline.delete();
						}
						continue;
					}
				}
			}

//			FIN YOANN
			addFileToDownloadList(rFile);

		}
		finished();
		/*}  catch (IOException ioe) {
			throw new BiomajBuildException(getProject(),ioe);
		} catch (BiomajException be) {
			throw new BuildException(be);
		} catch (ParseException pe) {
			throw new BiomajBuildException(getProject(),pe);
		}
		 */
	}

	protected String getLocalNameFromStateFile(RemoteFile rf,boolean logFilesIsOk) {
		//FileDesc res = null ;
		String defaultValue = rf.getAbsolutePath();

		//Si on ne trouve pas dans le fichier xml le fichier par defaut, le fichier en local porte 
		//le meme nom si no.extract=true et sans l extension de compression si no.extract=false

		if (!getProject().getProperties().containsKey(BiomajConst.noExtractProperty))
			defaultValue = BiomajUtils.getLocalFileName(rf.getAbsolutePath());
		else if ((!Boolean.valueOf(getProject().getProperty(BiomajConst.noExtractProperty)))) {
			defaultValue = BiomajUtils.getLocalFileName(rf.getAbsolutePath());
		}
		if (logFilesIsOk) {
			if (lastUpdateSession == null)
				return defaultValue;

			//res = lastUpdateSession.findFile(rf.getAbsolutePath(), rf.getDate().getTime(), rf.getSize(), Session.DOWNLOAD);
			String refHash = BiomajUtils.getHashFromRemoteFile(rf);
			/*
			if (res == null)
			{
				log(rf.getAbsolutePath()+" never logged in statefiles",Project.MSG_VERBOSE);
				return defaultValue;
			}
			 */
			Vector<FileDesc> lFd = lastUpdateSession.findDependancesFilesFromExtraction(refHash);
			if (lFd.size()==1) {
				log("Find !!!:"+lFd.get(0).getName());
				return lFd.get(0).getName();
			} else {
				BiomajLogger.getInstance().log("FileCheck::getLocalNameFromStateFile find more than one local file :");
				for (FileDesc f : lFd)
					BiomajLogger.getInstance().log(f.toString());

			}
		}

		return defaultValue;
	}


	/**
	 * Sets the file object in which the ant FTP task will store the remote ftp server directory listing.
	 * 
	 * @param file File to store the ftp server listing.
	 */
	public void setListingFile(File file) {
		listingfile = file;
	}

	/**
	 * Sets the pattern that will match the file on the local disk after it has been downloaded and extracted. The pattern format needs to follow the <code>java.util.regex.Pattern</code> specifications.
	 * @param s  Pattern to match the local files with.
	 * @see  java.util.regex.Pattern
	 * @uml.property  name="localFiles"
	 */
	public void setLocalFiles(String s) {
		localFiles = s;
	}

	/**
	 * Sets the file object which stores the filenames to be retrieved from the remote ftp server.
	 * @param file  File to store the ftp filenames to retrieve.
	 * @uml.property  name="output"
	 */
	public void setOutput(File file) {
		output = file;
	}

	/**
	 * Sets the online directory on the local disk for a database.
	 * @param dir  Local online directory.
	 * @uml.property  name="onlineDir"
	 */
	public void setOnlineDir(File dir) {
		onlineDir = dir;
	}

	/**
	 * Sets the offline directory on the local disk for a database.
	 * @param dir  Local offline directory.
	 * @uml.property  name="offlineDir"
	 */
	public void setOfflineDir(File dir) {
		offlineDir = dir;
	}

	/**
	 * Sets the file object used to store any filenames that need to be copied into the offline directory from the online directory before any post processing is done and it is moved online.
	 * @param file  File used to store a list of files that need to be copied locally.
	 * @uml.property  name="copyFileList"
	 */
	public void setCopyFileList(File file) {
		copyFileList = file;
	}

	/**
	 * Sets the file object used to store any filenames that need to be extracted.  This file is used as an includesfile for filesets in the extract target.
	 * @param file  The file to write the files requiring extraction to.
	 * @uml.property  name="extractFileList"
	 */
	public void setExtractFileList(File file) {
		extractFileList = file;
	}

	/**
	 * Sets the file object used to store the number of files needed for the download.
	 * 
	 * @param file The file object to store the property containing the number of files to download.
	 */
	public void setUpdateFile (File file) {
		up2dateFile = file;
	}

	/**
	 * Add a file in the download list and extract list
	 * @param parser
	 */
	private void addFileToDownloadList(RemoteFile rFile)  {


		try {
			//Log the actions.
			log(rFile.getName() + " added to the download list.",Project.MSG_VERBOSE);
			String extFile ;
			//Write the ftp url to the appropriate file and increment the file count.
			extFile=rFile.getAbsolutePath();


			/*	if (rFile.isLink()) 
				extFile=rFile.getLinkName();
			else */
			extFile=rFile.getAbsolutePath();

			//ftpFilesNeeded.write("locationOnServer="+location+",timeOnServer="+Long.toString(rFile.getDate().getTime())+",sizeOnServer="+Long.toString(rFile.getSize()));
			ftpFilesNeeded.write(rFile.toString());
			ftpFilesNeeded.newLine();

			log("file extract:"+extFile,Project.MSG_VERBOSE);
			log("rFile associated:"+rFile.toString(),Project.MSG_VERBOSE);
			addFileToExtractList(extFile);

			countDownload++;
		} catch (Exception ioe) {
			throw new BiomajBuildException(getProject(),ioe);
		}
	}

	/**
	 * Add a file in the extraction list
	 * @param parser
	 */

	private void addFileToExtractList(String fileName) {
		try {
			String extractName = BmajExtract.removeExtension(fileName);
			if (extractName.compareTo(fileName)==0) {
				log("no compressed format:"+fileName,Project.MSG_VERBOSE);
				return;
			}
			log(fileName + " added to the extract list.",Project.MSG_VERBOSE);
			extractFile.write(fileName);
			extractFile.newLine();
			countExtract++; 
		} catch (Exception e) {
			throw new BiomajBuildException(getProject(),"fileCheck.error.extract.file",fileName,e);
		}
	}

	/**
	 * Add a file in the copy list
	 * @param parser
	 */

	private void addFileToCopyList(String path) {
		try {
			//Secure system : we test if the file exist!
			File f = new File(path);
			if (!f.exists())
				throw new BiomajBuildException(getProject(),"fileCheck.error.add.file",path,new Exception());
			//log("AVANT PATH COPYLIST:"+path,Project.MSG_INFO);
			String newPath = path.replace(onlineDir.getCanonicalPath()+"/","");
			//log("PATH COPYLIST:"+path,Project.MSG_INFO);

			//Correction Bug 28 novembre , il faut creer les sous repertoires si il n existe pas dans le offline!
			BiomajUtils.createSubDirectories(BiomajUtils.getRelativeDirectory(offlineDir.getAbsolutePath()+"/"+newPath));

			copyFile.write(newPath);
			copyFile.newLine();
			countCopyLocalOnline++;
		} catch (IOException e) {
			throw new BiomajBuildException(getProject(),e);
		}
	}


	private void addFileToKeepOfflineList(String fileName)    {
		try {
			log(fileName + " added to the keep offline list.",Project.MSG_VERBOSE);
			keepOfflineFile.write(fileName);
			keepOfflineFile.newLine();
			countKeepOffline++; 
		} catch (IOException e) {
			throw new BiomajBuildException(getProject(),"fileCheck.error.keep.offline.file",fileName,e);
		}
	}

	/**
	 * Checks the attributes passed to filecheck task.
	 *
	 * @throws org.apache.tools.ant.BuildException If required attributes are not set.
	 */
	private void checkInput() {
		if (!listingfile.isFile() || !listingfile.canRead()) {
			throw new BiomajBuildException(getProject(),"fileCheck.error.correct.file",listingfile.getAbsolutePath(),new Exception());
		}

		InputValidation.checkString(getProject(),output, "the output file property");
		InputValidation.checkString(getProject(),keepOfflineFileList, "the output file property to keep file in offline directory");
		InputValidation.checkString(getProject(),localFiles, "the local ftp files regular expression");
		//InputValidation.checkString(ftpLocation,"the ftp server name and directory location");
		InputValidation.checkString(getProject(),onlineDir, "the online database directory");
		InputValidation.checkString(getProject(),copyFileList, "the local copy file list property");
		InputValidation.checkString(getProject(),up2dateFile, "the up2date file property");
		InputValidation.checkString(getProject(),extractFileList, "the local extraction file list property");

		//correction bug O.filangi delete \\s caractere in remote directory
		remoteDir = remoteDir.trim();
	}

	/**
	 * Creates the File objects needed for input/output for this class.
	 */
	private void setIO() {
		try {

			ftpListingFile = new BufferedReader(new FileReader(listingfile));
			ftpFilesNeeded = new BufferedWriter(new FileWriter(output));
			keepOfflineFile = new BufferedWriter(new FileWriter(keepOfflineFileList));
			copyFile = new BufferedWriter(new FileWriter(copyFileList));
			extractFile = new BufferedWriter(new FileWriter(extractFileList));
			up2dateFileWriter = new BufferedWriter(new FileWriter(up2dateFile));
		} catch (Exception e) {
			throw new BiomajBuildException(getProject(),e);
		} 
	}


	private void finished() {
		try {
//			The copy file is used as an includesfile in ant.  So when it is empty all files get included.
			//That is not what we want to happen so we use this line to prevent it.
			copyFile.write("#PLACE_HOLDER_TO_PREVENT_EMPTY_FILE");
			copyFile.newLine();

			extractFile.write("#PLACE_HOLDER_TO_PREVENT_EMPTY_FILE");
			extractFile.newLine();

			keepOfflineFile.write("#PLACE_HOLDER_TO_PREVENT_EMPTY_FILE");
			extractFile.newLine();

			//Close the file buffers and flush them out.
			copyFile.close();
			ftpFilesNeeded.close();
			extractFile.close();
			keepOfflineFile.close();

			//Set the count property to the final download count.
			up2dateFileWriter.write("num.new.files.download_and_extract=" + Integer.toString(countDownload)+"\n");
			up2dateFileWriter.write("remote.files.size=" + Long.toString(sizeToDownload)+"\n");
			up2dateFileWriter.write("num.new.files.extract=" + Integer.toString(countExtract)+"\n");
			up2dateFileWriter.write("num.new.files.offline=" + Integer.toString(countKeepOffline)+"\n");

			up2dateFileWriter.newLine();

			//On peut remplacer les test par Boolean.toString !!!!
			if (countDownload > 0)
				up2dateFileWriter.write(BiomajConst.filesDownloadNeedDynamicProperty+"=true\n");
			else
				up2dateFileWriter.write(BiomajConst.filesDownloadNeedDynamicProperty+"=false\n");

			getProject().setProperty(BiomajConst.countDownloadProperty, Integer.toString(countDownload));

			if (countExtract>0) 
				up2dateFileWriter.write(BiomajConst.filesExtractNeedDynamicProperty+"=true\n");
			else
				up2dateFileWriter.write(BiomajConst.filesExtractNeedDynamicProperty+"=false\n");

			getProject().setProperty(BiomajConst.countExtractProperty, Integer.toString(countExtract));

			//Pour pas se retrouver a creer une nouvelle version a chaque lancement de citrina
			//on test si de nouveaux fichiers sont gerer
			boolean copyNeeded  = false ;
			
			if ((countCopyLocalOnline > 0)&&(filesUncompressedOnOffline||(countExtract>0)||(countDownload > 0))) {
				copyNeeded  = true;
			}
			/*
			else {  //Modif 06/08/2007 : Prend en compte si une nouvelle version n a pas de nvx fichier
				//et a des fichiers effaces....
				if (countCopyLocalOnline < BiomajUtils.howManyFiles(onlineDir)) {
					//log("New version use a subversion of the current version!",Project.MSG_WARN);
					copyNeeded  = true ;
				} else {
					copyNeeded  = false ;
				}
			}*/

			up2dateFileWriter.write(BiomajConst.filesCopyNeedDynamicProperty+"="+Boolean.toString(copyNeeded)+"\n");
			getProject().setProperty(BiomajConst.filesCopyNeedDynamicProperty, Boolean.toString(copyNeeded));
			getProject().setProperty(BiomajConst.countLocalOnlineFileProperty, Integer.toString(countCopyLocalOnline));
			getProject().setProperty(BiomajConst.countLocalOfflineFileProperty, Integer.toString(countCopyLocalOffline));

			//on en profite pour mettre la valeur de la release distant!
			up2dateFileWriter.write(BiomajConst.remoteReleaseDynamicProperty+"="+getProject().getProperty(BiomajConst.releaseResultProperty)+"\n");

			up2dateFileWriter.close();

		} catch (Exception e) {
			throw new BiomajBuildException(getProject(),e);
		} 
	}



	@Override
	public String getTaskName() {
		return "check files";
	}



	/**
	 * @return  the port
	 * @uml.property  name="port"
	 */
	public int getPort() {
		return port;
	}

	/**
	 * @param port  the port to set
	 * @uml.property  name="port"
	 */
	public void setPort(int port) {
		this.port = port;
	}

	/**
	 * @return  the protocol
	 * @uml.property  name="protocol"
	 */
	public String getProtocol() {
		return protocol;
	}

	/**
	 * @param protocol  the protocol to set
	 * @uml.property  name="protocol"
	 */
	public void setProtocol(String protocol) {
		this.protocol = protocol;
	}

	/**
	 * @return  the remoteDir
	 * @uml.property  name="remoteDir"
	 */
	public String getRemoteDir() {
		return remoteDir;
	}

	/**
	 * @param remoteDir  the remoteDir to set
	 * @uml.property  name="remoteDir"
	 */
	public void setRemoteDir(String remoteDir) {
		this.remoteDir = remoteDir;
	}

	/**
	 * @return  the server
	 * @uml.property  name="server"
	 */
	public String getServer() {
		return server;
	}

	/**
	 * @param server  the server to set
	 * @uml.property  name="server"
	 */
	public void setServer(String server) {
		this.server = server;
	}
	
	public boolean isFromScratch() {
		return fromScratch;
	}
	
	public void setFromScratch(boolean fromScratch) {
		this.fromScratch = fromScratch;
	}

	/**
	 * @return  the keepOfflineFileList
	 * @uml.property  name="keepOfflineFileList"
	 */
	public File getKeepOfflineFileList() {
		return keepOfflineFileList;
	}

	/**
	 * @param keepOfflineFileList  the keepOfflineFileList to set
	 * @uml.property  name="keepOfflineFileList"
	 */
	public void setKeepOfflineFileList(File keepOfflineFileList) {
		this.keepOfflineFileList = keepOfflineFileList;
	}
}
