/******************************************************************************
 *                             Bin-prot                                       *
 *                                                                            *
 * Copyright (C) 2008- Jane Street Holding, LLC                               *
 *    Contact: opensource@janestreet.com                                      *
 *    WWW: http://www.janestreet.com/ocaml                                    *
 *    Author: Markus Mottl                                                    *
 *                                                                            *
 * This library is free software; you can redistribute it and/or              *
 * modify it under the terms of the GNU Lesser General Public                 *
 * License as published by the Free Software Foundation; either               *
 * version 2 of the License, or (at your option) any later version.           *
 *                                                                            *
 * This library is distributed in the hope that it will be useful,            *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this library; if not, write to the Free Software        *
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA  *
 *                                                                            *
 ******************************************************************************/

/* Common binary protocol definitions */

#ifndef COMMON_STUBS_H
#define COMMON_STUBS_H

#define _BSD_SOURCE
#include <sys/types.h>

#include <string.h>
#include <arpa/inet.h>

#include <caml/config.h>
#include <caml/memory.h>
#include <caml/alloc.h>
#include <caml/fail.h>
#include <caml/bigarray.h>
#include <caml/callback.h>
#include <caml/signals.h>

#ifdef ARCH_INT64_TYPE
#include "int64_native.h"
#else
#include "int64_emul.h"
#endif


/* Endianness- and alignment-independent integer marshalling functions */

#define le8dec(x) (*x)

#ifndef le16dec  /* FreeBSD standard functions not available */

#ifdef OS_DARWIN
#include <libkern/OSByteOrder.h>

#define le16dec(ptr) OSReadLittleInt16(ptr, 0)
#define le32dec(ptr) OSReadLittleInt32(ptr, 0)
#define le64dec(ptr) OSReadLittleInt64(ptr, 0)

#define be16dec(ptr) OSReadBigInt16(ptr, 0)
#define be32dec(ptr) OSReadBigInt32(ptr, 0)
#define be64dec(ptr) OSReadBigInt64(ptr, 0)

#define le16enc(ptr, n) OSWriteLittleInt16(ptr, 0, n)
#define le32enc(ptr, n) OSWriteLittleInt32(ptr, 0, n)
#define le64enc(ptr, n) OSWriteLittleInt64(ptr, 0, n)

#define be16enc(ptr, n) OSWriteBigInt16(ptr, 0, n)
#define be32enc(ptr, n) OSWriteBigInt32(ptr, 0, n)
#define be64enc(ptr, n) OSWriteBigInt64(ptr, 0, n)

#else  /* OS_DARWIN */

/* Emulate with memcpy and endianness conversion functions */

#define le16dec(ptr) \
  (__extension__ ({ uint16_t __n; memcpy(&__n, ptr, 2); le16toh(__n); }))
#define le32dec(ptr) \
  (__extension__ ({ uint32_t __n; memcpy(&__n, ptr, 4); le32toh(__n); }))
#define le64dec(ptr) \
  (__extension__ ({ uint64_t __n; memcpy(&__n, ptr, 8); le64toh(__n); }))

#define be16dec(ptr) \
  (__extension__ ({ uint16_t __n; memcpy(&__n, ptr, 2); be16toh(__n); }))
#define be32dec(ptr) \
  (__extension__ ({ uint32_t __n; memcpy(&__n, ptr, 4); be32toh(__n); }))
#define be64dec(ptr) \
  (__extension__ ({ uint64_t __n; memcpy(&__n, ptr, 8); be64toh(__n); }))

#define le16enc(ptr, n) \
  (__extension__ ({ uint16_t __n = htole16(n); memcpy(ptr, &__n, 2); }))
#define le32enc(ptr, n) \
  (__extension__ ({ uint32_t __n = htole32(n); memcpy(ptr, &__n, 4); }))
#define le64enc(ptr, n) \
  (__extension__ ({ uint64_t __n = htole64(n); memcpy(ptr, &__n, 8); }))

#define be16enc(ptr, n) \
  (__extension__ ({ uint16_t __n = htobe16(n); memcpy(ptr, &__n, 2); }))
#define be32enc(ptr, n) \
  (__extension__ ({ uint32_t __n = htobe32(n); memcpy(ptr, &__n, 4); }))
#define be64enc(ptr, n) \
  (__extension__ ({ uint64_t __n = htobe64(n); memcpy(ptr, &__n, 8); }))

#endif  /* OS_DARWIN */

#endif  /* le16dec */


/* Bin-prot integer codes */

#define CODE_NEG_INT8 (char) -1
#define CODE_INT16 (char) -2
#define CODE_INT32 (char) -3
#define CODE_INT64 (char) -4


/* Buffer short exception */

extern value *v_bin_prot_exc_Buffer_short;


/* GNU compiler pragmas */

#if __GNUC__ >= 3
# ifndef inline
#   define inline inline __attribute__ ((always_inline))
# endif
# ifndef __pure
#   define __pure __attribute__ ((pure))
# endif
# ifndef __const
#   define __const __attribute__ ((const))
# endif
# ifndef __malloc
#   define __malloc __attribute__ ((malloc))
# endif
# ifndef __unused
#   define __unused __attribute__ ((unused))
# endif
# ifndef __likely
#   define likely(x) __builtin_expect (!!(x), 1)
# endif
# ifndef __unlikely
#   define unlikely(x) __builtin_expect (!!(x), 0)
# endif
#else
# ifndef inline
#   define inline
# endif
# ifndef __pure
#   define __pure
# endif
# ifndef  __const
#   define __const
# endif
# ifndef  __malloc
#   define __malloc
# endif
# ifndef  __unused
#   define __unused
# endif
# ifndef  __likely
#   define likely(x) (x)
# endif
# ifndef  __unlikely
#   define unlikely(x) (x)
# endif
#endif

#endif /* COMMON_STUBS_H */
