/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <stdlib.h>

#include <kozo.h>

#include "bkl-entry.h"
#include "bkl-item.h"
#include "bkl-item-extended.h"
#include "bkl-item-audio.h"
#include "bkl-item-image.h"
#include "bkl-item-video.h"
#include "bkl-item-broken.h"

enum {
    PROP_0,
    PROP_URI,
    PROP_MIMETYPE,
    PROP_SIZE,
    PROP_MTIME
};

struct _BklItemPrivate {
    BklItemType type;

    char *uri;
    char *mimetype;

    goffset size;
    glong mtime;
};

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), BKL_TYPE_ITEM, BklItemPrivate))
G_DEFINE_TYPE (BklItem, bkl_item, G_TYPE_OBJECT);

static void
bkl_item_finalize (GObject *object)
{
    BklItem *self = (BklItem *) object;
    BklItemPrivate *priv = self->priv;

    g_free (priv->uri);
    g_free (priv->mimetype);

    g_signal_handlers_destroy (object);
    G_OBJECT_CLASS (bkl_item_parent_class)->finalize (object);
}

static void
bkl_item_dispose (GObject *object)
{
    G_OBJECT_CLASS (bkl_item_parent_class)->dispose (object);
}

static void
bkl_item_set_property (GObject      *object,
                       guint         prop_id,
                       const GValue *value,
                       GParamSpec   *pspec)
{
    switch (prop_id) {
    case PROP_URI:
        break;

    case PROP_MIMETYPE:
        break;

    case PROP_SIZE:
        break;

    case PROP_MTIME:
        break;

    default:
        break;
    }
}

static void
bkl_item_get_property (GObject    *object,
                       guint       prop_id,
                       GValue     *value,
                       GParamSpec *pspec)
{
    switch (prop_id) {
    case PROP_URI:
        break;

    case PROP_MIMETYPE:
        break;

    case PROP_SIZE:
        break;

    case PROP_MTIME:
        break;

    default:
        break;
    }
}

static void
bkl_item_class_init (BklItemClass *klass)
{
    GObjectClass *o_class = (GObjectClass *)klass;

    o_class->dispose = bkl_item_dispose;
    o_class->finalize = bkl_item_finalize;
    o_class->set_property = bkl_item_set_property;
    o_class->get_property = bkl_item_get_property;

    g_type_class_add_private (klass, sizeof (BklItemPrivate));

    g_object_class_install_property (o_class, PROP_URI,
                                     g_param_spec_string ("uri", "", "", "",
                                                          G_PARAM_READWRITE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_MIMETYPE,
                                     g_param_spec_string ("mimetype", "", "", "",
                                                          G_PARAM_READWRITE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_SIZE,
                                     g_param_spec_uint64 ("size", "", "",
                                                          0, G_MAXUINT64, 0,
                                                          G_PARAM_READWRITE |
                                                          G_PARAM_STATIC_STRINGS));

    g_object_class_install_property (o_class, PROP_MTIME,
                                     g_param_spec_long ("modification-time", "", "",
                                                        0, G_MAXLONG, 0,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_STRINGS));
}

static void
bkl_item_init (BklItem *self)
{
    self->priv = GET_PRIVATE (self);
    self->priv->type = BKL_ITEM_TYPE_NONE;
}

#define IS_EMPTY_STRING(s) ((s) == NULL || *(s) == '\0')
void
bkl_item_set_from_field (BklItem   *item,
                         KozoField *field)
{
    const char *data;

    data = kozo_field_get_value_string (field, BKL_FILE_FIELD_URI);
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_set_uri (item, data);
    }

    data = kozo_field_get_value_string (field, BKL_FILE_FIELD_MIMETYPE);
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_set_mimetype (item, data);
    }

    data = kozo_field_get_value_string (field, BKL_FILE_FIELD_SIZE);
    if (!IS_EMPTY_STRING (data)) {
        goffset size = strtoull (data, NULL, 10);
        bkl_item_set_size (item, size);
    }

    data = kozo_field_get_value_string (field, BKL_FILE_FIELD_MODIFICATION_TIME);
    if (!IS_EMPTY_STRING (data)) {
        glong mtime = strtol (data, NULL, 10);
        bkl_item_set_modification_time (item, mtime);
    }
}

#if __WORDSIZE == 64
#define G_GINT64_MODIFIER "l"
# else
#define G_GINT64_MODIFIER "ll"
# endif
static GSList *
get_fields (BklItem *item)
{
    BklItemPrivate *priv = item->priv;
    GSList *fields = NULL;

    fields = g_slist_prepend (fields, g_strdup_printf ("%ld", priv->mtime));
    fields = g_slist_prepend (fields, g_strdup_printf ("%" G_GINT64_MODIFIER "u", priv->size));
    fields = g_slist_prepend (fields, g_strdup (priv->mimetype));
    fields = g_slist_prepend (fields, g_strdup (priv->uri));

    return fields;
}

GSList *
bkl_item_get_fields (BklItem *item)
{
    BklItemPrivate *priv;
    GSList *fields = NULL;

    priv = item->priv;

    fields = g_slist_prepend
        (fields, kozo_field_new_string_list (BKL_FIELD_EXTENDED,
                                             bkl_item_extended_get_fields ((BklItemExtended *) item)));

    switch (priv->type) {
    case BKL_ITEM_TYPE_AUDIO: {
        GSList *audio_fields;

        audio_fields = bkl_item_audio_get_fields ((BklItemAudio *) item);
        fields = g_slist_prepend (fields,
                                  kozo_field_new_string_list (BKL_FIELD_AUDIO,
                                                              audio_fields));
        break;
    }

    case BKL_ITEM_TYPE_IMAGE: {
        GSList *image_fields;

        image_fields = bkl_item_image_get_fields ((BklItemImage *) item);
        fields = g_slist_prepend (fields,
                                  kozo_field_new_string_list (BKL_FIELD_IMAGE,
                                                              image_fields));
        break;
    }

    case BKL_ITEM_TYPE_VIDEO: {
        GSList *video_fields;

        video_fields = bkl_item_video_get_fields ((BklItemVideo *) item);
        fields = g_slist_prepend (fields,
                                  kozo_field_new_string_list (BKL_FIELD_VIDEO,
                                                              video_fields));
        break;
    }

    case BKL_ITEM_TYPE_BROKEN: {
        GSList *broken_fields;

        broken_fields = bkl_item_broken_get_fields ((BklItemBroken *) item);
        fields = g_slist_prepend (fields,
                                  kozo_field_new_string_list (BKL_FIELD_BROKEN,
                                                              broken_fields));
        break;
    }

    case BKL_ITEM_TYPE_PLAYLIST:
    default:
        break;
    }

    fields = g_slist_prepend (fields,
                              kozo_field_new_string_list (BKL_FIELD_FILE,
                                                          get_fields (item)));
    fields = g_slist_prepend (fields,
                              kozo_field_new_int (BKL_FIELD_TYPE, priv->type));

    return fields;
}

void
bkl_item_set_item_type (BklItem    *item,
                        BklItemType type)
{
    item->priv->type = type;
}

BklItemType
bkl_item_get_item_type (BklItem *item)
{
    return item->priv->type;
}

void
bkl_item_set_uri (BklItem    *item,
                  const char *uri)
{
    g_free (item->priv->uri);
    item->priv->uri = g_strdup (uri);
}

const char *
bkl_item_get_uri (BklItem *item)
{
    return item->priv->uri;
}

void
bkl_item_set_mimetype (BklItem    *item,
                       const char *mimetype)
{
    g_free (item->priv->mimetype);
    item->priv->mimetype = g_strdup (mimetype);
}

const char *
bkl_item_get_mimetype (BklItem *item)
{
    return item->priv->mimetype;
}

void
bkl_item_set_size (BklItem *item,
                   goffset  size)
{
    item->priv->size = size;
}

goffset
bkl_item_get_size (BklItem *item)
{
    return item->priv->size;
}

void
bkl_item_set_modification_time (BklItem *item,
                                glong    mtime)
{
    item->priv->mtime = mtime;
}

glong
bkl_item_get_modification_time (BklItem *item)
{
    return item->priv->mtime;
}

char *
bkl_item_to_string (BklItem *item)
{
    char *ret = NULL, *ret2;

    if (item == NULL) {
        return NULL;
    }

    /* FIXME: This should all be done via class methods shurely (sic)? */
    switch (item->priv->type) {
    case BKL_ITEM_TYPE_AUDIO:
        ret = bkl_item_audio_to_string ((BklItemAudio *) item);
        break;

    case BKL_ITEM_TYPE_IMAGE:
        ret = bkl_item_image_to_string ((BklItemImage *) item);
        break;

    case BKL_ITEM_TYPE_VIDEO:
        ret = bkl_item_video_to_string ((BklItemVideo *) item);
        break;

    default:
        return NULL;
    }

    ret2 = g_strdup_printf ("%s %s %s", ret, bkl_item_get_uri (item),
                            bkl_item_get_mimetype (item));
    g_free (ret);
    return ret2;
}
