/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include "session.h"
#include "constants.h"
#include "sqlite_reader.h"
#include "directories.h"
#include "gwrappers.h"
#include "utilities.h"


ustring session_filename ()
{
  return gw_build_filename (directories_get_temp (), "session.sqlite");
}


void session_create ()
{
  // Get filename of the database.
  ustring filename = session_filename ();
  // Remove the old db.
  unlink (filename.c_str());
  // Create new db with default values.
  Session session (0);
  session.searchresultstype (sstLoad);
  session.searchbibletimetype (sbttOpenModules);
  session.checksorttype (cstSort0);
  session.area_type (atRaw);
  session.line_cutter_for_hebrew_text_characters (80);
}


Session::Session (int dummy)
{
  // Check whether the session database is present, and create it if needed.
  ustring filename = session_filename ();
  if (!g_file_test (filename.c_str(), G_FILE_TEST_IS_REGULAR)) {
    create (filename);
  }
  // Open the db and keep it open throughout the lifetime of this object.
  error = NULL;
  try
  {
    rc = sqlite3_open (filename.c_str (), &db);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    sqlite3_busy_timeout (db, 1000);
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
}


Session::~Session ()
{
  sqlite3_close (db);
}


void Session::create (const ustring& filename)
{
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    rc = sqlite3_open (filename.c_str (), &db);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    sqlite3_busy_timeout (db, 1000);
    char * sql;
    sql = g_strdup_printf ("create table singles (key text, value text);");
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    g_free (sql);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    sql = g_strdup_printf ("create table lists (key text, value text, sequence integer);");
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    g_free (sql);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  sqlite3_close (db);
}


ustring Session::string_get (gchar * key)
{
  ustring value;
  try
  {
    char * sql;
    sqlite3_busy_timeout (db, 1000);
    SqliteReader sqlitereader (0);
    sql = g_strdup_printf ("select value from singles where key = '%s';", key);
    rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) throw runtime_error (error);
    if ((sqlitereader.ustring0.size() > 0)) value = sqlitereader.ustring0[0];
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  return value;
}


void Session::string_set (gchar * key, const ustring& value)
{
  try
  {
    char * sql;
    sql = g_strdup_printf ("delete from singles where key = '%s';", key);
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    sql = g_strdup_printf ("insert into singles values ('%s', '%s')", key, double_apostrophy (value).c_str());
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    g_free (sql);
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
}


bool Session::boolean_get (gchar * key)
{
  bool value = false;
  ustring s = string_get (key);
  if (!s.empty()) 
    value = convert_to_bool (s);
  return value;
}


void Session::boolean_set (gchar * key, bool value)
{
  string_set (key, convert_to_string (value));
}


int Session::integer_get (gchar * key)
{
  int value = 0;
  ustring s = string_get (key);
  if (!s.empty())
    value = convert_to_int (s);
  return value;
}


void Session::integer_set (gchar * key, int value)
{
  string_set (key, convert_to_string (value));
}


vector<ustring> Session::stringlist_get (gchar * key)
{
  vector<ustring> value;
  try
  {
    vector<unsigned int> sequence;
    char * sql;
    sqlite3_busy_timeout (db, 1000);
    SqliteReader sqlitereader (0);
    sql = g_strdup_printf ("select value, sequence from lists where key = '%s';", key);
    rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    value.assign (sqlitereader.ustring0.begin(), sqlitereader.ustring0.end());
    for (unsigned int i = 0; i < sqlitereader.ustring1.size(); i++) {
      sequence.push_back (convert_to_int (sqlitereader.ustring1[i]));
    }
    quick_sort (sequence, value, 0, sequence.size());
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  return value;
}


void Session::stringlist_set (gchar * key, const vector<ustring>& value)
{
  try
  {
    char * sql;
    sql = g_strdup_printf ("delete from lists where key = '%s';", key);
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    g_free (sql);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    for (unsigned int i = 0; i < value.size(); i++) {
      sql = g_strdup_printf ("insert into lists values ('%s', '%s', %d)", key, double_apostrophy (value[i]).c_str(), i);
      rc = sqlite3_exec (db, sql, NULL, NULL, &error);
      g_free (sql);
      if (rc) {
        throw runtime_error (sqlite3_errmsg(db));
      }
    }
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
}


vector<int> Session::integerlist_get (gchar * key)
{
  vector<ustring> value = stringlist_get (key);
  vector<int > il;
  for (unsigned int i = 0; i < value.size(); i++)
    il.push_back (convert_to_int (value[i]));
  return il;
}


void Session::integerlist_set (gchar * key, const vector<int>& value)
{
  vector<ustring> s;
  for (unsigned int i = 0; i < value.size(); i++)
    s.push_back (convert_to_string (value[i]));
  stringlist_set (key, s);
}


vector<bool> Session::booleanlist_get (gchar * key)
{
  vector<ustring> value = stringlist_get (key);
  vector<bool> b;
  for (unsigned int i = 0; i < value.size(); i++)
    b.push_back (convert_to_bool (value[i]));
  return b;
}


void Session::booleanlist_set (gchar * key, const vector<bool>& value)
{
  vector<ustring> s;
  for (unsigned int i = 0; i < value.size(); i++)
    s.push_back (convert_to_string (value[i]));
  stringlist_set (key, s);
}


ustring Session::searchword ()
{
  return string_get ("1");
}


void Session::searchword (const ustring& value)
{
  string_set ("1", value);
}


ustring Session::replaceword ()
{
  return string_get ("2");
}


void Session::replaceword (const ustring& value)
{
  string_set ("2", value);
}


bool Session::search_case_sensitive ()
{
  return boolean_get ("3");
}


void Session::search_case_sensitive (bool value)
{
  boolean_set ("3", value);
}


bool Session::search_current_book ()
{
  return boolean_get ("4");
}


void Session::search_current_book (bool value)
{
  boolean_set ("4", value);
}


bool Session::search_globbing ()
{
  return boolean_get ("5");
}


void Session::search_globbing (bool value)
{
  boolean_set ("5", value);
}


bool Session::search_start_word_match ()
{
  return boolean_get ("6");
}


void Session::search_start_word_match (bool value)
{
  boolean_set ("6", value);
}


bool Session::search_end_word_match ()
{
  return boolean_get ("7");
}


void Session::search_end_word_match (bool value)
{
  boolean_set ("7", value);
}


SearchResultsType Session::searchresultstype ()
{
  return SearchResultsType (integer_get ("8"));
}


void Session::searchresultstype (SearchResultsType value)
{
  integer_set ("8", int (value));
}


int Session::search_page ()
{
  return integer_get ("9");
}


void Session::search_page (int value)
{
  integer_set ("9", int (value));
}


SearchBibleTimeType Session::searchbibletimetype ()
{
  return SearchBibleTimeType (integer_get ("a"));
}


void Session::searchbibletimetype (SearchBibleTimeType value)
{
  integer_set ("a", int (value));
}


ustring Session::search_bibletime_bible ()
{
  return string_get ("b");
}


void Session::search_bibletime_bible (const ustring& value)
{
  string_set ("b", value);
}


ustring Session::search_bibletime_commentary ()
{
  return string_get ("c");
}


void Session::search_bibletime_commentary (const ustring& value)
{
  string_set ("c", value);
}


set<unsigned int> Session::selected_books ()
{
  set<unsigned int> value;
  vector<ustring> s = stringlist_get ("d");
  for (unsigned int i = 0; i < s.size(); i++)
    value.insert (convert_to_int (s[i]));
  return value;
}


void Session::selected_books (const set<unsigned int>& value)
{
  vector<unsigned int> s (value.begin(), value.end());
  vector<ustring> s2;
  for (unsigned int i = 0; i < s.size(); i++)
    s2.push_back (convert_to_string (s[i]));
  stringlist_set ("d", s2);
}


void Session::highlights_clear ()
{
  vector<ustring> s;
  vector<bool> b;
  vector<AreaType> a;
  highlight_words (s);
  highlight_casesensitives (b);
  highlight_globbings (b);
  highlight_matchbegins (b);
  highlight_matchends (b);
  highlight_area_type (a);
  highlight_area_id (b);
  highlight_area_intro (b);
  highlight_area_heading (b);
  highlight_area_chapter (b);
  highlight_area_study (b);
  highlight_area_notes (b);
  highlight_area_xref (b);
  highlight_area_verse (b);
}


void Session::highlights_add (const ustring& word, 
                              bool casesensitive, bool globbing, 
                              bool matchbegin, bool matchend,
                              AreaType areatype,
                              bool id, bool intro, bool heading, bool chapter, 
                              bool study, bool notes, bool xref, bool verse)
{
  vector <ustring> s = highlight_words ();
  s.push_back (word);
  highlight_words (s);

  vector <bool> b1 = highlight_casesensitives ();
  b1.push_back (casesensitive);
  highlight_casesensitives (b1);

  vector <bool> b2 = highlight_globbings ();
  b2.push_back (globbing);
  highlight_globbings (b2);
  
  vector <bool> b3 = highlight_matchbegins ();
  b3.push_back (matchbegin);
  highlight_matchbegins (b3);
  
  vector <bool> b4 = highlight_matchends ();
  b4.push_back (matchend);
  highlight_matchends (b4);
  
  vector <AreaType> at = highlight_area_type ();
  at.push_back (areatype);
  highlight_area_type (at);
  
  vector<bool> b5 = highlight_area_id ();
  b5.push_back (id);
  highlight_area_id (b5);
  
  vector<bool> b6 = highlight_area_intro ();
  b6.push_back (intro);
  highlight_area_intro (b6);
  
  vector<bool> b7 = highlight_area_heading ();
  b7.push_back (heading);
  highlight_area_heading (b7);
  
  vector<bool> b8 = highlight_area_chapter ();
  b8.push_back (chapter);
  highlight_area_chapter (b8);
  
  vector<bool> b9 = highlight_area_study ();
  b9.push_back (study);
  highlight_area_study (b9);
  
  vector<bool> b10 = highlight_area_notes ();
  b10.push_back (notes);
  highlight_area_notes (b10);
  
  vector<bool> b11 = highlight_area_xref ();
  b11.push_back (xref);
  highlight_area_xref (b11);
  
  vector<bool> b12 = highlight_area_verse ();
  b12.push_back (verse);
  highlight_area_verse (b12);
}


vector<ustring> Session::highlight_words ()
{
  return stringlist_get ("e");
}


void Session::highlight_words (const vector<ustring>& value)
{
  stringlist_set ("e", value);
}


vector<bool> Session::highlight_casesensitives ()
{
  return booleanlist_get ("f");
}


void Session::highlight_casesensitives (const vector<bool>& value)
{
  booleanlist_set ("f", value);
}


vector<bool> Session::highlight_globbings ()
{
  return booleanlist_get ("g");
}


void Session::highlight_globbings (const vector<bool>& value)
{
  booleanlist_set ("g", value);
}


vector<bool> Session::highlight_matchbegins ()
{
  return booleanlist_get ("h");
}


void Session::highlight_matchbegins (const vector<bool>& value)
{
  booleanlist_set ("h", value);
}


vector<bool> Session::highlight_matchends ()
{
  return booleanlist_get ("i");
}


void Session::highlight_matchends (const vector<bool>& value)
{
  booleanlist_set ("i", value);
}


vector <ustring> Session::import_references_searchwords ()
{
  return stringlist_get ("j");
}


void Session::import_references_searchwords (const vector <ustring>& value)
{
  stringlist_set ("j", value);
}


ustring Session::export_usfm_location ()
{
  return string_get ("k");
}


void Session::export_usfm_location (const ustring& value)
{
  string_set ("k", value);
}


vector<ustring> Session::completion_search ()
{
  return stringlist_get ("l");
}


void Session::completion_search (const vector<ustring>& value)
{
  stringlist_set ("l", value);
}


vector<ustring> Session::completion_replace ()
{
  return stringlist_get ("m");
}


void Session::completion_replace (const vector<ustring>& value)
{
  stringlist_set ("m", value);
}


vector<ustring> Session::completion_goto ()
{
  return stringlist_get ("n");
}


void Session::completion_goto (const vector<ustring>& value)
{
  stringlist_set ("n", value);
}


CheckSortType Session::checksorttype ()
{
  return CheckSortType (integer_get ("o"));
}


void Session::checksorttype (CheckSortType value)
{
  integer_set ("o", int (value));
}


AreaType Session::area_type ()
{
  return AreaType (integer_get ("q"));
}


void Session::area_type (AreaType value)
{
  integer_set ("q", int (value));
}


bool Session::area_id ()
{
  return boolean_get ("r");
}


void Session::area_id (bool value)
{
  boolean_set ("r", value);
}


bool Session::area_intro ()
{
  return boolean_get ("s");
}


void Session::area_intro (bool value)
{
  boolean_set ("s", value);
}


bool Session::area_heading ()
{
  return boolean_get ("t");
}


void Session::area_heading (bool value)
{
  boolean_set ("t", value);
}


bool Session::area_chapter ()
{
  return boolean_get ("u");
}


void Session::area_chapter (bool value)
{
  boolean_set ("u", value);
}


bool Session::area_study ()
{
  return boolean_get ("v");
}


void Session::area_study (bool value)
{
  boolean_set ("v", value);
}


bool Session::area_notes ()
{
  return boolean_get ("w");
}


void Session::area_notes (bool value)
{
  boolean_set ("w", value);
}


bool Session::area_xref ()
{
  return boolean_get ("x");
}


void Session::area_xref (bool value)
{
  boolean_set ("x", value);
}


bool Session::area_verse ()
{
  return boolean_get ("y");
}


void Session::area_verse (bool value)
{
  boolean_set ("y", value);
}


vector<ustring> Session::additional_printing_projects ()
{
  return stringlist_get ("z");
}


void Session::additional_printing_projects (const vector<ustring>& value)
{
  stringlist_set ("z", value);
}


int Session::line_cutter_for_hebrew_text_characters ()
{
  return integer_get ("00");
}


void Session::line_cutter_for_hebrew_text_characters (int value)
{
  integer_set ("00", value);
}


bool Session::search_current_chapter ()
{
  return boolean_get ("01");
}


void Session::search_current_chapter (bool value)
{
  boolean_set ("01", value);
}


vector<AreaType> Session::highlight_area_type ()
{
  vector<AreaType> at;
  vector <int> il = integerlist_get ("02");
  for (unsigned int i = 0; i < il.size(); i++)
    at.push_back ((AreaType) il[i]);
  return at;
}


void Session::highlight_area_type (const vector<AreaType>& value)
{
  vector <int> il;
  for (unsigned int i = 0; i < value.size(); i++)
    il.push_back ((int) value[i]);
  integerlist_set ("02", il);
}


vector<bool> Session::highlight_area_id ()
{
  return booleanlist_get ("03");
}


void Session::highlight_area_id (const vector<bool>& value)
{
  booleanlist_set ("03", value);
}


vector<bool> Session::highlight_area_intro ()
{
  return booleanlist_get ("04");
}


void Session::highlight_area_intro (const vector<bool>& value)
{
  booleanlist_set ("04", value);
}


vector<bool> Session::highlight_area_heading ()
{
  return booleanlist_get ("05");
}


void Session::highlight_area_heading (const vector<bool>& value)
{
  booleanlist_set ("05", value);
}


vector<bool> Session::highlight_area_chapter ()
{
  return booleanlist_get ("06");
}


void Session::highlight_area_chapter (const vector<bool>& value)
{
  booleanlist_set ("06", value);
}


vector<bool> Session::highlight_area_study ()
{
  return booleanlist_get ("07");
}


void Session::highlight_area_study (const vector<bool>& value)
{
  booleanlist_set ("07", value);
}


vector<bool> Session::highlight_area_notes ()
{
  return booleanlist_get ("08");
}


void Session::highlight_area_notes (const vector<bool>& value)
{
  booleanlist_set ("08", value);
}


vector<bool> Session::highlight_area_xref ()
{
  return booleanlist_get ("09");
}


void Session::highlight_area_xref (const vector<bool>& value)
{
  booleanlist_set ("09", value);
}


vector<bool> Session::highlight_area_verse ()
{
  return booleanlist_get ("0a");
}


void Session::highlight_area_verse (const vector<bool>& value)
{
  booleanlist_set ("0a", value);
}


bool Session::debug ()
{
  return boolean_get ("0b");
}


void Session::debug (bool value)
{
  boolean_set ("0b", value);
}


bool Session::window_initialized ()
{
  return boolean_get ("0c");
}


void Session::window_initialized (bool value)
{
  boolean_set ("0c", value);
}
