/*
 * Copyright (C) 2005-2008 Junjiro Okajima
 *
 * This program, aufs is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * superblock private data
 *
 * $Id: sbinfo.c,v 1.50 2008/04/13 23:44:46 sfjro Exp $
 */

#include "aufs.h"

struct au_branch *au_sbr(struct super_block *sb, aufs_bindex_t bindex)
{
	struct au_branch *br;

	SiMustAnyLock(sb);
	AuDebugOn(bindex < 0 || au_sbend(sb) < bindex);
	br = au_sbi(sb)->si_branch[0 + bindex];
	AuDebugOn(!br);
	return br;
}

au_gen_t au_sigen_inc(struct super_block *sb)
{
	au_gen_t gen;

	SiMustWriteLock(sb);
	gen = ++au_sbi(sb)->si_generation;
	au_update_digen(sb->s_root);
	au_update_iigen(sb->s_root->d_inode);
	sb->s_root->d_inode->i_version++;
	return gen;
}

int au_find_bindex(struct super_block *sb, struct au_branch *br)
{
	aufs_bindex_t bindex, bend;

	bend = au_sbend(sb);
	for (bindex = 0; bindex <= bend; bindex++)
		if (au_sbr(sb, bindex) == br)
			return bindex;
	return -1;
}

/* ---------------------------------------------------------------------- */

/* dentry and super_block lock. call at entry point */
void aufs_read_lock(struct dentry *dentry, int flags)
{
	si_read_lock(dentry->d_sb, flags);
	if (au_ftest_lock(flags, DW))
		di_write_lock_child(dentry);
	else
		di_read_lock_child(dentry, flags);
}

void aufs_read_unlock(struct dentry *dentry, int flags)
{
	if (au_ftest_lock(flags, DW))
		di_write_unlock(dentry);
	else
		di_read_unlock(dentry, flags);
	si_read_unlock(dentry->d_sb);
}

void aufs_write_lock(struct dentry *dentry)
{
	si_write_lock(dentry->d_sb);
	di_write_lock_child(dentry);
}

void aufs_write_unlock(struct dentry *dentry)
{
	di_write_unlock(dentry);
	si_write_unlock(dentry->d_sb);
}

void aufs_read_and_write_lock2(struct dentry *d1, struct dentry *d2, int flags)
{
	AuDebugOn(d1 == d2 || d1->d_sb != d2->d_sb);
	si_read_lock(d1->d_sb, flags);
	di_write_lock2_child(d1, d2, au_ftest_lock(flags, DIR));
}

void aufs_read_and_write_unlock2(struct dentry *d1, struct dentry *d2)
{
	AuDebugOn(d1 == d2 || d1->d_sb != d2->d_sb);
	di_write_unlock2(d1, d2);
	si_read_unlock(d1->d_sb);
}

/* ---------------------------------------------------------------------- */

aufs_bindex_t au_new_br_id(struct super_block *sb)
{
	aufs_bindex_t br_id;
	struct au_sbinfo *sbinfo;

	AuTraceEnter();
	SiMustWriteLock(sb);

	sbinfo = au_sbi(sb);
	while (1) {
		br_id = ++sbinfo->si_last_br_id;
		if (br_id && au_br_index(sb, br_id) < 0)
			return br_id;
	}
}
