/* $Id: Collision.cpp,v 1.6 2003/03/26 15:43:02 mrq Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
 #include <config.h>
#endif

#include <Ark/ArkModelState.h>
#include <Modules/Collision/Collision.h>


// IMPORTANT NOTE FOR LATER.......
// FIXME: don't forget that when doing collision : skel's hitboxes aren't
// scaled.for (i = 0; i < m_HitBoxes.size(); i++)
//     m_HitBoxes[i].Transform (mat);
   
namespace Ark
{

   CDModel::CDModel ()
   {
      m_BoneMatrices = NULL;
   }
   
   CDModel::~CDModel()
   {
      std::vector<CDSubmodel*>::iterator i;

      for (i = m_SubModels.begin(); i != m_SubModels.end(); i++)
      {
	 delete (*i);
	 *i = 0;
      }

      m_SubModels.clear();
      if (m_Model)
      {
	 //m_Model->Unref();
	 m_Model = NULL;
      }

      if (m_BoneMatrices)
      {
	 delete[] m_BoneMatrices;
	 m_BoneMatrices = NULL;
      }
   }

   void
   CDModel::Build (Model *model)
   {
      m_Model = model;
      ModelState mState (m_Model);
      mState.Play(ModelState::ONCE,"stand");
      mState.ComputeMatrix ();
      m_HasSkel = model->m_Skeleton != NULL;

      if (m_HasSkel)
      {
	 m_BoneMatrices = new Matrix44[model->m_Skeleton->m_Bones.size()];
	 
	 for (size_t i = 0; i < model->m_Skeleton->m_Bones.size(); ++i)
	    m_BoneMatrices[i] = mState.m_BoneMatrices[i];
      }
   
      for (size_t i = 0; i < model->m_SubModels.size(); i++)
	 m_SubModels.push_back (BuildSubmodel (&model->m_SubModels[i]));
   }

   static void CD_AppendPB (std::vector<udword> &triangles,
			    std::vector<udword> &materials,
			    PrimitiveBlock &pb,
			    int mat)
   {

      size_t j = 2;

      switch (pb.Type())
      {
	 case PRIM_TRIANGLES:
	    for (j = 0; j < pb.Size(); j++)
	    {
	       triangles.push_back (pb[j]);
	       if (j%3 == 0) materials.push_back (mat);
	    }
	    break;

	 case PRIM_TRIANGLE_STRIP:
	    for (j = 2; j < pb.Size(); j++)
	    {
	       if (j & 1) 
	       {
		  triangles.push_back (pb[j-2]);
		  triangles.push_back (pb[j-1]);
		  triangles.push_back (pb[j]);
		  materials.push_back (mat);
	       }
	       else
	       {
		  triangles.push_back (pb[j-2]);
		  triangles.push_back (pb[j]);
		  triangles.push_back (pb[j-1]);
		  materials.push_back (mat);
	       }
	    }
	    break;

	 case PRIM_TRIANGLE_FAN:
	    for (j = 2; j < pb.Size(); j++)
	    {
	       triangles.push_back (pb[0]);
	       triangles.push_back (pb[j-1]);
	       triangles.push_back (pb[j]);
	       materials.push_back (mat);
	    }
	    break;

	 default:
	    break;
      }
   }

   CDSubmodel*
   CDModel::BuildSubmodel (SubModel *mdl)
   {
      Opcode::OPCODECREATE opc;
      CDSubmodel *submdl = new CDSubmodel;
      submdl->m_ColModel = this;

      // Copy triangles... 
      size_t i;

      for (i = 0; i < mdl->m_Meshes.size(); i++)
      {
	 Mesh &mesh = mdl->m_Meshes[i];

	 for (BlockList::iterator pb = mesh.m_Blocks.begin();
	      pb != mesh.m_Blocks.end(); ++pb)
	 {
	    CD_AppendPB (submdl->m_Triangles,
			 submdl->m_Materials,
			 *pb,
			 mesh.m_Material);
	 }
      }

      opc.NbTris = submdl->m_Triangles.size()/3;
      opc.Tris = &submdl->m_Triangles[0];

      /// Copy vertices...
      std::vector<Opcode::Point> vertices;
      if (m_HasSkel)
      {
	 submdl->m_SkelVB.SetFormat(VertexBuffer::VB_HAS_COORD);
	 submdl->m_SkelVB.Resize(mdl->m_VB.Size());

	 for (i = 0; i < mdl->m_VB.Size(); i++)
	 {
	    Matrix44& bm = m_BoneMatrices[mdl->m_BoneBindings[i]];
	   
	    Vector3 coo = (mdl->m_SkinnedVB == NULL) ?
	       bm.Transform (mdl->m_VB.Coord(i)) :
	       bm.Transform (mdl->m_SkinnedVB->Coord(i));
	    vertices.push_back (Opcode::Point (&coo.X));
	    submdl->m_SkelVB.Coord(i) = coo;
	 }
      }
      else
      {
	 for (i = 0; i < mdl->m_VB.Size(); i++)
	    vertices.push_back (Opcode::Point (&mdl->m_VB.Coord(i).X));
      }

      opc.NbVerts = vertices.size();
      opc.Verts = &vertices[0];

      /// Create model
      Opcode::OPCODE_Model *cdmdl = new Opcode::OPCODE_Model;
      cdmdl->Build (opc);

      submdl->m_Model = mdl;
      submdl->m_CDModel = cdmdl;

      return submdl;
   }

   //////////////////////////////////////////////////////////////////////

   void CD_GetTriangle (uint triangleindex,
			Vector3 *points, 
			int *mat,
			CDSubmodel *mdl)
   {
      int tindex = 3 * triangleindex;

      if (mdl->m_ColModel->m_Model->m_Skeleton)
      {
	 for (size_t j = 0; j < 3; j++, tindex++)
	    points[j] = mdl->m_SkelVB.Coord(mdl->m_Triangles[tindex]);
      }
      else
	 for (size_t j = 0; j < 3; j++, tindex++)
	 {
	    points[j] = mdl->m_Model->m_VB.Coord
	       (mdl->m_Triangles[tindex]);
	 }
      
      if (mat != NULL)
	 *mat = mdl->m_Materials[triangleindex];
   }

   static void ColCallback (udword triangleindex,
			    Opcode::VertexPointers &triangle, 
			    udword userdata)
   {
      CDSubmodel* mdl = (CDSubmodel*) userdata;

      int tindex = 3 * triangleindex;
      for (size_t j = 0; j < 3; j++, tindex++)
      {
	 triangle.Vertex[j] = (Opcode::Point*)
	    &mdl->m_Model->m_VB.Coord (mdl->m_Triangles[tindex]);
      }
   }

   static void ColCallbackSkel (udword triangleindex,
				Opcode::VertexPointers &triangle, 
				udword userdata)
   {
      CDSubmodel* mdl = (CDSubmodel*) userdata;
      
      int tindex = 3 * triangleindex;
      for (size_t j = 0; j < 3; j++, tindex++)
	 triangle.Vertex[j] = (Opcode::Point*)
	    &mdl->m_SkelVB.Coord(mdl->m_Triangles[tindex]);
   }

   CDSystem::CDSystem()
   {
      m_TreeCollider.SetFirstContact (true);
      m_TreeCollider.SetFullBoxBoxTest (false);
      m_TreeCollider.SetFullPrimBoxTest (false);
      m_TreeCollider.SetCallbackObj0 (ColCallback);
      m_TreeCollider.SetCallbackObj1 (ColCallback);
   }

   /**
    * Destroy the given collision system. The collision models won't
    * be destroyed, so you have to be careful to delete them before
    * you call this destructor. Otherwise the effects are unpredictable.
    */
   CDSystem::~CDSystem ()
   {
   }

   /**
    * Returns true if a collision occurs between the two models
    * given as arguments. In such a case, the function fills the fields
    * of the pair structure.
    *
    * Remark : only the first collision is taken into account.
    */
   bool CDSystem::TestCollision (ModelState &ms0, ModelState &ms1,
				 ColPair &pair)
   {
      Model *m0 = ms0.GetModel();
      Model *m1 = ms1.GetModel();

      if (m0 == 0 || m1 == 0)
	 return false;

      CDModel *cm0 = (CDModel*) m0->m_CDModel;
      CDModel *cm1 = (CDModel*) m1->m_CDModel;

      if (cm0 == 0 || cm1 == 0)
	 return false;

      if (cm0->m_HasSkel) m_TreeCollider.SetCallbackObj0 (ColCallbackSkel);
      else m_TreeCollider.SetCallbackObj0 (ColCallback);

      if (cm1->m_HasSkel) m_TreeCollider.SetCallbackObj1 (ColCallbackSkel);
      else m_TreeCollider.SetCallbackObj1 (ColCallback);

      for (size_t i = 0; i < cm0->m_SubModels.size(); i++)
      {
	 for (size_t j = 0; j < cm1->m_SubModels.size(); j++)
	 {
	    /// FIXME: use temporal coherence to cache collision results...
	    Opcode::BVTCache colcache;
	    colcache.Model0 = cm0->m_SubModels[i]->m_CDModel;
	    colcache.Model1 = cm1->m_SubModels[j]->m_CDModel;

	    m_TreeCollider.SetUserData0 ((udword) cm0->m_SubModels[i]);
	    m_TreeCollider.SetUserData1 ((udword) cm1->m_SubModels[j]);

	    m_TreeCollider.Collide (colcache,
				    * ((Opcode::Matrix4x4*) &ms0.m_Matrix),
				    * ((Opcode::Matrix4x4*) &ms1.m_Matrix));

	    if (m_TreeCollider.GetContactStatus())
	    {
	       Pair *pairs = m_TreeCollider.GetPairs();

	       // Only the first collision interests us...
	       int idtri0 = pairs->id0, mat0;
	       int idtri1 = pairs->id1, mat1;

	       // Fill the collision structure
	       CD_GetTriangle (idtri0,
			       pair.m_Triangle0.m_Coords, &mat0,
			       cm0->m_SubModels[i]);
	       CD_GetTriangle (idtri1,
			       pair.m_Triangle1.m_Coords, &mat1,
			       cm1->m_SubModels[j]);

	       
	       for (size_t k = 0; k < 3; ++k)
	       {
		  pair.m_Triangle0.m_Coords[k] = ms0.m_Matrix.Transform
		     (pair.m_Triangle0.m_Coords[k]);
		  pair.m_Triangle1.m_Coords[k] = ms1.m_Matrix.Transform
		     (pair.m_Triangle1.m_Coords[k]);
	       }

	       pair.m_Triangle0.m_Material = ms0.m_Skin->GetMaterial(mat0);
	       pair.m_Triangle1.m_Material = ms1.m_Skin->GetMaterial(mat1);
	       if (cm0->m_Model == m0)
		  pair.m_BodyPart0 = m0->m_SubModels[i].m_Name;
	       if (cm1->m_Model == m1)
		  pair.m_BodyPart1 = m1->m_SubModels[i].m_Name;

	       return true; 
	    }
	 }
      }

      return false;
   }
   
   /**
    * Returns true if the ray hits the model given as first argument.
    * It also returns the position of the hit in the ray equation.
    */
   bool
   CDSystem::RayTrace (ModelState &m1, const Ray &ray,
		       Collision &col)
   {
      return m_Raytracer.RayTest (m1, ray, col);
   }

   /**
    * Return true if the given bounding box hits the model.
    */
   bool
   CDSystem::BoxTest (ModelState& model, const BBox& box)
   {
     BBox modelBox = model.ExtractBbox();

     // Return true in case of bounding box collision
     // \todo TODO: Test with model data too
     return
       (
	 ( (modelBox.m_Min.X <= box.m_Min.X) && (box.m_Min.X <= modelBox.m_Max.X) ||
	   (box.m_Min.X <= modelBox.m_Min.X) && (modelBox.m_Min.X <= box.m_Max.X) )
	 &&
	 ( (modelBox.m_Min.Y <= box.m_Min.Y) && (box.m_Min.Y <= modelBox.m_Max.Y) ||
	   (box.m_Min.Y <= modelBox.m_Min.Y) && (modelBox.m_Min.Y <= box.m_Max.Y) )
	 &&
	 ( (modelBox.m_Min.Z <= box.m_Min.Z) && (box.m_Min.Z <= modelBox.m_Max.Z) ||
	   (box.m_Min.Z <= modelBox.m_Min.Z) && (modelBox.m_Min.Z <= box.m_Max.Z) )
	);
   }
   

   /** Create a collision model from the given model. */
   ColModel *
   CDSystem::CreateModel (Model *from)
   {
      CDModel *mdl = new CDModel;

      mdl->Build (from);
      return mdl;
   }
};

#define ARK_MODULE
#include <Ark/ArkFactoryReg.h>

class CDSystemFactory : public Ark::ColSystemFactory
{
   public:
      virtual ~CDSystemFactory() {}
      virtual Ark::ColSystem *NewColSystem()
      {
	 return new Ark::CDSystem();
      }
};

ARK_REGISTER("ark::Collision::Opcode",CDSystemFactory);
