/* $Id: ArkLoader.h,v 1.10 2002/10/11 01:10:03 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARK_LOADER_H
#define ARK_LOADER_H

#include <Ark/Ark.h>
#include <Ark/ArkObject.h>
#include <Ark/ArkStream.h>
#include <Ark/ArkFileSys.h>
#include <Ark/ArkSystem.h>

#include <vector>

namespace Ark
{
   
   class ARK_DLL_API Progress
   {
      public:
	 virtual bool Set (const String &name, int percent) = 0;
   };
   
   //  =======================================================================
   /**
    * A loader is a class which have capabilities to load a object from a file.
    * There might be several loaders for the same kind of object (ie. image
    * formats), and a loader might be able to load several kind of objects
    * (ie packed models, containing geometry and textures)
    */
   //  =======================================================================
   
   class ARK_DLL_API Loader
   {
      public:
	 /**
	  *  Destroy the given loader.
	  */
	 virtual ~Loader ();
	 
	 /**
	  * Returns true if the file pointed to by \c name does really
	  * contains a media of the given type, and if this loader can load
	  * it.
	  */
	 virtual bool CanLoad (ObjectType type, Stream &str,
			       const String &name, const String &args) = 0;
	 
	 /**
	  * Load the file pointed to by \c name, and read an object in it.
	  * The read data will be put in the \c vis argument, if the loader
	  * supports its type (ie model, image, etc.).
	  * It will update the \c Progress every \c granularity percents.
	  * \code
	    Image img;
	    Stream file ("test.jpg", "rt");
	    bool result = loader->Load
                     (&img,  file, "?res=256x256", renderer.GetCache());
	  * \endcode
 	  */
	 virtual bool Load (Object *vis, Stream &str, 
		 const String &name,
		 const String &args,
			    Cache *cache = NULL,
			    Progress *progress = NULL,
			    int granularity = 0) = 0;
	 
	 /// Returns informations about the formats supported by this loader.
	  virtual String GetInformations() = 0;
   };
   
   //  ========================================================================
   /** A loader list contains an array of loader (!).  Its role is to maintain
    * a list of loaders and to find the right one when it is asked to load a
    * file.
    */
   //  ========================================================================
   class ARK_DLL_API Loaders
   {
      protected:
	  typedef std::vector< Loader* > LoaderList;
	  typedef LoaderList::iterator LoaderLI;

	  LoaderList m_Loaders;
	  
      public:
	 /** Initialize a loader list */
	 Loaders();
	 
	 /**
	  * This destructor clears the loader list, but it does not destroy
	  * each single loader. That might create a memory leak. As typically
	  * loaders have the same lifetime as the application that is not
	  * really important
	  */
	 ~Loaders();
	 
	 /**
	  * Find the first loader capable of loading the file pointed to by
	  * \c name; It will update the progress every \c granularity
	  * percents.
	  */
	 bool Load (Object *vis, const String &name,
		    Cache *cache = NULL,
		    Progress *progress = NULL, int granularity = 0);
	 
	 /**
	  * Add a loader to the list of file loaders. The loader should be
	  * available until the application is quitted, or it should Remove()
	  * itself when it is not available any more (ie if it in a shared 
	  * library).
	  */
	 bool Add (Loader *loader);
	 
	 /**
	  * Remove a loader from the loader list. The loader has to be
	  * destroyed  after that, if it isn't necessary any more.
	  */
	 bool Remove (Loader *loader);
   };
   
// namespace Ark 
};

#endif
