#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""Tests the PackageKit compatibilty layer."""

import logging
import os.path
import shutil
import subprocess
import tempfile
import time
import sys
import unittest

from gi.repository import GLib
from gi.repository import PackageKitGlib as pk

import aptdaemon.test

REPO_PATH = os.path.join(aptdaemon.test.get_tests_dir(), "repo")
DEBUG = True


class PackageKitTest(aptdaemon.test.AptDaemonTestCase):

    """Test the PackageKit compatibilty layer."""

    def setUp(self):
        """Setup a chroot, run the aptdaemon and a fake PolicyKit daemon."""
        # Setup chroot
        self.chroot = aptdaemon.test.Chroot()
        self.chroot.setup()
        self.addCleanup(self.chroot.remove)
        self.chroot.add_test_repository()
        # set up scratch dir
        self.workdir = tempfile.mkdtemp()
        # allow tests to add plugins, etc.
        self.orig_pythonpath = os.environ.get("PYTHONPATH")
        os.environ["PYTHONPATH"] = "%s:%s" % (self.workdir, os.environ.get("PYTHONPATH", ""))
        self.start_session_aptd(self.chroot.path)
        # Start the fake PolikcyKit daemon
        self.start_fake_polkitd()
        time.sleep(0.5)

    def tearDown(self):
        shutil.rmtree(self.workdir)
        if self.orig_pythonpath:
            os.environ["PYTHONPATH"] = self.orig_pythonpath

    def test_install(self):
        """Test installing a package."""
        self.chroot.add_test_repository()
        pkg_name = "silly-depend-base"

        client = pk.Client()

        # Resolve the id of the package
        res = client.resolve(pk.FilterEnum.NONE, [pkg_name], None,
                             lambda p, t, d: True, None)
        self.assertEqual(res.get_exit_code(), pk.ExitEnum.SUCCESS)
        ids = []
        for pkg in res.get_package_array():
            self.assertEqual(pkg.get_name(), pkg_name)
            ids.append(pkg.get_id())
            break
        else:
            self.fail("Failed to resolve %s" % pkg_name)

        # Simulate
        res = client.simulate_install_packages(ids, None,
                                               lambda p, t, d: True, None)
        self.assertEqual(res.get_exit_code(), pk.ExitEnum.SUCCESS)
        for pkg in res.get_package_array():
            self.assertEqual(pkg.get_name(), "silly-base")
            self.assertEqual(pkg.get_info(),
                             pk.info_enum_from_string("installing"))
            break
        else:
            self.fail("Failed to get dependencies of %s" % pkg_name)

        # Install
        res = client.install_packages(True, ids, None,
                                      lambda p, t, d: True, None)
        self.assertEqual(res.get_exit_code(), pk.ExitEnum.SUCCESS)

    def test_download(self):
        """Test downloading packages."""
        self.chroot.add_test_repository()
        pkg_filename = "silly-base_0.1-0update1_all.deb"
        pkg_id = "silly-base;0.1-0update1;all;"
        temp_dir = tempfile.mkdtemp(prefix="aptd-download-test-")

        client = pk.Client()
        res = client.download_packages([pkg_id], temp_dir,
                                       None, lambda p, t, d: True, None)
        self.assertEqual(res.get_exit_code(), pk.ExitEnum.SUCCESS)
        if not os.path.exists(os.path.join(temp_dir, pkg_filename)):
            self.fail("Failed to download the package")

        shutil.rmtree(temp_dir)

    def test_get_updates(self):
        """Test getting updates."""
        self.chroot.add_test_repository()
        pkg = "silly-base_0.1-0_all.deb"
        self.chroot.install_debfile(os.path.join(REPO_PATH, pkg), True)

        client = pk.Client()

        res = client.get_updates(pk.FilterEnum.NONE, None,
                                 lambda p, t, d: True, None)
        for pkg in res.get_package_array():
            self.assertEqual(pkg.get_name(), "silly-base")
            self.assertEqual(pkg.get_version(), "0.1-0update1")
            self.assertEqual(pkg.get_info(),
                             pk.info_enum_from_string("normal"))
            break
        else:
            self.fail("Failed to detect upgrade")
        self.assertEqual(res.get_exit_code(), pk.ExitEnum.SUCCESS)

    def test_get_updates_security(self):
        """Test if security updates are detected correctly."""
        self.chroot.add_test_repository()
        self.chroot.add_repository(os.path.join(aptdaemon.test.get_tests_dir(),
                                                "repo/security"))
        pkg = "silly-base_0.1-0_all.deb"
        self.chroot.install_debfile(os.path.join(REPO_PATH, pkg), True)

        client = pk.Client()

        res = client.get_updates(pk.FilterEnum.NONE, None,
                                 lambda p, t, d: True, None)
        for pkg in res.get_package_array():
            self.assertEqual(pkg.get_name(), "silly-base")
            self.assertEqual(pkg.get_version(), "0.1-0update1")
            self.assertEqual(pkg.get_info(),
                             pk.info_enum_from_string("security"))
            break
        else:
            self.fail("Failed to detect upgrade")
        self.assertEqual(res.get_exit_code(), pk.ExitEnum.SUCCESS)

    def test_get_updates_backports(self):
        """Test if backports are detected correctly."""
        self.chroot.add_repository(os.path.join(aptdaemon.test.get_tests_dir(),
                                                "repo/backports"))
        pkg = "silly-base_0.1-0_all.deb"
        self.chroot.install_debfile(os.path.join(REPO_PATH, pkg), True)

        client = pk.Client()

        res = client.get_updates(pk.FilterEnum.NONE, None,
                                 lambda p, t, d: True, None)
        for pkg in res.get_package_array():
            self.assertEqual(pkg.get_name(), "silly-base")
            self.assertEqual(pkg.get_version(), "0.1-0update1")
            self.assertEqual(pkg.get_info(),
                             pk.info_enum_from_string("enhancement"))
            break
        else:
            self.fail("Failed to detect upgrade")
        self.assertEqual(res.get_exit_code(), pk.ExitEnum.SUCCESS)


    def test_dependencies(self):
        """Test getting dependencies and dependants."""
        pkg_id_depend = "silly-depend-base;0.1-0;all;"
        pkg_id = "silly-base;0.1-0update1;all;"

        client = pk.Client()

        # Get depends
        res = client.get_depends(pk.FilterEnum.NONE, [pkg_id_depend], True,
                                 None, lambda p, t, d: True, None)
        self.assertEqual(res.get_exit_code(), pk.ExitEnum.SUCCESS)
        for pkg in res.get_package_array():
            self.assertEqual(pkg.get_id(), pkg_id)
            break
        else:
            self.fail("Failed to get dependencies of %s" % pkg_id_depend)

        # Get requires
        res = client.get_requires(pk.FilterEnum.NONE, [pkg_id], True,
                                  None, lambda p, t, d: True, None)
        self.assertEqual(res.get_exit_code(), pk.ExitEnum.SUCCESS)
        for pkg in res.get_package_array():
            self.assertEqual(pkg.get_id(), pkg_id_depend)
            break
        else:
            self.fail("Failed to get dependants of %s" % pkg_id)

    def test_what_provides_unsupported(self):
        """Test querying for provides for unsupported type."""

        client = pk.Client()

        try:
            client.what_provides(pk.FilterEnum.NONE, pk.ProvidesEnum.CODEC, 
                                 ["gstreamer0.10(decoder-audio/ac3)"],
                                 None, lambda p, t, d: True, None)
            self.fail("expected GLib.Error failure")
        except GLib.GError as e:
            self.assertTrue("codec" in str(e), e)
            self.assertTrue("not supported" in str(e), e)

    def test_what_provides_plugin(self):
        """Test querying for provides with plugins."""

        # add plugin for extra codecs
        f = open(os.path.join(self.workdir, "extra_codecs.py"), "w")
        f.write("""from packagekit import enums

def fake_what_provides(cache, type, search):
    if type in (enums.PROVIDES_CODEC, enums.PROVIDES_ANY):
        if search.startswith('gstreamer'):
            return [cache["gstreamer0.10-silly"]]
    raise NotImplementedError('cannot handle type ' + str(type))
""")
        f.close()
        os.mkdir(os.path.join(self.workdir, "extra_codecs-0.egg-info"))
        f = open(os.path.join(self.workdir, "extra_codecs-0.egg-info", 'entry_points.txt'), "w")
        f.write("[packagekit.apt.plugins]\nwhat_provides=extra_codecs:fake_what_provides\n")
        f.close()

        # invalid plugin, should not stop the valid ones
        os.mkdir(os.path.join(self.workdir, "nonexisting-1.egg-info"))
        f = open(os.path.join(self.workdir, "nonexisting-1.egg-info", 'entry_points.txt'), "w")
        f.write("[packagekit.apt.plugins]\nwhat_provides=nonexisting:what_provides\n")
        f.close()

        # another plugin to test chaining and a new type
        f = open(os.path.join(self.workdir, "more_stuff.py"), "w")
        f.write("""from packagekit import enums

def my_what_provides(cache, type, search):
    if type in (enums.PROVIDES_CODEC, enums.PROVIDES_ANY):
        if search.startswith('gstreamer'):
            return [cache["silly-base"]]
    if type in (enums.PROVIDES_LANGUAGE_SUPPORT, enums.PROVIDES_ANY):
        if search.startswith('locale('):
            return [cache["silly-important"]]
    raise NotImplementedError('cannot handle type ' + str(type))
""")
        f.close()
        os.mkdir(os.path.join(self.workdir, "more_stuff-0.egg-info"))
        f = open(os.path.join(self.workdir, "more_stuff-0.egg-info", 'entry_points.txt'), "w")
        f.write("[packagekit.apt.plugins]\nwhat_provides=more_stuff:my_what_provides\n")
        f.close()

        client = pk.Client()

        # search for CODEC
        res = client.what_provides(pk.FilterEnum.NONE, pk.ProvidesEnum.CODEC, 
                                   ["gstreamer0.10(decoder-audio/vorbis)"],
                                   None, lambda p, t, d: True, None)
        self.assertEqual(res.get_exit_code(), pk.ExitEnum.SUCCESS)
        pkgs = [p.get_id().split(";")[0] for p in res.get_package_array()]
        self.assertEqual(pkgs, ["gstreamer0.10-silly", "silly-base"])

        # search for LANGUAGE_SUPPORT
        res = client.what_provides(pk.FilterEnum.NONE, pk.ProvidesEnum.LANGUAGE_SUPPORT, 
                                   ["locale(de_DE)"],
                                   None, lambda p, t, d: True, None)
        self.assertEqual(res.get_exit_code(), pk.ExitEnum.SUCCESS)
        pkgs = [p.get_id().split(";")[0] for p in res.get_package_array()]
        self.assertEqual(pkgs, ["silly-important"])

        # search ANY
        res = client.what_provides(pk.FilterEnum.NONE, pk.ProvidesEnum.ANY, 
                                   ["gstreamer0.10(decoder-audio/vorbis)"],
                                   None, lambda p, t, d: True, None)
        self.assertEqual(res.get_exit_code(), pk.ExitEnum.SUCCESS)
        pkgs = [p.get_id().split(";")[0] for p in res.get_package_array()]
        self.assertEqual(pkgs, ["gstreamer0.10-silly", "silly-base"])

        res = client.what_provides(pk.FilterEnum.NONE, pk.ProvidesEnum.ANY, 
                                   ["locale(de_DE)"],
                                   None, lambda p, t, d: True, None)
        self.assertEqual(res.get_exit_code(), pk.ExitEnum.SUCCESS)
        pkgs = [p.get_id().split(";")[0] for p in res.get_package_array()]
        self.assertEqual(pkgs, ["silly-important"])

        res = client.what_provides(pk.FilterEnum.NONE, pk.ProvidesEnum.ANY, 
                                   ["modalias(pci:1)"],
                                   None, lambda p, t, d: True, None)
        self.assertEqual(res.get_exit_code(), pk.ExitEnum.SUCCESS)
        self.assertEqual(res.get_package_array(), [])

        # unsupported type with plugins
        try:
            client.what_provides(pk.FilterEnum.NONE, pk.ProvidesEnum.PLASMA_SERVICE, 
                                 ["plasma4(dataengine-weather)"],
                                 None, lambda p, t, d: True, None)
            self.fail("expected GLib.Error failure")
        except GLib.GError as e:
            self.assertTrue("plasma" in str(e), e)
            self.assertTrue("not supported" in str(e), e)

def start_dbus_daemon():
    """Start a dbus system daemon.

    The PackageKit gir client doesn't support a changing dbus-server address
    after initialisation, so we need to keep a persistent dbus daemon for
    the whole test suite.
    """
    config_path = os.path.join(aptdaemon.test.get_tests_dir(), "dbus.conf")
    proc = subprocess.Popen(["dbus-daemon", "--nofork", "--print-address",
                             "--config-file", config_path],
                            stdout=subprocess.PIPE)
    address = proc.stdout.readline().strip()
    os.environ["DBUS_SYSTEM_BUS_ADDRESS"] = address
    return proc

def setup():
    proc = start_dbus_daemon()
    global DBUS_PID
    DBUS_PID = proc.pid

def tearDown():
    os.kill(DBUS_PID, 15)

if __name__ == "__main__":
    if DEBUG:
        logging.basicConfig(level=logging.DEBUG)
    dbus_proc = start_dbus_daemon()
    try:
        unittest.main()
    finally:
        dbus_proc.kill()
        dbus_proc.wait()

# vim: ts=4 et sts=4
