
#define LOCAL_DEBUG
#include "debug.h"

#include "config.h"
#include "acfg.h"
#include "dlcon.h"

#include "fileitem.h"
#include "caddrinfo.h"
#include "sockio.h"

struct tDlJob
{
	UINT m_nSourceId;
	MYSTD::string m_sHost, m_sPath;
	acfg::tHostiVec * m_pHostiVec;
	tFileItemPtr m_pStorage;
	
	tDlJob(tFileItemPtr pFi, const string & sHost,
			const string & sPath) :
				m_nSourceId(0), // don't care
				m_sHost(sHost), m_sPath(sPath), m_pHostiVec(NULL),
				m_pStorage(pFi)
	{
	}

	tDlJob(tFileItemPtr pFi, acfg::tHostiVec * pBackends,
			const MYSTD::string & sPath) :
				m_nSourceId(-1), // becomes 0 RSN...
				m_sHost(""), m_sPath(sPath), m_pHostiVec(pBackends),
				m_pStorage(pFi)
	{
		ChangePeer();
	}
	
	bool ChangePeer()
	{
		if(!m_pHostiVec)
			return false;
		
		tStrPos nOldPrefixLen(0);
		if(m_nSourceId!=(UINT)-1)
			nOldPrefixLen=m_pHostiVec->at(m_nSourceId).sPath.length();
		
		m_nSourceId++;
		if(m_nSourceId >= m_pHostiVec->size())
			return false;
		m_sHost=m_pHostiVec->at(m_nSourceId).sHost;
		m_sPath.replace(0, nOldPrefixLen, m_pHostiVec->at(m_nSourceId).sPath);
		//cout << "got host: " << m_sHost << " und path: " << m_sPath << " aus " << m_pHostiVec->at(m_nSourceId).sPath<<endl;
		return true;
	}

private:
	// not to be copoied
	tDlJob(const tDlJob&);
	tDlJob & operator=(const tDlJob&);
};

dlcon::dlcon():
m_conFd(-1),
m_bIsResponsibleDler(false),
m_bReconnectASAP(false),
m_bOrphaned(false),
m_nRest(0),
m_DlState(STATE_GETHEADER),
m_proxy(NULL)
{
	ldbg("Creating dlcon");
	m_wakepipe[0]=m_wakepipe[1]=-1;
	m_proxy=acfg::proxy_info.sHost.empty() ? NULL : &acfg::proxy_info;
}


#if 0
void dlcon::Reset()
{
	/*
	if(m_conFd>=0)
		close(m_conFd);
	
	m_conFd=-1;
	*/
	m_bOrphaned=false;
	m_bIsAttached2Fitem=false;
	m_bReconnectASAP=false;
	m_nRest=0;
	m_DlState=STATE_GETHEADER;
	//m_pCachedFitem.reset(NULL);
}
#endif

inline void dlcon::_AddJob(tDlJob* todo)
{
	setLockGuard;
	m_qToReceive.push_back(todo);
	
	MakeRequest(todo);
	if (m_wakepipe[1]>=0)
		POKE(m_wakepipe[1]);
	
	//lockguard g(todo->m_pStorage.get());
	if(todo->m_pStorage->status < FIST_DLPENDING)
    	todo->m_pStorage->status=FIST_DLPENDING;

}
void dlcon::AddJob(tFileItemPtr m_pItem, 
		acfg::tHostiVec *pBackends, const MYSTD::string & sPatSuffix)
{
	lockguard g(m_pItem.get());
	_AddJob(new tDlJob(m_pItem, pBackends, sPatSuffix));	
}
void dlcon::AddJob(tFileItemPtr m_pItem, tHttpUrl hi)
{
	lockguard g(m_pItem.get());
	_AddJob(new tDlJob(m_pItem, hi.sHost, hi.sPath));
}

void dlcon::SignalStop()
{
	setLockGuard;
	m_bOrphaned=true;
	POKE(m_wakepipe[1]);
}

dlcon::~dlcon()
{
	ldbg("Destroying dlcon");
	if (m_conFd>=0)
	{
		shutdown(m_conFd, O_RDWR);
		close(m_conFd);
	}
	
	if (m_wakepipe[0]>=0)
		close(m_wakepipe[0]);

	if (m_wakepipe[1]>=0)
		close(m_wakepipe[1]);
		
	list<tDlJob*>::iterator it=m_qToReceive.begin();
	for(;it!=m_qToReceive.end();it++)
		delete *it;
	
	ldbg("destroying dlcon done. Any postponed tasks to do? ");
}

// PREREQUISITES: current host is failed host and is first, refered by m_recvPos
void dlcon::_CleanupForFailedHost(const string & sErrorMsg)
{
	// kick all jobs that belong that hostname, adapt the position marks
	setLockGuard;
	
	if(!m_qToReceive.empty())
	{
		if (m_bIsResponsibleDler)
			m_qToReceive.front()->m_pStorage->SetFailureMode(sErrorMsg);
			
		string sBadHost=m_qToReceive.front()->m_sHost;
	
		list<tDlJob*> tmp;
		
		for (list<tDlJob*>::iterator it = m_qToReceive.begin(); 
		it!=m_qToReceive.end() ; it++)
		{
			if ((*it)->m_sHost == sBadHost)
			{
				(*it)->m_pStorage->SetFailureMode(sErrorMsg);
				delete *it;
			}
			else
			{
				ldbg("host? " << sBadHost << " Kept job: " << (*it)->m_sHost << "/" << (*it)->m_sPath);
				tmp.push_back(*it);
			}
		}
		
		m_qToReceive.swap(tmp);
	}
	

	m_bIsResponsibleDler=false;
	ldbg("cleanup one, force reconnection, m_sConnectedHost");
	m_sConnectedHost.clear(); // will trigger the _Connect call 
}


/*!
 * 
 * Process incoming traffic and write it down to disk/downloaders.
 * 
 * @return: 
 *  - true: ok
 *  - false: not ok, reconnect if sErrorOut is empty, threat as fatal error otherwise 
 */
bool dlcon::_HandleIncoming(string & sErrorOut)
{

	#define THROW_INC_ERROR(x) { sErrorOut=x; return false; }

	// closed connection (null-read) or some other failure
	int r = m_InBuf.sysread(m_conFd);
	ldbg("dlcon::_HandleIncoming()~sysread: " <<r);
	if (r <= 0)
	{
		if(r==-EAGAIN /* retries done there || r==-EINTR */ )
			return true;
		else
			return false;
	}

	Reswitch:
	ldbg("switch: " << m_DlState);
	
	
	switch (m_DlState)
	{

	case (STATE_GETHEADER):
	{
		ldbg("STATE_GETHEADER");
		header h;
		if(m_InBuf.size()==0)
			return true;  // will come back
		dbgline;
		int l=h.LoadFromBuf(m_InBuf.rptr(), m_InBuf.size());
		if (0==l)
			return true; // will come back
		dbgline;
		if (0>l)		
		{
			THROW_INC_ERROR("500 Invalid header");
		}
		else if(h.type!=header::ANSWER)
		{
			THROW_INC_ERROR("500 Unexpected response type");
		}
		
		ldbg("GOT, parsed: " << h.frontLine);

		m_InBuf.drop(l);

		const char *p=h.h[header::CONNECTION] ? h.h[header::CONNECTION] : h.h[header::PROXY_CONNECTION];  
		if(p && 0==strcasecmp(p, "close") )
		{
			ldbg("Peer wants to close connection after request");
			m_bReconnectASAP=true;
		}
		p=h.h[header::TRANSFER_ENCODING];
		if (p && 0==strcasecmp(p, "chunked"))
			m_DlState=STATE_GETCHUNKHEAD;
		else
		{
			dbgline;
			p=h.h[header::CONTENT_LENGTH];
			if (!p) THROW_INC_ERROR("500 Missing Content-Length");
			// may support such endless stuff in the future but that's too unreliable for now
			m_nRest=atol(p);
			m_DlState=STATE_GETDATA;
		}
		
		{
			setLockGuard;
			if(m_qToReceive.empty()) THROW_INC_ERROR("Peer sends unexpected data");
			
			h.set(header::XORIG,
					string("http://", 7)
					+m_qToReceive.front()->m_sHost
					+m_qToReceive.front()->m_sPath);
			
			if ( ! m_qToReceive.front()->m_pStorage->StoreHeader(h))
			{
				ldbg("Item dl'ed by others or in error state --> drop it, reconnect");
				delete m_qToReceive.front();
				m_qToReceive.pop_front();
				// _Reconnect will follow and also regenerate requests
				return false;
			}
			m_bIsResponsibleDler=true;
			ldbg("Claimed: " << m_qToReceive.front()->m_pStorage->m_sKey);
		}

		goto Reswitch;

	}
	case (STATE_GETDATA_CHUNKED): // fall through, just send it back to header parser hereafter
		ldbg("STATE_GETDATA_CHUNKED (to STATE_GETDATA)");
	case (STATE_GETDATA):
	{
		ldbg("STATE_GETDATA");
		
		while(true)
		{
			off_t nToStore = min((off_t)m_InBuf.size(), m_nRest);
			ldbg("To store: " <<nToStore);
			if(0==nToStore)
				break;

			if(!m_qToReceive.front()->m_pStorage->StoreFileData(m_InBuf.rptr(),
					nToStore))
			{
				setLockGuard;
				delete m_qToReceive.front();
				m_qToReceive.pop_front();
				// _Connect will do the rest
				return false;
			}
			m_nRest-=nToStore;
			m_InBuf.drop(nToStore);
		}
			
		ldbg("Rest: " << m_nRest );
		
		if (m_nRest==0)
		{
			m_DlState = (STATE_GETDATA==m_DlState) ? STATE_FINISHJOB : STATE_GETCHUNKHEAD;
		}
		else
			return true; // will come back

		goto Reswitch;
	}
	case (STATE_FINISHJOB):
	{
		ldbg("STATE_FINISHJOB");
		m_DlState=STATE_GETHEADER;
		m_qToReceive.front()->m_pStorage->StoreFileData(NULL, 0);
		
		{
			setLockGuard;
			m_bIsResponsibleDler=false;
			delete m_qToReceive.front();
			m_qToReceive.pop_front();
			if(m_bReconnectASAP)
				return false; // reconnect, don't risk to hang here
		}
		
		goto Reswitch;
	}
	case (STATE_GETCHUNKHEAD):
	{
		ldbg("STATE_GETCHUNKHEAD");
		char *p=m_InBuf.c_str();
		char *e=strstr(p, "\r\n");
		if(e==p)
		{ // came back from reading, drop remaining junk? 
			m_InBuf.drop(2);
			p+=2;
			e=strstr(p, "\r\n");
		}
		dbgline;
		if(!e)
		{
			m_InBuf.move();
			return true; // get more data
		}
		unsigned int len(0);
		int n = sscanf(p, "%x", &len); 
		
		long nCheadSize=e-p+2;
		if(n==1 && len>0)
		{
			ldbg("ok, skip " << nCheadSize <<" bytes, " <<p);
			m_InBuf.drop(nCheadSize);
			m_nRest=len;
			m_DlState=STATE_GETDATA_CHUNKED;
		}
		else if(n==1)
		{
			// skip the additional \r\n of the null-sized part here as well
			ldbg("looks like the end, but needs to get everything into buffer to change the state reliably");
			if( m_InBuf.size() < nCheadSize+2 )
			{
				m_InBuf.move();
				return true;
			}
			if( ! (e[2]=='\r' && e[3]=='\n'))
			{
				aclog::err(m_qToReceive.front()->m_pStorage->m_sKey+" -- error in chunk format detected");
				return false;
			}
			
			m_InBuf.drop(nCheadSize+2);
			m_DlState=STATE_FINISHJOB;
		}
		else
			return false; // that's bad...
		goto Reswitch;
		break;
	}
	
	}
	return true;
}
		
void dlcon::MakeRequest(tDlJob* j) {
	
	if(m_sReqBufForHost.empty()) // req.queue not send before, mark as ours
		m_sReqBufForHost=j->m_sHost;
	else if(m_sReqBufForHost!=j->m_sHost)
	{
		ldbg("Request for a different host, will be placed after reconnecting");
		return;
	}
	
	string head("GET ");
	if (m_proxy)
	{
		head+="http://";
		head+=j->m_sHost;
	}
	head+= j->m_sPath +" HTTP/1.1\r\nHost: ";
	head+= j->m_sHost+"\r\n";//Accept: */*\r\n";
	head+= "Connection: keep-alive\r\n";
	if (m_proxy) // add auth (in sPath) if possible and other stuff
		head+=m_proxy->sPath+"Proxy-Connection: keep-alive\r\n";

	if (j->m_pStorage->m_nSizeSeen > 0)
	{
		bool bSetRange(false), bSetIfRange(false);
		
		//lockguard g(*(j->m_pStorage));
		const header *pHead = j->m_pStorage->GetHeader();
		const char *p=pHead->h[header::LAST_MODIFIED];
		
		if (j->m_pStorage->m_bCheckFreshness)
		{
			if (p)
			{
				bSetIfRange=true;
				bSetRange=true;
			}
			// else no date available, cannot rely on anything
		}
		else
		{ 
			/////////////// this was protection against broken stuff in the pool ////
			// static file type, date does not matter. check known content length, not risking "range not satisfiable" result
			//
			//off_t nContLen=atol(h.get("Content-Length"));
			//if (nContLen>0 && j->m_pStorage->m_nFileSize < nContLen)
			bSetRange=true;
		}
		
		if (bSetRange)
		{
			/* use APT's trick - set the starting position one byte lower - this way the server has to
			 * send at least one byte if the assumed position is correct, and we never get a
			 * 416 error (one byte waste is acceptable). */
			char buf[50];
			sprintf(buf, "Range: bytes=%ld-\r\n", j->m_pStorage->m_nSizeSeen - 1 );
			head+=buf;
		}
		if (bSetIfRange)
		{
			head += "If-Range: ";
			head += p;
			head += "\r\n";
		}
	}
	// Debian Apt-Cacher/" ACVERSION 
	head+="User-Agent: " + acfg::agentname + "\r\n\r\n";
	m_sSendBuf+=head;
	ldbg("Request cooked: " << head);
	//cerr << "To request: " << j->m_sPath<<endl; 
}

#define HANDLEERROR(x) { m_sErrorMsgBuf=x ; goto ERROR_SEE_BUF; } 
void dlcon::WorkLoop(bool bSingleRun)
{

	fd_set rfds, wfds;

	if (m_wakepipe[0]<0)
	{
		if (0==pipe(m_wakepipe))
		{
			set_nb(m_wakepipe[0]);
			set_nb(m_wakepipe[1]);
		}
		else
		{
			m_sErrorMsgBuf="500 Unable to create file descriptors";
			return;
		}
	}

	if (!m_InBuf.init(acfg::dlbufsize))
		m_sErrorMsgBuf="500 Out of memory";

	while (true)
	{
		// some flags temporarily stored to minimize locking
		
		string sReconnHost; // if non-empty -> reconnect ASAP, otherwise ignore
		bool bExpectIncomming(false), bExpectOutgoing(false)/*, bGenRequests(false)*/;
		int maxfd, r;

		if (!m_sErrorMsgBuf.empty())
		{
			goto ERROR_SEE_BUF;
		}

		{ // critical decissions about (re)connection control, loop exit 
			
			setLockGuard;
			// detect when all users of the assigned item are gone
			if(m_bIsResponsibleDler && !m_qToReceive.empty() &&
					// only possible when fitem was deregistred in the pool before
					m_qToReceive.front()->m_pStorage->status >= FIST_ERRNOUSER)
			{
				m_sConnectedHost=""; // force reconnection and maybe exit
			}
			
			if(m_qToReceive.empty())
			{
				if(m_bOrphaned || bSingleRun)
					return;
			}
			else
			{
				bExpectIncomming=true;
				if (m_qToReceive.front()->m_sHost != m_sConnectedHost)
					sReconnHost=m_qToReceive.front()->m_sHost;
			}
			bExpectOutgoing=!m_sSendBuf.empty();
		}

		if ( ! sReconnHost.empty())
		{
			if( ! _Connect(sReconnHost) ) // sets error msg on failures
			{
				ldbg("Connection failure, m_sConnectedHost");
				m_sConnectedHost.clear(); // be sure about that
				
				lockguard g(&__mutex);	
				if(m_qToReceive.front()->ChangePeer())
					m_sErrorMsgBuf.clear(); // there is another chance, forget the failure
				
				continue;
			}
			if (!bExpectOutgoing)
			{
				// has reconnected, should have sth. to send now
				setLockGuard;
				if(m_sSendBuf.empty())
				{
					if(m_bOrphaned || bSingleRun)
						return;
				}
				else bExpectOutgoing=true;
			}
		}

		FD_ZERO(&rfds);
		FD_ZERO(&wfds);
		if (m_conFd>=0)
		{
			if (bExpectIncomming)
				FD_SET(m_conFd, &rfds);
			if(bExpectOutgoing)
				FD_SET(m_conFd, &wfds);
		}

		FD_SET(m_wakepipe[0], &rfds);
		maxfd=MYSTD::max(m_wakepipe[0], m_conFd);

		ldbg("select dlcon");
		struct timeval tv;
		tv.tv_sec = 60;
		tv.tv_usec = 0;
		r=select(maxfd+1, &rfds, &wfds, NULL, &tv);
		if (r<0)
		{
			if (EINTR == errno)
				continue;
			ldbg("FAILURE: select, errno: " << errno);
			HANDLEERROR("500 Internal malfunction, code 100");
		}
		else if(r==0)
		{
			ldbg("Select timeout for " << m_sConnectedHost << ". Trigger reconnection..., m_sConnectedHost.clear()");
			m_sConnectedHost.clear();
			continue;
		}

		if (FD_ISSET(m_wakepipe[0], &rfds))
		{
			int tmp;
			while (read(m_wakepipe[0], &tmp, 1) > 0)
				;
			
			// woken up -> recheck from the start, read stuff must come first to do timeout detection
			continue;
			
		}

		if (m_conFd<0)
		{
			ldbg("connection FD not open");
			continue;
		}

		// now do the real IO

		// do read first - might reconnect on failed reading

		if (FD_ISSET(m_conFd, &rfds))
		{
			ldbg("Receiving data...");
			if (_HandleIncoming(m_sErrorMsgBuf) )
			{
				ldbg("data received, OK");
			}
			else if(m_sErrorMsgBuf.empty())
			{
				ldbg("Non-fatal return (timeout?). Trigger reconnection..., m_sConnectedHost unset!");
				m_sConnectedHost.clear();
				continue;
			}
			else
			{  // bad things, any backup servers?
				lockguard g(&__mutex);
				if( !m_qToReceive.empty() && m_qToReceive.front()->ChangePeer())
				{
					ldbg("Peer address changed, force reconnection, m_sConnectedHost");
					m_sConnectedHost.clear();
					continue;
				}
				goto ERROR_SEE_BUF;
			}
		}

		if (FD_ISSET(m_conFd, &wfds))
		{
			ldbg("Sending data...\n" << m_sSendBuf);
			int s=::send(m_conFd, m_sSendBuf.data(), m_sSendBuf.length(), MSG_NOSIGNAL);
			if (s<0)
			{
				if(errno==EAGAIN || errno==EINTR)
					s=0;
				else
					HANDLEERROR("500 Peer communication, code 101");
			}
			m_sSendBuf.erase(0, s);
		}

		continue;

		ERROR_SEE_BUF:
		// fatal error for that server
		ldbg("Got exception: "<<m_sErrorMsgBuf);
		_CleanupForFailedHost(m_sErrorMsgBuf.c_str()); // ... and drop the fitems for the dead server
		m_sErrorMsgBuf.clear();

	}
}

bool dlcon::_Connect(const string & sHostname)
{
	CAddrInfoPtr dns = m_proxy ? 
			_Resolve(m_proxy->sHost, m_proxy->sPort, m_sErrorMsgBuf) 
			: _Resolve(sHostname, acfg::remoteport, m_sErrorMsgBuf);
	if(!dns)
		return false;
	
	struct addrinfo *pInfo(dns->m_bestInfo);
	
	signal(SIGPIPE, SIG_IGN);
	if (m_conFd>=0)
	{
		ldbg("Socket was connected before, shutting down, m_sConnectedHost");
		shutdown(m_conFd, O_RDWR);
		close(m_conFd);
		m_sConnectedHost.clear();
	}
	ldbg("Creating socket...");
	m_conFd = socket(pInfo->ai_family, pInfo->ai_socktype, pInfo->ai_protocol);
	
#ifndef NO_TCP_TUNNING
	int dummy(1);
    setsockopt(m_conFd,SOL_SOCKET,SO_REUSEADDR,&dummy,sizeof(dummy));
    setsockopt(m_conFd, IPPROTO_TCP, TCP_NODELAY, &dummy, sizeof(dummy));
#endif
    
	int r=connect(m_conFd, pInfo->ai_addr, pInfo->ai_addrlen);
	if (r<0)
	{
		m_sErrorMsgBuf="500 Connection failure";
		ldbg("m_sConnectedHost: Force reconnect, con. failure");
		m_sConnectedHost.clear();
		return false;
	}
	
	ldbg("connect() ok");
	// ready for polling with select now
	set_nb(m_conFd);
	
	m_sConnectedHost=sHostname;
	m_DlState=STATE_GETHEADER;
	m_InBuf.clear();
	m_sSendBuf.clear();
	
	// prepare request calls and also cleanup the list if caller(s) left in the meantime
	setLockGuard;
	// detect when all users are gone
	if(m_bIsResponsibleDler && !m_qToReceive.empty() &&
			// only possible when fitem was deregistred in the pool before
			m_qToReceive.front()->m_pStorage->status >= FIST_ERRNOUSER)
	{
		delete m_qToReceive.front();
		m_qToReceive.pop_front();
		
		// experimental. Drop all lost file items -- when this one lost its user,
		// many others are likely to follow
		if(0)
		{
			lockguard g(mapLck);
			list<tDlJob*> tmp;
			for (list<tDlJob*>::iterator it=m_qToReceive.begin(); it
					!= m_qToReceive.end(); it++)
			{
				if ((*it)->m_pStorage->status != FIST_ERRNOUSER)
					tmp.push_back(*it);
				else
					delete *it;
			}
			m_qToReceive=tmp;
		}
	}
	m_sReqBufForHost.clear();
	for(list<tDlJob*>::iterator it=m_qToReceive.begin(); it!= m_qToReceive.end(); it++)
	{
		lockguard g((*it)->m_pStorage.get());
		MakeRequest(*it);
	}
		
	return true;
}


