// ---------------------------------------------------------------------------
// - Algebra.hpp                                                             -
// - afnix:mth module - algebraic algorithm definitions                      -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2011 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_ALGEBRA_HPP
#define  AFNIX_ALGEBRA_HPP

#ifndef  AFNIX_RVI_HPP
#include "Rvi.hpp"
#endif

#ifndef  AFNIX_RMI_HPP
#include "Rmi.hpp"
#endif
 
namespace afnix {

  /// The Algebra class is a class that implements various algorithms used
  /// in algebra calculus. This is the default implementation which is often
  /// optmized by underlying class.
  /// @author amaury darsch

  class Algebra {
  public:
    /// copy a vector into another one
    /// @param r the result vector
    /// @param x the vector argument
    static void cpy (Rvi& r, const Rvi& x);

    /// add a vector with a scalar
    /// @param r the result vector
    /// @param x the vector argument
    /// @param s the scalar factor
    static void add (Rvi& r, const Rvi& x, const t_real s);

    /// add a vector with another one
    /// @param r the result vector
    /// @param x the vector argument
    /// @param y the vector argument
    static void add (Rvi& r, const Rvi& x, const Rvi& y);

    /// add a vector with another scaled one
    /// @param r the result vector
    /// @param x the vector argument
    /// @param y the vector argument
    /// @param s the scalar factor
    static void add (Rvi& r, const Rvi& x, const Rvi& y, const t_real s);

    /// substract a vector with a scalar
    /// @param r the result vector
    /// @param x the vector argument
    /// @param s the scalar factor
    static void sub (Rvi& r, const Rvi& x, const t_real s);

    /// substract a vector with another one
    /// @param r the result vector
    /// @param x the vector argument
    /// @param y the vector argument
    static void sub (Rvi& r, const Rvi& x, const Rvi& y);

    /// multiply a vector with a scalar
    /// @param r the result vector
    /// @param x the vector argument
    /// @param s the scalar factor
    static void mul (Rvi& r, const Rvi& x, const t_real s);

    /// multiply a vector with another one
    /// @param r the result vector
    /// @param x the vector argument
    /// @param y the vector argument
    static void mul (Rvi& r, const Rvi& x, const Rvi& y);

    /// add equal with a vector
    /// @param r the result vector
    /// @param x the vector argument
    static void aeq (Rvi& r, const Rvi& x);

    /// add equal with a scaled vector
    /// @param r the result vector
    /// @param x the vector argument
    /// @param s the scalar factor
    static void aeq (Rvi& r, const Rvi& x, const t_real s);

    /// rescale equal with a vector
    /// @param r the result vector
    /// @param x the vector argument
    /// @param s the scalar factor
    static void req (Rvi& r, const Rvi& x, const t_real s);

    /// add a matrix with another one
    /// @param mr the result matrix
    /// @param mx the matrix argument
    /// @param my the matrix argument
    static void add (Rmi& r, const Rmi& x, const Rmi& y);

    /// substract a matrix with another one
    /// @param mr the result matrix
    /// @param mx the matrix argument
    /// @param my the matrix argument
    static void sub (Rmi& r, const Rmi& x, const Rmi& y);

    /// multiply a matrix with a scalar
    /// @param r the result vector
    /// @param m the matrix argument
    /// @param s the scalar factor
    static void mul (Rvi& r, const Rmi& m, const t_real s);

    /// multiply a matrix with a vector and a scaling factor
    /// @param r the result vector
    /// @param m the matrix argument
    /// @param x the vector argument
    /// @param s the scaling factor
    static void mul (Rvi& r, const Rmi& m, const Rvi& x, const t_real s);

    /// multiply two matrices
    /// @param mr the result matrix
    /// @param mx the matrix argument
    /// @param my the matrix argument
    static void mul (Rmi& mr, const Rmi& mx, const Rmi& my);

    /// create a random vector with bounds
    /// @param r    the result vector
    /// @param rmin the minimum value
    /// @param rmax the maximum value
    static void random (Rvi& r, const t_real rmin, const t_real rmax);

    /// create a random matrix with bounds
    /// @param mr   the result matrix
    /// @param ddom the diagonally dominant flag
    /// @param rmin the minimum value
    /// @param rmax the maximum value
    static void random (Rmi& mr, const bool ddom, const t_real rmin, 
			const t_real rmax);
  };
}

#endif
