// -*- C++ -*- (c) 2007,2008 Petr Rockai <me@mornfall.net>

#include <QtCore/QObject>
#include <QtCore/QMap>
#include <QtCore/QSet>
#include <QtCore/QDebug>
#include <QtCore/QTimer>
#include <QtCore/QCoreApplication>

#include <QtGui/QAbstractItemDelegate>
#include <QtGui/QAbstractItemView>
#include <QtGui/QItemDelegate>
#include <QtGui/QLabel>
#include <QtGui/QTreeView>

#include <set>

#include <wibble/test.h>

#ifndef EPT_EXTENDABLELIST_H
#define EPT_EXTENDABLELIST_H

namespace adept {

class ExtendingEditorBuddy;

/**
 * @class ExtendingDelegate
 * Delegate that renders an extendable item.
 */
class ExtendingDelegate : public QItemDelegate {
    Q_OBJECT
private:
    /** Set of items that are currently extended and maps to the
     * editor widgets */
    typedef QSet< QPersistentModelIndex > IndexSet;
    mutable IndexSet m_extended;
    mutable QMap< QPersistentModelIndex, QWidget * > m_editors;
    friend class ExtendingEditorBuddy;

public:
    ExtendingDelegate( QObject *p = 0 ) : QItemDelegate( p ) {}

    IndexSet::iterator extendedBegin() { return m_extended.begin(); }
    IndexSet::iterator extendedEnd() { return m_extended.end(); }

    virtual QSize retractedSizeHint( const QStyleOptionViewItem &opt,
                                     const QModelIndex &idx ) const = 0;

    virtual QSize expandedSizeHint( const QStyleOptionViewItem &opt,
                                    const QModelIndex &idx ) const;

    /** Overrides QAbstractItem::sizeHint(opt,idx) to check if the
     * item is expanded. */
    virtual QSize sizeHint( const QStyleOptionViewItem &opt,
                            const QModelIndex &idx ) const {
        if ( m_extended.contains( idx ) )
            return expandedSizeHint( opt, idx );
        else
            return retractedSizeHint( opt, idx );
    }

    bool extended( const QModelIndex &i ) const {
        return m_extended.contains( i );
    }

    virtual QWidget *createEditorWidget( QWidget *,
                                         const QStyleOptionViewItem &,
                                         const QModelIndex & ) const
    {
        return 0;
    }

    /** Overrides QItemDelegate::createEditor
        Inserts the item into the set of extended items and creates an editor
        widget for it */
    virtual QWidget *createEditor( QWidget *parent,
                                   const QStyleOptionViewItem &opt,
                                   const QModelIndex &idx ) const;

    virtual void updateEditorGeometry (
        QWidget * editor,
        const QStyleOptionViewItem & opt,
        const QModelIndex & idx ) const;

public Q_SLOTS:
    /** Removes the editor and its corresponding item's model index from the 
        extended items set */
    virtual void closeEditor( const QModelIndex &i );
Q_SIGNALS:
    void editorGeometryUpdated() const;
    void updateItem( const QModelIndex & ) const;
};


class ExtendableListView : public QTreeView {
    Q_OBJECT
    friend class ExtendingEditorBuddy;
    
public:
    ExtendableListView( ExtendingDelegate *delegate, QWidget *parent = 0 );
    void closeAllEditors() {
        closeAllEditorsRequested();
    }

    virtual void setModel( QAbstractItemModel *m ) {
        closeAllEditors();
        QTreeView::setModel( m );
    }

Q_SIGNALS:
    void closeAllEditorsRequested();
};


/**
 * @class ExtendingEditorBuddy
 * Helper class to help redrawing of extended delegates in the view.
 *
 * Create one and pass a view and delegate to the constructor.
 * It will tell the view to use the delegate, so
 * you don't call setItemDelegate on the view manually.
 * You don't need to delete the Buddy, since it
 * will be destroyed with the view automatically.
 */
class ExtendingEditorBuddy : public QObject {
    Q_OBJECT
private:
    ExtendableListView *m_view;
    ExtendingDelegate *m_delegate;
    /** List of items that need to be updated */
    QList< QPersistentModelIndex > m_update;
    bool m_updateDone;
    int m_updates;
    QTimer *m_updateTimer;

public:
    ExtendingEditorBuddy( ExtendableListView *v, ExtendingDelegate *d );
    ~ExtendingEditorBuddy() {
        delete m_delegate;
    }

public Q_SLOTS:
    virtual void update( const QModelIndex &m );

protected Q_SLOTS:
    void doUpdate();
    virtual void toggleExtended( const QModelIndex &m );

    void closeAll();
    void updateAll();
};

}

#endif

