!{\src2tex{textfont=tt}}
!!****f* ABINIT/wvl_init_type_wfs
!!
!! NAME
!! wvl_init_type_wfs
!!
!! FUNCTION
!! Allocate and compute the access keys for the wavefuncrions when the positions
!! of the atoms are given. The arrays to store wavefunctions and their gradients
!! are also allocated, use wvl_free_type_wfs() to free them after use.
!!
!! COPYRIGHT
!! Copyright (C) 1998-2007 ABINIT group (DC)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!!  dtset <type(dataset_type)>=internal variables used by wavelets, describing
!!   | wvl_internal=desciption of the wavelet box.
!!   | natom=number of atoms.
!!  mpi_enreg=informations about MPI parallelization
!!  psps <type(pseudopotential_type)>=variables related to pseudopotentials
!!  rprimd(3,3)=dimensional primitive translations in real space (bohr)
!!
!! OUTPUT
!!  wfs <type(wvl_projector_type)>=wavefunctions informations for wavelets.
!!   | keys=its access keys for compact storage.
!!
!! SIDE EFFECTS
!!  xred(3,natom)=reduced dimensionless atomic coordinates (in fact IN but here
!!                because of INOUT xredxcart() behavior).
!!
!! PARENTS
!!      gstate
!!
!! CHILDREN
!!      createwavefunctionsdescriptors,leave_new,wrtout,xredxcart
!!
!! SOURCE
#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine wvl_init_type_wfs(dtset, mpi_enreg, psps, rprimd, wfs, xred)

  use defs_basis
  use defs_datatypes
#if defined HAVE_BIGDFT
  use libbigdft
#endif

!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifdef HAVE_FORTRAN_INTERFACES
 use interfaces_01manage_mpi
 use interfaces_12geometry
#endif
!End of the abilint section

  implicit none

!Arguments ------------------------------------
  !scalars
  type(dataset_type), intent(in)            :: dtset
  type(MPI_type), intent(in)                :: mpi_enreg
  type(pseudopotential_type), intent(in)    :: psps
  type(wvl_wf_type), intent(inout)            :: wfs
  !arrays
  real(dp), intent(in)                      :: rprimd(3, 3)
  real(dp), intent(inout)                   :: xred(3, dtset%natom)

!Local variables-------------------------------
  !scalars
  integer               :: idata
  logical               :: parallel
  character(len = 20)   :: atomnames(100)
  character(len=500)    :: message
  !arrays
  real(dp), allocatable :: xcart(:,:)

! *********************************************************************

#if defined MPI
  parallel = .true.
#else
  parallel = .false.
#endif

#if defined HAVE_BIGDFT
  ! Consistency checks, are all pseudo true GTH pseudo with geometric informations?
  if (dtset%npsp /= dtset%ntypat) then
    write(message, '(a,a,a,a,I0,a,I0,a,a,a)' ) ch10,&
      &  ' wvl_init_wfs_type :  consistency checks failed,', ch10, &
      &  '  dtset%npsp (', dtset%npsp, ') /= dtset%ntypat (', dtset%ntypat, ').', ch10, &
      &  '  No alchemy pseudo are allowed with wavelets.'
    call wrtout(6,message,'COLL')
    call leave_new('COLL')
  end if
  do idata = 1, dtset%ntypat, 1
    if (.not. psps%gth_params%set(idata)) then
      write(message, '(a,a,a,a,I0,a,a,a)' ) ch10,&
        &  ' wvl_init_wfs_type :  consistency checks failed,', ch10, &
        &  '  no GTH parameters found for type number ', idata, '.', ch10, &
        &  '  Check your input pseudo files.'
      call wrtout(6,message,'COLL')
      call leave_new('COLL')
    end if
    if (.not. psps%gth_params%hasGeometry(idata)) then
      write(message, '(a,a,a,a,a,a)' ) ch10,&
        &  ' wvl_init_wfs_type :  consistency checks failed,', ch10, &
        &  '  the given GTH parameters has no geometry informations.', ch10, &
        &  '  Upgrade your input pseudo files to GTH with geometric informatoins.'
      call wrtout(6,message,'COLL')
      call leave_new('COLL')
    end if
  end do

  ! Store xcart for each atom
  allocate(xcart(3, dtset%natom))
  call xredxcart(dtset%natom, 1, rprimd, xcart, xred)
  
  ! Init basic values.
  wfs%mband_init = 0
  wfs%mbandp_init = 0

  ! Nullify possibly unset pointers
  wfs%psidst  => null()
  wfs%hpsidst => null()
  wfs%ads     => null()
  
  ! Static allocations.
  allocate(wfs%eval(dtset%mband))
  
  ! We store only one big segment for each grid resolution in the key access object.
  wfs%keys%ndata = 1
  ! 0 is for coarse grid and 1 for fine grid
  allocate(wfs%keys%nseg (0:1))
  allocate(wfs%keys%nvctr(0:1))
  allocate(wfs%ibyz_c(2,0:dtset%wvl_internal%nSize(2),0:dtset%wvl_internal%nSize(3)), &
         & wfs%ibxz_c(2,0:dtset%wvl_internal%nSize(1),0:dtset%wvl_internal%nSize(3)), &
         & wfs%ibxy_c(2,0:dtset%wvl_internal%nSize(1),0:dtset%wvl_internal%nSize(2)))
  allocate(wfs%ibyz_f(2,0:dtset%wvl_internal%nSize(2),0:dtset%wvl_internal%nSize(3)), &
         & wfs%ibxz_f(2,0:dtset%wvl_internal%nSize(1),0:dtset%wvl_internal%nSize(3)), &
         & wfs%ibxy_f(2,0:dtset%wvl_internal%nSize(1),0:dtset%wvl_internal%nSize(2)))
  write(message, '(a,a)' ) ch10,&
    &  ' wvl_init_wfs_type : Create access keys for wavefunctions.'
  call wrtout(6,message,'COLL')
  call createWavefunctionsDescriptors( &
       & parallel, mpi_enreg%me, mpi_enreg%nproc, dtset%nwfshist, &
       & dtset%wvl_internal%nSize(1), dtset%wvl_internal%nSize(2), &
       & dtset%wvl_internal%nSize(3), .false., dtset%wvl_hgrid, &
       & dtset%natom, dtset%ntypat, dtset%typat, atomnames, xcart, &
       & psps%gth_params%radii_cf, dtset%wvl_crmult, dtset%wvl_frmult, &
       & wfs%ibyz_c, wfs%ibxz_c, wfs%ibxy_c, &
       & wfs%ibyz_f, wfs%ibxz_f, wfs%ibxy_f, &
       & wfs%keys%nseg(0), wfs%keys%nseg(1), wfs%keys%nvctr(0), &
       & wfs%keys%nvctr(1), wfs%mvctrp, wfs%keys%keyg, wfs%keys%keyv, &
       & dtset%mband, wfs%mbandp, wfs%psi, wfs%hpsi, wfs%psit, &
       & wfs%psidst, wfs%hpsidst, wfs%ads)
  write(message, '(a,2I8)' ) &
    &  '  | all orbitals have coarse segments, elements:', &
    &  wfs%keys%nseg(0), wfs%keys%nvctr(0)
  call wrtout(6,message,'COLL')
  write(message, '(a,2I8)' ) &
    &  '  | all orbitals have fine   segments, elements:', &
    &  wfs%keys%nseg(1), 7 * wfs%keys%nvctr(1)
  call wrtout(6,message,'COLL')
  
  write(message, '(a,a,a,a,I0)' ) ch10, &
    &  ' wvl_init_wfs_type : allocate wavefunctions,', ch10, &
    &  '  size of the compressed array per proc: ', &
    & (wfs%keys%nvctr(0) + 7 * wfs%keys%nvctr(1)) * wfs%mbandp
  call wrtout(6,message,'COLL')

  ! Deallocations
  deallocate(xcart)

#else
  write(message, '(a,a,a,a)' ) ch10,&
    &  ' wvl_init_type_wfs : BigDFT library is not compiled.', ch10, &
    &  '   Action, used the flag --enable-bigdft when configuring.'
  call wrtout(6,message,'COLL')
  call leave_new('COLL')
#endif
end subroutine wvl_init_type_wfs
!!***
