!{\src2tex{textfont=tt}}
!!****f* ABINIT/wvl_setBoxGeometry
!! NAME
!! wvl_setBoxGeometry
!!
!! FUNCTION
!! When wavelets are used, the box definition needs to be changed.
!! The box size is recomputed knowing some psp informations such as
!! the radius for coarse and fine grid. Then, the atoms are translated
!! to be included in the new box. Finally the FFT grid is computed using
!! the fine wavelet mesh and a buffer characteristic of used wavelets plus
!! a buffer used to be multiple of 2, 3 or 5.
!!
!! COPYRIGHT
!! Copyright (C) 1998-2007 ABINIT group (DC)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!!  mpi_enreg=informations about MPI parallelization
!!  psps <type(pseudopotential_type)>=variables related to pseudopotentials
!!
!! OUTPUT
!!  acell(3)=the new cell definition.
!!  rprimd(3,3)=dimensional primitive translations in real space (bohr)
!!
!! SIDE EFFECTS
!!  dtset <type(dataset_type)>=internal variables used by wavelets, describing
!!                             the box are set. The FFT grid is also changed.
!!  xred(3,natom)=reduced dimensionless atomic coordinates
!!
!! PARENTS
!!
!! CHILDREN
!!
!! SOURCE
#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine wvl_setBoxGeometry(acell, dtset, mpi_enreg, psps, rprimd, xred)

  use defs_basis
  use defs_datatypes
#if defined HAVE_BIGDFT
  use libbigdft
#endif

!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifdef HAVE_FORTRAN_INTERFACES
 use interfaces_01manage_mpi
 use interfaces_12geometry, except_this_one => wvl_setBoxGeometry
 use interfaces_12poisson
#endif
!End of the abilint section

  implicit none

!Arguments ------------------------------------
  !scalars
  type(dataset_type), intent(inout)         :: dtset
  type(MPI_type), intent(in)                :: mpi_enreg
  type(pseudopotential_type), intent(in)    :: psps
  !arrays
  real(dp), intent(inout)                   :: acell(3)
  real(dp), intent(inout)                   :: rprimd(3, 3)
  real(dp), intent(inout)                   :: xred(3, dtset%natom)

!Local variables-------------------------------
  !scalars
  integer               :: i, iat
  real(dp)              :: cxmin, cxmax, cymin, cymax, czmin, czmax, rad
  character(len=500)    :: message
  real(dp), parameter   :: eps_mach = 1.d-12, onem = 1.d0 - eps_mach
  !arrays
  real(dp)              :: rprim(3, 3)
  real(dp), allocatable :: xcart(:,:)

! *********************************************************************

#if defined HAVE_BIGDFT
  write(message, '(a,a,a,a)' ) ch10,&
    &  ' wvl_setBoxGeometry : Changing the box for wavelets computation.'
  call wrtout(6,message,'COLL')

  ! Store xcart for each atom
  allocate(xcart(3, dtset%natom))
  call xredxcart(dtset%natom, 1, rprimd, xcart, xred)
  
  call system_size(dtset%natom, xcart, psps%gth_params%radii_cf(:, 1), dtset%wvl_crmult, &
                 & dtset%typat, dtset%ntypat, cxmin, cxmax, cymin, cymax, czmin, czmax)
  acell(1) = cxmax - cxmin
  acell(2) = cymax - cymin
  acell(3) = czmax - czmin

  ! shift atomic positions such that molecule is inside cell
  do iat = 1, dtset%natom, 1
    xcart(1, iat) = xcart(1, iat) - cxmin
    xcart(2, iat) = xcart(2, iat) - cymin
    xcart(3, iat) = xcart(3, iat) - czmin
  enddo
  write(message, '(a,3F12.6)' ) &
    &  '  | shifting coordinates: ', (/ -cxmin, -cymin, -czmin /)
  call wrtout(6,message,'COLL')

  
  ! Find grid sizes n1,n2,n3
  do i = 1, 3, 1
    dtset%wvl_internal%nSize(i) = int(acell(i) / dtset%wvl_hgrid)
    acell(i) = dtset%wvl_internal%nSize(i) * dtset%wvl_hgrid
  end do
  write(message, '(a,3F12.6)' ) &
    &  '  | acell is now:         ', acell
  call wrtout(6,message,'COLL')

  ! Fine grid size (needed for creation of input wavefunction, preconditioning)
  dtset%wvl_internal%fineGrid(1, :) = dtset%wvl_internal%nSize(:)
  dtset%wvl_internal%fineGrid(2, :) = 0
  do iat = 1, dtset%natom, 1
    rad = psps%gth_params%radii_cf(dtset%typat(iat), 2) * dtset%wvl_frmult
    do i = 1, 3, 1
      dtset%wvl_internal%fineGrid(1, i) = min(dtset%wvl_internal%fineGrid(1, i), &
                                            & int(onem + (xcart(i, iat) - rad) / dtset%wvl_hgrid))
      dtset%wvl_internal%fineGrid(2, i) = max(dtset%wvl_internal%fineGrid(2, i), &
                                            & int((xcart(i, iat) + rad) / dtset%wvl_hgrid))
    end do
  enddo
  write(message, '(a,2I5,a,a,2I5,a,a,2I5)' ) &
    &  '  | nfl1, nfu1:           ', dtset%wvl_internal%fineGrid(:, 1), ch10, &
    &  '  | nfl2, nfu2:           ', dtset%wvl_internal%fineGrid(:, 2), ch10, &
    &  '  | nfl3, nfu3:           ', dtset%wvl_internal%fineGrid(:, 3)
  call wrtout(6,message,'COLL')

  
  ! Change the metric to orthogonal one
  rprim(:, :) = real(0., dp)
  do i = 1, 3, 1
    rprim(i, i) = real(1., dp)
  end do
  call mkrdim(acell, rprim, rprimd)

  ! Save shifted atom positions into xred
  call xredxcart(dtset%natom, -1, rprimd, xcart, xred)
  deallocate(xcart)
  
  ! Change nfft and ngfft
  dtset%wvl_internal%nDpPoints = 1
  do i = 1, 3, 1
    dtset%wvl_internal%dpSize(i) = 2 * dtset%wvl_internal%nSize(i) + dtset%wvl_internal%buffer
    dtset%wvl_internal%nDpPoints = dtset%wvl_internal%nDpPoints * dtset%wvl_internal%dpSize(i)
  end do
  do i = 1, 3, 1
    call fourier_dim(dtset%wvl_internal%dpSize(i), dtset%ngfft(i))
  end do
  dtset%nfft = dtset%ngfft(1) * dtset%ngfft(2) * dtset%ngfft(3) / mpi_enreg%nproc_fft
  !Set up fft array dimensions ngfft(4,5,6) to avoid cache conflicts
  ! Code paste from getng()
  dtset%ngfft(4) = 2 * (dtset%ngfft(1) / 2) + 1
  dtset%ngfft(5) = 2 * (dtset%ngfft(2) / 2) + 1
  dtset%ngfft(6) = dtset%ngfft(3)
  if (mpi_enreg%paral_fft == 0) then
    dtset%ngfft(9)  = 0    ! paral_fft
    dtset%ngfft(10) = 1    ! nproc_fft
    dtset%ngfft(11) = 0    ! me_fft
    dtset%ngfft(12) = 0    ! n2proc
    dtset%ngfft(13) = 0    ! n3proc
  else
    dtset%ngfft(9)  = 1    ! paral_fft
    dtset%ngfft(10) = mpi_enreg%nproc_fft
    dtset%ngfft(11) = mpi_enreg%me_fft
    dtset%ngfft(12) = dtset%ngfft(2) / mpi_enreg%nproc_fft
    dtset%ngfft(13) = dtset%ngfft(3) / mpi_enreg%nproc_fft
  end if
  write(message, '(a,3I12)' ) &
    &  '  | ngfft(1:3) is now:    ', dtset%ngfft(1:3)
  call wrtout(6,message,'COLL')
  write(message, '(a,3I12)' ) &
    &  '  | ngfft(4:6) is now:    ', dtset%ngfft(4:6)
  call wrtout(6,message,'COLL')
  write(message, '(a,3I12)' ) &
    &  '  | box size for datas:   ', dtset%wvl_internal%dpSize
  call wrtout(6,message,'COLL')
  write(message, '(a,3I12)' ) &
    &  '  | box size for wavelets:', dtset%wvl_internal%nSize
  call wrtout(6,message,'COLL')


  ! Set mgfft
  dtset%mgfft= max(dtset%ngfft(1), dtset%ngfft(2), dtset%ngfft(3))
  
#else
  write(message, '(a,a,a,a)' ) ch10,&
    &  ' wvl_setBoxGeometry : BigDFT library is not compiled.', ch10, &
    &  '   Action, used the flag --enable-bigdft when configuring.'
  call wrtout(6,message,'COLL')
  call leave_new('COLL')
#endif
end subroutine wvl_setBoxGeometry
!!***
