/* =========================================================================
   Copyright (c) 2010-2011, Institute for Microelectronics,
                            Institute for Analysis and Scientific Computing,
                            TU Wien.

                            -----------------
                  ViennaCL - The Vienna Computing Library
                            -----------------

   Project Head:    Karl Rupp                   rupp@iue.tuwien.ac.at
               
   (A list of authors and contributors can be found in the PDF manual)

   License:         MIT (X11), see file LICENSE in the base directory
============================================================================= */

// activate ublas support in ViennaCL
#define VIENNACL_HAVE_UBLAS 

//
// include necessary system headers
//
#include <iostream>

//
// ViennaCL includes: We only need the qr-header
//
#include "viennacl/linalg/qr.hpp"

//
// Boost includes
//
#include <boost/numeric/ublas/vector.hpp>
#include <boost/numeric/ublas/matrix.hpp>
#include <boost/numeric/ublas/io.hpp>

/*
*   Tutorial: QR factorization of host-based (ublas-)matrices
*/

int main (int argc, const char * argv[])
{
  typedef float               ScalarType;     //feel free to change this to 'double' if supported by your hardware
  typedef boost::numeric::ublas::matrix<ScalarType>        MatrixType;
  typedef boost::numeric::ublas::vector<ScalarType>        VectorType;

  size_t rows = 7;
  size_t cols = 6;
  
  //
  // Create matrices with some data
  //
  MatrixType A(rows, cols);
  MatrixType Q(rows, rows);
  MatrixType R(rows, cols);
  
  for (size_t i=0; i<rows; ++i)
  {
    for (size_t j=0; j<cols; ++j)
    {
      A(i,j) = 1.0 + (i + 1)*(j+1);
      R(i,j) = 0.0;
    }
    
    for (size_t j=0; j<rows; ++j)
      Q(i,j) = 0.0;
  }
  
  // Print matrix before factorization
  std::cout << "A: " << A << std::endl << std::endl;
  
  //
  // Compute QR factorization of A. A is overwritten with Householder vectors. Coefficients are returned and a block size of 3 is used.
  // Note that at the moment the number of columns of A must be divisible by the block size
  //
  std::vector<ScalarType> betas = viennacl::linalg::inplace_qr(A, 3);
  
  std::cout << "Inplace QR-factored A: " << A << std::endl << std::endl;
  
  //
  // Recover full matrix orthogonal matrix Q and upper right matrix R
  //
  viennacl::linalg::recoverQ(A, betas, Q, R); 

  // Print for verification purposes:
  std::cout << "R after recovery: " << R << std::endl << std::endl;
  std::cout << "Q after recovery: " << Q << std::endl << std::endl;
  std::cout << "Q*Q^T: " << prod(Q, trans(Q)) << std::endl << std::endl;  //This should be identity up to round-off errors
  
  std::cout << "Q * R: " << prod(Q, R) << std::endl << std::endl; // this should be the initial matrix A up to round-off errors
    
  //
  //  That's it.
  //
  std::cout << "!!!! TUTORIAL COMPLETED SUCCESSFULLY !!!!" << std::endl;

  return EXIT_SUCCESS;
}

