package tijmp.ui;

import java.awt.Component;
import java.awt.Cursor;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTree;
import javax.swing.event.TreeExpansionEvent;
import javax.swing.event.TreeWillExpandListener;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.ExpandVetoException;
import javax.swing.tree.MutableTreeNode;
import javax.swing.tree.TreeNode;
import tijmp.OwnerInfo;
import tijmp.OwnerInfoHeader;
import tijmp.ProfilerHandler;

/** A panel that shows object owner information
 */
public class OwnerInfoTree extends JPanel {
    public OwnerInfoTree (ProfilerHandler ph, 
			  Map<Long, OwnerInfoHeader> owners, 
			  long[] startObjects) {
	JTree tree = new JTree ();
	DefaultMutableTreeNode root = new DefaultMutableTreeNode ();
	Object[] os = ph.getObjectsForTags (startObjects);
	for (int i = 0; i < startObjects.length; i++) 
	    if (os[i] != null)
		root.add (new OwnerTreeNode (ph, tree, owners, 
					     startObjects[i], os[i], null));
	tree.setModel (new DefaultTreeModel (root));
        tree.addTreeWillExpandListener (new TreeExpander ());

	JScrollPane sp = new JScrollPane (tree);
	tree.setRootVisible (false);
	tree.setCellRenderer (new OIRenderer ());

	GridBagLayout gb = new GridBagLayout ();
	GridBagConstraints c = new GridBagConstraints ();
	setLayout (gb);
	c.insets = new Insets (2, 2, 2, 2);

	c.gridx = 0; 
	c.gridy = 0;
	c.weightx = 1;
	c.weighty = 1;
	c.fill = GridBagConstraints.BOTH;
	add (sp, c);
    }
}

class TreeExpander implements TreeWillExpandListener {
    public void treeWillExpand (TreeExpansionEvent e) 
	throws ExpandVetoException {
        TreeNode tn = (TreeNode)e.getPath ().getLastPathComponent ();
	if (tn instanceof OwnerTreeNode) 
            ((OwnerTreeNode)tn).addSubNodes ();
    }
    
    public void treeWillCollapse (TreeExpansionEvent e) {
        TreeNode tn = (TreeNode)e.getPath ().getLastPathComponent ();
        if (tn instanceof OwnerTreeNode)
	    ((OwnerTreeNode)tn).removeSubNodes ();
    }
}

class OIRenderer extends DefaultTreeCellRenderer {
    private static final int JVMTI_HEAP_REFERENCE_FIELD = 2;
    private static final int JVMTI_HEAP_REFERENCE_ARRAY_ELEMENT = 3;
    private static final int JVMTI_HEAP_REFERENCE_STATIC_FIELD = 8;

    @Override 
    public Component getTreeCellRendererComponent (JTree tree, Object value,
						   boolean sel, 
						   boolean expanded, 
						   boolean leaf, int row, 
						   boolean hasFocus) {
	super.getTreeCellRendererComponent (tree, "", sel, expanded, 
					    leaf, row, hasFocus);
	if (value instanceof OwnerTreeNode) {
	    OwnerTreeNode otn = (OwnerTreeNode)value;
	    String cn = null;
	    String prefix = "";
	    OwnerInfo oi = otn.getOwnerInfo ();
	    if (oi != null) {
		Field f;
		String fname;
		switch (oi.getReferenceType ()) {
		case JVMTI_HEAP_REFERENCE_FIELD: 
		    f = otn.getField (false);
		    fname = f == null ? "<null>?" : f.getName ();
		    prefix = "field " + fname + "(" + 
			oi.getIndex () + ") in";
		    break;
		case JVMTI_HEAP_REFERENCE_ARRAY_ELEMENT:
		    prefix ="position " + oi.getIndex () + " in";
		    break;
		case JVMTI_HEAP_REFERENCE_STATIC_FIELD:
		    f = otn.getField (true);
		    fname = f == null ? "<null>?" : f.getName ();
		    prefix = "static field " + fname + "(" + 
			oi.getIndex () + ") in";
		    cn = getClassText (otn, true);
		    break;
		}
	    }
	    if (cn == null)
		cn = getClassText (otn, false);
	    setText (prefix + " " + cn);
	} else if (value instanceof DefaultMutableTreeNode) {
	    DefaultMutableTreeNode dmt = (DefaultMutableTreeNode)value;
	    setText (getClassText (dmt, false));
	} else {
	    setText (value.toString ());
	}
	return this;
    }

    private String getClassText (DefaultMutableTreeNode dmt, boolean isStatic) {
	Object obj = dmt.getUserObject ();
	if (obj == null)
	    return "<null>";
	if (isStatic) { 
	    Class<?> c = (Class<?>)obj;
	    return Translator.translate (c);
	} 
	Class<?> c = obj.getClass ();
	return Translator.translate (c);
    }
}

class OwnerTreeNode extends DefaultMutableTreeNode {
    private ProfilerHandler ph;
    private Map<Long, OwnerInfoHeader> owners;
    private long tag;
    private JTree tree;
    private OwnerInfo oi;
    private Field field;
    
    public OwnerTreeNode (ProfilerHandler ph,
			  JTree tree, Map<Long, OwnerInfoHeader> owners, 
			  long tag, Object node, OwnerInfo oi) {
	super (node, true);
	this.ph = ph;
	this.tree = tree;
	this.tag = tag;
	this.owners = owners;
	this.oi = oi;
    }

    public long getTag () {
	return tag;
    }

    public OwnerInfo getOwnerInfo () {
	return oi;
    }

    public Field getField (boolean objectIsClass) {
	if (field != null)
	    return field;
	if (oi == null)
	    return null;
	Object o = getUserObject ();
	if (o == null)
	    return null;
	List<Class<?>> ls = new ArrayList<Class<?>> ();
	List<Class<?>> is = new ArrayList<Class<?>> ();
	Class<?> c = o.getClass ();
	if (objectIsClass)
	    c = (Class<?>)o;
	while (c != null) {
	    ls.add (c);
	    for (Class<?> i : c.getInterfaces ()) {
		if (!is.contains (i))
		    is.add (i);
	    }
	    c = c.getSuperclass ();
	}
	ls.addAll (is);
	Collections.reverse (ls);
	
	int fieldIndex = oi.getIndex ();
	for (Class<?> cs : ls) {
	    Field[] fs = cs.getDeclaredFields ();
	    if (fs.length > fieldIndex)
		return fs[fieldIndex];
	    fieldIndex -= fs.length;
	}
	return null;
    }

    @Override public int getChildCount () {
	if (owners.containsKey (tag))
	    return Math.max (1, super.getChildCount ());
	return super.getChildCount ();
    }
    
    private boolean alreadyShownInParents (Object o) {
	DefaultMutableTreeNode parent = this;
	while (parent != null) {
	    if (parent.getUserObject () == o)
		return true;
	    parent = (DefaultMutableTreeNode)parent.getParent ();
	}
	return false;
    }

    public void addSubNodes () {
        try {
            tree.setCursor (new Cursor (Cursor.WAIT_CURSOR));
	    OwnerInfoHeader h = owners.get (tag);
	    long[] tags = h.getOwnerTags ();
	    Object[] os = ph.getObjectsForTags (tags);
	    DefaultTreeModel model = (DefaultTreeModel)tree.getModel ();
	    for (int i = 0; i < tags.length; i++) {
		if (os[i] != null) {
		    DefaultMutableTreeNode node = null;
		    if (alreadyShownInParents (os[i])) {
			node = new DefaultMutableTreeNode (os[i], false);
		    } else {
			OwnerInfo oi = h.getOwnerInfo (i);
			node = new OwnerTreeNode (ph, tree, owners, 
						  tags[i], os[i], oi);
		    }
		    model.insertNodeInto (node, this, super.getChildCount ());
		}
	    }
        } finally {
            tree.setCursor (null); // inherit
        }        
    }

    public void removeSubNodes () {
	DefaultTreeModel model = (DefaultTreeModel)tree.getModel ();
	while (super.getChildCount () > 0)
	    // use super since we lie on purpose :-)
	    model.removeNodeFromParent ((MutableTreeNode)getChildAt (0));
    }
}
