#ifndef __RPGGAME__
#define __RPGGAME__

#include "cube.h"

#define DEFAULTMAP "mansion"
#define DEFAULTMODEL "rc"
#define DEFAULTICON "data/rpg/hud/default"

enum                            // static entity types
{
	NOTUSED = ET_EMPTY,         // entity slot not in use in map
	LIGHT = ET_LIGHT,           // lightsource, attr1 = radius, attr2 = intensity
	MAPMODEL = ET_MAPMODEL,     // attr1 = angle, attr2 = idx
	PLAYERSTART,                // attr1 = angle, attr2 = team
	ENVMAP = ET_ENVMAP,         // attr1 = radius
	PARTICLES = ET_PARTICLES,
	MAPSOUND = ET_SOUND,
	SPOTLIGHT = ET_SPOTLIGHT,
	TELEDEST, //attr1 = yaw, attr2 = from
	JUMPPAD, //attr1 = Z, attr2 = Y, attr3 = X, attr4 = radius
	CHECKPOINT,
	SPAWN,
	RESERVED1, //backwards compatibility
	RESERVED2, //backwards compatibility
	BLIP,
	CAMERA,
	PLATFORMROUTE,
	MAXENTTYPES
};

struct rpgentity : extentity
{
	//extend if needed
};

enum
{
	STATUS_HEALTH = 0, //attributes
	STATUS_MANA,
	STATUS_MOVE,
	STATUS_CRIT,
	STATUS_HREGEN,
	STATUS_MREGEN,

	///WARNING: it is critical that these are in the same order as the stats in the stats struct
	STATUS_STRENGTH,
	STATUS_ENDURANCE,
	STATUS_AGILITY,
	STATUS_CHARISMA,
	STATUS_WISDOM,
	STATUS_INTELLIGENCE,
	STATUS_LUCK,

	///WARNING: it is critical that these are in the same order as the skills inside the stats struct
	STATUS_ARMOUR,
	STATUS_DIPLOMANCY,
	STATUS_MAGIC,
	STATUS_MARKSMAN,
	STATUS_MELEE,
	STATUS_STEALTH,

	///WARNING: it is critical that these are in the same order as the ATTACK_* enum below
	STATUS_FIRE_T, //these are thresholds
	STATUS_WATER_T,
	STATUS_AIR_T,
	STATUS_EARTH_T,
	STATUS_MAGIC_T,
	STATUS_SLASH_T,
	STATUS_BLUNT_T,
	STATUS_PIERCE_T,

	STATUS_FIRE_R, //these are resistances
	STATUS_WATER_R,
	STATUS_AIR_R,
	STATUS_EARTH_R,
	STATUS_MAGIC_R,
	STATUS_SLASH_R,
	STATUS_BLUNT_R,
	STATUS_PIERCE_R,

	STATUS_DISPELL, //negative strength dispells positive magics, positive strength bad magics
	STATUS_DOOM,
	STATUS_REFLECT, //reflects spells in the direction of the caster
	STATUS_INVIS,
	STATUS_LIGHT,
	STATUS_STUN,
	STATUS_POLYMORPH,
	STATUS_SILENCE,
	STATUS_LOCK, //objects
	STATUS_UNLOCK,
	STATUS_MAX
};

enum
{
	ATTACK_FIRE = 0,
	ATTACK_WATER,
	ATTACK_AIR,
	ATTACK_EARTH,
	ATTACK_MAGIC,
	ATTACK_SLASH,
	ATTACK_BLUNT,
	ATTACK_PIERCE,
	ATTACK_MAX
};

struct response
{
	const char *talk; //text the player can respond with
	int dest;
	const char *script; //script to execute when option is selected (optional)

	response(const char *t, int d, const char *s) : talk(newstring(t)), dest(d), script(s && *s ? newstring(s) : NULL) {}
	~response()
	{
		delete [] talk;
		delete [] script;
	}
};

struct rpgchat
{
	const char *talk, //text presented to player
		*script; //script that creates responses

	vector<response *> dests;

	void open()
	{
		execute(script);
	}
	void close()
	{
		dests.deletecontents();
	}

	rpgchat(const char *t, const char *s) : talk(t), script(s) {}
	~rpgchat()
	{
		dests.deletecontents();
		delete [] talk;
		delete [] script;
	}
};

enum
{
	SCR_APPROACH = 0,
	SCR_DEATH,
	SCR_DROP,
	SCR_EQUIP,
	SCR_INTERACT,
	SCR_SPAWN,
	SCR_USE, //includes casting
	SCR_COLLIDE, //see dynentcollide in rpg.cpp
	SCR_HIT,
	SCR_MAX

	/** These are still under consideration

	SCR_RESURRECT
	SCR_UPDATE - this might be too expensive/unfeasible

	*/
};

struct script
{
	int pos;
	vector<rpgchat *> dialogue;

	const char *scripts[SCR_MAX];

	void transfer(script &o)
	{
		loopv(dialogue)
			dialogue[i]->close();

		o = *this;
		o.pos = -1;
		loopi(SCR_MAX)
		{
			if(scripts[i])
				o.scripts[i] = newstring(scripts[i]);
		}
	}

	script() : pos(-1)
	{
		loopi(SCR_MAX) scripts[i] = NULL;
	}
	~script()
	{
		dialogue.deletecontents();
		loopi(SCR_MAX) delete[] scripts[i];
	}
};

enum
{
	MSCR_LOAD = 0,
	MSCR_ENTER, //executed when you enter a map, but only via teledest
	MSCR_MAX
};

struct mapscript
{
	const char *scripts[MSCR_MAX];

	void transfer(mapscript &o)
	{
		loopi(MSCR_MAX)
			if(scripts[i]) o.scripts[i] = newstring(scripts[i]);
	}

	mapscript()
	{
		loopi(MSCR_MAX) scripts[i] = NULL;
	}

	~mapscript()
	{
		loopi(MSCR_MAX) delete[] scripts[i];
	}
};

struct rpgent;
struct rpgchar;
struct rpgitem;

enum
{
	FX_NONE = 0,
	FX_TRAIL = 1<<0,
	FX_DYNLIGHT = 1<<1,
	FX_FLARE = 1<<2, //LENSFLARE
	FX_FIXEDFLARE = 1<<3,
	FX_SPIN = 1<<4,
	FX_MAX = 1<<5
};

struct effect
{
	int flags, decal;

	const char *mdl;
	int projpart, projpartcol;
	float projpartsize;

	int trailpart, trailpartcol, trailpartfade, trailpartgrav;
	float trailpartsize;

	int traillightradius;
	vec traillightcol;

	int deathpart, deathpartcol, deathpartfade, deathpartgrav;
	float deathpartsize;

	int deathlightflags, deathlightfade, deathlightradius, deathlightinitradius;
	vec deathlightcol, deathlightinitcol;

	void transfer(effect &o)
	{
		o = *this;
		if (mdl) o.mdl = newstring(mdl);
	}

	effect()
	{
		flags = FX_MAX - 1;
		decal = DECAL_BURN;

		mdl = NULL;
		projpart = PART_FIREBALL1;
		projpartcol = 0xFFBF00;
		projpartsize = 4;

		trailpart = PART_STEAM;
		trailpartcol = 0x7F7F7F;
		trailpartsize = 1.0f;
		trailpartfade = 500;
		trailpartgrav = 200;

		traillightradius = 64;
		traillightcol = vec(1, .75, 0);

		deathpart = PART_EXPLOSION;
		deathpartcol = 0xFFBF00;
		deathpartfade = 800;
		deathpartsize = 4.0f;
		deathpartgrav = 10;

		deathlightflags = DL_EXPAND;
		deathlightfade = 200;
		deathlightradius = 128;
		deathlightinitradius = 64;
		deathlightcol = deathlightinitcol = traillightcol;
	}
	~effect()
	{
		delete[] mdl;
	}
};

enum
{
	P_TIME = 1,		// dissapears after set time
	P_DIST = 2,		// dissapears after set distance
	P_RICOCHET = 4,		// bounces off walls
	P_VOLATILE = 8,		// takes effect at current position at end of life (mix with P_TIME or P_DIST)
	P_PROXIMITY = 16,	// takes effect at current position if there's something in range of the effect (provided they aren't coming towards the projectile)
	P_STATIONARY = 32,	// does not move (ie mines) - requires proximity or time
	P_MAX = 64
};

enum
{
	T_SINGLE = 0,
	T_MULTI = 1,
	T_AREA = 2,
	T_SELF = 3,
	//these 2 will simulate slashing weapons
	T_HORIZ = 4,
	T_VERT = 5,
	T_MAX
};

struct equipment;
struct projectile
{
	/**
		<NOTES>
		1) speed is always 100, this can be modulated by changing the magnitude of dir as neccesary
	*/
	rpgchar *owner;
	int item, use;
	int ammo, ause;

	vec o, dir, emitpos;
	int lastemit, gravity;
	bool deleted;
	entitylight light;

	int pflags, time, dist;
	int fx, radius;
	float elasticity;

	float charge;

	enum
	{
		INIT_FUZZY = 1,
		INIT_ADJUST = 2
	};

	void init(rpgchar *d, equipment *w, equipment *a, int flags, float speed = 1.0f);
	bool update();
	rpgent *travel(float &distance);
	void render(bool mainpass);
	void drawdeath();
	void dynlight();

	projectile() : owner(NULL), item(-1), use(-1), ammo(-1), ause(-1), o(vec(0, 0, 0)), dir(vec(0, 0, 0)), emitpos(vec(0, 0, 0)), lastemit(0), gravity(0), deleted(false), pflags(0), time(10000), dist(25000), fx(0), radius(32), elasticity(.8), charge (1.0f) {}
	~projectile() {}
};

struct status
{
	int type;

	virtual void dispell(float mag) {} // used by dispell to weaken effects - mainly used for healing or DoT
	virtual status *dup()=0; //duplicate

	status() : type(STATUS_HEALTH) {}
	virtual ~status() {}
};

struct status_generic : status
{
	int strength;

	status *dup()
	{
		return new status_generic(*this);
	}
	status_generic() : strength(0) {}
	~status_generic() {}
};

struct status_polymorph : status
{
	const char *mdl;

	status *dup()
	{
		status_polymorph *p = new status_polymorph();
		p->type = type;
		p->mdl = mdl ? newstring(mdl) : NULL;
		return p;
	}

	status_polymorph() : mdl(NULL) {}
	~status_polymorph() { delete[] mdl; }
};

struct status_light : status
{
	vec colour;
	int radius;

	status *dup() { return new status_light(*this); }

	status_light() : colour(vec(0, 0, 0)), radius(0) {}
	~status_light() {}
};

struct statusgroup
{
	vector<status *> effects;
	bool friendly;

	const char *icon, *name, *description;

	statusgroup() : friendly(false), icon(NULL), name(NULL), description(NULL) {}
	~statusgroup()
	{
		effects.deletecontents();
	}

	void transfer(statusgroup &o)
	{
		o = *this;
		if(icon) o.icon = newstring(icon);
		if(name) o.name = newstring(name);
		if(description) o.description = newstring(description);

		o.effects.setsize(0);

		loopv(effects)
		{
			o.effects.add(effects[i]->dup());
		}
	}
};

struct areaeffect
{
	rpgent *owner;
	vec o;
	int startmillis, lastemit, duration, group, radius, fx;
	float mul;

	bool update();
	void render();

	areaeffect() : owner(NULL), o(vec(0, 0, 0)), startmillis(lastmillis), lastemit(lastmillis), duration(0), group(0), radius(0), fx(0), mul(1.0f) {}
	~areaeffect() {}
};

struct victimeffect
{
	rpgent *owner;
	int startmillis, duration, group;
	float mul;

	bool update(rpgent *victim);
	victimeffect(rpgent *o = NULL, int d = 0, int g = 0, float m = 1.0f) : owner(o), startmillis(lastmillis), duration(d), group(g), mul(m) {}
	~victimeffect() {}
};

struct invstack
{
	int base;
	int quantity;

	invstack(int b = 0, int q = 0) : base(b), quantity(q) {}
};

enum
{
	ENT_INVALID = 0,
	ENT_CHAR,
	ENT_ITEM,
	ENT_OBJECT
};

struct equipment
{
	int base; int use;
	equipment(int b, int u = 0) : base(b), use(u) {}
};

struct use_weapon;

struct rpgent : dynent
{
	int lasttouch; //used for entity touches - mainly to avoid getting overpowered

	struct
	{
		vec4 light; //w == radius x y z == R G B
		float alpha;
		const char *mdl;
	} temp;

	///everything can suffer some status effects, whether this is just invisibility or something more sinister is up for debate
	vector<victimeffect *> seffects;

	///global
	virtual void update()=0;
	virtual void resetmdl()=0;
	virtual void render(bool mainpass)=0;
	virtual int getscript()=0;
	virtual vec blipcol() { return vec(1, 1, 1);}
	virtual const char *getname() const {return "noname";}
	virtual const int type()=0;

	///character/AI
	virtual void givexp(int xp) {}
	virtual void equip(int i, int u = 0) {}
	virtual void dequip(int i, int slots = 0) {}
	virtual void die(rpgent *killer = NULL) {}
	virtual void respawn() {}
	virtual void hit(rpgent *attacker, use_weapon *weapon, use_weapon *ammo, float mul, vec dir) = 0;
	virtual void applyeffects(statusgroup *g, float mag, bool instant, rpgent *owner = NULL) {}

	///inventory
	virtual int drop(int b, int q, bool spawn = true) {return 0;}
	virtual bool pickup(rpgent *item) {return false;}
	virtual void uncurse() {}
	virtual void curse() {}
	virtual int getitemcount(int i)=0;

	///read and write functions for savegames

	///{de,con}structors
	rpgent() : lasttouch(0)
	{
		temp.mdl = NULL;
		temp.light = vec4(0, 0, 0, 0);
		temp.alpha = 1;
	}
	virtual ~rpgent()
	{
		seffects.deletecontents();
	}
};

enum
{
	STAT_STRENGTH = 0,
	STAT_ENDURANCE,
	STAT_AGILITY,
	STAT_CHARISMA,
	STAT_WISDOM,
	STAT_INTELLIGENCE,
	STAT_LUCK,
	STAT_MAX
};

enum
{
	SKILL_ARMOUR = 0,
	SKILL_DIPLOMACY,
	SKILL_MAGIC,
	SKILL_MARKSMAN,
	SKILL_MELEE,
	SKILL_STEALTH,
	SKILL_MAX
};

struct stats
{
	//note delta vars are for spell effects
	///SPECIAL
	int experience, level, points;

	///PRIMARY STATS
	short baseattrs[STAT_MAX];
	short baseskills[SKILL_MAX];

	short deltaattrs[STAT_MAX];
	short deltaskills[SKILL_MAX];

	///DERIVED STATS
	short bonusthresh[ATTACK_MAX];
	short bonusresist[ATTACK_MAX];
	int bonushealth, bonusmana, bonusmovespeed, bonusjumpvel, bonuscarry, bonuscrit;
	float bonushregen, bonusmregen;

	short deltathresh[ATTACK_MAX];
	short deltaresist[ATTACK_MAX];
	int deltahealth, deltamana, deltamovespeed, deltajumpvel, deltacarry, deltacrit;
	float deltahregen, deltamregen;

	///QUERIES
	static int neededexp(int level) { return level * (level + 1) * 500; } //standard EXP formula, 0... 1000.... 3000.... 6000.... 10000....
	void givexp(int xp)
	{
		experience = max(0, experience + xp);
		while(experience >= neededexp(level))
		{
			level++;
			points += 5;
		}
	}

	//no safety checks are done, make sure to use the ENUM
	inline int getattr(int n) const {return max(0, baseattrs[n] + deltaattrs[n]);}
	inline int getskill(int n) const {return max(0, baseskills[n] + deltaskills[n]);}
	inline int critchance() const {return bonuscrit + deltacrit + getattr(STAT_LUCK) / 10;}

	int getmaxhp()
	{
		int amnt = bonushealth + deltahealth + (getattr(STAT_ENDURANCE) * 5 + getattr(STAT_STRENGTH)) * (10 + level) / 10.0f;
		return max(1, amnt); ///TODO allow maxhp of 0 and just kill the character?
	}

	float gethpregen()
	{
		return bonushregen + deltahregen + getattr(STAT_ENDURANCE) * 0.01 + getattr(STAT_STRENGTH) * 0.005;
	}

	int getmaxmp()
	{
		int amnt = bonusmana + deltamana + (getattr(STAT_INTELLIGENCE) * 2 + getattr(STAT_WISDOM) * 2) * (10 + level) / 10.0f;
		return max(0, amnt);
	}

	float getmpregen()
	{
		return bonusmregen + deltamregen + getattr(STAT_WISDOM)  * 0.05 + getattr(STAT_INTELLIGENCE) * 0.025;
	}

	int getmaxcarry()
	{
		int amnt = bonuscarry + deltacarry + getattr(STAT_STRENGTH) * 5;
		return max(0, amnt);
	}

	int getthreshold(int n)
	{
		//unlike resistances, thresholds are not defined by stats
		return bonusthresh[n] + deltathresh[n];
	}

	int getresistance(int n)
	{
		float amnt = bonusresist[n] + deltaresist[n] + getattr(STAT_LUCK) / 15.0f;
		switch(n)
		{
			case ATTACK_FIRE:
			case ATTACK_WATER:
				amnt += getattr(STAT_AGILITY) / 10.0f; break;
			case ATTACK_AIR:
			case ATTACK_EARTH:
				amnt += getattr(STAT_STRENGTH) / 10.0f; break;
			case ATTACK_MAGIC:
				amnt += (getattr(STAT_WISDOM) + getattr(STAT_INTELLIGENCE)) / 20.0f; break;
			case ATTACK_SLASH:
				amnt += getattr(STAT_AGILITY) / 20.0f; break;
			case ATTACK_BLUNT:
				amnt += getattr(STAT_STRENGTH) / 20.0f; break;
			case ATTACK_PIERCE:
				amnt += getattr(STAT_ENDURANCE) / 20.0f; break;
		}
		return min<int>(95, amnt);
	}

	float skillpotency(int n)
	{
		float amnt = .5;
		switch(n)
		{
			case -1: return 1; //special NO MODIFIERs case
			case SKILL_ARMOUR:
				amnt += (getattr(STAT_STRENGTH)      + getattr(STAT_ENDURANCE) * 2       + getskill(SKILL_ARMOUR) * 5) / 100.0f;
				break;
			case SKILL_DIPLOMACY:
				amnt += (getattr(STAT_CHARISMA) * 3                                      + getskill(SKILL_DIPLOMACY) * 5) / 100.0f;
				break;
			case SKILL_MAGIC:
				amnt += (getattr(STAT_WISDOM) * 1.75 + getattr(STAT_INTELLIGENCE) * 1.25 + getskill(SKILL_MAGIC) * 5) / 100.0f;
				break;
			case SKILL_MARKSMAN:
				amnt += (getattr(STAT_STRENGTH) * .5 + getattr(STAT_AGILITY) * 2.5       + getskill(SKILL_MARKSMAN) * 5) / 100.0f;
				break;
			case SKILL_MELEE:
				amnt += (getattr(STAT_AGILITY) * .5  + getattr(STAT_STRENGTH) * 2.5      + getskill(SKILL_MELEE) * 5) / 100.0f;
				break;
			case SKILL_STEALTH:
				amnt += (getattr(STAT_LUCK) * 1      + getattr(STAT_AGILITY) * 2         + getskill(SKILL_STEALTH) * 5) / 100.0f;
				break;
		}
		amnt += logf(1 + amnt);
		amnt *= (50 + rnd(101)) / 100.0f;
		if(rnd(100) < critchance())
		{
			amnt += 1;
			if(rnd(10000) < critchance())
			{
				amnt += 1;
				if(rnd(1000000) < critchance())
					amnt += 1;
			}
		}

		return amnt;
	}

	void setspeeds(float &maxspeed, float &jumpvel)
	{
		maxspeed = 40 + bonusmovespeed + deltamovespeed + getattr(STAT_AGILITY) / 8;
		jumpvel = 80 + bonusjumpvel + deltajumpvel + getattr(STAT_AGILITY) / 6;
	}

	void resetdeltas()
	{
		loopi(STAT_MAX)
			deltaattrs[i] = 0;
		loopi(SKILL_MAX)
			deltaskills[i] = 0;
		loopi(ATTACK_MAX)
		{
			deltathresh[i] = 0;
			deltaresist[i] = 0;
		}
		deltahealth = deltamana = deltamovespeed = deltajumpvel = deltacarry = deltacrit = 0;
		deltahregen = deltamregen = 0;
	}

	stats() : experience(0), level(1), points(0)
	{
		loopi(STAT_MAX)
			baseattrs[i] = 10;
		loopi(SKILL_MAX)
			baseskills[i] = 0;
		loopi(ATTACK_MAX)
		{
			bonusthresh[i] = 0;
			bonusresist[i] = 0;
		}
		bonushealth = bonusmana = bonusmovespeed = bonusjumpvel = bonuscarry = bonuscrit = 0;
		bonushregen = bonusmregen = 0;
		resetdeltas();
	}
	~stats() {}
};

struct statreq
{
	short attrs[STAT_MAX];
	short skills[SKILL_MAX];

	statreq()
	{
		loopi(STAT_MAX)
			attrs[i] = 0;
		loopi(SKILL_MAX)
			skills[i] = 0;
	}
	~statreq() {}

	bool meet(stats &st)
	{
		loopi(STAT_MAX)
		{
			if(attrs[i] > st.getattr(i))
				return false;
		}
		loopi(SKILL_MAX)
		{
			if(skills[i] > st.getskill(i))
				return false;
		}

		return true;
	}
};

enum
{
	USE_CONSUME = 0,
	USE_ARMOUR,
	USE_WEAPON,
	USE_MAX
};

enum
{
	SLOT_LHAND  = 1 << 0,
	SLOT_RHAND  = 1 << 1,
	SLOT_LEGS   = 1 << 2,
	SLOT_ARMS   = 1 << 3,
	SLOT_TORSO  = 1 << 4,
	SLOT_HEAD   = 1 << 5,
	SLOT_FEET   = 1 << 6,
	SLOT_QUIVER = 1 << 7,
	SLOT_MAX    = 1 << 8,
};

struct use
{
	int type; //refers to above
	int script;
	int status; //refers to status effect index
	int duration;
	int cooldown;
	float mul;

	virtual void apply(rpgchar *user);
	use(int s) : type(USE_CONSUME), script(s), status(0), duration(0), cooldown(500), mul(1.0f) {}
	~use() {}
};

struct use_armour : use
{
	statreq reqs;
	int slots;
	int skill;

	void apply(rpgchar *user);
	use_armour(int s) : use(s), reqs(statreq()), slots(0), skill(SKILL_ARMOUR) {type = USE_ARMOUR;}
	~use_armour() {}
};

struct use_weapon : use_armour
{
	int range;
	int angle;
	int lifetime;
	int effect;
	int cost; //mana/items consumed in use
	int flags;
	int element;
	int ammo; //refers to item in ammotype vector<>, -ves use mana
	int target; //self, others, etc
	int radius;
	int kickback;
	int charge; //how long it takes to charge and the cooldown
	float basecharge; //additional charge, the charge meter only displays the two below -
	float mincharge, maxcharge; // minimum charge to fire and maximum possible charge
	float elasticity, speed;

	void apply(rpgchar *user);
	use_weapon(int s) : use_armour(s), range(256), angle(60), lifetime(10000), effect(0), cost(10), flags(P_DIST|P_TIME), element(ATTACK_FIRE), ammo(-1), target(T_SINGLE), radius(32), kickback(10), charge(0), basecharge(.5f), mincharge(.5f), maxcharge(1.0f), elasticity(0.8), speed(1.0f) {type = USE_WEAPON;}
	~use_weapon() {}
};

enum
{
	ITEM_CONSUMABLE = 0,
	ITEM_ARMOUR,
	ITEM_WEAPON,
	ITEM_AMMO,
	ITEM_MISC,
	ITEM_MAX
};

struct item_base
{

	enum
	{
		F_QUEST = 1<<0, //just helps identifying important items - should safe guards be implemented?
		F_CURSED = 1<<1, //item glues itself to the wearer - holy (allows removal), or dispelling magic is needed
		F_TRUEVAL = 1<<2, //only trades for base price, ie money
		F_MAX = 1 << 3
	};
	const char *name;
	const char *icon;
	const char *description;
	const char *mdl;

	int script;
	int type;
	int flags;
	int worth;
	int weight;

	int cursebase; //item to swap to should this item become (un)cursed

	vector<use *> uses;

	void transfer(item_base &o)
	{
		delete[] o.mdl;

		o = *this;

		if(name) o.name = newstring(name);
		if(icon) o.icon = newstring(icon);
		if(description) o.description = newstring(description);
		o.name = newstring(mdl);
	}

	item_base() : name(NULL), icon(NULL), description(NULL), mdl(newstring(DEFAULTMODEL)), script(1), type(ITEM_MISC), flags(0), worth(0), weight(0), cursebase(-1) {}
	~item_base()
	{
		delete[] name;
		delete[] icon;
		delete[] description;
		delete[] mdl;
		uses.deletecontents();
	}
};

struct rpgitem : rpgent, invstack
{
	void update();
	void applyeffects(statusgroup *g, float mag, bool instant, rpgent *owner);

	void resetmdl();
	void render(bool mainpass);
	const char *getname() const;
	vec blipcol() { return vec(0, .75, 1);}
	void hit(rpgent *attacker, use_weapon *weapon, use_weapon *ammo, float mul, vec dir);
	const int type() {return ENT_ITEM;}
	int getscript();
	int getitemcount(int i);

	rpgitem() {physent::type = ENT_INANIMATE;}
};

struct recipe
{
	vector<invstack> ingredients, // meat.. veggies.. seasoning... butane...
		catalysts, // portable gas stove
		product; // stew! YEEAAH

	void transfer(recipe &o)
	{
		o.ingredients.setsize(0);
		o.catalysts.setsize(0);
		o.product.setsize(0);

		loopv(ingredients) o.ingredients.add(ingredients[i]);
		loopv(catalysts) o.catalysts.add(catalysts[i]);
		loopv(product) o.product.add(product[i]);
	}

	recipe() {}
	~recipe() {}
};

struct object_base
{
	const char *mdl, *name;
	int flags, weight, script;
	vector<invstack *> inventory;

	void transfer(object_base &o)
	{
		delete[] o.mdl;
		delete[] o.name;

		o = *this;
		o.mdl = newstring(mdl);
		if(name) o.name = newstring(name);

		o.inventory.setsize(0);
		loopv(inventory)
			o.inventory.add(new invstack(*inventory[i]));
	}

	object_base() : mdl(newstring(DEFAULTMODEL)), name(NULL), flags(0), weight(100), script(0) {}
	~object_base() { inventory.deletecontents(); }
};

struct rpgobject : rpgent, object_base
{
	enum
	{
		TRIGGERED = 1 << 0,
		CONTAINER = 1 << 1,
		LOCKABLE  = 1 << 2,
		MOVABLE   = 1 << 3,
		PLATFORM  = 1 << 4,
		F_MAX     = 1 << 5
	};

	int lasttrigger;
	vector<rpgent *> stack;

	const int type() {return ENT_OBJECT;}

	void resetmdl();
	void update();
	void applyeffects(statusgroup *g, float mag, bool instant, rpgent *owner);
	void render(bool mainpass);
	int getscript() {return script;}
	void hit(rpgent *attacker, use_weapon *weapon, use_weapon *ammo, float mul, vec dir);
	int drop(int b, int q, bool spawn);
	const char *getname() const;
	int getitemcount(int i);

	rpgobject() : lasttrigger(lastmillis) {physent::type = ENT_INANIMATE;}
	~rpgobject() {}
};

struct ammotype
{
	const char *name;
	vector<int> items;

	void transfer(ammotype &a)
	{
		if(name) a.name = newstring(name);
		a.items = items;
	}

	ammotype() : name(NULL) {}
	~ammotype() { delete[] name; }
};

struct faction
{
	const char *name, *logo;
	vector<short> relations;

	void setrelation(int i, int f)
	{
		if(i < 0)
			return;
		if(i >= relations.length())
		{
			int olen = relations.length();
			relations.growbuf(i + 1);
			relations.advance(relations.capacity() - olen); // this would max it out to capacity

			loopi(relations.length()-olen)
			relations[i + olen] = 50; //default to neutral
		}

		relations[i] = f;
	}

	int getrelation(int i)
	{
		if(relations.inrange(i))
			return relations[i];

		return 50; //neutral
	}

	void transfer(faction &d)
	{
		if(name) d.name = newstring(name);
		if(logo) d.logo = newstring(logo);

		d.relations = relations;
	}

	faction() : name(NULL), logo(NULL) {}
	~faction()
	{
		delete[] name;
		delete[] logo;
	}
};

//directives are sorted by priority, the most important ones are done first.
//for example, self defense is high priority while fleeing is an even higher priority,
//directives also include rudimentary things such as picking stuff up
struct directive
{
	directive() {}
	virtual ~directive() {}

	virtual int priority()=0;
	virtual int type()=0;
	virtual bool update(rpgchar *d)=0; // returns false when executed/finished, some (such as follow) don't finish unless explicitly cleared

	static int compare(directive *a, directive *b)
	{
		return clamp(1, -1, a->priority() - b->priority());
	}
};

struct char_base
{
	const char *name;
	const char *mdl;
	stats base;
	vector<invstack *> inventory;
	int script, faction;

	void transfer(char_base &o)
	{
		delete[] o.mdl;
		delete[] o.name;

		o = *this;
		if(name) o.name = newstring(name);
		o.mdl = newstring(mdl);
		o.inventory.setsize(0);

		loopv(inventory) o.inventory.add(new invstack(*inventory[i]));
	}

	char_base() : name(NULL), mdl(newstring(DEFAULTMODEL)), base(stats()), script(0), faction(0) {}
	~char_base()
	{
		delete[] name;
		delete[] mdl;
		inventory.deletecontents();
	}
};

struct rpgchar : rpgent, char_base
{
	vector<equipment> equipped;

	float health, mana;

	int lastaction;
	int charge;
	bool primary, secondary;
	bool lastprimary, lastsecondary;

	use_weapon *attack;

	//AI specific stuff, the player can access them during (and only during) cutscenes

	rpgent *target;
	vec dest;
	vector<ushort> route;
	vector<directive *> directives;

	//AI functions

	void followroute();
	bool cansee(rpgent *d);

	///global
	void resetmdl();
	void update();
	void render(bool mainpass);
	const char *getname() const;
	const int type() {return ENT_CHAR;}
	int getscript() {return script;}

	///character/AI
	void givexp(int xp) {base.givexp(xp);}
	void equip(int i, int u = 0);
	void dequip(int i, int slots = 0);
	void die(rpgent *killer);
	void respawn();
	void hit(rpgent *attacker, use_weapon *weapon, use_weapon *ammo, float mul, vec dir);
	void applyeffects(statusgroup *g, float mag, bool instant, rpgent *owner);

	///inventory
	int drop(int b, int q, bool spawn);
	bool pickup(rpgent *item);
	void uncurse();
	void curse();
	int getitemcount(int i);

	//character specific stuff
	bool checkammo(equipment &eq, equipment *quiver, bool remove = false);
	void doattack(equipment *eleft, equipment *eright, equipment *quiver);

	void cleanup();

	rpgchar() : lastaction(0), charge(0), primary(false), secondary(false), lastprimary(false), lastsecondary(false), attack(NULL), target(NULL)
	{
		health = base.getmaxhp();
		mana = base.getmaxmp();
	}
	~rpgchar()
	{
		directives.deletecontents();
	}
};

enum
{
	ACTION_TELEPORT = 0,
	ACTION_SPAWN,
	ACTION_SCRIPT
};

struct action
{
	virtual void exec()=0;
	virtual const int type()=0;

	action() {}
	virtual ~action() {}
};

//actions to execute when the player enters a map
struct action_teleport : action
{
	rpgent *ent;
	int dest;

	void exec();
	const int type() {return ACTION_TELEPORT;}

	action_teleport(rpgent *e, int d) : ent(e), dest(d) {}
	~action_teleport() {}
};

struct action_spawn : action
{
	// entity tag, entity type, base id
	int tag, ent, id, amount, qty;

	void exec();
	const int type() {return ACTION_SPAWN;}

	action_spawn(int ta, int en, int i, int amt, int q) : tag(ta), ent(en), id(i), amount(amt), qty(q) {}
	~action_spawn()	{}
};

struct action_script : action
{
	const char *script;

	void exec() { if(script) execute(script); }
	const int type() {return ACTION_SCRIPT;}

	action_script(const char *s) : script(newstring(s)) {}
	~action_script()
	{
		delete[] script;
	}
};

struct blip
{
	vec o;
	vec col;
	int size;
	//Texture *icon;

	blip(vec _o, vec _c = vec(1,1,1), int _s = 8) : o(_o), col(_c), size(_s) {}
	~blip() {}
};

struct mapinfo
{
	//the only reason name exists, is because it's too much effort to clear the keys, before deleting the tables
	const char *name;
	vector<rpgent *> objs;
	int script;
	bool loaded;
	vector<action *> loadactions;
	vector<projectile *> projs;
	vector<areaeffect *> aeffects;
	vector<blip> blips;

	mapinfo() : name(NULL), script(0), loaded(false) {}
	~mapinfo()
	{
		delete[] name;
		objs.deletecontents();
		loadactions.deletecontents();
		projs.deletecontents();
		aeffects.deletecontents();
	}
};

//this is compatible with the FPSGAME waypoints
struct waypoint
{
	waypoint *parent;
	vec o;
	vector<ushort> links;
	int score;

	waypoint() {}
	waypoint(const vec &o) : parent(NULL), o(o), score(-1) {}
};

#ifdef NO_DEBUG

#define DEBUG_WORLD   (false)
#define DEBUG_ENT     (false)
#define DEBUG_CONF    (false)
#define DEBUG_PROJ    (false)
#define DEBUG_SCRIPT  (false)
#define DEBUG_AI      (false)
#define DEBUG_IO      (false)
#define DEBUG_CAMERA  (false)

#else

#define DEBUG_WORLD   (game::debug & (1 << 0))
#define DEBUG_ENT     (game::debug & (1 << 1))
#define DEBUG_CONF    (game::debug & (1 << 2))
#define DEBUG_PROJ    (game::debug & (1 << 3))
#define DEBUG_SCRIPT  (game::debug & (1 << 4))
#define DEBUG_AI      (game::debug & (1 << 5))
#define DEBUG_IO      (game::debug & (1 << 6))
#define DEBUG_CAMERA  (game::debug & (1 << 7))

#endif
#define DEBUG_MAX ((1 << 8) - 1)

namespace rpgeffect
{
	enum
	{
		PROJ,
		TRAIL, // a trail
		TRAIL_SINGLE, //a trail which only emits once
		DEATH, // death cariant which only emits once
		DEATH_PROLONG // death variant which is intended for a delay that takes a while
	};
	extern void drawsphere(effect *e, vec &o, float radius, float size = 1, int type = PROJ, int elapse = 17);
	extern void drawsplash(effect *e, vec &o, float radius, float size = 1, int type = PROJ, int elapse = 17);
	extern bool drawline(effect *e, vec &from, vec &to, float size = 1, int type = PROJ, int elapse = 17);
	extern void drawaura(effect *e, rpgent *d, float size = 1, int type = PROJ, int elapse = 17);
	extern void drawcircle(effect *e, vec &o, vec dir, vec &axis, int angle, float radius, float size = 1, int type = PROJ, int elapse = 17);
	extern void drawcircle(rpgent *d, use_weapon *wep, float size = 1, int type = PROJ, int elapse = 17);
}

namespace ai
{
	extern vector<waypoint> waypoints;
	extern void loadwaypoints(const char *name, bool msg = false);
	extern void savewaypoints(const char *name);
	extern void findroute(int from, int to, vector<ushort> &route);
	extern waypoint *closestwaypoint(const vec &o);
	extern void renderwaypoints();
	extern void trydrop();
	extern void clearwaypoints();
}

namespace game
{
	extern vector<script *> scripts; ///scripts, includes dialogue
	extern vector<effect *> effects; ///pretty particle effects for spells and stuff
	extern vector<statusgroup *> statuses; ///TODO find a better name - status effect definitions to transfer onto victims
	extern vector<item_base *> items;
	extern vector<ammotype *> ammotypes;
	extern vector<char_base *> chars;
	extern vector<faction *> factions;
	extern vector<int> variables; //global variables
	extern vector<const char *> tips; //tips
	extern vector<object_base *> objects;
	extern vector<mapscript *> mapscripts;
	extern vector<recipe *> recipes;

	extern int debug;
	extern bool connected;
	extern bool transfer;
	extern rpgchar *player1;
	extern hashtable<const char *, mapinfo> *mapdata;
	extern mapinfo *curmap;

	extern void cleangame();
	extern mapinfo *accessmap(const char *name);
	extern bool cansave();
	extern bool intersect(rpgent *d, const vec &from, const vec &to, float &dist);
	extern rpgent *intersectclosest(const vec &from, const vec &to, rpgent *at, float &bestdist, float maxdist = 1e16);
}

namespace rpgscript
{
	extern rpgent *entsel;
	extern rpgent *hover;
	extern script *chatscript;

	extern void update();
	extern void clean();
	extern void doitemscript(invstack *invokee, rpgent *invoker, int slot);
	extern void doentscript(rpgent *invokee, rpgent *invoker, int slot);
	extern void domapscript(mapinfo *map, rpgent *victim, int slot);
}

namespace entities
{
	extern vector<extentity *> ents;
	extern vector<int> intents;
	extern void spawn(extentity &e, int ind, int type, int qty);
	extern void teleport(rpgent *d, int dest);
	extern void genentlist();
	extern void touchents(rpgent *d);
}

namespace camera
{
	extern bool cutscene;
	extern void cleanup(bool clear = false);
	extern void update();
	extern void render(int w, int h);

	extern physent *attached;
}

namespace rpggui
{
	bool open();
	void forcegui();
	void hotkey(int n);
}

enum
{
	S_JUMP = 0, S_LAND, S_RIFLE, S_TELEPORT, S_SPLASH1, S_SPLASH2, S_CG,
	S_RLFIRE, S_RUMBLE, S_JUMPPAD, S_WEAPLOAD, S_ITEMAMMO, S_ITEMHEALTH,
	S_ITEMARMOUR, S_ITEMPUP, S_ITEMSPAWN,  S_NOAMMO, S_PUPOUT,
	S_PAIN1, S_PAIN2, S_PAIN3, S_PAIN4, S_PAIN5, S_PAIN6,
	S_DIE1, S_DIE2,
	S_FLAUNCH, S_FEXPLODE,
	S_SG, S_PUNCH1,
	S_GRUNT1, S_GRUNT2, S_RLHIT,
	S_PAINO,
	S_PAINR, S_DEATHR,
	S_PAINE, S_DEATHE,
	S_PAINS, S_DEATHS,
	S_PAINB, S_DEATHB,
	S_PAINP, S_PIGGR2,
	S_PAINH, S_DEATHH,
	S_PAIND, S_DEATHD,
	S_PIGR1, S_ICEBALL, S_SLIMEBALL, S_PISTOL,

	S_V_BASECAP, S_V_BASELOST,
	S_V_FIGHT,
	S_V_BOOST, S_V_BOOST10,
	S_V_QUAD, S_V_QUAD10,
	S_V_RESPAWNPOINT,

	S_FLAGPICKUP,
	S_FLAGDROP,
	S_FLAGRETURN,
	S_FLAGSCORE,
	S_FLAGRESET,

	S_BURN,
	S_CHAINSAW_ATTACK,
	S_CHAINSAW_IDLE,

	S_HIT
};
#endif