//*****************************************************************************
//
// File:    DnaSequenceTree.java
// Package: edu.rit.phyl.pars
// Unit:    Class edu.rit.phyl.pars.DnaSequenceTree
//
// This Java source file is copyright (C) 2007 by Alan Kaminsky. All rights
// reserved. For further information, contact the author, Alan Kaminsky, at
// ark@cs.rit.edu.
//
// This Java source file is part of the Parallel Java Library ("PJ"). PJ is free
// software; you can redistribute it and/or modify it under the terms of the GNU
// General Public License as published by the Free Software Foundation; either
// version 3 of the License, or (at your option) any later version.
//
// PJ is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
// A PARTICULAR PURPOSE. See the GNU General Public License for more details.
//
// A copy of the GNU General Public License is provided in the file gpl.txt. You
// may also obtain a copy of the GNU General Public License on the World Wide
// Web at http://www.gnu.org/licenses/gpl.html.
//
//******************************************************************************

package edu.rit.phyl.pars;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;

/**
 * Class DnaSequenceTree provides a rooted bifurcated tree of {@linkplain
 * DnaSequence}s. Each tree node contains a DNA sequence, namely a sequence of
 * sites and an integer score. Each tree node has references to the two child
 * nodes and the parent node. There is also an integer score associated with the
 * whole tree.
 *
 * @author  Alan Kaminsky
 * @version 20-Dec-2007
 */
public class DnaSequenceTree
	implements Externalizable
	{

// Hidden data members.

	private static final long serialVersionUID = -1405663180316351649L;

	// Maximum number of tip nodes.
	int myTipCapacity;

	// Maximum number of nodes.
	int myNodeCapacity;

	// Actual number of tip nodes.
	int myTipCount;

	// Actual number of nodes.
	int myNodeCount;

	// The nodes themselves.
	Node[] myNode;

	// Index of the root node, or -1 if tree is empty.
	int myRoot;

	// The overall tree score.
	int myScore;

	// 128 bytes of extra padding to avert cache interference.
	private transient long p0, p1, p2, p3, p4, p5, p6, p7;
	private transient long p8, p9, pa, pb, pc, pd, pe, pf;

	// Default stringifier.
	static Stringifier theDefaultStringifier = new Stringifier()
		{
		public String toString
			(Node node)
			{
			StringBuilder buf = new StringBuilder();
			buf.append (node.mySequence);
			buf.append (':');
			buf.append (node.mySequence.myScore);
			return buf.toString();
			}
		};

// Exported helper classes.

	/**
	 * Class Node provides one node in a {@linkplain DnaSequenceTree}.
	 *
	 * @author  Alan Kaminsky
	 * @version 22-Apr-2007
	 */
	public class Node
		implements Externalizable
		{
		private static final long serialVersionUID = 8525724559158098803L;

		// This node's DNA sequence.
		private DnaSequence mySequence = new DnaSequence();

		// This node's parent node index, or -1 if this node is the root node.
		private int myParent = -1;

		// This node's first child node index, or -1 if this node is a tip node.
		private int myChild1 = -1;

		// This node's second child node index, or -1 if this node is a tip
		// node.
		private int myChild2 = -1;

		// 128 bytes of extra padding to avert cache interference.
		private transient long p0, p1, p2, p3, p4, p5, p6, p7;
		private transient long p8, p9, pa, pb, pc, pd, pe, pf;

		/**
		 * Create a new node.
		 */
		private Node()
			{
			}

		/**
		 * Create a new node that is a copy of the given node.
		 *
		 * @param  theNode  Node to copy.
		 */
		private Node
			(Node theNode)
			{
			copy (theNode);
			}

		/**
		 * Make this node be a copy of the given node.
		 *
		 * @param  theNode  Node to copy.
		 */
		private void copy
			(Node theNode)
			{
			this.mySequence.copy (theNode.mySequence);
			this.myParent = theNode.myParent;
			this.myChild1 = theNode.myChild1;
			this.myChild2 = theNode.myChild2;
			}

		/**
		 * Get this node's DNA sequence.
		 *
		 * @return  DNA sequence.
		 */
		public DnaSequence sequence()
			{
			return mySequence;
			}

		/**
		 * Get this node's parent node.
		 *
		 * @return  Parent node, or null if this node is the root node.
		 */
		public Node parent()
			{
			return myParent == -1 ? null : myNode[myParent];
			}

		/**
		 * Get this node's first child node.
		 *
		 * @return  First child node, or null if this node is a tip node.
		 */
		public Node child1()
			{
			return myChild1 == -1 ? null : myNode[myChild1];
			}

		/**
		 * Get this node's second child node.
		 *
		 * @return  Second child node, or null if this node is a tip node.
		 */
		public Node child2()
			{
			return myChild2 == -1 ? null : myNode[myChild2];
			}

		/**
		 * Write this node to the given object output stream.
		 *
		 * @param  out  Object output stream.
		 *
		 * @exception  IOException
		 *     Thrown if an I/O error occurred.
		 */
		public void writeExternal
			(ObjectOutput out)
			throws IOException
			{
			mySequence.writeExternal (out);
			out.writeInt (myParent);
			out.writeInt (myChild1);
			out.writeInt (myChild2);
			}

		/**
		 * Read this node from the given object input stream.
		 *
		 * @param  in  Object input stream.
		 *
		 * @exception  IOException
		 *     Thrown if an I/O error occurred.
		 * @exception  ClassNotFoundException
		 *     Thrown if a class needed to deserialize this node cannot be
		 *     found.
		 */
		public void readExternal
			(ObjectInput in)
			throws IOException, ClassNotFoundException
			{
			mySequence.readExternal (in);
			myParent = in.readInt();
			myChild1 = in.readInt();
			myChild2 = in.readInt();
			}
		}

	/**
	 * Interface Stringifier specifies the interface for an object that converts
	 * a DNA sequence tree node to a string.
	 *
	 * @author  Alan Kaminsky
	 * @version 21-Apr-2007
	 */
	public static interface Stringifier
		{
		/**
		 * Convert the given node to a string.
		 *
		 * @param  node  Node.
		 *
		 * @return  String version of <TT>node</TT>.
		 */
		public String toString
			(Node node);
		}

// Exported constructors.

	/**
	 * Construct a new DNA sequence tree with a capacity of 1. The tree is
	 * initially empty (has no nodes), and the tree's score is initially 0.
	 */
	public DnaSequenceTree()
		{
		this (1);
		}

	/**
	 * Construct a new DNA sequence tree with the given capacity. The tree is
	 * initially empty (has no nodes), and the tree's score is initially 0.
	 *
	 * @param  N  Capacity (maximum number of tip nodes).
	 *
	 * @exception  NegativeArraySizeException
	 *     (unchecked exception) Thrown if <TT>N</TT> &lt; 1.
	 */
	public DnaSequenceTree
		(int N)
		{
		allocate (N);
		}

	private void allocate
		(int N)
		{
		myTipCapacity = N;
		myNodeCapacity = 2*N-1;
		myTipCount = 0;
		myNodeCount = 0;
		myNode = new Node [myNodeCapacity];
		for (int i = 0; i < myNodeCapacity; ++ i)
			{
			myNode[i] = new Node();
			}
		myRoot = -1;
		myScore = 0;
		}

	/**
	 * Construct a new DNA sequence tree that is a copy of the given tree.
	 *
	 * @param  tree  DNA sequence tree to copy.
	 *
	 * @exception  NullPointerException
	 *     (unchecked exception) Thrown if <TT>tree</TT> is null.
	 */
	public DnaSequenceTree
		(DnaSequenceTree tree)
		{
		allocate (tree.myTipCapacity);
		copy (tree);
		}

// Exported operations.

	/**
	 * Get this DNA sequence tree's tip capacity.
	 *
	 * @return  Tip capacity (maximum number of tip nodes).
	 */
	public int tipCapacity()
		{
		return myTipCapacity;
		}

	/**
	 * Get this DNA sequence tree's node capacity. This includes tip nodes and
	 * interior nodes.
	 *
	 * @return  Node capacity (maximum number of nodes).
	 */
	public int nodeCapacity()
		{
		return myNodeCapacity;
		}

	/**
	 * Get this DNA sequence tree's tip count.
	 *
	 * @return  Tip count (actual number of tip nodes).
	 */
	public int tipCount()
		{
		return myTipCount;
		}

	/**
	 * Get this DNA sequence tree's node count. This includes tip nodes and
	 * interior nodes.
	 *
	 * @return  Node count (actual number of nodes).
	 */
	public int nodeCount()
		{
		return myNodeCount;
		}

	/**
	 * Determine if this DNA sequence tree is empty.
	 *
	 * @return  True if this DNA sequence tree is empty, false otherwise.
	 */
	public boolean isEmpty()
		{
		return myTipCount == 0;
		}

	/**
	 * Determine if this DNA sequence tree is full.
	 *
	 * @return  True if this DNA sequence tree is full, false otherwise.
	 */
	public boolean isFull()
		{
		return myTipCount == myTipCapacity;
		}

	/**
	 * Get this DNA sequence tree's root node.
	 *
	 * @return  Root node, or null if this tree is empty.
	 */
	public Node root()
		{
		return myRoot == -1 ? null : myNode[myRoot];
		}

	/**
	 * Get this DNA sequence tree's score.
	 *
	 * @return  Score.
	 */
	public int score()
		{
		return myScore;
		}

	/**
	 * Set this DNA sequence tree's score.
	 *
	 * @param  score  Score.
	 */
	public void score
		(int score)
		{
		myScore = score;
		}

	/**
	 * Clear this DNA sequence tree. Afterwards, the tree is empty (has no
	 * nodes), and the tree's score is 0.
	 */
	public void clear()
		{
		myTipCount = 0;
		myNodeCount = 0;
		myRoot = -1;
		myScore = 0;
		}

	/**
	 * Make this DNA sequence tree be a copy of the given tree.
	 *
	 * @param  tree  DNA sequence tree to copy.
	 *
	 * @exception  NullPointerException
	 *     (unchecked exception) Thrown if <TT>tree</TT> is null.
	 */
	public void copy
		(DnaSequenceTree tree)
		{
		if (this.myTipCapacity != tree.myTipCapacity)
			{
			allocate (tree.myTipCapacity);
			}
		this.myTipCount = tree.myTipCount;
		this.myNodeCount = tree.myNodeCount;
		for (int i = 0; i < this.myNodeCount; ++ i)
			{
			this.myNode[i].copy (tree.myNode[i]);
			}
		this.myRoot = tree.myRoot;
		this.myScore = tree.myScore;
		}

	/**
	 * Add a tip node to this DNA sequence tree. Different index values yield
	 * different tree topologies. The index must be in the range 0 ..
	 * <TT>nodeCount()</TT>-1.
	 * <P>
	 * The <TT>addTipNode()</TT> method adds two nodes to the tree: the new tip
	 * node itself, and the new tip node's parent node. The parent node's first
	 * child is one of the existing tree nodes, as determined by the
	 * <TT>index</TT> argument. The parent node's second child is the new tip
	 * node. The new tip node's DNA sequence is a copy of the <TT>seq</TT>
	 * argument. The parent node's DNA sequence is not set to anything. No other
	 * node's DNA sequence is altered. The new tip node is returned.
	 * <P>
	 * If this DNA sequence tree is empty, the <TT>index</TT> argument is
	 * ignored, and only one node is added to the tree, namely the new tip node.
	 * The new tip node is returned.
	 *
	 * @param  index  Index that determines the tree topology.
	 * @param  seq    DNA sequence for the new tip node.
	 *
	 * @return  New tip node.
	 *
	 * @exception  NullPointerException
	 *     (unchecked exception) Thrown if <TT>seq</TT> is null.
	 * @exception  IllegalStateException
	 *     (unchecked exception) Thrown if this DNA sequence tree is full.
	 * @exception  IndexOutOfBoundsException
	 *     (unchecked exception) Thrown if <TT>index</TT> is not in the range 0
	 *     .. <TT>nodeCount()</TT>-1.
	 */
	public Node addTipNode
		(int index,
		 DnaSequence seq)
		{
		Node tip, tipparent, child, grandparent;
		int tipindex, tipparentindex;

		// Special case if tree is empty.
		if (myTipCount == 0)
			{
			// Get tip node.
			tip = myNode[0];

			// Fill in tip node.
			tip.mySequence.copy (seq);
			tip.myParent = -1;
			tip.myChild1 = -1;
			tip.myChild2 = -1;

			// Root is the tip node.
			myRoot = 0;

			// Update counts.
			myTipCount = 1;
			myNodeCount = 1;
			}

		// Normal case if tree is not empty.
		else
			{
			// Verify preconditions.
			if (myTipCount == myTipCapacity)
				{
				throw new IllegalStateException
					("DnaSequenceTree.addTipNode(): tree full");
				}
			if (0 > index || index >= myNodeCount)
				{
				throw new IndexOutOfBoundsException
					("DnaSequenceTree.addTipNode(): index = " + index +
					 " out of bounds");
				}

			// Get tip, tip parent, and child nodes.
			tipindex = myNodeCount;
			tipparentindex = myNodeCount+1;
			tip = myNode[tipindex];
			tipparent = myNode[tipparentindex];
			child = myNode[index];

			// Fill in tip node.
			tip.mySequence.copy (seq);
			tip.myParent = tipparentindex;
			tip.myChild1 = -1;
			tip.myChild2 = -1;

			// Fill in parent node.
			tipparent.myParent = child.myParent;
			tipparent.myChild1 = index;
			tipparent.myChild2 = tipindex;

			// Splice parent node into tree.
			child.myParent = tipparentindex;
			if (tipparent.myParent == -1)
				{
				myRoot = tipparentindex;
				}
			else
				{
				grandparent = myNode[tipparent.myParent];
				if (grandparent.myChild1 == index)
					{
					grandparent.myChild1 = tipparentindex;
					}
				else
					{
					grandparent.myChild2 = tipparentindex;
					}
				}

			// Update counts.
			myTipCount += 1;
			myNodeCount += 2;
			}

		return tip;
		}

	/**
	 * Update the parsimony score of this tree using the Fitch algorithm.
	 * It is assumed that the <TT>tip</TT> has just been added as a tip node to
	 * this tree. Afterwards, each tip node's DNA sequence and score are
	 * unchanged; each interior node's DNA sequence is the combination of the
	 * two child nodes' DNA sequences according to the Fitch algorithm; each
	 * interior node's score is the number of state changes due to that node;
	 * and the tree's overall score is the total number of state changes.
	 *
	 * @param  tip   Tip node.
	 */
	public void updateFitchScore
		(DnaSequenceTree.Node tip)
		{
		// Fill in parent of tip node, if any.
		DnaSequenceTree.Node node = tip.parent();
		if (node != null)
			{
			node.mySequence.setFitchAncestor
				(node.child1().mySequence,
				 node.child2().mySequence);
			myScore += node.mySequence.myScore;
			node = node.parent();
			}

		// Update further parent nodes until the root is encountered.
		while (node != null)
			{
			// Get old number of state changes.
			int nChangesOld = node.mySequence.myScore;

			// Combine child nodes and compute new number of state changes.
			node.mySequence.setFitchAncestor
				(node.child1().mySequence,
				 node.child2().mySequence);

			// Update scores.
			int nChangesNew = node.mySequence.myScore;
			myScore = myScore - nChangesOld + nChangesNew;

			node = node.parent();
			}
		}

	/**
	 * Write this DNA sequence tree to the given object output stream.
	 *
	 * @param  out  Object output stream.
	 *
	 * @exception  IOException
	 *     Thrown if an I/O error occurred.
	 */
	public void writeExternal
		(ObjectOutput out)
		throws IOException
		{
		out.writeInt (myTipCapacity);
		out.writeInt (myTipCount);
		for (int i = 0; i < myNodeCount; ++ i)
			{
			myNode[i].writeExternal (out);
			}
		out.writeInt (myRoot);
		out.writeInt (myScore);
		}

	/**
	 * Read this DNA sequence tree from the given object input stream.
	 *
	 * @param  in  Object input stream.
	 *
	 * @exception  IOException
	 *     Thrown if an I/O error occurred.
	 * @exception  ClassNotFoundException
	 *     Thrown if a class needed to deserialize this DNA sequence tree cannot
	 *     be found.
	 */
	public void readExternal
		(ObjectInput in)
		throws IOException, ClassNotFoundException
		{
		int N = in.readInt();
		if (myTipCapacity != N) allocate (N);
		myTipCount = in.readInt();
		myNodeCount = 2*myTipCount-1;
		for (int i = 0; i < myNodeCount; ++ i)
			{
			myNode[i].readExternal (in);
			}
		myRoot = in.readInt();
		myScore = in.readInt();
		}

	/**
	 * Returns a string version of this DNA sequence tree. The returned string
	 * is in Newick Standard format.
	 * <P>
	 * Each tip node is converted to a string using the default stringifier. The
	 * initial default stringifier converts a node to a string which is the
	 * node's DNA sequence followed by its score; for example:
	 * <TT>"AGCT:42"</TT>. To change the default stringifier, use the
	 * <TT>defaultStringifier()</TT> method.
	 *
	 * @return  String version of this tree.
	 */
	public String toString()
		{
		return toString (theDefaultStringifier);
		}

	/**
	 * Returns a string version of this DNA sequence tree using the given
	 * stringifier. The returned string is in Newick Standard format.
	 * <P>
	 * Each tip node is converted to a string using <TT>stringifier</TT>.
	 *
	 * @param  stringifier  Stringifier.
	 *
	 * @return  String version of this tree.
	 */
	public String toString
		(Stringifier stringifier)
		{
		StringBuilder buf = new StringBuilder();
		if (myTipCount == 0)
			{
			buf.append ('(');
			buf.append (')');
			}
		else if (myTipCount == 1)
			{
			buf.append ('(');
			buf.append (stringifier.toString (myNode[0]));
			buf.append (')');
			}
		else
			{
			toString (stringifier, buf, myRoot);
			}
		return buf.toString();
		}

	private void toString
		(Stringifier stringifier,
		 StringBuilder buf,
		 int index)
		{
		Node node = myNode[index];
		if (node.myChild1 == -1)
			{
			buf.append (stringifier.toString (node));
			}
		else
			{
			buf.append ('(');
			toString (stringifier, buf, node.myChild1);
			buf.append (',');
			toString (stringifier, buf, node.myChild2);
			buf.append (')');
			}
		}

	/**
	 * Get the default stringifier.
	 *
	 * @return  Default stringifier.
	 */
	public static Stringifier defaultStringifier()
		{
		return theDefaultStringifier;
		}

	/**
	 * Set the default stringifier.
	 *
	 * @param  stringifier  Default stringifier.
	 *
	 * @exception  NullPointerException
	 *     (unchecked exception) Thrown if <TT>stringifier</TT> is null.
	 */
	public static void defaultStringifier
		(Stringifier stringifier)
		{
		if (stringifier == null)
			{
			throw new NullPointerException
				("DnaSequenceTree.defaultStringifier(): stringifier is null");
			}
		theDefaultStringifier = stringifier;
		}

	/**
	 * Dump this DNA sequence tree to the standard output. This method is
	 * intended for debugging.
	 */
	public void dump()
		{
		System.out.print ("DNA sequence tree, myTipCapacity=");
		System.out.print (myTipCapacity);
		System.out.print (", myNodeCapacity=");
		System.out.print (myNodeCapacity);
		System.out.print (", myTipCount=");
		System.out.print (myTipCount);
		System.out.print (", myNodeCount=");
		System.out.print (myNodeCount);
		System.out.print (", myRoot=");
		System.out.print (myRoot);
		System.out.print (", myScore=");
		System.out.print (myScore);
		System.out.println();
		if (root() != null) dumpnode (root(), 1);
		}

	private void dumpnode
		(Node node,
		 int level)
		{
		for (int i = 0; i < level; ++ i)
			{
			System.out.print (".\t");
			}
		System.out.print (node.mySequence.myScore);
		System.out.print (' ');
		System.out.print (node.mySequence);
		System.out.println();
		if (node.child1() != null) dumpnode (node.child1(), level+1);
		if (node.child2() != null) dumpnode (node.child2(), level+1);
		}

	}
