//******************************************************************************
//
// File:    DrawTree.java
// Package: edu.rit.phyl
// Unit:    Class edu.rit.phyl.DrawTree
//
// This Java source file is copyright (C) 2007 by Alan Kaminsky. All rights
// reserved. For further information, contact the author, Alan Kaminsky, at
// ark@cs.rit.edu.
//
// This Java source file is part of the Parallel Java Library ("PJ"). PJ is free
// software; you can redistribute it and/or modify it under the terms of the GNU
// General Public License as published by the Free Software Foundation; either
// version 3 of the License, or (at your option) any later version.
//
// PJ is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
// A PARTICULAR PURPOSE. See the GNU General Public License for more details.
//
// A copy of the GNU General Public License is provided in the file gpl.txt. You
// may also obtain a copy of the GNU General Public License on the World Wide
// Web at http://www.gnu.org/licenses/gpl.html.
//
//******************************************************************************

package edu.rit.phyl;

import edu.rit.draw.Drawing;

import edu.rit.draw.item.Line;
import edu.rit.draw.item.Point;
import edu.rit.draw.item.Text;

import java.io.File;

import java.util.LinkedList;

/**
 * Class DrawTree is a program that draws a phylogenetic tree. The tree is
 * specified on the command line in a subset of Newick Standard format. The
 * drawing is stored in a file specified on the command line as a serialized
 * {@linkplain edu.rit.draw.Drawing} object. The {@linkplain View} program can
 * be used to view the drawing file and save it in an image file in several
 * formats.
 * <P>
 * Usage: java edu.rit.phyl.DrawTree "<I>tree</I>" <I>file</I>
 * <BR><I>tree</I> = Tree in Newick Standard format
 * <BR><I>file</I> = Drawing file name
 * <P>
 * <B>Newick Standard format.</B> The subset of Newick Standard format the
 * DrawTree program supports is described by this BNF syntax. Nonterminal
 * symbols are in <I>italic</I> font, terminal symbols are in
 * <TT>typewriter</TT> font.
 * <P>
 * <I>tree</I> ::= <I>node</I>
 * <BR><I>node</I> ::= <TT>(</TT> <I>childList</I> <TT>)</TT>
 * <BR><I>childList</I> ::= <I>child</I> | <I>child</I> <TT>,</TT> <I>childList</I>
 * <BR><I>child</I> ::= <I>tip</I> | <I>node</I>
 * <BR><I>tip</I> ::= Any sequence of characters except <TT>(</TT> <TT>)</TT> <TT>,</TT>
 * <P>
 * Here is an example command to draw a tree and store it in the file <TT>"tree.dwg"</TT>:
 * <P>
 * <TT>java edu.rit.phyl.DrawTree "(Gibbon,(Orangutan,(Gorilla,(Chimp,Human))))" tree.dwg</TT>
 * <P>
 * Here is the command to view the tree drawing:
 * <P>
 * <TT>java edu.rit.draw.View tree.dwg</TT>
 * <P>
 * Here is the tree drawing saved as a PNG image:
 * <P>
 * <IMG SRC="doc-files/tree.png">
 *
 * @author  Alan Kaminsky
 * @version 13-Apr-2007
 */
public class DrawTree
	{

// Prevent construction.

	private DrawTree()
		{
		}

// Constants.

	// Vertical distance between tip strings.
	static final double V = 15.0;

	// Length of lines from a node to its descendents.
	static final double H = 36.0;

	// Gap from line to tip text.
	static final double GAP = 6.0;

// Global variables.

	// Tree string in Newick Standard format, and index of next character to
	// parse.
	static String treestring;
	static char[] tree;
	static int index = 0;

	// Number of tips.
	static int tipCount = 0;

// Main program.

	/**
	 * Main program.
	 */
	public static void main
		(String[] args)
		throws Exception
		{
		// Parse command line arguments.
		if (args.length != 2) usage();
		treestring = args[0];
		tree = args[0].toCharArray();
		File file = new File (args[1]);

		// Parse the tree, building up the drawing as it goes.
		Point root = parseNode();
		if (index != tree.length) syntaxError ("Extra characters");

		// Add final line to root.
		new Line().to(root).hby(-H).add();

		// Write drawing object into output file.
		Drawing.write (file);
		}

// Hidden operations.

	/**
	 * Parse a node.
	 *
	 * @return  Point to which to attach line from parent.
	 */
	private static Point parseNode()
		{
		if (currentChar() != '(') syntaxError ("( expected");
		++ index;
		Point p = parseChildList();
		if (currentChar() != ')') syntaxError (") expected");
		++ index;
		return p;
		}

	/**
	 * Parse a child list.
	 *
	 * @return  Point to which to attach line from parent.
	 */
	private static Point parseChildList()
		{
		LinkedList<Point> childList = new LinkedList<Point>();
		double x = 0.0;
		double ymin = 0.0;
		double ymax = 0.0;
		for (;;)
			{
			Point p = parseChild();
			childList.add (p);
			x = Math.min (x, p.x());
			if (currentChar() != ',') break;
			++ index;
			}
		x -= H;
		for (Point p : childList)
			{
			new Line().to(p).hto(x).add();
			}
		ymin = childList.get(0).y();
		ymax = childList.get(childList.size()-1).y();
		new Line().to(x,ymin).to(x,ymax).add();
		return new Point (x, (ymin+ymax)/2);
		}

	/**
	 * Parse a child.
	 *
	 * @return  Point to which to attach line from parent.
	 */
	private static Point parseChild()
		{
		if (currentChar() == '(')
			{
			return parseNode();
			}
		else
			{
			return parseTip();
			}
		}

	/**
	 * Parse a tip.
	 *
	 * @return  Point to which to attach line from parent.
	 */
	private static Point parseTip()
		{
		StringBuilder tip = new StringBuilder();
		char c;
		for (;;)
			{
			c = currentChar();
			if (c == '(' || c == ')' || c == ',') break;
			tip.append (c);
			++ index;
			}
		Point p = new Point (0, tipCount * V);
		++ tipCount;
		String s = tip.toString();
		if (s.length() > 0) new Text().text(s).w(p.e(GAP)).add();
		return p;
		}

	/**
	 * Get the current character being parsed, checking for end-of-string.
	 *
	 * @return  Character.
	 */
	private static char currentChar()
		{
		if (index >= tree.length) syntaxError ("Incomplete tree");
		return tree[index];
		}

	/**
	 * Report a syntax error.
	 *
	 * @param  msg  Error message.
	 */
	private static void syntaxError
		(String msg)
		{
		System.err.println ("Syntax error: " + msg);
		System.err.println (treestring);
		for (int i = 0; i < index; ++ i) System.err.print (' ');
		System.err.println ('^');
		System.exit (1);
		}

	/**
	 * Print a usage message and exit.
	 */
	private static void usage()
		{
		System.err.println ("Usage: java edu.rit.phyl.DrawTree \"<tree>\" <file>");
		System.err.println ("<tree> = Tree in Newick Standard format");
		System.err.println ("<file> = Drawing file name");
		System.exit (1);
		}

	}
