/***************************************************************************
 *   Copyright (C) 2003-2005 by Kevin Hessels                              *
 *   khessels@shaw.ca                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qdir.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qlayout.h>
#include <qregexp.h>
#include <qtooltip.h>


#include <kaboutapplication.h>
#include <kaboutdata.h>
#include <kaction.h>
#include <kactioncollection.h>
#include <kapplication.h>
#include <kdebug.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kpassivepopup.h>
#include <kpopupmenu.h>
#include <krun.h>
#include <kstandarddirs.h>
#include <kio/job.h>
#include <kio/jobclasses.h>


#include "kfolding.h"
#include "kfoldingconfigdata.h"
#include "kfoldingprocess.h"
#include "kfoldingprogresswidget.h"
#include "kfoldingqueuedialog.h"
#include "kfoldingsettingsdialog.h"
#include "kfoldingunitdialog.h"


kfolding::kfolding(const QString& configFile, Type type, int actions, QWidget *parent, const char *name)
    : KPanelApplet(configFile, type, actions, parent, name),
      _lastPercent( 0 ),
      _lastUnit( "" ),
      _actions( this, this ),
      _settingsDialogue( 0L ),
      _unitDialogue( 0L )
{
	setBackgroundOrigin( AncestorOrigin );
	resize( 1, 1 );
	
	// Get the current application configuration handle
	_configFileName =  configFile.latin1();
	_cfg = new kfoldingConfigData( config() );
	_cfg->load();
	
	_process = new kfoldingProcess( this, "kfoldingProcess", _cfg );
	_progressWidget = new kfoldingProgressWidget( this, "kfoldingProgressWidget", orientation(), _cfg->appletImage() );
	_progressWidget->installEventFilter( this );
	QHBoxLayout *layout = new QHBoxLayout( this );
	layout->addWidget( _progressWidget );
	
	createActions();
	setToolTip();
	performStartup();
	
	if ( kapp->authorizeKAction( "kicker_rmb" ) ) {
		_menu = new KPopupMenu();
		connect( _menu, SIGNAL( aboutToShow() ), this, SLOT( slotContextMenuAboutToShow() ) );
		//setCustomMenu( _menu );
	} // if
	
	connect( _process, SIGNAL( progressUpdated() ), this, SLOT( slotUpdateProgress() ) );
	connect( _process, SIGNAL( stateChanged( kfoldingProcess::State ) ), this, SLOT( slotStateChanged( kfoldingProcess::State ) ) );
	connect( _process, SIGNAL( processError( kfoldingProcess::Error ) ), this, SLOT( slotProcessError( kfoldingProcess::Error ) ) );
	
	// force the user to configure the app on the first run
	if ( _cfg->firstRun() )
		preferences();
	
	loadState();
	return;
} // kfolding ctor



kfolding::~kfolding()
{
	KGlobal::locale()->removeCatalogue( "Folding@home Applet" );

	_cfg->save();
	delete _cfg;
	delete _menu;
	return;
} // kfolding dtor



void kfolding::about()
{
	KAboutData about( "kfolding", I18N_NOOP( "Folding@home Applet" ), KFOLDING_VERSION,
			  I18N_NOOP( "Folding@home Applet for KDE" ),
			  KAboutData::License_GPL_V2, "(c) 2003-2005 Kevin Hessels");
	about.addAuthor( "Kevin Hessels", I18N_NOOP( "Primary author and maintainer" ), "khessels@shaw.ca", 0L );
	
	about.addCredit( "Richard P. Howell IV", I18N_NOOP( "Original OpenGL code, queue information" ), 0L, 0L );
	
	KAboutApplication aboutApp( &about );
	aboutApp.setCaption( i18n( "About Folding@home Applet" ) );
	aboutApp.exec();
	return;
} // about



void kfolding::slotContextMenuAboutToShow()
{
	_workUnitAction->setEnabled( workUnitAvailable() );
	_queueInfoAction->setEnabled( queueInfoAvailable() );


	KPopupMenu *status = new KPopupMenu( _menu );
	_menu->clear();
	
	if ( !_process->currentUnit().isEmpty() ) {
		_menu->insertTitle( _process->currentUnit() );
	}
	
	_process->start()->plug( _menu );
	_process->stop()->plug( _menu );
	
	// I'm not sure if this is ever going to work...
	// the FAH program actually execv()s a worker 'core' process which does all of the computing.
	// Sending the suspend signal to the parent process leaves the worker process unaffected. At 
	// the moment, there's no proper way to get the pid of worker process and thus, no way to suspend it.
	//_process->suspend()->plug( _menu );
	//_process->resume()->plug( _menu );
	
	_menu->insertSeparator();
	if ( status ) {
		status->insertItem( i18n( "My Folding" ), this, SLOT( slotFoldingHome() ) );
		status->insertItem( i18n( "User statistics" ), this, SLOT( slotUserStatistics() ) );
		status->insertItem( i18n( "Team statistics" ), this, SLOT( slotTeamStatistics() ) );
		_queueInfoAction->plug( status );
		status->insertItem( i18n( "Log file" ), this, SLOT( slotLogFile() ) );		
		_menu->insertItem( i18n( "Status" ), status );
	}

	_workUnitAction->plug( _menu );
	_menu->insertItem( SmallIcon( "configure" ), i18n( "Configure Folding@home Applet..." ), this, SLOT( preferences() ) );
	
	_menu->insertSeparator();
	
	_menu->insertItem( i18n( "About Folding@home Applet" ), this, SLOT( slotAbout() ) );
	return;
} // slotContextMenuAboutToShow



void kfolding::preferences()
{
	_settingsDialogue = new kfoldingSettingsDialogue( 0L, QString::fromLatin1( "settings" ), _cfg );
	
	connect( _settingsDialogue, SIGNAL( settingsChanged() ), this, SLOT( slotApplySettings() ) );
	connect( _settingsDialogue, SIGNAL( finished() ), this, SLOT( slotDestroySettings() ) );

	_settingsDialogue->show();
	return;
} // preferences



int kfolding::widthForHeight(int height) const
{
	return _progressWidget->widthForHeight( height );
} // widthForHeight



int kfolding::heightForWidth(int width) const
{
	return _progressWidget->heightForWidth( width );
} // heightForWidth



bool kfolding::eventFilter( QObject* o, QEvent* e )
{
	if ( e->type() == QEvent::MouseButtonPress ) {
		QMouseEvent* me = static_cast<QMouseEvent* >( e );
		if ( me->button() == QMouseEvent::RightButton ) {
			if ( !kapp->authorizeKAction( "kicker_rmb" ) )
				return false;
				
			_menu->exec( me->globalPos() );
			return true;
		} // if
	} // if
	return KPanelApplet::eventFilter( o, e );
} // eventFilter



void kfolding::positionChange( Position p )
{
	switch( p ) {
		case pTop:
		case pBottom:
			_progressWidget->setOrientation( Horizontal );
			break;
		case pLeft:
		case pRight:
			_progressWidget->setOrientation( Vertical );
			break;
		default:
			break;
	} // switch
	return;
} // positionChange



void kfolding::resizeEvent( QResizeEvent* )
{
	_progressWidget->update();
	return; 
} // resizeEvent



void kfolding::slotShowWorkUnit()
{
	_unitDialogue = new kfoldingUnitDialog( 0L, QString::fromLatin1( "workunit" ), _process->currentUnit(), _cfg );

	connect( _unitDialogue, SIGNAL( finished() ), this, SLOT( slotDestroyUnitDialogue() ) );
	_unitDialogue->show();
	return;
} // slotShowWorkUnit



void kfolding::slotApplySettings()
{
	createClientCfg();
	
	_progressWidget->setImage( _cfg->appletImage() );
	resize( _progressWidget->width(), _progressWidget->height() );
	
	if ( _process->processState() == kfoldingProcess::StateRunning || _process->processState() == kfoldingProcess::StateSuspended ) {
		_process->restart();
	} else if ( _process->processState() == kfoldingProcess::StateStopped ) {
		// just to re-read the configuration. Not ideal but works...
		_process->stopFolding();
	}
	
	slotUpdateProgress();
	return;
} // slotApplySettings



void kfolding::slotAbout()
{
	about();
	return;
} // slotAbout



void kfolding::slotFoldingHome()
{
	kapp->invokeBrowser( _cfg->workingDir() + QString::fromLatin1( "MyFolding.html" ) );
	return;
} // slotFoldingHome


void kfolding::slotTeamStatistics()
{
	kapp->invokeBrowser( QString::fromLatin1( "http://vspx27.stanford.edu/cgi-bin/main.py?qtype=teampage&teamnum=" ) + QString::number( _cfg->teamNumber() ) );
	return;
} // slotTeamStatistics



void kfolding::slotUserStatistics()
{
	kapp->invokeBrowser( QString::fromLatin1( "http://vspx27.stanford.edu/cgi-bin/main.py?qtype=userpage&username=" ) + _cfg->userName() );
	return;
} // slotUserStatistics


void kfolding::slotLogFile()
{
	QString logFile = _cfg->workingDir() + QString::fromLatin1( "FAHlog.txt" ); 
	QFileInfo fileInfo( logFile );
	
	if ( fileInfo.exists() ) {
		KURL url;
		url.setPath( logFile );
		KRun::runURL( url, QString::fromLatin1( "text/plain" ) );
	} else {
		KMessageBox::error( 0L, i18n( "A log file has not been created yet." ) );
	} // else
	return;
} // slotLogFile



void kfolding::slotUpdateProgress()
{
	kdDebug() << "slotUpdateProgress" << endl;
	_progressWidget->slotSetPercentage( _process->currentProgress() );
	setToolTip();

	// change from 100 to 0 indicates that we have finished a work unit
	if ( _lastPercent == 100 && _process->currentProgress() == 0 ) {
		KPassivePopup::message( QString::fromLatin1( "Work unit complete" ),
			QString::fromLatin1( "Work unit %1 has been completed.\nThank you for your contribution!" ).arg( _lastUnit ),
			BarIcon( QString::fromLatin1( "kfolding" ), KIcon::SizeMedium ), this );
	} // if
	_lastPercent = _process->currentProgress();
	_lastUnit = _process->currentUnit();
	return;
} // slotUpdateProgess



void kfolding::slotDestroySettings()
{
	if ( _settingsDialogue ) {
		_settingsDialogue->delayedDestruct();
		_settingsDialogue = 0L;
	} // if
	return;
} // slotDestroySettings



void kfolding::slotDestroyUnitDialogue()
{
	if ( _unitDialogue ) {
		if ( _unitDialogue->isVisible() )
			_unitDialogue->hide();
		
		delete _unitDialogue;
		_unitDialogue = 0L;
	} // if 
	return;
} // slotDestroyUnitDialogue



void kfolding::slotQueueInfo()
{
	kfoldingQueueDialog queueInfo( 0L, "queue_dialog", i18n( "Folding@home Queue" ), _cfg );
	queueInfo.exec();
	return;
} // slotQueueInfo



void kfolding::slotStateChanged( kfoldingProcess::State state )
{
	kdDebug() << "slotStateChanged: state = " << state << endl;
	_cfg->setState( ( int ) state );
	_cfg->save();
	
	setToolTip();
	return;
} // slotStateChanged



void kfolding::slotProcessError( kfoldingProcess::Error error )
{
	QString errorMessage;
		
	switch( error ) {
		case kfoldingProcess::ErrorNoExecutable:
			errorMessage = i18n( "No executable file for the Folding@home client software has been " \
					     "specified.  Please configure Folding@home Applet with the " \
					     "path to the executable." );
			break;
		case kfoldingProcess::ErrorBadExecutable:
			errorMessage = i18n( "The file specified as the Folding@home client software does not " \
					     "appear to be execuable or you do not have permission to run the file." );
			break;
		case kfoldingProcess::ErrorNotFoldingAtHome:
			errorMessage = i18n( "The file specified as the Folding@home client software is not a supported " \
					     "version.  You must be Folding@home client for Linux version 3 or 4." );
			break;
		case kfoldingProcess::ErrorAlreadyRunning:
			errorMessage = i18n( "Folding@home Applet has detected another instance of the " \
					     "Folding@home client software is already running in the specified " \
					     "work directory.  Please select a different work directory." );
			break;
		case kfoldingProcess::ErrorMachineId:
			errorMessage = i18n( "Folding@home Applet has detected another instance of the " \
					     "Folding@home client sofware running with the same Machine Id. " \
					     "Please select a different Machine Id.\n" \
					     "If you are sure no other instances are running, you can delete the file: /tmp/fah/f%1 and try again.").arg( _cfg->machineId() );
			break;
		case kfoldingProcess::ErrorUnknown:		// fall-through
		default:
			errorMessage = i18n( "An unspecified error has occurred while attempting to launch " \
					     "the Folding@home client application.  Please view the log file " \
					     "for details about the error." );
			break;
	} // switch
	KMessageBox::error( 0L, errorMessage, i18n( "Folding@home Applet" ) );
	return;
} // slotProcessError



void kfolding::performStartup()
{
	if ( !_cfg )
		return;
	
	// check to see that the working directory exists and create it if it doesn't.  If we are unable to
	// create the directory, set it to the default.
	QFileInfo fileInfo( _cfg->workingDir() );
	QDir d( _cfg->workingDir() );
	if ( !fileInfo.exists() ) {
		if ( !d.mkdir( _cfg->workingDir() ) ) {
			KMessageBox::error( 0L, i18n( "The selected working directory: \"%1\" could not be created\n" ).arg( _cfg->workingDir() ) +
						i18n( "The default directory: \"%2\" will be used instead." ).arg( KGlobal::dirs()->saveLocation( "data", "kfolding/", false ) ),
					    	i18n( "Folding@home Applet" ) );
			
			kdError() << "Unable to create directory: " << _cfg->workingDir() << endl;
			kdError() << "Working directory will be set to " << KGlobal::dirs()->saveLocation( "data", "kfolding/", false ) << endl;
			_cfg->setWorkingDir( KGlobal::dirs()->saveLocation( "data", "kfolding/", true ) );
		} // if
	} // if
	if ( !fileInfo.isWritable() ) {
		KMessageBox::error( 0L, i18n( "Permission denied to directory: \"%1\"\n" ).arg( _cfg->workingDir() ) +
				    i18n( "The default directory: \"%2\" will be used instead." ).arg( KGlobal::dirs()->saveLocation( "data", "kfolding/", false ) ),
				    i18n( "Folding@home Applet" ) );
		_cfg->setWorkingDir( KGlobal::dirs()->saveLocation( "data", "kfolding/", true ) );
	} // if
			

	if ( QFile::exists( _cfg->workingDir() + QString::fromLatin1( "client.cfg" ) ) ) {
		readClientCfg();
	} else {
		createClientCfg();
	}  // else

	return;
} // performStartup


void kfolding::createActions()
{
	_queueInfoAction = new KAction( i18n( "Queue information" ), 0, 0, this, SLOT( slotQueueInfo() ), &_actions, "queuewidget_action" );
	_workUnitAction = new KAction( i18n( "Display work unit..." ), 0, 0, this, SLOT( slotShowWorkUnit() ), &_actions, "workunit_action" );
	
	return;
} // createActions



void kfolding::loadState()
{
	if ( !_cfg || !_process )
		return;
		
	if ( _cfg->startOnLaunch() && !_cfg->firstRun() ) {
		_process->startFolding();
	} else if ( _cfg->restoreState() && !_cfg->firstRun() ) {
		kfoldingProcess::State state = static_cast<kfoldingProcess::State>( _cfg->state() );
		switch ( state ) {
			case kfoldingProcess::StateRunning:
				_process->startFolding();
				break;
			case kfoldingProcess::StateStopped:
				_process->stopFolding();
				break;
			case kfoldingProcess::StateSuspended:
			default:
				break;
		} // switch
	} else {
		_process->stopFolding();
	} // else
	return;
} // loadState



void kfolding::readClientCfg()
{
	int pos;

	QFile src( _cfg->workingDir() + QString::fromLatin1( "client.cfg" ) );
	QStringList lines;
	if ( src.open( IO_ReadOnly ) ) {
		kdDebug() << "client.cfg found. parsing" << endl;
		QRegExp rxUserName( "^username=(.*)$" );
		QRegExp rxTeamNumber( "^team=(.*)$" );
		QRegExp rxClientType( "^type=(.*)$" );
		QRegExp rxMachineId( "^machineid=(.*)$" );
		QRegExp rxBigPackets( "^bigpackets=(.*)$" );	

		QTextStream stream( &src );
		QString line;
		while( !stream.atEnd() ) {
			line = stream.readLine().stripWhiteSpace();
			
			pos = rxUserName.search( line );
			if ( pos > -1 ) {
				_cfg->setUserName( rxUserName.cap( 1 ) );
			}
			
			pos = rxTeamNumber.search( line );
			if ( pos > -1 ) {
				_cfg->setTeamNumber( rxTeamNumber.cap( 1 ).toInt() );
			}
			
			pos = rxClientType.search( line );
			if ( pos > -1 ) {
				_cfg->setClientType( rxClientType.cap( 1 ).toInt() );
			}
			
			pos = rxMachineId.search( line );
			if ( pos > -1 ) {
				_cfg->setMachineId( rxMachineId.cap( 1 ).toInt() );
			}
			
			pos = rxBigPackets.search( line );
			if ( pos > -1 ) {
				if ( rxBigPackets.cap( 1 ).stripWhiteSpace() == QString::fromLatin1( "yes" ) ) {
					_cfg->setBigPackets( true );
				} else {
					_cfg->setBigPackets( false );
				} // else
			} // if
		} // while
		src.close();
	} else {
		// couldn't open the client config for reading ??
		// try to create a new one 
		createClientCfg();
	}
	return;
} // readClientCfg



void kfolding::createClientCfg()
{
	// set the user name and group number into the client.cfg file.
	kdDebug() << "Editing client.cfg file" << endl;
	QFile src( locate( "data", "kfolding/default.cfg" ) );
	QStringList lines;
	if ( src.open( IO_ReadOnly ) ) {
		QRegExp rxUserName( "^username=(.*)$" );
		QRegExp rxTeamNumber( "^team=(.*)$" );
		QRegExp rxClientType( "^type=(.*)$" );
		QRegExp rxMachineId( "^machineid=(.*)$" );
		QRegExp rxBigPackets( "^bigpackets=(.*)$" );
		
		QTextStream stream( &src );
		QString line;
		while ( !stream.atEnd() ) {
			line = stream.readLine();
			if ( rxUserName.search( line ) != -1 ) {
				line = QString::fromLatin1( "username=" ) + _cfg->userName();
			}
			else if ( rxTeamNumber.search( line ) != -1 ) {
				line = QString::fromLatin1( "team=" ) + QString::number( _cfg->teamNumber() );
			}
			else if ( rxClientType.search( line ) != -1 ) {
				line = QString::fromLatin1( "type=" ) + QString::number( _cfg->clientType() );
			} 
			else if ( rxMachineId.search( line ) != -1 ) {
				line = QString::fromLatin1( "machineid=" ) + QString::number( _cfg->machineId() );
			}
			else if ( rxBigPackets.search( line ) != -1 ) {
				if ( _cfg->bigPackets() ) {
					line = QString::fromLatin1( "bigpackets=yes" );
				} else {
					line = QString::fromLatin1( "bigpackets=no" );
				}
			} // else
			lines += line;
		} // while
		src.close();
	} // if
		
	// write back
	QFile dest( _cfg->workingDir() + QString::fromLatin1( "client.cfg" ) );
	if ( dest.open( IO_WriteOnly ) ) {
		QTextStream stream( &dest );	
		for ( QStringList::iterator it = lines.begin(); it != lines.end(); ++it ) {
			stream << *it << endl;
		} // for
		dest.close();
	} // if
	return;
} // createClientCfg



void kfolding::setToolTip()
{
	QToolTip::remove( _progressWidget );
	QString state;
	switch( _process->processState() ) {
		case kfoldingProcess::StateRunning:
			state = i18n( "Running" );
			break;
		case kfoldingProcess::StateStopped:
			state = i18n( "Stopped" );
			break;
		case kfoldingProcess::StateSuspended:
			state = i18n( "Suspended" );
			break;
		default:
			break;
	} // switch
	
	if ( _process->currentUnit().isEmpty() ) {
		QToolTip::add( _progressWidget, i18n( "Folding@home: %1" ).arg( state ) );
	} else {
		QToolTip::add( _progressWidget, i18n( "Folding@home: %1\nWork unit: %2 (%3%)" ).arg( state ).arg( _process->currentUnit() ).arg( QString::number( _process->currentProgress() ) ) );
	} // else
	return;
} // setTooltip


bool kfolding::queueInfoAvailable()
{
	// First check to see if we have a queue.dat file available for reading	
	QString queueFilePath = _cfg->workingDir() + QString::fromLatin1( "queue.dat" );
	QFileInfo queueFile( queueFilePath );

	// Now, check to see if we have a qd executable to read the queue information
	QString qdPath = KStandardDirs::findExe( QString::fromLatin1( "qd" ) );
	
	if ( qdPath.isEmpty() || !queueFile.exists() ) {
		return false;
	}

	return true;
} // queueInfoAvailable


bool kfolding::workUnitAvailable()
{
	QString workUnitFile = _cfg->workingDir() + QString::fromLatin1( "work/current.xyz" );
	QFileInfo fileInfo( workUnitFile );

	if ( !fileInfo.exists() ) {
		return false;
	}
	return true;
} // workUnitAvailable



extern "C"
{
    KPanelApplet* init( QWidget *parent, const QString configFile)
    {
        KGlobal::locale()->insertCatalogue( "Folding@home Applet" );
        return new kfolding( configFile, KPanelApplet::Normal,
                             KPanelApplet::About | KPanelApplet::Preferences,
                             parent, "Folding@home Applet" );
    }
}

#include "kfolding.moc"
