package org.inria.bmajwatcher.server;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Hashtable;

import javax.naming.Context;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.DirContext;
import javax.naming.directory.InitialDirContext;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;

import org.apache.log4j.Logger;
import org.inria.bmajwatcher.client.services.AuthenticationService;
import org.inria.bmajwatcher.server.utils.Utils;

import com.google.gwt.user.server.rpc.RemoteServiceServlet;

/**
 * Service that grants admin rights.
 * 
 * @author rsabas
 * 
 */
public class AuthenticationServiceImpl extends RemoteServiceServlet implements
		AuthenticationService {

	private static final long serialVersionUID = 1L;
	private static final String FILE_NAME = ".bmajwatcherpasswd";
	private static final String LOGGED = "logged";
	private static Logger log = Logger.getLogger(AuthenticationServiceImpl.class);

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.inria.bmajwatcher.client.services.AuthenticationService#authenticate
	 * (java.lang.String, java.lang.String)
	 */
	@Override
	public boolean authenticate(String login, String passwd) {

		log.info("Logging in request");
		if (login == null || passwd == null)
			return false;

		boolean auth = false;
		String useLDAP = getServletContext().getInitParameter("USELDAP");
		// If LDAP option is set in context, use ldap server method
		if (useLDAP != null && useLDAP.equals("1")) {
			log.debug("Authenticate with ldap");
			if ((auth = authenticateWithLDAP(login, passwd)) == true) {
				getThreadLocalRequest().getSession().setAttribute(LOGGED, "true");
			}
			
		}
		if (!auth) {
			String useLocal = getServletContext().getInitParameter("USELOCAL");			
			if (useLocal != null && useLocal.equals("1")) {
				log.debug("Authenticate with local");
				auth = authenticateWithLocal(login, passwd);
			}
		}
		
		return auth;
		
	}
	
	private boolean authenticateWithLocal(String login, String passwd) {
		try {
			MessageDigest digest = MessageDigest.getInstance("SHA1");
			digest.update(login.getBytes());
			byte[] hashedLogin = digest.digest();

			digest.reset();
			digest.update(passwd.getBytes());
			byte[] hashedPasswd = digest.digest();
			
			if (checkId(Utils.getHexString(hashedLogin), Utils.getHexString(hashedPasswd))) {
				getThreadLocalRequest().getSession().setAttribute(LOGGED, "true");
				return true;
			}
			return false;

		} catch (NoSuchAlgorithmException e) {
			log.error(e.getMessage(), e);
			return false;
		}
	}

	/**
	 * Authenticates against a LDAP server
	 * 
	 * @param login
	 * @param passwd
	 * @return true if authentication succeeded
	 */
	private boolean authenticateWithLDAP(String login, String passwd) {
		// Set up the environment for creating the initial context
		Hashtable<String, String> env = new Hashtable<String, String>();
		env.put(Context.INITIAL_CONTEXT_FACTORY,
				"com.sun.jndi.ldap.LdapCtxFactory");
		String ldapHost = getServletContext().getInitParameter("LDAPHOST");
		String ldapDn = getServletContext().getInitParameter("LDAPDN");
		if (ldapHost == null || ldapDn == null)
			return false;
		env.put(Context.PROVIDER_URL, "ldap://" + ldapHost + ":389/" + ldapDn);
		// Authenticate as anonymous
		env.put(Context.SECURITY_AUTHENTICATION, "simple");

		// Create the initial context
		try {
			log.debug("connect to ldap");
			DirContext ctx = new InitialDirContext(env);
			String optFilter = "";
			String usercn = null;
			if (getServletContext().getInitParameter("OPTFILTER") != null)
				optFilter = getServletContext().getInitParameter("OPTFILTER");
			// Bind with filter

			String filter = "(&(uid=" + login + ")" + optFilter + ")";
			log.debug("Filter: " + filter);
			// limit returned attributes to those we care about
			String[] attrIDs = { "cn" };
			SearchControls ctls = new SearchControls();
			ctls.setReturningAttributes(attrIDs);
			// comment out next line to limit to one container otherwise it'll
			// walk down the tree
			ctls.setSearchScope(SearchControls.SUBTREE_SCOPE);
			ctls.setCountLimit(1);
			// Search for objects using filter and controls
			NamingEnumeration<SearchResult> answer = ctx.search("", filter, ctls);
			// cycle through result set
			while (answer.hasMore()) {
				SearchResult sr = (SearchResult) answer.next();
				usercn = sr.getName() + "," + ldapDn;
				ctx.close();
			}
			if (usercn == null)
				return false;
			env.put(Context.SECURITY_PRINCIPAL, usercn);
			env.put(Context.SECURITY_CREDENTIALS, passwd);
			log.debug("log with " + usercn);
			ctx = new InitialDirContext(env);
			ctx.close();

		} catch (NamingException e) {
			log.error("Ldap error: " + e.getMessage());
			return false;
		}

		return true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.inria.bmajwatcher.client.services.AuthenticationService#getUserAcces
	 * ()
	 */
	@Override
	public String getUserAcces() {
		Object o = getThreadLocalRequest().getSession().getAttribute(LOGGED);
		if (o != null)
			return o.toString();
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.inria.bmajwatcher.client.services.AuthenticationService#logout()
	 */
	@Override
	public void logout() {
		getThreadLocalRequest().getSession().invalidate();
	}

	/**
	 * Checks if login and password exist in the file.
	 * 
	 * @param login
	 * @param passwd
	 * @return
	 */
	private boolean checkId(String login, String passwd) {
		
		String path = getServletContext().getInitParameter("BIOMAJ_ROOT") + System.getProperty("file.separator") + FILE_NAME;
		if (new File(path).exists()) {
			try {
				BufferedReader br = new BufferedReader(new FileReader(path));
				String line = br.readLine();
				if (line != null) {
					String[] id = line.trim().split(":");
					return (id[0].equals(login) && id[1].equals(passwd));
				}
				return false;
	
			} catch (IOException ex) {
				log.error(ex.getMessage(), ex);
			}
		} else
			log.error("Password file not found : " + path);
		
		return false;
	}

}
