#!/usr/bin/perl -w
# $Id: petrovich.pl,v 1.8 2001/07/16 16:34:55 tdotkinch Exp $
#TODO: #includes in config file

use strict;
use Getopt::Long;

# autoflush pipes
$|++;

# globals
my (
    @dirs,            # array of dirs to recursively trample
    %exceptions,  # hash of exceptions from within @dirs
    %stored,        # info stored in the db about the file for verifying
    $hash,           # object for whichever digest type we have chosen
    $hash_type,   # string representing which hash type we want to use
    $db_file,        # file name for database - auto-gen if absent
    $config_file,   # name of the configuration file
    $verify,         # true if we are running in verify mode
    @stuff,         # stuff we are monitoring for changes
    $db_hash,     # the hash of the db file itself - generated from the filesystem
    $db_hash_verifier,  # the hash of the db file - user supplied
    $crap_os,      # 0 if running on UNIX - 1 otherwise

);

print "\nInitializing...\n";
# keep track of when the script started
my $start_time = time;

$crap_os = detect_os();

#process command line arguments
proc_args();

# parse the config file for directories and exceptions    
parse_config();

print "Opening $db_file\n";

# if we are not in verify mode we are in db creation mode
if (!defined($verify)) { 
    
    # thanks to RFP for sharing his ++31336 code with the rest of us
    *process = \&return_file_info;
    *cleanup = \&cleanup_db_gen;

    open(DB,">$db_file") or die "Can't open $db_file for writing: $!";
    
    #write the db's hash type to 1st line
    print DB "*$hash_type\n";

} else {

    # verify against an existing db

    # set the right function references    
    *process = \&verify_file_info;
    *cleanup = \&cleanup_db_verify;
    open(DB,"$db_file") or die "Can't read from $db_file: $!\n
    Maybe you should specify one with --db filename";
    # read the db into a hash for easy comparison with @current
    # the key is the filename - the value is the same as gen_file_info 
    while (<DB>) {

        # skip comments
        next if /^#/;
        
        # read hash type from first line
        if (s/^\*//) {
            chomp;
            $hash_type =$_;
            next;
        }
        
        # Otherwise, store the first field (filename) in $1
        /(.+?)\|/;
        if (!defined($1)) {
            print "bad line $_\n";
            next;
        } else {
            if ($crap_os) {        
                $stored{lc($1)} = $_;
            } else {
                $stored{$1} = $_;
            }
        }
    }    
}

# set the hash algorithm to use
SWITCH: {

        $hash_type =~ /^md2/i && do { 
            $hash = eval "use Digest::MD2;Digest::MD2->new;";
            die "You don't have the Digest::MD2 module installed" if $@;
            last SWITCH;
        };

        $hash_type =~ /^md5/i && do {
            $hash = eval "use Digest::MD5;Digest::MD5->new;";
            die "You don't have the Digest::MD5 module installed" if $@;
            last SWITCH; };

        $hash_type =~ /^sha1/i && do { 
            $hash = eval "use Digest::SHA1; Digest::SHA1->new;";
            die "You don't have the Digest::SHA1 module installed" if $@;            
            last SWITCH;
            };
}

# if a db hash was provided on the command line then check it
if (defined($db_hash_verifier)) {
    $db_hash = gimme_hash($db_file);
    die "hash you supplied doesn't match $db_file!" unless ($db_hash eq $db_hash_verifier);
}

foreach (@dirs) {
    print "Processing $_\n";
    recurse($_);
}

cleanup();

my $end_time = time;
print "$0 ran for " . ($end_time - $start_time) . " seconds\n";

sub detect_os {

    return 0 if !defined($ENV{'OS'});    
    if ($ENV{'OS'} =~ /win/i) {
        return 1;
    } else {
        return 0;
    }
    

}

sub cleanup_db_gen {

    # close the db
    close DB;
        
    # if in db generation mode gen a hash on the db
    $db_hash = gimme_hash($db_file);    
    # SPAM it to stdout
    print "$db_file\'s $hash_type hash is $db_hash\n";
    
}

sub cleanup_db_verify {

    foreach (keys %stored) {
        print "deleted: $_\n";
    }
    close DB;

}


sub verify_file_info {

my $file = shift;

# if the file doesn't exist in the db
unless (defined($stored{$file})) {

    # then it's been added since the last run
    print "added: $file\n";
    return;
} else {

    # check the version on the filesystem    
    my $this = gen_file_info($file);
    
    # check it against version in the db
    unless ($this eq $stored{$file}) {
        my $i;        
        print "changed: $file [ ";
        my @stored = split(/\|/,$stored{$file});
        my @current = split(/\|/,$this);
        for $i (1..$#stored) {
            unless ($stored[$i] eq $current[$i]) {
                print "$stuff[$i] ";
            }
        }
        print "]\n";

    }
    delete $stored{$file};
}
} # end sub

sub recurse {
    
    # recursively process all the files in a directory, excluding exceptions
    # need local copies of these vars so the sub can call itself
    my $dir = shift;
    my $file;
        # if it's an excluded dir skip it
        unless (defined($exceptions{$dir})) {
        (opendir(DIR, $dir)) or warn "could not open $dir: $!\n";
            foreach (readdir(DIR)) {

                  # fully qualify the path
                  $file = "$dir/$_";
                  $file = lc($file) if $crap_os;
                  next if ($_ eq '.' || $_ eq '..');  ## skip current and parent directories          
                  next if (!-f $file && !-d $file);   ## skip everything but directories and files
                  next if (-l $file);                    ## skip symlinks
                  
                  # if it's a directory step into it          
                  if (-d $file) {
                      closedir(DIR);              
                      recurse($file);
                  } else {

                  # if it's a regular file and not excluded print its info to the database                  
                  process($file) unless defined($exceptions{$file});
                  
                  }  
            } 
            closedir(DIR);
        } 

}

sub parse_config {
    # lines beginning with # are comments
    # lines beginning with + are included dirs
    # lines beginning with - are excluded dirs

    open(CONF, $config_file) or die "Can't open $config_file: $!";
    while (<CONF>) {

        # replace backslashes with slashes for consistency's sake
        s/\\/\//;
        
        # convert to lower case if running on windows
        $_ = lc($_) if $crap_os;
            
        # ignore comments and blank lines    
        next if /^#/;
        next if /^\s/;

        # + lines are dirs to process - strip the + add to @dirs    
        if (s/^\+//) {
            chomp;        
            push(@dirs,$_);
        
        # - lines are dirs to exclude (implies a + parent)
        # strip the - and add to %exceptions    
        } elsif (s/^-//) {
            chomp;
            $exceptions{$_} = 1;
        } else {
            print "\nERROR:Don't know what to do about config line:\n$_\n\n";    
        }

}
}

sub proc_args { 
    GetOptions(
        "verify"        => \$verify,
        "hash=s"      => \$hash_type,
        "conf=s"       => \$config_file,
        "db=s"          => \$db_file,
        "dbhash=s"   => \$db_hash_verifier,
    ) or die <<USAGE;
    to create a new db: $0 [ --hash md2|md5|sha1 ] [--conf configfile]
    to verify an existing: $0  --verify [--db filename] [--conf configfile] [--dbhash Base64hash]
USAGE
    # check sanity and provide reasonable defaults where appropriate
    # these are all globals
    
#    undef $db_hash_verifier; # for now cuz it's not working    
    $config_file = "/etc/petrovich.conf" unless defined($config_file);
    $hash_type = "md5" unless defined($hash_type);
    @stuff = ("filename", "mode","uid","gid","size","mtime","ctime","$hash_type");
    $db_file = "/var/db/petrovich/petrovich.db" unless defined($db_file);
    
# if running under windows lowercase everything
    $config_file = lc($config_file) if $crap_os;
    $db_file = lc($db_file)      if $crap_os;
}

sub return_file_info {
    my $this = shift;
    my $that = gen_file_info($this);
    if ($that) {
        print DB $that;
    } else {
        return 0;
    }
}

sub gen_file_info {
    my $this = shift; 
    my $ret;
    if (-e $this) {
    
      (my $mode,my $uid,my $gid,my $size,my $mtime,my $ctime) = (stat($this))[2,4,5,7,9,10];
      $ret = "$this|$mode|$uid|$gid|$size|$mtime|$ctime|";
      $ret .= gimme_hash($this);
      $ret .= "\n";
      return $ret;

    } else {
      return 0;
    }
}

sub gimme_hash {
    my $filename = shift;
    if (open(FILE, $filename)) {
            binmode(FILE);            
            $hash -> addfile(*FILE);
            return scalar($hash->b64digest);
} else {
    return 0;
}

}
