# frozen_string_literal: true

# Licensed to the Software Freedom Conservancy (SFC) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The SFC licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.

module Selenium
  module WebDriver
    #
    # Base class implementing default behavior of service_manager object,
    # responsible for starting and stopping driver implementations.
    #
    # @api private
    #
    class ServiceManager
      START_TIMEOUT = 20
      SOCKET_LOCK_TIMEOUT = 45
      STOP_TIMEOUT = 20

      #
      # End users should use a class method for the desired driver, rather than using this directly.
      #
      # @api private
      #

      def initialize(config)
        @executable_path = config.executable_path
        @host = Platform.localhost
        @port = config.port
        @extra_args = config.extra_args
        @shutdown_supported = config.shutdown_supported

        raise Error::WebDriverError, "invalid port: #{@port}" if @port < 1
      end

      def start
        raise "already started: #{uri.inspect} #{@executable_path.inspect}" if process_running?

        Platform.exit_hook(&method(:stop)) # make sure we don't leave the server running

        socket_lock.locked do
          find_free_port
          start_process
          connect_until_stable
        end
      end

      def stop
        return unless @shutdown_supported

        stop_server
        @process.poll_for_exit STOP_TIMEOUT
      rescue ChildProcess::TimeoutError
        nil # noop
      ensure
        stop_process
      end

      def uri
        @uri ||= URI.parse("http://#{@host}:#{@port}")
      end

      private

      def build_process(*command)
        WebDriver.logger.debug("Executing Process #{command}")
        @process = ChildProcess.build(*command)
        if WebDriver.logger.debug?
          @process.io.stdout = @process.io.stderr = WebDriver.logger.io
        elsif Platform.jruby?
          # Apparently we need to read the output of drivers on JRuby.
          @process.io.stdout = @process.io.stderr = File.new(Platform.null_device, 'w')
        end

        @process
      end

      def connect_to_server
        Net::HTTP.start(@host, @port) do |http|
          http.open_timeout = STOP_TIMEOUT / 2
          http.read_timeout = STOP_TIMEOUT / 2

          yield http
        end
      end

      def find_free_port
        @port = PortProber.above(@port)
      end

      def start_process
        @process = build_process(@executable_path, "--port=#{@port}", *@extra_args)
        # NOTE: this is a bug only in Windows 7
        @process.leader = true unless Platform.windows?
        @process.start
      end

      def stop_process
        return if process_exited?

        @process.stop STOP_TIMEOUT
        @process.io.stdout.close if Platform.jruby? && !WebDriver.logger.debug?
      end

      def stop_server
        return if process_exited?

        connect_to_server do |http|
          headers = WebDriver::Remote::Http::Common::DEFAULT_HEADERS.dup
          http.get('/shutdown', headers)
        end
      end

      def process_running?
        defined?(@process) && @process&.alive?
      end

      def process_exited?
        @process.nil? || @process.exited?
      end

      def connect_until_stable
        socket_poller = SocketPoller.new @host, @port, START_TIMEOUT
        return if socket_poller.connected?

        raise Error::WebDriverError, cannot_connect_error_text
      end

      def cannot_connect_error_text
        "unable to connect to #{@executable_path} #{@host}:#{@port}"
      end

      def socket_lock
        @socket_lock ||= SocketLock.new(@port - 1, SOCKET_LOCK_TIMEOUT)
      end
    end # Service
  end # WebDriver
end # Selenium
