/********************************************************************************************************
 * QRNA - Comparative analysis of biological sequences 
 *         with pair hidden Markov models, pair stochastic context-free
 *        grammars, and probabilistic evolutionary  models.
 *       
 * Version 2.0.0 (JUN 2003)
 *
 * Copyright (C) 2000-2003 Howard Hughes Medical Institute/Washington University School of Medicine
 * All Rights Reserved
 * 
 *     This source code is distributed under the terms of the
 *     GNU General Public License. See the files COPYING and LICENSE
 *     for details.
 ***********************************************************************************************************/

/* scorewithmodels.c
 *
 * ER, Fri Feb 25 22:00:06 CST 2000 [St. Louis]
 * 
 * 
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>

#include "funcs.h"
#include "globals.h"
#include "squid.h"
#include "structs.h"

#ifdef MEMDEBUG
#include "dbmalloc.h"
#endif

static void add_abs_to_end3s (int start, struct end3_s *end3s);
static void add_abs_to_ends (int start, struct end_s *ends);
static void cal_structure (FILE *ofp,
			   SQINFO sqinfoX, int *seqX,  
			   SQINFO sqinfoY, int *seqY, 
			   char *gss, int start, int L, 
			   struct rnamodel_s *rnamodel, 
			   struct rnascfg_s  *mx, 
			   struct sc3_s      *sc, 
			   int cyk, int logodds, int do_nus, int parse, int rnass, int traceback, struct end_s *rnaends, int isrnarev);
static void print_window_ends (FILE *ofp, int start, int leg, char *whichmodel, struct end_s *ends, int isreverse, int winnerstrand);
static void score_with_null(FILE *ofp, 
			    int *isegX, int *isegY, int len, 
			    struct nullmodel_s *null, 
			    double nullsc, 
			    double nullrevsc, 
			    int verbose);

static int score_global(FILE *ofp, 
			SQINFO sqinfoX, int *isegX, 
			SQINFO sqinfoY, int *isegY, 
			char *gss, int start, int win, 
			struct dos_s      d, 
			struct model_s   *model, 
			struct dpd_s     *dpd, 
			struct rnascfg_s *mx, 
			struct sc3_s     *sc, 
			struct ali_s     *ali, 
			int *ret_scfg_status, int *ret_scfgrv_status,
			int cyk, int fastintloop, int logodds, int ones, int parse, int rnass, int sweep, int traceback, 
			int doends, struct windowends_s *windowends);

static int score_viterbi_diag(FILE *ofp, 
			      FILE *regressfp, char *regressionfile, 
			      SQINFO sqinfoX, int *isegX, 
			      SQINFO sqinfoY, int *isegY,  
			      char *gss, int start, int win,
			      struct dos_s      d, 
			      struct model_s   *model,
			      struct dpd_s     *dpd, 
			      struct rnascfg_s *mx, 
			      struct sc3_s     *sc, 
			      struct ali_s     *ali, 
			      int *ret_scfg_status, int *ret_scfgrv_status,
			      int alignment, int cyk, int fastintloop, int logodds, int ones, int parse, int rnass, int sweep, int traceback, 
			      int doends, struct windowends_s *windowends);

static int score_forward_diag(FILE *ofp, 
			      SQINFO sqinfoX, int *isegX, 
			      SQINFO sqinfoY, int *isegY, 
			      char *gss, int start, int win, 
			      struct dos_s      d, 
			      struct model_s   *model, 
			      struct dpd_s     *dpd, 
			      struct rnascfg_s *mx, 
			      struct sc3_s     *sc, 
			      struct dpsc3_s   *ardiag, 
			      struct ali_s     *ali, 
			      int *ret_scfg_status, int *ret_scfgrv_status,
			      int cyk, int fastintloop, int logodds, int ones, int parse, int sweep, struct windowends_s *windowends);

static int score_semi(FILE *ofp, 
		      SQINFO sqinfoX, int *iseqX, int startX, int lenX, 
		      SQINFO sqinfoY, int *iseqY, int startY, int lenY,
		      char *gss, int start, int *ret_legnew, int Lw, 
		      struct dos_s      d,  
		      struct model_s   *model, 
		      struct dpd_s     *dpd, 
		      struct othdpf_s  *othdpf, 
		      struct rnascfg_s *mx, 
		      struct scores_s  *sc, 
		      struct ali_s     *ali, 
		      int alignment, int cyk, int fastintloop, int logodds, int ones, int parse, int rnass, int sweep, int traceback, 
		      int doends, struct windowends_s *windowends);

static int score_full(FILE *ofp, 
		      SQINFO sqinfoX, int *iseqX, int startX, int lenX, 
		      SQINFO sqinfoY, int *iseqY, int startY, int lenY,
		      char *gss, int start, int *ret_legnew, int Lw, 
		      struct dos_s     d, 
		      struct model_s   *model, 
		      struct dpf_s     *dpf, 
		      struct rnascfg_s *mx, 
		      struct scores_s  *sc, 
		      struct ali_s     *ali, 
		      int alignment, int cyk, int fastintloop, int logodds, int ones, int parse, int rnass, int sweep, int traceback,
		      int doends, struct windowends_s *windowends);

static void test_OTH_model(FILE *ofp, 
			   SQINFO sqinfoX, int *isegX, 
			   SQINFO sqinfoY, int *isegY, 
			   int win,  
			   struct othmodel_s *oth, 
			   struct othdpd_s   *othdp, 
			   double            *othsc, 
			   struct ali_s      *ali, 
			   int alignment, int traceback, 
			   struct windowends_s *windowends);

/* Function: AllocEnd()
 * Date:     ER, Thu Mar 13 10:10:01 CST 2003 [St. Louis] 
 * 
 * Purpose:  Allocates memory for structure end_s  
 * 
 * Returns:  en3scan_s are allocated 
 */  
struct end_s * 
AllocEnd()
{
  struct end_s *end;   
  int               x;
   
  end = (struct end_s *) MallocOrDie (sizeof(struct end_s));

  end->lend = (int *) MallocOrDie (sizeof(int) * MAX_NUM_ENDS);
  end->rend = (int *) MallocOrDie (sizeof(int) * MAX_NUM_ENDS);

  /* initialize */
  for (x = 0 ; x < MAX_NUM_ENDS; x++) {
    end->lend[x] = -1;
    end->rend[x] = -1;
  }
  
  return end;
}

/* Function: AllocEnds3()
 * Date:     ER, Wed Sep  5 13:10:37 CDT 2001  [St. Louis]
 *
 * Purpose:  Allocates memory for structure end3_s 
 *
 * Returns:  end3_s are allocated
 */
struct end3_s *
AllocEnds3()
{
  struct end3_s *ends;   
   
  ends = (struct end3_s *) MallocOrDie (sizeof(struct end3_s));

  ends->cod = AllocEnd();
  ends->oth = AllocEnd();
  ends->rna = AllocEnd();

  return ends;
}

struct windowends_s *
AllocWindowEnds()
{
  struct windowends_s *windowends;   

  windowends = (struct windowends_s *) MallocOrDie (sizeof(struct windowends_s));
   
  windowends->fwd = AllocEnds3();
  windowends->rev = AllocEnds3();

  return windowends;
}

/* Function: AllocScores()
 * Date:     ER, Sat Feb 26 13:53:21 CST 2000 [St. Louis]
 *
 * Purpose:  Allocates memory for all scores
 *
 * Returns:  scores are allocated
 */
struct scores_s *
AllocScores()
{
  struct scores_s *sc;       /* structure with dp matrices   */

  sc = (struct scores_s *) MallocOrDie (sizeof(struct scores_s));

  sc->global = AllocSc3();
  sc->vidiag = AllocSc3();
  sc->frdiag = AllocSc3();
  sc->vifull = AllocSc3();
  sc->frfull = AllocSc3();
  sc->visemi = AllocSc3();
  sc->frsemi = AllocSc3();

  sc->ardiag = (struct dpsc3_s *) MallocOrDie (sizeof(struct dpsc3_s));
  sc->arfull = (struct dpsc3_s *) MallocOrDie (sizeof(struct dpsc3_s));

  return sc;
}

struct sc3_s *
AllocSc3()
{
  struct sc3_s *sc3;

  sc3 = (struct sc3_s *) MallocOrDie (sizeof(struct sc3_s));
  sc3->oth = (struct sc2_s *) MallocOrDie (sizeof(struct sc2_s));
  sc3->cod = (struct sc2_s *) MallocOrDie (sizeof(struct sc2_s));
  sc3->rna = (struct sc2_s *) MallocOrDie (sizeof(struct sc2_s));

  /* initialize */
  sc3->oth->pl = 0.0;
  sc3->oth->mn = 0.0;
  sc3->cod->pl = 0.0;
  sc3->cod->mn = 0.0;
  sc3->rna->pl = 0.0;
  sc3->rna->mn = 0.0;

  return sc3;
}

void
FreeEnd(struct end_s *end)
{
  free(end->lend);
  free(end->rend);

  free(end);
}

void
FreeEnds3(struct end3_s *ends)
{
  FreeEnd(ends->oth);
  FreeEnd(ends->cod);
  FreeEnd(ends->rna);

  free(ends);
}

void
FreeWindowEnds(struct windowends_s *windowends)
{
  FreeEnds3(windowends->fwd);
  FreeEnds3(windowends->rev);

  free(windowends);
}

void
FreeSc3(struct sc3_s *sc3)
{
  free(sc3->oth);
  free(sc3->cod);
  free(sc3->rna);

  free(sc3);
}

void
FreeScores(struct scores_s *sc)
{
  FreeSc3(sc->global);
  FreeSc3(sc->vidiag);
  FreeSc3(sc->frdiag);
  FreeSc3(sc->vifull);
  FreeSc3(sc->frfull);
  FreeSc3(sc->visemi);
  FreeSc3(sc->frsemi);

  free(sc->ardiag);
  free(sc->arfull);

  free(sc);
}

void
PatternEnd(struct end_s *ends)
{
  int x;

  for (x = 0 ; x < MAX_NUM_ENDS; x++) {
    ends->lend[x] = -1;
    ends->rend[x] = -1;
  }
}

/* Function: PrintScores()
 * Date:     ER, Mon 12 April 1999 [St. Louis]
 *
 * Purpose:  print scores
 *
 * Args:     ofp, sqinfoX, sqinfoY, nullsc, pamsc, rnasc
 *
 * Returns:  void.
 */
void
PrintScores(FILE *ofp, struct sc3_s *sc, int doends, struct windowends_s *windowends, int start, int win, int ones)
{
  char   *winn;
  double  othsc, codsc, rnasc;
  double  post_o, post_c, post_r;  /* posterior probs in bits */
  double  sigm_o, sigm_c, sigm_r;  /* sigmoidal scores in bits */
  int     othwinisrev = FALSE;
  int     codwinisrev = FALSE;
  int     rnawinisrev = FALSE;

  if (doends) {
    if (sc->oth->pl < sc->oth->mn) othwinisrev = TRUE;
    if (sc->cod->pl < sc->cod->mn) codwinisrev = TRUE;
    if (sc->rna->pl < sc->rna->mn) rnawinisrev = TRUE;

    print_window_ends (ofp, start, win, "OTH", windowends->fwd->oth, FALSE, othwinisrev);
    if (!ones) print_window_ends (ofp, start, win, "OTH", windowends->rev->oth, TRUE,  othwinisrev);
    
    print_window_ends (ofp, start, win, "COD", windowends->fwd->cod, FALSE, codwinisrev);
    if (!ones) print_window_ends (ofp, start, win, "COD", windowends->rev->cod, TRUE,  codwinisrev);
    
    print_window_ends (ofp, start, win, "RNA", windowends->fwd->rna, FALSE, rnawinisrev);
    if (!ones) print_window_ends (ofp, start, win, "RNA", windowends->rev->rna, TRUE,  rnawinisrev);
  }
  
  if (!ones) {
    othsc = sc->oth->pl + LOG2(1.0 + EXP2(sc->oth->mn - sc->oth->pl)) - 1.0;
    codsc = sc->cod->pl + LOG2(1.0 + EXP2(sc->cod->mn - sc->cod->pl)) - 1.0;
    rnasc = sc->rna->pl + LOG2(1.0 + EXP2(sc->rna->mn - sc->rna->pl)) - 1.0;
  }
  else {
    othsc = sc->oth->pl;
    codsc = sc->cod->pl;
    rnasc = sc->rna->pl;
  }
 
  /* posteriors assuming flat priors for the models */
  post_o = PosteriorLog2(othsc, codsc, rnasc);
  post_c = PosteriorLog2(codsc, othsc, rnasc);
  post_r = PosteriorLog2(rnasc, othsc, codsc);

 /* sigmoidal scores assuming flat priors for the models */
  sigm_o = SigmoidalLog2(othsc, codsc, rnasc);
  sigm_c = SigmoidalLog2(codsc, othsc, rnasc);
  sigm_r = SigmoidalLog2(rnasc, othsc, codsc);
  
  winn = WinnerOfThree(sigm_o, sigm_c, sigm_r);
  
  fprintf(ofp, "winner = %s \n", winn);
  fprintf(ofp, "              OTH = %12.3f             COD = %12.3f             RNA = %12.3f \n", 
	  othsc, codsc, rnasc);
  fprintf(ofp, "   logoddspostOTH = %12.3f  logoddspostCOD = %12.3f  logoddspostRNA = %12.3f \n", 
	  post_o - post_o, post_c - post_o, post_r - post_o);  
  fprintf(ofp, "     sigmoidalOTH = %12.3f    sigmoidalCOD = %12.3f    sigmoidalRNA = %12.3f \n\n", 
	  sigm_o, sigm_c, sigm_r);
}

/* Function: WinnerOfThree()
 * Date:     ER,  Thu Aug  7 06:01:26 CDT 2003   [Benasque]
 *
 * Purpose:  calculates the winner score
 *
 * Args:     ofp, sqinfoX, sqinfoY, nullsc, pamsc, rnasc
 *
 * Returns:  winner
 */
char *
WinnerOfThree(double othsc, double codsc, double rnasc)
{
  char *win;

  if       (othsc >= codsc && othsc >= rnasc) win = "OTH";
  else if  (codsc >= othsc && codsc >= rnasc) win = "COD";
  else if  (rnasc >= othsc && rnasc >= codsc) win = "RNA";
  else Die ("who is the winner?");

  return win;

}

/* Function: WinnerScore()
 * Date:     ER, Mon Sep  3 16:03:32 CDT 2001    [St. Louis]
 *
 * Purpose:  calculates the winner score
 *
 * Args:     ofp, sqinfoX, sqinfoY, nullsc, pamsc, rnasc
 *
 * Returns:  winner
 */
char *
WinnerScore(struct sc3_s *sc, int ones)
{
  char *win;
  double othsc, codsc, rnasc;
  
  if (!ones) {
    othsc = LOG2(EXP2(sc->oth->pl) + EXP2(sc->oth->mn)) - 1.0;
    codsc = LOG2(EXP2(sc->cod->pl) + EXP2(sc->cod->mn)) - 1.0;
    rnasc = LOG2(EXP2(sc->rna->pl) + EXP2(sc->rna->mn)) - 1.0;
  }
  else {
    othsc = sc->oth->pl;
    codsc = sc->cod->pl;
    rnasc = sc->rna->pl;
  }
  
  if       (othsc >= codsc && othsc >= rnasc) win = "OTH";
  else if  (codsc >= othsc && codsc >= rnasc) win = "COD";
  else if  (rnasc >= othsc && rnasc >= codsc) win = "RNA";
  else Die ("who is the winner?");

  return win;
}

/* Function: ScoreWithModels()
 * Date:     ER, Sat Feb 26 10:39:26 CST 2000 [St. Louis]
 *
 * Purpose:  decide which scoring method to use. Called by main().
 *
 * Args:     
 *
 * Returns:  void.
 */
void
ScoreWithModels(FILE *ofp, 
		FILE *regressfp, char *regressionfile, 
		SQINFO sqinfoX, int *isegX, int *iseqX, 
		SQINFO sqinfoY, int *isegY, int *iseqY, 
		char *gss,
		int start, int win, int Lw, 
		struct dos_s      d, 
		struct model_s   *model, 
		struct dpd_s     *dpd, 
		struct dpf_s     *dpf, 
		struct rnascfg_s *mx, 
		struct scores_s  *sc, 
		struct ali_s     *ali, 
		int alignment, int cyk, int doends, int fastintloop, int logodds, int ones, int parse, int rnass, 
		int shuffle, int sre_shuffle, int sweep, int traceback, int verbose, int use_win_id, struct three_times_s time)
{
  struct windowends_s *windowends;
  int                  startX, endX;        /* first and last position of analysis for seqX   */
  int                  startY, endY;        /* first and last position of analysis for seqY   */
  int                  lenX, lenY;          /* len of seq's  without gaps                     */
  int                  fstX, fstY;          /* mapping of startX and startY to the seq without gaps */
  int                  leg_new;
  int                  is_significant;
  int                  scfg_status;
  int                  scfgrv_status;
  int                *iseqshX, *iseqshY;
  int                *seqX,    *seqY;
  double             *freqX,   *freqY;
  double             id, gap, mut;

  windowends = AllocWindowEnds();

  /* default values */
  
  is_significant = FALSE; /* TRUE = OTH score is significant, continue scoring with the COD and RNA models        */
  scfg_status    = FALSE; /* TRUE = the scfg part of the RNA model has already been calculated -- given strand    */
  scfgrv_status  = FALSE; /* TRUE = the scfg part of the RNA model has already been calculated -- reverse strand  */

  lenX = win;
  lenY = win; 
  
  if (sweep) startX = 0;
  else       startX = start;
  startY = start;
  endX = startX + lenX - 1;
  endY = startY + lenY - 1;

  freqX = AllocBaseFreq();
  freqY = AllocBaseFreq();
  
  
  /* (0) leg = minimum length of aligment 
     RemoveJointGaps(lenX, isegX+startX, lenY, isegY+startY, &win); 
     
     *
     * this function can be passed to analyze_2_sequences() in "qrna.c" if 
     * one wants to remove the common gaps of the two alignments globally.
     * The difference is only relevant when using a window.
     */
  
  /*if (verbose) 
    test_OTH_model(ofp, sqinfoX, isegX+startX, sqinfoY, isegY+startY, win, model->cod->COJ, 
		   dpd->oth, sc->ardiag->oth, ali, alignment, traceback, ends->oth);
  */

  /* if (shuffle == TRUE) shuffle aligment */
  if (shuffle || sre_shuffle) {
    AllocIntSeqs(win, &iseqshX, &iseqshY);
    DupIntSeq(isegX, iseqshX, startX+win-1, win-1);
    DupIntSeq(isegY, iseqshY, startY+win-1, win-1);  

    if (shuffle)     Shuffle2IntSequences(iseqshX, iseqshY, win, win-1, win-1, verbose);
    if (sre_shuffle) QRNAIntShuffle(iseqshX, iseqshY, win);

    seqX = iseqshX;
    seqY = iseqshY;
  }
  else {
    seqX = isegX+startX;
    seqY = isegY+startY;
  }

  PercIdSeqs(seqX, seqY, win-1, win-1, &id, &gap, &mut);

  /* (1) duplicate the region (start, start+win-1) of iseg into iseq. 
   *     Remove the gaps, and calculate lenX, and lenY.
   */
  DupIntSeq(seqX, iseqX, win-1, win-1);
  DupIntSeq(seqY, iseqY, win-1, win-1);  
      
  RemoveGaps(iseqX, win, &lenX, verbose); /* len of seqX without gaps */
  RemoveGaps(iseqY, win, &lenY, verbose); /* len of seqY without gaps */

  fstX = PosNoGaps(isegX, startX);
  fstY = PosNoGaps(isegY, startY);

  if (!sweep) {
    fprintf(ofp, "length alignment: %d (id=%.2f) (mut=%.2f) (gap=%.2f)", win, id, mut, gap);
    if      (shuffle)     fprintf(ofp, "(shuffled)\n");
    else if (sre_shuffle) fprintf(ofp, "(sre_shuffled)\n");
    else                  fprintf(ofp, "\n");
    
    BaseComp(ofp, iseqX, lenX-1, lenX-1, freqX);
    BaseComp(ofp, iseqY, lenY-1, lenY-1, freqY);

    fprintf(ofp, "posX: %d-%d [%d-%d](%d) -- (%.2f %.2f %.2f %.2f) \n", 
	    startX, endX, fstX, fstX+lenX-1, lenX, freqX[0], freqX[1], freqX[2], freqX[3]);
    fprintf(ofp, "posY: %d-%d [%d-%d](%d) -- (%.2f %.2f %.2f %.2f) \n", 
	    startY, endY, fstY, fstY+lenY-1, lenY, freqY[0], freqY[1], freqY[2], freqY[3]);
  }

  /* Regression test info.
   */
  if (regressionfile != NULL) 
    PrintScanBanner(regressfp, startX, endX, lenX, lenY, fstX, fstY, freqX, freqY, id, gap, mut, use_win_id, time);
  
  /* Calculate the null-model scores
   */
  score_with_null(ofp, seqX, seqY, win, model->null, sc->null, sc->nullrev, verbose);
  
  /* GLOBAL (global alignment   */
  if (d.global) 
    is_significant = 
      score_global(ofp, sqinfoX, seqX, sqinfoY, seqY, gss, start, win, d, model, dpd, 
		   mx, sc->global, ali, &scfg_status, &scfgrv_status, 
		   cyk, fastintloop, logodds, ones, parse, rnass, sweep, traceback, doends, windowends);
  
  /* VITERBI (diag local aligmnent) 
   */
  if (d.vidiag) 
    is_significant = 
      score_viterbi_diag(ofp, regressfp, regressionfile, sqinfoX, seqX, sqinfoY, seqY, gss, start, win, d, model, dpd, 
			 mx, sc->vidiag, ali, &scfg_status, &scfgrv_status, 
			 alignment, cyk, fastintloop, logodds, ones, parse, rnass, sweep, traceback, doends, windowends);
  
  /* FORWARD (diag local aligmnent)
   */
  if (d.frdiag) 
    is_significant = 
      score_forward_diag(ofp, sqinfoX, seqX, sqinfoY, seqY, gss, start, win, d, model, dpd, 
			 mx, sc->frdiag, sc->ardiag, ali, &scfg_status, &scfgrv_status, 
			 cyk, fastintloop, logodds, ones, parse, sweep, windowends);

  /* SEMI-DIAG local aligmnent. [Here is where we use iseq]  
   */
  if (d.visemi || d.frsemi)
    is_significant = 
      score_semi(ofp, sqinfoX, iseqX, 0, lenX, sqinfoY, iseqY, 0, lenY, gss, start, &leg_new, Lw, d, model, 
		 dpd, dpf->oth, mx, sc, ali, alignment, 
		 cyk, fastintloop, logodds, ones, parse, rnass, sweep, traceback, doends, windowends);
  

  /* FULL local aligmnent.  [Here is where we use iseq]
   */
  if (d.vifull || d.frfull)
    is_significant = 
      score_full(ofp, sqinfoX, iseqX, 0, lenX, sqinfoY, iseqY, 0, lenY, gss, start, &leg_new, Lw, d, model,
		 dpf, mx, sc, ali, alignment, 
		 cyk, fastintloop, logodds, ones, parse, rnass, sweep, traceback, doends, windowends);
  
  if (is_significant && sweep) {
    fprintf(ofp, "length alignment: %d ", win);
    if (shuffle) fprintf(ofp, "(shuffled)\n");
    else         fprintf(ofp, "\n");
    
    BaseComp(ofp, iseqX, lenX-1, lenX-1, freqX);
    BaseComp(ofp, iseqY, lenY-1, lenY-1, freqY);

   fprintf(ofp, "posX: %d-%d [%d-%d](%d) -- (%.2f %.2f %.2f %.2f) \n", 
	   startX, endX, fstX, fstX+lenX-1, lenX, freqX[0], freqX[1], freqX[2], freqX[3]);
   fprintf(ofp, "posY: %d-%d [%d-%d](%d) -- (%.2f %.2f %.2f %.2f) \n", 
	   startY, endY, fstY, fstY+lenY-1, lenY, freqY[0], freqY[1], freqY[2], freqY[3]);
  }
  
  if (!d.vidiag && !d.vifull && !d.visemi && alignment) 
    Print2Seqs(ofp, &sqinfoX, seqX, &sqinfoY, seqY, 0, win);
  
  if (shuffle) {
    free(iseqshX);
    free(iseqshY);
  }
  free(freqX); free(freqY);
  FreeWindowEnds(windowends);
}


void
add_abs_to_ends (int start, struct end_s *ends)
{
  int x;

      for (x = 0; x < MAX_NUM_ENDS; x++) 
	if (ends->lend[x] > -1 && ends->rend[x]-ends->lend[x] > 0) {
	  ends->lend[x] += start;
	  ends->rend[x] += start;
	}
}

void
add_abs_to_end3s (int start, struct end3_s *end3s)
{
  add_abs_to_ends (start, end3s->oth);
  add_abs_to_ends (start, end3s->cod);
  add_abs_to_ends (start, end3s->rna);
}

void
cal_structure (FILE *ofp,
	       SQINFO sqinfoX, int *seqX,  
	       SQINFO sqinfoY, int *seqY, 
	       char *gss, int start, int L, 
	       struct rnamodel_s *rnamodel, 
	       struct rnascfg_s  *mx, 
	       struct sc3_s      *sc, 
	       int cyk, int logodds, int do_nus, int parse, int rnass, int traceback, struct end_s *rnaends, int isrnarev)
{
  int d;
  
  d = abs(rnaends->rend[0] - rnaends->lend[0]);
  
  if (d > 0) {
    if (cyk && ((rnass) || traceback)) 
      {
	if (!parse)
	  CYKTracebackRNA(ofp, sqinfoX, seqX, sqinfoY, seqY, gss, start, L, rnamodel->pi2, mx, do_nus, logodds, TRUE, rnaends, isrnarev);
	else 
	  CYKParseTracebackRNA(ofp, sqinfoX, seqX, sqinfoY, seqY, gss, start, L, rnamodel->pi2, mx, do_nus, logodds, traceback, rnaends, isrnarev);
      }
    
    if (!cyk && rnass) 
      PosteriorRNA(ofp, sqinfoX, seqX, sqinfoY, seqY, gss, start, L, rnamodel, mx, do_nus, logodds, traceback, rnaends, isrnarev);
  }
}

void
score_with_null(FILE *ofp, int *isegX, int *isegY, int len, struct nullmodel_s *null, 
		double nullsc, double nullrevsc, int verbose)
{
  nullsc    = ScoreWithNullDiag(isegX, isegY, 0, len, null);
  nullrevsc = ScoreWithNullRevDiag(isegX, isegY, 0, len, null);
  if (verbose) PrintNull(ofp, null, nullsc, nullrevsc);
}

int
score_global(FILE *ofp, 
	     SQINFO sqinfoX, int *isegX, 
	     SQINFO sqinfoY, int *isegY, 
	     char *gss, int start, int win,
	     struct dos_s      d, 
	     struct model_s   *model, 
	     struct dpd_s     *dpd, 
	     struct rnascfg_s *mx, 
	     struct sc3_s     *sc, 
	     struct ali_s     *ali, 
	     int *ret_scfg_status, int *ret_scfgrv_status,
	     int cyk, int fastintloop, int logodds, int ones, int parse, int rnass, int sweep, int traceback, 
	     int doends, struct windowends_s *windowends)
{
  struct end_s *rnaends;
  int isrnarev = FALSE;
  int scfg_status;
  int scfgrv_status;
 
  scfg_status   = *ret_scfg_status;
  scfgrv_status = *ret_scfgrv_status;

  ScoreWithOTH2(ofp, isegX, isegY, 0, win, model->oth, sc->oth, ones, traceback);
  ScoreWithCOD(ofp, isegX, isegY, 0, win, model->cod, model->null, dpd->cod, sc->cod, logodds, ones, traceback);
  ScoreWithRNA(ofp, sqinfoX, isegX, sqinfoY, isegY, gss, 0, win, model, dpd->oth, mx, sc->rna,
	       ret_scfg_status, ret_scfgrv_status, cyk, fastintloop, logodds, d.nus, ones, parse, traceback);

  if (sc->rna->mn > sc->rna->pl) { isrnarev = TRUE;  rnaends = windowends->rev->rna; }
  else                           { isrnarev = FALSE; rnaends = windowends->fwd->rna; }

  /* since this is "global", ends are the total ends
   */
  windowends->fwd->cod->lend[0] = 0;
  windowends->rev->cod->lend[0] = 0;
  windowends->fwd->cod->rend[0] = win - 1;
  windowends->rev->cod->rend[0] = win - 1;
  
  windowends->fwd->oth->lend[0] = 0;
  windowends->rev->oth->lend[0] = 0;
  windowends->fwd->oth->rend[0] = win - 1;
  windowends->rev->oth->rend[0] = win - 1;
  
  windowends->fwd->rna->lend[0] = 0;
  windowends->rev->rna->lend[0] = 0;
  windowends->fwd->rna->rend[0] = win - 1;
  windowends->rev->rna->rend[0] = win - 1;
  
  cal_structure(ofp, sqinfoX, isegX, sqinfoY, isegY, gss, 0, win, model->rna, mx, sc, 
		cyk, logodds, d.nus, parse, rnass, traceback, rnaends, isrnarev);

  /* add absolute position to ends
   */
  add_abs_to_end3s(start, windowends->fwd);
  add_abs_to_end3s(start, windowends->rev);

  fprintf(ofp, "GLOBAL -- ");
  RNAbanner(ofp, cyk);

  PrintScores(ofp, sc, doends, windowends, start, start+win, ones);

  *ret_scfg_status   = scfg_status;
  *ret_scfgrv_status = scfgrv_status;

  return 1;
}

int
score_viterbi_diag(FILE *ofp, 
		   FILE *regressfp, char *regressionfile, 
		   SQINFO sqinfoX, int *isegX, 
		   SQINFO sqinfoY, int *isegY,  
		   char *gss, int start, int win,
		   struct dos_s      d, 
		   struct model_s   *model, 
		   struct dpd_s     *dpd,  
		   struct rnascfg_s *mx, 
		   struct sc3_s     *sc, 
		   struct ali_s     *ali, 
		   int *ret_scfg_status,int *ret_scfgrv_status,
		   int alignment, int cyk, int fastintloop, int logodds, int ones, int parse, int rnass, int sweep, int traceback, 
		   int doends, struct windowends_s *windowends)
{
  struct end_s *rnaends;
  int   isrnarev = FALSE;
  int   scfg_status;
  int   scfgrv_status;

  scfg_status   = *ret_scfg_status;
  scfgrv_status = *ret_scfgrv_status;

  ViterbiOTHDiag(ofp, sqinfoX, isegX, sqinfoY, isegY, 0, win, model->oth, dpd->oth, sc->oth, 
		 ali, alignment, ones, sweep, traceback, doends, windowends);
   /*if (sweep && sc->oth < 0.0) return 0;*/

  ViterbiCODDiag(ofp, sqinfoX, isegX, sqinfoY, isegY, 0, win, model->cod, dpd, model->null, sc->cod, 
		ali, alignment, logodds, ones, traceback, doends, windowends);  
  ViterbiRNADiag(ofp, sqinfoX, isegX, sqinfoY, isegY, gss, 0, win, model, dpd, mx, sc->rna, 
		 ali, &scfg_status, &scfgrv_status, alignment, cyk, fastintloop, logodds, d.nus, ones, parse, traceback, doends, windowends);
 
 /* Regression test info.
   */
  if (regressionfile != NULL) {
    fprintf(regressfp, "+> %f %f %f\n", sc->oth->pl,sc->cod->pl,sc->rna->pl);
    fprintf(regressfp, "-> %f %f %f\n", sc->oth->mn,sc->cod->mn,sc->rna->mn);
  }
  
  if (sc->rna->mn > sc->rna->pl) { isrnarev = TRUE;  rnaends = windowends->rev->rna; }
  else                           { isrnarev = FALSE; rnaends = windowends->fwd->rna; }
  
  cal_structure(ofp, sqinfoX, isegX, sqinfoY, isegY, gss, 0, win, model->rna, mx, sc, 
		cyk, logodds, d.nus, parse, rnass, traceback, rnaends, isrnarev);
  
  /* add absolute position to ends
   */
  add_abs_to_end3s(start, windowends->fwd);
  add_abs_to_end3s(start, windowends->rev);
 
  fprintf(ofp, "LOCAL_DIAG_VITERBI -- ");
  RNAbanner(ofp, cyk);
  PrintScores(ofp, sc, doends, windowends, start, start+win, ones);

  *ret_scfg_status   = scfg_status;
  *ret_scfgrv_status = scfgrv_status;

  return 1;
}

int
score_forward_diag(FILE *ofp, 
		   SQINFO sqinfoX, int *isegX, 
		   SQINFO sqinfoY, int *isegY,
		   char *gss, int start, int win, 
		   struct dos_s      d, 
		   struct model_s   *model, 
		   struct dpd_s     *dpd, 
		   struct rnascfg_s *mx, 
		   struct sc3_s     *sc, 
		   struct dpsc3_s   *ardiag, 
		   struct ali_s     *ali, 
		   int *ret_scfg_status, int *ret_scfgrv_status,
		   int cyk, int fastintloop, int logodds, int ones, int parse, int sweep, struct windowends_s *windowends)
{   
 int scfg_status;
 int scfgrv_status;

  scfg_status   = *ret_scfg_status;
  scfgrv_status = *ret_scfgrv_status;

  ForwardOTHDiag(ofp, isegX, isegY, 0, win, model->oth, dpd->oth, sc->oth, ardiag->oth, ones);
  /*if (sweep && sc->oth < 0.0) return 0;*/

  ForwardCODDiag(ofp, isegX, isegY, 0, win, model->cod, dpd, model->null, sc->cod, logodds, ardiag, ones);
  ForwardRNADiag(ofp, sqinfoX, isegX, sqinfoY, isegY, gss, 0, win, model, dpd, mx, sc->rna, ardiag, 
		 &scfg_status, &scfgrv_status, cyk, fastintloop, logodds, d.nus, ones, parse);
  
  fprintf(ofp, "LOCAL_DIAG_FORWARD -- ");
  RNAbanner(ofp, cyk);
  PrintScores(ofp, sc, FALSE, windowends, start, start+win, ones);

  *ret_scfg_status   = scfg_status;
  *ret_scfgrv_status = scfgrv_status;

  return 1;
}


/* Function: score_semi()
 * Date:     ER, Sat Feb 26 16:34:13 CST 2000 [St. Louis]
 *
 * Purpose:  this is a little trick, to not to have to implement the full COD/RNA.
 *      
 *           I take the "new aligment" generated by 
 *           ViterbiOTH_L2() and use it to calculate COD and RNA scores
 *           as if it where a "given alignment" therefore as a diag DP
 *
 * Args:     
 *
 * Returns:  void.
 */
int
score_semi(FILE *ofp, 
	   SQINFO sqinfoX, int *iseqX, int startX, int lenX, 
	   SQINFO sqinfoY, int *iseqY, int startY, int lenY,
	   char *gss, int start, int *ret_legnew, int Lw, 
	   struct dos_s      d, 
	   struct model_s   *model, 
	   struct dpd_s     *dpd, 
	   struct othdpf_s  *othdpf, 
	   struct rnascfg_s *mx, 
	   struct scores_s  *sc, 
	   struct ali_s     *ali, 
	   int alignment, int cyk, int fastintloop, int logodds, int ones, int parse, int rnass, int sweep, int traceback, 
	   int doends, struct windowends_s *windowends)
{
  struct end_s *rnaends;
  int     Ltot;       /* total space necessary to allocate a possible alignment of the two sequences */
  int     legnew;     /* length of new aligment generated by full viterbi                            */
  int     isrnarev      = FALSE;
  int     scfg_status   = FALSE;
  int     scfgrv_status = FALSE;
  double  id, gap, mut;

  /* (1) apply full viterbi according to the OTH model
   */
  ViterbiOTHFull(ofp, sqinfoX, iseqX, sqinfoY, iseqY, startX, startY, lenX, lenY,
		 &legnew, Lw, model->oth, othdpf, sc->vifull->oth, ali, alignment, ones, sweep, traceback, doends, windowends);
  
  /*if (sweep && sc->visemi->oth < -300.0) return 0;*/

  /* (2) dump the alignments in iseq and calculate the length of the alignment
   */

  Ltot = (lenX > lenY)? 2*lenX : 2*lenY;
  IntizeGapAsequence(ali->charX, Ltot-legnew, legnew, iseqX, FALSE);
  IntizeGapAsequence(ali->charY, Ltot-legnew, legnew, iseqY, FALSE);
  
  PercIdSeqs(iseqX, iseqY, legnew-1, legnew-1, &id, &gap, &mut);

  if (legnew > 2*Lw) 
    Die ("alignment is larger than allocated space\n");
    
  if (d.visemi) 
    {
      ViterbiCODDiag(ofp, sqinfoX, iseqX, sqinfoY, iseqY, 0, legnew, model->cod, dpd, model->null, sc->visemi->cod, 
		     ali, alignment, logodds, ones, traceback, doends, windowends);

      /*if (sweep && sc->visemi->oth < 0.0) return 0;*/

      ViterbiRNADiag(ofp, sqinfoX, iseqX, sqinfoY, iseqY, gss, 0, legnew, model, dpd, mx, sc->visemi->rna, ali, 
		     &scfg_status, &scfgrv_status, alignment, cyk, fastintloop, logodds, d.nus, ones, parse, traceback, doends, windowends);
      
      if (sc->visemi->rna->mn > sc->visemi->rna->pl) { isrnarev = TRUE;  rnaends = windowends->rev->rna; }
      else                                           { isrnarev = FALSE; rnaends = windowends->fwd->rna; }

      cal_structure(ofp, sqinfoX, iseqX, sqinfoY, iseqY, gss, 0, legnew, model->rna, mx, sc->visemi, 
		    cyk, logodds, d.nus, parse, rnass, traceback, rnaends, isrnarev);
 
      /* add absolute position to ends
       */
      add_abs_to_end3s(start, windowends->fwd);
      add_abs_to_end3s(start, windowends->rev);
      
      fprintf(ofp, "LOCAL_SEMI_VITERBI -- ");
      RNAbanner(ofp, cyk);
      fprintf(ofp, "length alignment: %d (id=%.2f) (gap=%.2f) (mut=%.2f)\n", legnew, id, gap, mut);
      PrintScores(ofp, sc->visemi, doends, windowends, start, start+legnew, ones);    
    }

  if (d.frsemi) 
    {
      ForwardOTHDiag(ofp, iseqX, iseqY, 0, legnew, model->oth, dpd->oth, sc->frsemi->oth, sc->ardiag->oth, ones);
      /*if (sweep && sc->frsemi->oth < -300.0) return 0;*/

      ForwardCODDiag(ofp, iseqX, iseqY, 0, legnew, model->cod, dpd, model->null, sc->frsemi->cod,
		     logodds, sc->ardiag, ones);
      ForwardRNADiag(ofp, sqinfoX, iseqX, sqinfoY, iseqY, gss, 0, legnew, model, dpd, mx, sc->frsemi->rna, sc->ardiag, 
		     &scfg_status, &scfgrv_status, cyk, fastintloop, logodds, d.nus, ones, parse);

      fprintf(ofp, "LOCAL_SEMI_FORWARD -- ");
      RNAbanner(ofp, cyk);
      PrintScores(ofp, sc->frsemi, doends, windowends, start, start+legnew, ones);
    }

  *ret_legnew = legnew;
  return 1;
}

int
score_full(FILE *ofp, 
	   SQINFO sqinfoX, int *iseqX, int startX, int lenX, 
	   SQINFO sqinfoY, int *iseqY, int startY, int lenY,
	   char *gss, int start, int *ret_legnew, int Lw, 
	   struct dos_s      d, 
	   struct model_s   *model, 
	   struct dpf_s     *dpf, 
	   struct rnascfg_s *mx, 
	   struct scores_s  *sc, 
	   struct ali_s     *ali, 
	   int alignment, int cyk, int fastintloop, int logodds, int ones, int parse, int rnass, int sweep, int traceback, 
	   int doends, struct windowends_s *windowends)
{
  int legnew;     /* length of new aligmnt generated by full viterbi  */

 /* apply full-viterbi/full-froward according to the three  models
   */
  if (d.vifull) {
    if (!d.visemi)
      ViterbiOTHFull(ofp, sqinfoX, iseqX, sqinfoY, iseqY, startX, startY, lenX, lenY, &legnew, Lw, 
		     model->oth, dpf->oth, sc->vifull->oth,
		     ali, alignment, ones, sweep, traceback, doends, windowends);
    else {
      sc->vifull->oth = sc->visemi->oth;
      legnew = *ret_legnew;
    }

  /* this one compares the different ways of calculating viterbiOTHFull
   */
  
   if (FALSE) 
      CompareOTHFull(ofp, sqinfoX, iseqX, sqinfoY, iseqY, startX, lenX, startY, lenY, 
		     &legnew, Lw, model->cod->COJ, dpf->oth, ali, windowends);

   ViterbiCODFull(ofp, sqinfoX, iseqX, sqinfoY, iseqY, startX, startY, lenX, lenY, &legnew, Lw,
		  model->cod, dpf, model->null, sc->vifull->cod,
		  ali, alignment, logodds, ones, traceback, doends, windowends);
     
    sc->vifull->rna->pl = 0.0;
    sc->vifull->rna->mn = 0.0;
    
    /* add absolute position to ends
     */
    add_abs_to_end3s(start, windowends->fwd);
    add_abs_to_end3s(start, windowends->rev);
    
    fprintf(ofp, "LOCAL_FULL_VITERBI --under construction -- ");
    RNAbanner(ofp, cyk);
    fprintf(ofp, "length of alignment: %d\n", legnew);
    PrintScores(ofp, sc->vifull, doends, windowends, start, start+legnew, ones);
  }
  
  if (d.frfull) {
    ForwardOTHFull(ofp, iseqX, iseqY, startX, startY, lenX, lenY, Lw, model->oth, dpf->oth, sc->frfull->oth, sc->arfull->oth, ones);

    sc->frfull->cod->pl = 0.0;
    sc->frfull->cod->mn = 0.0;
    sc->frfull->rna->pl = 0.0;
    sc->frfull->rna->mn = 0.0;
    
    fprintf(ofp, "LOCAL_FULL_FORWARD -- under construction -- ");
    RNAbanner(ofp, cyk);
    PrintScores(ofp, sc->frfull, doends, windowends, start, start+legnew, ones);
  }

  *ret_legnew = legnew;

  return 1;
}

void
test_OTH_model(FILE *ofp, 
	       SQINFO sqinfoX, int *isegX, 
	       SQINFO sqinfoY, int *isegY, 
	       int win,
	       struct othmodel_s *oth, 
	       struct othdpd_s   *othdp, 
	       double            *othsc, 
	       struct ali_s      *ali, 
	       int alignment, int traceback, struct windowends_s *windowends)
{
  struct end_s *othends;

  othends = windowends->fwd->oth;

  TestViterbiOTH(ofp, sqinfoX, isegX, sqinfoY, isegY, 0, win, oth, othdp, 
		 ali, alignment, traceback, othends);
  TestForwardOTH(ofp, isegX, isegY, 0, win, oth, othdp, othsc);
  CompareOTH(ofp, sqinfoX, isegX, sqinfoY, isegY, 0, win, oth, othdp, 
	     othsc, ali, alignment, traceback);
}


void
print_window_ends (FILE *ofp, int start, int leg, char *whichmodel, struct end_s *ends, int isreverse, int winnerstrand)
{
  char *strand;
  int   lend, rend;
  int   x;

  if (isreverse) { if (winnerstrand) strand = "*(-) "; else strand = " (-) "; }
  else           { if (winnerstrand) strand = " (+) "; else strand = "*(+) "; }

  fprintf(ofp, "%s ends %s=  ", whichmodel, strand);

  if (isreverse)
    {
      for (x = 0; x < MAX_NUM_ENDS; x++)
	{
	  if (ends->lend[x] > -1 && ends->rend[x]-ends->lend[x] > 0) {
	    
	    /* for the reversed strand, report the ends in the other (the "given")  strand
	     */
	    rend = start + leg - 1 - ends->lend[x];
	    lend = start + leg - 1 - ends->rend[x];
	    
	    ends->rend[x] = rend;
	    ends->lend[x] = lend;
	    
	    fprintf(ofp, "(%d..[%d]..%d) ", ends->lend[x], ends->rend[x]-ends->lend[x]+1, ends->rend[x]);
	  }
	}  
      fprintf(ofp, "\n");  
    }

  else
    {
      for (x = MAX_NUM_ENDS-1; x >= 0; x--) 
	if (ends->lend[x] > -1 && ends->rend[x]-ends->lend[x] > 0) {
 	  fprintf(ofp, "(%d..[%d]..%d) ", ends->lend[x], ends->rend[x]-ends->lend[x]+1, ends->rend[x]);
	}
      
      fprintf(ofp, "\n");  
    }
  
}






