/*
 * ndpi_dga_match.c.inc
 *
 * Copyright (C) 2011-22 - ntop.org
 *
 * nDPI is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * nDPI is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with nDPI.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

/* ******************************************************************** */

/*
  Tor
  The tor protocol uses SSL to contact peers so it could be exchanged with
  standard SSL. However the host names such as

  - www.fgd2iwya7vinfutj5wq5we.com
  - www.qbtxzhetq4s2f.com
  - www.fgd2iwya7vinfutj5wq5we.net

  See also DGA (Domain Generation Algorithm)

  In essence www.<name>.com|net

  To do things properly we should check if host name in the certificate
  exists or if the IP address of the SSL peer resolves to a name. Unfortunately
  for performance reasons we can't afford to do a DNS lookup in nDPI (however apps
  can do it if possible) and thus we have created a heuristic algorithm that tries
  to check the host name (in the SSL certificate) to see if it looks like a
  random name or a real name. We cannot use a dictionary (how can the kernel read a file?)
  and this we use bigrams distribution to decide if the string looks like a
  word or just random chars.

  http://www3.nd.edu/~busiforc/handouts/cryptography/Letter%20Frequencies.html

  DGA Datasets
  - https://data.netlab.360.com/dga/
  - https://github.com/baderj/domain_generation_algorithms
*/

static const char *ndpi_en_bigrams[] = {
  "aa", "ba", "ca", "da", "ea", "fa", "ga", "ha", "ia", "ja", "ka", "la", "ma", "na", "oa", "pa", "qa",
  "ra", "sa", "ta", "ua", "va", "wa", "xa", "ya", "za", "ab", "bb", "db", "eb", "fb", "gb", "hb", "ib",
  "kb", "lb", "mb", "nb", "ob", "pb", "rb", "sb", "tb", "ub", "wb", "yb", "ac", "bc", "cc", "dc", "ec",
  "fc", "gc", "hc", "ic", "kc", "lc", "mc", "nc", "oc", "pc", "rc", "sc", "tc", "uc", "wc", "xc", "yc",
  "ad", "bd", "cd", "dd", "ed", "fd", "gd", "hd", "id", "kd", "ld", "md", "nd", "od", "pd", "rd", "sd",
  "td", "ud", "wd", "xd", "yd", "zd", "ae", "be", "ce", "de", "ee", "fe", "ge", "he", "ie", "je", "ke",
  "le", "me", "ne", "oe", "pe", "re", "se", "te", "ue", "ve", "we", "xe", "ye", "ze", "af", "bf", "df",
  "ef", "ff", "gf", "hf", "if", "kf", "lf", "mf", "nf", "of", "pf", "rf", "sf", "tf", "uf", "wf", "xf",
  "yf", "zf", "ag", "bg", "dg", "eg", "fg", "gg", "hg", "ig", "kg", "lg", "ng", "og", "pg", "rg", "sg",
  "tg", "ug", "wg", "yg", "ah", "bh", "ch", "dh", "eh", "fh", "gh", "hh", "ih", "kh", "lh", "mh", "nh",
  "oh", "ph", "rh", "sh", "th", "uh", "wh", "xh", "yh", "ai", "bi", "ci", "di", "ei", "fi", "gi", "hi",
  "ii", "ji", "ki", "li", "mi", "ni", "oi", "pi", "qi", "ri", "si", "ti", "ui", "vi", "wi", "xi", "yi",
  "zi", "aj", "bj", "dj", "ej", "fj", "gj", "hj", "ij", "jj", "kj", "lj", "nj", "oj", "pj", "rj", "sj",
  "tj", "uj", "wj", "yj", "ak", "ck", "dk", "ek", "gk", "ik", "kk", "lk", "mk", "nk", "ok", "pk", "rk",
  "sk", "tk", "uk", "wk", "yk", "zk", "al", "bl", "cl", "dl", "el", "fl", "gl", "hl", "il", "kl", "ll",
  "ml", "nl", "ol", "pl", "rl", "sl", "tl", "ul", "vl", "wl", "xl", "yl", "zl", "am", "bm", "cm", "dm",
  "em", "fm", "gm", "hm", "im", "km", "lm", "mm", "nm", "om", "pm", "rm", "sm", "tm", "um", "wm", "xm",
  "ym", "zm", "an", "bn", "cn", "dn", "en", "fn", "gn", "hn", "in", "kn", "ln", "mn", "nn", "on", "pn",
  "rn", "sn", "tn", "un", "wn", "xn", "yn", "ao", "bo", "co", "do", "eo", "fo", "go", "ho", "io", "jo",
  "ko", "lo", "mo", "no", "oo", "po", "ro", "so", "to", "uo", "vo", "wo", "xo", "yo", "zo", "ap", "bp",
  "dp", "ep", "fp", "gp", "hp", "ip", "kp", "lp", "mp", "np", "op", "pp", "rp", "sp", "tp", "up", "wp",
  "xp", "yp", "zp", "aq", "cq", "dq", "eq", "hq", "iq", "nq", "oq", "qq", "rq", "sq", "uq", "xq", "ar",
  "br", "cr", "dr", "er", "fr", "gr", "hr", "ir", "kr", "lr", "mr", "nr", "or", "pr", "rr", "sr", "tr",
  "ur", "vr", "wr", "xr", "yr", "as", "bs", "cs", "ds", "es", "fs", "gs", "hs", "is", "ks", "ls", "ms",
  "ns", "os", "ps", "rs", "ss", "ts", "us", "vs", "ws", "xs", "ys", "at", "bt", "ct", "dt", "et", "ft",
  "gt", "ht", "it", "kt", "lt", "mt", "nt", "ot", "pt", "rt", "st", "tt", "ut", "wt", "xt", "yt", "zt",
  "au", "bu", "cu", "du", "eu", "fu", "gu", "hu", "iu", "ju", "ku", "lu", "mu", "nu", "ou", "pu", "qu",
  "ru", "su", "tu", "uu", "vu", "wu", "xu", "yu", "zu", "av", "bv", "dv", "ev", "iv", "lv", "mv", "nv",
  "ov", "rv", "sv", "tv", "uv", "vv", "zv", "aw", "bw", "dw", "ew", "fw", "gw", "hw", "iw", "kw", "lw",
  "mw", "nw", "ow", "pw", "rw", "sw", "tw", "uw", "ww", "xw", "yw", "zw", "ax", "ex", "ix", "nx", "ox",
  "rx", "ux", /* "xx", */ "yx", "ay", "by", "cy", "dy", "ey", "fy", "gy", "hy", "ky", "ly", "my", "ny", "oy",
  "py", "ry", "sy", "ty", "uy", "vy", "wy", "xy", "yy", "zy", "az", "bz", "cz", "dz", "ez", "gz", "iz",
  "lz", "nz", "oz", "pz", "rz", "tz", "uz", "zz",
  NULL };

/* ******************************************************************** */

/* http://practicalcryptography.com/media/cryptanalysis/files/english_trigrams.txt.zip */
/* Top EN trigrams */
static const char *ndpi_en_trigrams[] = {
"the", "and", "ing", "ent", "ion", "her", "for", "tha", "nth", "int", "ere", "tio", "ter", "est", "ers", "ati", "hat", "ate", "all", "eth", "hes", "ver", "his", "oft", "ith", "fth", "sth", "oth", "res", "ont", "dth", "are", "rea", "ear", "was", "sin", "sto", "tth", "sta", "thi", "tin", "ted", "ons", "edt", "wit", "san", "din", "ort", "con", "rth", "eve", "eco", "era", "ist", "ngt", "ast", "ill", "com", "ore", "ive", "nce", "one", "edi", "pro", "ess", "out", "ein", "att", "men", "hec", "esa", "hen", "ina", "eri", "ert", "ame", "iti", "ome", "son", "art", "man", "ean", "ona", "eof", "tor", "hea", "ran", "rin", "ine", "eda", "nto", "ave", "nin", "ove", "oun", "ain", "ant", "str", "eto", "hem", "sof", "per", "nde", "ste", "nte", "eas", "dto", "our", "red", "rom", "tof", "ght", "tot", "ese", "cha", "ica", "hei", "hin", "ide", "ndt", "han", "tan", "lin", "not", "der", "ect", "tra", "igh", "fro", "eat", "sti", "hep", "ndi", "ins", "she", "nal", "pla", "als", "een", "nti", "you", "lan", "und", "nda", "rat", "lea", "can", "has", "nds", "nga", "hel", "hed", "inc", "use", "esi", "gth", "asa", "het", "nts", "hav", "hew", "tho", "but", "nan", "ass", "hef", "ies", "ret", "end", "par", "wer", "cti", "ren", "rec", "cal", "its", "ree", "ene", "rst", "eal", "ana", "nst", "cou", "tur", "min", "ity", "yth", "hey", "eca", "oul", "lle", "ard", "rou", "anc", "ost", "pre", "age", "efo", "les", "ssi", "ema", "eso", "tat", "ath", "wor", "ust", "heb", "ewa", "sho", "ind", "sed", "hou", "lly", "uld", "ase", "ure", "ono", "ele", "enc", "nat", "ead", "whe", "ell", "ble", "kin", "ans", "tic", "ali", "sco", "ero", "whi", "ces", "own", "nta", "act", "ber", "ven", "tim", "don", "dan", "ose", "ice", "isa", "ton", "den", "ngs", "ugh", "nes", "lat", "tal", "edo", "ten", "ime", "eme", "ack", "tes", "ple", "ous", "off", "tto", "chi", "ani", "orm", "ned", "ens", "sha", "mor", "iss", "ite", "nge", "tis", "ora", "lli", "ede", "sse", "ade", "rie", "aid", "emo", "ral", "sit", "oin", "hth", "tre", "any", "ake", "ern", "mer", "ric", "dis", "ish", "oug", "ini", "ong", "ntr", "eli", "wil", "led", "sar", "how", "edb", "ich", "spe", "sea", "lit", "yin", "sai", "ndo", "gin", "shi", "ord", "mon", "ena", "new", "por", "ser", "ial", "ori", "tte", "mar", "epr", "ach", "har", "yea", "tri", "che", "tea", "unt", "omp", "who", "tar", "owe", "rit", "ded", "ors", "day", "hee", "thr", "eir", "ond", "mes", "efi", "had", "ner", "ela", "let", "lso", "ris", "ire", "isi", "met", "ars", "hic", "cen", "ari", "fin", "tob", "nsi", "las", "ope", "lar", "des", "fte", "nit", "sen", "ang", "som", "abo", "sio", "two", "ian", "eis", "tsa", "ngi", "uni", "ses", "rep", "rac", "top", "abl", "eti", "ebe", "eha", "now", "oni", "ves", "fir", "erc", "ofa", "ace", "sal", "get", "app", "ane", "rsa", "nof", "heh", "gre", "win", "car", "ete", "mat", "cho", "lay", "swe", "esp", "pri", "tiv", "rof", "gra", "llo", "cor", "eac", "nis", "dit", "gan", "gto", "eno", "bou", "obe", "esh", "tos", "ery", "rma", "ngo", "ewi", "ara", "rto", "rel", "oma", "ala", "asi", "tst", "utt", "irs", "yan", "lla", "sfo", "ork", "ett", "lth", "sid", "aso", "swi", "ita", "set", "twa", "erm", "epa", "ron", "tit", "aft", "dre", "tle", "mil", "dby", "ale", "pen", "bec", "mbe", "toa", "heg", "sch", "sis", "rti", "heo", "low", "lis", "oll", "war", "alt", "elo", "tro", "cat", "med", "lic", "hil", "ile", "tht", "rem", "rre", "ays", "oli", "rso", "nsa", "omm", "old", "cre", "ata", "ise", "cia", "pos", "ger", "sma", "uti", "sts", "sec", "sbe", "eni", "sre", "lon", "isc", "nse", "nor", "bee", "ano", "nco", "fer", "itt", "sno", "epo", "eon", "eds", "eam", "esc", "fic", "ech", "way", "ved", "ike", "alo", "yof", "ash", "ote", "ook", "eta", "erf", "onc", "emi", "eci", "ats", "erv", "rsi", "sst", "ili", "eed", "ary", "sso", "mth", "vel", "dat", "mea", "esu", "uri", "pan", "rch", "uth", "spo", "wou", "ffe", "sel", "rei", "rge", "rse", "tom", "usi", "ega", "sas", "ssa", "ato", "erw", "ood", "ama", "sat", "ece", "mpl", "tso", "gen", "arr", "dea", "sca", "dof", "ual", "dbe", "ewo", "nso", "rte", "vin", "adi", "ndw", "ndh", "edf", "swh", "see", "toc", "tch", "ewh", "eba", "onl", "tem", "dwi", "err", "lec", "lac", "eop", "tel", "ami", "ehe", "dfo", "ien", "uch", "ndc", "ely", "dst", "ici", "edw", "aus", "nfo", "ntt", "nne", "ewe", "sur", "exp", "bet", "ket", "inf", "etr", "yto", "rde", "rce", "omt", "evi", "vet", "pec", "rai", "arl", "yst", "sou", "him", "ref", "lik", "ges", "cto", "urn", "fou", "lls", "rne", "wha", "tow", "ndr", "das", "ssu", "spr", "opl", "rad", "esw", "onf", "col", "omi", "duc", "mos", "ark", "tai", "ick", "hos", "ult", "emb", "ign", "hoo", "too", "nea", "itw", "ppe", "ffi", "ula", "nam", "mis", "ced", "los", "gam", "nas", "reg", "lie", "olo", "twe", "ann", "tod", "ail", "ota", "iso", "aye", "tco", "gro", "cam", "efe", "onw", "bea", "ngl", "wee", "edu", "rew", "aga", "til", "ode", "ory", "erb", "bac", "len", "nly", "imp", "aro", "ehi", "amp", "mpa", "spa", "ivi", "ict", "stt", "net", "rta", "hre", "erh", "eet", "row", "sor", "epe", "tie", "nch", "tru", "mal", "cte", "ndm", "atc", "fac", "twi", "orc", "dco", "ndp", "fre", "ofs", "arg", "ima", "dec", "eng", "ria", "owi", "eit", "pol", "ked", "inn", "bli", "dow", "etw", "clu", "hol", "air", "rri", "rig", "sla", "rop", "ofi", "toh", "non", "ool", "isp", "mak", "rev", "equ", "lyt", "jus", "lya", "ole", "cit", "pea", "vea", "kno", "oca", "tak", "acc", "cer", "ada", "sup", "ngw", "peo", "dso", "rdi", "tou", "cau", "rov", "tfo", "stu", "pin", "tly", "ied", "ues", "tsi", "awa", "ler", "wel", "epl", "hor", "erp", "hro", "ute", "oss", "tbe", "tyo", "tog", "heu", "byt", "ngf", "ngh", "hal", "ban", "ism", "rol", "atu", "sam", "hom", "dhi", "ild", "gai", "rds", "edh", "ino", "ego", "egi", "llt", "arc", "ram", "oto", "cle", "dhe", "aco", "orl", "cas", "riv", "ofc", "orn", "rna", "urs", "avi", "nar", "ubl", "tti", "mem", "amo", "rme", "qui", "que", "nee", "ndf", "inh", "ink", "eho", "ncl", "ock", "vic", "rts", "ets", "leg", "cla", "tse", "eor", "nci", "sli", "erg", "did", "rco", "bas", "loc", "ebu", "hig", "asp", "ext", "egr", "dia", "tab", "bri", "cei", "ras", "dal", "alc", "run", "ppo", "ofh", "cke", "atw", "cea", "ott", "bal", "nsh", "mai", "dar", "asb", "rfo", "vis", "owa", "ude", "rsh", "ifi", "loo", "hit", "uar", "cet", "ovi", "ndb", "tol", "uto", "lem", "bro", "vid", "rio", "uct", "try", "efr", "sde", "dha", "bel", "bil", "tas", "suc", "ebo", "oro", "sev", "mit", "yco", "nwh", "neo", "rvi", "dev", "eun", "wan", "sig", "ths", "asc", "dsa", "tac", "emp", "ank", "rke", "nhe", "nme", "nre", "ros", "ngc", "may", "unc", "nic", "nni", "ura", "ful", "oco", "osi", "def", "ngr", "tec", "say", "nbe", "yre", "ntl", "dge", "sac", "sfr", "tsh", "isn", "ldi", "arm", "cur", "eag", "ndl", "eld", "smo", "org", "llb", "del", "poi", "iat", "urt", "ker", "sol", "tma", "mus", "bei", "qua", "lys", "dwa", "nma", "hip", "rod", "efa", "inv", "isl", "pon", "yon", "pas", "lia", "reo", "lif", "tia", "cri", "nsu", "rly", "lbe", "oha", "ndd", "ham", "ept", "ebr", "bes", "ngp", "yha", "tme", "lef", "add", "ofm", "mad", "dma", "dur", "mme", "mas", "ize", "ics", "ybe", "dno", "ege", "gat", "dde", "cul", "uta", "meo", "lco", "val", "atr", "ecu", "ged", "ndu", "yar", "det", "isf", "ofp", "spi", "rot", "rob", "ped", "lde", "oti", "eek", "dem", "bef", "mov", "els", "iou", "swa", "erl", "imi", "fam", "nex", "lof", "fan", "oes", "scr", "isw", "ull", "erd", "inm", "soc", "rni", "eel", "apa", "mic", "dep", "rda", "pit", "ogr", "sun", "esf", "ida", "gar", "cks", "rag", "geo", "goo", "lud", "nhi", "ife", "bra", "ddi", "pat", "ceo", "ngb", "kes", "yer", "nwa", "dra", "lyi", "odu", "pub", "inu", "orr", "aya", "lor", "ttl", "hon", "sab", "nha", "ldb", "liv", "edr", "seo", "oad", "rmo", "dle", "ecr", "udi", "gon", "alf", "die", "rwa", "lst", "lig", "rid", "rmi", "ofe", "urc", "sic", "ape", "ayi", "bot", "eye", "fri", "ila", "ryo", "ado", "alm", "sag", "nwi", "wed", "nno", "abi", "rve", "ees", "aki", "fie", "bor", "dou", "eep", "dif", "tet", "orp", "ito", "ayt", "onh", "oba", "tud", "vie", "nyo", "sbu", "dic", "dri", "ofo", "oan", "rya", "rus", "xpe", "nca", "nve", "ily", "hot", "nag", "uts", "lre", "rap", "ecl", "dse", "oup", "leo", "upp", "roa", "nad", "bus", "ntw", "alp", "urr", "yfo", "omo", "onm", "ntu", "ows", "iff", "rsw", "asw", "asn", "hap", "cce", "cro", "ier", "cco", "cap", "lev", "tsw", "ken", "oke", "owt", "asm", "ngm", "cts", "nou", "put", "reb", "tta", "edm", "lio", "ney", "onb", "sfi", "tag", "eem", "umb", "cra", "anu", "swo", "wev", "eig", "ssh", "eus", "eyo", "sim", "gle", "roc", "cie", "log", "reh", "wth", "slo", "eer", "ift", "hev", "fit", "tua", "epu", "gov", "got", "cin", "inw", "inp", "wes", "ago", "ams", "sap", "itu", "oac", "opp", "etu", "wat", "sus", "lai", "arn", "eea", "itc", "isr", "tca", "rld", "alr", "ask", "ctu", "lto", "oge", "oot", "eap", "pti", "edp", "fea", "fil", "twh", "ngu", "ems", "veb", "uil", "mot", "roo", "rpr", "rli", "mpo", "isb", "elf", "dfr", "sei", "oon", "sle", "mou", "nev", "nec", "clo", "iel", "lot", "usa", "apr", "edc", "rer", "lei", "fra", "wea", "nen", "alw", "esb", "bar", "eya", "nei", "nsw", "elp", "ola", "eak", "fen", "esn", "ryt", "orh", "anb", "lsa", "far", "giv", "pic", "cep", "toe", "dpr", "rim", "eab", "pac", "lti", "hof", "asu", "saf", "ews", "stw", "eex", "atm", "mpe", "tba", "tee", "inb", "eav", "dwh", "nom", "mun", "enh", "rro", "pet", "nsp", "mto", "agr", "lew", "rha", "lve", "alb", "odo", "tun", "lwa", "gue", "dca", "rar", "rwh", "pai", "hop", "rog", "nfr", "fal", "onn", "inl", "sep", "bat", "smi", "vem", "cus", "epi", "wal", "dsh", "lab", "thc", "apo", "aan", "odi", "fol", "gof", "squ", "sfa", "bur", "nac", "rga", "vil", "ngd", "oci", "hir", "rhe", "uss", "efu", "ywi", "esm", "ywa", "npr", "ume", "oda", "osa", "dro", "ghe", "ghi", "opr", "van", "lds", "kan", "beg", "rpo", "orw", "isd", "dmi", "goi", "fun", "oar", "ofr", "onp", "rib", "idi", "rwi", "lop", "hto", "edd", "stp", "mag", "sne", "sem", "eru", "adt", "bui", "mmi", "alk", "gal", "ibl", "opu", "ald", "oal", "ysa", "tsc", "idt", "pli", "muc", "num", "sba", "nel", "nia", "enu", "ruc", "lyo", "stm", "bre", "toi", "esd", "dmo", "ryi", "ots", "sda", "vol", "dam", "aca", "rbe", "pul", "won", "exa", "yed", "lse", "aci", "stc", "ofw", "lim", "atl", "ais", "flo", "cel", "rno", "cht", "sty", "boo", "aug", "rys", "upt", "sex", "spl", "rty", "eva", "ysi", "ofb", "ldr", "uit", "cli", "mei", "ior", "doe", "ira", "lyw", "igi", "siv", "nab", "law", "tpr", "lda", "ayo", "rki", "idn", "nio", "iva", "tya", "pho", "coa", "ilt", "chr", "ndg", "phi", "eff", "eyw", "raf", "yis", "asr", "nov", "lyb", "aut", "nap", "fee", "ndn", "gfo", "sub", "ski", "nty", "asf", "oor", "aff", "enn", "sdi", "hur", "stl", "lyc", "dir", "cut", "wen", "afe", "dne", "kth", "plo", "rab", "nfi", "yma", "ofl", "ung", "anw", "tmo", "enw", "gas", "tss", "jec", "foo", "gis", "vat", "ley", "tne", "ush", "rnm", "tsp", "nsc", "gea", "div", "sad", "dsi", "fco", "opo", "fhi", "edl", "yor", "yca", "iev", "ywe", "omb", "ird", "big", "tif", "dli", "rhi", "rca", "gha", "pth", "sgo", "ywh", "ava", "sib", "gwi", "sum", "atp", "leb", "tsu", "cki", "fec", "ncr", "rle", "sme", "cis", "tdo", "olu", "kea", "iew", "yse", "mod", "uce", "fai", "nig", "tye", "tli", "tsf", "rba", "dor", "aba", "enb", "bla", "rsc", "thh", "mpi", "ads", "mmu", "pow", "lfo", "gst", "asd", "egu", "snt", "tno", "adv", "iet", "bya", "nvi", "aha", "ofd", "cov", "pop", "rra", "nli", "lte", "uca", "dwe", "tei", "nue", "fla", "sdo", "ril", "iro", "iam", "tev", "esl", "mpr", "edg", "mew", "ico", "yal", "yle", "ntf", "stb", "lyd", "eft", "orb", "hte", "cem", "ben", "yas", "kee", "ntc", "ntb", "une", "nem", "eou", "ouc", "ppl", "thu", "tex", "icu", "rdo", "sew", "sul", "osp", "exc", "otb", "gne", "ids", "kil", "lov", "sys", "ims", "niv", "oom", "stf", "irc", "ohi", "nmo", "asl", "onv", "tde", "oce", "ebi", "pto", "ldh", "ype", "nie", "nsf", "elt", "ibe", "fig", "npa", "fat", "lym", "tyt", "typ", "dnt", "lue", "uro", "owh", "ohe", "blo", "wri", "hts", "rks", "deo", "hri", "sot", "lyr", "dus", "ayb", "chu", "tfr", "wom", "yde", "bin", "req", "gge", "mel", "ffo", "arb", "sop", "uma", "vio", "inr", "rtu", "ror", "taf", "dab", "mee", "lee", "inj", "mse", "vei", "bed", "bit", "hek", "ury", "ppr", "tep", "dac", "mac", "acr", "mma", "enr", "utw", "onr", "nju", "eys", "fas", "kto", "orf", "pte", "cid", "ysh", "var", "saw", "eaf", "tla", "cka", "sia", "obl", "uen", "rsu", "oki", "uat", "uet", "ajo", "cos", "ckt", "chw", "oas", "llh", "sbo", "few", "mst", "scu", "boa", "rry", "eev", "lma", "lyf", "veh", "ido", "mid", "ois", "teo", "sci", "owo", "uck", "dch", "irt", "opa", "rss", "agu", "rsp", "ule", "eki", "beh", "oye", "nwe", "dom", "ady", "ypr", "nki", "fes", "git", "nfe", "ssp", "lou", "ymo", "urg", "pra", "dme", "sow", "edn", "sav", "icl", "opi", "lyp", "kof", "nla", "ldt", "rav", "tfi", "dot", "rfa", "eau", "nho", "dba", "nim", "cew", "ske", "enf", "rho", "nna", "tpa", "wis", "dfi", "anh", "api", "rpa", "llc", "ipa", "adm", "gla", "tap", "abe", "ety", "gco", "rms", "ldn", "lam", "tam", "yat", "anp", "dla", "iri", "nbu", "emu", "ntp", "uns", "ols", "dpa", "lmo", "gua", "irl", "nef", "rbo", "pal", "urd", "lyh", "sce", "anf", "gni", "rip", "urp", "ify", "nle", "eph", "efl", "woo", "mmo", "iga", "joh", "sra", "isg", "tir", "ttr", "thm", "ydi", "fel", "jor", "dpo", "rtr", "sue", "yso", "dho", "lpr", "rla", "opt", "ksa", "cta", "mpt", "yhe", "dup", "nol", "oso", "hej", "alu", "gli", "goa", "itl", "cko", "thw", "bly", "dsu", "hwa", "tsb", "ofg", "rfi", "otr", "wna", "ohn", "gio", "pur", "mec", "nfl", "zed", "ilm", "dbu", "dun", "ogi", "fli", "ucc", "nba", "fle", "raw", "eiv", "deb", "eot", "unn", "ntm", "evo", "mig", "hai", "ldo", "oat", "nfa", "ssc", "lus", "naf", "iwa", "eup", "cip", "pay", "chs", "atf", "amb", "ody", "agi", "yme", "exi", "tpe", "rpe", "imm", "lye", "tbu", "ctt", "llp", "lha", "tdi", "ils", "eur", "lbu", "tyi", "hun", "hie", "gsa", "nif", "nks", "wnt", "atb", "ddo", "gel", "lep", "msa", "enm", "oms", "ywo", "rye", "dap", "osh", "hco", "apt", "irm", "maj", "ait", "yfr", "ocr", "gho", "yli", "tys", "atd", "upa", "fyo", "nun", "afr", "avo", "adb", "eim", "usl", "tpo", "lad", "owl", "htt", "bod", "dwo", "ffa", "liz", "och", "thl", "lau", "scl", "anm", "vot", "npo", "vit", "rdt", "hti", "daf", "mof", "ief", "fiv", "kel", "pot", "isu", "olv", "enp", "key", "rsf", "rlo", "ncy", "rgi", "chn", "una", "ilo", "ghs", "ods", "ony", "afo", "mef", "oub", "lsi", "thp", "sna", "loy", "pir", "tlo", "hta", "lag", "ych", "dtr", "rns", "cef", "hum", "ofn", "lex", "sef", "ndy", "amm", "gui", "cil", "dpe", "vec", "hus", "kst", "llw", "sgr", "esr", "aph", "uir", "awi", "tmi", "tsm", "ocu", "dru", "jun", "tew", "niz", "osu", "uff", "six", "quo", "nke", "hhi", "dua", "chh", "dyo", "eju", "sil", "tok", "iha", "phe", "wne", "gso", "dsp", "dlo", "nss", "rfr", "eec", "hma", "oit", "dad", "ips", "nbo", "gor", "lta", "rtt", "yno", "eyc", "tig", "wns", "dav", "ray", "hhe", "nmi", "ump", "iza", "noc", "lut", "veo", "tef", "ccu", "aim", "kat", "alg", "dsc", "occ", "ldw", "idd", "job", "anr", "nlo", "fis", "nyt", "ryw", "fhe", "eyh", "upe", "doc", "oru", "ctr", "adu", "ngg", "rsd", "ops", "tus", "syo", "mul", "ogo", "ups", "eck", "fed", "rtm", "teg", "chm", "sby", "tgo", "upo", "tyl", "rbu", "pie", "cec", "sua", "lel", "dee", "nby", "rex", "ued", "lca", "yne", "coo", "meb", "rey", "leh", "ceb", "mbi", "nus", "pme", "gol", "dio", "ldc", "ubs", "aig", "lum", "llf", "ylo", "ibu", "rwo", "std", "gna", "oct", "lib", "lme", "cad", "seb", "utu", "eby", "asg", "ydo", "glo", "mso", "aho", "ayw", "nwo", "ebl", "vir", "ryc", "oof", "lsh", "doi", "uis", "sov", "hca", "tad", "ckl", "lun", "oop", "rva", "gei", "nut", "ntd", "gri", "gul", "thb", "rtw", "rsb", "nua", "ias", "ugg", "rci", "awh", "eei", "mep", "sfu", "hno", "dex", "abr", "pof", "gem", "fma", "tut", "sbr", "aul", "vep", "ayf", "usp", "dly", "oph", "kso", "enl", "smu", "abu", "umm", "npe", "soi", "seh", "spu", "xam", "ybu", "lfa", "ycl", "aun", "pha", "mba", "ngn", "reu", "obs", "oks", "nau", "phy", "wid", "alh", "llm", "ipp", "adr", "chc", "dag", "bys", "jan", "gab", "usc", "nop", "dte", "soa", "afi", "ppi", "kis", "bab", "dva", "imt", "dei", "isk", "gma", "eke", "etc", "hle", "lyn", "lfi", "ewy", "thf", "gun", "sye", "rgo", "rif", "pes", "xpl", "aty", "thd", "nvo", "lid", "emt", "neg", "tsn", "rwe", "dfa", "sfe", "tup", "nos", "afa", "oje", "loa", "ypa", "lwi", "urb", "arp", "bon", "rul", "sro", "ibi", "dsw", "pio", "arw", "utc", "ply", "wyo", "aps", "ejo", "igu", "sly", "dvi", "hst", "rka", "oud", "roj", "nav", "aly", "obi", "ddr", "hoc", "yes", "asy", "tax", "icc", "ybo", "fur", "pis", "why", "dgo", "rpl", "tov", "ssf", "ofu", "mfo", "hoi", "irp", "soo", "tyc", "ige", "gus", "ypo", "lpa", "sju", "lyl", "nep", "nro", "hoh", "dpl", "oyo", "cca", "tpl", "itm", "gou", "eyr", "kon", "xte", "nsb", "gic", "mpu", "yla", "ium", "awe", "ryb", "fus", "ndj", "nsl", "fto", "oka", "zin", "ahe", "udg", "neb", "rum", "dol", "opm", "llr", "roy", "itr", "rbi", "kei", "tby", "lub", "ipl", "oil", "usu", "nct", "tay", "ysu", "ogy", "bam", "yhi", "ova", "hab", "upi", "due", "cop", "lpe", "exe", "oww", "nya", "ael", "dve", "esy", "dul", "joi", "ysc", "oen", "imo", "cru", "ygo", "rug", "uin", "rcu", "tyw", "vew", "bru", "hha", "svi", "pee", "fst", "dbo", "anl", "vef", "ntg", "nsm", "veg", "jac", "tsl", "lap", "sob", "thy", "oho", "gpr", "gag", "atn", "rsm", "zat", "eol", "rfe", "npl", "meh", "hsc", "weh", "idh", "iaa", "dta", "njo", "apl", "dbr", "utf", "env", "rgr", "ltu", "hod", "gfr", "lln", "nja", "mbl", "nbr", "cot", "dda", "ysp", "icr", "isy", "neu", "adc", "eny", "obo", "rsq", "gno", "sod", "eyd", "ssw", "teh", "gsi", "lhe", "dau", "fwh", "etb", "omh", "vai", "eyt", "lld", "nai", "aud", "lip", "yfi", "omy", "wro", "hfo", "rsl", "idg", "eaw", "lsc", "naw", "irf", "yac", "erk", "yit", "ayn", "tge", "oga", "yba", "dgr", "onj", "ryf", "ryp", "emy", "xis", "hme", "tsr", "uan", "itb", "pst", "nys", "lch", "bad", "sos", "boy", "oos", "vor", "oic", "uel", "ahi", "tfa", "ewr", "obu", "elv", "eov", "tbo", "flu", "ceh", "ssm", "dju", "riz", "ifo", "tau", "ayh", "cee", "urf", "wai", "utm", "uso", "xtr", "stg", "urv", "sve", "bul", "tsd", "acl", "yho", "eef", "moc", "tda", "ipt", "lts", "pap", "gir", "usb", "vey", "uic", "ekn", "yet", "psa", "lba", "rgu", "fse", "iod", "owd", "web", "dai", "dim", "sau", "yaf", "azi", "igg", "noo", "itf", "tga", "uli", "ltr", "osc", "lne", "chd", "xce", "oua", "eai", "stn", "iol", "gwa", "rcr", "gur", "utr", "lal", "eks", "aym", "soh", "llu", "jul", "iqu", "wew", "lsw", "utb", "kne", "shu", "voi", "teb", "pel", "uly", "kid", "bse", "iar", "hbo", "lpo", "rbr", "hmo", "fft", "ddl", "dqu", "hwe", "wst", "ytr", "yun", "omu", "hso", "pta", "sri", "oya", "sui", "sge", "utp", "yfa", "rcl", "gse", "adl", "sey", "uge", "kfo", "ryd", "rdw", "poo", "jur", "gwh", "rfu", "pus", "rnt", "itd", "atg", "lak", "fti", "dop", "lyg", "esk", "fpr", "pwi", "lwh", "upl", "doo", "fca", "awo", "icp", "tfe", "ndv", "arf", "gca", "eid", "erj", "onu", "ehu", "nka", "gew", "ymi", "pil", "cum", "edy", "ugu", "vee", "nra", "mys", "rmy", "irr", "tgr", "rup", "kep", "gmo", "gup", "cup", "ldm", "hsa", "ctl", "ndk", "fso", "ouw", "tbr", "keo", "kar", "utl", "sdu", "oex", "imb", "gbe", "obr", "nye", "ttw", "ewt", "ffs", "hwi", "neh", "chb", "lki", "gef", "fav", "oid", "rdr", "gme", "yda", "sga", "gac", "amt", "gia", "ghl", "ryl", "kha", "jam", "bir", "dov", "htw", "haw", "mwi", "noi", "chf", "aur", "gba", "fet", "bje", "fut", "llg", "otw", "kit", "enj", "gsh", "thn", "wif", "lob", "wei", "esg", "fde", "blu", "gav", "hys", "ckw", "fpe", "chp", "kle", "amu", "zon", "ngy", "lll", "sht", "seq", "byc", "jud", "god", "tyf", "rah", "guy", "wto", "dna", "nkt", "uha", "daw", "sdr", "bym", "yte", "bep", "rby", "utd", "nva", "oly", "ftw", "nob", "rym", "gdo", "lfr", "acu", "deg", "olt", "cif", "tty", "pag", "dtw", "fch", "byh", "via", "lph", "yab", "gap", "byp", "otc", "pse", "mco", "hpr", "sss", "hoa", "tcl", "zen", "owm", "dcr", "pou", "hac", "buy", "hse", "rtf", "nyw", "oap", "shm", "bum", "ums", "alv", "chl", "eja", "tju", "wle", "iec", "dcl", "dga", "lyu", "abs", "omf", "nri", "ryh", "bow", "yra", "ofj", "bst", "umi", "yen", "rok", "mur", "gly", "mwa", "wof", "sif", "rui", "yro", "imu", "iag", "lbo", "lsu", "lsp", "mps", "gec", "fmo", "elc", "msi", "sbi", "hni", "unl", "yex", "uty", "seu", "utn", "ouh", "lno", "htb", "lmi", "aln", "ofv", "idu", "ska", "bud", "sug", "mwh", "eew", "odr", "thg", "ksh", "ldl", "rtl", "wre", "tkn", "pso", "acy", "bsi", "gde", "iab", "dig", "ypl", "oys", "aka", "foc", "lga", "ghb", "adw", "rko", "ldp", "dsf", "iny", "upw", "eka", "omc", "ksi", "feb", "ssl", "cab", "lhi", "isv", "heq", "seg", "lty", "ilw", "ffr", "hce", "mbo", "nny", "wni", "hwh", "oxi", "aof", "mir", "fof", "lgo", "nym", "cir", "sru", "wec", "unk", "rsr", "lke", "wsa", "ayl", "gpo", "ybr", "hli", "geb", "mom", "yna", "edv", "sah", "aws", "eyb", "dew", "wic", "lho", "kem", "fmi", "ldf", "yap", "taw", "cth", "rgy", "dti", "gte", "viv", "ypi", "arv", "ipe", "yta", "kly", "kwi", "lav", "umo", "ofy", "rmu", "pun", "eym", "luc", "otl", "fta", "tna", "nyc", "mno", "tsg", "cyc", "olf", "fwa", "psi", "bol", "ddu", "dpu", "dsl", "adn", "siz", "djo", "lwo", "vas", "bis", "jap", "bem", "nqu", "gbu", "yri", "dfe", "haf", "eps", "nyp", "sie", "tue", "elm", "nnu", "npu", "dos", "fon", "amw", "dye", "tip", "arh", "wno", "fpo", "suf", "tts", "xpa", "kwa", "agg", "lvi", "toy", "nah", "mha", "mbu", "dmu", "hae", "ocl", "dog", "meg", "box", "usw", "eil", "rik", "lss", "dss", "bia", "rue", "ckb", "plu", "yad", "ewc", "aru", "rej", "bos", "uea", "nsd", "rjo", "ppa", "hla", "eyf", "nfu", "moo", "hbe", "hra", "dsm", "omw", "csa", "fpa", "idw", "fsa", "gad", "yss", "ewm", "mph", "toj", "ofk", "zer", "ayc", "ygr", "kab", "rtn", "tcr", "yge", "owc", "mhe", "tvi", "yus", "ysw", "sgi", "udo", "igr", "tyb", "tpu", "eje", "rdf", "odt", "gpa", "rut", "xec", "kas", "ifa", "urm", "yam", "owb", "lro", "rdl", "roi", "keh", "yot", "fly", "yti", "yby", "adh", "iwo", "lpl", "url", "ibr", "dah", "ctw", "eif", "cio", "ios", "otp", "ryr", "fem", "map", "hly", "tyh", "iot", "uid", "uag", "icy", "unf", "rau", "lwe", "nko", "ldg", "eah", "msw", "mie", "eyi", "hug", "usf", "tmu", "sir", "hsi", "fbe", "rae", "gyo", "mcc", "hsh", "dya", "fme", "joy", "ydr", "wso", "pau", "nsr", "cai", "gpe", "jer", "jou", "civ", "irb", "fab", "ncu", "gep", "ymp", "hio", "eeo", "obb", "owr", "ftr", "pok", "etf", "rkt", "udy", "fho", "dcu", "kor", "yag", "abb", "etm", "fne", "elb", "hdo", "skn", "tym", "aqu", "otm", "nyi", "irw", "msh", "rru", "gch", "lsf", "sph", "wwh", "ksw", "fno", "mbr", "sjo", "yel", "udd", "ntn", "odd", "goe", "beo", "ckh", "htr", "itn", "cog", "uer", "naa", "ogn", "hns", "hau", "irg", "eyl", "mre", "bay", "cof", "jon", "byr", "adf", "ptu", "kie", "hia", "fsu", "gsw", "ixt", "tyr", "yni", "dgi", "ggl", "oui", "okn", "hde", "icf", "mca", "nup", "hpo", "ldd", "cod", "avy", "gsu", "deh", "icd", "wnf", "uos", "otf", "wma", "eob", "ltt", "yfe", "fap", "byb", "ngv", "pfo", "hpa", "uls", "ckf", "mhi", "ulo", "fwe", "otu", "hag", "ptt", "wco", "cst", "emw", "ggr", "agn", "dys", "ipo", "edj", "elu", "kla", "ipi", "cky", "hoe", "bbe", "rdc", "ysl", "wos", "oko", "nmy", "adp", "fha", "tiz", "eyp", "ghw", "hyo", "ebs", "fsh", "lgr", "tjo", "uga", "ssr", "yev", "gfi", "iai", "usm", "okt", "rek", "tfu", "ngj", "rdu", "kew", "ggi", "egl", "aon", "bio", "shr", "gdi", "oea", "dki", "hmi", "lks", "bov", "kma", "xin", "asv", "gsc", "ysb", "uci", "rua", "tid", "shs", "icb", "xan", "rkw", "onk", "eeh", "iac", "lva", "dfu", "bso", "anj", "dja", "idl", "lpi", "ewp", "afu", "eww", "fif", "nmu", "hne", "tni", "gwe", "upf", "nsy", "kou", "byd", "cci", "ysm", "ymu", "etp", "ycr", "yye", "ius", "tah", "lom", "jos", "gwo", "sym", "ahu", "rls", "ayp", "fdi", "aar", "fid", "gtr", "cqu", "rdp", "beb", "pts", "pid", "mex", "coc", "fwi", "myo", "usd", "diu", "kec", "sky", "gti", "fru", "htf", "kup", "xim", "hdi", "elw", "kal", "irh", "byf", "oam", "hib", "kic", "rdb", "htl", "tqu", "otg", "eut", "ftt", "ube", "bun", "isj", "niq", "igo", "icm", "byi", "ueo", "dyi", "shc", "ssb", "tki", "xpo", "dsb", "xpr", "mob", "wye", "nru", "vac", "ntk", "wsp", "gaz", "kco", "tce", "dyt", "lah", "itp", "gns", "ghh", "bey", "tdr", "lft", "aks", "hch", "hda", "ngk", "geh", "ewb", "upr", "fba", "gsp", "beu", "dbl", "sbl", "fts", "tum", "lur", "rqu", "awn", "eum", "dbi", "mya", "icw", "ckn", "dil", "gos", "dut", "gil", "apu", "dpi", "doa", "rhu", "urh", "fad", "cag", "rtb", "orv", "gbo", "mam", "cow", "mol", "gaf", "axi", "raz", "mik", "dvo", "dfl", "atk", "lge", "msu", "pco", "kwh", "rph", "noa", "eoc", "fau", "uad", "asj", "sgu", "aby", "fgo", "enk", "htn", "rpu", "bai", "xic", "rtc", "hut", "mah", "eey", "ssy", "ckm", "rsn", "idb", "yim", "bye", "lfe", "oim", "voc", "sfl", "oml", "kli", "leu", "acq", "hid", "bag", "swr", "uno", "mab", "byl", "uve", "fei", "yga", "uee", "yve", "hwo", "xas", "hlo", "msc", "jew", "mix", "nyb", "ppy", "iki", "idf", "orj", "uys", "uei", "veu", "ckc", "mfr", "wnw", "pab", "adj", "rpi", "hba", "teu", "ubt", "gie", "yyo", "ckp", "owf", "ypu", "lka", "amc", "nir", "iaw", "wsh", "wol", "otd", "fwo", "kag", "ixe", "lms", "nkn", "ryg", "uke", "tbi", "eao", "lsb", "uba", "yee", "odf", "ayd", "byw", "xes", "ayr", "iph", "mia", "psh", "agl", "khe", "gga", "bew", "gee", "nod", "aas", "laf", "ogu", "urk", "gpl", "zes", "edk", "doy", "stv", "ifh", "ffl", "odw", "fix", "rdm", "ldu", "wot", "hgr", "tpm", "meu", "oum", "raa", "yju", "hfr", "myf", "ewl", "gya", "fsi", "oab", "sni", "xth", "hbu", "wnb", "ubj", "idm", "nid", "wca", "usr", "bom", "oaf", "oev", "xed", "myt", "obj", "hsp", "urw", "woy", "kre", "nbi", "ulf", "hpe", "gev", "jum", "uot", "emc", "htc", "eyg", "tpi", "umn", "shw", "rmt", "tve", "epp", "tsy", "rdh", "nsn", "egy", "tdu", "zet", "etl", "gim", "kyo", "fbo", "sva", "fsp", "idc", "rsy", "iea", "eow", "jim", "byo", "gfa", "mss", "yvi", "omd", "hho", "ubb", "olk", "nur", "yce", "tva", "afl", "bid", "dce", "pma", "tav", "wow", "ysf", "iba", "rox", "ogs", "oou", "hsu", "hyd", "zar", "omn", "odg", "ykn", "omr", "fbu", "rsg", "fex", "ufa", "cku", "nil", "tyd", "hya", "caa", "pbe", "lra", "dob", "zea", "lcl", "uwa", "joe", "ghp", "wet", "aza", "buc", "yop", "amh", "wnp", "bob", "lri", "ako", "hak", "hob", "ipm", "owp", "eeu", "hex", "hfi", "auc", "wop", "nyd", "dke", "fsc", "fag", "lbr", "ilh", "rak", "wwe", "ewf", "sks", "eeb", "pad", "max", "psw", "izi", "ksf", "gop", "rnc", "dgu", "psy", "uip", "egg", "tgi", "oqu", "ntv", "deu", "raq", "axe", "kra", "maa", "iin", "fgr", "dhu", "npi", "exh", "osl", "cac", "rju", "fbr", "ipw", "adg", "eaa", "ieg", "ivo", "mli", "itz", "chy", "osw", "rkc", "eyn", "xac", "cav", "kni", "yjo", "osm", "wag", "mde", "myc", "rvo", "ydu", "etn", "izo", "esv", "ewd", "nbl", "mne", "llk", "wsi", "gaw", "syn", "wnc", "yov", "kef", "tfl", "mwe", "gob", "iwi", "shl", "tuc", "ctf", "ssd", "gmi", "ttu", "eic", "nns", "shf", "dmy", "hyt", "mut", "tgu", "msf", "hpl", "uph", "rfl", "mda", "hif", "nsk", "aze", "wde", "dni", "fdo", "kbe", "idr", "lcr", "imw", "mmy", "rbl", "nkl", "msp", "ikn", "cya", "gru", "keb", "ckd", "nuf", "ovo", "wfo", "tmy", "elh", "wav", "odc", "svo", "xch", "lsm", "cso", "ilb", "kev", "dtu", "byg", "emm", "mum", "tcu", "iis", "htm", "ojo", "rtp", "wse", "gda", "oem", "poe", "mev", "uco", "shb", "tib", "dyw", "tbl", "byn", "gow", "aat", "kse", "gby", "nyf", "atv", "ouv", "iah", "rtg", "bbi", "coi", "olb", "upb", "nyr", "vou", "faf", "fue", "thk", "zan", "okl", "nyh", "ogl", "utg", "ryn", "bsa", "awt", "woc", "taa", "nhu", "ueb", "ouk", "lyk", "kss", "cpr", "csc", "gex", "mle", "ctc", "ggo", "oav", "rdd", "kho", "bvi", "idp", "syc", "hov", "rkn", "sak", "odb", "coh", "anv", "fda", "rud", "ebt", "eko", "upd", "stj", "atj", "aor", "wnh", "chg", "ilr", "wwi", "aal", "lcu", "irn", "jes", "hna", "mey", "pwa", "stk", "amf", "obt", "ytw", "dci", "ilv", "ksc", "ybi", "rja", "obv", "shp", "eeg", "uda", "daa", "keu", "unb", "aja", "fvi", "eos", "upc", "yki", "gif", "oze", "oym", "uew", "olc", "lpu", "csi", "wdo", "wod", "nsg", "lpt", "ilu", "oet", "kir", "lua", "exu", "rkf", "fev", "etd", "ksb", "lod", "lce", "ckr", "pia", "igs", "ckg", "thj", "yru", "tke", "dph", "ubi", "nak", "ucl", "maz", "ubm", "tci", "tiw", "elr", "ghc", "yaw", "gcl", "maf", "awk", "soe", "cob", "elg", "wsc", "oog", "hay", "tyg", "ugs", "cry", "wli", "gta", "lfu", "oir", "odp", "iap", "hyp", "nip", "lps", "byj", "wmo", "icg", "ulp", "rkh", "imf", "oak", "rub", "saa", "fmy", "xtt", "fbi", "lna", "hbr", "usn", "esj", "wir", "fna", "gsf", "uef", "nuc", "cch", "slu", "rsk", "emf", "unw", "kov", "wsw", "wim", "ygi", "olm", "ubu", "tsk", "nry", "itg", "bta", "hqu", "hdr", "oel", "unr", "sja", "sud", "mho", "kwe", "bie", "lgi", "mla", "unp", "fge", "oju", "ltw", "ifu", "riu", "dyn", "ipr", "yfu", "uec", "nje", "esq", "bby", "hah", "imn", "kim", "oag", "thv", "oob", "dsr", "wly", "ksl", "xer", "olw", "roe", "cse", "weg", "ctm", "lsl", "lmu", "pef", "dry", "rnh", "kfr", "ulu", "oty", "cue", "ssn", "wnl", "fga", "ifs", "khi", "smy", "hgo", "ilf", "kgr", "dum", "iju", "lsy", "myb", "ilc", "maw", "myh", "hfa", "pup", "wsu", "oov", "fpl", "gau", "kri", "dib", "ceg", "mwo", "dip", "ghf", "yah", "gib", "bak", "olp", "tuf", "ika", "rtd", "wak", "hog", "nyl", "ioc", "nkw", "iru", "ghm", "syl", "hru", "myw", "efs", "wup", "aml", "ybl", "nph", "iad", "loi", "kam", "hye", "jef", "iao", "xua", "gss", "ntj", "lkn", "dsd", "pda", "fcr", "kbu", "ioa", "gbr", "cyo", "nui", "amn", "eeq", "rps", "rii", "cub", "kba", "bic", "sko", "tsj", "lyv", "yqu", "pfr", "ndq", "xci", "htp", "alj", "iii", "laa", "rnf", "pov", "ilk", "rao", "ags", "usg", "dyb", "imh", "inq", "mok", "bev", "mlo", "kdo", "omg", "mch", "ukn", "gay", "fmu", "wwa", "ycu", "ksp", "fju", "wab", "wem", "amr", "cma", "xhi", "awr", "kca", "lby", "wpr", "wmu", "iem", "fcl", "lsd", "tsv", "cyt", "yem", "pwh", "ouf", "wbe", "ifw", "dwr", "cde", "nay", "gum", "lfw", "tja", "fyi", "aju", "pak", "ieb", "lek", "rgh", "eog", "kro", "ixi", "dje", "gsb", "oyi", "ugo", "rnb", "myp", "sya", "hty", "vad", "dod", "iow", "ysd", "rmc", "yif", "asq", "mym", "ctb", "jen", "ekt", "msb", "mgo", "mdo", "sok", "mau", "shh", "lfl", "kpa", "akt", "pei", "bsc", "ysr", "psc", "fgh", "mdi", "tub", "nwr", "fox", "xcl", "enz", "uwi", "nkr", "umt", "dsn", "ilp", "tyn", "iho", "uyi", "rnw", "laz", "opy", "aer", "yja", "raj", "gye", "wme", "gsy", "wbu", "iof", "hez", "oja", "owu", "ayg", "odl", "amd", "noh", "ecy", "otn", "afg", "jai", "noe", "cdo", "iaf", "eyv", "dyh", "lju", "ekl", "nky", "gyp", "lci", "okp", "eze", "ypt", "yhu", "rnp", "idy", "ogg", "uru", "bbl", "msl", "ffw", "woa", "xto", "umw", "jas", "lil", "wra", "poc", "gcr", "hyi", "ebb", "hvi", "ceu", "tph", "usy", "nrs", "htu", "euk", "myl", "xty", "lsr", "xts", "cfo", "wob", "dnu", "caf", "dka", "myi", "ddy", "odh", "jet", "rnu", "pca", "nkh", "okf", "owy", "fpu", "ibb", "pyo", "nza", "dyc", "sog", "awy", "gsl", "www", "ryu", "nik", "rkb", "cyi", "wbo", "irk", "yow", "yfl", "mra", "niw", "cey", "pgr", "dyf", "dsy", "mye", "azz", "wnm", "diw", "mfi", "ghr", "roh", "pss", "ghd", "hga", "hcl", "dub", "hbi", "eip", "dsk", "fot", "ftb", "lfc", "ghn", "sip", "onz", "nmc", "icn", "htd", "tvo", "eoi", "msm", "fhu", "oah", "xon", "wad", "nog", "kmo", "mna", "pew", "mif", "yup", "uou", "hul", "usk", "osb", "wep", "aku", "yav", "tul", "jin", "ffu", "otj", "pmo", "aac", "gnt", "doz", "mek", "wnr", "yke", "niu", "tnu", "yva", "gfu", "mrs", "hiv", "eiw", "gmu", "ysg", "yph", "gah", "geg", "anz", "iei", "dkn", "dds", "ixo", "yau", "akn", "lbi", "ccl", "rmw", "fye", "xof", "woh", "thq", "ipu", "rur", "gsm", "lej", "jea", "kad", "ctp", "upg", "jay", "imc", "uds", "mcl", "psp", "ayu", "gvi", "ygu", "bha", "uab", "oln", "yvo", "ulc", "snu", "kpr", "kai", "ldq", "igb", "rdn", "cev", "wla", "lru", "vev", "von", "kya", "amy", "uep", "unu", "bth", "zil", "bmi", "pip", "fja", "feo", "llv", "kpl", "buf", "wog", "syt", "hmu", "cui", "ghu", "mni", "icv", "ymb", "ftf", "mtr", "ssk", "ytu", "hfu", "ecc", "gfe", "fos", "myd", "oed", "gdr", "awf", "kia", "nyg", "yll", "cpa", "vig", "fjo", "wss", "lfs", "hfe", "etv", "sms", "emh", "kru", "lug", "cdi", "hcr", "jar", "cwa", "izz", "odm", "gce", "unh", "eez", "irv", "shn", "gbi", "deq", "psf", "hua", "kun", "rkm", "agh", "pep", "lyy", "pbu", "phr", "ysn", "kay", "img", "rje", "egs", "ahm", "ksu", "iwh", "hik", "mro", "duk", "myr", "ulm", "tze", "ewu", "awl", "xit", "rdg", "ewg", "opf", "oaw", "bik", "yci", "wlo", "ewj", "zab", "iog", "ksm", "cys", "ueh", "gpi", "olh", "tih", "kur", "shy", "hok", "wpo", "ieu", "igt", "emn", "oby", "nek", "klo", "ywr", "rnd", "isq", "ulg", "kaw", "psu", "kpo", "ysy", "fia", "gyi", "wba", "lbl", "pfi", "skt", "uie", "byu", "oyt", "sje", "nix", "gnu", "emr", "ohu", "lyj", "csp", "kos", "rez", "ryv", "fdr", "opc", "csh", "ljo", "wsf", "jok", "bof", "wdi", "onq", "bba", "kob", "rkp", "ulb", "hfl", "rkl", "wac", "okh", "pod", "gtw", "iry", "mby", "ooc", "lko", "iwe", "dak", "rwr", "akh", "fce", "uzz", "uwo", "liq", "mbs", "uya", "ldy", "rkr", "lkt", "psb", "rdy", "emd", "ylv", "bib", "eee", "kap", "xti", "gju", "ipb", "ugl", "gpu", "miz", "keg", "doh", "umu", "aes", "xfo", "dlu", "erq", "faw", "odn", "gnm", "hyw", "lax", "hby", "mte", "kac", "ifn", "nyn", "cpo", "oku", "owg", "pru", "twr", "neq", "kwo", "aic", "tvs", "kfa", "nao", "rcy", "nsv", "csw", "phs", "fim", "ixm", "paw", "kme", "wfr", "vag", "gyt", "gom", "osn", "iop", "eod", "sek", "wnd", "lqu", "hup", "msd", "ffb", "kna", "ksd", "ieh", "bco", "bry", "fah", "oau", "fop", "oow", "egm", "ifl", "iko", "dgl", "ffh", "ryy", "kde", "pam", "vej", "fki", "ubo", "hge", "uka", "ltd", "rgl", "eza", "gym", "otk", "lup", "nze", "omj", "wda", "upm", "azo", "shd", "css", "gjo", "syr", "oyd", "kaf", "uas", "igm", "nkf", "wpa", "awm", "umc", "eyu", "moi", "ngq", "ihe", "wam", "eyk", "yai", "pby", "aad", "aii", "gqu", "zeo", "wap", "cgr", "ezo", "hjo", "yid", "hju", "etg", "oue", "lgu", "trs", "cwi", "ctd", "imd", "ptf", "llj", "mta", "iya", "fbl", "mju", "uem", "tej", "wwo", "xha", "oxe", "bap", "egn", "axp", "kci", "kfi", "toq", "naz", "lvo", "kbo", "rmf", "ggs", "tka", "pde", "rml", "exo", "lnu", "utk", "wek", "haz", "ayy", "ptr", "ssg", "kti", "fva", "ffm", "msr", "gry", "dyp", "lux", "pga", "mfa", "ccr", "xco", "fff", "kby", "dsg", "abd", "yob", "byk", "uln", "adq", "oei", "ibo", "hao", "bma", "iml", "gbl", "rsv", "uor", "feu", "kch", "dyl", "fty", "bbo", "ltf", "bca", "bau", "alq", "unm", "pum", "ayj", "eoa", "tey", "cyb", "sox", "eml", "soy", "eia", "wmi", "ekw", "ltb", "epm", "boi", "osf", "myg", "xtu", "oer", "hko", "ulk", "zza", "nkb", "stq", "fci", "ipc", "mgr", "dym", "uki", "lfb", "uye", "kot", "bek", "htg", "wef", "tje", "wfa", "hmy", "pav", "akf", "mfe", "rbs", "yie", "hsw", "fpi", "ftc", "mup", "lja", "ync", "opw", "nlu", "axa", "osy", "iji", "utj", "bto", "cme", "okw", "hyb", "byv", "rmb", "fve", "ezi", "cwh", "ewz", "yoc", "jua", "gmy", "wsm", "uwe", "ysk", "imr", "caw", "goc", "mcg", "fug", "abc", "sut", "kyl", "jaz", "wze", "csu", "cyw", "sgl", "hsl", "gko", "fcu", "lsn", "fyt", "fph", "lui", "ahs", "lry", "zzl", "bah", "okc", "nyy", "rsj", "xmo", "ulw", "cej", "lhu", "haa", "kip", "aum", "wpe", "cbo", "aje", "lsk", "tug", "okm", "aia", "coe", "ptw", "cun", "nez", "ilg", "dvd", "mri", "rnr", "ftu", "hki", "ugi", "cmo", "pyt", "smc", "cyr", "ehy", "ynd", "zei", "saj", "nbc", "cne", "wch", "wje", "nae", "siw", "gve", "geu", "iia", "ldv", "xtd", "cba", "rhy", "rlu", "gdu", "ixa", "tox", "cyp", "skm", "eul", "ufo", "epc", "tao", "wte", "cwo", "bbc", "ndz", "akl", "hva", "ddh", "omv", "amg", "hud", "hub", "mpb", "mud", "hpi", "gsd", "ptc", "ciz", "psm", "azy", "ioi", "zle", "idj", "umf", "shg", "vek", "ggu", "liu", "xtm", "rmp", "zel", "rmr", "dyr", "pty", "skf", "nii", "xcu", "vul", "gwr", "mve", "goh", "lfh", "mge", "ipf", "pmt", "ldj", "cbe", "elk", "huc", "rkd", "mej", "peg", "gey", "eom", "oxf", "mim", "ofq", "aaf", "syd", "wig", "rir", "ixs", "pik", "kta", "hsm", "fef", "dih", "nib", "ksr", "nok", "nej", "mcd", "dty", "fdu", "hpu", "hve", "iae", "eyj", "ewn", "gcu", "iom", "iln", "chj", "wsb", "kol", "opb", "zie", "naj", "lfd", "woe", "ynt", "uez", "chk", "cbu", "igl", "ekh", "pps", "kpe", "dyd", "nih", "lir", "luk", "lpf", "gys", "cmi", "fje", "edq", "itj", "ryj", "iit", "woi", "yak", "fqu", "gut", "igp", "lff", "wks", "wnu", "oxa", "bwa", "ffc", "wsk", "ekr", "kus", "fgu", "vre", "uiv", "hnm", "vik", "osk", "xre", "pyr", "pba", "iep", "upu", "nzi", "okb", "cfi", "mog", "lsg", "cly", "aos", "fke", "moh", "wfi", "hja", "omk", "axo", "zha", "ktr", "ltc", "kda", "rgs", "mao", "ouo", "pya", "kgo", "tyu", "aub", "fka", "kys", "kaa", "smt", "mcr", "xtw", "tek", "hbl", "cvi", "rnl", "pne", "feh", "aap", "cfa", "lao", "nkm", "ioo", "fak", "eih", "gid", "ltl", "zal", "aen", "dmc", "aea", "nyu", "fya", "mop", "tiq", "oyc", "ija", "xtf", "pig", "acs", "pdo", "gki", "pch", "kah", "ieo", "dey", "rmh", "aiw", "ooh", "cim", "mav", "chv", "riw", "tlu", "bug", "nuo", "osd", "gsr", "hyh", "mti", "oai", "dox", "tmc", "hym", "ylu", "vab", "fsw", "olr", "dae", "eln", "cpl", "sej", "eiz", "gaa", "cna", "ksg", "apy", "gfl", "zec", "eij", "eud", "aqi", "ghg", "bdu", "orq", "ueg", "zis", "yef", "roz", "yol", "akr", "wbr", "glu", "ryk", "kdi", "uak", "paf", "yei", "gci", "ifr", "aww", "voy", "mce", "cek", "erz", "ldk", "bog", "ttp", "olg", "arj", "vam", "bsw", "ydn", "zoo", "pmi", "fsm", "zac", "gig", "nji", "dyg", "myn", "yos", "uyt", "ubw", "hss", "cyl", "ubc", "hdu", "ecs", "kbr", "wtr", "osq", "osr", "exs", "smr", "xta", "agm", "mga", "ynn", "lni", "rix", "fib", "wie", "bcs", "ehr", "tyy", "fni", "ymc", "nkc", "gnw", "wng", "kmi", "aji", "tcy", "wau", "pki", "fay", "gyc", "gva", "iau", "tgl", "oec", "cym", "akw", "wsr", "irj", "ulr", "miu", "axc", "shv", "ddt", "ekm", "awb", "oky", "uof", "msn", "leq", "usv", "igc", "epb", "egh", "rku", "ftp", "hsb", "dmr", "oyl", "gnc", "nox", "jak", "hcu", "oif", "rlf", "fvo", "ctn", "lmc", "eay", "csf", "aab", "hke", "lul", "mdu", "nul", "hci", "pno", "mug", "epw", "dao", "oxy", "dez", "psl", "wun", "cak", "xil", "pem", "kiw", "yka", "wex", "uev", "cah", "oef", "anx", "epf", "oza", "cua", "lfp", "kow", "uwh", "arq", "vau", "ubr", "tsq", "auk", "mae", "ahl", "otv", "bsh", "nzo", "utv", "sih", "lol", "wsl", "ffy", "mck", "fov", "peb", "waf", "gog", "aam", "bub", "yje", "mpm", "kul", "wfu", "ulh", "boe", "arz", "kdr", "noy", "ecd", "siu", "lmt", "ugb", "wgo", "eik", "ooo", "fum", "tyv", "hhu", "aed", "ymy", "ynu", "tui", "suk", "aik", "eae", "tww", "vow", "ezs", "csb", "baa", "kif", "oyf", "tii", "cda", "psd", "ovs", "uon", "inz", "xwi", "mci", "iob", "opd", "dek", "xca", "diz", "ekb", "vok", "fsy", "anq", "rnn", "ihi", "nsj", "sku", "wov", "exw", "jav", "oxs", "hyl", "moa", "igd", "xem", "lih", "hnn", "gvo", "ooa", "joa", "cpe", "gph", "cao", "eib", "gnp", "dej", "yil", "mvi", "hyc", "oyw", "nsq", "epd", "ucr", "umh", "zew", "wnn", "ahn", "ibs", "axr", "ikh", "aaa", "ahr", "rmm", "cga", "kte", "taj", "cfr", "gnf", "psr", "teq", "lwr", "hyn", "rmd", "cno", "ssq", "gyw", "foi", "liw", "ckj", "bss", "gja", "ukr", "nxi", "ixw", "zem", "aux", "wpl", "aht", "cex", "bbs", "avr", "wgr", "itv", "xia", "yrs", "gsk", "lmy", "kii", "ltm", "kfu", "rky", "rdq", "xie", "dgm", "pwo", "hry", "rtv", "yeo", "tik", "cyf", "mja", "nhl", "nku", "ooi", "ixy", "abh", "akd", "eku", "nkp", "ttt", "osg", "ffp", "fgl", "ahw", "fsl", "ifp", "ttc", "apm", "pdi", "pfu", "hph", "efc", "gae", "eoh", "eii", "kex", "elj", "gsn", "xat", "hgu", "lyz", "tiu", "xye", "dji", "bte", "etz", "drs", "zas", "rrs", "kop", "ksn", "aag", "ubd", "xpi", "cbs", "tza", "zim", "upy", "fep", "lmw", "gsg", "umd", "daj", "fup", "zsa", "ftm", "bfo", "rih", "epy", "ixp", "igf", "yze", "xma", "ouj", "pwe", "bwi", "lfm", "wip", "eoo", "pms", "bts", "tae", "jia", "ftl", "pbo", "kcl", "mly", "peh", "sik", "rza", "smw", "ayv", "rij", "wcl", "msg", "udl", "ufi", "alz", "wki", "cnn", "pyw", "ewv", "kom", "dzo", "umr", "aiv", "hvo", "skr", "obw", "rtz", "owj", "fob", "zzi", "uxu", "giu", "zam", "pev", "uai", "kyt", "yaa", "sao", "fby", "atz", "mjo", "idk", "cwe", "rdj", "sps", "oeu", "iak", "loh", "yec", "sii", "ooe", "hgi", "bdi", "mdr", "uyo", "pom", "cae", "xid", "coy", "zor", "lpm", "buz", "hsf", "uju", "uyw", "sbn", "fgi", "gyf", "nyk", "mue", "owk", "atq", "iun", "rng", "bej", "eji", "exm", "hii", "hka", "ghy", "eug", "adk", "ifm", "okr", "lii", "euc", "ifc", "xib", "aup", "mki", "aem", "obc", "aue", "mow", "dyk", "nyv", "kau", "aex", "sax", "oxo", "xur", "bwe", "rpm", "mcn", "wej", "jag", "jab", "mru", "rze", "tmr", "awc", "upn", "giz", "ofz", "dko", "feg", "enq", "mfu", "abw", "xio", "ssv", "spy", "gtu", "kum", "wti", "kke", "igw", "etj", "uml", "cze", "yya", "shk", "aeo", "xwe", "ioh", "pcl", "vib", "uac", "vaa", "acd", "ttm", "ubh", "ntq", "cyh", "zak", "cgo", "ekc", "ipg", "hnc", "hms", "lez", "axt", "kvi", "nnw", "smd", "kpi", "ffd", "btt", "xst", "lmm", "rtj", "evs", "ctg", "kej", "gao", "ebc", "yir", "kmu", "rkg", "ttb", "apb", "boc", "pfa", "ekf", "ixc", "saq", "mpf", "wta", "eaj", "ubp", "bho", "lae", "dpm", "lyq", "baf", "wsd", "yiw", "mrb", "wah", "wls", "oeb", "wok", "duo", "cbr", "hnb", "ffg", "ahy", "hyf", "kyw", "kaz", "kki", "xtb", "lkw", "ezw", "daq", "rlw", "dug", "imj", "tyj", "dik", "pyi", "rpt", "csm", "wjo", "uhe", "xwa", "mcm", "udr", "pbr", "fnu", "oia", "mns", "bwh", "eie", "efb", "kok", "cty", "itk", "ymn", "hje", "ufe", "mtu", "lpp", "syi", "cyn", "ptl", "nux", "rdv", "dud", "luf", "mrm", "vyw", "hnu", "paa", "eej", "eio", "ctv", "exf", "awp", "kka", "loe", "msy", "hnw", "uji", "avs", "hiw", "zai", "cig", "mio", "pue", "oxt", "rct", "uko", "agt", "xbo", "apc", "hnh", "wus", "ggy", "mpw", "rlt", "vak", "srs", "ckv", "pah", "emg", "ilj", "tko", "bde", "iov", "vap", "iks", "pey", "waw", "ugt", "ncs", "zwa", "odj", "nnc", "sae", "afs", "kek", "xra", "bsp", "udt", "zio", "rgw", "uva", "awd", "hih", "cic", "wfe", "csl", "vii", "ptb", "ekd", "nkd", "vya", "csd", "ckk", "cfe", "aky", "xor", "spn", "ngz", "euv", "dii", "udb", "ruf", "fwr", "oig", "nnb", "epg", "uia", "btr", "bsu", "byy", "wii", "oxc", "puc", "ygl", "pgo", "usj", "aev", "tzi", "cyd", "eok", "rdk", "bpr", "ayk", "cay", "ezu", "aip", "skh", "ueu", "wke", "gyb", "apw", "ucu", "kyr", "ttf", "ftd", "pez", "hsy", "syw", "ltp", "xwh", "aet", "mhu", "dix", "zef", "aou", "nnh", "bsf", "yog", "jol", "xse", "ipd", "dsv", "cfl", "tzs", "cpu", "geq", "tyk", "iex", "opk", "smf", "aol", "jad", "phd", "iee", "bpa", "mva", "hyr", "vst", "bti", "dps", "aif", "mfl", "gej", "bbr", "ikk", "yty", "dok", "etk", "oxw", "ysv", "dhy", "dsq", "vsh", "vsk", "ghv", "skw", "edz", "yuk", "aeu", "lmf", "axb", "zaa", "icj", "wdr", "akb", "ogh", "cdr", "ntz", "rji", "pex", "auf", "hoy", "kcr", "urj", "nug", "evr", "lmb", "bsb", "scs", "iav", "lix", "cox", "oyh", "nkg", "kav", "ozo", "hiz", "ahp", "wik", "kyi", "uvi", "ixf", "ltz", "ecb", "evy", "opg", "kga", "piz", "nnf", "uoi", "exr", "jui", "zeb", "meq", "weo", "csr", "ehm", "owv", "nmr", "mtw", "wsn", "oiw", "pim", "vys", "unj", "yps", "ogt", "wbi", "aua", "amj", "exb", "udw", "vla", "kak", "uja", "ruz", "iiw", "smb", "efd", "xle", "aqa", "gnb", "ghj", "aop", "xme", "cye", "dyu", "lpb", "vos", "juv", "tll", "hue", "ssj", "drm", "psn", "okd", "htv", "unv", "oyb", "fiw", "kye", "zhe", "wds", "huf", "zma", "yom", "boh", "wge", "khu", "wsy", "azu", "obh", "vez", "mih", "cmu", "ekp", "aib", "fax", "bdo", "cge", "acm", "nnt", "rgt", "ppu", "vse", "axw", "ecp", "fow", "yua", "akp", "kfe", "fkn", "apf", "wiz", "efm", "gke", "suz", "ezh", "mcu", "fry", "idv", "xtp", "waa", "uay", "wcr", "haj", "lsq", "ysj", "ixd", "udh", "xel", "lok", "kjo", "evu", "lje", "bme", "zep", "vth", "oey", "muh", "iej", "yur", "mke", "vro", "fsk", "iek", "xtg", "kdu", "hnd", "zeh", "ahb", "gka", "eaq", "puz", "zur", "nnd", "ifb", "lpc", "ixh", "spm", "faa", "syf", "fio", "koo", "ebp", "tzo", "nuk", "fuc", "lkh", "zth", "gyr", "vsa", "lpw", "wga", "naq", "uks", "dzi", "yza", "aai", "jaw", "ouu", "msk", "xar", "csy", "bch", "biz", "rrh", "yok", "yay", "uza", "uom", "jal", "pmu", "mpc", "zia", "yew", "orz", "rny", "mrc", "iku", "egt", "iim", "btl", "ovt", "ewk", "yko", "abt", "wiv", "usq", "xen", "ogw", "oji", "lmd", "uxi", "wby", "amv", "uku", "fae", "scy", "kja", "pmf", "yep", "weu", "axs", "xtc", "npm", "fkm", "efw", "mez", "zho", "ucs", "drh", "myk", "qin", "qan", "jah", "foa", "kyb", "yeq", "ibm", "uqu", "lsv", "fey", "zzo", "dtv", "toz", "pfe", "cib", "uej", "byz", "dze", "koh", "mii", "ioe", "gek", "ymr", "bcr", "vyc", "mbt", "mqu", "wve", "efp", "miw", "gak", "yic", "uho", "luo", "wyn", "jug", "lqa", "hnt", "oeh", "mgu", "bno", "mgl", "ypm", "oew", "tkm", "bhi", "ahh", "amk", "fko", "wnv", "rnj", "bue", "dsj", "zip", "beq", "jee", "hsd", "gje", "phu", "izu", "aot", "myv", "iio", "xvi", "dts", "yug", "opn", "dnr", "wvi", "hlu", "mka", "pix", "pdr", "hmc", "dui", "gyd", "ulv", "pfl", "smm", "ysq", "acb", "cgu", "gnh", "rlb", "kib", "hnf", "oep", "pht", "bbu", "obd", "xde", "jit", "udu", "cds", "syb", "drb", "hnp", "pcr", "snr", "pmm", "ttd", "zwh", "akm", "rlh", "kpu", "cph", "zag", "fls", "psg", "ubf", "utz", "fsn", "rzo", "akk", "sze", "fog", "aau", "mvp", "cpi", "xbu", "kih", "ksy", "zee", "nyj", "cve", "ffn", "ixg", "pge", "ohy", "apd", "sji", "afc", "nud", "zig", "zad", "acp", "daz", "kut", "tuo", "xcr", "aoi", "aav", "akc", "wru", "yns", "koc", "lfg", "mgi", "eue", "hsr", "smp", "hwr", "yia", "ugm", "bct", "lpy", "cni", "oxb", "baw", "agw", "kfl", "ruk", "ymm", "gps", "aio", "okg", "ahc", "bhe", "oyn", "hui", "gkn", "xne", "koa", "ptp", "fmr", "ciu", "bsd", "kju", "iic", "zli", "xyg", "rtk", "duf", "rpc", "drd", "bip", "ttg", "zap", "ozi", "ukh", "pdu", "gdp", "iby", "sfy", "fyf", "pke", "lmp", "gnd", "rmg", "joc", "uiz", "tvc", "pqu", "odv", "hyg", "myu", "arx", "qwh", "eox", "otq", "ugr", "ghq", "jil", "mmc", "xsa", "kyp", "hnl", "vco", "pmw", "dza", "kva", "foe", "wpi", "nhy", "hkn", "ixr", "dkr", "ixb", "soj", "veq", "sny", "fcc", "gyh", "ptm", "peu", "myj", "kbl", "xho", "mpp", "xga", "aov", "riy", "koi", "wbl", "smh", "obm", "lkl", "gnl", "sza", "wkn", "eaz", "hsn", "huk", "kve", "auo", "yig", "moz", "ryq", "qis", "kod", "lbs", "dcy", "nps", "ogb", "ktw", "cgi", "mrp", "ohl", "emv", "ijo", "xal", "taq", "kmy", "zze", "mtv", "noj", "htj", "onx", "wce", "mrh", "egw", "yeb", "vwa", "ddw", "zto", "kym", "kio", "aaw", "syp", "ecw", "zaw", "tds", "kyh", "iwr", "pci", "mib", "lkr", "pcs", "bne", "rgm", "aoc", "whu", "ksv", "xli", "axy", "yut", "lmr", "imk", "wfl", "fss", "rcs", "lsj", "uus", "hnr", "uum", "lgl", "wqu", "hij", "giw", "egb", "fmc", "kyc", "iyo", "agp", "uby", "klu", "yeu", "bna", "moe", "pog", "lmh", "uow", "bpo", "zue", "urq", "hoj", "sdy", "axl", "rgc", "iaz", "jel", "uex", "vyi", "cbc", "imv", "pys", "kje", "yao", "nub", "exl", "laq", "llq", "ezt", "cby", "umg", "dku", "zom", "rnv", "bop", "lcy", "ltg", "ksk", "tji", "imy", "rmn", "eky", "xot", "kmm", "zum", "haq", "ifd", "ocs", "kua", "mms", "uov", "ivy", "uhi", "lkm", "jra", "htk", "laj", "ltv", "kmh", "rax", "fsd", "uam", "kbi", "kik", "yih", "gmc", "ezb", "snd", "uny", "bwo", "yey", "cbi", "mkn", "pbi", "pcu", "shj", "xfi", "nrb", "oeg", "gnr", "yrd", "gyl", "ufr", "vaw", "zco", "kub", "uol", "mwr", "ctj", "aih", "obf", "jis", "kge", "vyt", "ocy", "ogd", "mip", "gaj", "cdu", "lkb", "pna", "eub", "bhu", "bsm", "xmi", "efj", "xag", "pye", "kyd", "mpd", "agd", "xus", "cva", "ooz", "yoa", "roq", "hsk", "hng", "vra", "yod", "aui", "ozz", "ohm", "zbe", "acw", "kgi", "zuk", "zah", "wey", "oaa", "pjo", "piv", "uxe", "xbr", "cfu", "odk", "zzy", "ceq", "pju", "hls", "gms", "ahd", "xbe", "vod", "egp", "abn", "pif", "kao", "pvi", "ucy", "tml", "gyu", "cuf", "buk", "oxh", "ecf", "axd", "moy", "bcn", "mnu", "tms", "nfc", "fih", "tzw", "ecm", "aec", "eax", "asz", "ftg", "drc", "ogm", "bae", "wny", "ukt", "tuk", "eua", "rkk", "guo", "ekk", "kms", "wju", "wva", "suv", "axm", "ebm", "aeg", "osv", "ugc", "bce", "pek", "ttv", "tzg", "aie", "nzy", "udc", "bsl", "auw", "aow", "fps", "wpu", "sgt", "ndx", "ehn", "knu", "ccs", "agb", "isz", "vah", "hnk", "nnm", "emk", "ecz", "rxi", "pph", "kkn", "ibn", "phw", "wya", "pao", "feq", "poa", "wkw", "ogc", "lkc", "csn", "nck", "aqw", "xwo", "pud", "tct", "rzi", "psk", "ohs", "ncc", "btw", "zwi", "ftn", "hgl", "fsf", "iih", "kqu", "bda", "ogf", "xso", "aoa", "yds", "ktu", "cjo", "jed", "pmc", "pmy", "ltn", "kcu", "oxn", "wsg", "bok", "jib", "ehl", "bsr", "cok", "wnj", "iaj", "vto", "dms", "skb", "lkf", "ngx", "phl", "aoh", "spc", "wgu", "aef", "spt", "rgb", "tez", "muk", "faj", "utq", "fsb", "xdi", "rkj", "exd", "rkv", "mrr", "esz", "nrh", "kce", "zge", "oay", "fhy", "wci", "mpg", "lza", "ihu", "tij", "ifg", "kyf", "xav", "yms", "ccc", "wiw", "irq", "rgp", "lky", "zhi", "okk", "ncb", "ikm", "mrd", "wmy", "bav", "zst", "yum", "xsp", "rpg", "mje", "gok", "wcu", "ugd", "wja", "noq", "inx", "oih", "suh", "bcc", "szo", "uze", "doj", "cud", "ncw", "gyn", "zca", "xii", "dcc", "sbc", "bao", "icz", "iid", "tvt", "ytv", "iye", "mrg", "msv", "vyr", "bgr", "rnk", "ehs", "cyg", "ekg", "eht", "ebh", "xfr", "vip", "bci", "umv", "aak", "kud", "axf", "mpy", "xdo", "tpw", "slt", "uim", "ogp", "skl", "bmo", "egf", "jev", "hji", "iib", "mmm", "oyu", "rfs", "bif", "xba", "lns", "goj", "tmp", "hyu", "hmr", "ddc", "jat", "wax", "uig", "ddm", "yoh", "yaj", "ttn", "khs", "pyf", "iox", "lzo", "spp", "dcs", "efh", "ikt", "upk", "mpk", "cuu", "uyb", "srh", "mrt", "xsh", "acf", "obp", "svs", "aqs", "vds", "miy", "nnp", "aij", "oyp", "mlb", "cuo", "lpd", "aiz", "dtd", "soq", "cdc", "vsc", "lij", "wdu", "rgf", "jeo", "agy", "bmw", "euf", "vyd", "mvo", "ovw", "xro", "uxo", "ofx", "uap", "ezz", "uaw", "yiv", "zya", "xfa", "rbt", "vha", "hrs", "cvo", "uod", "ezc", "mrk", "kko", "zre", "jub", "tku", "ghk", "exg", "mrw", "uyh", "nnr", "yeg", "zym", "pry", "uxa", "dkm", "ugp", "upj", "rgn", "csg", "uzu", "kae", "pyc", "chq", "scb", "tps", "vym", "aah", "ugw", "vpr", "vli", "hsg", "ksj", "scc", "egc", "zof", "vpa", "flc", "ftv", "kmc", "exy", "rbc", "yly", "cbl", "ipn", "lpg", "awg", "aei", "aob", "oux", "ptd", "kiv", "qae", "oeq", "gsq", "iif", "vyl", "uft", "zhu", "pug", "aok", "baz", "mcs", "pce", "yoi", "fld", "zbo", "bfr", "mcf", "eui", "goy", "mrl", "xsc", "nms", "loq", "tzh", "fuk", "bsn", "mnt", "pae", "ecn", "iip", "dij", "nzh", "yzi", "jig", "aom", "fsq", "ncd", "vch", "bcl", "apn", "yrn", "tcs", "vda", "avu", "noz", "xtl", "zol", "ohr", "rbh", "ozy", "acn", "njr", "oxl", "xda", "drj", "kaj", "ebn", "wia", "tvw", "dyj", "fcs", "vca", "piu", "pww", "lmg", "vwh", "uoa", "yji", "nmd", "ipy", "vyo", "kuw", "bim", "ppt", "bcw", "yip", "pgi", "ruh", "hiu", "gji", "vma", "nng", "frs", "oxp", "zir", "ahf", "ezr", "oyr", "hyy", "wlt", "lml", "vyb", "qwa", "uoy", "nyq", "zos", "rck", "phc", "kry", "drp", "wtw", "iir", "thz", "sxi", "bps", "baj", "eoe", "riq", "tvp", "sft", "tzk", "ixl", "koz", "rlc", "nax", "fsr", "uaa", "ujo", "eyy", "wgi", "zun", "snb", "ezd", "ebd", "ydw", "pmp", "pyl", "cnu", "epn", "oaq", "xla", "dkh", "fuj", "lku", "skp", "rgd", "agf", "wry", "oax", "ije", "xop", "btf", "pbl", "fys", "skc", "uah", "oaj", "dwy", "upv", "ouy", "iul", "bih", "npp", "fod", "ubg", "cza", "uey", "uun", "kgu", "dnd", "axh", "obg", "fej", "rll", "uyl", "rpp", "drr", "uop", "koe", "oxm", "wtu", "yng", "pyb", "vyf", "cuc", "vae", "ddv", "zde", "flt", "vri", "rcm", "vne", "faq", "sds", "pdf", "orx", "kmf", "fyw", "egd", "vog", "fcy", "mlu", "taz", "dhr", "zba", "mbh", "wvo", "gih", "rlm", "pyh", "auh", "auv", "itq", "pob", "zul", "tkr", "jih", "lhy", "oht", "lfy", "ykr", "drw", "fom", "zaf", "wnk", "jaf", "vsp", "kph", "lcd", "bpl", "woj", "uif", "sry", "obn", "drg", "scd", "evl", "caj", "gyg", "biw", "wsv", "ruv", "buo", "qur", "mbw", "suw", "zit", "gmr", "jod", "tzt", "dct", "syu", "ugf", "jir", "dgy", "uyf", "mrf", "mui", "vss", "ulj", "erx", "bcu", "ycy", "plc", "lkp", "myy", "skd", "bii", "zeg", "yls", "ukw", "msj", "mls", "dcb", "olj", "scm", "ebw", "gdy", "uaf", "uob", "oip", "cct", "dyv", "xfe", "zsc", "ffj", "apg", "oiv", "mua", "jaa", "lvd", "eux", "evw", "sbs", "csk", "adz", "xhe", "kvo", "pwr", "nnl", "ikr", "avv", "olz", "yeh", "vof", "xsu", "vaj", "ddb", "mko", "kmn", "nmp", "wph", "zbu", "elz", "uyc", "jha", "faz", "pja", "rcc", "btc", "opj", "abm", "sct", "ohw", "nhs", "sdw", "voo", "uzi", "sml", "nkj", "fku", "hmm", "pym", "zys", "zek", "avl", "ybc", "wlw", "uxt", "afp", "bez", "ovc", "gip", "jut", "ncn", "mcb", "jpm", "bni", "oii", "pbs", "ncp", "ctk", "gsv", "cja", "agc", "yhy", "srd", "vao", "vaf", "pej", "kwr", "ydb", "vue", "syh", "bfi", "fms", "vde", "lze", "yul", "drt", "srb", "osj", "npc", "phb", "ttk", "mcp", "udm", "fji", "lrs", "lfn", "viz", "vdi", "mbc", "tdw", "blv", "fyh", "ncm", "yaq", "ezm", "zra", "ezf", "exx", "bjo", "jup", "yku", "bex", "kig", "owq", "uvr", "awu", "cdp", "xpu", "pyd", "fok", "phm", "tmd", "oxv", "pii", "kuk", "vdr", "uym", "ynb", "oie", "mgm", "aep", "zte", "fij", "dax", "xab", "oxr", "icq", "luv", "tcc", "pgu", "pve", "gzh", "euw", "ruw", "php", "lkd", "kog", "trh", "hps", "lcs", "akg", "kuc", "mbb", "tvn", "osx", "hza", "upq", "rlp", "uyu", "wwf", "tcb", "tzp", "bfa", "wwr", "bpe", "wih", "sjr", "abf", "azt", "saz", "uoc", "yii", "zeu", "zfo", "udf", "pua", "xsi", "rmv", "ddn", "tcw", "spd", "zoi", "dbc", "skg", "ahg", "rhp", "gng", "xlo", "puf", "zwe", "mhz", "gub", "pyp", "luz", "wsj", "cii", "prs", "guc", "liy", "dky", "zpa", "scf", "vwi", "pni", "emj", "rfc", "ufc", "iey", "kyn", "pmd", "vom", "dcd", "nwy", "izh", "sgm", "jma", "sll", "biq", "vso", "isx", "uzh", "udp", "tvm", "ynm", "efn", "ynh", "dpc", "lzh", "ukc", "kiy", "tvr", "phn", "ahj", "aew", "iur", "cju", "okv", "vfo", "muf", "tzc", "tzm", "dpt", "yaz", "evc", "dmp", "tdy", "ikw", "pva", "zic", "nmt", "sgh", "pmj", "wws", "ymd", "ynw", "kmt", "rlr", "vex", "euo", "oyg", "snp", "izs", "wdw", "sld", "ptg", "snc", "cmp", "wka", "ixn", "xei", "vsi", "byq", "yzo", "sdc", "fze", "xex", "phk", "vyp", "hkm", "msq", "sgs", "yae", "rff", "zla", "cps", "xxx", "vyh", "gsj", "ezp", "pcc", "ppp", "gmt", "eiu", "nij", "huh", "tpc", "diy", "syg", "pvo", "ivs", "szi", "mmr", "evt", "kuh", "ufl", "khm", "ekj", "ebf", "acg", "qth", "yuc", "omq", "phh", "abp", "tyq", "yrh", "frh", "vvy", "tcm", "mts", "gty", "cwr", "ejr", "voa", "swu", "fmp", "fek", "iyu", "ehw", "dnc", "fuz", "scw", "tnd", "zmi", "wwc", "qia", "zda", "ddf", "ehd", "fiz", "mbp", "rpw", "scp", "dpp", "fyu", "ncf", "bve", "wlc", "wdt", "iay", "psv", "aiy", "ufu", "mrj", "oik", "muz", "hsq", "xet", "tcp", "iil", "phf", "spb", "svu", "aek", "drf", "fip", "flp", "duv", "oek", "efy", "nmm", "sfc", "nrd", "ipv", "edx", "ifj", "unq", "ddp", "niy", "yoo", "iig", "xya", "mbd", "tmt", "zwo", "ipk", "drk", "vpo", "ffk", "mmt", "ttj", "bcp", "ftk", "eix", "xun", "vsw", "ydy", "npt", "nbb", "uyn", "suo", "tdp", "rcb", "jog", "kgl", "kui", "zmo", "dmd", "xgo", "hiy", "vmo", "rrb", "bju", "azh", "rpd", "bua", "uzo", "jrw", "pmb", "ypn", "nvs", "afb", "ppm", "dcp", "vij", "azd", "tkh", "fza", "drl", "ovr", "miv", "jrs", "rpf", "zpr", "oeo", "ouz", "fkr", "waz", "uoh", "ikl", "exn", "lnc", "xad", "tiy", "tvb", "oib", "foh", "fud", "gts", "rbb", "kyu", "djs", "lbc", "usz", "iez", "rcd", "uek", "rft", "ubn", "iie", "ooy", "dld", "igy", "vdo", "coz", "uxs", "ppc", "gcy", "ewq", "aod", "evp", "kez", "zop", "qat", "ddj", "nkv", "gxi", "ahk", "qai", "wul", "ylt", "bax", "biu", "huy", "aqt", "maq", "ovp", "aeb", "oxd", "rjr", "dmm", "fmt", "eyq", "zyn", "ovd", "zyt", "yjr", "tgm", "ddd", "vim", "yry", "uvo", "bdr", "bcb", "ppw", "moj", "fao", "fcw", "boj", "nnn", "ovb", "hpm", "ugn", "tdb", "dcn", "nnj", "tcd", "afw", "epj", "xxi", "fkh", "bsg", "gnn", "vga", "bnp", "gud", "rvs", "jik", "stz", "sez", "rzy", "jst", "yny", "xno", "yrt", "joo", "zoa", "ajp", "rrd", "rpb", "hnj", "yib", "dny", "mft", "sfs", "azm", "xiv", "uav", "nzu", "umk", "zuc", "udn", "zlo", "vsm", "ehp", "opv", "hny", "zey", "xhu", "guj", "odz", "umy", "rrw", "xna", "dgh", "uyg", "nww", "keq", "ppg", "sls", "tzb", "vus", "gtv", "tfs", "lmv", "guz", "izm", "urz", "zsh", "efg", "jro", "rgg", "xsy", "kyg", "iok", "uzb", "rww", "dgw", "oao", "jax", "tzl", "zvo", "xap", "ovm", "gny", "sdn", "poh", "azl", "ubv", "foy", "zoe", "mrn", "omz", "ptn", "hrh", "paj", "kht", "rrt", "dcm", "xbi", "uoo", "lox", "pmn", "fii", "epk", "aiu", "hku", "zne", "piw", "ayz", "vye", "tvg", "ivu", "cms", "oio", "mnw", "wlg", "qas", "biv", "kmw", "jau", "sxa", "boz", "pmr", "idz", "mnc", "aay", "nll", "ciw", "ffv", "yio", "mry", "hmp", "btb", "kuo", "dtc", "gaq", "uhl", "esx", "sbp", "cmc", "evd", "duh", "czy", "smg", "hrm", "odq", "gzo", "vbe", "ocm", "ggw", "evm", "zao", "ouq", "xgr", "pct", "zki", "bmc", "ydh", "rlg", "nht", "zod", "tvu", "loz", "eqi", "mpn", "jwa", "ixu", "gii", "ukb", "mml", "ifk", "prc", "krs", "vsu", "trp", "wwt", "zqu", "uxb", "cyu", "lji", "flh", "dhs", "ayq", "hnv", "ozh", "jem", "paz", "xml", "lnw", "xxo", "bga", "bms", "bgo", "ukk", "sbb", "ekv", "vna", "pka", "bge", "tlt", "juk", "ghz", "wwm", "ocd", "rzh", "jov", "eoj", "dtt", "snf", "xod", "wgl", "nvu", "zse", "tvf", "sgy", "vle", "gmp", "vaz", "avn", "ajs", "qho", "dkl", "hcy", "unz", "tnt", "euh", "zdi", "puj", "bja", "ezl", "sqm", "ggh", "dls", "tbs", "pyg", "otz", "fcb", "ltj", "paq", "wao", "xby", "wur", "tpt", "sww", "ydc", "okj", "rbm", "hhy", "vlo", "tvd", "ajr", "tmm", "psj", "atx", "rrr", "dcw", "smn", "cgl", "fzi", "wdy", "bsy", "zsi", "rln", "rbp", "zko", "iiv", "dnb", "rcp", "tcf", "vvi", "luh", "lgh", "gku", "fct", "xru", "zot", "dzh", "izt", "bcd", "tls", "uyp", "ecg", "huo", "tsx", "zzs", "ivt", "woq", "zou", "pax", "buh", "hlt", "evf", "dll", "dns", "gwy", "khr", "aeh", "tdt", "tvl", "uwr", "kku", "pox", "tpp", "rbw", "uib", "ezv", "dbs", "tzr", "azs", "mox", "tbc", "ykm", "ukl", "cyy", "dcf", "pnc", "ccp", "hhh", "hrt", "buq", "spf", "jiw", "nlc", "nhd", "ruo", "yts", "qmi", "ydd", "pkn", "ccw", "dgs", "bpi", "luw", "tnc", "spw", "izw", "bcf", "pcm", "uyd", "pkw", "csv", "nkk", "apk", "zow", "kug", "uup", "uhu", "lmn", "sij", "kij", "sql", "mbf", "tdc", "uxw", "cpc", "vsb", "uae", "wlb", "yik", "hkh", "gmd", "xev", "srt", "nlt", "zzb", "ovh", "yub", "yld", "hpk", "dbh", "nfs", "hdt", "xqu", "ehb", "sdm", "tny", "acz", "wmc", "fdy", "epv", "cdt", "zni", "rcw", "tjr", "ypp", "ibh", "fgd", "bup", "mdc", "drn", "vmi", "jsa", "tlb", "tvh", "scn", "dpg", "ykh", "mty", "src", "ukp", "fyc", "lnd", "awv", "ryz", "nmb", "abk", "qar", "ovn", "yue", "jid", "tzu", "jot", "zyb", "tft", "awj", "nxa", "zev", "dmt", "bki", "hdd", "gza", "uzm", "tnb", "bns", "dlt", "ygh", "mnf", "pje", "aoz", "mct", "koy", "pns", "zyi", "evh", "zid", "yrg", "jii", "fyb", "flw", "rmj", "rsz", "nbh", "nuw", "ydm", "waj", "bsk", "xtn", "mnd", "jri", "vhs", "ksq", "sdv", "dfs", "zna", "ylc", "wlu", "bhp", "zzf", "ocw", "trm", "dfc", "fyp", "hsv", "cpm", "fcp", "nls", "gug", "zus", "crs", "ivr", "evg", "ibd", "obk", "spg", "vsf", "nvy", "rfw", "ynf", "zyo", "yyi", "ufm", "nml", "szt", "zri", "ydl", "jop", "ptv", "rmk", "zzw", "mmw", "jth", "hkr", "fsg", "smv", "sdh", "ovu", "eqa", "dxi", "xri", "nzs", "wty", "xfu", "flr", "grs", "pgs", "jow", "fik", "ddg", "lcc", "zog", "afn", "nlv", "ppf", "ebj", "bnt", "rfb", "vte", "hrb", "mds", "asx", "fzo", "zzt", "ymt", "srp", "kuz", "mku", "ipj", "dbb", "knr", "zyw", "zfe", "mbm", "ggp", "nsz", "amz", "dhd", "nhc", "muj", "uio", "xyo", "huw", "zsp", "oxg", "ohd", "gik", "zik", "kmr", "qwe", "pnu", "pcb", "ncz", "viw", "dzu", "izb", "ccd", "aoy", "nft", "hds", "zro", "hzo", "sbh", "zav", "sdp", "pml", "ggm", "uyr", "bcm", "hwy", "goz", "vwo", "djr", "ocb", "azr", "eiy", "zmu", "ymf", "tcn", "ukm", "hlc", "dhp", "oix", "wwb", "mmd", "eoz", "uiw", "tgh", "nld", "stx", "jja", "yyu", "tzf", "sxp", "hmt", "dyy", "xaf", "mza", "qco", "slr", "bbw", "vdc", "uvs", "ozu", "vni", "yud", "lpk", "vge", "tsz", "wyc", "ezg", "hrd", "aqb", "duw", "yuu", "dpw", "gmb", "enx", "rrc", "fcd", "ioj", "sdb", "dvs", "uhr", "trb", "ibc", "iut", "ovy", "mnb", "afm", "myq", "jie", "dmb", "tmb", "zme", "fmm", "ohc", "lzi", "qha", "ovl", "uky", "nfm", "crt", "avc", "wlx", "iuc", "jae", "sns", "aqc", "vbu", "tdm", "ohb", "xdr", "gbt", "syk", "kuy", "dft", "sdt", "ezn", "ckq", "ieq", "uxc", "jiv", "jeh", "xys", "ivp", "hyj", "exv", "vsd", "igv", "kml", "ftj", "akv", "jpa", "gpm", "bij", "thx", "tdh", "gze", "pyn", "cfs", "ynr", "ovg", "hlb", "fyd", "ziz", "yui", "zyg", "vta", "vno", "mcq", "irz", "cyv", "vsn", "dux", "tuw", "wwp", "trd", "ulz", "vfr", "ziw", "yvs", "vay", "zno", "ovf", "lfk", "ggt", "ixv", "tnf", "txi", "iud", "qub", "tlc", "nzl", "lgy", "vgu", "oae", "wcw", "nfk", "ifv", "mkh", "uhn", "twy", "pmh", "srw", "mji", "oej", "ctq", "rrm", "rcf", "slc", "umj", "bmu", "ygm", "mll", "hyk", "zzc", "bjp", "igj", "ozn", "uii", "yuw", "pzi", "xif", "lpn", "wdc", "rfm", "btu", "jre", "nrc", "hyv", "gjr", "dlb", "bke", "zso", "vsr", "npd", "hze", "hoq", "ydt", "flf", "xtv", "viu", "svp", "yvu", "fcn", "kty", "hdf", "ajm", "xea", "lbf", "idq", "ccf", "zfa", "osz", "jca", "ikb", "slb", "zch", "jic", "cyj", "syv", "vwe", "fym", "siq", "xsl", "diq", "puk", "hrl", "vav", "tld", "ohh", "mmp", "iup", "ipz", "gyz", "dnf", "oxx", "lng", "etq", "vhe", "jrt", "bfl", "dtb", "aao", "ecv", "ruj", "tky", "snl", "kuf", "zfi", "mcw", "cdm", "vun", "hld", "ibw", "soz", "uxd", "qyo", "tmw", "usx", "iaq", "hpc", "zzm", "cvs", "uxf", "aae", "elq", "rwy", "ruy", "rcn", "pdt", "oyk", "bpm", "zub", "txa", "ppd", "nwu", "aee", "zyc", "piq", "tbp", "vtr", "zfr", "hdy", "gcs", "aog", "jpu", "hdm", "xve", "vsl", "pcw", "rbf", "ynp", "ahv", "luj", "ukf", "hlw", "ydg", "vdp", "yks", "lrh", "ucf", "vih", "lnb", "koj", "hlp", "xou", "fll", "uik", "pcp", "nhr", "ltk", "kue", "vbr", "mhy", "tzd", "tuh", "ajj", "pui", "rpk", "bkh", "llz", "ykl", "btp", "yxi", "rht", "wwn", "ccb", "yck", "crc", "mij", "rlj", "dvu", "ovj", "pls", "nnk", "lfv", "rbn", "zov", "nbs", "wae", "mbn", "bfe", "fcf", "jco", "hox", "rdz", "ppb", "vyn", "dlc", "rxa", "nxm", "csq", "xva", "tng", "uok", "mmg", "jna", "exk", "ezy", "lnt", "ejs", "zdo", "tkl", "lfj", "zob", "lgb", "rrp", "evb", "kgb", "hpt", "dbp", "flb", "xgi", "hzi", "evj", "coj", "wrt", "gmm", "fcm", "yrr", "tmh", "ehc", "wsx", "nuu", "xki", "dff", "foz", "mkr", "lvs", "uxh", "ivc", "afd", "poz", "mmb", "rzu", "cez", "cje", "ihs", "gkr", "ajg", "xfl", "ohp", "zky", "hlh", "sbt", "nuh", "tux", "sbm", "phg", "fmb", "ntx", "cuy", "fua", "nnv", "dvr", "rbd", "cik", "ztr", "xyl", "hlg", "cdw", "lct", "oee", "weq", "tfc", "wbc", "rvl", "ctx", "lnh", "jge", "snh", "amq", "mmf", "hpp", "jda", "lbt", "ukg", "bbb", "yoy", "cmt", "gmw", "izc", "dxa", "vti", "ivw", "hpb", "pih", "shq", "efk", "wlf", "tns", "ysz", "vba", "prt", "ajd", "qif", "ooq", "wys", "ypc", "brc", "ibt", "zib", "hdc", "rpn", "btm", "nqi", "zii", "exq", "ccm", "gcc", "zyd", "vcr", "ejm", "dtm", "pmg", "jho", "ypd", "fyr", "csj", "fgm", "avp", "wll", "wyl", "dbm", "wlr", "tbh", "rvu", "bty", "xsm", "xmu", "nzw", "bbt", "chz", "afh", "peq", "glb", "fmw", "jao", "dht", "htq", "sng", "nxb", "abj", "nrt", "rhs", "igk", "pkr", "rqa", "yek", "nhb", "mhs", "vbo", "vjo", "jde", "npb", "xup", "ooj", "dpy", "qal", "bjr", "dtj", "qdo", "hlm", "slp", "zau", "zyk", "npy", "eqb", "qwi", "hdw", "aqh", "ptj", "uau", "hrc", "dng", "apj", "ocp", "psq", "kmd", "sws", "ktv", "ggc", "ikp", "sdf", "jrh", "suy", "uyy", "oyj", "pds", "rvy", "mmh", "nfb", "wdb", "swc", "qip", "vka", "mub", "zbr", "fhr", "zta", "sdl", "wut", "rwu", "grh", "qto", "qba", "hll", "bph", "ikc", "gkh", "xos", "yuh", "tnw", "wwd", "iuk", "rgv", "wwg", "sxv", "dww", "rnq", "ijk", "caz", "oaz", "ocf", "ejp", "sxs", "fbh", "aix", "ukd", "fbc", "vgo", "trn", "zgr", "loj", "oij", "wnq", "wlp", "pdp", "zyp", "lwy", "srm", "cyk", "xsw", "khz", "gij", "nlp", "iqa", "wrs", "mhm", "zay", "ycb", "evn", "azp", "lny", "jeb", "vob", "oyy", "sgp", "pgl", "htz", "kyk", "hsj", "iyi", "lkg", "hmd", "fxi", "puy", "djp", "hkl", "auz", "vyg", "cih", "kza", "guk", "gww", "mws", "ocn", "tix", "aqo", "igq", "avt", "sxm", "mnh", "frd", "acv", "avg", "zoh", "wsq", "vpe", "zga", "vho", "ynx", "crd", "vfi", "huj", "ebv", "iik", "zvi", "xyc", "uog", "azw", "dbt", "hgh", "jof", "xge", "ydp", "ymw", "zyh", "ucd", "szk", "nrw", "flg", "cml", "tmf", "rpy", "tbb", "fpm", "prp", "abg", "ijs", "drv", "xyt", "scg", "rrf", "szu", "cmd", "ejc", "iub", "jom", "vyu", "lbw", "dpd", "rfp", "duq", "skq", "twc", "sfm", "sdj", "alx", "uxl", "nqa", "zpi", "tuv", "gnv", "vsg", "jsi", "vop", "dpf", "grd", "ytt", "gbs", "zif", "skj", "sdd", "euz", "ruu", "dyq", "kkr", "iky", "fbs", "gtt", "trc", "hcc", "oxu", "opq", "nhw", "lnf", "pib", "czo", "vme", "rtq", "mpj", "avk", "nrp", "wlh", "tzv", "siy", "evv", "suu", "dlp", "jaj", "yzs", "brs", "yuz", "sbd", "wdh", "eoy", "tvv", "azn", "zgo", "bpu", "ebg", "oxj", "xbl", "gzi", "tdn", "iua", "zka", "lhp", "gyv", "gpp", "bva", "axn", "ggb", "gbp", "gbh", "zaz", "wcc", "dtp", "rfd", "mze", "zpo", "mtc", "nbn", "cuz", "uxp", "ivn", "swy", "ehh", "zae", "rcz", "zyl", "trt", "rbj", "plt", "crm", "mrv", "fdn", "lbh", "jto", "dml", "nlw", "jek", "frm", "nhp", "ylm", "wko", "gtc", "wij", "wwl", "nws", "azq", "nxs", "npg", "gmf", "ynl", "yky", "biy", "ajk", "kyv", "vpi", "uhs", "ijn", "ezk", "coq", "lnp", "fln", "xju", "vdt", "fpc", "qad", "sgd", "lcw", "njs", "exj", "cnb", "gqi", "axx", "bmp", "mdw", "gez", "ylb", "wtv", "apv", "oox", "nff", "uih", "dpb", "owz", "mdb", "fnb", "xdu", "mtp", "jfk", "bdy", "nyx", "iyy", "lkv", "qso", "cls", "pko", "wlm", "srf", "nbp", "brn", "cgw", "szc", "khw", "hjr", "pcg", "nxt", "jrg", "dgb", "dhm", "bka", "jpe", "zyf", "zuz", "nuv", "jwh", "yoj", "vpl", "mwc", "gwu", "xiw", "zoc", "zsu", "dfm", "jiu", "woz", "khy", "hcm", "ojr", "jrr", "jba", "fez", "ncg", "imq", "vdw", "zss", "qsa", "zly", "axg", "mnp", "ikf", "sqb", "ylp", "uoe", "xub", "tnh", "uzs", "fky", "zcl", "etx", "tfm", "ygy", "lbb", "jhe", "cfc", "fyl", "dtf", "aoo", "zej", "mcv", "ejt", "cdd", "sgw", "tpg", "hhs", "aaj", "nvl", "hmb", "ccy", "ssz", "buj", "mgs", "tpb", "jwi", "dgp", "ycs", "ebk", "ixk", "hcs", "tpd", "nzb", "izr", "dmv", "khl", "nlb", "vif", "uuk", "voe", "bnf", "skv", "zke", "nrm", "hfc", "aqf", "zpe", "vix", "ejd", "tyz", "tdf", "rnz", "grt", "vfa", "jeu", "fiu", "rux", "mcy", "hrw", "zjo", "wmd", "aez", "ydj", "fdh", "ajn", "bfu", "ylw", "qit", "mng", "rhm", "ozs", "cdb", "cbn", "wdf", "ajb", "wib", "lvy", "ueq", "lcm", "ljr", "clc", "xss", "hmw", "nhm", "nuy", "vps", "nwc", "efv", "afy", "szh", "gpc", "mzo", "kny", "mnm", "lcp", "jrb", "yml", "snm", "bgi", "mnl", "fww", "ckz", "auj", "ibf", "huv", "jrn", "vve", "aaz", "gmg", "nwt", "egk", "swt", "xir", "kiu", "lzb", "fdc", "cmy", "udv", "ajw", "xov", "lck", "zpl", "hwb", "rxs", "npf", "dgn", "wmr", "vja", "hdh", "ynj", "jfo", "muo", "qbs", "emq", "mzi", "ydf", "zdu", "vov", "pcf", "mnr", "jrc", "jka", "frb", "kyj", "cdf", "ajc", "avm", "gax", "tnm", "bnw", "jla", "dhc", "mld", "nrl", "jif", "lnm", "qic", "vyy", "swb", "ncv", "krz", "gmh", "fny", "avd", "scj", "gtd", "bmx", "ypy", "cbd", "mlt", "vyv", "axu", "zzp", "qig", "poy", "qma", "mny", "hlf", "yox", "jmo", "bsj", "vut", "hcb", "sgn", "tuy", "tgw", "mpv", "akj", "fpt", "lxx", "wvu", "mww", "mdy", "ybb", "xke", "dfb", "rzb", "ecj", "vva", "rhd", "fds", "cpp", "ajh", "ryx", "swp", "oxk", "fkl", "lbp", "mtb", "nxp", "izd", "avh", "yij", "ufs", "mfs", "fml", "bnc", "lcb", "fxa", "vgr", "fmf", "tgy", "cnc", "nbt", "gml", "pgt", "tdv", "nxe", "jsh", "bbh", "rfh", "ikd", "fmd", "ydv", "xym", "bnb", "ozl", "fbp", "ycm", "iqi", "dgt", "wyt", "nbm", "snw", "frc", "dwu", "cws", "tzy", "dbn", "vdb", "uur", "mlc", "dks", "vfw", "tgs", "xiu", "njc", "muw", "uzc", "xyi", "suj", "fwy", "sgc", "kmb", "cuw", "ycd", "rlk", "jsc", "jbu", "ivf", "rcg", "lxl", "iax", "dnh", "jio", "olq", "bnh", "uje", "ggd", "zbi", "zva", "tzn", "tws", "jue", "dtn", "ynk", "nmh", "guf", "jpr", "wkr", "sqf", "flm", "nzf", "jse", "rrn", "kts", "omx", "ylg", "ylr", "tbm", "nfd", "tlp", "nuz", "ivl", "yrm", "fyn", "cfm", "zex", "jme", "aeq", "mvs", "gls", "rrl", "yoe", "rbg", "qfo", "sxt", "jci", "ucb", "huz", "hlr", "jsp", "vfl", "kji", "kiz", "ahz", "kuj", "zcz", "ivd", "fsv", "xai", "twt", "jmi", "cpt", "vox", "hmg", "bux", "fdw", "dpn", "uhf", "bcg", "zhn", "afk", "zye", "vts", "yct", "vcl", "wym", "ycn", "kkh", "jdi", "sgb", "fnm", "hdv", "buw", "njh", "tpy", "hcp", "rvp", "dsz", "tdd", "tpf", "ovk", "grb", "rvt", "ldz", "iiu", "rgk", "gld", "cft", "lsz", "bnd", "jrm", "qir", "fbb", "cdy", "szy", "qca", "hix", "ohk", "xjo", "hax", "crp", "azb", "zzu", "ycc", "ubk", "tbn", "avw", "oiz", "kmp", "twu", "tdj", "svr", "wld", "njt", "dvl", "xtj", "hbh", "sbw", "mdt", "ajt", "egj", "dqa", "mgh", "grn", "ylh", "dfw", "ezj", "vci", "uxm", "hpd", "gnj", "gbc", "pdc", "vpc", "hky", "fuh", "yrc", "fnc", "azc", "joj", "uws", "mrz", "bnm", "vcs", "jta", "bhs", "mvc", "fzh", "jli", "hiq", "zyr", "uao", "njp", "fgw", "xxv", "uzt", "trk", "kpm", "nrf", "kcy", "szl", "jrd", "nmf", "kjr", "bpc", "xmm", "wdl", "rkz", "ohg", "zcr", "jko", "mtt", "tks", "yfc", "pfc", "dfd", "crh", "wyd", "xxa", "djd", "voh", "nvh", "qvi", "zih", "djc", "yax", "xyf", "mtm", "wdp", "vhi", "tgt", "ppn", "lbj", "fub", "fcz", "bgu", "bsv", "qun", "hdb", "fnf", "syj", "fnd", "dkw", "fyy", "cns", "tgd", "qft", "dtl", "ulq", "svt", "jux", "fyg", "dmg", "wts", "wua", "vph", "igz", "drz", "dmh", "dhb", "hdp", "sbf", "lmk", "lmj", "jbe", "xvo", "oyv", "izp", "mlp", "tbt", "yuj", "rgj", "rhr", "wps", "dkg", "mdh", "nzr", "ygn", "nfy", "mht", "clt", "ivb", "pfs", "bdp", "axv", "oyz", "dnm", "vuk", "fgt", "sdk", "izn", "nmw", "pnp", "fns", "nxr", "hoz", "hrp", "btd", "jch", "xol", "rlv", "phj", "pcd", "png", "uyv", "npw", "vrs", "xob", "pyv", "jep", "ihy", "pnt", "jle", "brt", "ggf", "duz", "hux", "ptk", "rxe", "lbk", "bls", "ioz", "byx", "lrt", "yrb", "wcs", "qst", "zaj", "vdf", "zdr", "vtu", "bsq", "wrc", "nmn", "hbc", "bpp", "tkg", "xgu", "uxk", "lrd", "grp", "dmw", "txv", "xje", "prd", "dmf", "zut", "xau", "lyx", "nwv", "nbj", "obz", "svc", "pvc", "scv", "lbd", "djm", "ikg", "rzw", "vpt", "nzc", "nsx", "egv", "dwt", "guw", "uzw", "tuz", "ywy", "dhw", "pyk", "jcp", "hcd", "bqu", "pji", "aqd", "mfm", "bpw", "cmw", "dhl", "twp", "xnu", "tnp", "sfb", "pgc", "qim", "sfw", "fhs", "xyw", "qab", "tmn", "aaq", "zsf", "udk",

#if 0
"glt", "gtm", "iij", "hfs", "uht", "pll", "srr", "tmg", "gyj", "zui", "vce", "qre", "izf", "nzm", "fpp", "zzr", "voj", "jip", "bbq", "trl", "nrn", "ehf", "uut", "rxb", "nmv", "sqa", "qpr", "vmw", "jhi", "xsf", "xip", "hbb", "mtl", "lhs", "ejb", "cfb", "dvc", "swf", "spx", "fws", "wum", "ozm", "ocq", "djt", "fpy", "pyu", "nxw", "khn", "bmt", "zuo", "zve", "tcg", "smk", "dcz", "frp", "sfd", "nzt", "mxi", "fpg", "nfp", "nrr", "lbm", "zuh", "sxr", "qbe", "fui", "gcd", "aqm", "gpt", "ytl", "cbm", "kcs", "ybp", "svy", "vje", "ykw", "sfp", "mdm", "sff", "kuu", "gtp", "duj", "btv", "wcd", "prf", "ivh", "hpw", "qaw", "xoc", "nmg", "hgm", "uij", "vur", "wdg", "pgw", "yzh", "izl", "zua", "wdm", "udz", "prm", "ilz", "evk", "vwr", "eyz", "spv", "djb", "trw", "bpt", "udj", "qbu", "ixj", "hvs", "izk", "xep", "slf", "mbg", "gbm", "fks", "rqi", "xix", "jbl", "ogj", "hbp", "ggn", "qor", "ylk", "zsl", "hct", "ybm", "huu", "zzh", "vnd", "uhy", "lcf", "aej", "frt", "vru", "twm", "wyr", "ivv", "aoj", "pyj", "ilq", "bds", "uzn", "prb", "ncj", "kzo", "bnu", "kaq", "buu", "swm", "fiq", "ppv", "qam", "jdu", "ojs", "slm", "nwf", "ikv", "czi", "vms", "gkm", "fzu", "pgf", "fgs", "qon", "iuw", "ejl", "nlr", "juj", "jgr", "mgt", "ccn", "zby", "prw", "czs", "pgp", "lrb", "cpd", "wkm", "aoe", "abv", "xps", "epq", "ybs", "rzs", "jrf", "tlh", "svl", "fuf", "yxa", "lrc", "cmb", "bix", "lxi", "lpv", "lcz", "lww", "dtk", "rsx", "lcn", "juc", "lnr", "quh", "rvr", "dbj", "ibp", "dlw", "wkh", "xig", "uoj", "ioy", "wrd", "fck", "knd", "dlr", "oiu", "jcr", "dmk", "xud", "xlu", "djh", "ypg", "iuj", "xph", "ehk", "umq", "sxc", "rkq", "baq", "bft", "xyb", "rcv", "njw", "aey", "ybh", "zpu", "dxm", "skk", "kvs", "wlv", "mlf", "xeg", "xom", "uzy", "ozt", "cnt", "ylf", "lgd", "imz", "tgb", "sxb", "mdp", "dws", "jrp", "iuy", "gvs", "pvs", "txt", "tfw", "qbt", "cji", "yiu", "ycz", "zyz", "tqb", "fuw", "auq", "nlh", "nhh", "ejv", "bkn", "uxr", "kyy", "hml", "hcn", "wgs", "uvu", "lbn", "nvr", "pnd", "dfg", "mfc", "wuh", "rws", "tgn", "kky", "wms", "hbs", "cwt", "pnr", "jdo", "rfg", "dnp", "ycp", "nxc", "miq", "dlm", "aqp", "wuc", "rrv", "glc", "yww", "yff", "dcg", "hdl", "fvu", "fhp", "qdi", "jbo", "gyk", "blt", "prr", "lvu", "rxu", "tuu", "jjo", "cmm", "bko", "dmn", "hmf", "bnx", "guh", "dfp", "ozw", "nwb", "fpn", "dqi", "gcm", "fkw", "rvc", "ufk", "hpf", "sjs", "dnw", "fdm", "ywc", "gox", "nyz", "lgm", "bvo", "wfc", "mmn", "grc", "dlh", "mkm", "ipq", "ocz", "vbi", "vdu", "hmh", "wkt", "prn", "grm", "ggg", "yxm", "ajl", "fdp", "ejh", "dxe", "dtg", "tkw", "tgp", "jez", "fqa", "uix", "vui", "fqi", "nrg", "jso", "fhm", "hpv", "dwb", "bbf", "mnn", "bmg", "zsm", "jyo", "hks", "uoz", "brd", "nhk", "gtz", "dbw", "lnn", "tjp", "zzg", "qbj", "pzo", "zsr", "ozc", "cmr", "ioq", "bbm", "nxf", "prh", "mdn", "eiq", "sjp", "kbp", "jaq", "dbd", "cmh", "gll", "yej", "luy", "kps", "xkm", "hpj", "jbr", "cnr", "ihl", "elx", "ijr", "bnr", "dxb", "twb", "hrr", "dgc", "cww", "sxe", "tdl", "tdg", "pcv", "btn", "kzi", "aox", "xyz", "nfh", "vsy", "slw", "ccg", "xsb", "wdd", "fwu", "avb", "tpn", "smj", "qaf", "czw", "rpv", "mhw", "vdd", "zgi", "dcv", "ztu", "pgm", "lzs", "gcp", "ejj", "nwm", "nbd", "uaj", "jpo", "cmf", "xja", "lnl", "crb", "zti", "sjc", "ojp", "gcb", "cdv", "rqb", "slv", "vpu", "vko", "nlm", "ejn", "rjp", "nxu", "mgp", "wfp", "wdn", "pdw", "lgs", "vki", "gtb", "tfd", "rjd", "caq", "wcy", "cdn", "dgk", "rhb", "fgy", "cgh", "avf", "jni", "bbd", "khc", "fbn", "bng", "uhh", "nzd", "wkc", "waq", "jga", "aqr", "zug", "hqa", "ytc", "hdn", "xsd", "xee", "txe", "lpj", "wcb", "djw", "ykj", "fwp", "srn", "uqi", "ymh", "pff", "fgc", "jsm", "mkl", "dwp", "tmv", "yrw", "vtw", "pvt", "jno", "wwk", "wtc", "ehg", "xtq", "mtn", "jss", "ltq", "jtw", "nlf", "mdd", "uxn", "sln", "pnw", "djj", "hgy", "bmd", "zru", "ygw", "xaw", "njd", "yuv", "blm", "ygg", "sbj", "cfd", "bwr", "puw", "bkg", "oqi", "wwu", "vcu", "iuf", "mtd", "dwc", "iqb", "kll", "qsh", "jru", "dbf", "phv", "mxa", "sfg", "jpl", "cuh", "tjs", "ocg", "cmg", "hgw", "fmh", "dkc", "krd", "kox", "dqb", "goq", "emx", "vhf", "iue", "srg", "rrg", "fbm", "yhr", "vpm", "llx", "yez", "ivk", "bbg", "vcc", "kfc", "acj", "trg", "nxo", "rhn", "yuf", "uaz", "krh", "ohf", "srl", "slg", "fdd", "ctz", "zij", "ymg", "vdv", "nxx", "cbb", "xue", "vua", "ohj", "aqe", "qut", "yfs", "iht", "lkj", "iui", "pgh", "ivm", "svm", "azk", "qus", "wio", "ihr", "iuh", "tnr", "nhn", "xey", "kpp", "wyi", "bpd", "vfe", "jne", "hdj", "zok", "rhc", "rfy", "hrn", "gdc", "uuu", "uqa", "jtr", "dcj", "trf", "ewx", "dnj", "gtj", "wwj", "ojc", "nkz", "wwv", "ukv", "vpw", "ucn", "vsv", "mtf", "ugy", "slh", "kkk", "jmu", "wgm", "ssx", "svh", "mlw", "fpd", "ogv", "npn", "yft", "mdf", "gds", "hht", "jsw", "ffq", "cpw", "jei", "gbb", "ckx", "gky", "fpf", "qof", "fng", "okq", "qla", "opz", "nwn", "dgd", "tbw", "wct", "pcy", "hnq", "dvp", "cuk", "cgt", "qiw", "fmn", "njb", "mhc", "fbt", "bbp", "yrf", "tln", "xwr", "utx", "rvn", "csx", "xry", "mgw", "juw", "yzu", "kls", "ozd", "jiy", "dvt", "vmu", "juh", "fmg", "zgu", "xyp", "dkb", "hft", "rjs", "pnm", "pfw", "dxr", "yyy", "mks", "jpi", "fkg", "xni", "jsu", "cwc", "prz", "prk", "sxx", "zsw", "trr", "ojt", "cjs", "uyj", "kze", "ivg", "vax", "fwc", "uhw", "nbf", "lht", "dlg", "aql", "ugk", "gnk", "vdm", "qbr", "hpg", "cgs", "nvp", "blc", "cnd", "kww", "zsd", "wpm", "rcj", "ypf", "uua", "syy", "nvt", "lzu", "klm", "yht", "rwc", "sxo", "crw", "wfs", "fyv", "crr", "xmr", "rxt", "vpn", "qls", "bnl", "amx", "ziv", "tbd", "rxm", "rrj", "zny", "ovv", "wuw", "wmt", "sjd", "ubz", "dlf", "sdg", "wpc", "tcz", "nwp", "mhl", "duy", "rjc", "txs", "gxu", "cij", "bza", "krt", "krp", "ybd", "uhm", "srv", "qse", "azg", "bmr", "yrp", "scz", "hzw", "djf", "cll", "vpp", "swl", "nuj", "njm", "cfp", "lgp", "tjh", "ogk", "dwm", "cdh", "spk", "nzp", "jmc", "frn", "ypb", "fdl", "qbc", "hzu", "kcn", "cax", "gtl", "mgb", "fgn", "brz", "bfc", "tnl", "tfb", "lhc", "jlo", "clw", "zsb", "qde", "lzw", "fsj", "hrf", "fjr", "uzd", "ixx", "fgp", "dkt", "bze", "kkl", "pnb", "hzt", "giy", "tgc", "kgm", "yoz", "dhn", "zja", "pvp", "izg", "bfs", "khb", "frf", "fdv", "doq", "plp", "wcp", "qsi", "sxw", "pgb", "nmk", "yfm", "shz", "gkl", "iys", "jsk", "sfh", "qbm", "zzd", "rvw", "khp", "frr", "nbw", "ejf", "wbs", "jke", "qio", "joz", "ejw", "whp", "oez", "gdn", "wtf", "pld", "tlw", "gyy", "kwu", "gtk", "plm", "nnz", "idx", "fvl", "xcy", "wpt", "qaa", "mln", "cgy", "xsr", "rdx", "fdt", "qii", "pft", "hln", "cbt", "kgt", "kcb", "yuo", "gcn", "gfc", "bhd", "kgw", "zcu", "xaa", "sck", "uyk", "gdv", "vsj", "vby", "qta", "ksz", "clb", "wfm", "tyx", "tjm", "msz", "ycf", "jcc", "rbk", "mhr", "tqa", "pxi", "ozb", "jwe", "plw", "nvc", "tfp", "lgw", "cpb", "szw", "jfr", "gdw", "wpp", "vrt", "pmv", "gqb", "hgs", "lkk", "khh", "wox", "crn", "uwm", "mwt", "hkd", "pyy", "lhd", "jij", "trz", "agk", "dhh", "yhs", "dvb", "hhp", "ykc", "tbf", "jds", "cbw", "ajf", "gpw", "wrm", "hww", "flq", "puh", "jva", "jya", "qli", "lgt", "gwb", "sxf", "zju", "ziy", "xah", "mvu", "qsc", "lgn", "gct", "ucp", "mmv", "tzj", "fpb", "zup", "gtf", "pku", "nqb", "qhe", "tuj", "qil", "pdl", "xkl", "rwb", "mgc", "iuz", "dkp", "wku", "jfi", "bnn", "rvm", "svg", "pcn", "ijd", "wud", "kdj", "czu", "afj", "cny", "xeo", "qib", "fvs", "fdj", "eqw", "ywu", "ymk", "xyr", "qte", "sqi", "ybn", "jrk", "vum", "zzz", "agv", "mdl", "hpy", "frw", "cdl", "uud", "mhb", "fcg", "ytm", "rvd", "mgd", "jji", "quy", "mxm", "wft", "svw", "fuu", "zuy", "tkt", "fnh", "dxp", "zfu", "ojm", "txr", "rhw", "fxs", "dxs", "wmp", "rwm", "mwm", "mwy", "yln", "wcf", "kns", "jrl", "zbl", "xow", "vtt", "knj", "jvo", "zce", "jki", "aqg", "xew", "bmh", "tff", "btg", "rwt", "hgp", "cvc", "swd", "ugv", "qid", "lrp", "nxh", "wkb", "tkb", "wiy", "ufw", "mky", "fnt", "tlm", "ddk", "jte", "iqs", "ykt", "mbk", "kcc", "hcf", "klt", "dnl", "bhr", "azf", "nxn", "szs", "iuo", "hzh", "fxe", "dvf", "hbf", "brg", "auu", "aqq", "zyu", "cng", "xef", "wzi", "wji", "pnh", "lhr", "kgh", "jcl", "efx", "xmc", "txm", "rpj", "syx", "cbp", "fxm", "bpb", "xxs", "wiu", "nhg", "dfh", "tjc", "rwd", "fhn", "cwm", "vyj", "hpn", "cgc", "lwu", "lvc", "jsf", "gvu", "djv", "ziu", "eqs", "mfy", "ijm", "dqv", "tjd", "sgf", "mhp", "lvt", "rzl", "wyf", "cvt", "yrk", "qsp", "yvl", "ejk", "ygd", "frg", "xul", "xsn", "rvh", "mgy", "pmk", "nlg", "hxi", "fuo", "tjb", "ftq", "gkg", "yhm", "jbi", "ytb", "pij", "gfs", "wkl", "qbb", "tvy", "mlm", "dvh", "khk", "tck", "plf", "dvm", "whs", "vct", "pdh", "jhu", "snj", "qah", "eqt", "clr", "wln", "iww", "wnz", "vji", "nhf", "mdg", "npv", "aqn", "njf", "gtg", "cpn", "wbb", "tkc", "qum", "xih", "wgc", "pfp", "rfn", "bzu", "bld", "agj", "txb", "jps", "iwu", "nwl", "bvr", "wzo", "luu", "ehv", "jwo", "ajy", "yhd", "lxa", "gpd", "wwy", "qld", "flv", "dln", "xms", "wgn", "tkp", "rfk", "snv", "clp", "nrv", "hgt", "axk", "sjh", "qwo", "qbd", "nvm", "fgb", "dpj", "rbv", "yym", "kcd", "wyb", "vry", "mlk", "qih", "hrg", "uul", "sjb", "gbd", "whl", "pdy", "hnz", "fdb", "afv", "kuv", "wuk", "uzl", "lvr", "yrl", "vuz", "jjh", "kgs", "yjp", "jdd", "vdh", "bjs", "yfy", "plb", "xgl", "wza", "snn", "rhl", "bny", "qiu", "gmj", "ktm", "kld", "hzb", "dxt", "kpt", "khd", "vpb", "rmq", "msx", "gmn", "tlr", "svd", "wrp", "pdb", "snk", "oqa", "tcv", "wgh", "xxc", "kbs", "jti", "dxc", "dbg", "vgi", "rxw", "wrb", "swg", "rrk", "pfm", "cfw", "vpf", "rjh", "pdd", "hhd", "gpy", "tnk", "tcj", "spj", "mjr", "cgm", "uxg", "qbi", "bzo", "grr", "bck", "lgc", "prl", "jgo", "hbt", "sxl", "wue", "gzu", "poj", "ojv", "kvd", "jpg", "fiy", "ygp", "wyw", "eoq", "wui", "wcm", "hmv", "dhg", "wtg", "vuv", "jku", "jeg", "aiq", "vbl", "rxr", "zez", "xka", "pnf", "grf", "fht", "vju", "emz", "lws", "krc", "hgb", "grw", "dwl", "xkn", "njn", "dkf", "nzn", "cuv", "azv", "iug", "vuo", "tbj", "tlg", "nkq", "pvr", "ptq", "iux", "vbs", "tmz", "rxp", "kbb", "xht", "cff", "fkp", "vud", "puo", "bcj", "rjj", "ocv", "gft", "euu", "vup", "jbs", "mfw", "hzs", "hhm", "gpg", "uuc", "svf", "hbm", "clm", "nln", "bgl", "gdh", "cpg", "wtt", "szp", "piy", "ifq", "pks", "jej", "fvr", "bmb", "zwr", "ucm", "ljs", "pjs", "yhp", "lzm", "jsb", "gpf", "tjf", "tfg", "prg", "wrz", "lsx", "mkt", "fkb", "unx", "pdm", "wub", "ikj", "xeu",
#endif

NULL
};

/* ******************************************************************** */

static const char *ndpi_en_impossible_bigrams[] = {
  "bk", "bq", "bx", "cb", "cf", "cg", "cj", "cp", /* "cv", */ "cw", "cx", "dx", "fk", "fq", "fv", "fx", /* "ee", removed it can be found in 'meeting' */
  "fz", "gq", "gv", "gx", "hh", "hk", "hv", "hx", "hz", "iy", "jb", /* "jc", jcrew.com */ "jd", "jf", "jg", "jh", "jk",
  "jl", "jm", "jn", "jp", "jq", /* "jr",*/  /* "js", */ "jt", "jv", "jw", "jx", "jy", "jz", /* "kg", */ "kq", "kv", "kx",
  "kz", "lq", "lx", /* "mg" tamgrt.com , */ "mj", /* "mq", mqtt */  "mx", "mz", "pq", "pv", "px", "qb", "qc", "qd", "qe", "qf", "ii",
  "qg", "qh", "qj", "qk", "ql", "qm", "qn", "qo", "qp", "qr", "qs", "qt", "qv", "qw", "qx", "qy", "uu",
  "qz", "sx", "sz", "tq", "tx", "vb", "vc", "vd", "vf", "vg", "vh", "vj", /* "vm" Used in VMs */ "vn", /* "vp", Removed for vpbank.com */ "bw", /* "vk", "zr" Removed for kavkazr */
  "vq", /* "vt", gvt1.com, gvt2.com */ "vw", "vx", "vz", "wq", "wv", "wx", "wz", /* "xb", foxbusiness.com */
  "xg", "xj", "xk", "xv", "xz", "xw", /* "yd", Removed for anywayanyday.com */ /*"yp", Removed for paypal */
  "yj", "yq", "yv", "yz", "yw", "zb", "zc", "zg", "zh", "zj", "zn", "zq", "zs", "zx", /* "wh", e.g. why */ "wk",
  "wb", "zk", "kp", "zk", "xy", "xx",
  NULL };
