/*
 * Copyright (c) 2019-2023, Intel Corporation. All rights reserved.
 * Copyright (c) 2024-2025, Altera Corporation. All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef SOCFPGA_SIP_SVC_H
#define SOCFPGA_SIP_SVC_H


/* SiP status response */
#define INTEL_SIP_SMC_STATUS_OK						0
#define INTEL_SIP_SMC_STATUS_BUSY					0x1
#define INTEL_SIP_SMC_STATUS_REJECTED					0x2
#define INTEL_SIP_SMC_STATUS_NO_RESPONSE				0x3
#define INTEL_SIP_SMC_STATUS_ERROR					0x4
#define INTEL_SIP_SMC_RSU_ERROR						0x7
#define INTEL_SIP_SMC_SEU_ERR_READ_ERROR				0x8

/* SiP mailbox error code */
#define GENERIC_RESPONSE_ERROR						0x3FF

/* SiP V2 command code range */
#define INTEL_SIP_SMC_CMD_MASK						0xFFFF
#define INTEL_SIP_SMC_CMD_V2_RANGE_BEGIN				0x400
#define INTEL_SIP_SMC_CMD_V2_RANGE_END					0x4FF

/* SiP V3 command code range */
#define INTEL_SIP_SMC_CMD_V3_RANGE_BEGIN				0x00C8
#define INTEL_SIP_SMC_CMD_V3_RANGE_END					0x01F4

/* SiP V2 protocol header */
#define INTEL_SIP_SMC_HEADER_JOB_ID_MASK				0xF
#define INTEL_SIP_SMC_HEADER_JOB_ID_OFFSET				0U
#define INTEL_SIP_SMC_HEADER_CID_MASK					0xF
#define INTEL_SIP_SMC_HEADER_CID_OFFSET					4U
#define INTEL_SIP_SMC_HEADER_VERSION_MASK				0xF
#define INTEL_SIP_SMC_HEADER_VERSION_OFFSET				60U

/* SMC SiP service function identifier for version 1 */

/* FPGA Reconfig */
#define INTEL_SIP_SMC_FPGA_CONFIG_START					0xC2000001
#define INTEL_SIP_SMC_FPGA_CONFIG_WRITE					0x42000002
#define INTEL_SIP_SMC_FPGA_CONFIG_COMPLETED_WRITE			0xC2000003
#define INTEL_SIP_SMC_FPGA_CONFIG_ISDONE				0xC2000004
#define INTEL_SIP_SMC_FPGA_CONFIG_GET_MEM				0xC2000005

/* FPGA Bitstream Flag */
#define FLAG_PARTIAL_CONFIG						BIT(0)
#define FLAG_AUTHENTICATION						BIT(1)
#define CONFIG_TEST_FLAG(_flag, _type)					(((flag) & FLAG_##_type) \
									== FLAG_##_type)

/* Secure Register Access */
#define INTEL_SIP_SMC_REG_READ						0xC2000007
#define INTEL_SIP_SMC_REG_WRITE						0xC2000008
#define INTEL_SIP_SMC_REG_UPDATE					0xC2000009

/* Remote System Update */
#define INTEL_SIP_SMC_RSU_STATUS					0xC200000B
#define INTEL_SIP_SMC_RSU_UPDATE					0xC200000C
#define INTEL_SIP_SMC_RSU_NOTIFY					0xC200000E
#define INTEL_SIP_SMC_RSU_RETRY_COUNTER					0xC200000F
#define INTEL_SIP_SMC_RSU_DCMF_VERSION					0xC2000010
#define INTEL_SIP_SMC_RSU_COPY_DCMF_VERSION				0xC2000011
#define INTEL_SIP_SMC_RSU_MAX_RETRY					0xC2000012
#define INTEL_SIP_SMC_RSU_COPY_MAX_RETRY				0xC2000013
#define INTEL_SIP_SMC_RSU_DCMF_STATUS					0xC2000014
#define INTEL_SIP_SMC_RSU_COPY_DCMF_STATUS				0xC2000015
#define INTEL_SIP_SMC_RSU_GET_DEVICE_INFO				0xC2000016

/* Hardware monitor */
#define INTEL_SIP_SMC_HWMON_READTEMP					0xC2000020
#define INTEL_SIP_SMC_HWMON_READVOLT					0xC2000021
#define TEMP_CHANNEL_MAX						(1 << 15)
#define VOLT_CHANNEL_MAX						(1 << 15)

/* ECC */
#define INTEL_SIP_SMC_ECC_DBE						0xC200000D

/* Generic Command */
#define INTEL_SIP_SMC_SERVICE_COMPLETED					0xC200001E
#define INTEL_SIP_SMC_FIRMWARE_VERSION					0xC200001F
#define INTEL_SIP_SMC_HPS_SET_BRIDGES					0xC2000032
#define INTEL_SIP_SMC_GET_ROM_PATCH_SHA384				0xC2000040

#define SERVICE_COMPLETED_MODE_ASYNC					0x00004F4E

/* Mailbox Command */
#define INTEL_SIP_SMC_MBOX_SEND_CMD					0xC200003C
#define INTEL_SIP_SMC_GET_USERCODE					0xC200003D

/* FPGA Crypto Services */
#define INTEL_SIP_SMC_FCS_RANDOM_NUMBER					0xC200005A
#define INTEL_SIP_SMC_FCS_RANDOM_NUMBER_EXT				0x4200008F
#define INTEL_SIP_SMC_FCS_CRYPTION					0x4200005B
#define INTEL_SIP_SMC_FCS_CRYPTION_EXT					0xC2000090
#define INTEL_SIP_SMC_FCS_SERVICE_REQUEST				0x4200005C
#define INTEL_SIP_SMC_FCS_SEND_CERTIFICATE				0x4200005D
#define INTEL_SIP_SMC_FCS_GET_PROVISION_DATA				0x4200005E
#define INTEL_SIP_SMC_FCS_CNTR_SET_PREAUTH				0xC200005F
#define INTEL_SIP_SMC_FCS_PSGSIGMA_TEARDOWN				0xC2000064
#define INTEL_SIP_SMC_FCS_CHIP_ID					0xC2000065
#define INTEL_SIP_SMC_FCS_ATTESTATION_SUBKEY				0xC2000066
#define INTEL_SIP_SMC_FCS_ATTESTATION_MEASUREMENTS			0xC2000067
#define INTEL_SIP_SMC_FCS_GET_ATTESTATION_CERT				0xC2000068
#define INTEL_SIP_SMC_FCS_CREATE_CERT_ON_RELOAD				0xC2000069
#define INTEL_SIP_SMC_FCS_OPEN_CS_SESSION				0xC200006E
#define INTEL_SIP_SMC_FCS_CLOSE_CS_SESSION				0xC200006F
#define INTEL_SIP_SMC_FCS_IMPORT_CS_KEY					0x42000070
#define INTEL_SIP_SMC_FCS_EXPORT_CS_KEY					0xC2000071
#define INTEL_SIP_SMC_FCS_REMOVE_CS_KEY					0xC2000072
#define INTEL_SIP_SMC_FCS_GET_CS_KEY_INFO				0xC2000073
#define INTEL_SIP_SMC_FCS_AES_CRYPT_INIT				0xC2000074
#define INTEL_SIP_SMC_FCS_AES_CRYPT_UPDATE				0x42000075
#define INTEL_SIP_SMC_FCS_AES_CRYPT_FINALIZE				0x42000076
#define INTEL_SIP_SMC_FCS_GET_DIGEST_INIT				0xC2000077
#define INTEL_SIP_SMC_FCS_GET_DIGEST_UPDATE				0xC2000078
#define INTEL_SIP_SMC_FCS_GET_DIGEST_FINALIZE				0xC2000079
#define INTEL_SIP_SMC_FCS_GET_DIGEST_SMMU_UPDATE			0x42000091
#define INTEL_SIP_SMC_FCS_GET_DIGEST_SMMU_FINALIZE			0x42000092
#define INTEL_SIP_SMC_FCS_MAC_VERIFY_INIT				0xC200007A
#define INTEL_SIP_SMC_FCS_MAC_VERIFY_UPDATE				0xC200007B
#define INTEL_SIP_SMC_FCS_MAC_VERIFY_FINALIZE				0xC200007C
#define INTEL_SIP_SMC_FCS_MAC_VERIFY_SMMU_UPDATE			0x42000093
#define INTEL_SIP_SMC_FCS_MAC_VERIFY_SMMU_FINALIZE			0x42000094
#define INTEL_SIP_SMC_FCS_ECDSA_HASH_SIGN_INIT				0xC200007D
#define INTEL_SIP_SMC_FCS_ECDSA_HASH_SIGN_FINALIZE			0xC200007F
#define INTEL_SIP_SMC_FCS_ECDSA_SHA2_DATA_SIGN_INIT			0xC2000080
#define INTEL_SIP_SMC_FCS_ECDSA_SHA2_DATA_SIGN_UPDATE			0xC2000081
#define INTEL_SIP_SMC_FCS_ECDSA_SHA2_DATA_SIGN_FINALIZE			0xC2000082
#define INTEL_SIP_SMC_FCS_ECDSA_SHA2_DATA_SIGN_SMMU_UPDATE		0x42000095
#define INTEL_SIP_SMC_FCS_ECDSA_SHA2_DATA_SIGN_SMMU_FINALIZE		0x42000096
#define INTEL_SIP_SMC_FCS_ECDSA_HASH_SIG_VERIFY_INIT			0xC2000083
#define INTEL_SIP_SMC_FCS_ECDSA_HASH_SIG_VERIFY_FINALIZE		0xC2000085
#define INTEL_SIP_SMC_FCS_ECDSA_SHA2_DATA_SIG_VERIFY_INIT		0xC2000086
#define INTEL_SIP_SMC_FCS_ECDSA_SHA2_DATA_SIG_VERIFY_UPDATE		0xC2000087
#define INTEL_SIP_SMC_FCS_ECDSA_SHA2_DATA_SIG_VERIFY_FINALIZE		0xC2000088
#define INTEL_SIP_SMC_FCS_ECDSA_SHA2_DATA_SIG_VERIFY_SMMU_UPDATE	0x42000097
#define INTEL_SIP_SMC_FCS_ECDSA_SHA2_DATA_SIG_VERIFY_SMMU_FINALIZE	0x42000098
#define INTEL_SIP_SMC_FCS_ECDSA_GET_PUBKEY_INIT				0xC2000089
#define INTEL_SIP_SMC_FCS_ECDSA_GET_PUBKEY_FINALIZE			0xC200008B
#define INTEL_SIP_SMC_FCS_ECDH_REQUEST_INIT				0xC200008C
#define INTEL_SIP_SMC_FCS_ECDH_REQUEST_FINALIZE				0xC200008E
#define INTEL_SIP_SMC_FCS_SDM_REMAPPER_CONFIG				0xC2000201

/* SEU ERR */
#define INTEL_SIP_SMC_SEU_ERR_STATUS					0xC2000099
#define INTEL_SIP_SMC_SAFE_INJECT_SEU_ERR				0xC200009A

/* ATF build version */
#define INTEL_SIP_SMC_ATF_BUILD_VER					0xC200009B

#define INTEL_SIP_SMC_FCS_SHA_MODE_MASK					0xF
#define INTEL_SIP_SMC_FCS_DIGEST_SIZE_MASK				0xF
#define INTEL_SIP_SMC_FCS_DIGEST_SIZE_OFFSET				4U
#define INTEL_SIP_SMC_FCS_ECC_ALGO_MASK					0xF

/* ECC DBE */
#define WARM_RESET_WFI_FLAG						BIT(31)
#define SYSMGR_ECC_DBE_COLD_RST_MASK					(SYSMGR_ECC_OCRAM_MASK |\
									SYSMGR_ECC_DDR0_MASK |\
									SYSMGR_ECC_DDR1_MASK)

/* Non-mailbox SMC Call */
#define INTEL_SIP_SMC_SVC_VERSION					0xC2000200

/**
 * SMC SiP service function identifier for version 2
 * Command code from 0x400 ~ 0x4FF
 */

/* V2: Non-mailbox function identifier */
#define INTEL_SIP_SMC_V2_GET_SVC_VERSION				0xC2000400
#define INTEL_SIP_SMC_V2_REG_READ					0xC2000401
#define INTEL_SIP_SMC_V2_REG_WRITE					0xC2000402
#define INTEL_SIP_SMC_V2_REG_UPDATE					0xC2000403
#define INTEL_SIP_SMC_V2_HPS_SET_BRIDGES				0xC2000404
#define INTEL_SIP_SMC_V2_RSU_UPDATE_ADDR				0xC2000405

/* V2: Mailbox function identifier */
#define INTEL_SIP_SMC_V2_MAILBOX_SEND_COMMAND				0xC2000420
#define INTEL_SIP_SMC_V2_MAILBOX_POLL_RESPONSE				0xC2000421

/* SMC function IDs for SiP Service queries */
#define SIP_SVC_CALL_COUNT						0x8200ff00
#define SIP_SVC_UID							0x8200ff01
#define SIP_SVC_VERSION							0x8200ff03

/* SiP Service Calls version numbers */
/*
 * Increase if there is any backward compatibility impact
 */
#define SIP_SVC_VERSION_MAJOR						3
/*
 * Increase if there is new SMC function ID being added
 */
#define SIP_SVC_VERSION_MINOR						1


/* Structure Definitions */
struct fpga_config_info {
	uint32_t addr;
	int size;
	int size_written;
	uint32_t write_requested;
	int subblocks_sent;
	int block_number;
};

typedef enum {
	NO_REQUEST = 0,
	RECONFIGURATION,
	BITSTREAM_AUTH
} config_type;

/* Function Definitions */
bool is_size_4_bytes_aligned(uint32_t size);
bool is_address_in_ddr_range(uint64_t addr, uint64_t size);

/* ECC DBE */
bool cold_reset_for_ecc_dbe(void);
uint32_t intel_ecc_dbe_notification(uint64_t dbe_value);

/* Secure register access */
uint32_t intel_secure_reg_read(uint64_t reg_addr, uint32_t *retval);
uint32_t intel_secure_reg_write(uint64_t reg_addr, uint32_t val,
				uint32_t *retval);
uint32_t intel_secure_reg_update(uint64_t reg_addr, uint32_t mask,
				 uint32_t val, uint32_t *retval);

/* Set RSU update address*/
uint32_t intel_rsu_update(uint64_t update_address);

/* Miscellaneous HPS services */
uint32_t intel_hps_set_bridges(uint64_t enable, uint64_t mask);

/* SiP Service handler for version 2 */
uintptr_t sip_smc_handler_v2(uint32_t smc_fid,
			 u_register_t x1,
			 u_register_t x2,
			 u_register_t x3,
			 u_register_t x4,
			 void *cookie,
			 void *handle,
			 u_register_t flags);


#if SIP_SVC_V3
#define SMC_RET_ARGS_ONE							(1)
#define SMC_RET_ARGS_TWO							(2)
#define SMC_RET_ARGS_THREE							(3)
#define SMC_RET_ARGS_FOUR							(4)
#define SMC_RET_ARGS_FIVE							(5)
#define SMC_RET_ARGS_SIX							(6)

/*
 * SiP SVC Version3 SMC Functions IDs
 */

/* Generic response POLL commands */
#define ALTERA_SIP_SMC_ASYNC_RESP_POLL						(0x420000C8)
#define ALTERA_SIP_SMC_ASYNC_RESP_POLL_ON_INTR					(0x420000C9)

/* QSPI related commands */
#define ALTERA_SIP_SMC_ASYNC_QSPI_OPEN						(0x420000CC)
#define ALTERA_SIP_SMC_ASYNC_QSPI_CLOSE						(0x420000CD)
#define ALTERA_SIP_SMC_ASYNC_QSPI_SET_CS					(0x420000CE)
#define ALTERA_SIP_SMC_ASYNC_QSPI_ERASE						(0x420000CF)
#define ALTERA_SIP_SMC_ASYNC_QSPI_WRITE						(0x420000D0)
#define ALTERA_SIP_SMC_ASYNC_QSPI_READ						(0x420000D1)
#define ALTERA_SIP_SMC_ASYNC_GET_DEVICE_IDENTITY				(0x420000D2)
#define ALTERA_SIP_SMC_ASYNC_GET_IDCODE						(0x420000D3)
#define ALTERA_SIP_SMC_ASYNC_QSPI_GET_DEV_INFO					(0x420000D4)

#define ALTERA_SIP_SMC_ASYNC_HWMON_READTEMP					(0x420000E8)
#define ALTERA_SIP_SMC_ASYNC_HWMON_READVOLT					(0x420000E9)

/* FCS crypto service VAB/SDOS commands */
#define ALTERA_SIP_SMC_ASYNC_FCS_RANDOM_NUMBER					(0x4200012C)
#define ALTERA_SIP_SMC_ASYNC_FCS_RANDOM_NUMBER_EXT				(0x4200012D)
#define ALTERA_SIP_SMC_ASYNC_FCS_CRYPTION					(0x4200012E)
#define ALTERA_SIP_SMC_ASYNC_FCS_CRYPTION_EXT					(0x4200012F)
#define ALTERA_SIP_SMC_ASYNC_FCS_SERVICE_REQUEST				(0x42000130)
#define ALTERA_SIP_SMC_ASYNC_FCS_SEND_CERTIFICATE				(0x42000131)
#define ALTERA_SIP_SMC_ASYNC_FCS_GET_PROVISION_DATA				(0x42000132)
#define ALTERA_SIP_SMC_ASYNC_FCS_CNTR_SET_PREAUTH				(0x42000133)
#define ALTERA_SIP_SMC_ASYNC_FCS_PSGSIGMA_TEARDOWN				(0x42000134)
#define ALTERA_SIP_SMC_ASYNC_FCS_CHIP_ID					(0x42000135)
#define ALTERA_SIP_SMC_ASYNC_FCS_ATTESTATION_SUBKEY				(0x42000136)
#define ALTERA_SIP_SMC_ASYNC_FCS_ATTESTATION_MEASUREMENTS			(0x42000137)
#define ALTERA_SIP_SMC_ASYNC_FCS_GET_ATTESTATION_CERT				(0x42000138)
#define ALTERA_SIP_SMC_ASYNC_FCS_CREATE_CERT_ON_RELOAD				(0x42000139)

/* FCS crypto service session management commands */
#define ALTERA_SIP_SMC_ASYNC_FCS_OPEN_CS_SESSION				(0x4200013A)
#define ALTERA_SIP_SMC_ASYNC_FCS_CLOSE_CS_SESSION				(0x4200013B)

/* FCS crypto service key management commands */
#define ALTERA_SIP_SMC_ASYNC_FCS_IMPORT_CS_KEY					(0x4200013C)
#define ALTERA_SIP_SMC_ASYNC_FCS_EXPORT_CS_KEY					(0x4200013D)
#define ALTERA_SIP_SMC_ASYNC_FCS_REMOVE_CS_KEY					(0x4200013E)
#define ALTERA_SIP_SMC_ASYNC_FCS_GET_CS_KEY_INFO				(0x4200013F)
#define ALTERA_SIP_SMC_ASYNC_FCS_CREATE_CS_KEY					(0x42000167)

/* FCS crypto service primitive commands */
#define ALTERA_SIP_SMC_ASYNC_FCS_AES_CRYPT_INIT					(0x42000140)
#define ALTERA_SIP_SMC_ASYNC_FCS_AES_CRYPT_UPDATE				(0x42000141)
#define ALTERA_SIP_SMC_ASYNC_FCS_AES_CRYPT_FINALIZE				(0x42000142)
#define ALTERA_SIP_SMC_ASYNC_FCS_GET_DIGEST_INIT				(0x42000143)
#define ALTERA_SIP_SMC_ASYNC_FCS_GET_DIGEST_UPDATE				(0x42000144)
#define ALTERA_SIP_SMC_ASYNC_FCS_GET_DIGEST_FINALIZE				(0x42000145)
#define ALTERA_SIP_SMC_ASYNC_FCS_GET_DIGEST_SMMU_UPDATE				(0x42000146)
#define ALTERA_SIP_SMC_ASYNC_FCS_GET_DIGEST_SMMU_FINALIZE			(0x42000147)
#define ALTERA_SIP_SMC_ASYNC_FCS_MAC_VERIFY_INIT				(0x42000148)
#define ALTERA_SIP_SMC_ASYNC_FCS_MAC_VERIFY_UPDATE				(0x42000149)
#define ALTERA_SIP_SMC_ASYNC_FCS_MAC_VERIFY_FINALIZE				(0x4200014A)
#define ALTERA_SIP_SMC_ASYNC_FCS_MAC_VERIFY_SMMU_UPDATE				(0x4200014B)
#define ALTERA_SIP_SMC_ASYNC_FCS_MAC_VERIFY_SMMU_FINALIZE			(0x4200014C)
#define ALTERA_SIP_SMC_ASYNC_FCS_ECDSA_HASH_SIGN_INIT				(0x4200014D)
#define ALTERA_SIP_SMC_ASYNC_FCS_ECDSA_HASH_SIGN_FINALIZE			(0x4200014E)
#define ALTERA_SIP_SMC_ASYNC_FCS_ECDSA_SHA2_DATA_SIGN_INIT			(0x4200014F)
#define ALTERA_SIP_SMC_ASYNC_FCS_ECDSA_SHA2_DATA_SIGN_UPDATE			(0x42000150)
#define ALTERA_SIP_SMC_ASYNC_FCS_ECDSA_SHA2_DATA_SIGN_FINALIZE			(0x42000151)
#define ALTERA_SIP_SMC_ASYNC_FCS_ECDSA_SHA2_DATA_SIGN_SMMU_UPDATE		(0x42000152)
#define ALTERA_SIP_SMC_ASYNC_FCS_ECDSA_SHA2_DATA_SIGN_SMMU_FINALIZE		(0x42000153)
#define ALTERA_SIP_SMC_ASYNC_FCS_ECDSA_HASH_SIG_VERIFY_INIT			(0x42000154)
#define ALTERA_SIP_SMC_ASYNC_FCS_ECDSA_HASH_SIG_VERIFY_FINALIZE			(0x42000155)
#define ALTERA_SIP_SMC_ASYNC_FCS_ECDSA_SHA2_DATA_SIG_VERIFY_INIT		(0x42000156)
#define ALTERA_SIP_SMC_ASYNC_FCS_ECDSA_SHA2_DATA_SIG_VERIFY_UPDATE		(0x42000157)
#define ALTERA_SIP_SMC_ASYNC_FCS_ECDSA_SHA2_DATA_SIG_VERIFY_FINALIZE		(0x42000158)
#define ALTERA_SIP_SMC_ASYNC_FCS_ECDSA_SHA2_DATA_SIG_VERIFY_SMMU_UPDATE		(0x42000159)
#define ALTERA_SIP_SMC_ASYNC_FCS_ECDSA_SHA2_DATA_SIG_VERIFY_SMMU_FINALIZE	(0x4200015A)
#define ALTERA_SIP_SMC_ASYNC_FCS_ECDSA_GET_PUBKEY_INIT				(0x42000160)
#define ALTERA_SIP_SMC_ASYNC_FCS_ECDSA_GET_PUBKEY_FINALIZE			(0x42000161)
#define ALTERA_SIP_SMC_ASYNC_FCS_ECDH_REQUEST_INIT				(0x42000162)
#define ALTERA_SIP_SMC_ASYNC_FCS_ECDH_REQUEST_FINALIZE				(0x42000163)
#define ALTERA_SIP_SMC_ASYNC_FCS_SDM_REMAPPER_CONFIG				(0x42000164)
#define ALTERA_SIP_SMC_ASYNC_MCTP_MSG						(0x42000165)
#define ALTERA_SIP_SMC_ASYNC_FCS_HKDF_REQUEST					(0x42000166)
#define ALTERA_SIP_SMC_ASYNC_FCS_CREATE_CS_KEY					(0x42000167)

#define GET_CLIENT_ID(x)							(((x) & 0xF0) >> 4)
#define GET_JOB_ID(x)								((x) & 0x0F)
#endif	/* SIP_SVC_V3 */

#endif /* SOCFPGA_SIP_SVC_H */
