#ifndef RAZERLIB_h
#define RAZERLIB_h 1

/*
    razertool - Tool for controlling Razer Copperhead(TM) mice
    Copyright (C) 2006  Christopher Lais

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <usb.h>

#define RAZERLIB_API		1

#define RAZERLIB_DEFAULT_TIMEOUT	1000

#define RAZERLIB_ERROR_BEGIN	(USB_ERROR_BEGIN+500000)
#define RAZERLIB_ERR_INTERNAL	(RAZERLIB_ERROR_BEGIN+0)
#define RAZERLIB_ERR_STATUS	(RAZERLIB_ERROR_BEGIN+1)
#define RAZERLIB_ERR_INTEGRITY	(RAZERLIB_ERROR_BEGIN+2)
#define RAZERLIB_ERR_PARSE	(RAZERLIB_ERROR_BEGIN+3)

#define USB_VENDOR_RAZER		0x1532
#define USB_PRODUCT_RAZER_COPPERHEAD	0x0101
#define USB_PRODUCT_RAZER_HABU		0x0009

#define RAZER_MAX_PROFILES	5
#define RAZER_MAX_BUTTONS	7

#define RAZER_MACRO_MAX_LEN	(0x28/2)
#define RAZER_CH_MACRO_MAX_LEN	(0x28/2)

enum {
/*0*/	RAZER_ACTION_NONE,
	RAZER_ACTION_CLICK,
	RAZER_ACTION_BUTTON1 = RAZER_ACTION_CLICK,
	RAZER_ACTION_MENU,
	RAZER_ACTION_BUTTON2 = RAZER_ACTION_MENU,
	RAZER_ACTION_UNIVERSAL_SCROLL,
	RAZER_ACTION_BUTTON3 = RAZER_ACTION_UNIVERSAL_SCROLL,
	RAZER_ACTION_DOUBLE_CLICK,
/*5*/	RAZER_ACTION_KEY,
	RAZER_ACTION_MACRO,
/*7*/	RAZER_ACTION_OFF,

	RAZER_ACTION_UNKNOWN_08,

/*9*/	RAZER_ACTION_ON_THE_FLY,

/*a*/	RAZER_ACTION_BUTTON4,
	RAZER_ACTION_BUTTON5,
	RAZER_ACTION_DPI_UP,
/*d*/	RAZER_ACTION_DPI_DOWN,

	RAZER_ACTION_DPI_400,
	RAZER_ACTION_DPI_800,
	RAZER_ACTION_DPI_1600,
/*11*/	RAZER_ACTION_DPI_2000,

/*12*/	RAZER_ACTION_PROFILE1,
	RAZER_ACTION_PROFILE2,
	RAZER_ACTION_PROFILE3,
	RAZER_ACTION_PROFILE4,
/*16*/	RAZER_ACTION_PROFILE5,

	RAZER_ACTION_COUNT, /* number of actions */
};

#define RAZER_MACRO_KEY_DOWN	0x00
#define RAZER_MACRO_KEY_UP	0x01
#define RAZER_MACRO_KEY_DOWNUP_MASK	0x01
#define RAZER_MACRO_KEY_DELAY_AFTER	0x80

enum {
	RAZER_DPI_INVALID,
	RAZER_DPI_2000,
	RAZER_DPI_1600,
	RAZER_DPI_800,
	RAZER_DPI_400,
	RAZER_DPI_COUNT,
};

enum {
	RAZER_HZ_INVALID,
	RAZER_HZ_1000,
	RAZER_HZ_500,
	RAZER_HZ_125,
	RAZER_HZ_COUNT,
};

typedef struct razer_macro_char_s razer_macro_char_t;
struct razer_macro_char_s {
	unsigned char key;
	unsigned char how;	/* RAZER_MACRO_* */
};

typedef struct razer_button_s razer_button_t;
struct razer_button_s {
	int number;
	int action;
	int macro_len;
	int macro_max_len;
	razer_macro_char_t macro[RAZER_MACRO_MAX_LEN];
};

#define RAZER_PROFILE_HAS_NUMBER	(1<<0)
#define RAZER_PROFILE_HAS_DPI		(1<<1)
#define RAZER_PROFILE_HAS_HZ		(1<<2)
#define RAZER_PROFILE_HAS_BUTTON	(1<<3)

typedef struct razer_profile_s razer_profile_t;
struct razer_profile_s {
	int flags;	/* RAZER_PROFILE_* */

	int number;	/* profile number; filled in on read, used for write */
	int dpi;	/* RAZER_DPI_* */
	int hz;		/* RAZER_HZ_* */
	razer_button_t button[RAZER_MAX_BUTTONS];
};

/* All these functions return either 0 (SUCCESS) or < 0 (ERROR) */
/* May return -errno, -(RAZER_ERR_*), or -(USB_ERROR_BEGIN+n) (use razer_strerror) */
/* Some of these functions may be NULL if they are not supported by the
   device, so check. */
typedef struct razer_s razer_t;
struct razer_s {
	struct usb_dev_handle *udev;

	const char *name; /* text description of device */
	int profiles;	/* number of profiles */
	int buttons;	/* number of buttons */

	int (*ping)(razer_t *r, int timeout);

	int (*read_profile_number)(razer_t *r, int *number, int timeout);
	int (*read_firmware_version)(razer_t *r, int *version, int timeout);

	int (*read_profile)(razer_t *r, razer_profile_t *profile, int timeout);
	int (*set_profile)(razer_t *r, int profile, int timeout);
	int (*select_profile)(razer_t *r, int profile, int timeout);
	int (*write_profile)(razer_t *r, razer_profile_t *prof, int timeout);

	int (*firmware_mode)(razer_t *r);

	/* New functions here */
};

extern int razerlib_debug;

/* Core functions */
int razer_usb_ping(struct usb_dev_handle *udev, int timeout);

struct usb_device *razer_find(struct usb_device *prev); /* prev == NULL for first */

razer_t *razer_open(struct usb_device *dev);
void razer_close(razer_t *r);

/* convenience macros */
#define razer_ping(r,timeout)	((r)->ping?(r)->ping(r,timeout):-ENOSYS)

#define razer_read_profile_number(r,number,timeout) \
	((r)->read_profile_number?(r)->read_profile_number(r,number,timeout):-ENOSYS)
#define razer_read_firmware_version(r,version,timeout) \
	((r)->read_firmware_version?(r)->read_firmware_version(r,version,timeout):-ENOSYS)

#define razer_read_profile(r,profile,timeout) \
	((r)->read_profile?(r)->read_profile(r,profile,timeout):-ENOSYS)
#define razer_set_profile(r,profile,timeout) \
	((r)->set_profile?(r)->set_profile(r,profile,timeout):-ENOSYS)
#define razer_select_profile(r,profile,timeout) \
	((r)->select_profile?(r)->select_profile(r,profile,timeout):-ENOSYS)
#define razer_write_profile(r,profile,timeout) \
	((r)->write_profile?(r)->write_profile(r,profile,timeout):-ENOSYS)
#define razer_firmware_mode(r) \
	((r)->firmware_mode?(r)->firmware_mode(r):-ENOSYS)

/* Utility functions */
const char *razer_strerror(int number);

int razer_2hex(const char two[2]);

int razer_translate_key(const char *key);
void razer_untranslate_key(int usbkey, char *buf, int buflen);

const char *razer_action_string(int action);
int razer_action_int(const char *str);
int razer_dpi_int(int dpi);
int razer_hz_int(int hz);

#endif
