/* get_local_stream.c
 *
 * Copyright (c) 1999 Scott Manley, Barath Raghavan, Jack Moffitt, and Alexander Havng
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License  
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */


/* this file spawns a process to write to a pipe and spwans l3enc to read */
/* from this pipe */
/* several encoders are listed..... */

/* Timing parameters used to synchronise output in NO_SOUNDCARD mode */
#define MAX_SYNC_HISTORY 256
#define MIN_SYNC_HISTORY 128
#define MAX_SYNC_LAG 32

#include "liveice.h"
#include "serverlib.h"
#include "audio_proc.h"

#ifdef SOUNDCARD_SUPPORT
#ifdef HAVE_SYS_SOUNDCARD_H
#include <sys/soundcard.h>
#else
#include <machine/soundcard.h>
#endif
#endif
#include <sys/ioctl.h>
#define BUF_LENGTH 256
#define BUF_SIZE BUF_LENGTH*2
#define BUFLEN 1024*32

extern int errno;

FILE *ds;

/* suggested legal sample rates */
int legal_rates[8] = { 8000, 11025, 16000, 22050, 24000, 32000, 44100, 48000 };


int get_legal_rate(int rate)
{
        float diff,bdiff;
	int best,i;
	bdiff=fabs((float)(rate-legal_rates[0])/legal_rates[0]);
	best=0;
	for(i=1;i<8;i++){
	       diff=fabs((float)(rate-legal_rates[i])/legal_rates[i]);
	       if(diff<bdiff){
	               bdiff=diff;
		       best=i;
	       }
	}
	return legal_rates[best];
}





/* Open the soundcard with the right params */
void check_soundcard(void)
{
        int format,stereo,speed,caps;
#ifdef SOUNDCARD_SUPPORT
	fprintf(stderr,"Initialising Soundcard\n");

	if((g_conf.audio_fd=open(g_conf.sound_input_file,O_RDWR))==-1)
	        fatal("Failed to open sound device");

        /* see if the card can do full_duplex */
        if(g_conf.full_duplex){
                ioctl(g_conf.audio_fd, SNDCTL_DSP_GETCAPS, &caps);
	        if(caps & DSP_CAP_DUPLEX){
#ifdef HAVE_SYS_SOUNDCARD_H
	                ioctl(g_conf.audio_fd, SNDCTL_DSP_SETDUPLEX, 0);  
#else
			ioctl(g_conf.audio_fd, DSP_CAP_DUPLEX, 0);
#endif
	        }
		else {
		        write_message("Sound Card Driver Can't Do Full Duplex",0);
			g_conf.full_duplex=0;
		}
	}

	ioctl(g_conf.audio_fd, SNDCTL_DSP_GETFMTS, &format);
	if(!(format&AFMT_S16_LE))
		fatal("16bit mode not supported by driver");

	format = AFMT_S16_LE;
	if(ioctl(g_conf.audio_fd,SNDCTL_DSP_SETFMT,&format)==-1)
		fatal("Failed to set data format\n");
	
	stereo = g_conf.stereo;
	if(ioctl(g_conf.audio_fd,SNDCTL_DSP_STEREO,&stereo)==-1)
		fatal("Something went wrong with the stereo setting!\n");

	speed=g_conf.sample_rate;
	if(ioctl(g_conf.audio_fd,SNDCTL_DSP_SPEED,&speed)==-1)
		fatal("Speed Setting failed\n");

	
	fprintf(stderr,"16Bit %dHz ",g_conf.sample_rate);
	if(g_conf.stereo)
	          fprintf(stderr,"Stereo ");
	if(g_conf.full_duplex)
	 	  fprintf(stderr,"Full Duplex ");
	fprintf(stderr,"\n");
	close(g_conf.audio_fd);
	g_conf.audio_fd=0;
#else
	write_message("This executable Doesn't Support Soundcards",0);
#endif


}


void open_soundcard(int flags){
       int format,stereo,speed;
#ifdef SOUNDCARD_SUPPORT
       write_message("Opening Soundcard",1);
        
	if((g_conf.audio_fd=open(g_conf.sound_input_file,flags))==-1)
	        fatal("Failed to open sound device");
	
        if(g_conf.full_duplex)
#ifdef HAVE_SYS_SOUNDCARD_H
	        ioctl(g_conf.audio_fd, SNDCTL_DSP_SETDUPLEX, 0);  
#else
		ioctl(g_conf.audio_fd, DSP_CAP_DUPLEX, 0);
#endif

	format = AFMT_S16_LE;
	if(ioctl(g_conf.audio_fd,SNDCTL_DSP_SETFMT,&format)==-1)
		fatal("Failed to set data format\n");
	
	stereo = g_conf.stereo;
	if(ioctl(g_conf.audio_fd,SNDCTL_DSP_STEREO,&stereo)==-1)
		fatal("Something went wrong with the stereo setting\n");

	speed=g_conf.sample_rate;
	if(ioctl(g_conf.audio_fd,SNDCTL_DSP_SPEED,&speed)==-1)
		fatal("Speed Setting failed\n");
#else
	write_message("This executable Doesn't Support Soundcards",0);
#endif
}

void close_soundcard(void)
{
#ifdef SOUNDCARD_SUPPORT
	write_message("Closing Soundcard",1);
	ioctl(g_conf.audio_fd, SNDCTL_DSP_RESET, 0);
        close(g_conf.audio_fd);
#else
	write_message("This executable Doesn't Support Soundcards",0);
#endif
	
}

/* the enlightenment sound daemon provides and alternative audio api      */
/* esd is a lousy implementation, but I've got support here via the esdmon */
/* binary */

int open_esdmon(){
	int pid;
	int strm[2];
	char rate[32];
	sprintf(rate,"%d",g_conf.sample_rate); 
	pipe(strm);
	pid=fork();
	if(pid){
		/* parent */
		close(strm[1]);
	} else {
		/* duplicate stdout */
		close(strm[0]);
		close(1);
		dup(strm[1]);
		close(strm[0]);
		close(strm[1]);
		if(g_conf.stereo)
			execlp("esdmon","esdmon","-r",rate,NULL);
		else
			execlp("esdmon","esdmon","-m","-r",rate,NULL);
		fatal("bugger - no converter - check you've got esd");
	}
	g_conf.audio_fd=strm[0];
	g_conf.rw_size=2048;
	return pid;
}

/* Just like ESDMON, but run an arbitrary command */
int open_command(){
	int pid;
	int strm[2];
	char *argv[100];
	int n;

	pipe(strm);
	pid=fork();
	if(pid){
		/* parent */
		close(strm[1]);
	} else {
		/* duplicate stdout */
		close(strm[0]);
		close(1);
		dup(strm[1]);
		close(strm[0]);
		close(strm[1]);
		for(n=1,argv[0]=strtok(g_conf.mixer_cmd," \t") ;
		    (argv[n]=strtok(NULL," \t")) ; ++n)
		    ;
		fprintf(stderr,"argv[0]=%s argv[1]=%s\n",argv[0],argv[1]);
		execv(argv[0],argv);
		fatal("bugger - no mixer - check you've got the command right");
	}
	g_conf.audio_fd=strm[0];
	g_conf.rw_size=2048;
	return pid;
}

void close_esdmon(int pid){
	close(g_conf.audio_fd);
	kill(pid,SIGKILL);
}

void read_esdmon(){
	short audiobuf[2048];
	int err;
	time_t now,last;
	FILE *inp;
	draw_simple_display();
	err=0;
	time(&last);
	inp=fdopen(g_conf.audio_fd,"rb");
	while((!g_conf.terminate_now)&&(!err)){
		if(fread(audiobuf,2,g_conf.rw_size,inp) <= 0) {
			fprintf(stderr,"No input!!\n");
			break;
		}
		err=write_streams(audiobuf,g_conf.rw_size);
		draw_volume_meter(audiobuf,g_conf.rw_size,g_conf.stereo);
		time(&now);
		if((g_conf.meta_data>0) && (now > (last + g_conf.meta_data*60))){
			update_all_servers(g_conf.icy_name,g_conf.icy_url);
			time(&last);
		}
		do_simple_command();
	}
}


void read_soundcard()
{
	short audiobuf[4096];
	int err;
	time_t now,last;

	draw_simple_display();
	err=0;
	time(&last);
	while((!g_conf.terminate_now)&&(!err)){
		read(g_conf.audio_fd,audiobuf,2*g_conf.rw_size);
		if(g_conf.full_duplex)
			write(g_conf.audio_fd,audiobuf,2*g_conf.rw_size);
		err=write_streams(audiobuf,g_conf.rw_size);
		draw_volume_meter(audiobuf,g_conf.rw_size,g_conf.stereo);
		time(&now);
		if((g_conf.meta_data>0) && (now > (last + g_conf.meta_data*60))){
			update_all_servers(g_conf.icy_name,g_conf.icy_url);
			time(&last);
		}
		do_simple_command();
	}
	close_soundcard();
}


	

/* Throttle the sample rate by sleeping  */
/* this can get really bursty at high sample rates - but this is only
used when we're not reading or writing to the card so there's nothing
unbuffered waiting for data */
/* This will break if anyone chages the time... so dont do that */
void synchronise_time(long samples){
	static time_t start=0;
	static unsigned long tot_samples=0;
	time_t now;
	long samplespersec=g_conf.sample_rate*(g_conf.stereo ? 2 : 1);
 
	if(start==0){
		time(&start);
		start-= MIN_SYNC_HISTORY;
		tot_samples = MIN_SYNC_HISTORY * samplespersec;
	}
	time(&now);
	tot_samples+=samples;
	if((tot_samples/(now-start))>samplespersec)
		usleep(1000000*samples/samplespersec);

	if((now-start)>MAX_SYNC_HISTORY){
		start+=MIN_SYNC_HISTORY;
		tot_samples-=(MAX_SYNC_HISTORY-MIN_SYNC_HISTORY)*samplespersec;
		if(tot_samples < (MIN_SYNC_HISTORY-MAX_SYNC_LAG)*samplespersec){
			tot_samples = (MIN_SYNC_HISTORY-MAX_SYNC_LAG)*samplespersec;

		}
	}
			
}

/* write data from the mixer to all the active output streams */
/* this is also a good place to check for termination */

int write_streams(short *buf,int num){
	int i;
	int insamp,outsamp;
	int inch,outch;
	time_t now;
	short tmp_buf[num]; /* the output should never end up longer than the input */
	inch=g_conf.stereo+1;
	insamp = num / inch;

	/*fwrite(buf,2,num,ds);*/
	for(i=0;i<MAX_ENCODER_STREAMS;i++){
		
		if(g_conf.e_str[i].enabled){
			outch   = g_conf.e_str[i].stereo+1;
			outsamp = insamp * (float)(g_conf.e_str[i].sample_rate) / (float)(g_conf.sample_rate);
			convert_audio(buf,tmp_buf,inch,outch,insamp,outsamp);
			fwrite(tmp_buf,2,outsamp*outch,g_conf.e_str[i].pipe);
		}
	}
	if(g_conf.end_time){
		time(&now);
		if(now>g_conf.end_time){
			write_message("End of the line - Animal go bye Bye!",0);
			return -1;			
		}
	}
	return 0;
}


int icy_login(int sd,enc_struct *e_stream) {
	char buffer[4096];

	/* write the password first */
	sprintf(buffer, "%s\n",e_stream->password);
	write(sd, buffer, strlen(buffer));
	
	/* now look for OK response */
	read(sd,buffer,3);  /* "OK\n" */
	buffer[2]=0;
	if (buffer[0] != 'O' && buffer[0] != 'o')
	{
		write_message("server didn't send OK response",0);
		return -1;
	}

	/* send the bitrate */
	if(g_conf.encoder!=XING_VBR){
		sprintf(buffer, "icy-br:%d\n", e_stream->bitrate / 1000);
	} else {
		sprintf(buffer, "icy-br:VBR%d\n",e_stream->vbr_quality);
		}
	write(sd, buffer, strlen(buffer));

	/* stream name */
	sprintf(buffer, "icy-name:%s\n",e_stream->name);
	write(sd, buffer, strlen(buffer));

	/* genre */
	sprintf(buffer, "icy-genre:%s\n",e_stream->genre);
	write(sd, buffer, strlen(buffer));

	/* URL */
	sprintf(buffer, "icy-url:%s\n",e_stream->url);
	write(sd, buffer, strlen(buffer));

	/* Public or private? */
	sprintf(buffer, "icy-pub:%d\n",e_stream->public);
	write(sd, buffer, strlen(buffer));

	/* Finally a newline */
	sprintf(buffer, "\n");
	write(sd, buffer, strlen(buffer));
	
	return 0;
}


int x_audio_login(int sd,enc_struct *e_stream){
	char buffer[4096];
        /* x-audiocast style headers */
	/* send password */
	sprintf(buffer,"SOURCE %s ",e_stream->password);
	write(sd, buffer, strlen(buffer));

	/* send the mountpoint string */
	sprintf(buffer,"/%s\n\n", (e_stream->mountpoint[0] == '/' ? &(e_stream->mountpoint[1]) : e_stream->mountpoint));
	write(sd, buffer, strlen(buffer));

	/* send the Bitrate */
	if(g_conf.encoder!=XING_VBR){
		sprintf(buffer, "x-audiocast-bitrate:%d\n", e_stream->bitrate / 1000);
	} else {
		sprintf(buffer,"x-audiocast-bitrate:VBR%d\n",e_stream->vbr_quality);
	}
	
	/* stream name */
	write(sd, buffer, strlen(buffer));
	sprintf(buffer, "x-audiocast-name:%s\n",e_stream->name);

	/* genre */
	write(sd, buffer, strlen(buffer));
	sprintf(buffer, "x-audiocast-genre:%s\n",e_stream->genre);

	/* URL */
	write(sd, buffer, strlen(buffer));
	sprintf(buffer, "x-audiocast-url:%s\n",e_stream->url);

	/* Public or private? */
	write(sd, buffer, strlen(buffer));
	sprintf(buffer, "x-audiocast-public:%d\n",e_stream->public);

	/* Description string */
	write(sd, buffer, strlen(buffer));
	sprintf(buffer, "x-audiocast-description:%s\n",e_stream->description);
	write(sd, buffer, strlen(buffer));	
	
	sprintf(buffer, "x-audiocast-contentid:%s\n",g_conf.random_content_id);
	write(sd, buffer, strlen(buffer));	
       
	
	if(e_stream->remote_dumpfile){
		sprintf(buffer,"x-audiocast-dumpfile:%s\n",e_stream->remote_dumpfile);
		write(sd, buffer, strlen(buffer));
	}
	sprintf(buffer, "\n");
	write(sd, buffer, strlen(buffer));
	return 0;
}


void send_livestream(int sd, enc_struct *e_stream)
{
	unsigned char buffer[MP_BUFF_SIZE];
	int len,err;
	FILE *in;
	FILE *out;
	FILE *record=NULL;
#ifndef HAVE_LIBCURSES
	fprintf(stderr, "opening pipe!...\n");
	fflush(stderr);

        fprintf(stderr, "writing password\n");
	fflush(stderr);
#endif

	if(g_conf.header_format==ICY_LOGIN){ 
		err=icy_login(sd,e_stream);
	} else { 
		err=x_audio_login(sd,e_stream);
	}
	 
	if ((in = fopen(e_stream->mpeg_pipe, "r")) == NULL)
		fatal("unable to open mpeg pipe");

	/*  now turn it into a stdio file... saves changing stuff so far */
	out=fdopen(sd,"wb");
	fread(buffer, 1, MP_BUFF_SIZE, in);
	/* If we've got a filename for recording then use it */
	if(e_stream->recording_file != NULL){
		record=fopen(e_stream->recording_file,"wb");
		}
	/* now run in a loop */
	while(!g_conf.terminate_now) {
	        arm(30);  /* timeout */
		len = fwrite(buffer,1,MP_BUFF_SIZE,out);
		if (len <= 0)
			fatal("sending data to remote server");
		if(e_stream->recording_file != NULL){
			len = fwrite(buffer,1,MP_BUFF_SIZE,record);
			if (len <=0){
				e_stream->recording_file=NULL;
				fclose(record);
			}
		}
		fread(buffer, 1, MP_BUFF_SIZE, in);
	}


}

int spawn_tcp_streamer(enc_struct *e_stream)
{
	sckt *sp;
	int sd;
	pid_t child;
	char mesg[4096];
	
	if(e_stream->enabled){
		signal(SIGPIPE,alarm_timeout);
		/* generate a socket*/
		sprintf(mesg,"opening connection to %s %d",e_stream->server,e_stream->port);
		write_message(mesg,0);
		if ((sp = sopen()) == 0)
			fatal("sopen() - get remote stream - couldn't create a socket....");
		write_message("Created Socket pair",1);
		
		/* Make contact with server */ 
		write_message("Attempting to Contact Server",0);
		if ((sd = sclient(sp, e_stream->server, e_stream->port)) == -1){
			write_message("oops.... some problem connecting, disabling stream",0);
			e_stream->enabled=0;
			sclose(sp);
			signal (SIGPIPE, kill_children);
			return 0;
		} else {
			write_message("connection successful: forking process",0);
			switch(child=fork()){
			case 0:
				arm(60); /* just in case the encoder dies */
				send_livestream(sd,e_stream);
				sclose(sp);
				exit(0);
				break;
			case -1:
				sclose(sp);
				fatal("unable to fork() tcp streamer......");
				break;
			default:
				sclose(sp);
				signal (SIGPIPE, kill_children);
				e_stream->tcp=child; 
			}
			return 1;
		}
	} else {
		return 0;
	}
}


void create_content_id(void){
  time_t now;
  
  g_conf.random_content_id=malloc(strlen(g_conf.icy_name)+32);
  time(&now);
  sprintf(g_conf.random_content_id,"%d%s",now,g_conf.icy_name);
  
}


int start_tcp_streams(void) {
	int i,a,tot;
	tot=0;
	/* new bit */
	create_content_id();
	
	for(i=0;i<MAX_ENCODER_STREAMS;i++){
		a=spawn_tcp_streamer(&(g_conf.e_str[i]));
		if(a){
			a=spawn_encoder_process(&(g_conf.e_str[i]));
			if(a) { 
				tot++;
				g_conf.e_str[i].pipe=fopen(g_conf.e_str[i].sound_pipe,"wb");
				if((g_conf.e_str[i].encoder==XING)||(g_conf.e_str[i].encoder==XING_VBR))
					write_wav_header(g_conf.e_str[i].pipe,g_conf.e_str[i].sample_rate,g_conf.e_str[i].stereo);
			} else {
				kill(g_conf.e_str[i].tcp,SIGALRM);
				g_conf.e_str[i].enabled=0;
			}
		}
	}
	return tot;
}


/* shuts down an exisitng stream */
void kill_stream(enc_struct *e_stream){

	if(e_stream->enabled){
		/* kill the encoder */
		kill(e_stream->enc,15);

		/* tell the TCP daemon it's time to go */
		kill(e_stream->tcp,SIGALRM);
		
		/* switch off the enabled variable */
		e_stream->enabled=0;
		
		/* clsoe the relevant file descriptor */
		fclose(e_stream->pipe);	
	}
}


/* shuts down all existing streams */
void kill_all_streams(void){
	int i;
	for(i=0;i<MAX_ENCODER_STREAMS;i++){
		kill_stream(&(g_conf.e_str[i]));
	}
}


void encoding(void)
{
	
	pid_t enc;
	write_message("hola, encoding...",1);

	fflush(stderr);
	signal(SIGCHLD,SIG_IGN);
	printf("%d\n",g_conf.mixer);

#ifdef SOUNDCARD_SUPPORT
	if(g_conf.soundcard && g_conf.mixer != ESDMON_MODE
	   && g_conf.mixer != COMMAND_MODE)
		check_soundcard();
#endif

	start_tcp_streams();
	fprintf(stderr,"Setting up Interface\n");
	set_interface();
	
	if(g_conf.mixer==MP3MIXER_MODE){
	  mp3mixer();
	} else if (g_conf.mixer==ESDMON_MODE){
		enc=open_esdmon();
		read_esdmon();
		close_esdmon(enc);
	} else if(g_conf.mixer == SHOUT_MODE){
	        shout_streamer();	  
	} else if(g_conf.mixer == COMMAND_MODE) {
		enc=open_command();
		read_esdmon();
		close_esdmon(enc);
	} else {
#ifdef SOUNDCARD_SUPPORT
		open_soundcard(O_RDWR);
		read_soundcard();
		close_soundcard();
#else
		write_message("Hmmmmm.... no soundcard support compiled",0);
#endif
	}
	kill_all_streams();
	fix_interface();
}
