/* display.c
 *
 * Copyright (c) 1999 Scott Manley, Barath Raghavan, Jack Moffitt, and Alexander Havng
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License  
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

/* Some changes added  by steve j. kondik <shade@chemlab.org> */

#include "liveice.h"
#include "mixer.h"

/* display handler - deals with terminal/curses output */

struct termios tsave;
int display_set=0;


/* sets canonical input mode */
void set_interface(void){
	int flags;

	if(g_conf.frontend == FE_NONE)
	  return;

	if(g_conf.frontend == FE_CURSES){
#ifdef HAVE_LIBCURSES
		/* Initialise the curses screen */
		if(isatty(0))
			tcgetattr(0,&tsave);
		g_conf.icescreen=initscr();
		g_conf.messagescreen=newwin(5,COLS-2,LINES-6,1);
		cbreak();
		noecho();
		nodelay(g_conf.icescreen,TRUE);
#else
/* otherwise put the terminal into canonical mode */
		struct termios tbuf;
		if(isatty(0)){
			tcgetattr(0,&tbuf);
			tsave=tbuf;
			tbuf.c_lflag &= ~(ECHO | ICANON | ISIG);
			tbuf.c_cc[VMIN] = tbuf.c_cc[VTIME] = 0;
			tcsetattr(0,TCSANOW,&tbuf);
		}
#endif
		display_set=1;
		if(!isatty(0)){
			flags=fcntl(0,F_GETFL);
			flags=flags | O_NONBLOCK;
			fcntl(0,F_SETFL,flags);
		}
	} else {
		
		flags=fcntl(0,F_GETFL);
		flags=flags | O_NONBLOCK;
		fcntl(0,F_SETFL,flags);
	}

}

/* outputs a message via the correct method */
void write_message(char *msg,int pri){
	if(g_conf.frontend == FE_NONE) {
		printf("%s\n",msg);
	} else if(g_conf.frontend == FE_PIPE){
		fprintf(stdout,"MSG %d %s\n",pri,msg);
	} else if(g_conf.debug>=pri){
#ifdef HAVE_LIBCURSES
		if(display_set){
			wmove(g_conf.messagescreen,0,0);
			wdeleteln(g_conf.messagescreen);
			mvwprintw(g_conf.messagescreen,4,0,msg);
			wrefresh(g_conf.messagescreen);
		} else {
			fprintf(stderr,"%s\n",msg);
		}
#else
		fprintf(stderr,"%s\n",msg);
#endif
	}
}

/* restore the settings */
void restore_interface(void){
	if(display_set){
#ifdef HAVE_LIBCURSES
		delwin(g_conf.messagescreen);
		clear();
		refresh();
		endwin();
		tcsetattr(0,TCSANOW,&tsave);
		printf("\n");
#else
		tcsetattr(0,TCSANOW,&tsave);
		printf("\n");
#endif
		display_set=0;
	} 
}

void fix_interface(void){
	if(display_set){
#ifdef HAVE_LIBCURSES
		delwin(g_conf.messagescreen);
		clear();
		endwin();
		refresh();
		tcsetattr(0,TCSANOW,&tsave);
		printf("\n");
#else
		tcsetattr(0,TCSANOW,&tsave);
		printf("\n"); /* hack */
#endif
		display_set=0;
	} 
}

char get_inp_char(void){
	if(g_conf.frontend == FE_NONE)
		return 0;
#ifdef HAVE_LIBCURSES
	return getch();
#else
	return getchar();
#endif
}

void init_streamdata(FILE *out,enc_struct *e_stream){
	fprintf(out,"FORMAT %d %d\n",e_stream->sample_rate,e_stream->stereo+1); 
	if(e_stream->encoder==XING_VBR){
		fprintf(out,"ENCODING %d %s %d\n",e_stream->encoder,e_stream->encoder_cmd,e_stream->vbr_quality); 
	} else {
		fprintf(out,"ENCODING %d %s %d\n",e_stream->encoder,e_stream->encoder_cmd,e_stream->bitrate); 
	}
	
	fprintf(out,"SERVER %s %d\n",e_stream->server,e_stream->port);
	fprintf(out,"NAME %s\n",e_stream->name);
	fprintf(out,"GENRE %s\n",e_stream->genre);
	fprintf(out,"URL %s\n",e_stream->url);
	fprintf(out,"DESCRIPTION %s\n",e_stream->description);
}

void draw_simple_display(void){ 
	int i;

	if(g_conf.frontend == FE_NONE)
	  return;

	if(g_conf.frontend == FE_CURSES){
#ifdef HAVE_LIBCURSES
		border(0,0,0,0,0,0,0,0);
		mvprintw(0,20,"LiveIce Streaming Mpeg Audio Generator");
		mvprintw(2,5,"Input Mode: Direct Soundcard [%s]",g_conf.sound_input_file);
		mvprintw(3,5,"Input Format: 16 Bit, %d Hz %s",g_conf.sample_rate,(g_conf.stereo ? "Stereo" : "Mono"));
		if(g_conf.encoder!=XING_VBR){
			mvprintw(5,5,"Output Format: %d Bps Mpeg Audio",g_conf.bitrate);
		} else {
			mvprintw(5,5,"Output Format: VBR Mpeg Audio q=%d",g_conf.vbr_quality);
		}
		mvprintw(6,5,"IceCast Server: %s:%d",g_conf.server,g_conf.port);
		mvprintw(7,5,"Icy-Name: %s",g_conf.icy_name);
		mvprintw(8,5,"Icy-Genre: %s",g_conf.icy_genre);
		mvprintw(9,5,"Icy-Url: %s",g_conf.icy_url);
		mvprintw(12,5,"Input-Level:");
		if(g_conf.stereo)
			mvprintw(13,5,"Input-Level:");
		mvprintw(LINES-2,20,"Press '+' to Finish");
		refresh();
#else
		fprintf(stderr,"\rSoundcard Reopened For Encoding\n");
		fprintf(stderr,"\rInput Format: 16Bit %dHz %s\n",g_conf.sample_rate,(g_conf.stereo ? "Stereo" : "Mono") );
		if(g_conf.encoder!=XING_VBR){
			fprintf(stderr,"\rOutput Format: %d Bps Mpeg Audio\n",g_conf.bitrate); 
		} else {
			fprintf(stderr,"\rOutput Format: VBR Mpeg Audio q=%d\n",g_conf.vbr_quality);
		}
		
		fprintf(stderr,"\rIceCast Server: %s:%d\n",g_conf.server,g_conf.port);
		fprintf(stderr,"\rMountpoint: %s\n",g_conf.mountpoint);
		fprintf(stderr,"\rName: %s\n",g_conf.icy_name);
		fprintf(stderr,"\rGenre: %s\n",g_conf.icy_genre);
		fprintf(stderr,"\rUrl: %s\n",g_conf.icy_url);
		fprintf(stderr,"\rDescription: %s\n\n",g_conf.description);
		fprintf(stderr,"\r Press '+' to Finish\n");
#endif
	} else {
		/* send data to the frontend */
		fprintf(stdout,"INIT\n");
		fprintf(stdout,"VERSION %s\n",LIVEICE_VERSION);
		fprintf(stdout,"SIMPLE_MODE\n");
		fprintf(stdout,"FORMAT %d %d\n",g_conf.sample_rate,g_conf.stereo+1);
		fprintf(stdout,"BEGIN_STREAM_LIST\n");
		for(i=0;i<MAX_ENCODER_STREAMS;i++){
			if(g_conf.e_str[i].enabled){
				fprintf(stdout,"STREAM %d\n",i);
				init_streamdata(stdout,&(g_conf.e_str[i]));
				fprintf(stdout,"END_STREAM\n");
			}
		}
		fprintf(stdout,"END_STREAM_LIST\n");
		fprintf(stdout,"END_INIT\n");
		fflush(stdout);
	}
}


void draw_mixer_display(void){
	int i;

	if(g_conf.frontend == FE_NONE)
	  return;

	if(g_conf.frontend == FE_CURSES){
#ifdef HAVE_LIBCURSES
	border(0,0,0,0,0,0,0,0);
	mvprintw(0,20,"LiveIce Streaming Mpeg Audio Generator");
	mvprintw(2,5,"Input Mode: MP3Mixer");
	mvprintw(3,5,"Input Format: 16 Bit, %d Hz %s",g_conf.sample_rate,(g_conf.stereo ? "Stereo" : "Mono"));
	if(g_conf.encoder!=XING_VBR){
		mvprintw(4,5,"Output Format: %d Bps Mpeg Audio",g_conf.bitrate);
	} else {
		mvprintw(4,5,"Output Format: VBR Mpeg Audio q=%d",g_conf.vbr_quality);
	}
	mvhline(6,1,0,COLS-2);
	mvprintw(7,5,"Soundcard");
	mvprintw(8,5,"Input-Level:");
	mvhline(9,1,0,COLS-2);
	/* Channel 1 goes here */
	mvhline(12,1,0,COLS-2);
	/* Channel 2 goes here */
	mvhline(15,1,0,COLS-2);
	/* mixer goes here */
	mvprintw(16,1,"Output-Level:");
	/* maybe a playlist manager or messge window.... */
	refresh();
#else
	fprintf(stderr,"\rSoundcard Reopened For Encoding\n");
	fprintf(stderr,"\rInput Format: 16Bit %dHz %s\n",g_conf.sample_rate,(g_conf.stereo ? "Stereo" : "Mono") );
 	if(g_conf.encoder!=XING_VBR){
		fprintf(stderr,"\rOutput Format: %d Bps Mpeg Audio\n",g_conf.bitrate); 
	} else {
		fprintf(stderr,"\rOutput Format: VBR Mpeg Audio q=%d\n",g_conf.vbr_quality);
	}
	fprintf(stderr,"\rIceCast Server: %s:%d\n",g_conf.server,g_conf.port);
	fprintf(stderr,"\rMountpoint: %s\n",g_conf.mountpoint);
	fprintf(stderr,"\rName: %s\n",g_conf.icy_name);
	fprintf(stderr,"\rGenre: %s\n",g_conf.icy_genre);
	fprintf(stderr,"\rUrl: %s\n",g_conf.icy_url);
	fprintf(stderr,"\rDescription: %s\n\n",g_conf.description);
	fprintf(stderr,"\r Press '+' to Finish\n");
#endif
	} else {
		fprintf(stdout,"INIT\n");
		fprintf(stdout,"VERSION %s\n",LIVEICE_VERSION);
		fprintf(stdout,"MIXER_MODE\n");
		fprintf(stdout,"FORMAT %d %d\n",g_conf.sample_rate,g_conf.stereo+1);
		fprintf(stdout,"BEGIN_STREAM_LIST\n");
		for(i=0;i<MAX_ENCODER_STREAMS;i++){
			if(g_conf.e_str[i].enabled){
				fprintf(stdout,"STREAM %d\n",i);
				init_streamdata(stdout,&(g_conf.e_str[i]));
				fprintf(stdout,"END_STREAM\n");
			}
		}
		fprintf(stdout,"END_STREAM_LIST\n");
		fprintf(stdout,"END_INIT\n");
		fflush(stdout);
	}
}


void init_channel_display(cptr ch){
#ifdef HAVE_LIBCURSES
	int idx;
#endif

 	if(g_conf.frontend == FE_NONE)
	  return;

#ifdef HAVE_LIBCURSES
	idx=strlen(ch->curnt->filename)-70;
	if(idx<0) idx=0;
	werase(ch->c_display);
	mvwprintw(ch->c_display,0,0,"%s : %s",ch->name,&(ch->curnt->filename[idx]));
	
	mvwprintw(ch->c_display,1,0,
		  "Input: %6d  Volume: %3d%%  Speed: %3d%%  Mode: %c%c",
		  0,(int)(ch->volume*100),(int)(ch->speed*100),
		  (ch->on ? '>' : '#'),(ch->stayon ? '*': ' '));
	wnoutrefresh(ch->c_display);
#endif
}

void update_channel_display(cptr ch,int num){
#ifdef HAVE_LIBCURSES
	int i,vol=0;
	for(i=0;i<num;i++)
		if(abs(ch->inbuf[i])>vol)
			vol=abs(ch->inbuf[i]);
	mvwprintw(ch->c_display,1,0,
		  "Input: %6d  Volume: %3d%%  Speed: %3d%%  Mode: %c%c",
		  vol,(int)(ch->volume*100),(int)(ch->speed*100),
		  (ch->on ? '>' : '#'),(ch->stayon ? '*': ' '));
	wnoutrefresh(ch->c_display);
#endif
}



void level_string(int vol,char *lvl){
	int i;
	for(i=0;i<vol*39/33000;i++)
		lvl[i]='#';
	for(;i<40;i++)
		lvl[i]=' ';
	lvl[40]=0;
	
}

#ifdef HAVE_LIBCURSES
void mv_draw_volume(int x,int y,int vol){
	char lvl[64];
	level_string(vol,lvl);
	mvprintw(x,y,"%s %s",lvl,(vol>32000 ? "*clip*" : "      "));
}
#endif

void draw_volume_meter(short *data,int num,int stereo){
	char lvl[64];
	int  i,vol=0,vol2=0;

	if(g_conf.frontend == FE_NONE)
	  return;

	if(g_conf.stereo){
		for(i=0;i<num;i+=2){
			if(abs(data[i])>vol)
				vol=abs(data[i]);
			if(abs(data[i+1])>vol2)
				vol2=abs(data[i+1]);
		}
		if(g_conf.frontend == FE_CURSES){
#ifdef HAVE_LIBCURSES
			level_string(vol,lvl);
			mvprintw(12,20,"%s",lvl);
			if(vol>32000)
				mvprintw(12,60,"*clip*");
			else
				mvprintw(12,60,"      ");
			level_string(vol2,lvl);
			mvprintw(13,20,"%s",lvl);
			if(vol2>32000)
				mvprintw(13,60,"*clip*");
			else
				mvprintw(13,60,"      ");    
			refresh();
#else
			fprintf(stderr,"\rLvl: L: %6d R: %6d",vol,vol2);
			if((vol>32000)||(vol2>32000))
				fprintf(stderr," *clip* ");
			else
				fprintf(stderr,"        ");
#endif
		} else {
			fprintf(stdout,"VOLUME %d %d\n",vol,vol2);
			fflush(stdout);
		}
	} else {
		for(i=0;i<num;i++){
			if(abs(data[i])>vol)
				vol=abs(data[i]);
		}	
		if(g_conf.frontend == FE_CURSES){
#ifdef HAVE_LIBCURSES
			level_string(vol,lvl);
			mvprintw(12,20,"%s",lvl);
			if(vol>32000)
				mvprintw(12,60,"*clip*");
			else
				mvprintw(12,60,"      ");
			level_string(vol2,lvl);
			mvprintw(13,20,"%s",lvl);
			if(vol2>32000)
 				mvprintw(13,60,"*clip*");
 			else
 				mvprintw(13,60,"      ");
			refresh();
#else
			fprintf(stderr,"\rLvl: %6d ",vol);
			if(vol>32000)
				fprintf(stderr," *clip* ");
			else
				fprintf(stderr,"        ");
#endif
		} else {
			fprintf(stdout,"VOLUME %d %d\n",vol,vol);
			fflush(stdout);
		}
	}
}


 











