// Syd: rock-solid application kernel
// lib/src/syd_test.go: Tests for Go bindings of libsyd, the syd API C Library
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: LGPL-3.0

// Package syd provides Go bindings for the libsyd C library.
package syd

import (
	"errors"
	"fmt"
	"io/ioutil"
	"os"
	"path/filepath"
	"reflect"
	"strconv"
	"strings"
	"syscall"
	"testing"
	"time"
)

// Function to check if a slice contains a specific string
func contains(slice []string, value string) bool {
	for _, item := range slice {
		if item == value {
			return true
		}
	}
	return false
}

func Test_01_Api(t *testing.T) {
	api, err := Api()
	if err != nil {
		t.Fatalf("Api failed: %v", err)
	}
	if api != 3 {
		t.Errorf("Api is not 3!")
	}
}

func Test_02_Stat(t *testing.T) {
	// Walk Test
	state := EnabledWalk()
	if err := EnableWalk(); err != nil {
		t.Fatalf("EnableWalk failed: %v", err)
	}
	if enabled := EnabledWalk(); !enabled {
		t.Error("Expected Walk to be enabled")
	}
	if err := DisableWalk(); err != nil {
		t.Fatalf("DisableWalk failed: %v", err)
	}
	if enabled := EnabledWalk(); enabled {
		t.Error("Expected Walk to be disabled")
	}
	if state {
		EnableWalk()
	} else {
		DisableWalk()
	}

	// Stat Test
	state = EnabledStat()
	if err := EnableStat(); err != nil {
		t.Fatalf("EnableStat failed: %v", err)
	}
	if enabled := EnabledStat(); !enabled {
		t.Error("Expected Stat to be enabled")
	}
	if err := DisableStat(); err != nil {
		t.Fatalf("DisableStat failed: %v", err)
	}
	if enabled := EnabledStat(); enabled {
		t.Error("Expected Stat to be disabled")
	}
	if state {
		EnableStat()
	} else {
		DisableStat()
	}

	// Read Test
	state = EnabledRead()
	if err := EnableRead(); err != nil {
		t.Fatalf("EnableRead failed: %v", err)
	}
	if enabled := EnabledRead(); !enabled {
		t.Error("Expected Read to be enabled")
	}
	if err := DisableRead(); err != nil {
		t.Fatalf("DisableRead failed: %v", err)
	}
	if enabled := EnabledRead(); enabled {
		t.Error("Expected Read to be disabled")
	}
	if state {
		EnableRead()
	} else {
		DisableRead()
	}

	// Write Test
	state = EnabledWrite()
	if err := EnableWrite(); err != nil {
		t.Fatalf("EnableWrite failed: %v", err)
	}
	if enabled := EnabledWrite(); !enabled {
		t.Error("Expected Write to be enabled")
	}
	if err := DisableWrite(); err != nil {
		t.Fatalf("DisableWrite failed: %v", err)
	}
	if enabled := EnabledWrite(); enabled {
		t.Error("Expected Write to be disabled")
	}
	if state {
		EnableWrite()
	} else {
		DisableWrite()
	}

	// Exec Test
	state = EnabledExec()
	if err := EnableExec(); err != nil {
		t.Fatalf("EnableExec failed: %v", err)
	}
	if enabled := EnabledExec(); !enabled {
		t.Error("Expected Exec to be enabled")
	}
	if err := DisableExec(); err != nil {
		t.Fatalf("DisableExec failed: %v", err)
	}
	if enabled := EnabledExec(); enabled {
		t.Error("Expected Exec to be disabled")
	}
	if state {
		EnableExec()
	} else {
		DisableExec()
	}

	// Ioctl Test
	state = EnabledIoctl()
	if err := EnableIoctl(); err != nil {
		t.Fatalf("EnableIoctl failed: %v", err)
	}
	if enabled := EnabledIoctl(); !enabled {
		t.Error("Expected Ioctl to be enabled")
	}
	if err := DisableIoctl(); err != nil {
		t.Fatalf("DisableIoctl failed: %v", err)
	}
	if enabled := EnabledIoctl(); enabled {
		t.Error("Expected Ioctl to be disabled")
	}
	if state {
		EnableIoctl()
	} else {
		DisableIoctl()
	}

	// Create Test
	state = EnabledCreate()
	if err := EnableCreate(); err != nil {
		t.Fatalf("EnableCreate failed: %v", err)
	}
	if enabled := EnabledCreate(); !enabled {
		t.Error("Expected Create to be enabled")
	}
	if err := DisableCreate(); err != nil {
		t.Fatalf("DisableCreate failed: %v", err)
	}
	if enabled := EnabledCreate(); enabled {
		t.Error("Expected Create to be disabled")
	}
	if state {
		EnableCreate()
	} else {
		DisableCreate()
	}

	// Delete Test
	state = EnabledDelete()
	if err := EnableDelete(); err != nil {
		t.Fatalf("EnableDelete failed: %v", err)
	}
	if enabled := EnabledDelete(); !enabled {
		t.Error("Expected Delete to be enabled")
	}
	if err := DisableDelete(); err != nil {
		t.Fatalf("DisableDelete failed: %v", err)
	}
	if enabled := EnabledDelete(); enabled {
		t.Error("Expected Delete to be disabled")
	}
	if state {
		EnableDelete()
	} else {
		DisableDelete()
	}

	// Rename Test
	state = EnabledRename()
	if err := EnableRename(); err != nil {
		t.Fatalf("EnableRename failed: %v", err)
	}
	if enabled := EnabledRename(); !enabled {
		t.Error("Expected Rename to be enabled")
	}
	if err := DisableRename(); err != nil {
		t.Fatalf("DisableRename failed: %v", err)
	}
	if enabled := EnabledRename(); enabled {
		t.Error("Expected Rename to be disabled")
	}
	if state {
		EnableRename()
	} else {
		DisableRename()
	}

	// Symlink Test
	state = EnabledSymlink()
	if err := EnableSymlink(); err != nil {
		t.Fatalf("EnableSymlink failed: %v", err)
	}
	if enabled := EnabledSymlink(); !enabled {
		t.Error("Expected Symlink to be enabled")
	}
	if err := DisableSymlink(); err != nil {
		t.Fatalf("DisableSymlink failed: %v", err)
	}
	if enabled := EnabledSymlink(); enabled {
		t.Error("Expected Symlink to be disabled")
	}
	if state {
		EnableSymlink()
	} else {
		DisableSymlink()
	}

	// Truncate Test
	state = EnabledTruncate()
	if err := EnableTruncate(); err != nil {
		t.Fatalf("EnableTruncate failed: %v", err)
	}
	if enabled := EnabledTruncate(); !enabled {
		t.Error("Expected Truncate to be enabled")
	}
	if err := DisableTruncate(); err != nil {
		t.Fatalf("DisableTruncate failed: %v", err)
	}
	if enabled := EnabledTruncate(); enabled {
		t.Error("Expected Truncate to be disabled")
	}
	if state {
		EnableTruncate()
	} else {
		DisableTruncate()
	}

	// Chdir Test
	state = EnabledChdir()
	if err := EnableChdir(); err != nil {
		t.Fatalf("EnableChdir failed: %v", err)
	}
	if enabled := EnabledChdir(); !enabled {
		t.Error("Expected Chdir to be enabled")
	}
	if err := DisableChdir(); err != nil {
		t.Fatalf("DisableChdir failed: %v", err)
	}
	if enabled := EnabledChdir(); enabled {
		t.Error("Expected Chdir to be disabled")
	}
	if state {
		EnableChdir()
	} else {
		DisableChdir()
	}

	// Readdir Test
	state = EnabledReaddir()
	if err := EnableReaddir(); err != nil {
		t.Fatalf("EnableReaddir failed: %v", err)
	}
	if enabled := EnabledReaddir(); !enabled {
		t.Error("Expected Readdir to be enabled")
	}
	if err := DisableReaddir(); err != nil {
		t.Fatalf("DisableReaddir failed: %v", err)
	}
	if enabled := EnabledReaddir(); enabled {
		t.Error("Expected Readdir to be disabled")
	}
	if state {
		EnableReaddir()
	} else {
		DisableReaddir()
	}

	// Mkdir Test
	state = EnabledMkdir()
	if err := EnableMkdir(); err != nil {
		t.Fatalf("EnableMkdir failed: %v", err)
	}
	if enabled := EnabledMkdir(); !enabled {
		t.Error("Expected Mkdir to be enabled")
	}
	if err := DisableMkdir(); err != nil {
		t.Fatalf("DisableMkdir failed: %v", err)
	}
	if enabled := EnabledMkdir(); enabled {
		t.Error("Expected Mkdir to be disabled")
	}
	if state {
		EnableMkdir()
	} else {
		DisableMkdir()
	}

	// Rmdir Test
	state = EnabledRmdir()
	if err := EnableRmdir(); err != nil {
		t.Fatalf("EnableRmdir failed: %v", err)
	}
	if enabled := EnabledRmdir(); !enabled {
		t.Error("Expected Rmdir to be enabled")
	}
	if err := DisableRmdir(); err != nil {
		t.Fatalf("DisableRmdir failed: %v", err)
	}
	if enabled := EnabledRmdir(); enabled {
		t.Error("Expected Rmdir to be disabled")
	}
	if state {
		EnableRmdir()
	} else {
		DisableRmdir()
	}

	// Chown Test
	state = EnabledChown()
	if err := EnableChown(); err != nil {
		t.Fatalf("EnableChown failed: %v", err)
	}
	if enabled := EnabledChown(); !enabled {
		t.Error("Expected Chown to be enabled")
	}
	if err := DisableChown(); err != nil {
		t.Fatalf("DisableChown failed: %v", err)
	}
	if enabled := EnabledChown(); enabled {
		t.Error("Expected Chown to be disabled")
	}
	if state {
		EnableChown()
	} else {
		DisableChown()
	}

	// Chgrp Test
	state = EnabledChgrp()
	if err := EnableChgrp(); err != nil {
		t.Fatalf("EnableChgrp failed: %v", err)
	}
	if enabled := EnabledChgrp(); !enabled {
		t.Error("Expected Chgrp to be enabled")
	}
	if err := DisableChgrp(); err != nil {
		t.Fatalf("DisableChgrp failed: %v", err)
	}
	if enabled := EnabledChgrp(); enabled {
		t.Error("Expected Chgrp to be disabled")
	}
	if state {
		EnableChgrp()
	} else {
		DisableChgrp()
	}

	// Chmod Test
	state = EnabledChmod()
	if err := EnableChmod(); err != nil {
		t.Fatalf("EnableChmod failed: %v", err)
	}
	if enabled := EnabledChmod(); !enabled {
		t.Error("Expected Chmod to be enabled")
	}
	if err := DisableChmod(); err != nil {
		t.Fatalf("DisableChmod failed: %v", err)
	}
	if enabled := EnabledChmod(); enabled {
		t.Error("Expected Chmod to be disabled")
	}
	if state {
		EnableChmod()
	} else {
		DisableChmod()
	}

	// Chattr Test
	state = EnabledChattr()
	if err := EnableChattr(); err != nil {
		t.Fatalf("EnableChattr failed: %v", err)
	}
	if enabled := EnabledChattr(); !enabled {
		t.Error("Expected Chattr to be enabled")
	}
	if err := DisableChattr(); err != nil {
		t.Fatalf("DisableChattr failed: %v", err)
	}
	if enabled := EnabledChattr(); enabled {
		t.Error("Expected Chattr to be disabled")
	}
	if state {
		EnableChattr()
	} else {
		DisableChattr()
	}

	/* Chroot is startup only since 3.32.4
	// Chroot Test
	state = EnabledChroot()
	if err := EnableChroot(); err != nil {
		t.Fatalf("EnableChroot failed: %v", err)
	}
	if enabled := EnabledChroot(); !enabled {
		t.Error("Expected Chroot to be enabled")
	}
	if err := DisableChroot(); err != nil {
		t.Fatalf("DisableChroot failed: %v", err)
	}
	if enabled := EnabledChroot(); enabled {
		t.Error("Expected Chroot to be disabled")
	}
	if state {
		EnableChroot()
	} else {
		DisableChroot()
	}
	*/

	// Utime Test
	state = EnabledUtime()
	if err := EnableUtime(); err != nil {
		t.Fatalf("EnableUtime failed: %v", err)
	}
	if enabled := EnabledUtime(); !enabled {
		t.Error("Expected Utime to be enabled")
	}
	if err := DisableUtime(); err != nil {
		t.Fatalf("DisableUtime failed: %v", err)
	}
	if enabled := EnabledUtime(); enabled {
		t.Error("Expected Utime to be disabled")
	}
	if state {
		EnableUtime()
	} else {
		DisableUtime()
	}

	// Mkbdev Test
	state = EnabledMkbdev()
	if err := EnableMkbdev(); err != nil {
		t.Fatalf("EnableMkbdev failed: %v", err)
	}
	if enabled := EnabledMkbdev(); !enabled {
		t.Error("Expected Mkbdev to be enabled")
	}
	if err := DisableMkbdev(); err != nil {
		t.Fatalf("DisableMkbdev failed: %v", err)
	}
	if enabled := EnabledMkbdev(); enabled {
		t.Error("Expected Mkbdev to be disabled")
	}
	if state {
		EnableMkbdev()
	} else {
		DisableMkbdev()
	}

	// Mkcdev Test
	state = EnabledMkcdev()
	if err := EnableMkcdev(); err != nil {
		t.Fatalf("EnableMkcdev failed: %v", err)
	}
	if enabled := EnabledMkcdev(); !enabled {
		t.Error("Expected Mkcdev to be enabled")
	}
	if err := DisableMkcdev(); err != nil {
		t.Fatalf("DisableMkcdev failed: %v", err)
	}
	if enabled := EnabledMkcdev(); enabled {
		t.Error("Expected Mkcdev to be disabled")
	}
	if state {
		EnableMkcdev()
	} else {
		DisableMkcdev()
	}

	// Mkfifo Test
	state = EnabledMkfifo()
	if err := EnableMkfifo(); err != nil {
		t.Fatalf("EnableMkfifo failed: %v", err)
	}
	if enabled := EnabledMkfifo(); !enabled {
		t.Error("Expected Mkfifo to be enabled")
	}
	if err := DisableMkfifo(); err != nil {
		t.Fatalf("DisableMkfifo failed: %v", err)
	}
	if enabled := EnabledMkfifo(); enabled {
		t.Error("Expected Mkfifo to be disabled")
	}
	if state {
		EnableMkfifo()
	} else {
		DisableMkfifo()
	}

	// Mktemp Test
	state = EnabledMktemp()
	if err := EnableMktemp(); err != nil {
		t.Fatalf("EnableMktemp failed: %v", err)
	}
	if enabled := EnabledMktemp(); !enabled {
		t.Error("Expected Mktemp to be enabled")
	}
	if err := DisableMktemp(); err != nil {
		t.Fatalf("DisableMktemp failed: %v", err)
	}
	if enabled := EnabledMktemp(); enabled {
		t.Error("Expected Mktemp to be disabled")
	}
	if state {
		EnableMktemp()
	} else {
		DisableMktemp()
	}

	// Net Test
	state = EnabledNet()
	if err := EnableNet(); err != nil {
		t.Fatalf("EnableNet failed: %v", err)
	}
	if enabled := EnabledNet(); !enabled {
		t.Error("Expected Net to be enabled")
	}
	if err := DisableNet(); err != nil {
		t.Fatalf("DisableNet failed: %v", err)
	}
	if enabled := EnabledNet(); enabled {
		t.Error("Expected Net to be disabled")
	}
	if state {
		EnableNet()
	} else {
		DisableNet()
	}

	// Force Test
	state = EnabledForce()
	if err := EnableForce(); err != nil {
		t.Fatalf("EnableForce failed: %v", err)
	}
	if enabled := EnabledForce(); !enabled {
		t.Error("Expected Force to be enabled")
	}
	if err := DisableForce(); err != nil {
		t.Fatalf("DisableForce failed: %v", err)
	}
	if enabled := EnabledForce(); enabled {
		t.Error("Expected Force to be disabled")
	}
	if state {
		EnableForce()
	} else {
		DisableForce()
	}

	// Mem Test
	state = EnabledMem()
	if err := EnableMem(); err != nil {
		t.Fatalf("EnableMem failed: %v", err)
	}
	if enabled := EnabledMem(); !enabled {
		t.Error("Expected Mem to be enabled")
	}
	if err := DisableMem(); err != nil {
		t.Fatalf("DisableMem failed: %v", err)
	}
	if enabled := EnabledMem(); enabled {
		t.Error("Expected Mem to be disabled")
	}
	if state {
		EnableMem()
	} else {
		DisableMem()
	}

	// Pid Test
	state = EnabledPid()
	if err := EnablePid(); err != nil {
		t.Fatalf("EnablePid failed: %v", err)
	}
	if enabled := EnabledPid(); !enabled {
		t.Error("Expected Pid to be enabled")
	}
	if err := DisablePid(); err != nil {
		t.Fatalf("DisablePid failed: %v", err)
	}
	if enabled := EnabledPid(); enabled {
		t.Error("Expected Pid to be disabled")
	}
	if state {
		EnablePid()
	} else {
		DisablePid()
	}

	// TPE Test
	state = EnabledTPE()
	if err := EnableTPE(); err != nil {
		t.Fatalf("EnableTPE failed: %v", err)
	}
	if enabled := EnabledTPE(); !enabled {
		t.Error("Expected TPE to be enabled")
	}
	if err := DisableTPE(); err != nil {
		t.Fatalf("DisableTPE failed: %v", err)
	}
	if enabled := EnabledTPE(); enabled {
		t.Error("Expected TPE to be disabled")
	}
	if state {
		EnableTPE()
	} else {
		DisableTPE()
	}

	// Testing memory limits
	sandbox, err := Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	memMaxOrig := sandbox.MemMax
	memVmMaxOrig := sandbox.MemVmMax
	pidMaxOrig := sandbox.PidMax

	// Test setting MemMax
	if err := MemMax("1G"); err != nil {
		t.Fatalf("MemMax(1G) failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.MemMax != 1024*1024*1024 {
		t.Errorf("Expected MemMax to be %d, got %d", 1024*1024*1024, sandbox.MemMax)
	}
	MemMax(strconv.FormatInt(memMaxOrig, 10)) // Resetting to original

	// Similar tests for MemVmMax...
	if err := MemVmMax("1G"); err != nil {
		t.Fatalf("MemVmMax(1G) failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.MemVmMax != 1024*1024*1024 {
		t.Errorf("Expected MemVmMax to be %d, got %d", 1024*1024*1024, sandbox.MemVmMax)
	}
	MemVmMax(strconv.FormatInt(memVmMaxOrig, 10)) // Resetting to original

	// Test setting PidMax
	if err := PidMax(-1); err == nil {
		t.Error("Expected PidMax(-1) to fail")
	}
	if err := PidMax(4096); err != nil {
		t.Fatalf("PidMax(4096) failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.PidMax != 4096 {
		t.Errorf("Expected PidMax to be 4096, got %d", sandbox.PidMax)
	}
	PidMax(pidMaxOrig) // Resetting to original

	// Test appending to IoctlDeny
	if err := IoctlDeny(0xdeadca11); err != nil {
		t.Errorf("IoctlDeny(0xdeadca11) failed: %v", err)
	}
}

func Test_03_Default(t *testing.T) {
	sandbox, err := Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultWalk != "deny" {
		t.Errorf("Expected DefaultWalk to be Deny, got %s", sandbox.DefaultWalk)
	}
	if err := DefaultWalk(ActionAllow); err != nil {
		t.Errorf("DefaultWalk failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultWalk != "allow" {
		t.Errorf("Expected DefaultWalk to be Allow, got %s", sandbox.DefaultWalk)
	}
	if err := DefaultWalk(ActionWarn); err != nil {
		t.Errorf("DefaultWalk failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultWalk != "warn" {
		t.Errorf("Expected DefaultWalk to be Warn, got %s", sandbox.DefaultWalk)
	}
	if err := DefaultWalk(ActionFilter); err != nil {
		t.Errorf("DefaultWalk failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultWalk != "filter" {
		t.Errorf("Expected DefaultWalk to be Filter, got %s", sandbox.DefaultWalk)
	}
	if err := DefaultWalk(ActionPanic); err != nil {
		t.Errorf("DefaultWalk failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultWalk != "panic" {
		t.Errorf("Expected DefaultWalk to be Panic, got %s", sandbox.DefaultWalk)
	}
	if err := DefaultWalk(ActionStop); err != nil {
		t.Errorf("DefaultWalk failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultWalk != "stop" {
		t.Errorf("Expected DefaultWalk to be Stop, got %s", sandbox.DefaultWalk)
	}
	if err := DefaultWalk(ActionAbort); err != nil {
		t.Errorf("DefaultWalk failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultWalk != "abort" {
		t.Errorf("Expected DefaultWalk to be Abort, got %s", sandbox.DefaultWalk)
	}
	if err := DefaultWalk(ActionKill); err != nil {
		t.Errorf("DefaultWalk failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultWalk != "kill" {
		t.Errorf("Expected DefaultWalk to be Kill, got %s", sandbox.DefaultWalk)
	}
	if err := DefaultWalk(ActionExit); err != nil {
		t.Errorf("DefaultWalk failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultWalk != "exit" {
		t.Errorf("Expected DefaultWalk to be Exit, got %s", sandbox.DefaultWalk)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultWalk(ActionDeny); err != nil {
		t.Errorf("DefaultWalk failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultWalk != "deny" {
		t.Errorf("Expected DefaultWalk to be Deny, got %s", sandbox.DefaultWalk)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultStat != "deny" {
		t.Errorf("Expected DefaultStat to be Deny, got %s", sandbox.DefaultStat)
	}
	if err := DefaultStat(ActionAllow); err != nil {
		t.Errorf("DefaultStat failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultStat != "allow" {
		t.Errorf("Expected DefaultStat to be Allow, got %s", sandbox.DefaultStat)
	}
	if err := DefaultStat(ActionWarn); err != nil {
		t.Errorf("DefaultStat failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultStat != "warn" {
		t.Errorf("Expected DefaultStat to be Warn, got %s", sandbox.DefaultStat)
	}
	if err := DefaultStat(ActionFilter); err != nil {
		t.Errorf("DefaultStat failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultStat != "filter" {
		t.Errorf("Expected DefaultStat to be Filter, got %s", sandbox.DefaultStat)
	}
	if err := DefaultStat(ActionPanic); err != nil {
		t.Errorf("DefaultStat failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultStat != "panic" {
		t.Errorf("Expected DefaultStat to be Panic, got %s", sandbox.DefaultStat)
	}
	if err := DefaultStat(ActionStop); err != nil {
		t.Errorf("DefaultStat failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultStat != "stop" {
		t.Errorf("Expected DefaultStat to be Stop, got %s", sandbox.DefaultStat)
	}
	if err := DefaultStat(ActionAbort); err != nil {
		t.Errorf("DefaultStat failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultStat != "abort" {
		t.Errorf("Expected DefaultStat to be Abort, got %s", sandbox.DefaultStat)
	}
	if err := DefaultStat(ActionKill); err != nil {
		t.Errorf("DefaultStat failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultStat != "kill" {
		t.Errorf("Expected DefaultStat to be Kill, got %s", sandbox.DefaultStat)
	}
	if err := DefaultStat(ActionExit); err != nil {
		t.Errorf("DefaultStat failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultStat != "exit" {
		t.Errorf("Expected DefaultStat to be Exit, got %s", sandbox.DefaultStat)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultStat(ActionDeny); err != nil {
		t.Errorf("DefaultStat failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultStat != "deny" {
		t.Errorf("Expected DefaultStat to be Deny, got %s", sandbox.DefaultStat)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRead != "deny" {
		t.Errorf("Expected DefaultRead to be Deny, got %s", sandbox.DefaultRead)
	}
	if err := DefaultRead(ActionAllow); err != nil {
		t.Errorf("DefaultRead failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRead != "allow" {
		t.Errorf("Expected DefaultRead to be Allow, got %s", sandbox.DefaultRead)
	}
	if err := DefaultRead(ActionWarn); err != nil {
		t.Errorf("DefaultRead failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRead != "warn" {
		t.Errorf("Expected DefaultRead to be Warn, got %s", sandbox.DefaultRead)
	}
	if err := DefaultRead(ActionFilter); err != nil {
		t.Errorf("DefaultRead failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRead != "filter" {
		t.Errorf("Expected DefaultRead to be Filter, got %s", sandbox.DefaultRead)
	}
	if err := DefaultRead(ActionPanic); err != nil {
		t.Errorf("DefaultRead failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRead != "panic" {
		t.Errorf("Expected DefaultRead to be Panic, got %s", sandbox.DefaultRead)
	}
	if err := DefaultRead(ActionStop); err != nil {
		t.Errorf("DefaultRead failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRead != "stop" {
		t.Errorf("Expected DefaultRead to be Stop, got %s", sandbox.DefaultRead)
	}
	if err := DefaultRead(ActionAbort); err != nil {
		t.Errorf("DefaultRead failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRead != "abort" {
		t.Errorf("Expected DefaultRead to be Abort, got %s", sandbox.DefaultRead)
	}
	if err := DefaultRead(ActionKill); err != nil {
		t.Errorf("DefaultRead failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRead != "kill" {
		t.Errorf("Expected DefaultRead to be Kill, got %s", sandbox.DefaultRead)
	}
	if err := DefaultRead(ActionExit); err != nil {
		t.Errorf("DefaultRead failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRead != "exit" {
		t.Errorf("Expected DefaultRead to be Exit, got %s", sandbox.DefaultRead)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultRead(ActionDeny); err != nil {
		t.Errorf("DefaultRead failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRead != "deny" {
		t.Errorf("Expected DefaultRead to be Deny, got %s", sandbox.DefaultRead)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultWrite != "deny" {
		t.Errorf("Expected DefaultWrite to be Deny, got %s", sandbox.DefaultWrite)
	}
	if err := DefaultWrite(ActionAllow); err != nil {
		t.Errorf("DefaultWrite failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultWrite != "allow" {
		t.Errorf("Expected DefaultWrite to be Allow, got %s", sandbox.DefaultWrite)
	}
	if err := DefaultWrite(ActionWarn); err != nil {
		t.Errorf("DefaultWrite failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultWrite != "warn" {
		t.Errorf("Expected DefaultWrite to be Warn, got %s", sandbox.DefaultWrite)
	}
	if err := DefaultWrite(ActionFilter); err != nil {
		t.Errorf("DefaultWrite failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultWrite != "filter" {
		t.Errorf("Expected DefaultWrite to be Filter, got %s", sandbox.DefaultWrite)
	}
	if err := DefaultWrite(ActionPanic); err != nil {
		t.Errorf("DefaultWrite failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultWrite != "panic" {
		t.Errorf("Expected DefaultWrite to be Panic, got %s", sandbox.DefaultWrite)
	}
	if err := DefaultWrite(ActionStop); err != nil {
		t.Errorf("DefaultWrite failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultWrite != "stop" {
		t.Errorf("Expected DefaultWrite to be Stop, got %s", sandbox.DefaultWrite)
	}
	if err := DefaultWrite(ActionAbort); err != nil {
		t.Errorf("DefaultWrite failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultWrite != "abort" {
		t.Errorf("Expected DefaultWrite to be Abort, got %s", sandbox.DefaultWrite)
	}
	if err := DefaultWrite(ActionKill); err != nil {
		t.Errorf("DefaultWrite failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultWrite != "kill" {
		t.Errorf("Expected DefaultWrite to be Kill, got %s", sandbox.DefaultWrite)
	}
	if err := DefaultWrite(ActionExit); err != nil {
		t.Errorf("DefaultWrite failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultWrite != "exit" {
		t.Errorf("Expected DefaultWrite to be Exit, got %s", sandbox.DefaultWrite)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultWrite(ActionDeny); err != nil {
		t.Errorf("DefaultWrite failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultWrite != "deny" {
		t.Errorf("Expected DefaultWrite to be Deny, got %s", sandbox.DefaultWrite)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultExec != "deny" {
		t.Errorf("Expected DefaultExec to be Deny, got %s", sandbox.DefaultExec)
	}
	if err := DefaultExec(ActionAllow); err != nil {
		t.Errorf("DefaultExec failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultExec != "allow" {
		t.Errorf("Expected DefaultExec to be Allow, got %s", sandbox.DefaultExec)
	}
	if err := DefaultExec(ActionWarn); err != nil {
		t.Errorf("DefaultExec failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultExec != "warn" {
		t.Errorf("Expected DefaultExec to be Warn, got %s", sandbox.DefaultExec)
	}
	if err := DefaultExec(ActionFilter); err != nil {
		t.Errorf("DefaultExec failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultExec != "filter" {
		t.Errorf("Expected DefaultExec to be Filter, got %s", sandbox.DefaultExec)
	}
	if err := DefaultExec(ActionPanic); err != nil {
		t.Errorf("DefaultExec failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultExec != "panic" {
		t.Errorf("Expected DefaultExec to be Panic, got %s", sandbox.DefaultExec)
	}
	if err := DefaultExec(ActionStop); err != nil {
		t.Errorf("DefaultExec failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultExec != "stop" {
		t.Errorf("Expected DefaultExec to be Stop, got %s", sandbox.DefaultExec)
	}
	if err := DefaultExec(ActionAbort); err != nil {
		t.Errorf("DefaultExec failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultExec != "abort" {
		t.Errorf("Expected DefaultExec to be Abort, got %s", sandbox.DefaultExec)
	}
	if err := DefaultExec(ActionKill); err != nil {
		t.Errorf("DefaultExec failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultExec != "kill" {
		t.Errorf("Expected DefaultExec to be Kill, got %s", sandbox.DefaultExec)
	}
	if err := DefaultExec(ActionExit); err != nil {
		t.Errorf("DefaultExec failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultExec != "exit" {
		t.Errorf("Expected DefaultExec to be Exit, got %s", sandbox.DefaultExec)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultExec(ActionDeny); err != nil {
		t.Errorf("DefaultExec failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultExec != "deny" {
		t.Errorf("Expected DefaultExec to be Deny, got %s", sandbox.DefaultExec)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultIoctl != "deny" {
		t.Errorf("Expected DefaultIoctl to be Deny, got %s", sandbox.DefaultIoctl)
	}
	if err := DefaultIoctl(ActionAllow); err != nil {
		t.Errorf("DefaultIoctl failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultIoctl != "allow" {
		t.Errorf("Expected DefaultIoctl to be Allow, got %s", sandbox.DefaultIoctl)
	}
	if err := DefaultIoctl(ActionWarn); err != nil {
		t.Errorf("DefaultIoctl failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultIoctl != "warn" {
		t.Errorf("Expected DefaultIoctl to be Warn, got %s", sandbox.DefaultIoctl)
	}
	if err := DefaultIoctl(ActionFilter); err != nil {
		t.Errorf("DefaultIoctl failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultIoctl != "filter" {
		t.Errorf("Expected DefaultIoctl to be Filter, got %s", sandbox.DefaultIoctl)
	}
	if err := DefaultIoctl(ActionPanic); err != nil {
		t.Errorf("DefaultIoctl failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultIoctl != "panic" {
		t.Errorf("Expected DefaultIoctl to be Panic, got %s", sandbox.DefaultIoctl)
	}
	if err := DefaultIoctl(ActionStop); err != nil {
		t.Errorf("DefaultIoctl failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultIoctl != "stop" {
		t.Errorf("Expected DefaultIoctl to be Stop, got %s", sandbox.DefaultIoctl)
	}
	if err := DefaultIoctl(ActionAbort); err != nil {
		t.Errorf("DefaultIoctl failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultIoctl != "abort" {
		t.Errorf("Expected DefaultIoctl to be Abort, got %s", sandbox.DefaultIoctl)
	}
	if err := DefaultIoctl(ActionKill); err != nil {
		t.Errorf("DefaultIoctl failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultIoctl != "kill" {
		t.Errorf("Expected DefaultIoctl to be Kill, got %s", sandbox.DefaultIoctl)
	}
	if err := DefaultIoctl(ActionExit); err != nil {
		t.Errorf("DefaultIoctl failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultIoctl != "exit" {
		t.Errorf("Expected DefaultIoctl to be Exit, got %s", sandbox.DefaultIoctl)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultIoctl(ActionDeny); err != nil {
		t.Errorf("DefaultIoctl failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultIoctl != "deny" {
		t.Errorf("Expected DefaultIoctl to be Deny, got %s", sandbox.DefaultIoctl)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultCreate != "deny" {
		t.Errorf("Expected DefaultCreate to be Deny, got %s", sandbox.DefaultCreate)
	}
	if err := DefaultCreate(ActionAllow); err != nil {
		t.Errorf("DefaultCreate failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultCreate != "allow" {
		t.Errorf("Expected DefaultCreate to be Allow, got %s", sandbox.DefaultCreate)
	}
	if err := DefaultCreate(ActionWarn); err != nil {
		t.Errorf("DefaultCreate failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultCreate != "warn" {
		t.Errorf("Expected DefaultCreate to be Warn, got %s", sandbox.DefaultCreate)
	}
	if err := DefaultCreate(ActionFilter); err != nil {
		t.Errorf("DefaultCreate failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultCreate != "filter" {
		t.Errorf("Expected DefaultCreate to be Filter, got %s", sandbox.DefaultCreate)
	}
	if err := DefaultCreate(ActionPanic); err != nil {
		t.Errorf("DefaultCreate failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultCreate != "panic" {
		t.Errorf("Expected DefaultCreate to be Panic, got %s", sandbox.DefaultCreate)
	}
	if err := DefaultCreate(ActionStop); err != nil {
		t.Errorf("DefaultCreate failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultCreate != "stop" {
		t.Errorf("Expected DefaultCreate to be Stop, got %s", sandbox.DefaultCreate)
	}
	if err := DefaultCreate(ActionAbort); err != nil {
		t.Errorf("DefaultCreate failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultCreate != "abort" {
		t.Errorf("Expected DefaultCreate to be Abort, got %s", sandbox.DefaultCreate)
	}
	if err := DefaultCreate(ActionKill); err != nil {
		t.Errorf("DefaultCreate failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultCreate != "kill" {
		t.Errorf("Expected DefaultCreate to be Kill, got %s", sandbox.DefaultCreate)
	}
	if err := DefaultCreate(ActionExit); err != nil {
		t.Errorf("DefaultCreate failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultCreate != "exit" {
		t.Errorf("Expected DefaultCreate to be Exit, got %s", sandbox.DefaultCreate)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultCreate(ActionDeny); err != nil {
		t.Errorf("DefaultCreate failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultCreate != "deny" {
		t.Errorf("Expected DefaultCreate to be Deny, got %s", sandbox.DefaultCreate)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultDelete != "deny" {
		t.Errorf("Expected DefaultDelete to be Deny, got %s", sandbox.DefaultDelete)
	}
	if err := DefaultDelete(ActionAllow); err != nil {
		t.Errorf("DefaultDelete failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultDelete != "allow" {
		t.Errorf("Expected DefaultDelete to be Allow, got %s", sandbox.DefaultDelete)
	}
	if err := DefaultDelete(ActionWarn); err != nil {
		t.Errorf("DefaultDelete failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultDelete != "warn" {
		t.Errorf("Expected DefaultDelete to be Warn, got %s", sandbox.DefaultDelete)
	}
	if err := DefaultDelete(ActionFilter); err != nil {
		t.Errorf("DefaultDelete failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultDelete != "filter" {
		t.Errorf("Expected DefaultDelete to be Filter, got %s", sandbox.DefaultDelete)
	}
	if err := DefaultDelete(ActionPanic); err != nil {
		t.Errorf("DefaultDelete failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultDelete != "panic" {
		t.Errorf("Expected DefaultDelete to be Panic, got %s", sandbox.DefaultDelete)
	}
	if err := DefaultDelete(ActionStop); err != nil {
		t.Errorf("DefaultDelete failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultDelete != "stop" {
		t.Errorf("Expected DefaultDelete to be Stop, got %s", sandbox.DefaultDelete)
	}
	if err := DefaultDelete(ActionAbort); err != nil {
		t.Errorf("DefaultDelete failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultDelete != "abort" {
		t.Errorf("Expected DefaultDelete to be Abort, got %s", sandbox.DefaultDelete)
	}
	if err := DefaultDelete(ActionKill); err != nil {
		t.Errorf("DefaultDelete failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultDelete != "kill" {
		t.Errorf("Expected DefaultDelete to be Kill, got %s", sandbox.DefaultDelete)
	}
	if err := DefaultDelete(ActionExit); err != nil {
		t.Errorf("DefaultDelete failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultDelete != "exit" {
		t.Errorf("Expected DefaultDelete to be Exit, got %s", sandbox.DefaultDelete)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultDelete(ActionDeny); err != nil {
		t.Errorf("DefaultDelete failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultDelete != "deny" {
		t.Errorf("Expected DefaultDelete to be Deny, got %s", sandbox.DefaultDelete)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRename != "deny" {
		t.Errorf("Expected DefaultRename to be Deny, got %s", sandbox.DefaultRename)
	}
	if err := DefaultRename(ActionAllow); err != nil {
		t.Errorf("DefaultRename failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRename != "allow" {
		t.Errorf("Expected DefaultRename to be Allow, got %s", sandbox.DefaultRename)
	}
	if err := DefaultRename(ActionWarn); err != nil {
		t.Errorf("DefaultRename failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRename != "warn" {
		t.Errorf("Expected DefaultRename to be Warn, got %s", sandbox.DefaultRename)
	}
	if err := DefaultRename(ActionFilter); err != nil {
		t.Errorf("DefaultRename failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRename != "filter" {
		t.Errorf("Expected DefaultRename to be Filter, got %s", sandbox.DefaultRename)
	}
	if err := DefaultRename(ActionPanic); err != nil {
		t.Errorf("DefaultRename failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRename != "panic" {
		t.Errorf("Expected DefaultRename to be Panic, got %s", sandbox.DefaultRename)
	}
	if err := DefaultRename(ActionStop); err != nil {
		t.Errorf("DefaultRename failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRename != "stop" {
		t.Errorf("Expected DefaultRename to be Stop, got %s", sandbox.DefaultRename)
	}
	if err := DefaultRename(ActionAbort); err != nil {
		t.Errorf("DefaultRename failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRename != "abort" {
		t.Errorf("Expected DefaultRename to be Abort, got %s", sandbox.DefaultRename)
	}
	if err := DefaultRename(ActionKill); err != nil {
		t.Errorf("DefaultRename failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRename != "kill" {
		t.Errorf("Expected DefaultRename to be Kill, got %s", sandbox.DefaultRename)
	}
	if err := DefaultRename(ActionExit); err != nil {
		t.Errorf("DefaultRename failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRename != "exit" {
		t.Errorf("Expected DefaultRename to be Exit, got %s", sandbox.DefaultRename)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultRename(ActionDeny); err != nil {
		t.Errorf("DefaultRename failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRename != "deny" {
		t.Errorf("Expected DefaultRename to be Deny, got %s", sandbox.DefaultRename)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultSymlink != "deny" {
		t.Errorf("Expected DefaultSymlink to be Deny, got %s", sandbox.DefaultSymlink)
	}
	if err := DefaultSymlink(ActionAllow); err != nil {
		t.Errorf("DefaultSymlink failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultSymlink != "allow" {
		t.Errorf("Expected DefaultSymlink to be Allow, got %s", sandbox.DefaultSymlink)
	}
	if err := DefaultSymlink(ActionWarn); err != nil {
		t.Errorf("DefaultSymlink failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultSymlink != "warn" {
		t.Errorf("Expected DefaultSymlink to be Warn, got %s", sandbox.DefaultSymlink)
	}
	if err := DefaultSymlink(ActionFilter); err != nil {
		t.Errorf("DefaultSymlink failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultSymlink != "filter" {
		t.Errorf("Expected DefaultSymlink to be Filter, got %s", sandbox.DefaultSymlink)
	}
	if err := DefaultSymlink(ActionPanic); err != nil {
		t.Errorf("DefaultSymlink failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultSymlink != "panic" {
		t.Errorf("Expected DefaultSymlink to be Panic, got %s", sandbox.DefaultSymlink)
	}
	if err := DefaultSymlink(ActionStop); err != nil {
		t.Errorf("DefaultSymlink failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultSymlink != "stop" {
		t.Errorf("Expected DefaultSymlink to be Stop, got %s", sandbox.DefaultSymlink)
	}
	if err := DefaultSymlink(ActionAbort); err != nil {
		t.Errorf("DefaultSymlink failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultSymlink != "abort" {
		t.Errorf("Expected DefaultSymlink to be Abort, got %s", sandbox.DefaultSymlink)
	}
	if err := DefaultSymlink(ActionKill); err != nil {
		t.Errorf("DefaultSymlink failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultSymlink != "kill" {
		t.Errorf("Expected DefaultSymlink to be Kill, got %s", sandbox.DefaultSymlink)
	}
	if err := DefaultSymlink(ActionExit); err != nil {
		t.Errorf("DefaultSymlink failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultSymlink != "exit" {
		t.Errorf("Expected DefaultSymlink to be Exit, got %s", sandbox.DefaultSymlink)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultSymlink(ActionDeny); err != nil {
		t.Errorf("DefaultSymlink failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultSymlink != "deny" {
		t.Errorf("Expected DefaultSymlink to be Deny, got %s", sandbox.DefaultSymlink)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultTruncate != "deny" {
		t.Errorf("Expected DefaultTruncate to be Deny, got %s", sandbox.DefaultTruncate)
	}
	if err := DefaultTruncate(ActionAllow); err != nil {
		t.Errorf("DefaultTruncate failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultTruncate != "allow" {
		t.Errorf("Expected DefaultTruncate to be Allow, got %s", sandbox.DefaultTruncate)
	}
	if err := DefaultTruncate(ActionWarn); err != nil {
		t.Errorf("DefaultTruncate failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultTruncate != "warn" {
		t.Errorf("Expected DefaultTruncate to be Warn, got %s", sandbox.DefaultTruncate)
	}
	if err := DefaultTruncate(ActionFilter); err != nil {
		t.Errorf("DefaultTruncate failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultTruncate != "filter" {
		t.Errorf("Expected DefaultTruncate to be Filter, got %s", sandbox.DefaultTruncate)
	}
	if err := DefaultTruncate(ActionPanic); err != nil {
		t.Errorf("DefaultTruncate failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultTruncate != "panic" {
		t.Errorf("Expected DefaultTruncate to be Panic, got %s", sandbox.DefaultTruncate)
	}
	if err := DefaultTruncate(ActionStop); err != nil {
		t.Errorf("DefaultTruncate failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultTruncate != "stop" {
		t.Errorf("Expected DefaultTruncate to be Stop, got %s", sandbox.DefaultTruncate)
	}
	if err := DefaultTruncate(ActionAbort); err != nil {
		t.Errorf("DefaultTruncate failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultTruncate != "abort" {
		t.Errorf("Expected DefaultTruncate to be Abort, got %s", sandbox.DefaultTruncate)
	}
	if err := DefaultTruncate(ActionKill); err != nil {
		t.Errorf("DefaultTruncate failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultTruncate != "kill" {
		t.Errorf("Expected DefaultTruncate to be Kill, got %s", sandbox.DefaultTruncate)
	}
	if err := DefaultTruncate(ActionExit); err != nil {
		t.Errorf("DefaultTruncate failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultTruncate != "exit" {
		t.Errorf("Expected DefaultTruncate to be Exit, got %s", sandbox.DefaultTruncate)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultTruncate(ActionDeny); err != nil {
		t.Errorf("DefaultTruncate failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultTruncate != "deny" {
		t.Errorf("Expected DefaultTruncate to be Deny, got %s", sandbox.DefaultTruncate)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChdir != "deny" {
		t.Errorf("Expected DefaultChdir to be Deny, got %s", sandbox.DefaultChdir)
	}
	if err := DefaultChdir(ActionAllow); err != nil {
		t.Errorf("DefaultChdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChdir != "allow" {
		t.Errorf("Expected DefaultChdir to be Allow, got %s", sandbox.DefaultChdir)
	}
	if err := DefaultChdir(ActionWarn); err != nil {
		t.Errorf("DefaultChdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChdir != "warn" {
		t.Errorf("Expected DefaultChdir to be Warn, got %s", sandbox.DefaultChdir)
	}
	if err := DefaultChdir(ActionFilter); err != nil {
		t.Errorf("DefaultChdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChdir != "filter" {
		t.Errorf("Expected DefaultChdir to be Filter, got %s", sandbox.DefaultChdir)
	}
	if err := DefaultChdir(ActionPanic); err != nil {
		t.Errorf("DefaultChdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChdir != "panic" {
		t.Errorf("Expected DefaultChdir to be Panic, got %s", sandbox.DefaultChdir)
	}
	if err := DefaultChdir(ActionStop); err != nil {
		t.Errorf("DefaultChdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChdir != "stop" {
		t.Errorf("Expected DefaultChdir to be Stop, got %s", sandbox.DefaultChdir)
	}
	if err := DefaultChdir(ActionAbort); err != nil {
		t.Errorf("DefaultChdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChdir != "abort" {
		t.Errorf("Expected DefaultChdir to be Abort, got %s", sandbox.DefaultChdir)
	}
	if err := DefaultChdir(ActionKill); err != nil {
		t.Errorf("DefaultChdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChdir != "kill" {
		t.Errorf("Expected DefaultChdir to be Kill, got %s", sandbox.DefaultChdir)
	}
	if err := DefaultChdir(ActionExit); err != nil {
		t.Errorf("DefaultChdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChdir != "exit" {
		t.Errorf("Expected DefaultChdir to be Exit, got %s", sandbox.DefaultChdir)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultChdir(ActionDeny); err != nil {
		t.Errorf("DefaultChdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChdir != "deny" {
		t.Errorf("Expected DefaultChdir to be Deny, got %s", sandbox.DefaultChdir)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultReaddir != "deny" {
		t.Errorf("Expected DefaultReaddir to be Deny, got %s", sandbox.DefaultReaddir)
	}
	if err := DefaultReaddir(ActionAllow); err != nil {
		t.Errorf("DefaultReaddir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultReaddir != "allow" {
		t.Errorf("Expected DefaultReaddir to be Allow, got %s", sandbox.DefaultReaddir)
	}
	if err := DefaultReaddir(ActionWarn); err != nil {
		t.Errorf("DefaultReaddir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultReaddir != "warn" {
		t.Errorf("Expected DefaultReaddir to be Warn, got %s", sandbox.DefaultReaddir)
	}
	if err := DefaultReaddir(ActionFilter); err != nil {
		t.Errorf("DefaultReaddir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultReaddir != "filter" {
		t.Errorf("Expected DefaultReaddir to be Filter, got %s", sandbox.DefaultReaddir)
	}
	if err := DefaultReaddir(ActionPanic); err != nil {
		t.Errorf("DefaultReaddir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultReaddir != "panic" {
		t.Errorf("Expected DefaultReaddir to be Panic, got %s", sandbox.DefaultReaddir)
	}
	if err := DefaultReaddir(ActionStop); err != nil {
		t.Errorf("DefaultReaddir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultReaddir != "stop" {
		t.Errorf("Expected DefaultReaddir to be Stop, got %s", sandbox.DefaultReaddir)
	}
	if err := DefaultReaddir(ActionAbort); err != nil {
		t.Errorf("DefaultReaddir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultReaddir != "abort" {
		t.Errorf("Expected DefaultReaddir to be Abort, got %s", sandbox.DefaultReaddir)
	}
	if err := DefaultReaddir(ActionKill); err != nil {
		t.Errorf("DefaultReaddir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultReaddir != "kill" {
		t.Errorf("Expected DefaultReaddir to be Kill, got %s", sandbox.DefaultReaddir)
	}
	if err := DefaultReaddir(ActionExit); err != nil {
		t.Errorf("DefaultReaddir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultReaddir != "exit" {
		t.Errorf("Expected DefaultReaddir to be Exit, got %s", sandbox.DefaultReaddir)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultReaddir(ActionDeny); err != nil {
		t.Errorf("DefaultReaddir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultReaddir != "deny" {
		t.Errorf("Expected DefaultReaddir to be Deny, got %s", sandbox.DefaultReaddir)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkdir != "deny" {
		t.Errorf("Expected DefaultMkdir to be Deny, got %s", sandbox.DefaultMkdir)
	}
	if err := DefaultMkdir(ActionAllow); err != nil {
		t.Errorf("DefaultMkdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkdir != "allow" {
		t.Errorf("Expected DefaultMkdir to be Allow, got %s", sandbox.DefaultMkdir)
	}
	if err := DefaultMkdir(ActionWarn); err != nil {
		t.Errorf("DefaultMkdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkdir != "warn" {
		t.Errorf("Expected DefaultMkdir to be Warn, got %s", sandbox.DefaultMkdir)
	}
	if err := DefaultMkdir(ActionFilter); err != nil {
		t.Errorf("DefaultMkdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkdir != "filter" {
		t.Errorf("Expected DefaultMkdir to be Filter, got %s", sandbox.DefaultMkdir)
	}
	if err := DefaultMkdir(ActionPanic); err != nil {
		t.Errorf("DefaultMkdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkdir != "panic" {
		t.Errorf("Expected DefaultMkdir to be Panic, got %s", sandbox.DefaultMkdir)
	}
	if err := DefaultMkdir(ActionStop); err != nil {
		t.Errorf("DefaultMkdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkdir != "stop" {
		t.Errorf("Expected DefaultMkdir to be Stop, got %s", sandbox.DefaultMkdir)
	}
	if err := DefaultMkdir(ActionAbort); err != nil {
		t.Errorf("DefaultMkdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkdir != "abort" {
		t.Errorf("Expected DefaultMkdir to be Abort, got %s", sandbox.DefaultMkdir)
	}
	if err := DefaultMkdir(ActionKill); err != nil {
		t.Errorf("DefaultMkdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkdir != "kill" {
		t.Errorf("Expected DefaultMkdir to be Kill, got %s", sandbox.DefaultMkdir)
	}
	if err := DefaultMkdir(ActionExit); err != nil {
		t.Errorf("DefaultMkdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkdir != "exit" {
		t.Errorf("Expected DefaultMkdir to be Exit, got %s", sandbox.DefaultMkdir)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultMkdir(ActionDeny); err != nil {
		t.Errorf("DefaultMkdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkdir != "deny" {
		t.Errorf("Expected DefaultMkdir to be Deny, got %s", sandbox.DefaultMkdir)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRmdir != "deny" {
		t.Errorf("Expected DefaultRmdir to be Deny, got %s", sandbox.DefaultRmdir)
	}
	if err := DefaultRmdir(ActionAllow); err != nil {
		t.Errorf("DefaultRmdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRmdir != "allow" {
		t.Errorf("Expected DefaultRmdir to be Allow, got %s", sandbox.DefaultRmdir)
	}
	if err := DefaultRmdir(ActionWarn); err != nil {
		t.Errorf("DefaultRmdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRmdir != "warn" {
		t.Errorf("Expected DefaultRmdir to be Warn, got %s", sandbox.DefaultRmdir)
	}
	if err := DefaultRmdir(ActionFilter); err != nil {
		t.Errorf("DefaultRmdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRmdir != "filter" {
		t.Errorf("Expected DefaultRmdir to be Filter, got %s", sandbox.DefaultRmdir)
	}
	if err := DefaultRmdir(ActionPanic); err != nil {
		t.Errorf("DefaultRmdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRmdir != "panic" {
		t.Errorf("Expected DefaultRmdir to be Panic, got %s", sandbox.DefaultRmdir)
	}
	if err := DefaultRmdir(ActionStop); err != nil {
		t.Errorf("DefaultRmdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRmdir != "stop" {
		t.Errorf("Expected DefaultRmdir to be Stop, got %s", sandbox.DefaultRmdir)
	}
	if err := DefaultRmdir(ActionAbort); err != nil {
		t.Errorf("DefaultRmdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRmdir != "abort" {
		t.Errorf("Expected DefaultRmdir to be Abort, got %s", sandbox.DefaultRmdir)
	}
	if err := DefaultRmdir(ActionKill); err != nil {
		t.Errorf("DefaultRmdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRmdir != "kill" {
		t.Errorf("Expected DefaultRmdir to be Kill, got %s", sandbox.DefaultRmdir)
	}
	if err := DefaultRmdir(ActionExit); err != nil {
		t.Errorf("DefaultRmdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRmdir != "exit" {
		t.Errorf("Expected DefaultRmdir to be Exit, got %s", sandbox.DefaultRmdir)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultRmdir(ActionDeny); err != nil {
		t.Errorf("DefaultRmdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRmdir != "deny" {
		t.Errorf("Expected DefaultRmdir to be Deny, got %s", sandbox.DefaultRmdir)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRmdir != "deny" {
		t.Errorf("Expected DefaultRmdir to be Deny, got %s", sandbox.DefaultRmdir)
	}
	if err := DefaultRmdir(ActionAllow); err != nil {
		t.Errorf("DefaultRmdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRmdir != "allow" {
		t.Errorf("Expected DefaultRmdir to be Allow, got %s", sandbox.DefaultRmdir)
	}
	if err := DefaultRmdir(ActionWarn); err != nil {
		t.Errorf("DefaultRmdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRmdir != "warn" {
		t.Errorf("Expected DefaultRmdir to be Warn, got %s", sandbox.DefaultRmdir)
	}
	if err := DefaultRmdir(ActionFilter); err != nil {
		t.Errorf("DefaultRmdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRmdir != "filter" {
		t.Errorf("Expected DefaultRmdir to be Filter, got %s", sandbox.DefaultRmdir)
	}
	if err := DefaultRmdir(ActionPanic); err != nil {
		t.Errorf("DefaultRmdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRmdir != "panic" {
		t.Errorf("Expected DefaultRmdir to be Panic, got %s", sandbox.DefaultRmdir)
	}
	if err := DefaultRmdir(ActionStop); err != nil {
		t.Errorf("DefaultRmdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRmdir != "stop" {
		t.Errorf("Expected DefaultRmdir to be Stop, got %s", sandbox.DefaultRmdir)
	}
	if err := DefaultRmdir(ActionAbort); err != nil {
		t.Errorf("DefaultRmdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRmdir != "abort" {
		t.Errorf("Expected DefaultRmdir to be Abort, got %s", sandbox.DefaultRmdir)
	}
	if err := DefaultRmdir(ActionKill); err != nil {
		t.Errorf("DefaultRmdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRmdir != "kill" {
		t.Errorf("Expected DefaultRmdir to be Kill, got %s", sandbox.DefaultRmdir)
	}
	if err := DefaultRmdir(ActionExit); err != nil {
		t.Errorf("DefaultRmdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRmdir != "exit" {
		t.Errorf("Expected DefaultRmdir to be Exit, got %s", sandbox.DefaultRmdir)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultRmdir(ActionDeny); err != nil {
		t.Errorf("DefaultRmdir failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultRmdir != "deny" {
		t.Errorf("Expected DefaultRmdir to be Deny, got %s", sandbox.DefaultRmdir)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChown != "deny" {
		t.Errorf("Expected DefaultChown to be Deny, got %s", sandbox.DefaultChown)
	}
	if err := DefaultChown(ActionAllow); err != nil {
		t.Errorf("DefaultChown failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChown != "allow" {
		t.Errorf("Expected DefaultChown to be Allow, got %s", sandbox.DefaultChown)
	}
	if err := DefaultChown(ActionWarn); err != nil {
		t.Errorf("DefaultChown failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChown != "warn" {
		t.Errorf("Expected DefaultChown to be Warn, got %s", sandbox.DefaultChown)
	}
	if err := DefaultChown(ActionFilter); err != nil {
		t.Errorf("DefaultChown failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChown != "filter" {
		t.Errorf("Expected DefaultChown to be Filter, got %s", sandbox.DefaultChown)
	}
	if err := DefaultChown(ActionPanic); err != nil {
		t.Errorf("DefaultChown failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChown != "panic" {
		t.Errorf("Expected DefaultChown to be Panic, got %s", sandbox.DefaultChown)
	}
	if err := DefaultChown(ActionStop); err != nil {
		t.Errorf("DefaultChown failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChown != "stop" {
		t.Errorf("Expected DefaultChown to be Stop, got %s", sandbox.DefaultChown)
	}
	if err := DefaultChown(ActionAbort); err != nil {
		t.Errorf("DefaultChown failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChown != "abort" {
		t.Errorf("Expected DefaultChown to be Abort, got %s", sandbox.DefaultChown)
	}
	if err := DefaultChown(ActionKill); err != nil {
		t.Errorf("DefaultChown failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChown != "kill" {
		t.Errorf("Expected DefaultChown to be Kill, got %s", sandbox.DefaultChown)
	}
	if err := DefaultChown(ActionExit); err != nil {
		t.Errorf("DefaultChown failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChown != "exit" {
		t.Errorf("Expected DefaultChown to be Exit, got %s", sandbox.DefaultChown)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultChown(ActionDeny); err != nil {
		t.Errorf("DefaultChown failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChown != "deny" {
		t.Errorf("Expected DefaultChown to be Deny, got %s", sandbox.DefaultChown)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChgrp != "deny" {
		t.Errorf("Expected DefaultChgrp to be Deny, got %s", sandbox.DefaultChgrp)
	}
	if err := DefaultChgrp(ActionAllow); err != nil {
		t.Errorf("DefaultChgrp failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChgrp != "allow" {
		t.Errorf("Expected DefaultChgrp to be Allow, got %s", sandbox.DefaultChgrp)
	}
	if err := DefaultChgrp(ActionWarn); err != nil {
		t.Errorf("DefaultChgrp failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChgrp != "warn" {
		t.Errorf("Expected DefaultChgrp to be Warn, got %s", sandbox.DefaultChgrp)
	}
	if err := DefaultChgrp(ActionFilter); err != nil {
		t.Errorf("DefaultChgrp failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChgrp != "filter" {
		t.Errorf("Expected DefaultChgrp to be Filter, got %s", sandbox.DefaultChgrp)
	}
	if err := DefaultChgrp(ActionPanic); err != nil {
		t.Errorf("DefaultChgrp failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChgrp != "panic" {
		t.Errorf("Expected DefaultChgrp to be Panic, got %s", sandbox.DefaultChgrp)
	}
	if err := DefaultChgrp(ActionStop); err != nil {
		t.Errorf("DefaultChgrp failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChgrp != "stop" {
		t.Errorf("Expected DefaultChgrp to be Stop, got %s", sandbox.DefaultChgrp)
	}
	if err := DefaultChgrp(ActionAbort); err != nil {
		t.Errorf("DefaultChgrp failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChgrp != "abort" {
		t.Errorf("Expected DefaultChgrp to be Abort, got %s", sandbox.DefaultChgrp)
	}
	if err := DefaultChgrp(ActionKill); err != nil {
		t.Errorf("DefaultChgrp failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChgrp != "kill" {
		t.Errorf("Expected DefaultChgrp to be Kill, got %s", sandbox.DefaultChgrp)
	}
	if err := DefaultChgrp(ActionExit); err != nil {
		t.Errorf("DefaultChgrp failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChgrp != "exit" {
		t.Errorf("Expected DefaultChgrp to be Exit, got %s", sandbox.DefaultChgrp)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultChgrp(ActionDeny); err != nil {
		t.Errorf("DefaultChgrp failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChgrp != "deny" {
		t.Errorf("Expected DefaultChgrp to be Deny, got %s", sandbox.DefaultChgrp)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChmod != "deny" {
		t.Errorf("Expected DefaultChmod to be Deny, got %s", sandbox.DefaultChmod)
	}
	if err := DefaultChmod(ActionAllow); err != nil {
		t.Errorf("DefaultChmod failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChmod != "allow" {
		t.Errorf("Expected DefaultChmod to be Allow, got %s", sandbox.DefaultChmod)
	}
	if err := DefaultChmod(ActionWarn); err != nil {
		t.Errorf("DefaultChmod failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChmod != "warn" {
		t.Errorf("Expected DefaultChmod to be Warn, got %s", sandbox.DefaultChmod)
	}
	if err := DefaultChmod(ActionFilter); err != nil {
		t.Errorf("DefaultChmod failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChmod != "filter" {
		t.Errorf("Expected DefaultChmod to be Filter, got %s", sandbox.DefaultChmod)
	}
	if err := DefaultChmod(ActionPanic); err != nil {
		t.Errorf("DefaultChmod failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChmod != "panic" {
		t.Errorf("Expected DefaultChmod to be Panic, got %s", sandbox.DefaultChmod)
	}
	if err := DefaultChmod(ActionStop); err != nil {
		t.Errorf("DefaultChmod failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChmod != "stop" {
		t.Errorf("Expected DefaultChmod to be Stop, got %s", sandbox.DefaultChmod)
	}
	if err := DefaultChmod(ActionAbort); err != nil {
		t.Errorf("DefaultChmod failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChmod != "abort" {
		t.Errorf("Expected DefaultChmod to be Abort, got %s", sandbox.DefaultChmod)
	}
	if err := DefaultChmod(ActionKill); err != nil {
		t.Errorf("DefaultChmod failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChmod != "kill" {
		t.Errorf("Expected DefaultChmod to be Kill, got %s", sandbox.DefaultChmod)
	}
	if err := DefaultChmod(ActionExit); err != nil {
		t.Errorf("DefaultChmod failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChmod != "exit" {
		t.Errorf("Expected DefaultChmod to be Exit, got %s", sandbox.DefaultChmod)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultChmod(ActionDeny); err != nil {
		t.Errorf("DefaultChmod failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChmod != "deny" {
		t.Errorf("Expected DefaultChmod to be Deny, got %s", sandbox.DefaultChmod)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChattr != "deny" {
		t.Errorf("Expected DefaultChattr to be Deny, got %s", sandbox.DefaultChattr)
	}
	if err := DefaultChattr(ActionAllow); err != nil {
		t.Errorf("DefaultChattr failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChattr != "allow" {
		t.Errorf("Expected DefaultChattr to be Allow, got %s", sandbox.DefaultChattr)
	}
	if err := DefaultChattr(ActionWarn); err != nil {
		t.Errorf("DefaultChattr failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChattr != "warn" {
		t.Errorf("Expected DefaultChattr to be Warn, got %s", sandbox.DefaultChattr)
	}
	if err := DefaultChattr(ActionFilter); err != nil {
		t.Errorf("DefaultChattr failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChattr != "filter" {
		t.Errorf("Expected DefaultChattr to be Filter, got %s", sandbox.DefaultChattr)
	}
	if err := DefaultChattr(ActionPanic); err != nil {
		t.Errorf("DefaultChattr failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChattr != "panic" {
		t.Errorf("Expected DefaultChattr to be Panic, got %s", sandbox.DefaultChattr)
	}
	if err := DefaultChattr(ActionStop); err != nil {
		t.Errorf("DefaultChattr failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChattr != "stop" {
		t.Errorf("Expected DefaultChattr to be Stop, got %s", sandbox.DefaultChattr)
	}
	if err := DefaultChattr(ActionAbort); err != nil {
		t.Errorf("DefaultChattr failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChattr != "abort" {
		t.Errorf("Expected DefaultChattr to be Abort, got %s", sandbox.DefaultChattr)
	}
	if err := DefaultChattr(ActionKill); err != nil {
		t.Errorf("DefaultChattr failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChattr != "kill" {
		t.Errorf("Expected DefaultChattr to be Kill, got %s", sandbox.DefaultChattr)
	}
	if err := DefaultChattr(ActionExit); err != nil {
		t.Errorf("DefaultChattr failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChattr != "exit" {
		t.Errorf("Expected DefaultChattr to be Exit, got %s", sandbox.DefaultChattr)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultChattr(ActionDeny); err != nil {
		t.Errorf("DefaultChattr failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChattr != "deny" {
		t.Errorf("Expected DefaultChattr to be Deny, got %s", sandbox.DefaultChattr)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChroot != "deny" {
		t.Errorf("Expected DefaultChroot to be Deny, got %s", sandbox.DefaultChroot)
	}
	if err := DefaultChroot(ActionAllow); err != nil {
		t.Errorf("DefaultChroot failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChroot != "allow" {
		t.Errorf("Expected DefaultChroot to be Allow, got %s", sandbox.DefaultChroot)
	}
	if err := DefaultChroot(ActionWarn); err != nil {
		t.Errorf("DefaultChroot failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChroot != "warn" {
		t.Errorf("Expected DefaultChroot to be Warn, got %s", sandbox.DefaultChroot)
	}
	if err := DefaultChroot(ActionFilter); err != nil {
		t.Errorf("DefaultChroot failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChroot != "filter" {
		t.Errorf("Expected DefaultChroot to be Filter, got %s", sandbox.DefaultChroot)
	}
	if err := DefaultChroot(ActionPanic); err != nil {
		t.Errorf("DefaultChroot failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChroot != "panic" {
		t.Errorf("Expected DefaultChroot to be Panic, got %s", sandbox.DefaultChroot)
	}
	if err := DefaultChroot(ActionStop); err != nil {
		t.Errorf("DefaultChroot failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChroot != "stop" {
		t.Errorf("Expected DefaultChroot to be Stop, got %s", sandbox.DefaultChroot)
	}
	if err := DefaultChroot(ActionAbort); err != nil {
		t.Errorf("DefaultChroot failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChroot != "abort" {
		t.Errorf("Expected DefaultChroot to be Abort, got %s", sandbox.DefaultChroot)
	}
	if err := DefaultChroot(ActionKill); err != nil {
		t.Errorf("DefaultChroot failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChroot != "kill" {
		t.Errorf("Expected DefaultChroot to be Kill, got %s", sandbox.DefaultChroot)
	}
	if err := DefaultChroot(ActionExit); err != nil {
		t.Errorf("DefaultChroot failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChroot != "exit" {
		t.Errorf("Expected DefaultChroot to be Exit, got %s", sandbox.DefaultChroot)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultChroot(ActionDeny); err != nil {
		t.Errorf("DefaultChroot failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultChroot != "deny" {
		t.Errorf("Expected DefaultChroot to be Deny, got %s", sandbox.DefaultChroot)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultUtime != "deny" {
		t.Errorf("Expected DefaultUtime to be Deny, got %s", sandbox.DefaultUtime)
	}
	if err := DefaultUtime(ActionAllow); err != nil {
		t.Errorf("DefaultUtime failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultUtime != "allow" {
		t.Errorf("Expected DefaultUtime to be Allow, got %s", sandbox.DefaultUtime)
	}
	if err := DefaultUtime(ActionWarn); err != nil {
		t.Errorf("DefaultUtime failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultUtime != "warn" {
		t.Errorf("Expected DefaultUtime to be Warn, got %s", sandbox.DefaultUtime)
	}
	if err := DefaultUtime(ActionFilter); err != nil {
		t.Errorf("DefaultUtime failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultUtime != "filter" {
		t.Errorf("Expected DefaultUtime to be Filter, got %s", sandbox.DefaultUtime)
	}
	if err := DefaultUtime(ActionPanic); err != nil {
		t.Errorf("DefaultUtime failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultUtime != "panic" {
		t.Errorf("Expected DefaultUtime to be Panic, got %s", sandbox.DefaultUtime)
	}
	if err := DefaultUtime(ActionStop); err != nil {
		t.Errorf("DefaultUtime failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultUtime != "stop" {
		t.Errorf("Expected DefaultUtime to be Stop, got %s", sandbox.DefaultUtime)
	}
	if err := DefaultUtime(ActionAbort); err != nil {
		t.Errorf("DefaultUtime failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultUtime != "abort" {
		t.Errorf("Expected DefaultUtime to be Abort, got %s", sandbox.DefaultUtime)
	}
	if err := DefaultUtime(ActionKill); err != nil {
		t.Errorf("DefaultUtime failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultUtime != "kill" {
		t.Errorf("Expected DefaultUtime to be Kill, got %s", sandbox.DefaultUtime)
	}
	if err := DefaultUtime(ActionExit); err != nil {
		t.Errorf("DefaultUtime failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultUtime != "exit" {
		t.Errorf("Expected DefaultUtime to be Exit, got %s", sandbox.DefaultUtime)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultUtime(ActionDeny); err != nil {
		t.Errorf("DefaultUtime failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultUtime != "deny" {
		t.Errorf("Expected DefaultUtime to be Deny, got %s", sandbox.DefaultUtime)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkbdev != "deny" {
		t.Errorf("Expected DefaultMkbdev to be Deny, got %s", sandbox.DefaultMkbdev)
	}
	if err := DefaultMkbdev(ActionAllow); err != nil {
		t.Errorf("DefaultMkbdev failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkbdev != "allow" {
		t.Errorf("Expected DefaultMkbdev to be Allow, got %s", sandbox.DefaultMkbdev)
	}
	if err := DefaultMkbdev(ActionWarn); err != nil {
		t.Errorf("DefaultMkbdev failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkbdev != "warn" {
		t.Errorf("Expected DefaultMkbdev to be Warn, got %s", sandbox.DefaultMkbdev)
	}
	if err := DefaultMkbdev(ActionFilter); err != nil {
		t.Errorf("DefaultMkbdev failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkbdev != "filter" {
		t.Errorf("Expected DefaultMkbdev to be Filter, got %s", sandbox.DefaultMkbdev)
	}
	if err := DefaultMkbdev(ActionPanic); err != nil {
		t.Errorf("DefaultMkbdev failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkbdev != "panic" {
		t.Errorf("Expected DefaultMkbdev to be Panic, got %s", sandbox.DefaultMkbdev)
	}
	if err := DefaultMkbdev(ActionStop); err != nil {
		t.Errorf("DefaultMkbdev failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkbdev != "stop" {
		t.Errorf("Expected DefaultMkbdev to be Stop, got %s", sandbox.DefaultMkbdev)
	}
	if err := DefaultMkbdev(ActionAbort); err != nil {
		t.Errorf("DefaultMkbdev failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkbdev != "abort" {
		t.Errorf("Expected DefaultMkbdev to be Abort, got %s", sandbox.DefaultMkbdev)
	}
	if err := DefaultMkbdev(ActionKill); err != nil {
		t.Errorf("DefaultMkbdev failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkbdev != "kill" {
		t.Errorf("Expected DefaultMkbdev to be Kill, got %s", sandbox.DefaultMkbdev)
	}
	if err := DefaultMkbdev(ActionExit); err != nil {
		t.Errorf("DefaultMkbdev failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkbdev != "exit" {
		t.Errorf("Expected DefaultMkbdev to be Exit, got %s", sandbox.DefaultMkbdev)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultMkbdev(ActionDeny); err != nil {
		t.Errorf("DefaultMkbdev failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkbdev != "deny" {
		t.Errorf("Expected DefaultMkbdev to be Deny, got %s", sandbox.DefaultMkbdev)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkcdev != "deny" {
		t.Errorf("Expected DefaultMkcdev to be Deny, got %s", sandbox.DefaultMkcdev)
	}
	if err := DefaultMkcdev(ActionAllow); err != nil {
		t.Errorf("DefaultMkcdev failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkcdev != "allow" {
		t.Errorf("Expected DefaultMkcdev to be Allow, got %s", sandbox.DefaultMkcdev)
	}
	if err := DefaultMkcdev(ActionWarn); err != nil {
		t.Errorf("DefaultMkcdev failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkcdev != "warn" {
		t.Errorf("Expected DefaultMkcdev to be Warn, got %s", sandbox.DefaultMkcdev)
	}
	if err := DefaultMkcdev(ActionFilter); err != nil {
		t.Errorf("DefaultMkcdev failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkcdev != "filter" {
		t.Errorf("Expected DefaultMkcdev to be Filter, got %s", sandbox.DefaultMkcdev)
	}
	if err := DefaultMkcdev(ActionPanic); err != nil {
		t.Errorf("DefaultMkcdev failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkcdev != "panic" {
		t.Errorf("Expected DefaultMkcdev to be Panic, got %s", sandbox.DefaultMkcdev)
	}
	if err := DefaultMkcdev(ActionStop); err != nil {
		t.Errorf("DefaultMkcdev failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkcdev != "stop" {
		t.Errorf("Expected DefaultMkcdev to be Stop, got %s", sandbox.DefaultMkcdev)
	}
	if err := DefaultMkcdev(ActionAbort); err != nil {
		t.Errorf("DefaultMkcdev failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkcdev != "abort" {
		t.Errorf("Expected DefaultMkcdev to be Abort, got %s", sandbox.DefaultMkcdev)
	}
	if err := DefaultMkcdev(ActionKill); err != nil {
		t.Errorf("DefaultMkcdev failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkcdev != "kill" {
		t.Errorf("Expected DefaultMkcdev to be Kill, got %s", sandbox.DefaultMkcdev)
	}
	if err := DefaultMkcdev(ActionExit); err != nil {
		t.Errorf("DefaultMkcdev failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkcdev != "exit" {
		t.Errorf("Expected DefaultMkcdev to be Exit, got %s", sandbox.DefaultMkcdev)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultMkcdev(ActionDeny); err != nil {
		t.Errorf("DefaultMkcdev failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkcdev != "deny" {
		t.Errorf("Expected DefaultMkcdev to be Deny, got %s", sandbox.DefaultMkcdev)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkfifo != "deny" {
		t.Errorf("Expected DefaultMkfifo to be Deny, got %s", sandbox.DefaultMkfifo)
	}
	if err := DefaultMkfifo(ActionAllow); err != nil {
		t.Errorf("DefaultMkfifo failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkfifo != "allow" {
		t.Errorf("Expected DefaultMkfifo to be Allow, got %s", sandbox.DefaultMkfifo)
	}
	if err := DefaultMkfifo(ActionWarn); err != nil {
		t.Errorf("DefaultMkfifo failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkfifo != "warn" {
		t.Errorf("Expected DefaultMkfifo to be Warn, got %s", sandbox.DefaultMkfifo)
	}
	if err := DefaultMkfifo(ActionFilter); err != nil {
		t.Errorf("DefaultMkfifo failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkfifo != "filter" {
		t.Errorf("Expected DefaultMkfifo to be Filter, got %s", sandbox.DefaultMkfifo)
	}
	if err := DefaultMkfifo(ActionPanic); err != nil {
		t.Errorf("DefaultMkfifo failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkfifo != "panic" {
		t.Errorf("Expected DefaultMkfifo to be Panic, got %s", sandbox.DefaultMkfifo)
	}
	if err := DefaultMkfifo(ActionStop); err != nil {
		t.Errorf("DefaultMkfifo failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkfifo != "stop" {
		t.Errorf("Expected DefaultMkfifo to be Stop, got %s", sandbox.DefaultMkfifo)
	}
	if err := DefaultMkfifo(ActionAbort); err != nil {
		t.Errorf("DefaultMkfifo failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkfifo != "abort" {
		t.Errorf("Expected DefaultMkfifo to be Abort, got %s", sandbox.DefaultMkfifo)
	}
	if err := DefaultMkfifo(ActionKill); err != nil {
		t.Errorf("DefaultMkfifo failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkfifo != "kill" {
		t.Errorf("Expected DefaultMkfifo to be Kill, got %s", sandbox.DefaultMkfifo)
	}
	if err := DefaultMkfifo(ActionExit); err != nil {
		t.Errorf("DefaultMkfifo failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkfifo != "exit" {
		t.Errorf("Expected DefaultMkfifo to be Exit, got %s", sandbox.DefaultMkfifo)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultMkfifo(ActionDeny); err != nil {
		t.Errorf("DefaultMkfifo failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMkfifo != "deny" {
		t.Errorf("Expected DefaultMkfifo to be Deny, got %s", sandbox.DefaultMkfifo)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMktemp != "deny" {
		t.Errorf("Expected DefaultMktemp to be Deny, got %s", sandbox.DefaultMktemp)
	}
	if err := DefaultMktemp(ActionAllow); err != nil {
		t.Errorf("DefaultMktemp failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMktemp != "allow" {
		t.Errorf("Expected DefaultMktemp to be Allow, got %s", sandbox.DefaultMktemp)
	}
	if err := DefaultMktemp(ActionWarn); err != nil {
		t.Errorf("DefaultMktemp failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMktemp != "warn" {
		t.Errorf("Expected DefaultMktemp to be Warn, got %s", sandbox.DefaultMktemp)
	}
	if err := DefaultMktemp(ActionFilter); err != nil {
		t.Errorf("DefaultMktemp failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMktemp != "filter" {
		t.Errorf("Expected DefaultMktemp to be Filter, got %s", sandbox.DefaultMktemp)
	}
	if err := DefaultMktemp(ActionPanic); err != nil {
		t.Errorf("DefaultMktemp failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMktemp != "panic" {
		t.Errorf("Expected DefaultMktemp to be Panic, got %s", sandbox.DefaultMktemp)
	}
	if err := DefaultMktemp(ActionStop); err != nil {
		t.Errorf("DefaultMktemp failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMktemp != "stop" {
		t.Errorf("Expected DefaultMktemp to be Stop, got %s", sandbox.DefaultMktemp)
	}
	if err := DefaultMktemp(ActionAbort); err != nil {
		t.Errorf("DefaultMktemp failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMktemp != "abort" {
		t.Errorf("Expected DefaultMktemp to be Abort, got %s", sandbox.DefaultMktemp)
	}
	if err := DefaultMktemp(ActionKill); err != nil {
		t.Errorf("DefaultMktemp failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMktemp != "kill" {
		t.Errorf("Expected DefaultMktemp to be Kill, got %s", sandbox.DefaultMktemp)
	}
	if err := DefaultMktemp(ActionExit); err != nil {
		t.Errorf("DefaultMktemp failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMktemp != "exit" {
		t.Errorf("Expected DefaultMktemp to be Exit, got %s", sandbox.DefaultMktemp)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultMktemp(ActionDeny); err != nil {
		t.Errorf("DefaultMktemp failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMktemp != "deny" {
		t.Errorf("Expected DefaultMktemp to be Deny, got %s", sandbox.DefaultMktemp)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultForce != "deny" {
		t.Errorf("Expected DefaultForce to be Deny, got %s", sandbox.DefaultForce)
	}
	if err := DefaultForce(ActionAllow); err == nil {
		t.Errorf("DefaultForce did not fail with ActionAllow")
	} else if !errors.Is(err, syscall.EINVAL) {
		t.Errorf("Expected EINVAL error, got: %v", err)
	}
	if err := DefaultForce(ActionWarn); err != nil {
		t.Errorf("DefaultForce failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultForce != "warn" {
		t.Errorf("Expected DefaultForce to be Warn, got %s", sandbox.DefaultForce)
	}
	if err := DefaultForce(ActionFilter); err != nil {
		t.Errorf("DefaultForce failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultForce != "filter" {
		t.Errorf("Expected DefaultForce to be Filter, got %s", sandbox.DefaultForce)
	}
	if err := DefaultForce(ActionPanic); err != nil {
		t.Errorf("DefaultForce failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultForce != "panic" {
		t.Errorf("Expected DefaultForce to be Panic, got %s", sandbox.DefaultForce)
	}
	if err := DefaultForce(ActionStop); err != nil {
		t.Errorf("DefaultForce failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultForce != "stop" {
		t.Errorf("Expected DefaultForce to be Stop, got %s", sandbox.DefaultForce)
	}
	if err := DefaultForce(ActionAbort); err != nil {
		t.Errorf("DefaultForce failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultForce != "abort" {
		t.Errorf("Expected DefaultForce to be Abort, got %s", sandbox.DefaultForce)
	}
	if err := DefaultForce(ActionKill); err != nil {
		t.Errorf("DefaultForce failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultForce != "kill" {
		t.Errorf("Expected DefaultForce to be Kill, got %s", sandbox.DefaultForce)
	}
	if err := DefaultForce(ActionExit); err != nil {
		t.Errorf("DefaultForce failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultForce != "exit" {
		t.Errorf("Expected DefaultForce to be Exit, got %s", sandbox.DefaultForce)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultForce(ActionDeny); err != nil {
		t.Errorf("DefaultForce failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultForce != "deny" {
		t.Errorf("Expected DefaultForce to be Deny, got %s", sandbox.DefaultForce)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMem != "kill" {
		t.Errorf("Expected DefaultMem to be Deny, got %s", sandbox.DefaultMem)
	}
	if err := DefaultMem(ActionAllow); err == nil {
		t.Errorf("DefaultMem did not fail with ActionAllow")
	} else if !errors.Is(err, syscall.EINVAL) {
		t.Errorf("Expected EINVAL error, got: %v", err)
	}
	if err := DefaultMem(ActionWarn); err != nil {
		t.Errorf("DefaultMem failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMem != "warn" {
		t.Errorf("Expected DefaultMem to be Warn, got %s", sandbox.DefaultMem)
	}
	if err := DefaultMem(ActionFilter); err != nil {
		t.Errorf("DefaultMem failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMem != "filter" {
		t.Errorf("Expected DefaultMem to be Filter, got %s", sandbox.DefaultMem)
	}
	if err := DefaultMem(ActionDeny); err != nil {
		t.Errorf("DefaultMem failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMem != "deny" {
		t.Errorf("Expected DefaultMem to be Filter, got %s", sandbox.DefaultMem)
	}
	if err := DefaultMem(ActionPanic); err != nil {
		t.Errorf("DefaultMem failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMem != "panic" {
		t.Errorf("Expected DefaultMem to be Panic, got %s", sandbox.DefaultMem)
	}
	if err := DefaultMem(ActionStop); err != nil {
		t.Errorf("DefaultMem failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMem != "stop" {
		t.Errorf("Expected DefaultMem to be Stop, got %s", sandbox.DefaultMem)
	}
	if err := DefaultMem(ActionAbort); err != nil {
		t.Errorf("DefaultMem failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMem != "abort" {
		t.Errorf("Expected DefaultMem to be Abort, got %s", sandbox.DefaultMem)
	}
	if err := DefaultMem(ActionExit); err != nil {
		t.Errorf("DefaultMem failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMem != "exit" {
		t.Errorf("Expected DefaultMem to be Exit, got %s", sandbox.DefaultMem)
	}
	// Ensure we reset to Kill last, so other tests are uneffected.
	if err := DefaultMem(ActionKill); err != nil {
		t.Errorf("DefaultMem failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultMem != "kill" {
		t.Errorf("Expected DefaultMem to be Deny, got %s", sandbox.DefaultMem)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultPid != "kill" {
		t.Errorf("Expected DefaultPid to be Kill, got %s", sandbox.DefaultPid)
	}
	if err := DefaultPid(ActionAllow); err == nil {
		t.Errorf("DefaultPid did not fail with ActionAllow")
	} else if !errors.Is(err, syscall.EINVAL) {
		t.Errorf("Expected EINVAL error, got: %v", err)
	}
	if err := DefaultPid(ActionWarn); err != nil {
		t.Errorf("DefaultPid failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultPid != "warn" {
		t.Errorf("Expected DefaultPid to be Warn, got %s", sandbox.DefaultPid)
	}
	if err := DefaultPid(ActionFilter); err != nil {
		t.Errorf("DefaultPid failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultPid != "filter" {
		t.Errorf("Expected DefaultPid to be Filter, got %s", sandbox.DefaultPid)
	}
	if err := DefaultPid(ActionDeny); err == nil {
		t.Errorf("DefaultPid did not fail with ActionDeny")
	} else if !errors.Is(err, syscall.EINVAL) {
		t.Errorf("Expected EINVAL error, got: %v", err)
	}
	if err := DefaultPid(ActionPanic); err == nil {
		t.Errorf("DefaultPid did not fail with ActionPanic")
	} else if !errors.Is(err, syscall.EINVAL) {
		t.Errorf("Expected EINVAL error, got: %v", err)
	}
	if err := DefaultPid(ActionStop); err != nil {
		t.Errorf("DefaultPid failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultPid != "stop" {
		t.Errorf("Expected DefaultPid to be Stop, got %s", sandbox.DefaultPid)
	}
	if err := DefaultPid(ActionAbort); err != nil {
		t.Errorf("DefaultPid failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultPid != "abort" {
		t.Errorf("Expected DefaultPid to be Abort, got %s", sandbox.DefaultPid)
	}
	if err := DefaultPid(ActionExit); err != nil {
		t.Errorf("DefaultPid failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultPid != "exit" {
		t.Errorf("Expected DefaultPid to be Exit, got %s", sandbox.DefaultPid)
	}
	// Ensure we reset to Kill last, so other tests are uneffected.
	if err := DefaultPid(ActionKill); err != nil {
		t.Errorf("DefaultPid failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultPid != "kill" {
		t.Errorf("Expected DefaultPid to be Kill, got %s", sandbox.DefaultPid)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultTPE != "deny" {
		t.Errorf("Expected DefaultTPE to be Deny, got %s", sandbox.DefaultTPE)
	}
	if err := DefaultTPE(ActionAllow); err == nil {
		t.Errorf("DefaultTPE did not fail with ActionAllow")
	} else if !errors.Is(err, syscall.EINVAL) {
		t.Errorf("Expected EINVAL error, got: %v", err)
	}
	if err := DefaultTPE(ActionWarn); err != nil {
		t.Errorf("DefaultTPE failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultTPE != "warn" {
		t.Errorf("Expected DefaultTPE to be Warn, got %s", sandbox.DefaultTPE)
	}
	if err := DefaultTPE(ActionFilter); err != nil {
		t.Errorf("DefaultTPE failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultTPE != "filter" {
		t.Errorf("Expected DefaultTPE to be Filter, got %s", sandbox.DefaultTPE)
	}
	if err := DefaultTPE(ActionPanic); err != nil {
		t.Errorf("DefaultTPE failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultTPE != "panic" {
		t.Errorf("Expected DefaultTPE to be Panic, got %s", sandbox.DefaultTPE)
	}
	if err := DefaultTPE(ActionStop); err != nil {
		t.Errorf("DefaultTPE failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultTPE != "stop" {
		t.Errorf("Expected DefaultTPE to be Stop, got %s", sandbox.DefaultTPE)
	}
	if err := DefaultTPE(ActionAbort); err != nil {
		t.Errorf("DefaultTPE failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultTPE != "abort" {
		t.Errorf("Expected DefaultTPE to be Abort, got %s", sandbox.DefaultTPE)
	}
	if err := DefaultTPE(ActionKill); err != nil {
		t.Errorf("DefaultTPE failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultTPE != "kill" {
		t.Errorf("Expected DefaultTPE to be Kill, got %s", sandbox.DefaultTPE)
	}
	if err := DefaultTPE(ActionExit); err != nil {
		t.Errorf("DefaultTPE failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultTPE != "exit" {
		t.Errorf("Expected DefaultTPE to be Exit, got %s", sandbox.DefaultTPE)
	}
	// Ensure we reset to Deny last, so other tests are uneffected.
	if err := DefaultTPE(ActionDeny); err != nil {
		t.Errorf("DefaultTPE failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.DefaultTPE != "deny" {
		t.Errorf("Expected DefaultTPE to be Deny, got %s", sandbox.DefaultTPE)
	}
}

func Test_04_Glob(t *testing.T) {
	path := "/tmp/gosyd"

	testCases := []struct {
		AddFunc     func(Action, string) error
		DelFunc     func(Action, string) error
		RemFunc     func(Action, string) error
		Act         Action
		ActStr, Cap string
	}{
		{WalkAdd, WalkDel, WalkRem, ActionAllow, "allow", "walk"},
		{WalkAdd, WalkDel, WalkRem, ActionWarn, "warn", "walk"},
		{WalkAdd, WalkDel, WalkRem, ActionFilter, "filter", "walk"},
		{WalkAdd, WalkDel, WalkRem, ActionPanic, "panic", "walk"},
		{WalkAdd, WalkDel, WalkRem, ActionStop, "stop", "walk"},
		{WalkAdd, WalkDel, WalkRem, ActionAbort, "abort", "walk"},
		{WalkAdd, WalkDel, WalkRem, ActionKill, "kill", "walk"},
		{WalkAdd, WalkDel, WalkRem, ActionExit, "exit", "walk"},
		{WalkAdd, WalkDel, WalkRem, ActionDeny, "deny", "walk"},

		{StatAdd, StatDel, StatRem, ActionAllow, "allow", "stat"},
		{StatAdd, StatDel, StatRem, ActionWarn, "warn", "stat"},
		{StatAdd, StatDel, StatRem, ActionFilter, "filter", "stat"},
		{StatAdd, StatDel, StatRem, ActionPanic, "panic", "stat"},
		{StatAdd, StatDel, StatRem, ActionStop, "stop", "stat"},
		{StatAdd, StatDel, StatRem, ActionAbort, "abort", "stat"},
		{StatAdd, StatDel, StatRem, ActionKill, "kill", "stat"},
		{StatAdd, StatDel, StatRem, ActionExit, "exit", "stat"},
		{StatAdd, StatDel, StatRem, ActionDeny, "deny", "stat"},

		{ReadAdd, ReadDel, ReadRem, ActionAllow, "allow", "read"},
		{ReadAdd, ReadDel, ReadRem, ActionWarn, "warn", "read"},
		{ReadAdd, ReadDel, ReadRem, ActionFilter, "filter", "read"},
		{ReadAdd, ReadDel, ReadRem, ActionPanic, "panic", "read"},
		{ReadAdd, ReadDel, ReadRem, ActionStop, "stop", "read"},
		{ReadAdd, ReadDel, ReadRem, ActionAbort, "abort", "read"},
		{ReadAdd, ReadDel, ReadRem, ActionKill, "kill", "read"},
		{ReadAdd, ReadDel, ReadRem, ActionExit, "exit", "read"},
		{ReadAdd, ReadDel, ReadRem, ActionDeny, "deny", "read"},

		{WriteAdd, WriteDel, WriteRem, ActionAllow, "allow", "write"},
		{WriteAdd, WriteDel, WriteRem, ActionWarn, "warn", "write"},
		{WriteAdd, WriteDel, WriteRem, ActionFilter, "filter", "write"},
		{WriteAdd, WriteDel, WriteRem, ActionPanic, "panic", "write"},
		{WriteAdd, WriteDel, WriteRem, ActionStop, "stop", "write"},
		{WriteAdd, WriteDel, WriteRem, ActionAbort, "abort", "write"},
		{WriteAdd, WriteDel, WriteRem, ActionKill, "kill", "write"},
		{WriteAdd, WriteDel, WriteRem, ActionExit, "exit", "write"},
		{WriteAdd, WriteDel, WriteRem, ActionDeny, "deny", "write"},

		{ExecAdd, ExecDel, ExecRem, ActionAllow, "allow", "exec"},
		{ExecAdd, ExecDel, ExecRem, ActionWarn, "warn", "exec"},
		{ExecAdd, ExecDel, ExecRem, ActionFilter, "filter", "exec"},
		{ExecAdd, ExecDel, ExecRem, ActionPanic, "panic", "exec"},
		{ExecAdd, ExecDel, ExecRem, ActionStop, "stop", "exec"},
		{ExecAdd, ExecDel, ExecRem, ActionAbort, "abort", "exec"},
		{ExecAdd, ExecDel, ExecRem, ActionKill, "kill", "exec"},
		{ExecAdd, ExecDel, ExecRem, ActionExit, "exit", "exec"},
		{ExecAdd, ExecDel, ExecRem, ActionDeny, "deny", "exec"},

		{IoctlAdd, IoctlDel, IoctlRem, ActionAllow, "allow", "ioctl"},
		{IoctlAdd, IoctlDel, IoctlRem, ActionWarn, "warn", "ioctl"},
		{IoctlAdd, IoctlDel, IoctlRem, ActionFilter, "filter", "ioctl"},
		{IoctlAdd, IoctlDel, IoctlRem, ActionPanic, "panic", "ioctl"},
		{IoctlAdd, IoctlDel, IoctlRem, ActionStop, "stop", "ioctl"},
		{IoctlAdd, IoctlDel, IoctlRem, ActionAbort, "abort", "ioctl"},
		{IoctlAdd, IoctlDel, IoctlRem, ActionKill, "kill", "ioctl"},
		{IoctlAdd, IoctlDel, IoctlRem, ActionExit, "exit", "ioctl"},
		{IoctlAdd, IoctlDel, IoctlRem, ActionDeny, "deny", "ioctl"},

		{CreateAdd, CreateDel, CreateRem, ActionAllow, "allow", "create"},
		{CreateAdd, CreateDel, CreateRem, ActionWarn, "warn", "create"},
		{CreateAdd, CreateDel, CreateRem, ActionFilter, "filter", "create"},
		{CreateAdd, CreateDel, CreateRem, ActionPanic, "panic", "create"},
		{CreateAdd, CreateDel, CreateRem, ActionStop, "stop", "create"},
		{CreateAdd, CreateDel, CreateRem, ActionAbort, "abort", "create"},
		{CreateAdd, CreateDel, CreateRem, ActionKill, "kill", "create"},
		{CreateAdd, CreateDel, CreateRem, ActionExit, "exit", "create"},
		{CreateAdd, CreateDel, CreateRem, ActionDeny, "deny", "create"},

		{DeleteAdd, DeleteDel, DeleteRem, ActionAllow, "allow", "delete"},
		{DeleteAdd, DeleteDel, DeleteRem, ActionWarn, "warn", "delete"},
		{DeleteAdd, DeleteDel, DeleteRem, ActionFilter, "filter", "delete"},
		{DeleteAdd, DeleteDel, DeleteRem, ActionPanic, "panic", "delete"},
		{DeleteAdd, DeleteDel, DeleteRem, ActionStop, "stop", "delete"},
		{DeleteAdd, DeleteDel, DeleteRem, ActionAbort, "abort", "delete"},
		{DeleteAdd, DeleteDel, DeleteRem, ActionKill, "kill", "delete"},
		{DeleteAdd, DeleteDel, DeleteRem, ActionExit, "exit", "delete"},
		{DeleteAdd, DeleteDel, DeleteRem, ActionDeny, "deny", "delete"},

		{RenameAdd, RenameDel, RenameRem, ActionAllow, "allow", "rename"},
		{RenameAdd, RenameDel, RenameRem, ActionWarn, "warn", "rename"},
		{RenameAdd, RenameDel, RenameRem, ActionFilter, "filter", "rename"},
		{RenameAdd, RenameDel, RenameRem, ActionPanic, "panic", "rename"},
		{RenameAdd, RenameDel, RenameRem, ActionStop, "stop", "rename"},
		{RenameAdd, RenameDel, RenameRem, ActionAbort, "abort", "rename"},
		{RenameAdd, RenameDel, RenameRem, ActionKill, "kill", "rename"},
		{RenameAdd, RenameDel, RenameRem, ActionExit, "exit", "rename"},
		{RenameAdd, RenameDel, RenameRem, ActionDeny, "deny", "rename"},

		{SymlinkAdd, SymlinkDel, SymlinkRem, ActionAllow, "allow", "symlink"},
		{SymlinkAdd, SymlinkDel, SymlinkRem, ActionWarn, "warn", "symlink"},
		{SymlinkAdd, SymlinkDel, SymlinkRem, ActionFilter, "filter", "symlink"},
		{SymlinkAdd, SymlinkDel, SymlinkRem, ActionPanic, "panic", "symlink"},
		{SymlinkAdd, SymlinkDel, SymlinkRem, ActionStop, "stop", "symlink"},
		{SymlinkAdd, SymlinkDel, SymlinkRem, ActionAbort, "abort", "symlink"},
		{SymlinkAdd, SymlinkDel, SymlinkRem, ActionKill, "kill", "symlink"},
		{SymlinkAdd, SymlinkDel, SymlinkRem, ActionExit, "exit", "symlink"},
		{SymlinkAdd, SymlinkDel, SymlinkRem, ActionDeny, "deny", "symlink"},

		{TruncateAdd, TruncateDel, TruncateRem, ActionAllow, "allow", "truncate"},
		{TruncateAdd, TruncateDel, TruncateRem, ActionWarn, "warn", "truncate"},
		{TruncateAdd, TruncateDel, TruncateRem, ActionFilter, "filter", "truncate"},
		{TruncateAdd, TruncateDel, TruncateRem, ActionPanic, "panic", "truncate"},
		{TruncateAdd, TruncateDel, TruncateRem, ActionStop, "stop", "truncate"},
		{TruncateAdd, TruncateDel, TruncateRem, ActionAbort, "abort", "truncate"},
		{TruncateAdd, TruncateDel, TruncateRem, ActionKill, "kill", "truncate"},
		{TruncateAdd, TruncateDel, TruncateRem, ActionExit, "exit", "truncate"},
		{TruncateAdd, TruncateDel, TruncateRem, ActionDeny, "deny", "truncate"},

		{ChdirAdd, ChdirDel, ChdirRem, ActionAllow, "allow", "chdir"},
		{ChdirAdd, ChdirDel, ChdirRem, ActionWarn, "warn", "chdir"},
		{ChdirAdd, ChdirDel, ChdirRem, ActionFilter, "filter", "chdir"},
		{ChdirAdd, ChdirDel, ChdirRem, ActionPanic, "panic", "chdir"},
		{ChdirAdd, ChdirDel, ChdirRem, ActionStop, "stop", "chdir"},
		{ChdirAdd, ChdirDel, ChdirRem, ActionAbort, "abort", "chdir"},
		{ChdirAdd, ChdirDel, ChdirRem, ActionKill, "kill", "chdir"},
		{ChdirAdd, ChdirDel, ChdirRem, ActionExit, "exit", "chdir"},
		{ChdirAdd, ChdirDel, ChdirRem, ActionDeny, "deny", "chdir"},

		{ReaddirAdd, ReaddirDel, ReaddirRem, ActionAllow, "allow", "readdir"},
		{ReaddirAdd, ReaddirDel, ReaddirRem, ActionWarn, "warn", "readdir"},
		{ReaddirAdd, ReaddirDel, ReaddirRem, ActionFilter, "filter", "readdir"},
		{ReaddirAdd, ReaddirDel, ReaddirRem, ActionPanic, "panic", "readdir"},
		{ReaddirAdd, ReaddirDel, ReaddirRem, ActionStop, "stop", "readdir"},
		{ReaddirAdd, ReaddirDel, ReaddirRem, ActionAbort, "abort", "readdir"},
		{ReaddirAdd, ReaddirDel, ReaddirRem, ActionKill, "kill", "readdir"},
		{ReaddirAdd, ReaddirDel, ReaddirRem, ActionExit, "exit", "readdir"},
		{ReaddirAdd, ReaddirDel, ReaddirRem, ActionDeny, "deny", "readdir"},

		{MkdirAdd, MkdirDel, MkdirRem, ActionAllow, "allow", "mkdir"},
		{MkdirAdd, MkdirDel, MkdirRem, ActionWarn, "warn", "mkdir"},
		{MkdirAdd, MkdirDel, MkdirRem, ActionFilter, "filter", "mkdir"},
		{MkdirAdd, MkdirDel, MkdirRem, ActionPanic, "panic", "mkdir"},
		{MkdirAdd, MkdirDel, MkdirRem, ActionStop, "stop", "mkdir"},
		{MkdirAdd, MkdirDel, MkdirRem, ActionAbort, "abort", "mkdir"},
		{MkdirAdd, MkdirDel, MkdirRem, ActionKill, "kill", "mkdir"},
		{MkdirAdd, MkdirDel, MkdirRem, ActionExit, "exit", "mkdir"},
		{MkdirAdd, MkdirDel, MkdirRem, ActionDeny, "deny", "mkdir"},

		{RmdirAdd, RmdirDel, RmdirRem, ActionAllow, "allow", "rmdir"},
		{RmdirAdd, RmdirDel, RmdirRem, ActionWarn, "warn", "rmdir"},
		{RmdirAdd, RmdirDel, RmdirRem, ActionFilter, "filter", "rmdir"},
		{RmdirAdd, RmdirDel, RmdirRem, ActionPanic, "panic", "rmdir"},
		{RmdirAdd, RmdirDel, RmdirRem, ActionStop, "stop", "rmdir"},
		{RmdirAdd, RmdirDel, RmdirRem, ActionAbort, "abort", "rmdir"},
		{RmdirAdd, RmdirDel, RmdirRem, ActionKill, "kill", "rmdir"},
		{RmdirAdd, RmdirDel, RmdirRem, ActionExit, "exit", "rmdir"},
		{RmdirAdd, RmdirDel, RmdirRem, ActionDeny, "deny", "rmdir"},

		{ChownAdd, ChownDel, ChownRem, ActionAllow, "allow", "chown"},
		{ChownAdd, ChownDel, ChownRem, ActionWarn, "warn", "chown"},
		{ChownAdd, ChownDel, ChownRem, ActionFilter, "filter", "chown"},
		{ChownAdd, ChownDel, ChownRem, ActionPanic, "panic", "chown"},
		{ChownAdd, ChownDel, ChownRem, ActionStop, "stop", "chown"},
		{ChownAdd, ChownDel, ChownRem, ActionAbort, "abort", "chown"},
		{ChownAdd, ChownDel, ChownRem, ActionKill, "kill", "chown"},
		{ChownAdd, ChownDel, ChownRem, ActionExit, "exit", "chown"},
		{ChownAdd, ChownDel, ChownRem, ActionDeny, "deny", "chown"},

		{ChgrpAdd, ChgrpDel, ChgrpRem, ActionAllow, "allow", "chgrp"},
		{ChgrpAdd, ChgrpDel, ChgrpRem, ActionWarn, "warn", "chgrp"},
		{ChgrpAdd, ChgrpDel, ChgrpRem, ActionFilter, "filter", "chgrp"},
		{ChgrpAdd, ChgrpDel, ChgrpRem, ActionPanic, "panic", "chgrp"},
		{ChgrpAdd, ChgrpDel, ChgrpRem, ActionStop, "stop", "chgrp"},
		{ChgrpAdd, ChgrpDel, ChgrpRem, ActionAbort, "abort", "chgrp"},
		{ChgrpAdd, ChgrpDel, ChgrpRem, ActionKill, "kill", "chgrp"},
		{ChgrpAdd, ChgrpDel, ChgrpRem, ActionExit, "exit", "chgrp"},
		{ChgrpAdd, ChgrpDel, ChgrpRem, ActionDeny, "deny", "chgrp"},

		{ChmodAdd, ChmodDel, ChmodRem, ActionAllow, "allow", "chmod"},
		{ChmodAdd, ChmodDel, ChmodRem, ActionWarn, "warn", "chmod"},
		{ChmodAdd, ChmodDel, ChmodRem, ActionFilter, "filter", "chmod"},
		{ChmodAdd, ChmodDel, ChmodRem, ActionPanic, "panic", "chmod"},
		{ChmodAdd, ChmodDel, ChmodRem, ActionStop, "stop", "chmod"},
		{ChmodAdd, ChmodDel, ChmodRem, ActionAbort, "abort", "chmod"},
		{ChmodAdd, ChmodDel, ChmodRem, ActionKill, "kill", "chmod"},
		{ChmodAdd, ChmodDel, ChmodRem, ActionExit, "exit", "chmod"},
		{ChmodAdd, ChmodDel, ChmodRem, ActionDeny, "deny", "chmod"},

		{ChattrAdd, ChattrDel, ChattrRem, ActionAllow, "allow", "chattr"},
		{ChattrAdd, ChattrDel, ChattrRem, ActionWarn, "warn", "chattr"},
		{ChattrAdd, ChattrDel, ChattrRem, ActionFilter, "filter", "chattr"},
		{ChattrAdd, ChattrDel, ChattrRem, ActionPanic, "panic", "chattr"},
		{ChattrAdd, ChattrDel, ChattrRem, ActionStop, "stop", "chattr"},
		{ChattrAdd, ChattrDel, ChattrRem, ActionAbort, "abort", "chattr"},
		{ChattrAdd, ChattrDel, ChattrRem, ActionKill, "kill", "chattr"},
		{ChattrAdd, ChattrDel, ChattrRem, ActionExit, "exit", "chattr"},
		{ChattrAdd, ChattrDel, ChattrRem, ActionDeny, "deny", "chattr"},

		{ChrootAdd, ChrootDel, ChrootRem, ActionAllow, "allow", "chroot"},
		{ChrootAdd, ChrootDel, ChrootRem, ActionWarn, "warn", "chroot"},
		{ChrootAdd, ChrootDel, ChrootRem, ActionFilter, "filter", "chroot"},
		{ChrootAdd, ChrootDel, ChrootRem, ActionPanic, "panic", "chroot"},
		{ChrootAdd, ChrootDel, ChrootRem, ActionStop, "stop", "chroot"},
		{ChrootAdd, ChrootDel, ChrootRem, ActionAbort, "abort", "chroot"},
		{ChrootAdd, ChrootDel, ChrootRem, ActionKill, "kill", "chroot"},
		{ChrootAdd, ChrootDel, ChrootRem, ActionExit, "exit", "chroot"},
		{ChrootAdd, ChrootDel, ChrootRem, ActionDeny, "deny", "chroot"},

		{UtimeAdd, UtimeDel, UtimeRem, ActionAllow, "allow", "utime"},
		{UtimeAdd, UtimeDel, UtimeRem, ActionWarn, "warn", "utime"},
		{UtimeAdd, UtimeDel, UtimeRem, ActionFilter, "filter", "utime"},
		{UtimeAdd, UtimeDel, UtimeRem, ActionPanic, "panic", "utime"},
		{UtimeAdd, UtimeDel, UtimeRem, ActionStop, "stop", "utime"},
		{UtimeAdd, UtimeDel, UtimeRem, ActionAbort, "abort", "utime"},
		{UtimeAdd, UtimeDel, UtimeRem, ActionKill, "kill", "utime"},
		{UtimeAdd, UtimeDel, UtimeRem, ActionExit, "exit", "utime"},
		{UtimeAdd, UtimeDel, UtimeRem, ActionDeny, "deny", "utime"},

		{MkbdevAdd, MkbdevDel, MkbdevRem, ActionAllow, "allow", "mkbdev"},
		{MkbdevAdd, MkbdevDel, MkbdevRem, ActionWarn, "warn", "mkbdev"},
		{MkbdevAdd, MkbdevDel, MkbdevRem, ActionFilter, "filter", "mkbdev"},
		{MkbdevAdd, MkbdevDel, MkbdevRem, ActionPanic, "panic", "mkbdev"},
		{MkbdevAdd, MkbdevDel, MkbdevRem, ActionStop, "stop", "mkbdev"},
		{MkbdevAdd, MkbdevDel, MkbdevRem, ActionAbort, "abort", "mkbdev"},
		{MkbdevAdd, MkbdevDel, MkbdevRem, ActionKill, "kill", "mkbdev"},
		{MkbdevAdd, MkbdevDel, MkbdevRem, ActionExit, "exit", "mkbdev"},
		{MkbdevAdd, MkbdevDel, MkbdevRem, ActionDeny, "deny", "mkbdev"},

		{MkcdevAdd, MkcdevDel, MkcdevRem, ActionAllow, "allow", "mkcdev"},
		{MkcdevAdd, MkcdevDel, MkcdevRem, ActionWarn, "warn", "mkcdev"},
		{MkcdevAdd, MkcdevDel, MkcdevRem, ActionFilter, "filter", "mkcdev"},
		{MkcdevAdd, MkcdevDel, MkcdevRem, ActionPanic, "panic", "mkcdev"},
		{MkcdevAdd, MkcdevDel, MkcdevRem, ActionStop, "stop", "mkcdev"},
		{MkcdevAdd, MkcdevDel, MkcdevRem, ActionAbort, "abort", "mkcdev"},
		{MkcdevAdd, MkcdevDel, MkcdevRem, ActionKill, "kill", "mkcdev"},
		{MkcdevAdd, MkcdevDel, MkcdevRem, ActionExit, "exit", "mkcdev"},
		{MkcdevAdd, MkcdevDel, MkcdevRem, ActionDeny, "deny", "mkcdev"},

		{MkfifoAdd, MkfifoDel, MkfifoRem, ActionAllow, "allow", "mkfifo"},
		{MkfifoAdd, MkfifoDel, MkfifoRem, ActionWarn, "warn", "mkfifo"},
		{MkfifoAdd, MkfifoDel, MkfifoRem, ActionFilter, "filter", "mkfifo"},
		{MkfifoAdd, MkfifoDel, MkfifoRem, ActionPanic, "panic", "mkfifo"},
		{MkfifoAdd, MkfifoDel, MkfifoRem, ActionStop, "stop", "mkfifo"},
		{MkfifoAdd, MkfifoDel, MkfifoRem, ActionAbort, "abort", "mkfifo"},
		{MkfifoAdd, MkfifoDel, MkfifoRem, ActionKill, "kill", "mkfifo"},
		{MkfifoAdd, MkfifoDel, MkfifoRem, ActionExit, "exit", "mkfifo"},
		{MkfifoAdd, MkfifoDel, MkfifoRem, ActionDeny, "deny", "mkfifo"},

		{MktempAdd, MktempDel, MktempRem, ActionAllow, "allow", "mktemp"},
		{MktempAdd, MktempDel, MktempRem, ActionWarn, "warn", "mktemp"},
		{MktempAdd, MktempDel, MktempRem, ActionFilter, "filter", "mktemp"},
		{MktempAdd, MktempDel, MktempRem, ActionPanic, "panic", "mktemp"},
		{MktempAdd, MktempDel, MktempRem, ActionStop, "stop", "mktemp"},
		{MktempAdd, MktempDel, MktempRem, ActionAbort, "abort", "mktemp"},
		{MktempAdd, MktempDel, MktempRem, ActionKill, "kill", "mktemp"},
		{MktempAdd, MktempDel, MktempRem, ActionExit, "exit", "mktemp"},
		{MktempAdd, MktempDel, MktempRem, ActionDeny, "deny", "mktemp"},

		{NetSendFdAdd, NetSendFdDel, NetSendFdRem, ActionAllow, "allow", "net/sendfd"},
		{NetSendFdAdd, NetSendFdDel, NetSendFdRem, ActionWarn, "warn", "net/sendfd"},
		{NetSendFdAdd, NetSendFdDel, NetSendFdRem, ActionFilter, "filter", "net/sendfd"},
		{NetSendFdAdd, NetSendFdDel, NetSendFdRem, ActionPanic, "panic", "net/sendfd"},
		{NetSendFdAdd, NetSendFdDel, NetSendFdRem, ActionStop, "stop", "net/sendfd"},
		{NetSendFdAdd, NetSendFdDel, NetSendFdRem, ActionAbort, "abort", "net/sendfd"},
		{NetSendFdAdd, NetSendFdDel, NetSendFdRem, ActionKill, "kill", "net/sendfd"},
		{NetSendFdAdd, NetSendFdDel, NetSendFdRem, ActionExit, "exit", "net/sendfd"},
		{NetSendFdAdd, NetSendFdDel, NetSendFdRem, ActionDeny, "deny", "net/sendfd"},
	}

	for _, tc := range testCases {
		testName := tc.ActStr + tc.Cap
		rule := GlobRule{Act: tc.ActStr, Cap: tc.Cap, Pat: path}

		// Test Add
		err := tc.AddFunc(tc.Act, path)
		if err != nil {
			t.Fatalf("%sAdd failed: %v", testName, err)
		}
		sandbox, err := Info()
		if err != nil {
			t.Fatalf("Info failed: %v", err)
		}
		idx := findGlob(sandbox.GlobRules, rule)
		if idx != len(sandbox.GlobRules)-1 {
			t.Errorf("Expected %s rule to be last, got index %d", testName, idx)
		}

		// Test Del
		err = tc.DelFunc(tc.Act, path)
		if err != nil {
			t.Fatalf("%sDel failed: %v", testName, err)
		}
		sandbox, err = Info()
		if err != nil {
			t.Fatalf("Info failed: %v", err)
		}
		idx = findGlob(sandbox.GlobRules, rule)
		if idx != -1 {
			t.Errorf("Expected %s rule to be absent, got index %d", testName, idx)
		}

		// Test Add, Add, Add, Rem
		err = tc.AddFunc(tc.Act, path)
		if err != nil {
			t.Fatalf("%sAdd failed: %v", testName, err)
		}
		err = tc.AddFunc(tc.Act, path)
		if err != nil {
			t.Fatalf("%sAdd failed: %v", testName, err)
		}
		err = tc.AddFunc(tc.Act, path)
		if err != nil {
			t.Fatalf("%sAdd failed: %v", testName, err)
		}
		err = tc.RemFunc(tc.Act, path)
		if err != nil {
			t.Fatalf("%sRem failed: %v", testName, err)
		}
		sandbox, err = Info()
		if err != nil {
			t.Fatalf("Info failed: %v", err)
		}
		idx = findGlob(sandbox.GlobRules, rule)
		if idx != -1 {
			t.Errorf("Expected %s rule to be absent after %sRem, got index %d", testName, testName, idx)
		}
	}
}

func Test_05_Cidr_Port_Single(t *testing.T) {
	host := "127.3.1.4/8"
	port := 16
	addr := host + "!" + fmt.Sprint(port)
	aarg := string(addr)

	testCases := []struct {
		AddFunc     func(Action, string) error
		DelFunc     func(Action, string) error
		RemFunc     func(Action, string) error
		Act         Action
		ActStr, Cap string
	}{
		{NetBindAdd, NetBindDel, NetBindRem, ActionAllow, "allow", "net/bind"},
		{NetBindAdd, NetBindDel, NetBindRem, ActionWarn, "warn", "net/bind"},
		{NetBindAdd, NetBindDel, NetBindRem, ActionFilter, "filter", "net/bind"},
		{NetBindAdd, NetBindDel, NetBindRem, ActionDeny, "deny", "net/bind"},
		{NetBindAdd, NetBindDel, NetBindRem, ActionStop, "stop", "net/bind"},
		{NetBindAdd, NetBindDel, NetBindRem, ActionKill, "kill", "net/bind"},
		{NetBindAdd, NetBindDel, NetBindRem, ActionExit, "exit", "net/bind"},

		{NetConnectAdd, NetConnectDel, NetConnectRem, ActionAllow, "allow", "net/connect"},
		{NetConnectAdd, NetConnectDel, NetConnectRem, ActionWarn, "warn", "net/connect"},
		{NetConnectAdd, NetConnectDel, NetConnectRem, ActionFilter, "filter", "net/connect"},
		{NetConnectAdd, NetConnectDel, NetConnectRem, ActionDeny, "deny", "net/connect"},
		{NetConnectAdd, NetConnectDel, NetConnectRem, ActionStop, "stop", "net/connect"},
		{NetConnectAdd, NetConnectDel, NetConnectRem, ActionKill, "kill", "net/connect"},
		{NetConnectAdd, NetConnectDel, NetConnectRem, ActionExit, "exit", "net/connect"},
	}

	for _, tc := range testCases {
		testName := tc.ActStr + tc.Cap
		rule := CidrRule{
			Act: tc.ActStr,
			Cap: tc.Cap,
			Pat: Pattern{Addr: host, Port: port},
		}

		// Test Add
		err := tc.AddFunc(tc.Act, aarg)
		if err != nil {
			t.Fatalf("%sAdd failed: %v", testName, err)
		}
		sandbox, err := Info()
		if err != nil {
			t.Fatalf("Info failed: %v", err)
		}
		idx := findCidr(sandbox.CidrRules, rule.Pat)
		if idx != len(sandbox.CidrRules)-1 {
			t.Errorf("Expected %s rule to be last, got index %d. CIDR Rules: %+v", testName, idx, sandbox.CidrRules)
		}

		// Test Del
		err = tc.DelFunc(tc.Act, aarg)
		if err != nil {
			t.Fatalf("%sDel failed: %v", testName, err)
		}
		sandbox, err = Info()
		if err != nil {
			t.Fatalf("Info failed: %v", err)
		}
		idx = findCidr(sandbox.CidrRules, rule.Pat)
		if idx != -1 {
			t.Errorf("Expected %s rule to be absent, got index %d", testName, idx)
		}

		// Test Add, Add, Add, Rem
		err = tc.AddFunc(tc.Act, aarg)
		if err != nil {
			t.Fatalf("%sAdd failed: %v", testName, err)
		}
		err = tc.AddFunc(tc.Act, aarg)
		if err != nil {
			t.Fatalf("%sAdd failed: %v", testName, err)
		}
		err = tc.AddFunc(tc.Act, aarg)
		if err != nil {
			t.Fatalf("%sAdd failed: %v", testName, err)
		}
		err = tc.RemFunc(tc.Act, aarg)
		if err != nil {
			t.Fatalf("%sRem failed: %v", testName, err)
		}
		sandbox, err = Info()
		if err != nil {
			t.Fatalf("Info failed: %v", err)
		}
		idx = findCidr(sandbox.CidrRules, rule.Pat)
		if idx != -1 {
			t.Errorf("Expected %s rule to be absent, got index %d", testName, idx)
		}
	}
}

func Test_06_Cidr_Port_Double(t *testing.T) {
	host := "127.3.1.4/8"
	port := [2]int{1024, 65535}
	addr := host + "!" + fmt.Sprint(port[0]) + "-" + fmt.Sprint(port[1])
	aarg := string(addr)

	testCases := []struct {
		AddFunc     func(Action, string) error
		DelFunc     func(Action, string) error
		RemFunc     func(Action, string) error
		Act         Action
		ActStr, Cap string
	}{
		{NetBindAdd, NetBindDel, NetBindRem, ActionAllow, "allow", "net/bind"},
		{NetBindAdd, NetBindDel, NetBindRem, ActionWarn, "warn", "net/bind"},
		{NetBindAdd, NetBindDel, NetBindRem, ActionFilter, "filter", "net/bind"},
		{NetBindAdd, NetBindDel, NetBindRem, ActionDeny, "deny", "net/bind"},
		{NetBindAdd, NetBindDel, NetBindRem, ActionStop, "stop", "net/bind"},
		{NetBindAdd, NetBindDel, NetBindRem, ActionKill, "kill", "net/bind"},
		{NetBindAdd, NetBindDel, NetBindRem, ActionExit, "exit", "net/bind"},

		{NetConnectAdd, NetConnectDel, NetConnectRem, ActionAllow, "allow", "net/connect"},
		{NetConnectAdd, NetConnectDel, NetConnectRem, ActionWarn, "warn", "net/connect"},
		{NetConnectAdd, NetConnectDel, NetConnectRem, ActionFilter, "filter", "net/connect"},
		{NetConnectAdd, NetConnectDel, NetConnectRem, ActionDeny, "deny", "net/connect"},
		{NetConnectAdd, NetConnectDel, NetConnectRem, ActionStop, "stop", "net/connect"},
		{NetConnectAdd, NetConnectDel, NetConnectRem, ActionKill, "kill", "net/connect"},
		{NetConnectAdd, NetConnectDel, NetConnectRem, ActionExit, "exit", "net/connect"},
	}

	for _, tc := range testCases {
		testName := tc.ActStr + tc.Cap
		rule := CidrRule{
			Act: tc.ActStr,
			Cap: tc.Cap,
			Pat: Pattern{Addr: host, Port: port},
		}

		// Test Add
		err := tc.AddFunc(tc.Act, aarg)
		if err != nil {
			t.Fatalf("%sAdd failed: %v", testName, err)
		}
		sandbox, err := Info()
		if err != nil {
			t.Fatalf("Info failed: %v", err)
		}
		idx := findCidr(sandbox.CidrRules, rule.Pat)
		if idx != len(sandbox.CidrRules)-1 {
			t.Errorf("Expected %s rule to be last, got index %d. CIDR Rules: %+v", testName, idx, sandbox.CidrRules)
		}

		// Test Del
		err = tc.DelFunc(tc.Act, aarg)
		if err != nil {
			t.Fatalf("%sDel failed: %v", testName, err)
		}
		sandbox, err = Info()
		if err != nil {
			t.Fatalf("Info failed: %v", err)
		}
		idx = findCidr(sandbox.CidrRules, rule.Pat)
		if idx != -1 {
			t.Errorf("Expected %s rule to be absent, got index %d", testName, idx)
		}

		// Test Add, Add, Add, Rem
		err = tc.AddFunc(tc.Act, aarg)
		if err != nil {
			t.Fatalf("%sAdd failed: %v", testName, err)
		}
		err = tc.AddFunc(tc.Act, aarg)
		if err != nil {
			t.Fatalf("%sAdd failed: %v", testName, err)
		}
		err = tc.AddFunc(tc.Act, aarg)
		if err != nil {
			t.Fatalf("%sAdd failed: %v", testName, err)
		}
		err = tc.RemFunc(tc.Act, aarg)
		if err != nil {
			t.Fatalf("%sRem failed: %v", testName, err)
		}
		sandbox, err = Info()
		if err != nil {
			t.Fatalf("Info failed: %v", err)
		}
		idx = findCidr(sandbox.CidrRules, rule.Pat)
		if idx != -1 {
			t.Errorf("Expected %s rule to be absent, got index %d", testName, idx)
		}
	}
}

func Test_07_Force(t *testing.T) {
	path := "/tmp/gosyd"
	hash := strings.Repeat("0", 128)
	rule := ForceRule{Act: "kill", Sha: hash, Pat: path}

	// Assert ForceAdd
	err := ForceAdd(path, hash, ActionKill)
	if err != nil {
		t.Fatalf("ForceAdd failed: %v", err)
	}

	sandbox, err := Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}

	if !findForce(sandbox.ForceRules, rule) {
		t.Errorf("Expected rule to be present")
	}

	// Assert ForceDel
	err = ForceDel(path)
	if err != nil {
		t.Fatalf("ForceDel failed: %v", err)
	}

	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}

	if findForce(sandbox.ForceRules, rule) {
		t.Errorf("Expected rule to be absent")
	}

	// Assert ForceClr
	path_1 := "/tmp/gosyd_1"
	path_2 := "/tmp/gosyd_2"
	err = ForceAdd(path_1, hash, ActionWarn)
	if err != nil {
		t.Fatalf("ForceAdd failed: %v", err)
	}
	err = ForceAdd(path_2, hash, ActionKill)
	if err != nil {
		t.Fatalf("ForceAdd failed: %v", err)
	}
	err = ForceClr()
	if err != nil {
		t.Fatalf("ForceClr failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	num_rules := len(sandbox.ForceRules)
	if num_rules != 0 {
		t.Errorf("Expected empty list, got %d elements!", num_rules)
	}
}

func Test_08_SegvGuard(t *testing.T) {
	sandbox, err := Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	segvGuardExpiryOrig := sandbox.SegvGuardExpiry
	segvGuardSuspensionOrig := sandbox.SegvGuardSuspension
	segvGuardMaxCrashesOrig := sandbox.SegvGuardMaxCrashes

	// Test setting SegvGuardExpiry
	if err := SegvGuardExpiry(4096); err != nil {
		t.Fatalf("SegvGuardExpiry(4096) failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.SegvGuardExpiry != 4096 {
		t.Errorf("Expected SegvGuardExpiry to be 4096, got %d", sandbox.SegvGuardExpiry)
	}
	SegvGuardExpiry(segvGuardExpiryOrig) // Resetting to original

	// Test setting SegvGuardSuspension
	if err := SegvGuardSuspension(4096); err != nil {
		t.Fatalf("SegvGuardSuspension(4096) failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.SegvGuardSuspension != 4096 {
		t.Errorf("Expected SegvGuardSuspension to be 4096, got %d", sandbox.SegvGuardSuspension)
	}
	SegvGuardSuspension(segvGuardSuspensionOrig) // Resetting to original

	// Test setting SegvGuardMaxCrashes
	if err := SegvGuardMaxCrashes(42); err != nil {
		t.Fatalf("SegvGuardMaxCrashes(42) failed: %v", err)
	}
	sandbox, err = Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}
	if sandbox.SegvGuardMaxCrashes != 42 {
		t.Errorf("Expected SegvGuardMaxCrashes to be 42, got %d", sandbox.SegvGuardMaxCrashes)
	}
	SegvGuardMaxCrashes(segvGuardMaxCrashesOrig) // Resetting to original
}

func Test_09_Exec(t *testing.T) {
	// Create a temporary directory
	tempDir, err := ioutil.TempDir("", "syd_test")
	if err != nil {
		t.Fatalf("Failed to create temporary directory: %v", err)
	}
	defer os.RemoveAll(tempDir) // Clean up

	// Path to the temporary file
	tempFile := filepath.Join(tempDir, "file")

	// Prepare command and arguments
	file := "/bin/sh"
	argv := []string{"-c", "echo 42 > " + tempFile}

	// Call Exec
	err = Exec(file, argv)
	if err != nil {
		t.Fatalf("Exec failed: %v", err)
	}

	// Wait for the command to execute
	time.Sleep(3 * time.Second)

	// Assert the contents of the file
	contents, err := ioutil.ReadFile(tempFile)
	if err != nil {
		t.Fatalf("Failed to read from temporary file: %v", err)
	}

	if strings.TrimSpace(string(contents)) != "42" {
		t.Errorf("Expected file contents to be '42', got '%s'", contents)
	}
}

func Test_10_Load(t *testing.T) {
	// Create a temporary file
	tempFile, err := ioutil.TempFile("", "syd_test")
	if err != nil {
		t.Fatalf("Failed to create temporary file: %v", err)
	}
	defer os.Remove(tempFile.Name()) // Clean up

	// Write test data to the temporary file
	_, err = tempFile.WriteString("pid/max:77\n")
	if err != nil {
		t.Fatalf("Failed to write to temporary file: %v", err)
	}

	// Seek back to the beginning of the file
	if _, err := tempFile.Seek(0, 0); err != nil {
		t.Fatalf("Failed to seek to beginning of the file: %v", err)
	}

	// Load the configuration from the temporary file
	if err := Load(int(tempFile.Fd())); err != nil {
		t.Fatalf("Load failed: %v", err)
	}

	// Retrieve the information using Info
	sandbox, err := Info()
	if err != nil {
		t.Fatalf("Info failed: %v", err)
	}

	// Assert the pid_max value
	expectedPidMax := 77
	if sandbox.PidMax != expectedPidMax {
		t.Errorf("Expected pid_max to be %d, got %d", expectedPidMax, sandbox.PidMax)
	}
}

func Test_11_Lock(t *testing.T) {
	if err := Lock(LockOff); err != nil {
		errno := err.(syscall.Errno)
		t.Errorf("Lock(LockOff): %v", errno)
	}
	if err := Lock(LockExec); err != nil {
		errno := err.(syscall.Errno)
		t.Errorf("Lock(LockExec): %v", errno)
	}

	if err := Lock(LockOff); err != nil {
		errno := err.(syscall.Errno)
		if errno != syscall.ENOENT {
			t.Errorf("Lock(LockOff): %v", errno)
		}
	} else {
		t.Errorf("Lock(LockOff): 0")
	}

	if err := Lock(LockExec); err != nil {
		errno := err.(syscall.Errno)
		if errno != syscall.ENOENT {
			t.Errorf("Lock(LockExec): %v", errno)
		}
	} else {
		t.Errorf("Lock(LockExec): 0")
	}

	if err := Lock(LockOn); err != nil {
		errno := err.(syscall.Errno)
		if errno != syscall.ENOENT {
			t.Errorf("Lock(LockOn): %v", errno)
		}
	} else {
		t.Errorf("Lock(LockOn): 0")
	}
}

// findForce searches for a rule in the list of ForceRules and returns true if it's found.
func findForce(rules []ForceRule, rule ForceRule) bool {
	for _, r := range rules {
		if reflect.DeepEqual(r, rule) {
			return true
		}
	}
	return false
}

// findFilter searches for a rule in the reversed list of FilterRules and returns its index.
func findFilter(rules []FilterRule, rule FilterRule) int {
	for idx, r := range rules {
		if reflect.DeepEqual(r, rule) {
			return idx
		}
	}
	return -1
}

// findGlob searches for a rule in the reversed list of GlobRules and returns its index.
func findGlob(rules []GlobRule, rule GlobRule) int {
	for idx, r := range rules {
		if reflect.DeepEqual(r, rule) {
			return idx
		}
	}
	return -1
}

func findCidr(rules []CidrRule, pattern Pattern) int {
	for idx, rule := range rules {
		if rule.Pat.Addr == pattern.Addr {
			fmt.Printf("rule: '%+v' == pat: '%+v'\n", rule.Pat.Addr, pattern.Addr)
		}
		if rule.Pat.Addr == pattern.Addr && comparePorts(rule.Pat.Port, pattern.Port) {
			return idx
		} else {
			fmt.Printf("port: '%+v' != pat: '%+v'\n", rule.Pat.Port, pattern.Port)
		}
	}
	return -1
}

func comparePorts(port1, port2 interface{}) bool {
	convertFloatSliceToIntSlice := func(floatSlice []interface{}) []int {
		intSlice := make([]int, len(floatSlice))
		for i, v := range floatSlice {
			if fv, ok := v.(float64); ok {
				intSlice[i] = int(fv)
			} else {
				fmt.Printf("Element in slice is not a float64: %v\n", v)
				return nil
			}
		}
		return intSlice
	}

	switch p1 := port1.(type) {
	case []interface{}:
		convertedP1 := convertFloatSliceToIntSlice(p1)
		if convertedP1 == nil {
			fmt.Printf("Failed to convert []interface{} to []int for Port1\n")
			return false
		}
		return comparePorts(convertedP1, port2)
	case float64:
		return comparePorts(int(p1), port2)
	case int:
		switch p2 := port2.(type) {
		case float64:
			return p1 == int(p2)
		case int:
			return p1 == p2
		case []int:
			return len(p2) == 2 && p1 >= p2[0] && p1 <= p2[1]
		case [2]int:
			return p1 >= p2[0] && p1 <= p2[1]
		case []interface{}:
			convertedP2 := convertFloatSliceToIntSlice(p2)
			if convertedP2 == nil {
				fmt.Printf("Failed to convert []interface{} to []int for Port2\n")
				return false
			}
			return comparePorts(p1, convertedP2)
		default:
			fmt.Printf("Pattern Port2 is of unexpected type %T\n", port2)
		}
	case []int:
		switch p2 := port2.(type) {
		case float64:
			return len(p1) == 1 && p1[0] == int(p2)
		case int:
			return len(p1) == 2 && p2 >= p1[0] && p2 <= p1[1]
		case []int:
			return reflect.DeepEqual(p1, p2)
		case [2]int:
			return reflect.DeepEqual(p1, p2[:])
		case []interface{}:
			convertedP2 := convertFloatSliceToIntSlice(p2)
			if convertedP2 == nil {
				fmt.Printf("Failed to convert []interface{} to []int for Port2\n")
				return false
			}
			return reflect.DeepEqual(p1, convertedP2)
		default:
			fmt.Printf("Pattern Port2 is of unexpected type %T\n", port2)
		}
	case [2]int:
		switch p2 := port2.(type) {
		case float64:
			return len(p1) == 1 && p1[0] == int(p2)
		case int:
			return p2 >= p1[0] && p2 <= p1[1]
		case []int:
			return reflect.DeepEqual(p1[:], p2)
		case [2]int:
			return reflect.DeepEqual(p1, p2)
		case []interface{}:
			convertedP2 := convertFloatSliceToIntSlice(p2)
			if convertedP2 == nil {
				fmt.Printf("Failed to convert []interface{} to []int for Port2\n")
				return false
			}
			return reflect.DeepEqual(p1[:], convertedP2)
		default:
			fmt.Printf("Pattern Port2 is of unexpected type %T\n", port2)
		}
	default:
		fmt.Printf("Rule Port1 is of unexpected type %T\n", port1)
	}
	return false
}
