/*
  mecab-skkserv -- MeCab-based SKK server

  $Id: mecab-skkserv.cpp,v 1.3 2006/04/15 17:18:25 taku Exp $;

  Copyright (C) 2005  Taku Kudo <taku@chasen.org>
  All rights reserved.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later verjsion.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.
*/

#include "config.h"
#include <mecab.h>
#include <string.h>
#include <stdlib.h>
#include <iostream>
#include <vector>
#include <string>
#include <cstring>
#include <stdexcept>
#include <set>

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#if defined HAVE_GETOPT_H && defined HAVE_GETOPT_LONG
#include <getopt.h>
#else
#include "getopt.h"
#endif

#define STDIN (fileno(stdin))
#define BUFSIZE 8192
#define DEFAULT_CAND_SIZE 20
#define DEFAULT_SEARCH_SIZE 100

#define USAGE argv[0] << " [-d dict_dir] [-c candidate size] [-s search size]"

using namespace std;

int main (int argc, char **argv)
{
  try {

    char *arg [8] = {"mecab", "-Oime",
		     "-l1", "-d", NULL, NULL};

    extern char *optarg;
    int opt; 
    std::string dict_dir     = DEFAULT_DICT_DIR;
    unsigned int cand_size   = DEFAULT_CAND_SIZE;
    unsigned int search_size = DEFAULT_SEARCH_SIZE;

    while ((opt = getopt(argc,argv,"d:s:c:hv")) != -1) {
      switch(opt) {
      case 'd': dict_dir = std::string (optarg); break;
      case 's': search_size = atoi (optarg); break;
      case 'c': cand_size   = atoi (optarg); break;
      case 'h': std::cout << USAGE << std::endl; return -1;
      case 'v': std::cout << PACKAGE << " of " << VERSION << std::endl;  return -1;
      default:
	std::cout << USAGE << std::endl; return -1;
      }
    }

    if (cand_size > 100)   throw std::runtime_error ("candidate size is too large, exit");
    if (cand_size <= 0)    throw std::runtime_error ("candidate size is too small, exit");
    if (search_size <= 0)  throw std::runtime_error ("search size is too small, exit");
    if (search_size > 500) throw std::runtime_error ("search size is too large, exit");
    
    arg[4] = (char *)dict_dir.c_str();

    MeCab::Tagger *tagger = MeCab::createTagger(5, arg);
    if (!tagger) {
       std::cerr << MeCab::getTaggerError();
       return -1;
    }

    char buf[BUFSIZE];
     
    while (true) {

      int length = read (STDIN, &buf[0], BUFSIZE-1);
      if (length < 0) continue;
      if (length == 0) break;

      switch (buf[0]) {

      case '0':
	return 0;
	break;
	 
      case  '1':
	{
	  char *input = buf + 1;
	  char *ptr;
	  for (ptr = input; *ptr != ' ' && ptr != buf + length - 1; ++ptr) {};
	   
	  if (ptr == input) {
	    std::cout << '4' << std::endl;
	    continue;
	  }
	   
	  *ptr = '\0';
	  tagger->parseNBestInit (input);
	  std::set <std::string> candhash;
	  std::vector <std::string> cand;
   
	  for (unsigned int n = 0; n < search_size; ++n) {
	    char *result = (char *)tagger->next ();
	    if (!result) break;
	    unsigned int len = strlen (result);
	    result[len-1] = '\0';
	    if (candhash.find (result) == candhash.end()) {
	      cand.push_back (result);
	      candhash.insert (result);
	      if (cand.size () >= cand_size) break;
	    }
	  }
   
	  if (cand.size() == 0) {
	    std::cout << '4' << input << std::endl;
	  } else {
	    std::cout << '1';
	    for (std::vector<std::string>::iterator it = cand.begin(); it != cand.end(); ++it)
	      std::cout << '/' << *it; 
	    std::cout << '/' << std::endl;
	  }
	}
       
	break;
	 
      case '2':
	std::cout << PACKAGE << '-' << VERSION << std::flush;
	break;
       
      case '3':
	std::cout << "localhost:127.0.0.1: " << std::flush;
	break;
      }
    }

    delete tagger;
  }

  catch (std::exception &e) {
    std::cerr << e.what () << std::endl;;
    return -1;
  }
   
  return 0;
}
