/*
process_wave.c

Copyright (C) Jan Panteltje  2004 and later

Font reading etc from Linux mplayer

subtitler-yuv is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.
  
subtitler-yuv is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
  
You should have received a copy of the GNU General Public License
along with GNU Make; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
*/

#include "process_wave.h"

static int line_number;

char *strsave(char *s) /*save char array s somewhere*/
{
char *p;
 
p = malloc( strlen(s) +  1);
if(p)
	{
	strcpy(p, s);

	return p;
	}

return 0;
} /* end function strsave */


int readline_ppml(FILE *file, char *contents)
{
int a, c, i;
int status;

if(debug_flag)
	{
	fprintf(stderr, "readline_ppml(): arg file=%lu\n", (long)file);
	}

/* a back-slash before a \n signals a continuation of the current line */
#define HAVE_BACKSLASH		1

status = 0;
i = 0;
while(1)
	{
	if(i > READSIZE - 1) break;

	while(1)
		{
		c = getc(file);
		a = ferror(file);
		if(a)
			{
			perror("readline():");
			continue;
			}
		break;
		}
	if(feof(file) )
		{
		fclose(file);
		contents[i] = 0;/* EOF marker */

		line_number++;
		return(EOF);
		}

	if(c == '\\') status = HAVE_BACKSLASH;

	else if(c == '\n')
		{
		line_number++;
		if(status == HAVE_BACKSLASH)
			{
			/* continue line */
			status = 0;

			/* scip the back-slash AND the \n */
			if(i > 0) i--;
			continue;
			}
		else
			{
			contents[i] = 0; /* string termination */
			
			return 1; /* end of line */
			}
		} 
	else
        {
        status = 0;
        }

	contents[i] = c;
	i++;
	} /* end for all chars */
/*
mmm since we are here, the line must be quite long, possibly something
is wrong.
Since I do not always check for a return 0 in the use uf this function,
just to be safe, gona force a string termination.
This prevents stack overflow, and variables getting overwritten.
*/
contents[i] = 0;/* force string termination */

line_number++;

if(debug_flag)
	{
	fprintf(stderr, \
	"readline_ppml(): line %d to long, returning 0 contents=%s\n",\
	line_number, contents);
	}

return 0;
}/* end function readline_ppml */


int print_usage()
{
fprintf(stderr,\
"Usage:\n\
process_wave [-d] [-h] [-i input.wave] [-o output.wav] -p filename.ppml\n\n\
\t-d  print debug messages (o.a. function calls and arguments).\n\
\t-f  frames per second (default 25.0).\n\
\t-h  print command line options (this help), and exit.\n\
\t-i  input audio wave filename (default stdin).\n\
\t-o  output audio filename (default stdout).\n\
\t-p  filename.ppml the file with the subtitles and commands.\n\
");

return 1;
} /* end function print_usage */



int main(int argc, char **argv)
{
int a;
char *input_filename = "/dev/fd/0";
char *output_filename = "/dev/fd/1";
char *ppml_filename = 0;
FILE *pinfile, *poutfile, *pppmlfile;
double frames_per_second;
double frame_duration;
char temp[READSIZE];
int frame_number = 0;
int skip;
char *ptr = 0;
wave_header *in_header = 0;
size_t data_bytes_read;
int header_size;
int byte_p_sec;
int byte_p_spl;
int sample_fq;
int bit_p_spl;
size_t input_data_length, output_data_length, input_file_length, ta, tread;
int previous_frame;
char *audio_buffer;
size_t audio_buffer_size;
int frames;
double audio_samples_per_frame;
int old_skip_state;
int channels = 0;

/* ID */
fprintf(stderr, "\nprocess_wave-%s copyright Jan Panteltje 2004 and later.\n\n", VERSION);

/* defaults */
input_filename = strsave("/dev/fd/0");
if(! input_filename)
	{
	fprintf(stderr, "process_wave: could not allocate space for input_filename, aborting.\n");

	exit(1);
	}

output_filename = strsave("/dev/fd/1");
if(! output_filename)
	{
	fprintf(stderr, "process_wave: could not allocate space for output_filename, aborting.\n");

	exit(1);
	}

frames_per_second = 25.0;
skip = 0;
/* end defaults */

/* get command line options, these override the defaults */
while(1)
	{
	a = getopt(argc, argv, "df:hi:o:p:");
	if(a == -1) break;
	switch(a)
		{
        case 'd':
			debug_flag = 1;
			break;
		case 'f':
			frames_per_second = atof(optarg);
			if(frames_per_second < 0.0)
				{
				fprintf(stderr, "process_wave: frames_per_second must be > 0, aborting.\n"); 	

				exit(1);
				}
			break;
		case 'h':
			print_usage();
			exit(0);	
			break;
		case 'o':
			free(output_filename);
			output_filename = strsave(optarg);
			if(! output_filename)
				{
				fprintf(stderr, "process_wave: could not allocate space for output_filename, aborting.\n");

				exit(1);
				}
			break;
		case 'p':
			ppml_filename = strsave(optarg);
			if(! ppml_filename)
				{
				fprintf(stderr, "process_wave: could not allocate space for ppml_filename ,aborting.\n");
				exit(1);
				}			
			break;
		case 'i':
			free(input_filename);
			input_filename = strsave(optarg);
			if(! input_filename)
				{
				fprintf(stderr, "process_wave: could not allocate space for input_filename, aborting.\n");

				exit(1);
				}
			break;
		default:
			print_usage();
			break;
		}/* end switch a */
	}/* end while getopt() */

if(! ppml_filename)
	{
	fprintf(stderr, "process_wave: No ppml file specified (-p option) use -h for help, aborting.\n");

	exit(1);
	}

pppmlfile = fopen(ppml_filename, "r");
if(! pppmlfile)
	{
	fprintf(stderr, "process_wave: Could not open ppml file %s for read, aborting.\n", ppml_filename);

	exit(1);
	}

pinfile = fopen(input_filename, "r");
if(! pinfile)
	{
	fprintf(stderr, "process_wave: Could not open input file %s for read, aborting.\n", input_filename);

	exit(1);
	}

poutfile = fopen(output_filename, "w");
if(! poutfile)
	{
	fprintf(stderr, "process_wave: Could not open output file %s for read, aborting.\n", output_filename);

	exit(1);
	}

header_size = sizeof(wave_header);

data_bytes_read = 0;

in_header= malloc(header_size);
if(! in_header)
	{
	fprintf(stderr, "process_wave: Could not allocate space for in_header, aborting.\n");

	exit(1);
	}

a = fread(in_header, sizeof(char), header_size, pinfile);
if(a < header_size)
	{
	fprintf(stderr, "process_wave: Could not read wave header for file %s, aborting.\n", input_filename);

	exit(1);
	}

if(debug_flag)
	{
	fprintf(stderr,\
	"process_wave: header_size=%d data_bytes_read=%lu\n", header_size, (unsigned long)data_bytes_read);
	}

input_file_length = in_header -> length;
byte_p_sec = in_header -> byte_p_sec;
byte_p_spl = in_header -> byte_p_spl;
sample_fq = in_header -> sample_fq;
bit_p_spl = in_header -> bit_p_spl;
channels = in_header -> modus;
input_data_length = in_header -> data_length;

if(debug_flag)
	{
	fprintf(stderr, "process_wave: input file length=%lu\n", (unsigned long)input_file_length);
	fprintf(stderr, "process_wave: input data length=%lu\n", (unsigned long)input_data_length);
	fprintf(stderr, "process_wave: channels=%d\n", channels);
	fprintf(stderr, "process_wave: sample frequency=%d\n", sample_fq);
	fprintf(stderr, "process_wave: bytes per sample=%d\n", byte_p_spl);
	fprintf(stderr, "process_wave: bits per sample=%d\n", bit_p_spl);
	fprintf(stderr, "process_wave: bytes per second=%d\n", byte_p_sec);
	fprintf(stderr, "process_wave: frames per second=%.6f\n", frames_per_second);
	}

/* report */
fprintf(stderr, "process_wave: frames per second=%.6f\n", frames_per_second);
fprintf(stderr, "process_wave: input data length=%lu\n", (unsigned long)input_data_length);
fprintf(stderr, "process_wave: channels=%d\n", channels);
fprintf(stderr, "process_wave: sample frequency=%d\n", sample_fq);
fprintf(stderr, "process_wave: bits per sample=%d\n", bit_p_spl);

/*
calculate audio samples per video frame 
For PAL this is 1764
If this number is no integer (NTSC) we will have to make it one, and will randomly drift out of sync...
*/
frame_duration = 1.0 / frames_per_second;
audio_samples_per_frame = frame_duration * sample_fq;

if(debug_flag)
	{
	fprintf(stderr, "process_wave: frame_duration=%.3f\n", frame_duration);
	fprintf(stderr, "process_wave: audio_samples_per_frame=%.3f\n", audio_samples_per_frame);
	}

/* copy input file header to output file */
ta = fwrite(in_header, sizeof(char), header_size, poutfile);
if(ta != header_size)
    {   
    fprintf(stderr,\
    "process_wave: could only write %lu of %d bytes of out header, aborting.\n",\
	(unsigned long)ta, header_size);   

    exit(1); 
    }

/* report */
fprintf(stderr, "process_wave: processing ppml file.\n");

old_skip_state = 0;
output_data_length = 0;
previous_frame = 0;
line_number = 0;
while(1)
	{
	a = readline_ppml(pppmlfile, temp);
	if(a == EOF) break;

	/* skip emty lines */
	if(temp[0] == 0) continue;

	/* skip lines starting with ';' these are comments */
	if(temp[0] == ';') continue;

	/* skip object definitions */
	if(temp[0] == '*') continue;

	/* line now must start with a frame number */
	sscanf(temp, "%d", 	&frame_number);

	/* transcode starts at frame 1, but this is for subtitler-yuv */
/*	if(frame_number == 0) continue;*/

	/* point to separating space */	
	ptr = temp;
	while(*ptr)
		{
		if(*ptr == ' ')
			{
			/* point past space to arguments */
			ptr++;
			
			if(debug_flag)
				{
				fprintf(stderr, "process_wave: Found space, ptr=%s\n", ptr);
				}
				
			break;
			}

		ptr++;
		}

	/* skip if no arguments */	
	if(! ptr) continue;

	/* scan arguments */
	while(*ptr)
		{
		/* look for 'skip=' */		
		if(strncmp(ptr, "skip=1", 6) == 0)
			{
			skip = 1;

			break;
			}
		if(strncmp(ptr, "skip=0", 6) == 0)
			{
			skip = 0;

			break;
			}
	
		ptr++;
		} /* end if arguments */

	if(debug_flag)
		{
		fprintf(stderr,\
		"process_wave: previous_frame=%d frame_number=%d skip=%d\n",\
		previous_frame, frame_number, skip);
		}

	if(skip != old_skip_state)
		{
		fprintf(stderr, "process_wave: frame=%d change: skip=%d\n", frame_number, skip);

		old_skip_state = skip;
		}

	/* copy audio from previous_frame to this frame_number */

	/* calculate frames */
	frames = frame_number - previous_frame;

	/* calculate buffer size */
	audio_buffer_size = (int) (audio_samples_per_frame * frames * byte_p_spl * channels);

	if(debug_flag)
		{
		fprintf(stderr,\
		"process_wave: frames=%d byte_p_spl=%d audio_samples_per_frame=%.2f audio_buffer_size=%lu\n",\
		frames, byte_p_spl, audio_samples_per_frame, (unsigned long)audio_buffer_size);
		fprintf(stderr, "process_wave: audio_buffer_size=%lu\n", (unsigned long)audio_buffer_size);
		}	

	if(frames != 0)
		{
		/* if not enough bytes in input audio file, use a smaller buffer */
		ta = input_data_length - data_bytes_read;

		/* test for all read */
		if(ta == 0)
			{
			skip = 1;
			
			break;
			}

		/* test for last bit */
		if(ta < audio_buffer_size)
			{
			audio_buffer_size = ta;

//			if(debug_flag)
				{
				fprintf(stderr,\
				"process_wave: ta=%lu new audio_buffer_size=%lu\n",\
				(unsigned long)ta, (unsigned long)audio_buffer_size);
				}
			}
		
		/* allocate buffer */
		audio_buffer = malloc(audio_buffer_size);
		if(! audio_buffer)
			{
			fprintf(stderr,\
			"process_wave: could not allocate space %d bytes for audio buffer, aborting.\n",\
			audio_buffer_size);
	
			exit(1);
			}
	
		/* read data */
		tread = fread(audio_buffer, sizeof(char), audio_buffer_size, pinfile);
		if( feof(pinfile) )
			{
			fprintf(stderr, "process_wave: frame_number=%d EOF detected in input audio file\n", frame_number);

			break;
			}

		if(tread != audio_buffer_size)
			{
			perror("fread audiobuffer");

			fprintf(stderr,\
			"process_wave: fread could only read %lu of %lu bytes\n",\
			(unsigned long)tread, (unsigned long)audio_buffer_size); 

//			exit(1);
			}

		data_bytes_read += tread;

		if(debug_flag)
			{
			fprintf(stderr, "process_wave: data_bytes_read=%lu\n", (unsigned long)data_bytes_read);		
			}

		if(skip)
			{
/*
fprintf(stderr, "process_wave: WAS SEEK\n");
			
			a = fseek(pinfile, (long)audio_buffer_size, SEEK_CUR);
			if(a == -1)
				{      
				perror("fseek failed");
 
	   			fprintf(stderr,\
				"process_wave: Could not seek %lu bytes from current position in input file, aborting.\n",\
				(unsigned long)audio_buffer_size);

				exit(1);
				}
*/

			} /* end if skip */
		else /* ! skip */
			{
			/* write data */

			ta = fwrite(audio_buffer, sizeof(char), tread, poutfile);
			if(ta != tread)
				{
				perror("fwrite audio buffer");

				fprintf(stderr,\
				"process_wave: could only write %lu of %lu bytes to output file, aborting.\n",\
				(unsigned long)ta, (unsigned long)tread);

				exit(1);
				}	

			output_data_length += ta;

			} /* end if copy */

		free(audio_buffer);
		} /* end if frames != 0 */

	previous_frame = frame_number;

	} /* end while read lines from ppml file */

if(debug_flag)
	{
	fprintf(stderr, "process_wave: %d lines read from ppml file\n", line_number);
	}

/* keep copying bytes until EOF if last state was not skip */

audio_buffer_size = input_data_length - data_bytes_read;

if(debug_flag)
	{
	fprintf(stderr,\
	"process_wave: data written after processing all entries in .ppml file=%lu\n",\
	(unsigned long)output_data_length);
	}

if( (! skip) && (! feof(pinfile) ) && (audio_buffer_size != 0) )
	{
	if(debug_flag)
		{
		fprintf(stderr,\
		"process_wave: data left after processing all entries in .ppml file=%lu\n",\
		(unsigned long)audio_buffer_size);
		}

	/* report */
	fprintf(stderr,\
	"process_wave: last frame ppml file = %d skip=0, writing remaining %lu bytes.\n",\
	frame_number, (unsigned long)audio_buffer_size);

	/* allocate buffer */

	audio_buffer = malloc(audio_buffer_size);

	if(! audio_buffer)
		{
		fprintf(stderr,\
		"process_wave: could not allocate space %lu bytes for audio buffer, aborting.\n",\
		(unsigned long)audio_buffer_size);
	
		exit(1);
		}
	
	/* read data */
	tread = fread(audio_buffer, sizeof(char), audio_buffer_size, pinfile);
	if(feof(pinfile) )
		{
		fprintf(stderr, "process_wave: EOF detected in input audio file\n");
		}

	if(tread != audio_buffer_size)
		{
		perror("fread audiobuffer");

		fprintf(stderr,\
		"process_wave: fread could only read %lu bytes of %lu.\n",\
		(unsigned long)tread, (unsigned long)audio_buffer_size); 

//		exit(1);
		}

	data_bytes_read += tread;

	if(debug_flag)
		{
		fprintf(stderr, "process_wave: data_bytes_read=%lu\n", (unsigned long)data_bytes_read);		
		}

	/* write data */
	ta = fwrite(audio_buffer, sizeof(char), tread, poutfile);
	if(ta != tread)
		{
		perror("fwrite audio buffer");

		fprintf(stderr,\
		"process_wave: could only write %lu of %lu bytes to output file, aborting.\n",\
		(unsigned long)ta, (unsigned long)tread);

		exit(1);
		}	

	output_data_length += ta;
		
	free(audio_buffer);
	} /* end if any data left */

/* report */
fprintf(stderr, "process_wave: output_data_length=%lu\n", (unsigned long)output_data_length);		

/* modify output header */

a = fseek(poutfile, 0, SEEK_SET);
if(a == -1)
	{      
    fprintf(stderr, "process_wave: Could not seek to start of output wave file, aborting.\n");

	exit(1);
	}

/* set length fields */
in_header -> length = output_data_length + header_size - 8;
in_header -> data_length = (unsigned long)output_data_length;

/* re-write mofified in_header as out wave header */
ta = fwrite(in_header, sizeof(char), header_size, poutfile);
if(ta != header_size)
	{
	fprintf(stderr,\
	"process_wave: could not write modified output wave header, could not set length fields in header, using 1 Gb.\n");
	}

free(in_header);

fclose(pinfile);
fclose(poutfile);

fprintf(stderr, "process_wave: ready.\n");

/* exit OK */
exit(0);
} /* end function main */


