//
//  Copyright (C) 2023-2025  Nick Gasson
//
//  This program is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//

#include "util.h"
#include "common.h"
#include "diag.h"
#include "hash.h"
#include "ident.h"
#include "mir/mir-node.h"
#include "mir/mir-unit.h"
#include "type.h"
#include "vlog/vlog-defs.h"
#include "vlog/vlog-node.h"
#include "vlog/vlog-number.h"
#include "vlog/vlog-phase.h"
#include "vlog/vlog-util.h"

#include <assert.h>
#include <inttypes.h>
#include <string.h>
#include <stdlib.h>

typedef struct {
   mir_type_t  type;
   unsigned    size;
} type_info_t;

typedef struct {
   mir_value_t  nets;
   mir_value_t  offset;
   mir_value_t  in_range;
   unsigned     size;
} vlog_lvalue_t;

STATIC_ASSERT(sizeof(vlog_lvalue_t) <= 16);

typedef struct {
   mir_unit_t *mu;
   ihash_t    *temps;
} vlog_gen_t;

#define PUSH_DEBUG_INFO(mu, v)                                          \
   __attribute__((cleanup(_mir_pop_debug_info), unused))                \
   const mir_saved_loc_t _old_loc =                                     \
      _mir_push_debug_info((mu), vlog_loc((v)));

static void vlog_lower_stmts(vlog_gen_t *g, vlog_node_t v);
static mir_value_t vlog_lower_rvalue(vlog_gen_t *g, vlog_node_t v);

static const type_info_t *vlog_type_info(vlog_gen_t *g, vlog_node_t v)
{
   assert(vlog_kind(v) == V_DATA_TYPE);

   type_info_t *ti = mir_get_priv(g->mu, v);
   if (ti == NULL) {
      ti = mir_malloc(g->mu, sizeof(type_info_t));
      ti->size = vlog_size(v);
      ti->type = mir_vec4_type(g->mu, ti->size, false);
   }

   return ti;
}

static mir_value_t vlog_get_temp(vlog_gen_t *g, mir_type_t type)
{
   if (g->temps == NULL)
      g->temps = ihash_new(16);
   else {
      mir_value_t exist = { .bits = (uintptr_t)ihash_get(g->temps, type.bits) };
      if (!mir_is_null(exist))
         return exist;
   }

   mir_value_t temp = mir_add_var(g->mu, type, MIR_NULL_STAMP,
                                  ident_uniq("tmp"), MIR_VAR_TEMP);

   ihash_put(g->temps, type.bits, (void *)(uintptr_t)temp.bits);
   return temp;
}

static mir_value_t vlog_lower_array_off(vlog_gen_t *g, vlog_node_t r,
                                        vlog_node_t v)
{
   mir_value_t index = vlog_lower_rvalue(g, v);

   mir_type_t index_type = mir_get_type(g->mu, index);
   if (mir_get_class(g->mu, index_type) == MIR_TYPE_VEC4) {
      // TODO: check X/Z handling
      const int size = mir_get_size(g->mu, index_type);
      mir_type_t vec2 = mir_vec2_type(g->mu, size, false);
      index = mir_build_cast(g->mu, vec2, index);
   }

   mir_type_t t_offset = mir_offset_type(g->mu);
   mir_value_t cast = mir_build_cast(g->mu, t_offset, index);

   assert(vlog_kind(r) == V_DIMENSION);

   int64_t lconst, rconst;
   vlog_bounds(r, &lconst, &rconst);

   mir_value_t left = mir_const(g->mu, t_offset, lconst);

   if (lconst < rconst)
      return mir_build_sub(g->mu, t_offset, cast, left);
   else
      return mir_build_sub(g->mu, t_offset, left, cast);
}

static vlog_lvalue_t vlog_lower_lvalue_ref(vlog_gen_t *g, vlog_node_t v)
{
   vlog_node_t decl = vlog_ref(v);
   if (vlog_kind(decl) == V_PORT_DECL)
      decl = vlog_ref(decl);

   int hops;
   mir_value_t var = mir_search_object(g->mu, decl, &hops);
   assert(!mir_is_null(var));

   assert(hops > 0);
   mir_value_t upref = mir_build_var_upref(g->mu, hops, var.id);

   const type_info_t *ti = vlog_type_info(g, vlog_type(decl));

   mir_type_t t_bool = mir_bool_type(g->mu);
   mir_type_t t_offset = mir_offset_type(g->mu);

   vlog_lvalue_t result = {
      .nets     = mir_build_load(g->mu, upref),
      .size     = ti->size,
      .offset   = mir_const(g->mu, t_offset, 0),
      .in_range = mir_const(g->mu, t_bool, 1),
   };
   return result;
}

static vlog_lvalue_t vlog_lower_lvalue(vlog_gen_t *g, vlog_node_t v)
{
   PUSH_DEBUG_INFO(g->mu, v);

   switch (vlog_kind(v)) {
   case V_REF:
      return vlog_lower_lvalue_ref(g, v);
   case V_BIT_SELECT:
      {
         vlog_node_t prefix = vlog_value(v);
         assert(vlog_kind(prefix) == V_REF);

         vlog_lvalue_t ref = vlog_lower_lvalue_ref(g, prefix);

         vlog_node_t decl = vlog_ref(prefix), dt = vlog_type(decl);

         const int nunpacked = vlog_ranges(decl);
         const int nparams = vlog_params(v);
         assert(nparams <= vlog_ranges(dt) + nunpacked);

         unsigned size = vlog_size(decl) * ref.size;

         mir_type_t t_offset = mir_offset_type(g->mu);
         mir_value_t zero = mir_const(g->mu, t_offset, 0), off = zero;
         mir_value_t in_range = ref.in_range;

         for (int i = 0; i < nparams; i++) {
            vlog_node_t dim;
            if (i < nunpacked)
               dim = vlog_range(decl, i);
            else
               dim = vlog_range(dt, i - nunpacked);

            const unsigned dim_size = vlog_size(dim);
            assert(size % dim_size == 0);
            size /= dim_size;

            mir_value_t this_off =
               vlog_lower_array_off(g, dim, vlog_param(v, i));

            mir_value_t count = mir_const(g->mu, t_offset, dim_size);
            mir_value_t cmp_low =
               mir_build_cmp(g->mu, MIR_CMP_GEQ, this_off, zero);
            mir_value_t cmp_high =
               mir_build_cmp(g->mu, MIR_CMP_LT, this_off, count);
            mir_value_t this_in_range = mir_build_and(g->mu, cmp_low, cmp_high);

            if (size != 1) {
               mir_value_t scale = mir_const(g->mu, t_offset, size);
               this_off = mir_build_mul(g->mu, t_offset, this_off, scale);
            }

            in_range = mir_build_and(g->mu, in_range, this_in_range);
            off = mir_build_add(g->mu, t_offset, off, this_off);
         }

         vlog_lvalue_t lvalue = {
            .nets     = ref.nets,
            .offset   = off,
            .size     = size,
            .in_range = in_range,
         };
         return lvalue;
      }
   case V_PART_SELECT:
      {
         vlog_lvalue_t prefix = vlog_lower_lvalue(g, vlog_value(v));

         vlog_node_t dt = vlog_type(vlog_ref(vlog_value(v)));
         vlog_node_t dim = vlog_range(dt, 0);

         mir_value_t off = vlog_lower_array_off(g, dim, vlog_left(v));

         mir_type_t t_offset = mir_offset_type(g->mu);

         vlog_lvalue_t lvalue = {
            .nets     = prefix.nets,
            .offset   = mir_build_add(g->mu, t_offset, off, prefix.offset),
            .size     = vlog_size(v),
            .in_range = prefix.in_range,   // XXX
         };
         return lvalue;
      }
   default:
      CANNOT_HANDLE(v);
   }
}

static void vlog_assign_variable(vlog_gen_t *g, vlog_node_t target,
                                 mir_value_t value)
{
   assert(mir_is_vector(g->mu, value));

   vlog_lvalue_t lvalue = vlog_lower_lvalue(g, target);

   mir_type_t t_offset = mir_offset_type(g->mu);
   mir_type_t t_vec = mir_vec4_type(g->mu, lvalue.size, false);

   mir_value_t resize = mir_build_cast(g->mu, t_vec, value);
   mir_value_t count = mir_const(g->mu, t_offset, lvalue.size);

   mir_block_t resume_bb = mir_add_block(g->mu);

   int64_t in_range_const;
   if (mir_get_const(g->mu, lvalue.in_range, &in_range_const)) {
      if (!in_range_const) {
         mir_comment(g->mu, "Out-of-range assignment");
         return;
      }
   }
   else {
      mir_block_t guarded_bb = mir_add_block(g->mu);
      mir_build_cond(g->mu, lvalue.in_range, guarded_bb, resume_bb);

      mir_set_cursor(g->mu, guarded_bb, MIR_APPEND);
   }

   mir_value_t dst = mir_build_array_ref(g->mu, lvalue.nets, lvalue.offset);

   mir_value_t tmp = MIR_NULL_VALUE;
   if (lvalue.size > 1) {
      mir_type_t t_elem = mir_logic_type(g->mu);
      mir_type_t t_array = mir_carray_type(g->mu, lvalue.size, t_elem);
      tmp = vlog_get_temp(g, t_array);
   }

   mir_value_t unpacked = mir_build_unpack(g->mu, resize, 0, tmp);

   mir_build_deposit_signal(g->mu, dst, count, unpacked);

   // Delay one delta cycle to see the update

   mir_value_t zero_time = mir_const(g->mu, mir_time_type(g->mu), 0);

   mir_build_wait(g->mu, resume_bb, zero_time);

   mir_set_cursor(g->mu, resume_bb, MIR_APPEND);
}

static mir_value_t vlog_lower_unary(vlog_gen_t *g, vlog_node_t v)
{
   mir_value_t input = vlog_lower_rvalue(g, vlog_value(v));
   mir_type_t type = mir_get_type(g->mu, input);

   mir_vec_op_t mop;
   switch (vlog_subkind(v)) {
   case V_UNARY_BITNEG: mop = MIR_VEC_BIT_NOT; break;
   case V_UNARY_NOT:    mop = MIR_VEC_LOG_NOT; break;
   case V_UNARY_NEG:    mop = MIR_VEC_SUB; break;
   case V_UNARY_OR:     mop = MIR_VEC_BIT_OR; break;
   default:
      CANNOT_HANDLE(v);
   }

   return mir_build_unary(g->mu, mop, type, input);
}

static mir_value_t vlog_lower_binary(vlog_gen_t *g, vlog_node_t v)
{
   mir_value_t left = vlog_lower_rvalue(g, vlog_left(v));
   mir_value_t right = vlog_lower_rvalue(g, vlog_right(v));

   assert(mir_is_vector(g->mu, left));
   assert(mir_is_vector(g->mu, right));

   mir_type_t ltype = mir_get_type(g->mu, left);
   mir_type_t rtype = mir_get_type(g->mu, right);

   const mir_class_t lclass = mir_get_class(g->mu, ltype);
   const mir_class_t rclass = mir_get_class(g->mu, rtype);

   const int lsize = mir_get_size(g->mu, ltype);
   const int rsize = mir_get_size(g->mu, rtype);

   mir_type_t type;
   if (lclass == MIR_TYPE_VEC4 || rclass == MIR_TYPE_VEC4)
      type = mir_vec4_type(g->mu, MAX(lsize, rsize), false);
   else
      type = mir_vec2_type(g->mu, MAX(lsize, rsize), false);

   mir_value_t lcast = mir_build_cast(g->mu, type, left);
   mir_value_t rcast = mir_build_cast(g->mu, type, right);

   mir_vec_op_t mop;
   switch (vlog_subkind(v)) {
   case V_BINARY_AND:      mop = MIR_VEC_BIT_AND; break;
   case V_BINARY_OR:       mop = MIR_VEC_BIT_OR; break;
   case V_BINARY_LOG_AND:  mop = MIR_VEC_LOG_AND; break;
   case V_BINARY_LOG_OR:   mop = MIR_VEC_LOG_OR; break;
   case V_BINARY_LT:       mop = MIR_VEC_LT; break;
   case V_BINARY_LEQ:      mop = MIR_VEC_LEQ; break;
   case V_BINARY_GT:       mop = MIR_VEC_GT; break;
   case V_BINARY_GEQ:      mop = MIR_VEC_GEQ; break;
   case V_BINARY_LOG_EQ:   mop = MIR_VEC_LOG_EQ; break;
   case V_BINARY_LOG_NEQ:  mop = MIR_VEC_LOG_NEQ; break;
   case V_BINARY_CASE_EQ:  mop = MIR_VEC_CASE_EQ; break;
   case V_BINARY_CASE_NEQ: mop = MIR_VEC_CASE_NEQ; break;
   case V_BINARY_PLUS:     mop = MIR_VEC_ADD; break;
   case V_BINARY_MINUS:    mop = MIR_VEC_SUB; break;
   case V_BINARY_TIMES:    mop = MIR_VEC_MUL; break;
   case V_BINARY_SHIFT_LL: mop = MIR_VEC_SLL; break;
   case V_BINARY_SHIFT_RL: mop = MIR_VEC_SRL; break;
   default:
      CANNOT_HANDLE(v);
   }

   return mir_build_binary(g->mu, mop, type, lcast, rcast);
}

static mir_value_t vlog_lower_systf_param(vlog_gen_t *g, vlog_node_t v)
{
   switch (vlog_kind(v)) {
   case V_REF:
   case V_STRING:
   case V_NUMBER:
   case V_EMPTY:
      return MIR_NULL_VALUE;
   case V_UNARY:
   case V_BINARY:
   case V_SYS_FCALL:
   case V_PREFIX:
   case V_POSTFIX:
   case V_BIT_SELECT:
      // TODO: these should not be evaluated until vpi_get_value is called
      return vlog_lower_rvalue(g, v);
   default:
      CANNOT_HANDLE(v);
   }
}

static mir_value_t vlog_lower_sys_tfcall(vlog_gen_t *g, vlog_node_t v)
{
   const int nparams = vlog_params(v);
   mir_value_t *args LOCAL =
      xmalloc_array((nparams * 2) + 1, sizeof(mir_value_t));
   int actual = 0;
   mir_type_t t_offset = mir_offset_type(g->mu);
   for (int i = 0; i < nparams; i++) {
      mir_value_t arg = vlog_lower_systf_param(g, vlog_param(v, i));
      if (mir_is_null(arg))
         continue;

      mir_type_t type = mir_get_type(g->mu, arg);
      switch (mir_get_class(g->mu, type)) {
      case MIR_TYPE_VEC4:
         args[actual++] = mir_const(g->mu, t_offset, mir_get_size(g->mu, type));
         args[actual++] = arg;
         break;
      case MIR_TYPE_VEC2:
         {
            // TODO: remove the cast
            const int size = mir_get_size(g->mu, type);
            mir_type_t t_vec4 = mir_vec4_type(g->mu, size, false);
            args[actual++] = mir_const(g->mu, t_offset, size);
            args[actual++] = mir_build_cast(g->mu, t_vec4, arg);
         }
         break;
      default:
         should_not_reach_here();
      }
   }

   mir_value_t locus = mir_build_locus(g->mu, vlog_to_object(v));

   mir_type_t type = MIR_NULL_TYPE;
   if (vlog_kind(v) == V_SYS_FCALL)
      type = mir_vec2_type(g->mu, 64, false);  // XXX: hack for $time

   return mir_build_syscall(g->mu, vlog_ident(v), type, MIR_NULL_STAMP,
                            locus, args, actual);
}

static mir_value_t vlog_lower_resolved(mir_unit_t *mu, vlog_node_t v)
{
   switch (vlog_kind(v)) {
   case V_PORT_DECL:
   case V_REF:
      return vlog_lower_resolved(mu, vlog_ref(v));
   case V_VAR_DECL:
   case V_NET_DECL:
      {
         int hops;
         mir_value_t var = mir_search_object(mu, v, &hops);
         assert(!mir_is_null(var));

         assert(hops > 0);
         mir_value_t upref = mir_build_var_upref(mu, hops, var.id);
         mir_value_t nets = mir_build_load(mu, upref);
         return mir_build_resolved(mu, nets);
      }
   default:
      CANNOT_HANDLE(v);
   }
}

static mir_value_t vlog_lower_truthy(vlog_gen_t *g, vlog_node_t v)
{
   mir_value_t test = vlog_lower_rvalue(g, v);
   assert(mir_is_vector(g->mu, test));

   mir_type_t type = mir_get_type(g->mu, test);
   mir_value_t zero = mir_const_vec(g->mu, type, 0, 0);
   return mir_build_cmp(g->mu, MIR_CMP_NEQ, test, zero);
}

static mir_value_t vlog_lower_bit_select(vlog_gen_t *g, vlog_node_t v)
{
   vlog_node_t value = vlog_value(v);
   assert(vlog_kind(value) == V_REF);

   mir_value_t data = vlog_lower_resolved(g->mu, value);

   vlog_node_t decl = vlog_ref(value), dt = vlog_type(decl);

   const int nunpacked = vlog_ranges(decl);
   const int nparams = vlog_params(v);
   assert(nparams <= vlog_ranges(dt) + nunpacked);

   unsigned size = vlog_size(decl) * vlog_size(dt);

   mir_type_t t_offset = mir_offset_type(g->mu);
   mir_value_t zero = mir_const(g->mu, t_offset, 0), off = zero;
   mir_value_t in_range = mir_const(g->mu, mir_bool_type(g->mu), 1);

   for (int i = 0; i < nparams; i++) {
      vlog_node_t dim;
      if (i < nunpacked)
         dim = vlog_range(decl, i);
      else
         dim = vlog_range(dt, i - nunpacked);

      const unsigned dim_size = vlog_size(dim);
      assert(size % dim_size == 0);
      size /= dim_size;

      mir_value_t this_off = vlog_lower_array_off(g, dim, vlog_param(v, i));

      mir_value_t count = mir_const(g->mu, t_offset, dim_size);

      mir_value_t cmp_low = mir_build_cmp(g->mu, MIR_CMP_GEQ, this_off, zero);
      mir_value_t cmp_high = mir_build_cmp(g->mu, MIR_CMP_LT, this_off, count);
      mir_value_t this_in_range = mir_build_and(g->mu, cmp_low, cmp_high);

      if (size != 1) {
         mir_value_t scale = mir_const(g->mu, t_offset, size);
         this_off = mir_build_mul(g->mu, t_offset, this_off, scale);
      }

      in_range = mir_build_and(g->mu, in_range, this_in_range);
      off = mir_build_add(g->mu, t_offset, off, this_off);
   }

   mir_type_t type = mir_vec4_type(g->mu, size, false);

   mir_block_t merge_bb = MIR_NULL_BLOCK;
   mir_value_t tmp;
   int64_t in_range_const;
   if (mir_get_const(g->mu, in_range, &in_range_const)) {
      if (!in_range_const)
         return mir_const_vec(g->mu, type, 1, 1);
   }
   else {
      // TODO: use a phi node here
      tmp = mir_add_var(g->mu, type, MIR_NULL_STAMP,
                        ident_uniq("tmp"), MIR_VAR_TEMP);
      mir_build_store(g->mu, tmp, mir_const_vec(g->mu, type, 1, 1));

      mir_block_t guarded_bb = mir_add_block(g->mu);
      merge_bb = mir_add_block(g->mu);

      mir_build_cond(g->mu, in_range, guarded_bb, merge_bb);

      mir_set_cursor(g->mu, guarded_bb, MIR_APPEND);
   }

   mir_value_t ptr = mir_build_array_ref(g->mu, data, off), packed;
   if (size == 1) {
      mir_value_t bit = mir_build_load(g->mu, ptr);
      packed = mir_build_pack(g->mu, type, bit);
   }
   else
      packed = mir_build_pack(g->mu, type, ptr);

   if (mir_is_null(merge_bb))
      return packed;
   else {
      mir_build_store(g->mu, tmp, packed);
      mir_build_jump(g->mu, merge_bb);

      mir_set_cursor(g->mu, merge_bb, MIR_APPEND);

      return mir_build_load(g->mu, tmp);
   }
}

static mir_value_t vlog_lower_rvalue(vlog_gen_t *g, vlog_node_t v)
{
   PUSH_DEBUG_INFO(g->mu, v);

   switch (vlog_kind(v)) {
   case V_REF:
      {
         vlog_node_t decl = vlog_ref(v);
         if (vlog_kind(decl) == V_LOCALPARAM)
            return vlog_lower_rvalue(g, vlog_value(decl));

         mir_value_t data = vlog_lower_resolved(g->mu, decl);

         const type_info_t *ti = vlog_type_info(g, vlog_type(decl));

         if (ti->size == 1)
            data = mir_build_load(g->mu, data);

         return mir_build_pack(g->mu, ti->type, data);
      }
   case V_EVENT:
      {
         vlog_node_t value = vlog_value(v);

         mir_type_t t_offset = mir_offset_type(g->mu);

         vlog_lvalue_t lvalue = vlog_lower_lvalue(g, value);
         mir_value_t count = mir_const(g->mu, t_offset, lvalue.size);

         // XXX: check in range
         mir_value_t nets =
            mir_build_array_ref(g->mu, lvalue.nets, lvalue.offset);

         mir_value_t event = mir_build_event_flag(g->mu, nets, count);

         const v_event_kind_t ekind = vlog_subkind(v);
         if (ekind == V_EVENT_LEVEL)
            return event;

         mir_type_t t_logic = mir_vec4_type(g->mu, 1, false);

         mir_value_t level =
            mir_const_vec(g->mu, t_logic, ekind == V_EVENT_POSEDGE, 0);
         mir_value_t rvalue = vlog_lower_rvalue(g, value);
         mir_value_t cmp = mir_build_cmp(g->mu, MIR_CMP_EQ, rvalue, level);

         return mir_build_and(g->mu, event, cmp);
      }
   case V_NUMBER:
      {

         number_t num = vlog_number(v);
         const int width = number_width(num);
         assert(width < 64);

         if (number_is_defined(num)) {
            mir_type_t t_vec2 = mir_vec2_type(g->mu, width, false);
            return mir_const_vec(g->mu, t_vec2, number_integer(num), 0);
         }
         else {
            uint64_t abits = 0, bbits = 0;
            for (int i = 0; i < width; i++) {
               vlog_logic_t bit = number_bit(num, width - i - 1);
               abits = (abits << 1) | (bit & 1);
               bbits = (bbits << 1) | ((bit >> 1) & 1);
            }

            mir_type_t t_vec4 = mir_vec4_type(g->mu, width, false);
            return mir_const_vec(g->mu, t_vec4, abits, bbits);
         }
      }
   case V_BINARY:
      return vlog_lower_binary(g, v);
   case V_UNARY:
      return vlog_lower_unary(g, v);
   case V_SYS_FCALL:
      return vlog_lower_sys_tfcall(g, v);
   case V_BIT_SELECT:
      return vlog_lower_bit_select(g, v);
   case V_PART_SELECT:
      {
         mir_value_t base = vlog_lower_resolved(g->mu, vlog_value(v));

         vlog_node_t dt = vlog_type(vlog_ref(vlog_value(v)));
         vlog_node_t dim = vlog_range(dt, 0);

         mir_value_t off = vlog_lower_array_off(g, dim, vlog_left(v));
         mir_value_t ptr = mir_build_array_ref(g->mu, base, off);

         mir_type_t t_vec = mir_vec4_type(g->mu, vlog_size(v), false);
         return mir_build_pack(g->mu, t_vec, ptr);
      }
   case V_CONCAT:
      {
         int size = 0, repeat = 1, pos = 0;
         const int nparams = vlog_params(v);
         mir_value_t *inputs LOCAL =
            xmalloc_array(nparams, sizeof(mir_value_t));

         for (int i = 0; i < nparams; i++) {
            inputs[i] = vlog_lower_rvalue(g, vlog_param(v, i));
            assert(mir_is_vector(g->mu, inputs[i]));
            size += mir_get_size(g->mu, mir_get_type(g->mu, inputs[i]));
         }

         if (vlog_has_value(v))
            size *= (repeat = MAX(0, vlog_get_const(vlog_value(v))));

         mir_type_t type = mir_vec4_type(g->mu, size, false);
         mir_value_t result = mir_const_vec(g->mu, type, 0, 0);

         for (int i = 0; i < repeat; i++) {
            for (int j = nparams - 1; j >= 0; j--) {
               // TODO: add a vector-insert operation
               mir_value_t cast = mir_build_cast(g->mu, type, inputs[j]);
               mir_value_t amount = mir_const_vec(g->mu, type, pos, 0);
               mir_value_t shift =
                  mir_build_binary(g->mu, MIR_VEC_SLL, type, cast, amount);
               result = mir_build_binary(g->mu, MIR_VEC_BIT_OR, type,
                                         result, shift);
               pos += mir_get_size(g->mu, mir_get_type(g->mu, inputs[j]));
            }
         }

         return result;
      }
   case V_PREFIX:
      {
         vlog_node_t target = vlog_target(v);

         mir_value_t prev = vlog_lower_rvalue(g, target);
         mir_type_t type = mir_get_type(g->mu, prev);
         mir_value_t one = mir_const_vec(g->mu, type, 1, 0);
         mir_value_t inc =
            mir_build_binary(g->mu, MIR_VEC_ADD, type, prev, one);

         // Must save/restore around blocking assignment
         mir_value_t tmp = mir_add_var(g->mu, type, MIR_NULL_STAMP,
                                       ident_uniq("prefix"), MIR_VAR_TEMP);
         mir_build_store(g->mu, tmp, inc);

         vlog_assign_variable(g, target, inc);

         return mir_build_load(g->mu, tmp);
      }
   case V_POSTFIX:
      {
         vlog_node_t target = vlog_target(v);

         mir_value_t prev = vlog_lower_rvalue(g, target);
         mir_type_t type = mir_get_type(g->mu, prev);
         mir_value_t one = mir_const_vec(g->mu, type, 1, 0);
         mir_value_t inc =
            mir_build_binary(g->mu, MIR_VEC_ADD, type, prev, one);

         vlog_assign_variable(g, target, inc);

         return inc;
      }
   case V_COND_EXPR:
      {
         // TODO: check semantics for return type
         // TODO: do not evaluate both sides

         mir_value_t cmp = vlog_lower_truthy(g, vlog_value(v));
         mir_value_t left = vlog_lower_rvalue(g, vlog_left(v));
         mir_value_t right = vlog_lower_rvalue(g, vlog_right(v));

         mir_type_t type = mir_get_type(g->mu, left);
         return mir_build_select(g->mu, type, cmp, left, right);
      }
   default:
      CANNOT_HANDLE(v);
   }
}

static mir_value_t vlog_lower_time(vlog_gen_t *g, vlog_node_t v)
{
   assert(vlog_kind(v) == V_NUMBER);

   mir_type_t t_time = mir_time_type(g->mu);
   number_t num = vlog_number(v);

   return mir_const(g->mu, t_time, number_integer(num));
}

static void vlog_lower_sensitivity(vlog_gen_t *g, vlog_node_t v)
{
   const vlog_kind_t kind = vlog_kind(v);
   switch (kind) {
   case V_REF:
      {
         switch (vlog_kind(vlog_ref(v))) {
         case V_PORT_DECL:
         case V_NET_DECL:
         case V_VAR_DECL:
            {
               mir_type_t t_offset = mir_offset_type(g->mu);

               vlog_lvalue_t lvalue = vlog_lower_lvalue(g, v);
               mir_value_t count = mir_const(g->mu, t_offset, lvalue.size);

               mir_build_sched_event(g->mu, lvalue.nets, count);
            }
            break;
         case V_PARAM_DECL:
         case V_LOCALPARAM:
            break;
         default:
            CANNOT_HANDLE(v);
         }
      }
      break;
   case V_BIT_SELECT:
   case V_PART_SELECT:
      {
         vlog_node_t prefix = vlog_longest_static_prefix(v);
         if (prefix == v) {
            mir_type_t t_offset = mir_offset_type(g->mu);

            vlog_lvalue_t lvalue = vlog_lower_lvalue(g, v);

            mir_value_t count = mir_const(g->mu, t_offset, lvalue.size);

            // XXX: check in range
            mir_value_t nets =
               mir_build_array_ref(g->mu, lvalue.nets, lvalue.offset);

            mir_build_sched_event(g->mu, nets, count);
         }
         else
            vlog_lower_sensitivity(g, prefix);

         if (kind == V_BIT_SELECT) {
            const int nparams = vlog_params(v);
            for (int i = 0; i < nparams; i++)
               vlog_lower_sensitivity(g, vlog_param(v, i));
         }
         else
            vlog_lower_sensitivity(g, vlog_left(v));
      }
      break;
   case V_EVENT:
      vlog_lower_sensitivity(g, vlog_value(v));
      break;
   case V_NUMBER:
      break;
   case V_BINARY:
      vlog_lower_sensitivity(g, vlog_left(v));
      vlog_lower_sensitivity(g, vlog_right(v));
      break;
   case V_UNARY:
      vlog_lower_sensitivity(g, vlog_value(v));
      break;
   case V_COND_EXPR:
      vlog_lower_sensitivity(g, vlog_value(v));
      vlog_lower_sensitivity(g, vlog_left(v));
      vlog_lower_sensitivity(g, vlog_right(v));
      break;
   case V_CONCAT:
      {
         const int nparams = vlog_params(v);
         for (int i = 0; i < nparams; i++)
            vlog_lower_sensitivity(g, vlog_param(v, i));
      }
      break;
   default:
      CANNOT_HANDLE(v);
   }
}

static void vlog_lower_timing(vlog_gen_t *g, vlog_node_t v, bool is_static)
{
   mir_block_t true_bb = mir_add_block(g->mu), false_bb = MIR_NULL_BLOCK;

   vlog_node_t ctrl = vlog_value(v);
   switch (vlog_kind(ctrl)) {
   case V_DELAY_CONTROL:
      {
         mir_type_t t_time = mir_time_type(g->mu);
         mir_value_t delay = vlog_lower_rvalue(g, vlog_value(ctrl));
         mir_value_t cast = mir_build_cast(g->mu, t_time, delay);

         mir_build_wait(g->mu, true_bb, cast);

         mir_set_cursor(g->mu, true_bb, MIR_APPEND);
      }
      break;
   case V_EVENT_CONTROL:
      {
         mir_value_t test = MIR_NULL_VALUE;
         const int nparams = vlog_params(ctrl);
         if (nparams > 0) {
            test = vlog_lower_rvalue(g, vlog_param(ctrl, 0));
            for (int i = 1; i < nparams; i++) {
               mir_value_t sub = vlog_lower_rvalue(g, vlog_param(ctrl, i));
               test = mir_build_or(g->mu, test, sub);
            }
         }

         false_bb = mir_add_block(g->mu);

         if (mir_is_null(test))
            mir_build_jump(g->mu, false_bb);
         else
            mir_build_cond(g->mu, test, true_bb, false_bb);

         mir_set_cursor(g->mu, true_bb, MIR_APPEND);
      }
      break;
   default:
      CANNOT_HANDLE(ctrl);
   }

   vlog_lower_stmts(g, v);

   if (!mir_is_null(false_bb)) {
      if (!mir_block_finished(g->mu, mir_get_cursor(g->mu, NULL)))
         mir_build_jump(g->mu, false_bb);

      mir_set_cursor(g->mu, false_bb, MIR_APPEND);
   }
}

static void vlog_lower_blocking_assignment(vlog_gen_t *g, vlog_node_t v)
{
   if (vlog_has_delay(v)) {
      vlog_node_t delay = vlog_delay(v);
      assert(vlog_kind(delay) == V_DELAY_CONTROL);

      mir_block_t delay_bb = mir_add_block(g->mu);

      mir_build_wait(g->mu, delay_bb, vlog_lower_time(g, vlog_value(delay)));

      mir_set_cursor(g->mu, delay_bb, MIR_APPEND);
   }

   mir_value_t value = vlog_lower_rvalue(g, vlog_value(v));
   vlog_assign_variable(g, vlog_target(v), value);
}

static void vlog_lower_non_blocking_assignment(vlog_gen_t *g, vlog_node_t v)
{
   vlog_node_t target = vlog_target(v);

   vlog_lvalue_t lvalue = vlog_lower_lvalue(g, target);

   // XXX: check in range
   mir_value_t nets = mir_build_array_ref(g->mu, lvalue.nets, lvalue.offset);

   mir_value_t value = vlog_lower_rvalue(g, vlog_value(v));
   assert(mir_is_vector(g->mu, value));

   mir_type_t t_offset = mir_offset_type(g->mu);
   mir_type_t t_vec = mir_vec4_type(g->mu, lvalue.size, false);

   mir_value_t resize = mir_build_cast(g->mu, t_vec, value);
   mir_value_t count = mir_const(g->mu, t_offset, lvalue.size);

   mir_value_t tmp = MIR_NULL_VALUE;
   if (lvalue.size > 1) {
      mir_type_t t_elem = mir_logic_type(g->mu);
      mir_type_t t_array = mir_carray_type(g->mu, lvalue.size, t_elem);
      tmp = vlog_get_temp(g, t_array);
   }

   const uint8_t strength = vlog_is_net(target) ? ST_STRONG : 0;
   mir_value_t unpacked = mir_build_unpack(g->mu, resize, strength, tmp);

   mir_type_t t_time = mir_time_type(g->mu);
   mir_value_t reject = mir_const(g->mu, t_time, 0);

   mir_value_t after;
   if (vlog_has_delay(v)) {
      vlog_node_t delay = vlog_delay(v);
      assert(vlog_kind(delay) == V_DELAY_CONTROL);

      after = vlog_lower_time(g, vlog_value(delay));
   }
   else
      after = mir_const(g->mu, t_time, 0);

   mir_build_sched_waveform(g->mu, nets, count, unpacked, reject, after);
}

static void vlog_lower_if(vlog_gen_t *g, vlog_node_t v)
{
   mir_block_t exit_bb = MIR_NULL_BLOCK;

   const int nconds = vlog_conds(v);
   for (int i = 0; i < nconds; i++) {
      vlog_node_t c = vlog_cond(v, i);
      mir_block_t next_bb = MIR_NULL_BLOCK;

      if (vlog_has_value(c)) {
         mir_value_t cmp = vlog_lower_truthy(g, vlog_value(c));

         mir_block_t btrue = mir_add_block(g->mu);

         if (i == nconds - 1) {
            if (mir_is_null(exit_bb))
               exit_bb = mir_add_block(g->mu);
            next_bb = exit_bb;
         }
         else
            next_bb = mir_add_block(g->mu);

         mir_build_cond(g->mu, cmp, btrue, next_bb);

         mir_set_cursor(g->mu, btrue, MIR_APPEND);
      }

      vlog_lower_stmts(g, c);

      if (!mir_block_finished(g->mu, MIR_NULL_BLOCK)) {
         if (mir_is_null(exit_bb))
            exit_bb = mir_add_block(g->mu);
         mir_build_jump(g->mu, exit_bb);
      }

      if (mir_is_null(next_bb))
         break;

      mir_set_cursor(g->mu, next_bb, MIR_APPEND);
   }

   if (!mir_is_null(exit_bb))
      mir_set_cursor(g->mu, exit_bb, MIR_APPEND);
}

static void vlog_lower_forever(vlog_gen_t *g, vlog_node_t v)
{
   mir_block_t body_bb = mir_add_block(g->mu);
   mir_build_jump(g->mu, body_bb);

   mir_set_cursor(g->mu, body_bb, MIR_APPEND);

   vlog_lower_stmts(g, v);

   mir_build_jump(g->mu, body_bb);
}

static void vlog_lower_for_loop(vlog_gen_t *g, vlog_node_t v)
{
   mir_comment(g->mu, "Begin for loop");

   vlog_node_t init = vlog_left(v);
   assert(vlog_kind(init) == V_FOR_INIT);

   assert(vlog_decls(init) == 0);   // TODO

   vlog_lower_stmts(g, init);

   mir_block_t body_bb = mir_add_block(g->mu);
   mir_block_t step_bb = mir_add_block(g->mu);
   mir_block_t test_bb = mir_add_block(g->mu);
   mir_block_t exit_bb = mir_add_block(g->mu);

   mir_build_jump(g->mu, test_bb);

   mir_set_cursor(g->mu, test_bb, MIR_APPEND);

   mir_comment(g->mu, "For loop test");

   mir_value_t test = vlog_lower_rvalue(g, vlog_value(v));
   assert(mir_is_vector(g->mu, test));

   mir_value_t zero = mir_const_vec(g->mu, mir_get_type(g->mu, test), 0, 0);
   mir_value_t cmp = mir_build_cmp(g->mu, MIR_CMP_NEQ, test, zero);
   mir_build_cond(g->mu, cmp, body_bb, exit_bb);

   mir_set_cursor(g->mu, body_bb, MIR_APPEND);

   mir_comment(g->mu, "For loop body");

   vlog_lower_stmts(g, v);

   if (!mir_block_finished(g->mu, MIR_NULL_BLOCK))
      mir_build_jump(g->mu, step_bb);

   mir_set_cursor(g->mu, step_bb, MIR_APPEND);

   mir_comment(g->mu, "For loop step");

   vlog_node_t step = vlog_right(v);
   assert(vlog_kind(step) == V_FOR_STEP);

   const int nstmts = vlog_stmts(step);
   for (int i = 0; i < nstmts; i++) {
      vlog_node_t s = vlog_stmt(step, i);
      if (vlog_kind(s) == V_BASSIGN)  // XXX: should be op assign
         vlog_lower_blocking_assignment(g, s);
      else
         vlog_lower_rvalue(g, s);
   }

   mir_build_jump(g->mu, test_bb);

   mir_set_cursor(g->mu, exit_bb, MIR_APPEND);

   mir_comment(g->mu, "End for loop");
}

static void vlog_lower_case(vlog_gen_t *g, vlog_node_t v)
{
   mir_comment(g->mu, "Begin case statement");

   mir_value_t value = vlog_lower_rvalue(g, vlog_value(v));
   mir_type_t type = mir_get_type(g->mu, value);

   // TODO: use a parallel case for small integer types

   const int nitems = vlog_stmts(v);
   mir_block_t *blocks LOCAL = xmalloc_array(nitems, sizeof(mir_block_t));

   mir_type_t t_logic = mir_vec4_type(g->mu, 1, false);
   mir_value_t zero = mir_const_vec(g->mu, t_logic, 0, 0);

   for (int i = 0; i < nitems; i++) {
      vlog_node_t item = vlog_stmt(v, i);
      assert(vlog_kind(item) == V_CASE_ITEM);

      blocks[i] = mir_add_block(g->mu);

      mir_block_t else_bb = mir_add_block(g->mu);

      mir_value_t comb = MIR_NULL_VALUE;
      const int nparams = vlog_params(item);
      for (int j = 0; j < nparams; j++) {
         mir_value_t test = vlog_lower_rvalue(g, vlog_param(item, j));
         mir_value_t cast = mir_build_cast(g->mu, type, test);
         mir_value_t case_eq =
            mir_build_binary(g->mu, MIR_VEC_CASE_EQ, type, cast, value);
         mir_value_t cmp = mir_build_cmp(g->mu, MIR_CMP_NEQ, case_eq, zero);

         if (mir_is_null(comb))
            comb = cmp;
         else
            comb = mir_build_or(g->mu, comb, cmp);
      }

      if (mir_is_null(comb))
         mir_build_jump(g->mu, blocks[i]);
      else
         mir_build_cond(g->mu, comb, blocks[i], else_bb);

      mir_set_cursor(g->mu, else_bb, MIR_APPEND);
   }

   mir_block_t exit_bb = mir_get_cursor(g->mu, NULL);

   for (int i = 0; i < nitems; i++) {
      vlog_node_t item = vlog_stmt(v, i);
      assert(vlog_kind(item) == V_CASE_ITEM);

      mir_set_cursor(g->mu, blocks[i], MIR_APPEND);
      vlog_lower_stmts(g, item);

      if (!mir_block_finished(g->mu, MIR_NULL_BLOCK))
         mir_build_jump(g->mu, exit_bb);
   }

   mir_set_cursor(g->mu, exit_bb, MIR_APPEND);

   mir_comment(g->mu, "End case statement");
}

static void vlog_lower_stmts(vlog_gen_t *g, vlog_node_t v)
{
   const int nstmts = vlog_stmts(v);
   for (int i = 0; i < nstmts; i++) {
      vlog_node_t s = vlog_stmt(v, i);
      mir_set_loc(g->mu, vlog_loc(s));

      switch (vlog_kind(s)) {
      case V_TIMING:
         vlog_lower_timing(g, s, false);
         break;
      case V_BASSIGN:
         vlog_lower_blocking_assignment(g, s);
         break;
      case V_NBASSIGN:
         vlog_lower_non_blocking_assignment(g, s);
         break;
      case V_BLOCK:
         vlog_lower_stmts(g, s);
         break;
      case V_SYS_TCALL:
         vlog_lower_sys_tfcall(g, s);
         break;
      case V_IF:
         vlog_lower_if(g, s);
         break;
      case V_FOREVER:
         vlog_lower_forever(g, s);
         break;
      case V_FOR_LOOP:
         vlog_lower_for_loop(g, s);
         break;
      case V_CASE:
         vlog_lower_case(g, s);
         break;
      default:
         CANNOT_HANDLE(s);
      }
   }
}

static void vlog_lower_driver(vlog_gen_t *g, vlog_node_t v)
{
   mir_type_t t_offset = mir_offset_type(g->mu);

   vlog_node_t prefix = vlog_longest_static_prefix(v);

   vlog_lvalue_t target = vlog_lower_lvalue(g, prefix);

   // XXX: check in range
   mir_value_t nets = mir_build_array_ref(g->mu, target.nets, target.offset);

   mir_value_t one = mir_const(g->mu, t_offset, target.size);

   mir_build_drive_signal(g->mu, nets, one);
}

static void vlog_driver_cb(vlog_node_t v, void *context)
{
   vlog_gen_t *g = context;

   switch (vlog_kind(v)) {
   case V_NBASSIGN:
   case V_ASSIGN:
      vlog_lower_driver(g, vlog_target(v));
      break;
   default:
      break;
   }
}

static void vlog_lower_always(vlog_gen_t *g, vlog_node_t v)
{
   mir_block_t start_bb = mir_add_block(g->mu);
   assert(start_bb.id == 1);

   vlog_visit(v, vlog_driver_cb, g);

   vlog_node_t timing = NULL, s0 = vlog_stmt(v, 0);
   if (vlog_kind(s0) == V_TIMING) {
      timing = s0;

      vlog_node_t ctrl = vlog_value(timing);
      assert(vlog_kind(ctrl) == V_EVENT_CONTROL);

      const int nparams = vlog_params(ctrl);
      for (int i = 0; i < nparams; i++)
         vlog_lower_sensitivity(g, vlog_param(ctrl, i));
   }

   mir_build_return(g->mu, MIR_NULL_VALUE);

   mir_set_cursor(g->mu, start_bb, MIR_APPEND);

   if (timing != NULL)
      vlog_lower_timing(g, timing, true);
   else
      vlog_lower_stmts(g, v);

   mir_build_wait(g->mu, start_bb, MIR_NULL_VALUE);
}

static void vlog_lower_initial(vlog_gen_t *g, vlog_node_t v)
{
   mir_block_t start_bb = mir_add_block(g->mu);
   assert(start_bb.id == 1);

   vlog_visit(v, vlog_driver_cb, g);

   mir_build_return(g->mu, MIR_NULL_VALUE);

   mir_set_cursor(g->mu, start_bb, MIR_APPEND);

   vlog_lower_stmts(g, v);

   if (!mir_block_finished(g->mu, MIR_NULL_BLOCK))
      mir_build_return(g->mu, MIR_NULL_VALUE);
}

static void vlog_lower_continuous_assign(vlog_gen_t *g, vlog_node_t v)
{
   mir_block_t start_bb = mir_add_block(g->mu);
   assert(start_bb.id == 1);

   vlog_visit(v, vlog_driver_cb, g);

   vlog_lower_sensitivity(g, vlog_value(v));

   mir_build_return(g->mu, MIR_NULL_VALUE);

   mir_set_cursor(g->mu, start_bb, MIR_APPEND);

   vlog_lower_non_blocking_assignment(g, v);

   mir_build_wait(g->mu, start_bb, MIR_NULL_VALUE);
}

static void vlog_lower_gate_inst(vlog_gen_t *g, vlog_node_t v)
{
   mir_block_t start_bb = mir_add_block(g->mu);
   assert(start_bb.id == 1);

   vlog_lower_driver(g, vlog_target(v));

   mir_type_t t_offset = mir_offset_type(g->mu);
   mir_type_t t_time = mir_time_type(g->mu);
   mir_type_t t_logic = mir_vec4_type(g->mu, 1, false);

   const int nparams = vlog_params(v);
   int first_term = 0;
   for (int i = 0; i < nparams; i++) {
      vlog_node_t p = vlog_param(v, i);
      if (vlog_kind(p) == V_STRENGTH)
         first_term = i + 1;
      else {
         vlog_lvalue_t lvalue = vlog_lower_lvalue(g, p);
         mir_value_t count = mir_const(g->mu, t_offset, lvalue.size);
         mir_build_sched_event(g->mu, lvalue.nets, count);
      }
   }

   mir_build_return(g->mu, MIR_NULL_VALUE);

   mir_set_cursor(g->mu, start_bb, MIR_APPEND);

   bool negate = false;
   uint8_t strength = ST_STRONG;
   mir_value_t value = MIR_NULL_VALUE;
   const vlog_gate_kind_t kind = vlog_subkind(v);
   switch (kind) {
   case V_GATE_PULLUP:
   case V_GATE_PULLDOWN:
      strength = vlog_subkind(vlog_param(v, 0));
      value = mir_const_vec(g->mu, t_logic, kind == V_GATE_PULLUP, 0);
      break;

   case V_GATE_NAND:
   case V_GATE_NOR:
   case V_GATE_XNOR:
      negate = true;
   case V_GATE_AND:
   case V_GATE_OR:
   case V_GATE_XOR:
      {
         static const mir_vec_op_t op_map[] = {
            [V_GATE_AND] = MIR_VEC_BIT_AND,
            [V_GATE_NAND] = MIR_VEC_BIT_AND,
            [V_GATE_OR] = MIR_VEC_BIT_OR,
            [V_GATE_NOR] = MIR_VEC_BIT_OR,
            [V_GATE_XOR] = MIR_VEC_BIT_XOR,
            [V_GATE_XNOR] = MIR_VEC_BIT_XOR,
         };

         value = vlog_lower_rvalue(g, vlog_param(v, first_term));

         const int nelems = nparams - first_term;
         for (int i = 1; i < nelems; i++) {
            vlog_node_t p = vlog_param(v, first_term + i);
            mir_value_t arg = vlog_lower_rvalue(g, p);
            value = mir_build_binary(g->mu, op_map[kind], t_logic, value, arg);
         }

         if (negate)
            value = mir_build_unary(g->mu, MIR_VEC_BIT_NOT, t_logic, value);
      }
      break;

   case V_GATE_NOT:
      {
         const int nparams = vlog_params(v);
         mir_value_t input = vlog_lower_rvalue(g, vlog_param(v, nparams - 1));
         value = mir_build_unary(g->mu, MIR_VEC_BIT_NOT, t_logic, input);
      }
      break;

   default:
      CANNOT_HANDLE(v);
   }

   mir_value_t unpacked =
      mir_build_unpack(g->mu, value, strength, MIR_NULL_VALUE);

   mir_value_t reject = mir_const(g->mu, t_time, 0);
   mir_value_t after = mir_const(g->mu, t_time, 0);

   vlog_lvalue_t lvalue = vlog_lower_lvalue(g, vlog_target(v));
   mir_value_t count = mir_const(g->mu, t_offset, lvalue.size);

   // XXX: check in range
   mir_value_t nets = mir_build_array_ref(g->mu, lvalue.nets, lvalue.offset);

   mir_build_sched_waveform(g->mu, nets, count, unpacked, reject, after);
   mir_build_wait(g->mu, start_bb, MIR_NULL_VALUE);
}

static void vlog_lower_cleanup(vlog_gen_t *g)
{
   if (g->temps != NULL)
      ihash_free(g->temps);
}

void vlog_lower_deferred(mir_unit_t *mu, object_t *obj)
{
   vlog_node_t v = vlog_from_object(obj);
   assert(v != NULL);

   vlog_gen_t g = {
      .mu = mu,
   };

   switch (vlog_kind(v)) {
   case V_ALWAYS:
      vlog_lower_always(&g, v);
      break;
   case V_INITIAL:
      vlog_lower_initial(&g, v);
      break;
   case V_ASSIGN:
      vlog_lower_continuous_assign(&g, v);
      break;
   case V_GATE_INST:
      vlog_lower_gate_inst(&g, v);
      break;
   default:
      CANNOT_HANDLE(v);
   }

   vlog_lower_cleanup(&g);
}

static void vlog_lower_net_decl(mir_unit_t *mu, vlog_node_t v, tree_t wrap,
                                mir_value_t resfn)
{
   mir_type_t t_net_value = mir_int_type(mu, 0, 255);
   mir_type_t t_net_signal = mir_signal_type(mu, t_net_value);
   mir_type_t t_offset = mir_offset_type(mu);

   assert(!vlog_has_value(v));   // Should have been replaced with assign

   mir_value_t value = mir_const(mu, t_net_value, LOGIC_X);
   mir_value_t count = mir_const(mu, t_offset, vlog_size(vlog_type(v)));
   mir_value_t size = mir_const(mu, t_offset, 1);
   mir_value_t flags = mir_const(mu, t_offset, 0);
   mir_value_t locus = mir_build_locus(mu, tree_to_object(wrap));

   mir_value_t signal = mir_build_init_signal(mu, t_net_value, count, size,
                                              value, flags, locus,
                                              MIR_NULL_VALUE);

   mir_build_resolve_signal(mu, signal, resfn);

   mir_value_t var = mir_add_var(mu, t_net_signal, MIR_NULL_STAMP,
                                 vlog_ident(v), MIR_VAR_SIGNAL);
   mir_build_store(mu, var, signal);

   mir_put_object(mu, v, var);
}

static void vlog_lower_var_decl(vlog_gen_t *g, vlog_node_t v, tree_t wrap)
{
   mir_type_t t_logic = mir_int_type(g->mu, 0, 3);
   mir_type_t t_logic_signal = mir_signal_type(g->mu, t_logic);
   mir_type_t t_offset = mir_offset_type(g->mu);

   const type_info_t *ti = vlog_type_info(g, vlog_type(v));
   const int total_size = ti->size * vlog_size(v);

   mir_value_t value;
   if (vlog_has_value(v)) {
      mir_value_t tmp = MIR_NULL_VALUE;
      if (ti->size > 1) {
         mir_type_t t_elem = mir_logic_type(g->mu);
         mir_type_t t_array = mir_carray_type(g->mu, total_size, t_elem);
         tmp = vlog_get_temp(g, t_array);
      }

      mir_value_t packed = vlog_lower_rvalue(g, vlog_value(v));
      mir_value_t cast = mir_build_cast(g->mu, ti->type, packed);
      value = mir_build_unpack(g->mu, cast, 0, tmp);
   }
   else
      value = mir_const(g->mu, t_logic, LOGIC_X);

   mir_value_t count = mir_const(g->mu, t_offset, total_size);
   mir_value_t size = mir_const(g->mu, t_offset, 1);
   mir_value_t flags = mir_const(g->mu, t_offset, 0);
   mir_value_t locus = mir_build_locus(g->mu, tree_to_object(wrap));

   mir_value_t signal = mir_build_init_signal(g->mu, t_logic, count, size,
                                              value, flags, locus,
                                              MIR_NULL_VALUE);

   mir_value_t var = mir_add_var(g->mu, t_logic_signal, MIR_NULL_STAMP,
                                 vlog_ident(v), MIR_VAR_SIGNAL);
   mir_build_store(g->mu, var, signal);

   mir_put_object(g->mu, v, var);
}

static mir_type_t vlog_lower_vhdl_type(mir_unit_t *mu, type_t type)
{
   if (type_eq(type, ieee_type(IEEE_STD_ULOGIC)))
      return mir_int_type(mu, 0, 8);
   else if (type_eq(type, ieee_type(IEEE_STD_ULOGIC_VECTOR)))
      return mir_uarray_type(mu, 1, mir_int_type(mu, 0, 8));
   else if (type_eq(type, verilog_type(VERILOG_NET_VALUE)))
      return mir_int_type(mu, 0, 255);
   else if (type_eq(type, verilog_type(VERILOG_LOGIC)))
      return mir_int_type(mu, 0, 3);
   else if (type_eq(type, verilog_type(VERILOG_WIRE_ARRAY))
            || type_eq(type, verilog_type(VERILOG_NET_ARRAY)))
      return mir_uarray_type(mu, 1, mir_int_type(mu, 0, 3));

   fatal_trace("cannot lower VHDL type %s", type_pp(type));
}

static void vlog_lower_converter(mir_unit_t *mu, tree_t cf, mir_value_t in,
                                 mir_value_t out)
{
   tree_t decl = tree_ref(cf);
   assert(tree_kind(decl) == T_FUNC_DECL);

   type_t rtype = type_result(tree_type(decl));

   // Dummy return value to force function calling convention
   mir_type_t t_offset = mir_offset_type(mu);
   mir_set_result(mu, t_offset);

   mir_type_t t_context = mir_context_type(mu, mir_get_parent(mu));
   mir_add_param(mu, t_context, MIR_NULL_STAMP, ident_new("context"));

   mir_type_t t_conv = mir_conversion_type(mu);
   mir_value_t conv =
      mir_add_param(mu, t_conv, MIR_NULL_STAMP, ident_new("cf"));

   mir_type_t t_out = vlog_lower_vhdl_type(mu, rtype);

   ident_t func = tree_ident2(tree_ref(cf));

   mir_value_t pkg = mir_build_link_package(mu, well_known(W_NVC_VERILOG));

   mir_value_t count = MIR_NULL_VALUE;
   if (mir_is(mu, in, MIR_TYPE_UARRAY)) {
      count = mir_build_uarray_len(mu, in, 0);
      in = mir_build_unwrap(mu, in);
   }
   else if (type_is_array(rtype)) {
      int64_t length;
      if (!folded_length(range_of(tree_type(tree_value(cf)), 0), &length))
         should_not_reach_here();

      count = mir_const(mu, t_offset, length);
   }

   mir_value_t resolved = mir_build_resolved(mu, in);

   mir_value_t arg;
   if (!mir_is_null(count)) {
      mir_dim_t dims[] = {
         { .left  = mir_const(mu, t_offset, 1),
           .right = count,
           .dir   = mir_const(mu, mir_bool_type(mu), RANGE_TO),
         }
      };
      arg = mir_build_wrap(mu, resolved, dims, 1);
   }
   else {
      arg = mir_build_load(mu, resolved);
      count = mir_const(mu, t_offset, 1);
   }

   mir_value_t args[] = { pkg, arg };

   mir_value_t result = mir_build_fcall(mu, func, t_out, MIR_NULL_STAMP,
                                        args, ARRAY_LEN(args));

   if (mir_is(mu, result, MIR_TYPE_UARRAY))
      result = mir_build_unwrap(mu, result);

   if (mir_is(mu, out, MIR_TYPE_UARRAY))
      out = mir_build_unwrap(mu, out);

   mir_build_put_conversion(mu, conv, out, count, result);

   mir_build_return(mu, mir_const(mu, t_offset, 0));
}

static void vlog_lower_convert_in(mir_unit_t *mu, object_t *obj)
{
   tree_t map = tree_from_object(obj);
   assert(tree_kind(map) == T_PARAM);

   tree_t cf = tree_value(map);
   assert(tree_kind(cf) == T_CONV_FUNC);

   mir_context_t *mc = mir_get_context(mu);

   ident_t parent1 = mir_get_parent(mu);
   mir_shape_t *shape1 = mir_get_shape(mc, parent1);

   ident_t parent2 = mir_get_shape_parent(shape1);
   mir_shape_t *shape2 = mir_get_shape(mc, parent2);

   tree_t arg = tree_value(cf);
   assert(tree_kind(arg) == T_REF);

   int nth = mir_find_slot(shape2, tree_ident(tree_ref(arg)));
   assert(nth >= 0);

   mir_value_t in_upref = mir_build_var_upref(mu, 2, nth);
   mir_value_t in_nets = mir_build_load(mu, in_upref);

   int hops;
   mir_value_t var = mir_search_object(mu, map, &hops);
   assert(!mir_is_null(var));
   assert(hops == 1);

   mir_value_t out_upref = mir_build_var_upref(mu, hops, var.id);
   mir_value_t out_nets = mir_build_load(mu, out_upref);

   vlog_lower_converter(mu, cf, in_nets, out_nets);
}

static void vlog_lower_convert_out(mir_unit_t *mu, object_t *obj)
{
   tree_t map = tree_from_object(obj);
   assert(tree_kind(map) == T_PARAM);

   tree_t cf = tree_name(map);
   assert(tree_kind(cf) == T_CONV_FUNC);

   mir_context_t *mc = mir_get_context(mu);

   ident_t parent1 = mir_get_parent(mu);
   mir_shape_t *shape1 = mir_get_shape(mc, parent1);

   ident_t parent2 = mir_get_shape_parent(shape1);
   mir_shape_t *shape2 = mir_get_shape(mc, parent2);

   int hops;
   mir_value_t var = mir_search_object(mu, map, &hops);
   assert(!mir_is_null(var));
   assert(hops == 1);

   mir_value_t in_upref = mir_build_var_upref(mu, hops, var.id);
   mir_value_t in_nets = mir_build_load(mu, in_upref);

   tree_t dst = tree_value(map);
   assert(tree_kind(dst) == T_REF);

   int nth = mir_find_slot(shape2, tree_ident(tree_ref(dst)));
   assert(nth >= 0);

   mir_value_t out_upref = mir_build_var_upref(mu, 2, nth);
   mir_value_t out_nets = mir_build_load(mu, out_upref);

   vlog_lower_converter(mu, cf, in_nets, out_nets);
}

void vlog_lower_block(mir_context_t *mc, ident_t parent, tree_t b)
{
   tree_t hier = tree_decl(b, 0);
   assert(tree_kind(hier) == T_HIER);

   mir_shape_t *shape = mir_get_shape(mc, parent);
   ident_t qual = tree_ident2(hier);
   mir_unit_t *mu = mir_unit_new(mc, qual, tree_to_object(b),
                                 MIR_UNIT_INSTANCE, shape);

   tree_t wrap = tree_ref(hier);
   assert(tree_kind(wrap) == T_VERILOG);

   vlog_node_t body = tree_vlog(wrap);
   assert(vlog_kind(body) == V_INST_BODY);

   hash_t *map = hash_new(16);

   const int vhdl_ndecls = tree_decls(b);
   const int vlog_ndecls = vlog_decls(body);

   for (int i = 1, pos = 0; i < vhdl_ndecls; i++) {
      tree_t t = tree_decl(b, i);
      ident_t id = tree_ident(t);
      for (; pos < vlog_ndecls; pos++) {
         vlog_node_t v = vlog_decl(body, pos);
         if (vlog_ident(v) == id) {
            hash_put(map, v, t);
            break;
         }
      }

      if (pos == vlog_ndecls)
         fatal_trace("missing VHDL signal for %s", istr(id));
   }

   const int nports = vlog_ports(body);
   assert(tree_ports(b) == nports);
   assert(tree_params(b) == nports);

   for (int i = 0; i < nports; i++)
      hash_put(map, vlog_ref(vlog_ref(vlog_port(body, i))), tree_port(b, i));

   ident_t pkg_name = well_known(W_NVC_VERILOG);
   mir_value_t pkg = mir_build_package_init(mu, pkg_name, MIR_NULL_VALUE);

   mir_type_t t_net_value = mir_int_type(mu, 0, 255);
   mir_type_t t_resolution = mir_resolution_type(mu, t_net_value);

   ident_t var_name = ident_new("NVC.VERILOG.T_WIRE$resolution");
   mir_value_t resfn =
      mir_build_link_var(mu, pkg_name, pkg, var_name, t_resolution);

   vlog_gen_t g = {
      .mu = mu,
   };

   for (int i = 0; i < vlog_ndecls; i++) {
      vlog_node_t d = vlog_decl(body, i);

      switch (vlog_kind(d)) {
      case V_PORT_DECL:
         break;   // Translated below
      case V_NET_DECL:
         vlog_lower_net_decl(mu, d, hash_get(map, d), resfn);
         break;
      case V_VAR_DECL:
         vlog_lower_var_decl(&g, d, hash_get(map, d));
         break;
      case V_LOCALPARAM:
         break;  // Always inlined for now
      default:
         CANNOT_HANDLE(d);
      }
   }

   mir_value_t self = mir_build_context_upref(mu, 0);
   mir_type_t t_self = mir_context_type(mu, qual);
   mir_type_t t_offset = mir_offset_type(mu);

   for (int i = 0; i < nports; i++) {
      vlog_node_t ref = vlog_port(body, i);
      assert(vlog_kind(ref) == V_REF);

      vlog_node_t port = vlog_ref(ref);
      assert(vlog_kind(port) == V_PORT_DECL);

      int hops;
      mir_value_t var = mir_search_object(mu, vlog_ref(port), &hops);
      assert(!mir_is_null(var));
      assert(hops == 0);

      mir_put_object(mu, port, var);

      mir_value_t count = mir_const(mu, t_offset, vlog_size(vlog_type(port)));

      tree_t map = tree_param(b, i);
      tree_t value = tree_value(map);

      mir_value_t in_conv = MIR_NULL_VALUE;
      if (tree_kind(value) == T_CONV_FUNC) {
         mir_put_object(mu, map, var);

         ident_t func = ident_sprintf("%s.%s$verilog_convert_in",
                                      istr(qual), istr(vlog_ident(port)));
         mir_defer(mc, func, qual, MIR_UNIT_FUNCTION, vlog_lower_convert_in,
                      tree_to_object(map));

         mir_value_t closure =
            mir_build_closure(mu, func, self, t_self, t_offset);
         in_conv = mir_build_port_conversion(mu, closure, closure);

         value = tree_value(value);
      }

      assert(tree_kind(value) == T_REF);

      int nth = mir_find_slot(shape, tree_ident(tree_ref(value)));
      assert(nth >= 0);

      mir_value_t upref = mir_build_var_upref(mu, 1, nth);
      mir_value_t dst_nets = mir_build_load(mu, upref);

      if (mir_is(mu, dst_nets, MIR_TYPE_UARRAY))
         dst_nets = mir_build_unwrap(mu, dst_nets);

      mir_value_t out_conv = MIR_NULL_VALUE;
      if (tree_subkind(map) == P_NAMED) {
         tree_t name = tree_name(map);
         if (tree_kind(name) == T_CONV_FUNC) {
            mir_put_object(mu, map, var);

            ident_t func = ident_sprintf("%s.%s$verilog_convert_out",
                                         istr(qual), istr(vlog_ident(port)));
            mir_defer(mc, func, qual, MIR_UNIT_FUNCTION, vlog_lower_convert_out,
                      tree_to_object(map));

            mir_value_t closure =
               mir_build_closure(mu, func, self, t_self, t_offset);
            out_conv = mir_build_port_conversion(mu, closure, closure);
         }
      }

      mir_value_t src_nets = mir_build_load(mu, var);

      if (mir_is(mu, src_nets, MIR_TYPE_UARRAY))
         src_nets = mir_build_unwrap(mu, src_nets);

      switch (vlog_subkind(port)) {
      case V_PORT_INPUT:
         if (mir_is_null(in_conv))
            mir_build_map_signal(mu, dst_nets, src_nets, count);
         else {
            mir_build_convert_out(mu, in_conv, src_nets, count);
            mir_build_convert_in(mu, in_conv, dst_nets, count);
         }
         break;
      case V_PORT_OUTPUT:
         if (mir_is_null(out_conv))
            mir_build_map_signal(mu, src_nets, dst_nets, count);
         else {
            mir_build_convert_out(mu, out_conv, dst_nets, count);
            mir_build_convert_in(mu, out_conv, src_nets, count);
         }
         break;
      default:
         CANNOT_HANDLE(port);
      }
   }

   mir_build_return(mu, MIR_NULL_VALUE);

   vlog_lower_cleanup(&g);

   hash_free(map);

   mir_optimise(mu, MIR_PASS_O1);
   mir_put_unit(mc, mu);
}
