/*++

Copyright (C) 2018 Autodesk Inc. (Original Author)

All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

--*/

//////////////////////////////////////////////////////////////////////////////////////////////////////
// buildbindingccppdynamic.go
// functions to generate C and C++-bindings of a library's API in form of dynamically loaded functions
// handles. It provides bindings that link explicitly (also known as run-time dynamic library loading)
// and implicitly (load-time dynamic library loading).
//////////////////////////////////////////////////////////////////////////////////////////////////////

package main

import (
	"fmt"
	"log"
	"path"
	"path/filepath"
	"strings"
)

// BuildBindingCExplicit builds dyanmic C-bindings of a library's API in form of explicitly loaded function handles.
func BuildBindingCExplicit(component ComponentDefinition, outputFolder string, outputFolderExample string, indentString string) error {
	forceRecreation := false

	namespace := component.NameSpace
	libraryname := component.LibraryName
	baseName := component.BaseName

	DynamicCHeader := path.Join(outputFolder, baseName+"_dynamic.h")
	log.Printf("Creating \"%s\"", DynamicCHeader)
	dynhfile, err := CreateLanguageFile(DynamicCHeader, indentString)
	if err != nil {
		return err
	}
	dynhfile.WriteCLicenseHeader(component,
		fmt.Sprintf("This is an autogenerated plain C Header file in order to allow an easy\n use of %s", libraryname),
		true)
	err = buildDynamicCCPPHeader(component, dynhfile, namespace, baseName, false, false)
	if err != nil {
		return err
	}

	DynamicCImpl := path.Join(outputFolder, baseName+"_dynamic.cc")
	log.Printf("Creating \"%s\"", DynamicCImpl)
	dyncppfile, err := CreateLanguageFile(DynamicCImpl, indentString)
	if err != nil {
		return err
	}
	dyncppfile.WriteCLicenseHeader(component,
		fmt.Sprintf("This is an autogenerated plain C Header file in order to allow an easy\n use of %s", libraryname),
		true)

	err = buildDynamicCImplementation(component, dyncppfile, namespace, baseName, true)
	if err != nil {
		return err
	}

	if len(outputFolderExample) > 0 {
		CExample := path.Join(outputFolderExample, namespace+"_example"+".c")
		if forceRecreation || !FileExists(CExample) {
			log.Printf("Creating \"%s\"", CExample)
			cexamplefile, err := CreateLanguageFile(CExample, indentString)
			if err != nil {
				return err
			}
			cexamplefile.WriteCLicenseHeader(component,
				fmt.Sprintf("This is an autogenerated C application that demonstrates the\n usage of the C bindings of %s", libraryname),
				true)
			buildDynamicCExample(component, cexamplefile, outputFolder, "")
		} else {
			log.Printf("Omitting recreation of C-example file \"%s\"", CExample)
		}

		CPPCMake := path.Join(outputFolderExample, "CMakeLists.txt")
		if forceRecreation || !FileExists(CPPCMake) {
			log.Printf("Creating \"%s\"", CPPCMake)
			cppcmake, err := CreateLanguageFile(CPPCMake, "  ")
			if err != nil {
				return err
			}
			cppcmake.WriteCMakeLicenseHeader(component,
				fmt.Sprintf("This is an autogenerated CMake Project that demonstrates the\n usage of the C bindings of %s", libraryname),
				true)
			buildCDynamicExampleCMake(component, cppcmake, outputFolder, outputFolderExample, true)
		} else {
			log.Printf("Omitting recreation of C++-example CMakeLists-file \"%s\"", CPPCMake)
		}
	}
	return nil
}

// BuildBindingCppImplicit builds dynamic C++-bindings of a library's API in form of implicitly linked functions handles.
func BuildBindingCppImplicit(component ComponentDefinition, outputFolder string, outputFolderExample string, indentString string, ClassIdentifier string) error {
	forceRecreation := false
	ExplicitLinking := false

	namespace := component.NameSpace
	libraryname := component.LibraryName
	baseName := component.BaseName

	CppHeader := path.Join(outputFolder, baseName+"_implicit.hpp")
	log.Printf("Creating \"%s\"", CppHeader)
	hppfile, err := CreateLanguageFile(CppHeader, indentString)
	if err != nil {
		return err
	}
	hppfile.WriteCLicenseHeader(component,
		fmt.Sprintf("This is an autogenerated C++-Header file in order to allow an easy\n use of %s", libraryname),
		true)
	err = buildCppHeader(component, hppfile, namespace, baseName, ClassIdentifier, ExplicitLinking)
	if err != nil {
		return err
	}

	if len(outputFolderExample) > 0 {
		CPPExample := path.Join(outputFolderExample, namespace+"_example"+".cpp")
		if forceRecreation || !FileExists(CPPExample) {
			log.Printf("Creating \"%s\"", CPPExample)
			cppexamplefile, err := CreateLanguageFile(CPPExample, indentString)
			if err != nil {
				return err
			}
			cppexamplefile.WriteCLicenseHeader(component,
				fmt.Sprintf("This is an autogenerated C++ application that demonstrates the\n usage of the C++ bindings of %s", libraryname),
				true)
			buildDynamicCppExample(component, cppexamplefile, outputFolder, ClassIdentifier, ExplicitLinking)
		} else {
			log.Printf("Omitting recreation of C++-example file \"%s\"", CPPExample)
		}

		CPPCMake := path.Join(outputFolderExample, "CMakeLists.txt")
		if forceRecreation || !FileExists(CPPCMake) {
			log.Printf("Creating \"%s\"", CPPCMake)
			cppcmake, err := CreateLanguageFile(CPPCMake, "  ")
			if err != nil {
				return err
			}
			cppcmake.WriteCMakeLicenseHeader(component,
				fmt.Sprintf("This is an autogenerated CMake Project that demonstrates the\n usage of the C++ bindings of %s", libraryname),
				true)
			buildCppDynamicExampleCMake(component, cppcmake, outputFolder, outputFolderExample, ExplicitLinking)
		} else {
			log.Printf("Omitting recreation of C++-example CMakeLists-file \"%s\"", CPPCMake)
		}
	}
	return nil
}

func buildDynamicCCPPHeader(component ComponentDefinition, w LanguageWriter, NameSpace string, BaseName string,
	headerOnly bool, useCPPTypes bool) error {

	sIncludeGuard := "__" + strings.ToUpper(NameSpace) + "_DYNAMICHEADER"
	if useCPPTypes {
		sIncludeGuard += "_CPPTYPES"
	}
	w.Writeln("#ifndef %s", sIncludeGuard)
	w.Writeln("#define %s", sIncludeGuard)
	w.Writeln("")

	if useCPPTypes {
		w.Writeln("#include \"%s_types.hpp\"", BaseName)
	} else {
		w.Writeln("#include \"%s_types.h\"", BaseName)
	}
	w.Writeln("")
	for _, subComponent := range(component.ImportedComponentDefinitions) {
		w.Writeln("#include \"%s_types.hpp\"", subComponent.BaseName)
	}
	w.Writeln("")

	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]

		w.Writeln("")
		w.Writeln("/*************************************************************************************************************************")
		w.Writeln(" Class definition for %s", class.ClassName)
		w.Writeln("**************************************************************************************************************************/")

		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]
			WriteCCPPAbiMethod(method, w, NameSpace, class.ClassName, false, true, useCPPTypes)
		}
	}

	w.Writeln("")
	w.Writeln("/*************************************************************************************************************************")
	w.Writeln(" Global functions")
	w.Writeln("**************************************************************************************************************************/")

	global := component.Global
	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]
		err := WriteCCPPAbiMethod(method, w, NameSpace, "Wrapper", true, true, useCPPTypes)
		if err != nil {
			return err
		}
	}

	w.Writeln("")
	w.Writeln("/*************************************************************************************************************************")
	w.Writeln(" Function Table Structure")
	w.Writeln("**************************************************************************************************************************/")
	w.Writeln("")
	w.Writeln("typedef struct {")
	w.Writeln("  void * m_LibraryHandle;")

	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]
		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]
			w.Writeln("  P%s%s_%sPtr m_%s_%s;", NameSpace, class.ClassName, method.MethodName, class.ClassName, method.MethodName)
		}
	}

	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]
		w.Writeln("  P%s%sPtr m_%s;", NameSpace, method.MethodName, method.MethodName)
	}

	w.Writeln("} s%sDynamicWrapperTable;", NameSpace)
	w.Writeln("")

	if !headerOnly {
		w.Writeln("/*************************************************************************************************************************")
		w.Writeln(" Load DLL dynamically")
		w.Writeln("**************************************************************************************************************************/")

		w.Writeln("%sResult Init%sWrapperTable(s%sDynamicWrapperTable * pWrapperTable);", NameSpace, NameSpace, NameSpace)
		w.Writeln("%sResult Release%sWrapperTable(s%sDynamicWrapperTable * pWrapperTable);", NameSpace, NameSpace, NameSpace)
		w.Writeln("%sResult Load%sWrapperTable(s%sDynamicWrapperTable * pWrapperTable, const char * pLibraryFileName);", NameSpace, NameSpace, NameSpace)

		w.Writeln("")
	}

	w.Writeln("#endif // %s", sIncludeGuard)
	w.Writeln("")

	return nil
}

func buildDynamicCInitTableCode(component ComponentDefinition, w LanguageWriter, NameSpace string, BaseName string, useStrictC bool) error {
	global := component.Global

	nullPtrStr := "nullptr"
	if (useStrictC) {
		nullPtrStr = "NULL"
	}

	w.Writeln("if (pWrapperTable == %s)", nullPtrStr)
	w.Writeln("  return %s_ERROR_INVALIDPARAM;", strings.ToUpper(NameSpace))
	w.Writeln("")
	
	w.Writeln("pWrapperTable->m_LibraryHandle = %s;", nullPtrStr)

	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]
		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]
			w.Writeln("pWrapperTable->m_%s_%s = %s;", class.ClassName, method.MethodName, nullPtrStr)
		}
	}

	global = component.Global
	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]
		w.Writeln("pWrapperTable->m_%s = %s;", method.MethodName, nullPtrStr)
	}

	w.Writeln("")
	w.Writeln("return %s_SUCCESS;", strings.ToUpper(NameSpace))

	return nil
}

func buildDynamicCReleaseTableCode(component ComponentDefinition, w LanguageWriter, NameSpace string, BaseName string, initWrapperFunctionName string, useStrictC bool) error {
	nullPtrStr := "nullptr"
	if (useStrictC) {
		nullPtrStr = "NULL"
	}

	w.Writeln("if (pWrapperTable == %s)", nullPtrStr)
	w.Writeln("  return %s_ERROR_INVALIDPARAM;", strings.ToUpper(NameSpace))
	w.Writeln("")
	w.Writeln("if (pWrapperTable->m_LibraryHandle != %s) {", nullPtrStr)
	w.Writeln("#ifdef _WIN32")
	w.Writeln("  HMODULE hModule = (HMODULE) pWrapperTable->m_LibraryHandle;")
	w.Writeln("  FreeLibrary(hModule);")
	w.Writeln("#else // _WIN32")
	w.Writeln("  dlclose(pWrapperTable->m_LibraryHandle);")
	w.Writeln("#endif // _WIN32")
	w.Writeln("  return %s(pWrapperTable);", initWrapperFunctionName)
	w.Writeln("}")
	w.Writeln("")
	w.Writeln("return %s_SUCCESS;", strings.ToUpper(NameSpace))

	return nil
}

func writeLoadingOfMethodFromSymbolLookupMethod(w LanguageWriter, methodName string, NameSpace string, useStrictC bool) {
	nullPtrStr := "nullptr"
	if (useStrictC) {
		nullPtrStr = "NULL"
	}

	w.Writeln("eLookupError = (*pLookup)(\"%s_%s\", (void**)&(pWrapperTable->m_%s));", strings.ToLower(NameSpace), strings.ToLower(methodName), methodName)
		
	w.Writeln("if ( (eLookupError != 0) || (pWrapperTable->m_%s == %s) )", methodName, nullPtrStr)
	w.Writeln("  return %s_ERROR_COULDNOTFINDLIBRARYEXPORT;", strings.ToUpper(NameSpace))
	w.Writeln("")
}

// WriteLoadingOfMethod the loading of a method from a library into a LanguagWriter
func WriteLoadingOfMethod(class ComponentDefinitionClass, method ComponentDefinitionMethod, w LanguageWriter, NameSpace string, useStrictC bool) {
	nullPtrStr := "nullptr"
	if (useStrictC) {
		nullPtrStr = "NULL"
	}

	w.Writeln("#ifdef _WIN32")
	w.Writeln("pWrapperTable->m_%s_%s = (P%s%s_%sPtr) GetProcAddress(hLibrary, \"%s_%s_%s\");", class.ClassName, method.MethodName, NameSpace, class.ClassName, method.MethodName, strings.ToLower(NameSpace), strings.ToLower(class.ClassName), strings.ToLower(method.MethodName))
	w.Writeln("#else // _WIN32")
	w.Writeln("pWrapperTable->m_%s_%s = (P%s%s_%sPtr) dlsym(hLibrary, \"%s_%s_%s\");", class.ClassName, method.MethodName, NameSpace, class.ClassName, method.MethodName, strings.ToLower(NameSpace), strings.ToLower(class.ClassName), strings.ToLower(method.MethodName))
	w.Writeln("dlerror();")
	w.Writeln("#endif // _WIN32")
	w.Writeln("if (pWrapperTable->m_%s_%s == %s)", class.ClassName, method.MethodName, nullPtrStr)
	w.Writeln("  return %s_ERROR_COULDNOTFINDLIBRARYEXPORT;", strings.ToUpper(NameSpace))
	w.Writeln("")
}


func buildDynamicCLoadTableFromSymbolLookupMethodCode(component ComponentDefinition, w LanguageWriter, NameSpace string, BaseName string, useStrictC bool) error {
	global := component.Global

	nullPtrStr := "nullptr"
	if (useStrictC) {
		nullPtrStr = "NULL"
	}

	w.Writeln("if (pWrapperTable == %s)", nullPtrStr)
	w.Writeln("  return %s_ERROR_INVALIDPARAM;", strings.ToUpper(NameSpace))
	w.Writeln("if (pSymbolLookupMethod == %s)", nullPtrStr)
	w.Writeln("  return %s_ERROR_INVALIDPARAM;", strings.ToUpper(NameSpace))
	w.Writeln("")
	w.Writeln("typedef %sResult(*SymbolLookupType)(const char*, void**);", NameSpace)
	w.Writeln("")
	w.Writeln("SymbolLookupType pLookup = (SymbolLookupType)pSymbolLookupMethod;")
	w.Writeln("")
	w.Writeln("%sResult eLookupError = %s_SUCCESS;", NameSpace, strings.ToUpper(NameSpace))
	
	
	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]
		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]
			methodName := class.ClassName + "_" + method.MethodName
			writeLoadingOfMethodFromSymbolLookupMethod(w, methodName, NameSpace, useStrictC)
		}
	}

	global = component.Global
	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]
		methodName := method.MethodName
		writeLoadingOfMethodFromSymbolLookupMethod(w, methodName, NameSpace, useStrictC)
	}

	w.Writeln("return %s_SUCCESS;", strings.ToUpper(NameSpace))

	return nil
}

func buildDynamicCLoadTableCode(component ComponentDefinition, w LanguageWriter, NameSpace string, BaseName string, useStrictC bool) error {
	global := component.Global

	nullPtrStr := "nullptr"
	if (useStrictC) {
		nullPtrStr = "NULL"
	}

	w.Writeln("if (pWrapperTable == %s)", nullPtrStr)
	w.Writeln("  return %s_ERROR_INVALIDPARAM;", strings.ToUpper(NameSpace))
	w.Writeln("if (pLibraryFileName == %s)", nullPtrStr)
	w.Writeln("  return %s_ERROR_INVALIDPARAM;", strings.ToUpper(NameSpace))

	w.Writeln("")

	w.Writeln("#ifdef _WIN32")
	w.Writeln("// Convert filename to UTF16-string")
	w.Writeln("int nLength = (int)strlen(pLibraryFileName);")
	w.Writeln("int nBufferSize = nLength * 2 + 2;")
	if (!useStrictC) {
		w.Writeln("std::vector<wchar_t> wsLibraryFileName(nBufferSize);")
		w.Writeln("int nResult = MultiByteToWideChar(CP_UTF8, 0, pLibraryFileName, nLength, &wsLibraryFileName[0], nBufferSize);")
		w.Writeln("if (nResult == 0)")
		w.Writeln("  return %s_ERROR_COULDNOTLOADLIBRARY;", strings.ToUpper(NameSpace))
		w.Writeln("")
		w.Writeln("HMODULE hLibrary = LoadLibraryW(wsLibraryFileName.data());")
	} else {
		w.Writeln("wchar_t* wsLibraryFileName = malloc(nBufferSize*sizeof(wchar_t));")
		w.Writeln("memset(wsLibraryFileName, 0, nBufferSize*sizeof(wchar_t));")
		w.Writeln("int nResult = MultiByteToWideChar(CP_UTF8, 0, pLibraryFileName, nLength, wsLibraryFileName, nBufferSize);")
		w.Writeln("if (nResult == 0) {")
		w.Writeln("  free(wsLibraryFileName);")
		w.Writeln("  return %s_ERROR_COULDNOTLOADLIBRARY;", strings.ToUpper(NameSpace))
		w.Writeln("}")
		w.Writeln("")
		w.Writeln("HMODULE hLibrary = LoadLibraryW(wsLibraryFileName);")
		w.Writeln("free(wsLibraryFileName);")
	}
	
	w.Writeln("if (hLibrary == 0) ")
	w.Writeln("  return %s_ERROR_COULDNOTLOADLIBRARY;", strings.ToUpper(NameSpace))
	w.Writeln("#else // _WIN32")
	w.Writeln("void* hLibrary = dlopen(pLibraryFileName, RTLD_LAZY);")
	w.Writeln("if (hLibrary == 0) ")
	w.Writeln("  return %s_ERROR_COULDNOTLOADLIBRARY;", strings.ToUpper(NameSpace))
	w.Writeln("dlerror();")
	w.Writeln("#endif // _WIN32")
	w.Writeln("")

	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]
		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]
			WriteLoadingOfMethod(class, method, w, NameSpace, useStrictC)
		}
	}

	global = component.Global
	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]
		w.Writeln("#ifdef _WIN32")
		w.Writeln("pWrapperTable->m_%s = (P%s%sPtr) GetProcAddress(hLibrary, \"%s_%s\");", method.MethodName, NameSpace, method.MethodName, strings.ToLower(NameSpace), strings.ToLower(method.MethodName))
		w.Writeln("#else // _WIN32")
		w.Writeln("pWrapperTable->m_%s = (P%s%sPtr) dlsym(hLibrary, \"%s_%s\");", method.MethodName, NameSpace, method.MethodName, strings.ToLower(NameSpace), strings.ToLower(method.MethodName))
		w.Writeln("dlerror();")
		w.Writeln("#endif // _WIN32")

		w.Writeln("if (pWrapperTable->m_%s == %s)", method.MethodName, nullPtrStr)
		w.Writeln("  return %s_ERROR_COULDNOTFINDLIBRARYEXPORT;", strings.ToUpper(NameSpace))
		w.Writeln("")
	}

	w.Writeln("pWrapperTable->m_LibraryHandle = hLibrary;")
	w.Writeln("return %s_SUCCESS;", strings.ToUpper(NameSpace))

	return nil
}

func buildDynamicCImplementation(component ComponentDefinition, w LanguageWriter, NameSpace string, BaseName string, useStrictC bool) error {
	w.Writeln("#include \"%s_types.h\"", BaseName)
	w.Writeln("#include \"%s_dynamic.h\"", BaseName)

	w.Writeln("#ifdef _WIN32")
	if (!useStrictC) {
		w.Writeln("#include <vector>")
	}
	w.Writeln("#include <windows.h>")

	w.Writeln("#else // _WIN32")
	w.Writeln("#include <dlfcn.h>")
	w.Writeln("#endif // _WIN32")

	w.Writeln("")
	w.Writeln("%sResult Init%sWrapperTable(s%sDynamicWrapperTable * pWrapperTable)", NameSpace, NameSpace, NameSpace)
	w.Writeln("{")

	w.AddIndentationLevel(1)
	buildDynamicCInitTableCode(component, w, NameSpace, BaseName, useStrictC)
	w.AddIndentationLevel(-1)

	w.Writeln("}")

	w.Writeln("")
	w.Writeln("%sResult Release%sWrapperTable(s%sDynamicWrapperTable * pWrapperTable)", NameSpace, NameSpace, NameSpace)
	w.Writeln("{")

	w.AddIndentationLevel(1)
	buildDynamicCReleaseTableCode(component, w, NameSpace, BaseName, "Init"+NameSpace+"WrapperTable", useStrictC)
	w.AddIndentationLevel(-1)

	w.Writeln("}")

	w.Writeln("")
	w.Writeln("%sResult Load%sWrapperTable(s%sDynamicWrapperTable * pWrapperTable, const char * pLibraryFileName)", NameSpace, NameSpace, NameSpace)
	w.Writeln("{")

	w.AddIndentationLevel(1)
	buildDynamicCLoadTableCode(component, w, NameSpace, BaseName, useStrictC)
	w.AddIndentationLevel(-1)

	w.Writeln("}")
	w.Writeln("")

	return nil
}

func writeDynamicCPPMethodDeclaration(method ComponentDefinitionMethod, w LanguageWriter, NameSpace string, ClassIdentifier string, ClassName string) error {
	parameters := ""
	returntype := "void"

	for k := 0; k < len(method.Params); k++ {

		param := method.Params[k]
		variableName := getBindingCppVariableName(param)

		switch param.ParamPass {
		case "in":
			if parameters != "" {
				parameters = parameters + ", "
			}
			cppParamType := getBindingCppParamType(param.ParamType, param.ParamClass, NameSpace, ClassIdentifier, true)

			switch param.ParamType {
			case "string":
				parameters = parameters + fmt.Sprintf("const %s & %s", cppParamType, variableName)
			case "struct":
				parameters = parameters + fmt.Sprintf("const %s & %s", cppParamType, variableName)
			case "structarray", "basicarray":
				parameters = parameters + fmt.Sprintf("const %s & %s", cppParamType, variableName)
			case "class", "optionalclass":
				parameters = parameters + fmt.Sprintf("%s %s", cppParamType, variableName)
			default:
				parameters = parameters + fmt.Sprintf("const %s %s", cppParamType, variableName)
			}
		case "out":
			cppParamType := getBindingCppParamType(param.ParamType, param.ParamClass, NameSpace, ClassIdentifier, false)
			if parameters != "" {
				parameters = parameters + ", "
			}
			parameters = parameters + fmt.Sprintf("%s & %s", cppParamType, variableName)
		case "return":
			returntype = getBindingCppParamType(param.ParamType, param.ParamClass, NameSpace, ClassIdentifier, false)
		default:
			return fmt.Errorf("invalid method parameter passing \"%s\" for %s.%s(%s)", param.ParamPass, ClassName, method.MethodName, param.ParamName)
		}
	}

	w.Writeln("  inline %s %s(%s);", returntype, method.MethodName, parameters)

	return nil
}

func writeDynamicCPPMethod(method ComponentDefinitionMethod, w LanguageWriter, NameSpace string, ClassIdentifier string, ClassName string,
	implementationLines []string, isGlobal bool, includeComments bool, doNotThrow bool, useCPPTypes bool, ExplicitLinking bool) error {

	CMethodName := ""
	requiresInitCall := false
	initCallParameters := "" // usually used to check sizes of buffers
	callParameters := ""
	checkErrorCodeBegin := ""
	checkErrorCodeEnd := ")"
	makeSharedParameter := ""

	if isGlobal {
		if ExplicitLinking {
			CMethodName = fmt.Sprintf("m_WrapperTable.m_%s", method.MethodName)
		} else {
			CMethodName = fmt.Sprintf("%s_%s", strings.ToLower(NameSpace), strings.ToLower(method.MethodName))
		}
		checkErrorCodeBegin = "CheckError(nullptr,"
		makeSharedParameter = "this"
	} else {
		if ExplicitLinking {
			CMethodName = fmt.Sprintf("m_pWrapper->m_WrapperTable.m_%s_%s", ClassName, method.MethodName)
		} else {
			CMethodName = fmt.Sprintf("%s_%s_%s", strings.ToLower(NameSpace), strings.ToLower(ClassName), strings.ToLower(method.MethodName))
		}
		callParameters = "m_pHandle"
		initCallParameters = "m_pHandle"
		checkErrorCodeBegin = "CheckError("
		makeSharedParameter = "m_pWrapper"
	}
	if doNotThrow {
		checkErrorCodeBegin = ""
		checkErrorCodeEnd = ""
	}

	parameters := ""
	returntype := "void"

	definitionCodeLines := []string{}
	functionCodeLines := []string{}
	returnCodeLines := []string{}
	commentcodeLines := []string{}
	postCallCodeLines := []string{}

	cppClassPrefix := "C"
	if !useCPPTypes {
		cppClassPrefix += NameSpace
	}
	cppClassName := cppClassPrefix + ClassIdentifier + ClassName

	for k := 0; k < len(method.Params); k++ {
		param := method.Params[k]
		variableName := getBindingCppVariableName(param)

		callParameter := ""
		initCallParameter := ""

		switch param.ParamPass {
		case "in":
			if parameters != "" {
				parameters = parameters + ", "
			}
			cppParamType := getBindingCppParamType(param.ParamType, param.ParamClass, NameSpace, ClassIdentifier, true)
			commentcodeLines = append(commentcodeLines, fmt.Sprintf("* @param[in] %s - %s", variableName, param.ParamDescription))

			switch param.ParamType {
			case "string":
				callParameter = variableName + ".c_str()"
				initCallParameter = callParameter
				parameters = parameters + fmt.Sprintf("const %s & %s", cppParamType, variableName)
			case "struct":
				callParameter = "&" + variableName
				initCallParameter = callParameter
				parameters = parameters + fmt.Sprintf("const %s & %s", cppParamType, variableName)
			case "structarray", "basicarray":
				callParameter = fmt.Sprintf("(%s_uint64)%s.size(), %s.data()", NameSpace, variableName, variableName)
				initCallParameter = callParameter
				parameters = parameters + fmt.Sprintf("const %s & %s", cppParamType, variableName)
			case "class", "optionalclass":
				paramNameSpace, _, _ := decomposeParamClassName(param.ParamClass)
				if len(paramNameSpace) == 0 {
					paramNameSpace = NameSpace
				}

				definitionCodeLines = append(definitionCodeLines, fmt.Sprintf("%sHandle h%s = nullptr;", paramNameSpace, param.ParamName))
				definitionCodeLines = append(definitionCodeLines, fmt.Sprintf("if (%s != nullptr) {", variableName))
				definitionCodeLines = append(definitionCodeLines, fmt.Sprintf("  h%s = %s->GetHandle();", param.ParamName, variableName))
				definitionCodeLines = append(definitionCodeLines, fmt.Sprintf("};"))
				callParameter = "h" + param.ParamName
				initCallParameter = callParameter
				parameters = parameters + fmt.Sprintf("%s %s", cppParamType, variableName)

			default:
				callParameter = variableName
				initCallParameter = callParameter
				parameters = parameters + fmt.Sprintf("const %s %s", cppParamType, variableName)
			}

		case "out":
			cppParamType := getBindingCppParamType(param.ParamType, param.ParamClass, NameSpace, ClassIdentifier, false)
			commentcodeLines = append(commentcodeLines, fmt.Sprintf("* @param[out] %s - %s", variableName, param.ParamDescription))

			if parameters != "" {
				parameters = parameters + ", "
			}
			parameters = parameters + fmt.Sprintf("%s & %s", cppParamType, variableName)

			switch param.ParamType {

			case "string":
				requiresInitCall = true
				definitionCodeLines = append(definitionCodeLines, fmt.Sprintf("%s_uint32 bytesNeeded%s = 0;", NameSpace, param.ParamName))
				definitionCodeLines = append(definitionCodeLines, fmt.Sprintf("%s_uint32 bytesWritten%s = 0;", NameSpace, param.ParamName))
				initCallParameter = fmt.Sprintf("0, &bytesNeeded%s, nullptr", param.ParamName)

				functionCodeLines = append(functionCodeLines, fmt.Sprintf("std::vector<char> buffer%s(bytesNeeded%s);", param.ParamName, param.ParamName))

				callParameter = fmt.Sprintf("bytesNeeded%s, &bytesWritten%s, &buffer%s[0]", param.ParamName, param.ParamName, param.ParamName)

				postCallCodeLines = append(postCallCodeLines, fmt.Sprintf("s%s = std::string(&buffer%s[0]);", param.ParamName, param.ParamName))

			case "class", "optionalclass":
				paramNameSpace, _, _ := decomposeParamClassName(param.ParamClass)
				if len(paramNameSpace) == 0 {
					paramNameSpace = NameSpace
				}

				definitionCodeLines = append(definitionCodeLines, fmt.Sprintf("%sHandle h%s = nullptr;", paramNameSpace, param.ParamName))
				callParameter = fmt.Sprintf("&h%s", param.ParamName)
				initCallParameter = callParameter

				if (param.ParamType == "optionalclass") {
					postCallCodeLines = append(postCallCodeLines, fmt.Sprintf("if (h%s) {", param.ParamName))
					postCallCodeLines = append(postCallCodeLines, fmt.Sprintf("  p%s = std::make_shared<%s%s%s>(%s, h%s);", param.ParamName, cppClassPrefix, ClassIdentifier, param.ParamClass, makeSharedParameter, param.ParamName))
					postCallCodeLines = append(postCallCodeLines, fmt.Sprintf("} else {"))
					postCallCodeLines = append(postCallCodeLines, fmt.Sprintf("  p%s = nullptr;", param.ParamName))
					postCallCodeLines = append(postCallCodeLines, fmt.Sprintf("}"))
				} else {
					postCallCodeLines = append(postCallCodeLines, fmt.Sprintf("if (!h%s) {", param.ParamName))
					postCallCodeLines = append(postCallCodeLines, fmt.Sprintf("  %s%s_ERROR_INVALIDPARAM%s;", checkErrorCodeBegin, strings.ToUpper(NameSpace), checkErrorCodeEnd))
					postCallCodeLines = append(postCallCodeLines, fmt.Sprintf("} else {"))
					postCallCodeLines = append(postCallCodeLines, fmt.Sprintf("  p%s = std::make_shared<%s%s%s>(%s, h%s);", param.ParamName, cppClassPrefix, ClassIdentifier, param.ParamClass, makeSharedParameter, param.ParamName))
					postCallCodeLines = append(postCallCodeLines, fmt.Sprintf("}"))
				}

			case "structarray", "basicarray":
				requiresInitCall = true
				definitionCodeLines = append(definitionCodeLines, fmt.Sprintf("%s_uint64 elementsNeeded%s = 0;", NameSpace, param.ParamName))
				definitionCodeLines = append(definitionCodeLines, fmt.Sprintf("%s_uint64 elementsWritten%s = 0;", NameSpace, param.ParamName))
				initCallParameter = fmt.Sprintf("0, &elementsNeeded%s, nullptr", param.ParamName)

				functionCodeLines = append(functionCodeLines, fmt.Sprintf("%s.resize((size_t) elementsNeeded%s);", variableName, param.ParamName))
				callParameter = fmt.Sprintf("elementsNeeded%s, &elementsWritten%s, %s.data()", param.ParamName, param.ParamName, variableName)

			default:
				callParameter = "&" + variableName
				initCallParameter = callParameter
			}

		case "return":
			commentcodeLines = append(commentcodeLines, fmt.Sprintf("* @return %s", param.ParamDescription))
			returntype = getBindingCppParamType(param.ParamType, param.ParamClass, NameSpace, ClassIdentifier, false)

			switch param.ParamType {
			case "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64", "bool", "single", "double", "pointer":
				callParameter = fmt.Sprintf("&result%s", param.ParamName)
				initCallParameter = callParameter
				definitionCodeLines = append(definitionCodeLines, fmt.Sprintf("%s result%s = 0;", returntype, param.ParamName))
				returnCodeLines = append(returnCodeLines, fmt.Sprintf("return result%s;", param.ParamName))

			case "string":
				requiresInitCall = true
				definitionCodeLines = append(definitionCodeLines, fmt.Sprintf("%s_uint32 bytesNeeded%s = 0;", NameSpace, param.ParamName))
				definitionCodeLines = append(definitionCodeLines, fmt.Sprintf("%s_uint32 bytesWritten%s = 0;", NameSpace, param.ParamName))
				initCallParameter = fmt.Sprintf("0, &bytesNeeded%s, nullptr", param.ParamName)

				functionCodeLines = append(functionCodeLines, fmt.Sprintf("std::vector<char> buffer%s(bytesNeeded%s);", param.ParamName, param.ParamName))

				callParameter = fmt.Sprintf("bytesNeeded%s, &bytesWritten%s, &buffer%s[0]", param.ParamName, param.ParamName, param.ParamName)

				returnCodeLines = append(returnCodeLines, fmt.Sprintf("return std::string(&buffer%s[0]);", param.ParamName))

			case "enum":
				callParameter = fmt.Sprintf("&result%s", param.ParamName)
				initCallParameter = callParameter
				definitionCodeLines = append(definitionCodeLines, fmt.Sprintf("e%s result%s = (e%s) 0;", param.ParamClass, param.ParamName, param.ParamClass))
				returnCodeLines = append(returnCodeLines, fmt.Sprintf("return result%s;", param.ParamName))

			case "struct":
				callParameter = fmt.Sprintf("&result%s", param.ParamName)
				initCallParameter = callParameter
				definitionCodeLines = append(definitionCodeLines, fmt.Sprintf("s%s result%s;", param.ParamClass, param.ParamName))
				returnCodeLines = append(returnCodeLines, fmt.Sprintf("return result%s;", param.ParamName))

			case "class", "optionalclass":
				paramNameSpace, paramClassName, _ := decomposeParamClassName(param.ParamClass)
				paramNameSpaceCPP, _, _ := decomposeParamClassNameCPP(param.ParamClass)
				CPPClass := cppClassPrefix + ClassIdentifier + paramClassName
				if len(paramNameSpace) == 0 {
					paramNameSpace = NameSpace
				} else {
					CPPClass = paramNameSpaceCPP + CPPClass
					makeSharedParameter = makeSharedParameter + "->m_p" + paramNameSpace + "Wrapper.get()"
				}
				
				definitionCodeLines = append(definitionCodeLines, fmt.Sprintf("%sHandle h%s = nullptr;", NameSpace, param.ParamName))
				callParameter = fmt.Sprintf("&h%s", param.ParamName)
				initCallParameter = callParameter
				
				if (param.ParamType == "optionalclass") {
					returnCodeLines = append(returnCodeLines, fmt.Sprintf("if (h%s) {", param.ParamName))
					returnCodeLines = append(returnCodeLines, fmt.Sprintf("  return std::make_shared<%s>(%s, h%s);", CPPClass, makeSharedParameter, param.ParamName))
					returnCodeLines = append(returnCodeLines, fmt.Sprintf("} else {"))
					returnCodeLines = append(returnCodeLines, fmt.Sprintf("  return nullptr;"))
					returnCodeLines = append(returnCodeLines, fmt.Sprintf("}"))
				} else {
					returnCodeLines = append(returnCodeLines, fmt.Sprintf("if (!h%s) {", param.ParamName))
					returnCodeLines = append(returnCodeLines, fmt.Sprintf("  %s%s_ERROR_INVALIDPARAM%s;", checkErrorCodeBegin, strings.ToUpper(NameSpace), checkErrorCodeEnd))
					returnCodeLines = append(returnCodeLines, fmt.Sprintf("}"))
					returnCodeLines = append(returnCodeLines, fmt.Sprintf("return std::make_shared<%s>(%s, h%s);", CPPClass, makeSharedParameter, param.ParamName))
				}

			case "basicarray":
				return fmt.Errorf("can not return basicarray \"%s\" for %s.%s(%s)", param.ParamPass, ClassName, method.MethodName, param.ParamName)

			case "structarray":
				return fmt.Errorf("can not return structarray \"%s\" for %s.%s(%s)", param.ParamPass, ClassName, method.MethodName, param.ParamName)

			default:
				return fmt.Errorf("invalid method parameter type \"%s\" for %s.%s(%s)", param.ParamType, ClassName, method.MethodName, param.ParamName)
			}

		default:
			return fmt.Errorf("invalid method parameter passing \"%s\" for %s.%s(%s)", param.ParamPass, ClassName, method.MethodName, param.ParamName)
		}

		if callParameters != "" {
			callParameters = callParameters + ", "
		}
		callParameters = callParameters + callParameter
		if initCallParameters != "" {
			initCallParameters = initCallParameters + ", "
		}
		initCallParameters = initCallParameters + initCallParameter
	}

	w.Writeln("  ")
	if includeComments {
		w.Writeln("  /**")
		w.Writeln("  * %s::%s - %s", cppClassName, method.MethodName, method.MethodDescription)
		w.Writelns("  ", commentcodeLines)
		w.Writeln("  */")
	}

	nameSpaceQualifier := cppClassName + "::"

	if isGlobal {
		w.Writeln("  inline %s %s%s(%s)", returntype, nameSpaceQualifier, method.MethodName, parameters)
	} else {
		w.Writeln("  %s %s%s(%s)", returntype, nameSpaceQualifier, method.MethodName, parameters)
	}

	w.Writeln("  {")
	w.Writelns("    ", definitionCodeLines)
	if requiresInitCall {
		w.Writeln("    %s%s(%s)%s;", checkErrorCodeBegin, CMethodName, initCallParameters, checkErrorCodeEnd)
	}
	w.Writelns("    ", functionCodeLines)
	w.Writeln("    %s%s(%s)%s;", checkErrorCodeBegin, CMethodName, callParameters, checkErrorCodeEnd)
	w.Writelns("    ", postCallCodeLines)

	if (len(implementationLines) >0) {
		w.Writeln("    ")
		w.Writelns("    ", implementationLines)
	}

	if (len(returnCodeLines) >0) {
		w.Writeln("    ")
		w.Writelns("    ", returnCodeLines)
	}
	w.Writeln("  }")

	return nil
}

func writeDynamicCppBaseClassMethods(component ComponentDefinition, baseClass ComponentDefinitionClass, w LanguageWriter, NameSpace string, BaseName string, cppClassPrefix string, ClassIdentifier string) error {
	cppBaseClassName := cppClassPrefix + ClassIdentifier + baseClass.ClassName
	w.Writeln("protected:")
	w.Writeln("  /* Wrapper Object that created the class. */")
	w.Writeln("  %s%sWrapper * m_pWrapper;", cppClassPrefix, ClassIdentifier)
	w.Writeln("  /* Handle to Instance in library*/")
	w.Writeln("  %sHandle m_pHandle;", NameSpace)
	w.Writeln("")
	w.Writeln("  /* Checks for an Error code and raises Exceptions */")
	w.Writeln("  void CheckError(%sResult nResult)", NameSpace)
	w.Writeln("  {")
	w.Writeln("    if (m_pWrapper != nullptr)")
	w.Writeln("      m_pWrapper->CheckError(this, nResult);")
	w.Writeln("  }")
	w.Writeln("public:")
	w.Writeln("  /**")
	w.Writeln("  * %s::%s - Constructor for Base class.", cppBaseClassName, cppBaseClassName)
	w.Writeln("  */")
	w.Writeln("  %s(%s%sWrapper * pWrapper, %sHandle pHandle)", cppBaseClassName, cppClassPrefix, ClassIdentifier, NameSpace)
	w.Writeln("    : m_pWrapper(pWrapper), m_pHandle(pHandle)")
	w.Writeln("  {")
	w.Writeln("  }")
	w.Writeln("")
	w.Writeln("  /**")
	w.Writeln("  * %s::~%s - Destructor for Base class.", cppBaseClassName, cppBaseClassName)
	w.Writeln("  */")
	w.Writeln("  virtual ~%s()", cppBaseClassName)
	w.Writeln("  {")
	w.Writeln("    if (m_pWrapper != nullptr)")
	w.Writeln("      m_pWrapper->%s(this);", component.Global.ReleaseMethod)
	w.Writeln("    m_pWrapper = nullptr;")
	w.Writeln("  }")
	w.Writeln("")
	w.Writeln("  /**")
	w.Writeln("  * %s::GetHandle - Returns handle to instance.", cppBaseClassName)
	w.Writeln("  */")
	w.Writeln("  %sHandle GetHandle()", NameSpace)
	w.Writeln("  {")
	w.Writeln("    return m_pHandle;")
	w.Writeln("  }")

	w.Writeln("  ")
	w.Writeln("  friend class CWrapper;")
	return nil
}


func buildBindingCPPAllForwardDeclarations(component ComponentDefinition, w LanguageWriter, NameSpace string, cppClassPrefix string, ClassIdentifier string) {
	w.Writeln("/*************************************************************************************************************************")
	w.Writeln(" Forward Declaration of all classes")
	w.Writeln("**************************************************************************************************************************/")
	w.Writeln("class %s%sWrapper;", cppClassPrefix, ClassIdentifier)
	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]
		className := cppClassPrefix + ClassIdentifier + class.ClassName
		w.Writeln("class %s;", className)
	}
	if (strings.Compare(ClassIdentifier, NameSpace) != 0) {
		w.Writeln("")
		w.Writeln("/*************************************************************************************************************************")
		w.Writeln(" Declaration of deprecated class types")
		w.Writeln("**************************************************************************************************************************/")
		w.Writeln("typedef %s%sWrapper %s%sWrapper;", cppClassPrefix, ClassIdentifier, cppClassPrefix, NameSpace)
		for i := 0; i < len(component.Classes); i++ {
			class := component.Classes[i]
			className := cppClassPrefix + ClassIdentifier + class.ClassName
			w.Writeln("typedef %s %s%s%s;", className, cppClassPrefix, NameSpace, class.ClassName)
		}
		w.Writeln("")
	}

	w.Writeln("/*************************************************************************************************************************")
	w.Writeln(" Declaration of shared pointer types")
	w.Writeln("**************************************************************************************************************************/")
	w.Writeln("typedef std::shared_ptr<%s%sWrapper> P%sWrapper;", cppClassPrefix, ClassIdentifier, ClassIdentifier)
	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]
		className := cppClassPrefix + ClassIdentifier + class.ClassName
		w.Writeln("typedef std::shared_ptr<%s> P%s%s;", className, ClassIdentifier, class.ClassName)
	}
	
	if (strings.Compare(ClassIdentifier, NameSpace) != 0) {
		w.Writeln("")
		w.Writeln("/*************************************************************************************************************************")
		w.Writeln(" Declaration of deprecated shared pointer types")
		w.Writeln("**************************************************************************************************************************/")
		w.Writeln("typedef P%sWrapper P%sWrapper;", ClassIdentifier, NameSpace)
		for i := 0; i < len(component.Classes); i++ {
			class := component.Classes[i]
			w.Writeln("typedef P%s%s P%s%s;", ClassIdentifier, class.ClassName, NameSpace, class.ClassName)
		}
		w.Writeln("")
	}
}

func writeCPPInputVector(w LanguageWriter, NameSpace string, ClassIdentifier string) error {
	w.Writeln("/*************************************************************************************************************************")
	w.Writeln(" Class C%sInputVector", ClassIdentifier)
	w.Writeln("**************************************************************************************************************************/")
	w.Writeln("template <typename T>")
	w.Writeln("class C%sInputVector {", ClassIdentifier)
	w.Writeln("private:")
	w.Writeln("  ")
	w.Writeln("  const T* m_data;")
	w.Writeln("  size_t m_size;")
	w.Writeln("  ")
	w.Writeln("public:")
	w.Writeln("  ")
	w.Writeln("  C%sInputVector( const std::vector<T>& vec)", ClassIdentifier)
	w.Writeln("    : m_data( vec.data() ), m_size( vec.size() )")
	w.Writeln("  {")
	w.Writeln("  }")
	w.Writeln("  ")
	w.Writeln("  C%sInputVector( const T* in_data, size_t in_size)", ClassIdentifier)
	w.Writeln("    : m_data( in_data ), m_size(in_size )")
	w.Writeln("  {")
	w.Writeln("  }")
	w.Writeln("  ")
	w.Writeln("  const T* data() const")
	w.Writeln("  {")
	w.Writeln("    return m_data;")
	w.Writeln("  }")
	w.Writeln("  ")
	w.Writeln("  size_t size() const")
	w.Writeln("  {")
	w.Writeln("    return m_size;")
	w.Writeln("  }")
	w.Writeln("  ")
	w.Writeln("};")
	w.Writeln("")
	if (strings.Compare(ClassIdentifier, NameSpace) != 0) {
		w.Writeln("// declare deprecated class name")
		w.Writeln("template<typename T>")
		w.Writeln("using C%sInputVector = C%sInputVector<T>;", NameSpace, ClassIdentifier)
	}
	return nil
}

func decomposeParamClassNameCPP(paramClassName string) (string, string, error) {
	paramNameSpace, paramClassName, err := decomposeParamClassName(paramClassName)
	if (err != nil) {
		return "", "", err
	}
	if (len(paramNameSpace) >0 ) {
		paramNameSpace = paramNameSpace + "::"
	}
	return paramNameSpace, paramClassName, err
}

func getBindingCppParamType(paramType string, paramClass string, NameSpace string, ClassIdentifier string, isInput bool) string {

	paramNameSpace, paramClassName, _ := decomposeParamClassNameCPP(paramClass)

	cppClassPrefix := "C"
	switch paramType {
	case "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64", "single", "double":
		return fmt.Sprintf("%s_%s", NameSpace, paramType)
	case "string":
		return fmt.Sprintf("std::string")
	case "bool":
		return fmt.Sprintf("bool")
	case "pointer":
		return fmt.Sprintf("%s_pvoid", NameSpace)
	case "basicarray":
		cppBasicType := ""
		switch paramClass {
		case "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64", "single", "double",
			"bool", "pointer":
			cppBasicType = getBindingCppParamType(paramClass, "", NameSpace, ClassIdentifier, isInput)
		default:
			log.Fatal("Invalid parameter type: ", paramClass)
		}
		if isInput {
			return fmt.Sprintf("C%sInputVector<%s>", ClassIdentifier, cppBasicType)
		}
		return fmt.Sprintf("std::vector<%s>", cppBasicType)
	case "structarray":
		typeName := paramNameSpace + "s"+paramClassName
		if isInput {
			return fmt.Sprintf("C%sInputVector<%s>", ClassIdentifier, typeName)
		}
		return fmt.Sprintf("std::vector<%s>", typeName)
	case "enum":
		return fmt.Sprintf(paramNameSpace + "e"+paramClassName)
	case "struct":
		return fmt.Sprintf(paramNameSpace + "s"+paramClassName)
	case "class", "optionalclass":
		if isInput {
			return fmt.Sprintf("%s%s%s%s *", paramNameSpace, cppClassPrefix, ClassIdentifier, paramClassName)
		}
		return fmt.Sprintf("%sP%s%s", paramNameSpace, ClassIdentifier, paramClassName)
	case "functiontype":
		return fmt.Sprintf(paramNameSpace + paramClassName)
	}
	log.Fatal("Invalid parameter type: ", paramType)
	return ""
}

func getBindingCppVariableName(param ComponentDefinitionParam) string {
	switch param.ParamType {
	case "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64":
		return "n" + param.ParamName
	case "string":
		return "s" + param.ParamName
	case "bool":
		return "b" + param.ParamName
	case "single":
		return "f" + param.ParamName
	case "basicarray", "structarray":
		return param.ParamName + "Buffer"
	case "double":
		return "d" + param.ParamName
	case "pointer":
		return "p" + param.ParamName
	case "enum":
		return "e" + param.ParamName
	case "struct":
		return param.ParamName
	case "class", "optionalclass":
		return "p" + param.ParamName
	case "functiontype":
		return fmt.Sprintf("p%s", param.ParamName)
	}

	log.Fatal("Invalid parameter type: ", param.ParamType)

	return ""
}


func buildCppHeader(component ComponentDefinition, w LanguageWriter, NameSpace string, BaseName string, ClassIdentifier string, ExplicitLinking bool) error {
	useCPPTypes := true

	global := component.Global

	cppClassPrefix := "C"
	baseClass := component.baseClass()
	cppBaseClassName := cppClassPrefix + ClassIdentifier + baseClass.ClassName

	sIncludeGuard := "";
	
	if ExplicitLinking {
		sIncludeGuard = "__" + strings.ToUpper(NameSpace) + "_CPPHEADER_DYNAMIC"
	} else {
		sIncludeGuard = "__" + strings.ToUpper(NameSpace) + "_CPPHEADER_IMPLICIT"
	}
	
	if useCPPTypes {
		sIncludeGuard += "_CPP"
	}
	w.Writeln("#ifndef %s", sIncludeGuard)
	w.Writeln("#define %s", sIncludeGuard)
	w.Writeln("")

	w.Writeln("#include \"%s_types.hpp\"", BaseName)
	
	if ExplicitLinking {
		w.Writeln("#include \"%s_dynamic.h\"", BaseName)
	} else {
		w.Writeln("#include \"%s_abi.hpp\"", BaseName)
	}
	
	w.Writeln("")
	for _, subComponent := range(component.ImportedComponentDefinitions) {
		w.Writeln("#include \"%s_dynamic.hpp\"", subComponent.BaseName)
	}
	w.Writeln("")

	w.Writeln("#ifdef _WIN32")
	w.Writeln("#include <windows.h>")
	w.Writeln("#else // _WIN32")
	w.Writeln("#include <dlfcn.h>")
	w.Writeln("#endif // _WIN32")
	w.Writeln("#include <string>")
	w.Writeln("#include <memory>")
	w.Writeln("#include <vector>")
	w.Writeln("#include <exception>")
	w.Writeln("")

	w.Writeln("namespace %s {", NameSpace)
	w.Writeln("")

	buildBindingCPPAllForwardDeclarations(component, w, NameSpace, cppClassPrefix, ClassIdentifier)

	w.Writeln("")
	w.Writeln("/*************************************************************************************************************************")
	w.Writeln(" Class E%sException ", NameSpace)
	w.Writeln("**************************************************************************************************************************/")
	w.Writeln("class E%sException : public std::exception {", NameSpace)
	w.Writeln("protected:")
	w.Writeln("  /**")
	w.Writeln("  * Error code for the Exception.")
	w.Writeln("  */")
	w.Writeln("  %sResult m_errorCode;", NameSpace)
	w.Writeln("  /**")
	w.Writeln("  * Error message for the Exception.")
	w.Writeln("  */")
	w.Writeln("  std::string m_errorMessage;")
	w.Writeln("")
	w.Writeln("public:")
	w.Writeln("  /**")
	w.Writeln("  * Exception Constructor.")
	w.Writeln("  */")
	w.Writeln("  E%sException(%sResult errorCode, const std::string & sErrorMessage)", NameSpace, NameSpace)
	w.Writeln("    : m_errorMessage(\"%s Error \" + std::to_string(errorCode) + \" (\" + sErrorMessage + \")\")", NameSpace)
	w.Writeln("  {")
	w.Writeln("    m_errorCode = errorCode;")
	w.Writeln("  }")
	w.Writeln("")
	w.Writeln("  /**")
	w.Writeln("  * Returns error code")
	w.Writeln("  */")
	w.Writeln("  %sResult getErrorCode() const noexcept", NameSpace)
	w.Writeln("  {")
	w.Writeln("    return m_errorCode;")
	w.Writeln("  }")
	w.Writeln("")
	w.Writeln("  /**")
	w.Writeln("  * Returns error message")
	w.Writeln("  */")
	w.Writeln("  const char* what() const noexcept")
	w.Writeln("  {")
	w.Writeln("    return m_errorMessage.c_str();")
	w.Writeln("  }")
	w.Writeln("")

	w.Writeln("};")

	w.Writeln("")

	err := writeCPPInputVector(w, NameSpace, ClassIdentifier)
	if err != nil {
		return err
	}
	w.Writeln("")

	w.Writeln("/*************************************************************************************************************************")
	w.Writeln(" Class %s%sWrapper ", cppClassPrefix, ClassIdentifier)
	w.Writeln("**************************************************************************************************************************/")

	w.Writeln("class %s%sWrapper {", cppClassPrefix, ClassIdentifier)
	w.Writeln("public:")
	w.Writeln("  ")
	
	if ExplicitLinking {
		w.Writeln("  %s%sWrapper(void* pSymbolLookupMethod)", cppClassPrefix, ClassIdentifier)
		w.Writeln("  {")
		w.Writeln("    CheckError(nullptr, initWrapperTable(&m_WrapperTable));")
		w.Writeln("    CheckError(nullptr, loadWrapperTableFromSymbolLookupMethod(&m_WrapperTable, pSymbolLookupMethod));")
		w.Writeln("    ")
		w.Writeln("    CheckError(nullptr, checkBinaryVersion());")
		w.Writeln("  }")
		w.Writeln("  ")

		w.Writeln("  %s%sWrapper(const std::string &sFileName)", cppClassPrefix, ClassIdentifier)
		w.Writeln("  {")
		w.Writeln("    CheckError(nullptr, initWrapperTable(&m_WrapperTable));")
		w.Writeln("    CheckError(nullptr, loadWrapperTable(&m_WrapperTable, sFileName.c_str()));")
		w.Writeln("    ")
		w.Writeln("    CheckError(nullptr, checkBinaryVersion());")
		w.Writeln("  }")
		w.Writeln("  ")

		w.Writeln("  static P%sWrapper loadLibrary(const std::string &sFileName)", ClassIdentifier)
		w.Writeln("  {")
		w.Writeln("    return std::make_shared<%s%sWrapper>(sFileName);", cppClassPrefix, ClassIdentifier)
		w.Writeln("  }")
		w.Writeln("  ")

		w.Writeln("  static P%sWrapper loadLibraryFromSymbolLookupMethod(void* pSymbolLookupMethod)", ClassIdentifier)
		w.Writeln("  {")
		w.Writeln("    return std::make_shared<%s%sWrapper>(pSymbolLookupMethod);", cppClassPrefix, ClassIdentifier)
		w.Writeln("  }")
		w.Writeln("  ")

		w.Writeln("  ~%s%sWrapper()", cppClassPrefix, ClassIdentifier)
		w.Writeln("  {")
		w.Writeln("    releaseWrapperTable(&m_WrapperTable);")
		w.Writeln("  }")
	} else {
		w.Writeln("  %s%sWrapper()", cppClassPrefix, ClassIdentifier)
		w.Writeln("  {")
		w.Writeln("  }")
		w.Writeln("  ")

		w.Writeln("  ~%s%sWrapper()", cppClassPrefix, ClassIdentifier)
		w.Writeln("  {")
		w.Writeln("  }")

		w.Writeln("  static inline P%sWrapper loadLibrary()", ClassIdentifier)
		w.Writeln("  {")
		w.Writeln("    return std::make_shared<%s%sWrapper>();", cppClassPrefix, ClassIdentifier)
		w.Writeln("  }")
	}
	
	w.Writeln("  ")
	w.Writeln("  inline void CheckError(%s * pBaseClass, %sResult nResult);", cppBaseClassName, NameSpace)
	w.Writeln("")

	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]

		err := writeDynamicCPPMethodDeclaration(method, w, NameSpace, ClassIdentifier, "Wrapper")
		if err != nil {
			return err
		}
	}

	w.Writeln("")
	w.Writeln("private:")
	if ExplicitLinking {
		w.Writeln("  s%sDynamicWrapperTable m_WrapperTable;", NameSpace)
	}

	if len(component.ImportedComponentDefinitions) > 0 {
		w.Writeln("  // Injected Components")
		for _, subComponent := range(component.ImportedComponentDefinitions) {
			subNameSpace := subComponent.NameSpace
			w.Writeln("  %s::PWrapper m_p%sWrapper;", subNameSpace, subNameSpace)
		}
		w.Writeln("")
	}
	
	
	w.Writeln("  ")
	w.Writeln("  %sResult checkBinaryVersion()", NameSpace)
	w.Writeln("  {")
	w.Writeln("    %s_uint32 nMajor, nMinor, nMicro;", NameSpace)
	w.Writeln("    %s(nMajor, nMinor, nMicro);", global.VersionMethod)
	w.Writeln("    if ( (nMajor != %s_VERSION_MAJOR) || (nMinor < %s_VERSION_MINOR) ) {", strings.ToUpper(NameSpace), strings.ToUpper(NameSpace))
	w.Writeln("      return %s_ERROR_INCOMPATIBLEBINARYVERSION;", strings.ToUpper(NameSpace))
	w.Writeln("    }")
	w.Writeln("    return %s_SUCCESS;", strings.ToUpper(NameSpace))
	w.Writeln("  }")

	if ExplicitLinking {
		w.Writeln("  %sResult initWrapperTable(s%sDynamicWrapperTable * pWrapperTable);", NameSpace, NameSpace)
		w.Writeln("  %sResult releaseWrapperTable(s%sDynamicWrapperTable * pWrapperTable);", NameSpace, NameSpace)
		w.Writeln("  %sResult loadWrapperTable(s%sDynamicWrapperTable * pWrapperTable, const char * pLibraryFileName);", NameSpace, NameSpace)
		w.Writeln("  %sResult loadWrapperTableFromSymbolLookupMethod(s%sDynamicWrapperTable * pWrapperTable, void* pSymbolLookupMethod);", NameSpace, NameSpace)
	}
	w.Writeln("")

	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]
		cppClassName := cppClassPrefix + ClassIdentifier + class.ClassName
		w.Writeln("  friend class %s;", cppClassName)
	}

	w.Writeln("")
	w.Writeln("};")
	w.Writeln("")

	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]
		cppClassName := cppClassPrefix + ClassIdentifier + class.ClassName

		cppParentClassName := ""
		inheritanceSpecifier := ""
		if !component.isBaseClass(class) {
			if class.ParentClass == "" {
				cppParentClassName = cppClassPrefix + ClassIdentifier + component.Global.BaseClassName
			} else {
				cppParentClassName = cppClassPrefix + ClassIdentifier+ class.ParentClass
			}
			inheritanceSpecifier = fmt.Sprintf(": public %s ", cppParentClassName)
		}

		w.Writeln("  ")
		w.Writeln("/*************************************************************************************************************************")
		w.Writeln(" Class %s ", cppClassName)
		w.Writeln("**************************************************************************************************************************/")
		w.Writeln("class %s %s{", cppClassName, inheritanceSpecifier)
		w.Writeln("public:")
		w.Writeln("  ")
		if !component.isBaseClass(class) {
			w.Writeln("  /**")
			w.Writeln("  * %s::%s - Constructor for %s class.", cppClassName, cppClassName, class.ClassName)
			w.Writeln("  */")
			w.Writeln("  %s(%s%sWrapper* pWrapper, %sHandle pHandle)", cppClassName, cppClassPrefix, ClassIdentifier, NameSpace)
			if cppParentClassName != "" {
				w.Writeln("    : %s(pWrapper, pHandle)", cppParentClassName)
			}
			w.Writeln("  {")
			w.Writeln("  }")
			w.Writeln("  ")
		} else {
			err = writeDynamicCppBaseClassMethods(component, baseClass, w, NameSpace, BaseName, cppClassPrefix, ClassIdentifier)
			if err != nil {
				return err
			}
		}

		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]
			err = writeDynamicCPPMethodDeclaration(method, w, NameSpace, ClassIdentifier, cppClassName)
			if err != nil {
				return err
			}
		}
		w.Writeln("};")
	}

	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]

		isSpecialFunction, err := CheckHeaderSpecialFunction(method, global);
		if err != nil {
			return err
		}
		
		implementationLines := make([]string, 0)
		if (isSpecialFunction == eSpecialMethodInjection) {
			implementationLines = append(implementationLines, "bool bNameSpaceFound = false;")
			sParamName := "s" + method.Params[0].ParamName
			for _, subComponent := range(component.ImportedComponentDefinitions) {
				theNameSpace := subComponent.NameSpace
				implementationLines = append(implementationLines, fmt.Sprintf("if (%s == \"%s\") {", sParamName, theNameSpace))
				implementationLines = append(implementationLines, fmt.Sprintf("  if (m_p%sWrapper != nullptr) {", theNameSpace))
				implementationLines = append(implementationLines, fmt.Sprintf("    throw E%sException(%s_ERROR_COULDNOTLOADLIBRARY, \"Library with namespace \" + %s + \" is already registered.\");", NameSpace, strings.ToUpper(NameSpace), sParamName) )
				implementationLines = append(implementationLines, fmt.Sprintf("  }"))

				implementationLines = append(implementationLines, fmt.Sprintf("  m_p%sWrapper = %s::CWrapper::loadLibraryFromSymbolLookupMethod(p%s);", theNameSpace, theNameSpace, method.Params[1].ParamName))
				implementationLines = append(implementationLines, fmt.Sprintf("  bNameSpaceFound = true;"))
				implementationLines = append(implementationLines, fmt.Sprintf("}"))
			}
			implementationLines = append(implementationLines, "if (!bNameSpaceFound)")
			implementationLines = append(implementationLines, fmt.Sprintf("  throw E%sException(%s_ERROR_COULDNOTLOADLIBRARY, \"Unknown namespace \" + %s);", NameSpace, strings.ToUpper(NameSpace), sParamName ))
		}

		err = writeDynamicCPPMethod(method, w, NameSpace, ClassIdentifier, "Wrapper", implementationLines, true, true, false, useCPPTypes, ExplicitLinking)
		if err != nil {
			return err
		}

	}

	w.Writeln("  ")
	w.Writeln("  inline void C%sWrapper::CheckError(%s * pBaseClass, %sResult nResult)", ClassIdentifier, cppBaseClassName, NameSpace)
	w.Writeln("  {")
	w.Writeln("    if (nResult != 0) {")
	w.Writeln("      std::string sErrorMessage;")
	w.Writeln("      if (pBaseClass != nullptr) {")
	w.Writeln("        %s(pBaseClass, sErrorMessage);", component.Global.ErrorMethod)
	w.Writeln("      }")
	w.Writeln("      throw E%sException(nResult, sErrorMessage);", NameSpace)
	w.Writeln("    }")
	w.Writeln("  }")
	w.Writeln("  ")

	w.Writeln("")
	
	if ExplicitLinking {
		w.Writeln("  inline %sResult %s%sWrapper::initWrapperTable(s%sDynamicWrapperTable * pWrapperTable)", NameSpace, cppClassPrefix, ClassIdentifier, NameSpace)
		w.Writeln("  {")

		w.AddIndentationLevel(2)
		buildDynamicCInitTableCode(component, w, NameSpace, BaseName, false)
		w.AddIndentationLevel(-2)

		w.Writeln("  }")
		w.Writeln("")

		w.Writeln("  inline %sResult %s%sWrapper::releaseWrapperTable(s%sDynamicWrapperTable * pWrapperTable)", NameSpace, cppClassPrefix, ClassIdentifier, NameSpace)
		w.Writeln("  {")

		w.AddIndentationLevel(2)
		buildDynamicCReleaseTableCode(component, w, NameSpace, BaseName, "initWrapperTable", false)
		w.AddIndentationLevel(-2)

		w.Writeln("  }")
		w.Writeln("")

		w.Writeln("  inline %sResult %s%sWrapper::loadWrapperTable(s%sDynamicWrapperTable * pWrapperTable, const char * pLibraryFileName)", NameSpace, cppClassPrefix, ClassIdentifier, NameSpace)
		w.Writeln("  {")

		w.AddIndentationLevel(2)
		buildDynamicCLoadTableCode(component, w, NameSpace, BaseName, false)
		w.AddIndentationLevel(-2)

		w.Writeln("  }")
		

		w.Writeln("")
		w.Writeln("  inline %sResult %s%sWrapper::loadWrapperTableFromSymbolLookupMethod(s%sDynamicWrapperTable * pWrapperTable, void* pSymbolLookupMethod)", NameSpace, cppClassPrefix, ClassIdentifier, NameSpace)
		w.Writeln("{")
	
		w.AddIndentationLevel(2)
		buildDynamicCLoadTableFromSymbolLookupMethodCode(component, w, NameSpace, BaseName, false)
		w.AddIndentationLevel(-2)
	
		w.Writeln("}")
		w.Writeln("")

		w.Writeln("  ")
	}


	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]
		cppClassName := cppClassPrefix + class.ClassName
		w.Writeln("  ")
		w.Writeln("  /**")
		w.Writeln("   * Method definitions for class %s", cppClassName)
		w.Writeln("   */")
		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]
			err := writeDynamicCPPMethod(method, w, NameSpace, ClassIdentifier, class.ClassName, make([]string,0), false, true, false, useCPPTypes, ExplicitLinking)
			if err != nil {
				return err
			}
		}
	}

	w.Writeln("")

	w.Writeln("} // namespace %s", NameSpace)
	w.Writeln("")

	w.Writeln("#endif // %s", sIncludeGuard)
	w.Writeln("")

	return nil
}

// BuildBindingCppExplicit builds headeronly C++-bindings of a library's API in form of expliclty loaded function handles.
func BuildBindingCppExplicit(component ComponentDefinition, outputFolder string, outputFolderExample string,
	indentString string, ClassIdentifier string) error {
	forceRecreation := false
	ExplicitLinking := true
	namespace := component.NameSpace
	libraryname := component.LibraryName
	baseName := component.BaseName

	DynamicCHeader := path.Join(outputFolder, baseName+"_dynamic.h")
	log.Printf("Creating \"%s\"", DynamicCHeader)
	dynhfile, err := CreateLanguageFile(DynamicCHeader, indentString)
	if err != nil {
		return err
	}
	dynhfile.WriteCLicenseHeader(component,
		fmt.Sprintf("This is an autogenerated C++-Header file in order to allow an easy\n use of %s", libraryname),
		true)
	err = buildDynamicCCPPHeader(component, dynhfile, namespace, baseName, true, true)
	if err != nil {
		return err
	}

	DynamicCppHeader := path.Join(outputFolder, baseName+"_dynamic.hpp")
	log.Printf("Creating \"%s\"", DynamicCppHeader)
	dynhppfile, err := CreateLanguageFile(DynamicCppHeader, indentString)
	if err != nil {
		return err
	}
	dynhppfile.WriteCLicenseHeader(component,
		fmt.Sprintf("This is an autogenerated C++-Header file in order to allow an easy\n use of %s", libraryname),
		true)
	err = buildCppHeader(component, dynhppfile, namespace, baseName, ClassIdentifier, ExplicitLinking)
	if err != nil {
		return err
	}

	if len(outputFolderExample) > 0 {
		DynamicCPPExample := path.Join(outputFolderExample, namespace+"_example"+".cpp")
		if forceRecreation || !FileExists(DynamicCPPExample) {
			log.Printf("Creating \"%s\"", DynamicCPPExample)
			dyncppexamplefile, err := CreateLanguageFile(DynamicCPPExample, indentString)
			if err != nil {
				return err
			}
			dyncppexamplefile.WriteCLicenseHeader(component,
				fmt.Sprintf("This is an autogenerated C++ application that demonstrates the\n usage of the Dynamic C++ bindings of %s", libraryname),
				true)
			buildDynamicCppExample(component, dyncppexamplefile, outputFolder, ClassIdentifier, ExplicitLinking)
		} else {
			log.Printf("Omitting recreation of C++Dynamic example file \"%s\"", DynamicCPPExample)
		}

		DynamicCPPCMake := path.Join(outputFolderExample, "CMakeLists.txt")
		if forceRecreation || !FileExists(DynamicCPPCMake) {
			log.Printf("Creating \"%s\"", DynamicCPPCMake)
			dyncppcmake, err := CreateLanguageFile(DynamicCPPCMake, "  ")
			if err != nil {
				return err
			}
			dyncppcmake.WriteCMakeLicenseHeader(component,
				fmt.Sprintf("This is an autogenerated CMake Project that demonstrates the\n usage of the Dynamic C++ bindings of %s", libraryname),
				true)
				buildCppDynamicExampleCMake(component, dyncppcmake, outputFolder, outputFolderExample, ExplicitLinking)
		} else {
			log.Printf("Omitting recreation of C++Dynamic example file \"%s\"", DynamicCPPCMake)
		}
	}

	return nil
}


func buildDynamicCppExample(componentdefinition ComponentDefinition, w LanguageWriter, outputFolder string, ClassIdentifier string, ExplicitLinking bool) error {
	NameSpace := componentdefinition.NameSpace
	BaseName := componentdefinition.BaseName

	w.Writeln("#include <iostream>")
	if (ExplicitLinking) {
		w.Writeln("#include \"%s_dynamic.hpp\"", strings.ToLower(BaseName))
	} else {
		w.Writeln("#include \"%s_implicit.hpp\"", strings.ToLower(BaseName))
	}
	
	w.Writeln("")
	w.Writeln("")

	w.Writeln("int main()")
	w.Writeln("{")
	w.Writeln("  try")
	w.Writeln("  {")
	if (ExplicitLinking) {
		w.Writeln("    std::string libpath = (\"\"); // TODO: put the location of the %s-library file here.", NameSpace)
		w.Writeln("    auto wrapper = %s::C%sWrapper::loadLibrary(libpath + \"/%s.\"); // TODO: add correct suffix of the library", NameSpace, ClassIdentifier, BaseName,)
	} else {
		w.Writeln("    auto wrapper = %s::C%sWrapper::loadLibrary();", NameSpace, ClassIdentifier)
	}
	w.Writeln("    %s_uint32 nMajor, nMinor, nMicro;", NameSpace)
	w.Writeln("    wrapper->%s(nMajor, nMinor, nMicro);", componentdefinition.Global.VersionMethod)
	w.Writeln("    std::cout << \"%s.Version = \" << nMajor << \".\" << nMinor << \".\" << nMicro;", NameSpace)
	if len(componentdefinition.Global.PrereleaseMethod)>0 {
		w.Writeln("    std::string sPreReleaseInfo;")
		w.Writeln("    if (wrapper->%s(sPreReleaseInfo)) {", componentdefinition.Global.PrereleaseMethod)
		w.Writeln("      std::cout << \"-\" << sPreReleaseInfo;")
		w.Writeln("    }")
	}
	if len(componentdefinition.Global.BuildinfoMethod)>0 {
		w.Writeln("    std::string sBuildInfo;")
		w.Writeln("    if (wrapper->%s(sBuildInfo)) {", componentdefinition.Global.BuildinfoMethod)
		w.Writeln("      std::cout << \"+\" << sBuildInfo;")
		w.Writeln("    }")
	}
	w.Writeln("    std::cout << std::endl;")
	w.Writeln("  }")
	w.Writeln("  catch (std::exception &e)")
	w.Writeln("  {")
	w.Writeln("    std::cout << e.what() << std::endl;")
	w.Writeln("    return 1;")
	w.Writeln("  }")
	w.Writeln("  return 0;")
	w.Writeln("}")
	w.Writeln("")

	return nil
}

func buildCppDynamicExampleCMake(componentdefinition ComponentDefinition, w LanguageWriter, outputFolder string, outputFolderExample string, ExplicitLinking bool) error {
	NameSpace := componentdefinition.NameSpace
	BaseName := componentdefinition.BaseName

	w.Writeln("cmake_minimum_required(VERSION 3.5)")

	w.Writeln("")
	projectName := fmt.Sprintf("%sExample_CPP", NameSpace)
	if ExplicitLinking {
		projectName += "Dynamic"
	} else {
		projectName += "Implicit"
	}

	bindingFolder, err := filepath.Rel(outputFolderExample, outputFolder)
	if (err != nil) {
		return err
	}
	bindingFolder = strings.Replace(bindingFolder, "\\", "/", -1)
	cmakeBindingFolder := "CMAKE_CURRENT_BINDING_DIR"
	w.Writeln("set(%s ${CMAKE_CURRENT_SOURCE_DIR}/%s)", cmakeBindingFolder, bindingFolder)

	w.Writeln("project(%s)", projectName)
	w.Writeln("set(CMAKE_CXX_STANDARD 11)")
	w.Writeln("add_executable(%s \"${CMAKE_CURRENT_SOURCE_DIR}/%s_example.cpp\")", projectName, NameSpace)
	if (ExplicitLinking || (len(componentdefinition.ImportedComponentDefinitions)>0)) {
		w.Writeln("if (UNIX)")
		w.Writeln("  target_link_libraries(%s ${CMAKE_DL_LIBS})", projectName)
		w.Writeln("endif (UNIX)")
	} else {
		linkFolder := "${CMAKE_CURRENT_SOURCE_DIR}/../../Implementations/*/*/*"
		w.Writeln("find_library(%sLOCATION %s \"%s\")", strings.ToUpper(BaseName), BaseName, linkFolder)
		w.Writeln("target_link_libraries(%s ${%sLOCATION})", projectName, strings.ToUpper(BaseName))
	}
	w.Writeln("target_include_directories(%s PRIVATE \"${%s}\")", projectName, cmakeBindingFolder)
	for _, subComponent := range(componentdefinition.ImportedComponentDefinitions) {
		w.Writeln("target_include_directories(%s PRIVATE \"${%s}/../../../%s_component/Bindings/CppDynamic\")", projectName, cmakeBindingFolder, subComponent.NameSpace)
	}
	return nil
}


func buildDynamicCExample(componentdefinition ComponentDefinition, w LanguageWriter, outputFolder string, ClassIdentifier string) error {
	NameSpace := componentdefinition.NameSpace
	BaseName := componentdefinition.BaseName

	w.Writeln("#include <stdio.h>")
	w.Writeln("#include <stdlib.h>")
	w.Writeln("#include \"%s_dynamic.h\"", strings.ToLower(BaseName))
	
	w.Writeln("")
	w.Writeln("")
	w.Writeln("void releaseWrapper(s%sDynamicWrapperTable* pWrapperTable) {", NameSpace)
	w.Writeln("  %sResult eResult = Release%sWrapperTable(pWrapperTable);", NameSpace, NameSpace)
	w.Writeln("  if (%s_SUCCESS != eResult) {", strings.ToUpper(NameSpace))
	w.Writeln("    printf_s(\"Failed to release wrapper table\\n\");")
	w.Writeln("  }")
	w.Writeln("}")
	w.Writeln("")

	w.Writeln("int main()")
	w.Writeln("{")
	w.Writeln("  // TODO: put a path to %s binary file here:", componentdefinition.LibraryName)
	w.Writeln("  const char* libpath = \"\";")
	w.Writeln("  s%sDynamicWrapperTable sWrapperTable;", NameSpace)
	w.Writeln("  %sResult eResult = %s_SUCCESS;", NameSpace, strings.ToUpper(NameSpace))
	w.Writeln("  ")
	w.Writeln("  eResult = Init%sWrapperTable(&sWrapperTable);", NameSpace)
	w.Writeln("  if (%s_SUCCESS != eResult) {", strings.ToUpper(NameSpace))
	w.Writeln("    printf_s(\"Failed to initialize wrapper table\\n\");")
	w.Writeln("    return eResult;")
	w.Writeln("  }")
	w.Writeln("  ")
	w.Writeln("  eResult = Load%sWrapperTable(&sWrapperTable, libpath);", NameSpace)
	w.Writeln("  if (%s_SUCCESS != eResult) {", strings.ToUpper(NameSpace))
	w.Writeln("    printf_s(\"Failed to load %s-binary\\n\");", BaseName)
	w.Writeln("    return eResult;")
	w.Writeln("  }")

	w.Writeln("  %s_uint32 nMajor, nMinor, nMicro;", NameSpace)
	w.Writeln("  eResult = sWrapperTable.m_%s(&nMajor, &nMinor, &nMicro);", componentdefinition.Global.VersionMethod)
	w.Writeln("  if (%s_SUCCESS != eResult) {", strings.ToUpper(NameSpace))
	w.Writeln("    printf_s(\"Failed to get version\\n\");")
	w.Writeln("    releaseWrapper(&sWrapperTable);")
	w.Writeln("    return eResult;")
	w.Writeln("  }")
	w.Writeln("  printf_s(\"%s.Version = %%d.%%d.%%d\", nMajor, nMinor, nMicro);", NameSpace)
	w.Writeln("  ")
	if len(componentdefinition.Global.PrereleaseMethod)>0 || len(componentdefinition.Global.BuildinfoMethod)>0 {
		w.Writeln("  %s_uint32 nBufferRequired = 0;", NameSpace)
		w.Writeln("  %s_uint8* theString = NULL;", NameSpace)
		w.Writeln("  bool bHasInfo = false;", NameSpace)
	}
	if len(componentdefinition.Global.PrereleaseMethod)>0 {
		w.Writeln("  eResult = sWrapperTable.m_%s(&bHasInfo, 0, &nBufferRequired, theString);", componentdefinition.Global.PrereleaseMethod)
		w.Writeln("  if (%s_SUCCESS != eResult) {", strings.ToUpper(NameSpace))
		w.Writeln("	   releaseWrapper(&sWrapperTable);")
		w.Writeln("    return eResult;")
		w.Writeln("  }")
		w.Writeln("  if (bHasInfo && (nBufferRequired > 0)) {")
		w.Writeln("    theString = malloc(sizeof(%s_uint8)*(nBufferRequired+1));", NameSpace)
		w.Writeln("    theString[nBufferRequired] = 0;")
		w.Writeln("    eResult = sWrapperTable.m_%s(&bHasInfo, nBufferRequired + 1, &nBufferRequired, theString);", componentdefinition.Global.PrereleaseMethod)
		w.Writeln("    if (%s_SUCCESS != eResult) {", strings.ToUpper(NameSpace))
		w.Writeln("      printf_s(\"Failed to get prerelease information\\n\"");
		w.Writeln("      releaseWrapper(&sWrapperTable);")
		w.Writeln("      free(theString);")
		w.Writeln("      return eResult;")
		w.Writeln("    }")
		w.Writeln("    printf_s(\"-%%s\", theString);")
		w.Writeln("    free(theString);")
		w.Writeln("    theString = NULL;")
		w.Writeln("  }")
		w.Writeln("  ")
	}
	if len(componentdefinition.Global.BuildinfoMethod)>0 {
		w.Writeln("  eResult = sWrapperTable.m_%s(&bHasInfo, 0, &nBufferRequired, theString);", componentdefinition.Global.BuildinfoMethod)
		w.Writeln("  if (%s_SUCCESS != eResult) {", strings.ToUpper(NameSpace))
		w.Writeln("	   releaseWrapper(&sWrapperTable);")
		w.Writeln("    return eResult;")
		w.Writeln("  }")
		w.Writeln("  if (bHasInfo && (nBufferRequired > 0)) {")
		w.Writeln("    theString = malloc(sizeof(%s_uint8)*(nBufferRequired+1));", NameSpace)
		w.Writeln("    theString[nBufferRequired] = 0;")
		w.Writeln("    eResult = sWrapperTable.m_%s(&bHasInfo, nBufferRequired + 1, &nBufferRequired, theString);", componentdefinition.Global.BuildinfoMethod)
		w.Writeln("    if (%s_SUCCESS != eResult) {", strings.ToUpper(NameSpace))
		w.Writeln("      printf_s(\"Failed to get build information\\n\"");
		w.Writeln("      releaseWrapper(&sWrapperTable);")
		w.Writeln("      free(theString);")
		w.Writeln("      return eResult;")
		w.Writeln("    }")
		w.Writeln("    printf_s(\"+%%s\", theString);")
		w.Writeln("    free(theString);")
		w.Writeln("    theString = NULL;")
		w.Writeln("  }")
		w.Writeln("  ")
	}
	w.Writeln("  printf_s(\"\\n\");")
	w.Writeln("  ")
	w.Writeln("  eResult = Release%sWrapperTable(&sWrapperTable);", NameSpace)
	w.Writeln("  if (%s_SUCCESS != eResult) {", strings.ToUpper(NameSpace))
	w.Writeln("    printf_s(\"Failed to release wrapper table\\n\");")
	w.Writeln("    return eResult;")
	w.Writeln("  }")
	w.Writeln("  ")
	w.Writeln("  return 0;")
	w.Writeln("}")
	w.Writeln("")
	return nil
}

func buildCDynamicExampleCMake(componentdefinition ComponentDefinition, w LanguageWriter, outputFolder string, outputFolderExample string, ExplicitLinking bool) error {
	NameSpace := componentdefinition.NameSpace
	BaseName := componentdefinition.BaseName

	w.Writeln("cmake_minimum_required(VERSION 3.5)")

	w.Writeln("")
	projectName := fmt.Sprintf("%sExample_C", NameSpace)
	if ExplicitLinking {
		projectName += "Dynamic"
	} else {
		projectName += "Implicit"
	}

	bindingFolder, err := filepath.Rel(outputFolderExample, outputFolder)
	if (err != nil) {
		return err
	}
	bindingFolder = strings.Replace(bindingFolder, "\\", "/", -1)
	cmakeBindingFolder := "CMAKE_CURRENT_BINDING_DIR"
	w.Writeln("set(%s ${CMAKE_CURRENT_SOURCE_DIR}/%s)", cmakeBindingFolder, bindingFolder)

	w.Writeln("")
	w.Writeln("project(%s C)", projectName)
	w.Writeln("")
	bindingSource := ""
	if (ExplicitLinking) {
		bindingSource = "\"${"+cmakeBindingFolder+"}/"+strings.ToLower(NameSpace)+"_dynamic.cc\""
		w.Writeln("SET_SOURCE_FILES_PROPERTIES(%s PROPERTIES LANGUAGE C)", bindingSource)
		bindingSource += "\n"
	}

	w.Writeln("add_executable(%s\n  \"${CMAKE_CURRENT_SOURCE_DIR}/%s_example.c\"\n  %s\n)", projectName, NameSpace, bindingSource)
	w.Writeln("set_property(TARGET %s PROPERTY C_STANDARD 99)", projectName)
	if (ExplicitLinking) {
		w.Writeln("if (UNIX)")
		w.Writeln("  target_link_libraries(%s ${CMAKE_DL_LIBS})", projectName)
		w.Writeln("endif (UNIX)")
	} else {
		linkFolder := strings.Replace(outputFolder, string(filepath.Separator), "/", -2) + "/../../Implementations/*/*/*"
		w.Writeln("find_library(%sLOCATION %s \"%s\")", strings.ToUpper(BaseName), BaseName, linkFolder)
		w.Writeln("target_link_libraries(%s ${%sLOCATION})", projectName, strings.ToUpper(BaseName))
	}
	w.Writeln("target_include_directories(%s PRIVATE \"${%s}\")", projectName, cmakeBindingFolder)
	return nil
}
