// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief B -> c l nu moments
  class BELLE_2021_I1917200 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(BELLE_2021_I1917200);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // projections
      declare(UnstableParticles(Cuts::abspid==511 ||
    				Cuts::abspid==521),"UFS");
      // histos
      for(unsigned int il=0;il<2;++il) {
	for(unsigned int ix=0;ix<4;++ix) {
	  book(_p[il][ix],4*il+1+ix,1,1);
	}
      }
    }

    void findDecayProducts(Particle parent, Particles & em, Particles & ep,
			   Particles & nue, Particles & nueBar, bool & charm) {
      for(const Particle & p : parent.children()) {
	if(PID::isCharmHadron(p.pid())) {
	  charm=true;
	}
	else if(p.pid() == PID::EMINUS || p.pid()==PID::MUON) {
	  em.push_back(p);
	}
	else if(p.pid() == PID::EPLUS || p.pid()==PID::ANTIMUON) {
	  ep.push_back(p);
	}
	else if(p.pid() == PID::NU_E  || p.pid()==PID::NU_MU) {
	  nue.push_back(p);
	}
	else if(p.pid() == PID::NU_EBAR || p.pid()==PID::NU_MUBAR) {
	  nueBar.push_back(p);
	}
	else if(PID::isBottomHadron(p.pid())) {
	  findDecayProducts(p,em,ep,nue,nueBar,charm);
	}
	else if(!PID::isHadron(p.pid())) {
	  findDecayProducts(p,em,ep,nue,nueBar,charm);
	}
      }
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {
      // find and loop over Upslion(4S)
      for (const Particle& p : apply<UnstableParticles>(event, "UFS").particles()) {
	if(p.children().empty() ||
	   (p.children().size()==1 && p.children()[1].abspid()==p.abspid()))
	  continue;
	// find decay products
	bool charm = false;
	Particles em,ep,nue,nueBar;
	findDecayProducts(p,em,ep,nue,nueBar,charm);
	if(!charm) continue;
	FourMomentum pl,pnu;
	unsigned int il = 0;
	if(em.size()==1 && nueBar.size()==1 && em[0].pid()+1==-nueBar[0].pid()) {
	  pl  = em[0].momentum();
	  pnu = nueBar[0].momentum();
	  if(em[0].abspid()==13) il=1;
	}
	else if(ep.size()==1 && nue.size()==1 && nue[0].pid()==-ep[0].pid()+1) {
	  pl  = ep[0].momentum();
	  pnu = nue[0].momentum();
	  if(ep[0].abspid()==13) il=1;
       	}
	else
	  continue;
	double q2 = (pl+pnu).mass2();
	vector<double> q2n(4);
	for(unsigned int ix=0;ix<4;++ix) q2n[ix] = pow(q2,1+ix);
	if(q2<3.0) continue;
	double q2cut=3.0;
	for(unsigned int ibin=0;ibin<15;++ibin) {
	  if(q2>q2cut) for(unsigned int ix=0;ix<4;++ix) _p[il][ix]->fill(q2cut,q2n[ix]);
	  else break;
	  q2cut+=0.5;
	}
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      for(unsigned int il=0;il<2;++il) {
	Scatter2DPtr tmp[3];
	for(unsigned int ix=0;ix<3;++ix) book(tmp[ix],3*il+9+ix,1,1);
	for(unsigned int iy=0;iy<_p[il][0]->numBins();++iy) {
	  double x  = 0.5*(_p[il][0]->xEdges()[iy+1]+_p[il][0]->xEdges()[iy]);
	  double dx = 0.5*(_p[il][0]->xEdges()[iy+1]-_p[il][0]->xEdges()[iy]);
	  double q2  = _p[il][0]->bins()[iy].mean()  ,  q4 = _p[il][1]->bins()[iy].mean(),
	    q6  = _p[il][2]->bins()[iy].mean()  ,  q8 = _p[il][3]->bins()[iy].mean();
	  double dq2 = _p[il][0]->bins()[iy].stdErr(), dq4 = _p[il][1]->bins()[iy].stdErr(),
	    dq6 = _p[il][2]->bins()[iy].stdErr(), dq8 = _p[il][3]->bins()[iy].stdErr();
	  // q4 case
	  double value = q4 - sqr(q2);
	  double error =sqr(dq4) + 4*sqr(dq2*q2);
	  tmp[0]->addPoint(x,value,make_pair(dx,dx),make_pair(error,error));
	  // q6 case
	  value = q6 + q2*(2*sqr(q2) - 3*q4);
	  error = sqr(dq6) + 9*(sqr(dq4*q2) + sqr(dq2)*sqr(-2*sqr(q2) + q4));
	  tmp[1]->addPoint(x,value,make_pair(dx,dx),make_pair(error,error));
	  // q8 case
	  value = q8 + q2*(-3*q2*sqr(q2) + 6*q2*q4 - 4*q6);
	  error = sqr(dq8) + 16*sqr(dq6*q2) + 36*sqr(dq4*sqr(q2)) + 16*sqr(dq2*(3*q2*sqr(q2) - 3*q2*q4 + q6));
	  tmp[2]->addPoint(x,value,make_pair(dx,dx),make_pair(error,error));
	}
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    Profile1DPtr _p[2][4];
    /// @}


  };


  RIVET_DECLARE_PLUGIN(BELLE_2021_I1917200);

}
