/* wmpasman
 * Copyright © 1999-2014  Brad Jorsch <anomie@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "config.h"

#include <stdarg.h>
#ifdef HAVE_PRCTL
#include <sys/prctl.h>
#else
#include <sys/resources.h>
#endif

#include <gtk/gtk.h>

#define SECRET_API_SUBJECT_TO_CHANGE "Grr..."
#include <libsecret/secret.h>

#define KEYRING_NAME "wmpasman"

#include "wmpasman.h"
#include "die.h"
#include "dock.h"
#include "keyring.h"

static SecretService *service = NULL;
static SecretCollection *collection = NULL;
password_t current_password = NULL;
static SecretItem *goto_item_on_reload = NULL;

static const SecretSchema schema = {
    .name = "net.sourceforge.users.anomie.wmpasman",
    .flags = SECRET_SCHEMA_NONE,
    .attributes = {
        { "line1", SECRET_SCHEMA_ATTRIBUTE_STRING },
        { "line2", SECRET_SCHEMA_ATTRIBUTE_STRING },
        { NULL, 0 }
    }
};

typedef struct {
    void (*done)(GError *);
    GCancellable *cancel;
} *keyring_cb_data;

typedef struct {
    GCallback next;
    gchar *line1, *line2;
    SecretValue *v;
    GList *lock;
    keyring_cb_data cb;
    GError *err;
} *password_edit_data;

static gboolean do_secret_service_get(gpointer);
static void get_collection_from_service(keyring_cb_data);
static void collections_changed_cb(GObject *, GParamSpec *, gpointer);
static void collection_items_changed_cb(SecretCollection *, GParamSpec *, gpointer);
static void collection_locked_changed_cb(SecretCollection *, GParamSpec *, gpointer);
static void item_attributes_changed_cb(SecretItem *, GParamSpec *, gpointer);

/*** Utility functions ***/

static void prefix_error(GError **err, const gchar *format, ...) G_GNUC_PRINTF(2, 3);
static void prefix_error(GError **err, const gchar *format, ...) {
    va_list ap;
    va_start(ap, format);
    gchar *prefix = g_strdup_vprintf(format, ap);
    va_end (ap);

    if (!*err) {
        g_set_error(err, APP_GENERIC_ERROR, 0, "%s", prefix);
    } else {
        gchar *oldstring = (*err)->message;
        (*err)->message = g_strconcat(prefix, ": ", oldstring, NULL);
        g_free(oldstring);
    }

    g_free(prefix);
}

static const gchar *item_sortkey(SecretItem *a) {
    const gchar *ret = g_object_get_data(G_OBJECT(a), "sortkey");
    if (!ret) {
        GHashTable *aa = secret_item_get_attributes(a);
        gchar *line1 = g_hash_table_lookup(aa, "line1");
        gchar *line2 = g_hash_table_lookup(aa, "line2");
        if (!line1 || !line2) {
            // Item probably isn't actually loaded yet. Don't cache the
            // sortkey.
            g_hash_table_unref(aa);
            return "";
        }
        warn(DEBUG_DEBUG, "Calculating sortkey for %s / %s", line1, line2);
        gchar *tmp1 = g_strjoin("\n", line1, line2, NULL);
        gchar *tmp2 = g_utf8_casefold(tmp1, -1);
        ret = g_utf8_collate_key(tmp2, -1);
        g_object_set_data_full(G_OBJECT(a), "sortkey", (gpointer)ret, g_free);
        g_free(tmp2);
        g_free(tmp1);
        g_hash_table_unref(aa);

        // Listen for change events, which would change the sort key, which
        // would require resorting.
        if (!g_object_get_data(G_OBJECT(a), "signal_attached")) {
            warn(DEBUG_DEBUG, "Connecting signal");
            g_signal_connect(G_OBJECT(a), "notify::attributes", G_CALLBACK(item_attributes_changed_cb), NULL);
            g_object_set_data(G_OBJECT(a), "signal_attached", GINT_TO_POINTER(1));
        }
    }
    return ret;
}

static gint item_compare(SecretItem *a, SecretItem *b) {
    const gchar *aa = item_sortkey(a);
    const gchar *bb = item_sortkey(b);
    return g_strcmp0(aa, bb);
}

static keyring_cb_data make_keyring_cb_data(void (*done)(GError *), GCancellable *cancel) {
    keyring_cb_data d = g_malloc(sizeof(*d));
    d->done = done;
    if (cancel) {
        d->cancel = g_object_ref(cancel);
    } else {
        d->cancel = NULL;
    }
    return d;
}

static void keyring_cb_data_done(keyring_cb_data d, GError *err) {
    if (err) {
        warn(DEBUG_WARN, "%s", err->message);
    }
    if (d) {
        d->done(err);
        if (d->cancel) {
            g_object_unref(d->cancel);
        }
        g_free(d);
    }
}

static password_edit_data make_password_edit_data(void (*done)(GError *), GCancellable *cancel) {
    password_edit_data d = g_malloc(sizeof(*d));
    d->line1 = NULL;
    d->line2 = NULL;
    d->v = NULL;
    d->lock = NULL;
    d->cb = make_keyring_cb_data(done, cancel);
    return d;
}

static GHashTable *password_edit_data_to_hashtable(password_edit_data d) {
    GHashTable *aa = g_hash_table_new(g_str_hash, g_str_equal);
    g_hash_table_insert(aa, "line1", d->line1);
    g_hash_table_insert(aa, "line2", d->line2);
    return aa;
}

static void free_password_edit_data(password_edit_data d, GError *err) {
    g_free(d->line1);
    g_free(d->line2);
    if (d->v) {
        secret_value_unref(d->v);
    }
    if (d->lock) {
        g_list_free_full(d->lock, g_object_unref);
    }
    keyring_cb_data_done(d->cb, err);
    g_free(d);
}

static void free_current_password() {
    if (current_password) {
        g_free((void *)current_password->line1);
        g_free((void *)current_password->line2);
        g_list_free_full(g_list_first(current_password->data), g_object_unref);
        g_clear_pointer(&current_password, g_free);
        app_update_state();
    }
}

static void update_current_password() {
    if (current_password) {
        GHashTable *aa = secret_item_get_attributes(SECRET_ITEM(current_password->data->data));
        g_free((void *)current_password->line1);
        current_password->line1 = g_strdup(g_hash_table_lookup(aa, "line1"));
        g_free((void *)current_password->line2);
        current_password->line2 = g_strdup(g_hash_table_lookup(aa, "line2"));
        g_hash_table_unref(aa);
        app_update_state();
    }
}

static void free_collection() {
    free_current_password();
    if (collection) {
        g_signal_handlers_disconnect_by_func(G_OBJECT(collection), G_CALLBACK(collections_changed_cb), NULL);
        g_signal_handlers_disconnect_by_func(G_OBJECT(collection), G_CALLBACK(collection_items_changed_cb), NULL);
        g_signal_handlers_disconnect_by_func(G_OBJECT(collection), G_CALLBACK(collection_locked_changed_cb), NULL);
        g_clear_pointer(&collection, g_object_unref);
    }
}

/*** All sorts of callbacks ***/

static void load_items_from_collection() {
    warn(DEBUG_DEBUG, "Reloading items from collection");

    SecretItem *item = NULL;
    if (current_password) {
        item = SECRET_ITEM(current_password->data->data);
        g_object_ref(item);
    }

    GList *items;
    if (secret_collection_get_locked(collection)) {
        // We don't load the list of items when locked.
        items = NULL;
    } else {
        items = secret_collection_get_items(collection);
    }

    if (!items) {
        free_current_password();
        g_clear_pointer(&item, g_object_unref);
        return;
    }
    items = g_list_sort(items, (GCompareFunc)item_compare);
    if (warn_level >= DEBUG_DEBUG) {
        fprintf(stderr, "Item list:");
        for (GList *p = items; p; p = p->next) {
            SecretItem *i = SECRET_ITEM(p->data);
            GHashTable *aa = secret_item_get_attributes(i);
            fprintf(stderr, " <%s / %s>",
                (char *)g_hash_table_lookup(aa, "line1"),
                (char *)g_hash_table_lookup(aa, "line2")
            );
            g_hash_table_unref(aa);
        }
        fprintf(stderr, "\n");
    }

    if (current_password) {
        g_list_free_full(g_list_first(current_password->data), g_object_unref);
        current_password->data = NULL;
    } else {
        current_password = g_malloc(sizeof(*current_password));
        current_password->line1 = NULL;
        current_password->line2 = NULL;
        current_password->data = NULL;
    }

    GList *cur = NULL;
    if (goto_item_on_reload) {
        const gchar *path = g_dbus_proxy_get_object_path(G_DBUS_PROXY(goto_item_on_reload));
        for (GList *p = items; p && !cur; p = p->next) {
            const gchar *path2 = g_dbus_proxy_get_object_path(G_DBUS_PROXY(p->data));
            if (!g_strcmp0(path, path2)) {
                cur = p;
            }
        }
        if (cur) {
            g_clear_object(&item);
            g_clear_object(&goto_item_on_reload);
        }
    }
    if (item) {
        // 1. Keep the current item by path, if possible.
        const gchar *path = g_dbus_proxy_get_object_path(G_DBUS_PROXY(item));
        for (GList *p = items; p && !cur; p = p->next) {
            const gchar *path2 = g_dbus_proxy_get_object_path(G_DBUS_PROXY(p->data));
            if (!g_strcmp0(path, path2)) {
                cur = p;
            }
        }

        // 2. Keep the current item by line1/line2, if possible.
        // 3. Find the first item preceeding the old line1/line2.
        if (!cur) {
            GList *cur2 = items;
            GHashTable *aa = secret_item_get_attributes(item);
            for (GList *p = items; p && !cur; p = p->next) {
                SecretItem *item2 = SECRET_ITEM(p->data);
                GHashTable *bb = secret_item_get_attributes(item2);
                if (!g_strcmp0(g_hash_table_lookup(aa, "line1"), g_hash_table_lookup(bb, "line1")) &&
                    !g_strcmp0(g_hash_table_lookup(aa, "line2"), g_hash_table_lookup(bb, "line2"))
                ) {
                    cur = p;
                } else if(item_compare(item2, item) <= 0) {
                    cur2 = p;
                }
                g_hash_table_unref(bb);
            }
            g_hash_table_unref(aa);
            if (!cur) {
                cur = cur2;
            }
        }
        g_object_unref(item);
    }
    if (!cur) {
        cur = items;
    }

    current_password->data = cur;
    update_current_password();
}

static void item_attributes_changed_cb(SecretItem *item G_GNUC_UNUSED, GParamSpec *ps G_GNUC_UNUSED, gpointer d G_GNUC_UNUSED) {
    warn(DEBUG_DEBUG, "Item attributes changed for %s, clearing sortkey and reloading list of items (for sorting)", g_dbus_proxy_get_object_path(G_DBUS_PROXY(item)));
    g_object_set_data(G_OBJECT(item), "sortkey", NULL);
    load_items_from_collection();
}

static void secret_collection_load_items_cb(SecretCollection *c, GAsyncResult *res, keyring_cb_data d) {
    if (c != collection) {
        secret_collection_load_items_finish(c, res, NULL);
        return;
    }

    GError *err = NULL;
    if (!secret_collection_load_items_finish(collection, res, &err)) {
        prefix_error(&err, "Failed to fetch list of passwords from keyring");
        keyring_cb_data_done(d, err);
        g_clear_error(&err);
        return;
    }
    g_clear_error(&err);

    warn(DEBUG_DEBUG, "Items loaded from collection %s", g_dbus_proxy_get_object_path(G_DBUS_PROXY(collection)));
    load_items_from_collection();
    keyring_cb_data_done(d, NULL);
    app_update_state();
}

static void collection_items_changed_cb(SecretCollection *c, GParamSpec *ps G_GNUC_UNUSED, gpointer p G_GNUC_UNUSED) {
    if (c != collection) {
        g_signal_handlers_disconnect_by_func(G_OBJECT(c), G_CALLBACK(collection_items_changed_cb), NULL);
        return;
    }
    warn(DEBUG_DEBUG, "Collection %s items changed!", g_dbus_proxy_get_object_path(G_DBUS_PROXY(collection)));
    load_items_from_collection();
    app_update_state();
}

static void collection_locked_changed_cb(SecretCollection *c, GParamSpec *ps G_GNUC_UNUSED, gpointer p G_GNUC_UNUSED) {
    if (c != collection) {
        g_signal_handlers_disconnect_by_func(G_OBJECT(c), G_CALLBACK(collection_locked_changed_cb), NULL);
        return;
    }

    gboolean locked = secret_collection_get_locked(collection);
    warn(DEBUG_DEBUG, "Collection %s is now %s!", g_dbus_proxy_get_object_path(G_DBUS_PROXY(collection)), locked ? "locked" : "unlocked");
    if (!locked) {
        load_items_from_collection();
    }
    app_update_state();
}

static void collections_changed_cb(GObject *obj, GParamSpec *ps G_GNUC_UNUSED, gpointer p G_GNUC_UNUSED) {
    if (!collection) {
        return;
    }

    // This function does double duty as a handler for collections-changed on
    // the service and label-changed on the collection. For the latter,
    // sometimes it fires a "label-changed" event despite the label still being
    // KEYRING_NAME, which we want to ignore.
    if (obj == G_OBJECT(collection)) {
        gchar *label = secret_collection_get_label(collection);
        warn(DEBUG_DEBUG, "Collection %s label changed to %s", g_dbus_proxy_get_object_path(G_DBUS_PROXY(collection)), label);
        if (!g_strcmp0(label, KEYRING_NAME)) {
            // Spurious notification, ignore it
            g_free(label);
            return;
        }
        g_free(label);
    } else {
        warn(DEBUG_DEBUG, "List of collections changed!");
    }

    // Make sure our collection still exists. If not, try to re-get it.
    const gchar *path = g_dbus_proxy_get_object_path(G_DBUS_PROXY(collection));
    GList *collections = secret_service_get_collections(service);
    for (GList *cur = collections; cur && !collection; cur = cur->next) {
        gchar *label = secret_collection_get_label(SECRET_COLLECTION(cur->data));
        if (!g_strcmp0(label, KEYRING_NAME)) {
            const gchar *path2 = g_dbus_object_get_object_path(G_DBUS_OBJECT(cur->data));
            if (!g_strcmp0(path, path2)) {
                return;
            }
        }
        g_free(label);
    }
    g_list_free_full(collections, g_object_unref);

    warn(DEBUG_INFO, "Collection %s disappeared or is no longer named '" KEYRING_NAME "'", g_dbus_proxy_get_object_path(G_DBUS_PROXY(collection)));
    free_collection();
    app_update_state();
    get_collection_from_service(NULL);
}

static void unlock_collection_cb(GObject *obj G_GNUC_UNUSED, GAsyncResult *res, keyring_cb_data d) {
    GError *err = NULL;
    if (!secret_service_unlock_finish(service, res, NULL, &err)) {
        prefix_error(&err, "Failed to unlock keyring");
        keyring_cb_data_done(d, err);
        g_clear_error(&err);
        return;
    }
    g_clear_error(&err);

    warn(DEBUG_DEBUG, "Collection %s unlocked! Loading items...", g_dbus_proxy_get_object_path(G_DBUS_PROXY(collection)));
    secret_collection_load_items(collection, d ? d->cancel : NULL, (GAsyncReadyCallback)secret_collection_load_items_cb, d);
}

static void handle_collection(keyring_cb_data d) {
    warn(DEBUG_DEBUG, "Setting up signal handlers for collection %s", g_dbus_proxy_get_object_path(G_DBUS_PROXY(collection)));
    g_signal_connect(G_OBJECT(collection), "notify::label", G_CALLBACK(collections_changed_cb), NULL);
    g_signal_connect(G_OBJECT(collection), "notify::items", G_CALLBACK(collection_items_changed_cb), NULL);
    g_signal_connect(G_OBJECT(collection), "notify::locked", G_CALLBACK(collection_locked_changed_cb), NULL);

    if (secret_collection_get_locked(collection)) {
        if (d) {
            warn(DEBUG_DEBUG, "Unlocking collection %s", g_dbus_proxy_get_object_path(G_DBUS_PROXY(collection)));
            GList *list = g_list_prepend(NULL, collection);
            secret_service_unlock(service, list, d->cancel, (GAsyncReadyCallback)unlock_collection_cb, d);
            g_list_free(list);
        } else {
            warn(DEBUG_DEBUG, "Collection %s is locked and we're non-interactive", g_dbus_proxy_get_object_path(G_DBUS_PROXY(collection)));
        }
    } else {
        warn(DEBUG_DEBUG, "Loading items for collection %s", g_dbus_proxy_get_object_path(G_DBUS_PROXY(collection)));
        secret_collection_load_items(collection, d ? d->cancel : NULL, (GAsyncReadyCallback)secret_collection_load_items_cb, d);
    }
}

static void secret_collection_create_cb(GObject *obj G_GNUC_UNUSED, GAsyncResult *res, keyring_cb_data d) {
    GError *err = NULL;
    collection = secret_collection_create_finish(res, &err);
    if (!collection) {
        prefix_error(&err, "Collection creation failed");
        keyring_cb_data_done(d, err);
        g_clear_error(&err);
        return;
    }
    g_clear_error(&err);

    gchar *label = secret_collection_get_label(collection);
    if (g_strcmp0(label, KEYRING_NAME)) {
        g_set_error(&err, APP_GENERIC_ERROR, 0, "Created collection is not named '" KEYRING_NAME "'");
        keyring_cb_data_done(d, err);
        g_clear_error(&err);
        return;
    }

    handle_collection(d);
}

static void get_collection_from_service(keyring_cb_data d) {
    GList *collections = secret_service_get_collections(service);
    for (GList *cur = collections; cur && !collection; cur = cur->next) {
        gchar *label = secret_collection_get_label(SECRET_COLLECTION(cur->data));
        if (!g_strcmp0(label, KEYRING_NAME)) {
            collection = SECRET_COLLECTION(cur->data);
            g_object_ref(collection);
        }
        g_free(label);
    }
    g_list_free_full(collections, g_object_unref);

    if (collection) {
        warn(DEBUG_DEBUG, "Using existing collection %s", g_dbus_proxy_get_object_path(G_DBUS_PROXY(collection)));
        handle_collection(d);
    } else if (d) {
        // Only do this if we're called from app_activate, because this will
        // probably prompt the user.
        warn(DEBUG_DEBUG, "Creating new '" KEYRING_NAME "' collection");
        secret_collection_create(service, KEYRING_NAME, NULL, SECRET_COLLECTION_CREATE_NONE, d->cancel, (GAsyncReadyCallback)secret_collection_create_cb, d);
    } else {
        warn(DEBUG_DEBUG, "No '" KEYRING_NAME "' collection exists and we're non-interactive");
    }
}

static void secret_service_get_cb(GObject *obj G_GNUC_UNUSED, GAsyncResult *res, gpointer p G_GNUC_UNUSED) {
    GError *err = NULL;
    service = secret_service_get_finish(res, &err);
    if (!service) {
        warn(DEBUG_WARN, "Failed to connect to Secrets service: %s", err ? err->message : "<unknown error>" );
        g_timeout_add_seconds(5, do_secret_service_get, NULL);
        return;
    }
    g_clear_error(&err);

    warn(DEBUG_DEBUG, "Secrets service connected!");
    app_update_state();
    g_signal_connect(G_OBJECT(service), "notify::collections", G_CALLBACK(collections_changed_cb), NULL);
    get_collection_from_service(NULL);
}

static gboolean do_secret_service_get(gpointer d G_GNUC_UNUSED) {
    warn(DEBUG_DEBUG, "Connecting to Secrets service");
    secret_service_get(SECRET_SERVICE_OPEN_SESSION | SECRET_SERVICE_LOAD_COLLECTIONS, NULL, (GAsyncReadyCallback)secret_service_get_cb, NULL);
    return FALSE;
}

static void lock_collection_cb(GObject *obj G_GNUC_UNUSED, GAsyncResult *res, keyring_cb_data d) {
    GError *err = NULL;
    if (!secret_service_lock_finish(service, res, NULL, &err)) {
        prefix_error(&err, "Failed to lock keyring");
    } else {
        warn(DEBUG_DEBUG, "Collection locked!");
        g_clear_error(&err);
    }
    keyring_cb_data_done(d, err);
    g_clear_error(&err);
}

static void secret_item_load_secret_cb(SecretItem *item, GAsyncResult *res, keyring_cb_data d) {
    GError *err = NULL;
    if (!secret_item_load_secret_finish(item, res, &err)) {
        prefix_error(&err, "Failed to load password");
    } else {
        warn(DEBUG_DEBUG, "Password secret loaded!");
        g_clear_error(&err);
    }
    keyring_cb_data_done(d, err);
    g_clear_error(&err);
}

static void secret_service_unlock_cb(GObject *obj G_GNUC_UNUSED, GAsyncResult *res, keyring_cb_data d) {
    GError *err = NULL;
    GList *list = NULL;
    if (!secret_service_unlock_finish(service, res, &list, &err)) {
        prefix_error(&err, "Failed to unlock password");
        keyring_cb_data_done(d, err);
        g_clear_error(&err);
        return;
    }
    g_clear_error(&err);

    warn(DEBUG_DEBUG, "Password item unlocked! Loading secret...");
    secret_item_load_secret(SECRET_ITEM(list->data), d->cancel, (GAsyncReadyCallback)secret_item_load_secret_cb, d);

    g_list_free_full(list, g_object_unref);
}

static void secret_collection_delete_cb(SecretCollection *c, GAsyncResult *res, void (*done)(GError *)) {
    GError *err = NULL;
    if (!secret_collection_delete_finish(c, res, &err)) {
        warn(DEBUG_DEBUG, "Collection delete failed: %s", err ? err->message : "<unknown error>");
        done(err);
    } else {
        warn(DEBUG_DEBUG, "Collection delete succeeded");
        done(NULL);
    }
    g_clear_error(&err);
}

static void password_edit_unlock_cb(GObject *obj G_GNUC_UNUSED, GAsyncResult *res, password_edit_data d) {
    GError *err = NULL;
    GList *list = NULL;
    gint expect = g_list_length(d->lock);
    gint num = secret_service_unlock_finish(service, res, &list, &err);
    if (num < expect) {
        prefix_error(&err, "Failed to unlock %d of %d keyrings/passwords", expect-num, expect);
        free_password_edit_data(d, err);
        g_clear_error(&err);
        return;
    }
    g_clear_error(&err);
    g_list_free_full(list, g_object_unref);

    ((void (*)(GObject *, password_edit_data))d->next)(d->lock->data, d);
}

static void secret_item_create_cb(GObject *obj G_GNUC_UNUSED, GAsyncResult *res, password_edit_data d) {
    GError *err = NULL;
    SecretItem *item = secret_item_create_finish(res, &err);
    if (!item) {
        prefix_error(&err, "Failed to create password");
    } else {
        warn(DEBUG_DEBUG, "Password created as %s", g_dbus_proxy_get_object_path(G_DBUS_PROXY(item)));
        g_clear_error(&err);
        g_clear_object(&goto_item_on_reload);
        goto_item_on_reload = item;
        load_items_from_collection();
    }
    free_password_edit_data(d, err);
    g_clear_error(&err);
}

static void do_password_create(SecretCollection *c G_GNUC_UNUSED, password_edit_data d) {
    GHashTable *ht = password_edit_data_to_hashtable(d);
    gchar *label = g_strjoin(" / ", d->line1, d->line2, NULL);
    warn(DEBUG_DEBUG, "Creating new password item for %s...", label);
    secret_item_create(collection, &schema, ht, label, d->v, SECRET_ITEM_CREATE_NONE, d->cb->cancel, (GAsyncReadyCallback)secret_item_create_cb, d);
    g_hash_table_unref(ht);
    g_free(label);
}

static void do_password_edit_cb3(SecretItem *item, GAsyncResult *res, password_edit_data d) {
    GError *err = NULL;
    if (!secret_item_set_label_finish(item, res, &err)) {
        warn(DEBUG_WARN, "Failed to set password label: %s", err ? err->message : "<unknown error>");
    } else {
        warn(DEBUG_DEBUG, "Password label set!");
    }
    g_clear_error(&err);
    free_password_edit_data(d, NULL);
}

static void do_password_edit_cb2(SecretItem *item, GAsyncResult *res, password_edit_data d) {
    GError *err = NULL;
    if (!secret_item_set_attributes_finish(item, res, &err)) {
        prefix_error(&err, "Failed to set password display lines (secret was saved)");
        free_password_edit_data(d, err);
        g_clear_error(&err);
        return;
    }
    g_clear_error(&err);

    warn(DEBUG_DEBUG, "Password attributes set! Setting label...");
    gchar *label = g_strjoin(" / ", d->line1, d->line2, NULL);
    secret_item_set_label(item, label, d->cb->cancel, (GAsyncReadyCallback)do_password_edit_cb3, d);
    g_free(label);
}

static void do_password_edit_cb1(SecretItem *item, GAsyncResult *res, password_edit_data d) {
    GError *err = NULL;
    if (!secret_item_set_secret_finish(item, res, &err)) {
        prefix_error(&err, "Failed to set password secret");
        free_password_edit_data(d, err);
        g_clear_error(&err);
        return;
    }
    g_clear_error(&err);

    warn(DEBUG_DEBUG, "Password secret set! Setting attributes...");
    GHashTable *ht = password_edit_data_to_hashtable(d);
    secret_item_set_attributes(item, &schema, ht, d->cb->cancel, (GAsyncReadyCallback)do_password_edit_cb2, d);
    g_hash_table_unref(ht);
}

static void do_password_edit(SecretItem *item, password_edit_data d) {
    warn(DEBUG_DEBUG, "Setting password secret for %s...", g_dbus_proxy_get_object_path(G_DBUS_PROXY(item)));
    secret_item_set_secret(item, d->v, d->cb->cancel, (GAsyncReadyCallback)do_password_edit_cb1, d);
}

static void secret_item_delete_cb(SecretItem *item, GAsyncResult *res, password_edit_data d) {
    GError *err = NULL;
    if (!secret_item_delete_finish(item, res, &err)) {
        prefix_error(&err, "Failed to delete password");
    } else {
        warn(DEBUG_DEBUG, "Password item deleted!");
        g_clear_error(&err);
    }
    free_password_edit_data(d, err);
    g_clear_error(&err);
}

static void do_password_delete(SecretItem *item, password_edit_data d) {
    warn(DEBUG_DEBUG, "Deleting password item %s...", g_dbus_proxy_get_object_path(G_DBUS_PROXY(item)));
    secret_item_delete(item, d->cb->cancel, (GAsyncReadyCallback)secret_item_delete_cb, d);
}

static void password_get_edit_data_cb2(SecretItem *item, GAsyncResult *res, GCancellable *cancel) {
    void (*done)(GObject *, const gchar *, const gchar *, SecretValue *, GError *) = g_object_get_data(G_OBJECT(cancel), "wmpasman_done");
    GError *err = NULL;
    if (!secret_item_load_secret_finish(item, res, &err)) {
        prefix_error(&err, "Failed to load password");
        done(NULL, NULL, NULL, NULL, err);
    } else {
        warn(DEBUG_DEBUG, "Loaded password edit data!");
        GHashTable *aa = secret_item_get_attributes(item);
        SecretValue *v = secret_item_get_secret(item);
        done(G_OBJECT(item), g_hash_table_lookup(aa, "line1"), g_hash_table_lookup(aa, "line2"), v, NULL);
        g_hash_table_unref(aa);
        secret_value_unref(v);
    }
    g_object_unref(cancel);
}

static void password_get_edit_data_cb(GObject *obj G_GNUC_UNUSED, GAsyncResult *res, GCancellable *cancel) {
    GError *err = NULL;
    GList *list = NULL;
    if (!secret_service_unlock_finish(service, res, &list, &err)) {
        prefix_error(&err, "Failed to unlock password");
        void (*done)(GObject *, const gchar *, const gchar *, SecretValue *, GError *) = g_object_get_data(G_OBJECT(cancel), "wmpasman_done");
        done(NULL, NULL, NULL, NULL, err);
        g_object_unref(cancel);
        g_clear_error(&err);
        return;
    }
    g_clear_error(&err);

    warn(DEBUG_DEBUG, "Loading secret for password %s...", g_dbus_proxy_get_object_path(G_DBUS_PROXY(list->data)));
    secret_item_load_secret(SECRET_ITEM(list->data), cancel, (GAsyncReadyCallback)password_get_edit_data_cb2, cancel);

    g_list_free_full(list, g_object_unref);
}


/*** public functions ***/

void keyring_init(gboolean allow_core_files) {
    warn(DEBUG_DEBUG, "Disabling core files...");
    errno=0;
#ifdef HAVE_PRCTL
    if (prctl(PR_SET_DUMPABLE, 0))
#else
    struct rlimit r;
    r.rlim_cur=r.rlim_max=0;
    if (setrlimit(RLIMIT_CORE, &r))
#endif
    {
        if (allow_core_files) {
            die("Could not disable core files: %s", strerror(errno));
        } else {
            warn(DEBUG_WARN, "Could not disable core files: %s", strerror(errno));
        }
    } else {
        warn(DEBUG_INFO, "Core files disabled");
    }

    // It seems this is currently useless, since SecretValue offers no way to
    // not fallback to insecure memory. Keep it anyway in case that changes.
    /*
    if (!allow_insecure_memory) {
        warn(DEBUG_DEBUG, "Checking whether secure memory can be allocated...");
        gpointer p = g_malloc(1024);
        SecretValue *secret = secret_value_new(p, 1024, "application/octet-stream");
        if (!secret) {
            die("Could not allocate 1024 bytes of secure memory (as a test)");
        }
        secret_value_unref(secret);
        g_free(p);
        warn(DEBUG_INFO, "Secure memory can be allocated");
    }
    */

    do_secret_service_get(NULL);
}

gboolean keyring_ready(void) {
    return service != NULL;
}

gboolean keyring_unlocked(void) {
    return collection != NULL && !secret_collection_get_locked(collection);
}

void keyring_unlock(GCancellable *cancel, void (*done)(GError *)) {
    if (!collection) {
        get_collection_from_service(make_keyring_cb_data(done, cancel));
    } else if (secret_collection_get_locked(collection)) {
        warn(DEBUG_DEBUG, "Unlocking collection %s", g_dbus_proxy_get_object_path(G_DBUS_PROXY(collection)));
        GList *list = g_list_prepend(NULL, collection);
        secret_service_unlock(service, list, cancel, (GAsyncReadyCallback)unlock_collection_cb, make_keyring_cb_data(done, cancel));
        g_list_free(list);
    } else {
        done(NULL);
    }
}

void keyring_lock(GCancellable *cancel, void (*done)(GError *)) {
    if (collection && !secret_collection_get_locked(collection)) {
        warn(DEBUG_DEBUG, "Locking collection %s", g_dbus_proxy_get_object_path(G_DBUS_PROXY(collection)));
        GList *list = g_list_prepend(NULL, collection);
        secret_service_lock(service, list, cancel, (GAsyncReadyCallback)lock_collection_cb, make_keyring_cb_data(done, cancel));
        g_list_free(list);
    } else {
        done(NULL);
    }
}

void keyring_lock_sync(void) {
    if (collection && !secret_collection_get_locked(collection)) {
        warn(DEBUG_DEBUG, "Locking collection %s", g_dbus_proxy_get_object_path(G_DBUS_PROXY(collection)));
        GError *err = NULL;
        GList *list = g_list_prepend(NULL, collection);
        if (!secret_service_lock_sync(service, list, NULL, NULL, &err)) {
            warn(DEBUG_ERROR, "Could not lock collection: %s", err ? err->message : "<unknown error>");
        }
        g_list_free(list);
        g_clear_error(&err);
    }
}

void keyring_activate(GCancellable *cancel, void (*done)(GError *)) {
    warn(DEBUG_DEBUG, "Unlocking password item %s", g_dbus_proxy_get_object_path(G_DBUS_PROXY(current_password->data->data)));
    GList *list = g_list_prepend(NULL, SECRET_ITEM(current_password->data->data));
    secret_service_unlock(service, list, cancel, (GAsyncReadyCallback)secret_service_unlock_cb, make_keyring_cb_data(done, cancel));
    g_list_free(list);
}

void keyring_deactivate(void) {
    if (current_password) {
        warn(DEBUG_DEBUG, "\"Locking\" password item %s", g_dbus_proxy_get_object_path(G_DBUS_PROXY(current_password->data->data)));
        // Unfortunately, there's no secret_item_unload_secret().
    }
}


void password_next() {
    if (!current_password) {
        return;
    }

    app_deactivate();
    GList *cur = current_password->data;
    current_password->data = cur->next ? cur->next : g_list_first(cur);
    update_current_password();
}

void password_prev() {
    if (!current_password) {
        return;
    }

    app_deactivate();
    GList *cur = current_password->data;
    current_password->data = cur->prev ? cur->prev : g_list_last(cur);
    update_current_password();
}

void password_create(const gchar *line1, const gchar *line2, SecretValue *password, GCancellable *cancel, void (*done)(GError *)) {
    password_edit_data d = make_password_edit_data(done, cancel);
    d->line1 = g_strdup(line1);
    d->line2 = g_strdup(line2);
    d->v = secret_value_ref(password);
    d->next = G_CALLBACK(do_password_create);
    d->lock = g_list_prepend(NULL, g_object_ref(collection));
    warn(DEBUG_DEBUG, "Unlocking collection %s (even though it shouldn't be necessary)", g_dbus_proxy_get_object_path(G_DBUS_PROXY(collection)));
    secret_service_unlock(service, d->lock, d->cb->cancel, (GAsyncReadyCallback)password_edit_unlock_cb, d);
}

void password_edit(GObject *pw, const gchar *line1, const gchar *line2, SecretValue *password, GCancellable *cancel, void (*done)(GError *)) {
    password_edit_data d = make_password_edit_data(done, cancel);
    d->line1 = g_strdup(line1);
    d->line2 = g_strdup(line2);
    d->v = secret_value_ref(password);
    d->next = G_CALLBACK(do_password_edit);
    d->lock = g_list_prepend(NULL, g_object_ref(SECRET_ITEM(pw)));
    warn(DEBUG_DEBUG, "Unlocking password item %s", g_dbus_proxy_get_object_path(G_DBUS_PROXY(pw)));
    secret_service_unlock(service, d->lock, d->cb->cancel, (GAsyncReadyCallback)password_edit_unlock_cb, d);
}

void password_delete(GCancellable *cancel, void (*done)(GError *)) {
    password_edit_data d = make_password_edit_data(done, cancel);
    d->next = G_CALLBACK(do_password_delete);
    d->lock = g_list_prepend(NULL, g_object_ref(current_password->data->data));
    warn(DEBUG_DEBUG, "Unlocking password item %s", g_dbus_proxy_get_object_path(G_DBUS_PROXY(current_password->data->data)));
    secret_service_unlock(service, d->lock, d->cb->cancel, (GAsyncReadyCallback)password_edit_unlock_cb, d);
}

void keyring_delete(GCancellable *cancel, void (*done)(GError *)) {
    warn(DEBUG_DEBUG, "Deleting collection %s", g_dbus_proxy_get_object_path(G_DBUS_PROXY(collection)));
    secret_collection_delete(collection, cancel, (GAsyncReadyCallback)secret_collection_delete_cb, done);
}

void password_to_selection(GtkSelectionData *selection_data) {
    GError *err = NULL;
    SecretValue *v = secret_item_get_secret(SECRET_ITEM(current_password->data->data));
    if (!v) {
        g_set_error(&err, APP_GENERIC_ERROR, 0, "Password data is currently unavailable");
        error_alert("%s", err);
        g_clear_error(&err);
        return;
    }

    const gchar *ct = secret_value_get_content_type(v);
    if (g_strcmp0(ct, "text/plain") &&
        g_strcmp0(ct, "text/plain; charset=utf-8")
    ) {
        g_set_error(&err, APP_GENERIC_ERROR, 0, "Password data content type \"%s\" is not supported", ct);
    } else {
        const gchar *text = secret_value_get(v, NULL);
        if (!gtk_selection_data_set_text(selection_data, text, -1)) {
            g_set_error(&err, APP_GENERIC_ERROR, 0, "Failed to set selection text into clipboard");
        }
    }
    if (err) {
        error_alert("%s", err);
        g_clear_error(&err);
    }
    secret_value_unref(v);
}

void password_get_edit_data(GCancellable *cancel, void (*done)(GObject *pw, const gchar *line1, const gchar *line2, SecretValue *password, GError *err)) {
    g_object_set_data(G_OBJECT(cancel), "wmpasman_done", done);

    warn(DEBUG_DEBUG, "Unlocking password item %s", g_dbus_proxy_get_object_path(G_DBUS_PROXY(current_password->data->data)));
    g_object_ref(cancel);
    GList *list = g_list_prepend(NULL, SECRET_ITEM(current_password->data->data));
    secret_service_unlock(service, list, cancel, (GAsyncReadyCallback)password_get_edit_data_cb, cancel);
    g_list_free(list);
}
