/*
 * GQradio
 * (C) 2002 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */

#include "gqradio.h"
#include "display.h"

#include "io_radio.h"
#include "mixer.h"
#include "preferences.h"
#include "preset.h"
#include "ui2_button.h"
#include "ui2_dial.h"
#include "ui2_item.h"
#include "ui2_main.h"
#include "ui2_list.h"
#include "ui2_number.h"
#include "ui2_slider.h"
#include "ui2_text.h"
#include "ui2_widget.h"

#include "window.h"


/* glue between SLIK and rest of app */


/*
 *-----------------------------------------------------------------------------
 * display widget callbacks - radio
 *-----------------------------------------------------------------------------
 */

/* seeking */

static void button_seek_up_press_cb(ButtonData *button, const gchar *key, gpointer data)
{
	seek_start(TRUE);
}

static void button_seek_up_release_cb(ButtonData *button, const gchar *key, gpointer data)
{
	seek_stop();
}

static void button_seek_down_press_cb(ButtonData *button, const gchar *key, gpointer data)
{
	seek_start(FALSE);
}

static void button_seek_down_release_cb(ButtonData *button, const gchar *key, gpointer data)
{
	seek_stop();
}

static void button_seek_up_preset_press_cb(ButtonData *button, const gchar *key, gpointer data)
{
	seek_start_by_mode(SEEK_MODE_PRESET, TRUE);
}

static void button_seek_up_preset_release_cb(ButtonData *button, const gchar *key, gpointer data)
{
	seek_stop();
}

static void button_seek_down_preset_press_cb(ButtonData *button, const gchar *key, gpointer data)
{
	seek_start_by_mode(SEEK_MODE_PRESET, FALSE);
}

static void button_seek_down_preset_release_cb(ButtonData *button, const gchar *key, gpointer data)
{
	seek_stop();
}

static void button_seek_up_manual_press_cb(ButtonData *button, const gchar *key, gpointer data)
{
	seek_start_by_mode(SEEK_MODE_MANUAL, TRUE);
}

static void button_seek_up_manual_release_cb(ButtonData *button, const gchar *key, gpointer data)
{
	seek_stop();
}

static void button_seek_down_manual_press_cb(ButtonData *button, const gchar *key, gpointer data)
{
	seek_start_by_mode(SEEK_MODE_MANUAL, FALSE);
}

static void button_seek_down_manual_release_cb(ButtonData *button, const gchar *key, gpointer data)
{
	seek_stop();
}

static void button_seek_up_auto_press_cb(ButtonData *button, const gchar *key, gpointer data)
{
	seek_start_by_mode(SEEK_MODE_AUTO, TRUE);
}

static void button_seek_up_auto_release_cb(ButtonData *button, const gchar *key, gpointer data)
{
	seek_stop();
}

static void button_seek_down_auto_press_cb(ButtonData *button, const gchar *key, gpointer data)
{
	seek_start_by_mode(SEEK_MODE_AUTO, FALSE);
}

static void button_seek_down_auto_release_cb(ButtonData *button, const gchar *key, gpointer data)
{
	seek_stop();
}

/* modes, options */

static gint button_mute_status_cb(ButtonData *button, const gchar *key, gpointer data)
{
	return muted;
}

static void button_mute_click_cb(ButtonData *button, const gchar *key, gpointer data)
{
	mute_toggle();
}

static void button_mode_click_cb(ButtonData *button, const gchar *key, gpointer data)
{
	mode_toggle();
}

static gint button_preset_set_status_cb(ButtonData *button, const gchar *key, gpointer data)
{
	return FALSE;
}

static void button_preset_set_click_cb(ButtonData *button, const gchar *key, gpointer data)
{
	preset_button_click();
}

static void button_preset_clear_click_cb(ButtonData *button, const gchar *key, gpointer data)
{
	preset_clear_one(preset);
}

static void button_preset_clear_all_click_cb(ButtonData *button, const gchar *key, gpointer data)
{
	preset_clear_all();
}

static gint button_preset_scan_status_cb(ButtonData *button, const gchar *key, gpointer data)
{
	return FALSE;
}

static void button_preset_scan_click_cb(ButtonData *button, const gchar *key, gpointer data)
{
	preset_auto_scan_click();
}

static void button_exit_click_cb(ButtonData *button, const gchar *key, gpointer data)
{
	gqradio_exit(TRUE);
}

static void button_exit_nomute_click_cb(ButtonData *button, const gchar *key, gpointer data)
{
	gqradio_exit(FALSE);
}

static void button_about_click_cb(ButtonData *button, const gchar *key, gpointer data)
{
	show_about();
}

static void button_config_click_cb(ButtonData *button, const gchar *key, gpointer data)
{
	config_window_show(0);
}

static void button_editor_click_cb(ButtonData *button, const gchar *key, gpointer data)
{
	edit_window(main_window);
}

static void button_mixer_click_cb(ButtonData *button, const gchar *key, gpointer data)
{
	mixer_run();
}

static const gchar *description_status_cb(TextData *text, const gchar *key, gpointer data)
{
	return preset_get_description(preset);
}

static gint item_mute_cb(ItemData *item, const gchar *key, gpointer data)
{
	return (muted);
}

static gint item_stereo_cb(ItemData *item, const gchar *key, gpointer data)
{
	return (stereo);
}

static gint item_mode_cb(ItemData *item, const gchar *key, gpointer data)
{
	return (gint)seek_mode + 1;;
}

static gint item_signal_cb(ItemData *item, const gchar *key, gpointer data)
{
        return item_section_by_percent(item, (gfloat)signal_strength / 100.0);
}

static gint number_freq_major_cb(NumberData *number, const gchar *key, gpointer data)
{
	return radio_freq_get_major(frequency);
}

static gint number_freq_minor_cb(NumberData *number, const gchar *key, gpointer data)
{
	return radio_freq_get_minor(frequency);
}

static gint number_preset_hide_cb(gpointer data)
{
	ui_widget_hide_by_key(main_window, "preset", number_type_id());
	return FALSE;
}

static gint number_preset_cb(NumberData *number, const gchar *key, gpointer data)
{
	if (preset < 0 && !preset_mode_get())
		{
		/* can't do this here */
		g_idle_add(number_preset_hide_cb, NULL);
		}

	return preset + 1;
}

static gint presets_length_cb(ListData *list, const gchar *key, gpointer data)
{
	return PRESET_LIST_SIZE;
}

static gint presets_row_info_cb(ListData *list, const gchar *key, gint row, ListRowData *rd, gpointer data)
{
	gchar *buf;

	buf = g_strdup_printf("%d.", row + 1);
	list_row_column_set_text(list, rd, "number", buf);
	g_free(buf);

	if (preset_is_set(row))
		{
		buf = radio_freq_to_text(preset_get_freq(row));
		list_row_column_set_text(list, rd, "frequency", buf);
		g_free(buf);
		list_row_column_set_text(list, rd, "description", preset_get_description(row));
		}

	return TRUE;
}

static void presets_row_click_cb(ListData *list, const gchar *key, gint row, ListRowData *rd,
				 gint button, gpointer data)
{
	preset_select(row);
}

/*
 *-----------------------------------------------------------------------------
 * display widget callbacks - mixer
 *-----------------------------------------------------------------------------
 */

static void button_volume_up_press_cb(ButtonData *button, const gchar *key, gpointer data)
{
	btn_volume_up_pressed();
}

static void button_volume_down_press_cb(ButtonData *button, const gchar *key, gpointer data)
{
	btn_volume_down_pressed();
}

static void button_volume_release_cb(ButtonData *button, const gchar *key, gpointer data)
{
	btn_volume_released();
}

static void button_balance_left_press_cb(ButtonData *button, const gchar *key, gpointer data)
{
	btn_balance_left_pressed();
}

static void button_balance_right_press_cb(ButtonData *button, const gchar *key, gpointer data)
{
	btn_balance_right_pressed();
}

static void button_balance_release_cb(ButtonData *button, const gchar *key, gpointer data)
{
	btn_balance_released();
}

static gint item_balance_status_cb(ItemData *item, const gchar *key, gpointer data)
{
	return item_section_by_percent(item, (float)get_balance() / 100.0);
}

static gint item_volume_status_cb(ItemData *item, const gchar *key, gpointer data)
{
	return item_section_by_percent(item, (float)get_volume() / 100.0);
}

static gfloat slider_volume_status_cb(SliderData *slider, const gchar *key, gpointer data)
{
	return (gfloat)get_volume() / 100.0;
}

static void slider_volume_drag_cb(SliderData *slider, const gchar *key, gfloat value, gpointer data)
{
	volume_adjust(value);
}

static gfloat slider_balance_status_cb(SliderData *slider, const gchar *key, gpointer data)
{
	return (gfloat)get_balance() / 100.0;
}

static void slider_balance_drag_cb(SliderData *slider, const gchar *key, gfloat value, gpointer data)
{
	balance_adjust(value);
}

static gfloat dial_volume_status_cb(DialData *dial, const gchar *key, gpointer data)
{
	return (gfloat)get_volume() / 100.0;
}

static void dial_volume_drag_cb(DialData *dial, const gchar *key, gfloat value, gpointer data)
{
	volume_adjust(value);
}

static gfloat dial_balance_status_cb(DialData *dial, const gchar *key, gpointer data)
{
	return (gfloat)get_balance() / 100.0;
}

static void dial_balance_drag_cb(DialData *dial, const gchar *key, gfloat value, gpointer data)
{
	balance_adjust(value);
}

/*
 *-----------------------------------------------------------------------------
 * display widget registry
 *-----------------------------------------------------------------------------
 */

void display_register_widgets(UIData *ui)
{
	button_register_key("seek.up", ui,
			    NULL, NULL,
			    NULL, NULL,
			    button_seek_up_press_cb, NULL,
			    button_seek_up_release_cb, NULL);
	button_register_key("seek.down", ui,
			    NULL, NULL,
			    NULL, NULL,
			    button_seek_down_press_cb, NULL,
			    button_seek_down_release_cb, NULL);

	button_register_key("seek.up.preset", ui,
			    NULL, NULL,
			    NULL, NULL,
			    button_seek_up_preset_press_cb, NULL,
			    button_seek_up_preset_release_cb, NULL);
	button_register_key("seek.down.preset", ui,
			    NULL, NULL,
			    NULL, NULL,
			    button_seek_down_preset_press_cb, NULL,
			    button_seek_down_preset_release_cb, NULL);

	button_register_key("seek.up.manual", ui,
			    NULL, NULL,
			    NULL, NULL,
			    button_seek_up_manual_press_cb, NULL,
			    button_seek_up_manual_release_cb, NULL);
	button_register_key("seek.down.manual", ui,
			    NULL, NULL,
			    NULL, NULL,
			    button_seek_down_manual_press_cb, NULL,
			    button_seek_down_manual_release_cb, NULL);

	button_register_key("seek.up.auto", ui,
			    NULL, NULL,
			    NULL, NULL,
			    button_seek_up_auto_press_cb, NULL,
			    button_seek_up_auto_release_cb, NULL);
	button_register_key("seek.down.auto", ui,
			    NULL, NULL,
			    NULL, NULL,
			    button_seek_down_auto_press_cb, NULL,
			    button_seek_down_auto_release_cb, NULL);

	button_register_key("mute", ui,
			    button_mute_status_cb, NULL,
			    button_mute_click_cb, NULL,
			    NULL, NULL,
			    NULL, NULL);
	button_register_key("mode", ui,
			    NULL, NULL,
			    button_mode_click_cb, NULL,
			    NULL, NULL,
			    NULL, NULL);

	button_register_key("preset.set", ui,
			    button_preset_set_status_cb, NULL,
			    button_preset_set_click_cb, NULL,
			    NULL, NULL,
			    NULL, NULL);
	button_register_key("preset.clear", ui,
			    NULL, NULL,
			    button_preset_clear_click_cb, NULL,
			    NULL, NULL,
			    NULL, NULL);
	button_register_key("preset.clear_all", ui,
			    NULL, NULL,
			    button_preset_clear_all_click_cb, NULL,
			    NULL, NULL,
			    NULL, NULL);
	button_register_key("preset.scan", ui,
			    button_preset_scan_status_cb, NULL,
			    button_preset_scan_click_cb, NULL,
			    NULL, NULL,
			    NULL, NULL);

	button_register_key("exit", ui,
			    NULL, NULL,
			    button_exit_click_cb, NULL,
			    NULL, NULL,
			    NULL, NULL);
	button_register_key("exit.nomute", ui,
			    NULL, NULL,
			    button_exit_nomute_click_cb, NULL,
			    NULL, NULL,
			    NULL, NULL);
	button_register_key("about", ui,
			    NULL, NULL,
			    button_about_click_cb, NULL,
			    NULL, NULL,
			    NULL, NULL);

	button_register_key("config", ui,
			    NULL, NULL,
			    button_config_click_cb, NULL,
			    NULL, NULL,
			    NULL, NULL);
	button_register_key("skin_editor", ui,
			    NULL, NULL,
			    button_editor_click_cb, NULL,
			    NULL, NULL,
			    NULL, NULL);

	button_register_key("mixer", ui,
			    NULL, NULL,
			    button_mixer_click_cb, NULL,
			    NULL, NULL,
			    NULL, NULL);
	button_register_key("volume.up", ui,
			    NULL, NULL,
			    NULL, NULL,
			    button_volume_up_press_cb, NULL,
			    button_volume_release_cb, NULL);
	button_register_key("volume.down", ui,
			    NULL, NULL,
			    NULL, NULL,
			    button_volume_down_press_cb, NULL,
			    button_volume_release_cb, NULL);
	button_register_key("balance.left", ui,
			    NULL, NULL,
			    NULL, NULL,
			    button_balance_left_press_cb, NULL,
			    button_balance_release_cb, NULL);
	button_register_key("balance.right", ui,
			    NULL, NULL,
			    NULL, NULL,
			    button_balance_right_press_cb, NULL,
			    button_balance_release_cb, NULL);

	item_register_key("volume", ui,
			  item_volume_status_cb, NULL);
	item_register_key("balance", ui,
			  item_balance_status_cb, NULL);

	item_register_key("mute", ui,
			  item_mute_cb, NULL);
	item_register_key("stereo", ui,
			  item_stereo_cb, NULL);
	item_register_key("mode", ui,
			  item_mode_cb, NULL);
	item_register_key("signal", ui,
			  item_signal_cb, NULL);
	item_register_key("scan", ui,
			  NULL, NULL);
	item_register_key("preset.set", ui,
			  NULL, NULL);
	item_register_key("preset.scan", ui,
			  NULL, NULL);

	number_register_key("freq.major", ui,
			    number_freq_major_cb, NULL);
	number_register_key("freq.minor", ui,
			    number_freq_minor_cb, NULL);
	number_register_key("preset", ui,
			    number_preset_cb, NULL);


	slider_register_key("volume", ui,
			    slider_volume_status_cb, NULL,
			    slider_volume_drag_cb, NULL,
			    NULL, NULL,
			    slider_volume_drag_cb, NULL);

	slider_register_key("balance", ui,
			    slider_balance_status_cb, NULL,
			    slider_balance_drag_cb, NULL,
			    NULL, NULL,
			    slider_balance_drag_cb, NULL);
	dial_register_key("volume", ui,
			  dial_volume_status_cb, NULL,
			  dial_volume_drag_cb, NULL,
			  NULL, NULL,
			  dial_volume_drag_cb, NULL);

	dial_register_key("balance", ui,
			  dial_balance_status_cb, NULL,
			  dial_balance_drag_cb, NULL,
			  NULL, NULL,
			  dial_balance_drag_cb, NULL);

	text_register_key("preset.description", ui, description_status_cb, NULL);

	list_register_key("presets", ui,
			  presets_length_cb, NULL,
			  presets_row_info_cb, NULL,
			  presets_row_click_cb, NULL,
			  NULL, NULL,
			  NULL, NULL);
}

/*
 *-----------------------------------------------------------------------------
 * display widget update functions (public)
 *-----------------------------------------------------------------------------
 */

void display_update_status(void)
{
	button_state_set("mute", main_window, muted);
	item_section_set("mute", main_window, muted);
	item_section_set("stereo", main_window, stereo);
	item_section_set_by_percent("signal", main_window, (gfloat)signal_strength / 100.0);
}

void display_set_mode(gint set)
{
	item_section_set("mode", main_window, (set) ? (gint)seek_mode + 1 : 0);
}

void display_set_preset_scan(gint set)
{
	item_animation_step("preset.scan", main_window, FALSE, !set);
	button_state_set("preset.scan", main_window, set);
}

void display_set_frequency(guint32 freq)
{
	number_value_set("freq.major", main_window, radio_freq_get_major(freq));
	number_value_set("freq.minor", main_window, radio_freq_get_minor(freq));
}

void display_hide_minor_digits(gint hide)
{
	if (hide)
		{
		ui_widget_hide_by_key(main_window, "freq.minor", number_type_id());
		}
	else
		{
		ui_widget_show_by_key(main_window, "freq.minor", number_type_id());
		}
}

void display_increment_scanner(gint reset)
{
	item_animation_step("scan", main_window, FALSE, reset);
}

void display_update_preset(gint flash, gint n)
{
	static gint preset_flash_point = 0;
	static gint preset_flash_on = TRUE;

	if (flash)
		{
		if (preset_flash_point != n)
			{
			preset_flash_on = TRUE;
			}
		else
			{
			preset_flash_on = !preset_flash_on;
			}
		preset_flash_point = n;

		item_section_set("preset.set", main_window, preset_flash_on);
		button_state_set("preset.set", main_window, preset_flash_on);
		}
	else
		{
		preset_flash_on = TRUE;
		n = preset;

		button_state_set("preset.set", main_window, FALSE);
		item_section_set("preset.set", main_window, FALSE);
		display_set_description(NULL);
		}

	if (preset_flash_on && n >= 0)
		{
		ui_widget_show_by_key(main_window, "preset", number_type_id());
		number_value_set("preset", main_window, n + 1);
		}
	else
		{
		ui_widget_hide_by_key(main_window, "preset", number_type_id());
		}

}

void display_preset_list_update(gint n)
{
	if (!main_window) return;

	if (n < 0)
		{
		list_refresh("presets", main_window);
		display_set_description(NULL);
		}
	else
		{
		list_row_update("presets", main_window, n);
		if (n == preset) display_set_description(NULL);
		}
}

void display_set_description(const gchar *text)
{
	if (!text) text = preset_get_description(preset);
	text_set_text("preset.description", main_window, text);
	text_set_autoscroll("preset.description", main_window, description_scrolls_always);

}

void display_set_volume(void)
{
	gfloat val;

	val = (float)get_volume() / 100.0;

	item_section_set_by_percent("volume", main_window, val);
	dial_value_set("volume", main_window, val);
	slider_value_set("volume", main_window, val);
}

void display_set_balance(void)
{
	gfloat val;

	val = (float)get_balance() / 100.0;

	item_section_set_by_percent("balance", main_window, val);
	dial_value_set("balance", main_window, val);
	slider_value_set("balance", main_window, val);
}

void display_scroll_preset_list(gint direction_up)
{
	list_scroll_row("presets", main_window, (direction_up) ? -1 : 1);
}



