/*
 *
 * This file is part of yacas.
 * Yacas is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesset General Public License as
 * published by the Free Software Foundation, either version 2.1
 * of the License, or (at your option) any later version.
 *
 * Yacas is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with yacas.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "yacas/mp/zz.hpp"

#include <gtest/gtest.h>

using namespace yacas::mp;

TEST(YMP_ZZTest, construction)
{
    ASSERT_EQ(ZZ("0"), ZZ(0));
    ASSERT_EQ(ZZ("123"), ZZ(123));
    ASSERT_EQ(ZZ(" 123"), ZZ(123));
    ASSERT_EQ(ZZ("+123"), ZZ(123));
    ASSERT_EQ(ZZ("FF", 16), ZZ("255"));
    ASSERT_EQ(ZZ("-FF", 16), ZZ("-255"));
    ASSERT_EQ(ZZ("ff", 16), ZZ("255"));
    ASSERT_EQ(ZZ("-ff", 16), ZZ("-255"));
    ASSERT_EQ(ZZ("deadbeef", 16), ZZ("3735928559"));
    ASSERT_EQ(ZZ("-deadbeef", 16), ZZ("-3735928559"));
    ASSERT_THROW(ZZ(""), ZZ::ParseError);
    ASSERT_THROW(ZZ(" "), ZZ::ParseError);
    ASSERT_THROW(ZZ("deadbeef", 15), ZZ::ParseError);
}

TEST(YMP_ZZTest, is_zero)
{
    ASSERT_TRUE(ZZ(0).is_zero());
    ASSERT_FALSE(ZZ(1).is_zero());
    ASSERT_FALSE(ZZ(-1).is_zero());
    ASSERT_FALSE(ZZ("4294967296").is_zero());
    ASSERT_TRUE(ZZ("-0").is_zero());
}

TEST(YMP_ZZTest, comparison)
{
    ASSERT_TRUE(ZZ("735") == ZZ(735));
    ASSERT_TRUE(ZZ("-735") == ZZ(-735));
    ASSERT_TRUE(ZZ("-735") != ZZ(735));
    ASSERT_TRUE(ZZ("736") != ZZ("735"));
    ASSERT_TRUE(ZZ("736") > ZZ("735"));
    ASSERT_TRUE(ZZ("736") >= ZZ("735"));
    ASSERT_TRUE(ZZ("736") > ZZ("-736"));
    ASSERT_TRUE(ZZ("-736") <= ZZ("735"));
}

TEST(YMP_ZZTest, parity)
{
    ASSERT_TRUE(ZZ(0).is_even());
    ASSERT_FALSE(ZZ(1).is_even());
    ASSERT_FALSE(ZZ(-1).is_even());
    ASSERT_TRUE(ZZ(2).is_even());
    ASSERT_TRUE(ZZ(-2).is_even());
    ASSERT_FALSE(ZZ(3).is_even());
    ASSERT_FALSE(ZZ(-3).is_even());
    ASSERT_TRUE(ZZ("167170571948282915479450721253708836308").is_even());
    ASSERT_TRUE(ZZ("-167170571948282915479450721253708836308").is_even());
    ASSERT_FALSE(ZZ("167170571948282915479450721253708836309").is_even());
    ASSERT_FALSE(ZZ("-167170571948282915479450721253708836309").is_even());
}

TEST(YMP_ZZTest, to_string)
{
    ASSERT_EQ(ZZ(0).to_string(10), "0");
    ASSERT_EQ(ZZ(1).to_string(10), "1");
    ASSERT_EQ(ZZ(-1).to_string(10), "-1");
    ASSERT_EQ(ZZ(2).to_string(10), "2");
    ASSERT_EQ(ZZ(-2).to_string(10), "-2");
    ASSERT_EQ(ZZ("18446744073709551616").to_string(), "18446744073709551616");
    ASSERT_EQ(ZZ("-18446744073709551616").to_string(), "-18446744073709551616");

    ASSERT_EQ(ZZ("121932631356500531591068431703703700581771069347203169112635269").to_string(), "121932631356500531591068431703703700581771069347203169112635269");
    ASSERT_EQ(ZZ("-121932631356500531591068431703703700581771069347203169112635269").to_string(), "-121932631356500531591068431703703700581771069347203169112635269");

    ASSERT_EQ(ZZ("3735928559").to_string(16), "deadbeef");
    ASSERT_EQ(ZZ("-3735928559").to_string(16), "-deadbeef");
}

TEST(YMP_ZZTest, io)
{
    std::ostringstream os;

    os << std::hex << ZZ("3735928559");
    ASSERT_EQ(os.str(), "deadbeef");
    os.str("");
    os.clear();
    os << std::oct << ZZ("3735928559");
    ASSERT_EQ(os.str(), "33653337357");
}

TEST(YMP_ZZTest, shift)
{
    ZZ a;

    a = ZZ(5);
    a <<= 1;
    ASSERT_EQ(a, ZZ(10));

    a = ZZ(-5);
    a <<= 1;
    ASSERT_EQ(a, ZZ(-10));

    a >>= 2;
    ASSERT_EQ(a, ZZ(-2));

    a = ZZ("-121932631356500531591068431703703700581771069347203169112635268");
    a >>= 1;
    ASSERT_EQ(a, ZZ("-60966315678250265795534215851851850290885534673601584556317634"));
}



TEST(YMP_ZZTest, sqr)
{
    ZZ a;

    a = ZZ(5);
    a.sqr();
    ASSERT_EQ(a, ZZ("25"));

    a = ZZ("-1562953088685537097059913193583733507649747348170814625952201");
    a.sqr();
    ASSERT_EQ(a, ZZ("2442822357431660390046655205292608987306861461763843847861569410791671541750358984338534812651100785465086339385936744401"));

    a = ZZ("16677035576003637250463766760907486818436058618557682023339705449125918899");
    a.sqr();
    ASSERT_EQ(a, ZZ("278123515603290968886766134770831823603135705647199088974965688745779144002689087632472856070865997170461249543456775022822347627260616871125372201"));
    a.sqr();
    ASSERT_EQ(a, ZZ("77352689931534035033046061911187956966109490783091765407180918697187558629814838215212696461020230376125043081229650956490719928164351366033600480221268981451258346147413175925170392414485100148544496206236851953940761239321222390198418165678165748977200849860077295483412916485724330783584401"));
}

TEST(YMP_ZZTest, add)
{
    ZZ a(10);
    a += 10;
    ASSERT_EQ(a, 20);
    a += -21;
    ASSERT_EQ(a, -1);
    a += 0;
    ASSERT_EQ(a, -1);
    a += -2;
    ASSERT_EQ(a, -3);
    a += 4;
    ASSERT_EQ(a, 1);

    ZZ x("123456789123456789123456789");
    ZZ y("-987654321987654321987654321987654321");

    x += y;

    ASSERT_EQ(x, ZZ("-987654321864197532864197532864197532"));
    x += ZZ("987654321864197532864197532864197534");
    ASSERT_EQ(x, 2);
}


TEST(YMP_ZZTest, mul)
{
    ZZ a(10);
    a *= 10;
    ASSERT_EQ(a, ZZ(100));
    a *= 10;
    ASSERT_EQ(a, ZZ(1000));
    a *= 10;
    ASSERT_EQ(a, ZZ(10000));
    a *= 100;
    ASSERT_EQ(a, ZZ(1000000));
    a *= 100;
    ASSERT_EQ(a, ZZ(100000000));
    a *= -100;
    ASSERT_EQ(a, ZZ("-10000000000"));
    a *= -100;
    ASSERT_EQ(a, ZZ("1000000000000"));
    a *= 100;
    ASSERT_EQ(a, ZZ("100000000000000"));
    a *= -753;
    ASSERT_EQ(a, ZZ("-75300000000000000"));
    ZZ b(-10);
    a *= b;
    ASSERT_EQ(a, ZZ("753000000000000000"));

    ZZ x("4294967296");
    ZZ y("-4294967296");
    x *= y;
    ASSERT_EQ(x, ZZ("-18446744073709551616"));

    x = ZZ("-123456789123456789123456789");
    y = ZZ("-987654321987654321987654321987654321");
    x *= y;
    ASSERT_EQ(x, ZZ("121932631356500531591068431703703700581771069347203169112635269"));
}

TEST(YMP_ZZTest, pow)
{
    ZZ a("-1234567890123456789");

    a.pow(7);
    ASSERT_EQ(a, ZZ("-4371241899268725428364208289519510588539212553598950486912858825153547618526426094549436384682321156604105518810510686881926429"));
}

TEST(YMP_ZZTest, div)
{
    ZZ a(-123);

    ASSERT_THROW(a /= 0, std::domain_error);
    ASSERT_EQ(a /= 1, ZZ(-123));
    ASSERT_EQ(a /= -123, ZZ::ONE);
    a = ZZ("123");
    ASSERT_EQ(a /= 10, ZZ(12));
    ASSERT_EQ(a /= -6, ZZ(-2));

    a = ZZ("-12314324325435664576576568787687576435342431432545345");
    a /= 743;
    ASSERT_EQ(a, ZZ("-16573787786589050574127279660413965592654685642725"));
    a = ZZ(-123);

    ASSERT_THROW(a /= ZZ::ZERO, std::domain_error);
    ASSERT_EQ((a /= ZZ(1)), ZZ(-123));
    ASSERT_EQ((a /= ZZ(123)), ZZ(-1));
    a = ZZ("-123");
    ASSERT_EQ((a /= ZZ(10)), ZZ(-12));
    ASSERT_EQ((a /= ZZ(-6)), ZZ(2));
    a = ZZ("-1125899906842624");
    ZZ b("-562949953421312");
    a /= b;
    ASSERT_EQ(a, ZZ::TWO);
    a = ZZ("79228162514264337593543950335");
    b = ZZ("39614081257132168796771975168");
    a /= b;
    ASSERT_EQ(a, ZZ::ONE);
    a = ZZ("-79228162514264337593543950336");
    b = ZZ("-39614081257132168796771975168");
    a /= b;
    ASSERT_EQ(a, ZZ::TWO);
    a = ZZ("-12314324325435664576576568787687576435342431432545345");
    b = ZZ("42345346452431432543645634543455434");
    a /= b;
    ASSERT_EQ(a, ZZ("-290807027385380782"));

    a = ZZ(1);
    for (int i = 1; i <= 180; ++i)
        a *= ZZ(-i);

    ASSERT_EQ(a, ZZ("200896062499134299656951336898466838917540340798867777940435335160044860953395980941180138112097309735631594101037399609671032132186331495273609598531966730972945653558819806475064353856858157445040809209560358463319644664891114256430017824141796753818192338642302693327818731986039603200000000000000000000000000000000000000000000"));

    b = ZZ(-1);
    for (int i = 1; i <= 170; ++i)
        b *= ZZ(i);

    ASSERT_EQ(b, ZZ("-7257415615307998967396728211129263114716991681296451376543577798900561843401706157852350749242617459511490991237838520776666022565442753025328900773207510902400430280058295603966612599658257104398558294257568966313439612262571094946806711205568880457193340212661452800000000000000000000000000000000000000000"));

    a /= b;
    ASSERT_EQ(a, ZZ("-27681487894311318144000"));

    a = ZZ("1234567890123456789");
    b = ZZ("-987654321");
    a /= b;
    ASSERT_EQ(a, ZZ("-1249999988"));

    a = ZZ("-12345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789");
    b = ZZ("123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789");

    a /= b;

    ASSERT_EQ(a, ZZ("-100000000000000000000000000000000000000000000000000000000000000000000000000000000"));
}

TEST(YMP_ZZTest, rem)
{
    ZZ a(3);
    a %= ZZ(2);
    ASSERT_EQ(a, ZZ("1"));

    a = ZZ("12314324325435664576576568787687576435342431432545345");
    a %= 743;
    ASSERT_EQ(a, ZZ(670));

    a = ZZ("1234567890123456789");
    ZZ b("987654321");
    a %= b;
    ASSERT_EQ(a, ZZ("725308641"));
}

TEST(YMP_ZZTest, no_digits)
{
    ASSERT_EQ(ZZ().no_digits(), 1);
    ASSERT_EQ(ZZ(1).no_digits(), 1);
    ASSERT_EQ(ZZ(2).no_digits(), 1);
    ASSERT_EQ(ZZ(3).no_digits(), 1);
    ASSERT_EQ(ZZ(4).no_digits(), 1);
    ASSERT_EQ(ZZ(5).no_digits(), 1);
    ASSERT_EQ(ZZ(6).no_digits(), 1);
    ASSERT_EQ(ZZ(7).no_digits(), 1);
    ASSERT_EQ(ZZ(8).no_digits(), 1);
    ASSERT_EQ(ZZ(9).no_digits(), 1);
    ASSERT_EQ(ZZ(10).no_digits(), 2);
    ASSERT_EQ(ZZ(99).no_digits(), 2);
    ASSERT_EQ(ZZ(100).no_digits(), 3);
    ZZ a = ZZ("1234567890123456789");
    ASSERT_EQ(a.no_digits(), 19);
    a = ZZ("-12314324325435664576576568787687576435342431432545345");
    ASSERT_EQ(a.no_digits(), 53);
    a = ZZ(-10);
    a.pow(93);
    ASSERT_EQ(a.no_digits(), 94);
}

TEST(YMP_ZZTest, gcd)
{
    ZZ a(-6);
    ZZ b(4);

    ZZ c;

    c = gcd(a, b);

    ASSERT_EQ(c, ZZ(2));
}
