/*  Pasang Emas. Enjoy a unique traditional game of Brunei.
    Copyright (C) 2010  Nor Jaidi Tuah

    This file is part of Pasang Emas.
      
    Pasang Emas is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
namespace Pasang {

/**
 * Extends Game to support undo and keep track of the number of wins
 */
public class GameSeries : Game {
    /**
     * When false, player 0 is drawn on top.
     * When true, player 1 is drawn on top.
     */
    public bool rotated;

    /**
     * num_wins[0] for the TOP player
     * num_wins[1] for the BOTTOM player
     */
    public int num_wins[2];
    public int num_rounds;

    /**
     * Information needed to support "Undo move".
     * recording records the current round, with record[0] storing the starting game pattern.
     */
    private string[] recording = {};
    private int undo_num_wins[2];
    private int undo_num_rounds;

    /**
     * Start a new series of games rounds
     */
    public void start_new_series () {
        num_rounds = 0;
        num_wins[0] = 0;
        num_wins[1] = 0;
    }

    /**
     * Update the number of wins. Must be called once only after each move
     */
    public void update_num_wins () {
        if (stage == Stage.GAME_OVER) {
            num_rounds++;
            // Is the top player playing first (0) or second (1)?
            int top = rotated ? 1 : 0;
            int bottom = 1 - top;
            int diff = score[top] - score[bottom];
            if (diff > 0) num_wins[0]++;
            else if (diff < 0) num_wins[1]++;
        }
    }

    /**
     * Start a game and record its starting pattern
     */
    public void start_recorded (string starting_pattern) {
        start (starting_pattern);
        recording = {starting_pattern};
        undo_num_rounds = num_rounds;   // A restore point, in case of undo after the game has completed
        undo_num_wins = num_wins;
    }

    /**
     * Perform a move and record it
     */
    public void perform_recorded (Move move, bool full_move) {
        perform (move, full_move);
        if (stage != Stage.SUB_MOVE && stage != Stage.SUB_SELECT) {
            recording += move.notation;
        }        
    }

    /**
     * Undo a move (by replaying the game)
     */
    public void undo () 
        requires (recording.length > 0) {
        if (recording.length == 1) return;   // recording[0] is the starting pattern and cannot be undone
        num_rounds = undo_num_rounds;
        num_wins = undo_num_wins;
        if (stage != Stage.SUB_MOVE && stage != Stage.SUB_SELECT) {
            recording.length --;
        }
        // Replay
        start (recording[0]);
        for (int i=1; i < recording.length; i++) {
            generate_moves ();
            perform_from_notation (recording[i]);
        }
    }

    /**
     * Return the starting pattern used to start the game
     */
    public string starting_pattern () {
        assert (recording.length > 0);
        return recording[0];
    }
}//class GameSeries
}//namespace
// vim: tabstop=4: expandtab: textwidth=100: autoindent:
