package processors

import (
	"path/filepath"
	"strings"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestIsAutogeneratedDetection(t *testing.T) {
	all := `
	// generated by stringer -type Pill pill.go; DO NOT EDIT

// Code generated by "stringer -type Pill pill.go"; DO NOT EDIT

// Code generated by vfsgen; DO NOT EDIT

// Created by cgo -godefs - DO NOT EDIT

/* Created by cgo - DO NOT EDIT. */

// Generated by stringer -i a.out.go -o anames.go -p ppc64
// Do not edit.

// DO NOT EDIT
// generated by: x86map -fmt=decoder ../x86.csv

// DO NOT EDIT.
// Generate with: go run gen.go -full -output md5block.go

// generated by "go run gen.go". DO NOT EDIT.

// DO NOT EDIT. This file is generated by mksyntaxgo from the RE2 distribution.

// GENERATED BY make_perl_groups.pl; DO NOT EDIT.

// generated by mknacl.sh - do not edit

// DO NOT EDIT ** This file was generated with the bake tool ** DO NOT EDIT //

// Generated by running
//  maketables --tables=all --data=http://www.unicode.org/Public/8.0.0/ucd/UnicodeData.txt
// --casefolding=http://www.unicode.org/Public/8.0.0/ucd/CaseFolding.txt
// DO NOT EDIT

/*
* CODE GENERATED AUTOMATICALLY WITH github.com/ernesto-jimenez/gogen/unmarshalmap
* THIS FILE SHOULD NOT BE EDITED BY HAND
*/

// AUTOGENERATED FILE: easyjson file.go
`

	generatedCases := strings.Split(all, "\n\n")
	for _, gc := range generatedCases {
		isGenerated := isGeneratedFileByComment(gc)
		assert.True(t, isGenerated)
	}

	notGeneratedCases := []string{
		"code not generated by",
		"test",
	}
	for _, ngc := range notGeneratedCases {
		isGenerated := isGeneratedFileByComment(ngc)
		assert.False(t, isGenerated)
	}
}

func TestGetDoc(t *testing.T) {
	testCases := []struct {
		fpath string
		doc   string
	}{
		{
			fpath: filepath.Join("testdata", "autogen_exclude.go"),
			doc: `first line
second line
third line
this text also
and this text also`,
		},
		{
			fpath: filepath.Join("testdata", "autogen_exclude_doc.go"),
			doc:   `DO NOT EDIT`,
		},
		{
			fpath: filepath.Join("testdata", "autogen_exclude_block_comment.go"),
			doc: `* first line
 *
 * second line
 * third line
and this text also
this type of block comment also
this one line comment also`,
		},
	}

	for _, tc := range testCases {
		doc, err := getDoc(tc.fpath)
		require.NoError(t, err)
		assert.Equal(t, tc.doc, doc)
	}
}

// Issue 954: Some lines can be very long, e.g. auto-generated
// embedded resources. Reported on file of 86.2KB.
func TestGetDocFileWithLongLine(t *testing.T) {
	fpath := filepath.Join("testdata", "autogen_exclude_long_line.go")
	_, err := getDoc(fpath)
	assert.NoError(t, err)
}
