/*
 * trajet.c --- verify trajectories.
 *
 * Copyright (c) 1997-2002 by Pascal Wassong All Rights Reserved.
 *
 * Time-stamp: <2002-08-15 13:10:52 Pascal>
 *
 * This file is part of Natch.
 *
 * Natch is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Natch is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include	"common.h"

#include	"trajet.h"

#include	"pcpj.h"
#include	"distance.h"
#include	"main.h"
#include	"precise_trajectories.h"

/*--------------------------------------------------------------------------*/

static bool_t	pawn_trajectories_are_not_correct( void );
static void	update_trajectories_due_to_stationary_pieces( void );

/*--------------------------------------------------------------------------*/

void
trajet(void)
{
    if (pawn_trajectories_are_not_correct())
    {
	return;
    }

    update_trajectories_due_to_stationary_pieces();
}

bool_t
trajet_traite_nouvelle_case_interdite(
    const exploration_t*	datas,
    square_t			nouvelle_case_interdite )
{
    int			i ;
    const piece_t*	p ;
    distance_t		distance_table[ 6 * SIZE_PIECE ];

    CasesInterdites[ nouvelle_case_interdite ] = TRUE;

    distance_create_table( distance_table );
    distance_push( distance_table );

    for ( i = 0, p = &( datas->pieces[ INDEX_ROI_BLANC ] ) ;
	  i < datas->nb_moving_pieces ;
	  i++, p++ )
    {
	int		indice_destination;
	piece_type_t	type_piece = p->typePiece;
	square_t	position_actuelle =
	    datas->pieces_squares[ i + INDEX_ROI_BLANC ];

	if (p->indDestAct == p->nbDestination - 1 &&
	    position_actuelle == p->caseArrivee)
	{
	    continue;
	}

	for (indice_destination = p->indDestAct;
	     indice_destination < p->nbDestination;
	     indice_destination++)
	{
	    if (p->casePromotion != CASE_PAS_DE_PROMOTION &&
		position_actuelle == p->casePromotion)
	    {
		/* Si la pice retourne sur sa case de promotion,
		 * cela ne pose aucun problme. */
		type_piece = p->typePromotion;
	    }

	    if (position_actuelle                  != CASE_ARRIVEE_QUELCONQUE &&
		p->destination[indice_destination] != CASE_ARRIVEE_QUELCONQUE )
	    {
		distance_t	delta = distance_delta(
		    type_piece,
		    p->camp,
		    position_actuelle,
		    p->destination[indice_destination]);
		if (delta > 0)
		{
		    if (( p->camp == BLANC && delta > NbCoupsBlancsRestants ) ||
			( p->camp == NOIR  && delta > NbCoupsNoirsRestants  ) )
		    {
/*  fprintf(MainFD, */
/*  	"Position OUT a cause de la piece : 0x%02x. Type = %d.\n", */
/*  	p->caseInitiale, p->typePiece); */
/*  fprintf(MainFD, */
/*  	"position_actuelle = 0x%02x. destination = 0x%02x. " */
/*  	position_actuelle, p->destination[indice_destination]); */
/*  fprintf(MainFD, */
/*  	"distance avant = %d, apres = %d. Delta = %d\n", */
/*  	p->distance[indice_destination], */
/*  	distance(type_piece, p->camp, position_actuelle, */
/*  		 p->destination[indice_destination]), */
/*  	delta); */
			distance_pop();
			CasesInterdites[ nouvelle_case_interdite ] = FALSE ;
			return FALSE ;
		    }
		}
	    }
	    position_actuelle = p->destination[ indice_destination ];
	}
    }

    distance_pop();
    CasesInterdites[ nouvelle_case_interdite ] = FALSE ;
    return TRUE;
}

/*--------------------------------------------------------------------------*/

static bool_t
pawn_trajectories_are_not_correct( void )
{
    int			i, j;
    piece_t*		piece;
    square_t		initial_square;
    square_t		destination_square;
    column_t		depX, arrX;
    square_delta_t	deltaY;
    piece_index_t	index_white, index_black;

    for (i=0, piece=ListeSures; i<NbSures+NbRestantes; i++, piece++)
    {
	if (piece->typePiece != PION)	continue;

	if (piece->nbDestination == 1 &&
	    piece->caseInitiale  == piece->destination[0])
	{
	    continue;
	}

	if (piece->camp == BLANC)	deltaY = 16;
	else				deltaY = -16;

	initial_square = piece->caseInitiale;
	depX = column(initial_square);

	/* Test to avoid a promoted pawn to be captured by a pawn bloquing
	 * his way to promotion.
	 */
	if ( piece->destination[ 0 ]
	     == piece->casePromotion /* Pawn promotes ... */
	     && piece->pieceCapturee[ 0 ]
	     == PAS_DE_CAPTURE	/* ... wihtout capture ...   */
	     && piece->pieceCapturante
	     != PIECE_PAS_CAPTUREE ) /* ... and is then captured. */
	{
	    piece_t*	pc = TabIndexPiece[ piece->pieceCapturante ];
	    if ( column( pc->caseInitiale ) == depX
		 && pc->pieceCapturee[ 0 ]  == piece->index
		 && pc->typePiece           == PION )
	    {
		return TRUE;
	    }
	}

	for (j=0; j<piece->nbDestination; j++)
	{
	    square_t	verified_square;
	    row_t	arrY;

	    destination_square = piece->destination[j];
	    arrX = column(destination_square);

	    /* Verify that if a pawn captures, he moves to an adjacent column.*/
	    if (piece->pieceCapturee[j] == PAS_DE_CAPTURE)
	    {
		if (depX != column(destination_square))
		{
		    return TRUE;
		}
	    }
	    else
	    {
		if (arrX - depX != 1 && depX - arrX != 1)
		{
		    return TRUE;
		}
	    }

	    /* Verify that a pawn does use a forbidden square. */
	    arrY = row(destination_square);
	    /* Peut etre considere comme inutile car la case d'arrivee
	     * n'est jamais une case interdite. Si une prise y a lieu
	     * alors la piece prise peut l'atteindre. Sinon c'est soit
	     * la case d'arrivee du pion, soit sa case de promotion qui
	     * ne peuvent pas etre interdite non plus.
	     * En fait ce test sert pour les pions immobiles, ceux qui
	     * se sont fait prendre sur place !?.
	     */
	    if (piece->pieceCapturee[j] == PAS_DE_CAPTURE)	arrY += deltaY;

	    verified_square = initial_square + deltaY;
	    /* arrY may have the value 0xF0 (due to `arrY += deltaY') ! */
	    while ((verified_square & 0xF0) != arrY)
	    {
		if (CasesInterdites[verified_square])
		{
		    return TRUE;
		}
		verified_square += deltaY;
	    }

	    if (piece->casePromotion == destination_square)	break;

	    initial_square = destination_square;
	    depX = column(initial_square);
	}
    }

    /* Verify that 2 pawns on the same column don't collide */
    for (index_white=INDEX_PION_BLANC_A, index_black=INDEX_PION_NOIR_A;
	 index_white<=INDEX_PION_BLANC_H;
	 index_white++, index_black++)
    {
	piece_t*	white_piece = TabIndexPiece[index_white];
	piece_t*	black_piece = TabIndexPiece[index_black];

	/* Test isn't done if pawn doesn't promote AND captures OR is captured*/
	if ((white_piece->destination[0] != white_piece->casePromotion &&
	     (white_piece->pieceCapturee[0] != PAS_DE_CAPTURE ||
	      (white_piece->nbDestination == 1 &&
	       white_piece->pieceCapturante != PIECE_PAS_CAPTUREE) ) ) ||
	    (black_piece->destination[0] != black_piece->casePromotion &&
	     (black_piece->pieceCapturee[0] != PAS_DE_CAPTURE ||
	      (black_piece->nbDestination == 1 &&
	       black_piece->pieceCapturante != PIECE_PAS_CAPTUREE) ) ) )
	{
	    continue;
	}

	if (row(white_piece->destination[0]) > row(black_piece->destination[0]))
	{
	    return TRUE; 
	}
    }

    return FALSE;
}

static void
update_trajectories_due_to_stationary_pieces( void )
{
    typedef struct
    {
	int		piece_indice;
	UCHAR		destination_index;
	distance_t	previous_distance;
    } new_distance_t;
    const int		max_nb_of_new_distances = 20;
    new_distance_t	new_distances[max_nb_of_new_distances];

    /* Cannot be > NbCoupsBlancsRestants + NbCoupsNoirsRestants */
    int			nb_new_distances = 0;

    piece_t*		piece;
    square_t		new_forbidden_squares[16];
    int			nb_new_forbidden_squares = 0;
    int			i, indice;
    int			white_spare_moves;
    int			black_spare_moves;
    distance_t		distance_table[6 * SIZE_PIECE];

    for (indice = 0, piece = &ListeSures[indice];
	 indice < NbSures + NbRestantes;
	 indice++, piece++)
    {
	if (piece->nbDestination   != 1                   ||
	    piece->caseArrivee     != piece->caseInitiale ||
	    piece->pieceCapturante != PIECE_PAS_CAPTUREE  )
	{
	    continue;
	}

	if ( ( piece->typePiece == ROI || piece->typePiece == TOUR) &&
	     ( piece->castling == KING_SIDE || piece->castling == QUEEN_SIDE ) )
	{
	    continue;
	}

	if ((piece->camp == BLANC && NbCoupsBlancsRestants < 2) ||
	    (piece->camp == NOIR  && NbCoupsNoirsRestants  < 2) )
	{
	    CasesInterdites[piece->caseInitiale] = TRUE;
	    new_forbidden_squares[nb_new_forbidden_squares] =
		piece->caseInitiale;
	    nb_new_forbidden_squares++;
	    continue;
	}
    }

    distance_create_table(distance_table);
    distance_push(distance_table);

    white_spare_moves = NbCoupsBlancsRestants;
    black_spare_moves = NbCoupsNoirsRestants;

    for (indice = 0, piece = &ListeSures[indice];
	 indice < NbSures + NbRestantes;
	 indice++, piece++)
    {
	int		destination_index;
	square_t	current_position = piece->caseInitiale;
	piece_type_t	type_piece       = piece->typePiece;

	if (piece->nbDestination == 1 &&
	    piece->caseInitiale  == piece->caseArrivee)
	{
	    continue;
	}

	for (destination_index = 0;
	     destination_index < piece->nbDestination;
	     destination_index++)
	{
	    if (piece->casePromotion != CASE_PAS_DE_PROMOTION &&
		current_position     == piece->casePromotion)
	    {
		/* Si la pice retourne sur sa case de promotion,
		 * cela ne pose aucun problme. */
		type_piece = piece->typePromotion;
	    }

	    /* We don't make the test if :
	     *	- the origin square is any square,
	     *	- the destination square is any square,
	     *	- a king is castling.
	     */
	    if ( current_position != CASE_ARRIVEE_QUELCONQUE &&
		 piece->destination[destination_index] !=
		 CASE_ARRIVEE_QUELCONQUE &&
		( piece->typePiece != ROI || piece->castling == NO_CASTLING ) )
	    {
		distance_t	new_distance = distance(
		    type_piece,
		    piece->camp,
		    current_position,
		    piece->destination[destination_index]);
		if (new_distance > piece->distance[destination_index])
		{
		    distance_t	delta =
			new_distance - piece->distance[destination_index];
		    if ( ( piece->camp == BLANC &&
			   delta > NbCoupsBlancsRestants ) ||
			 ( piece->camp == NOIR  &&
			   delta > NbCoupsNoirsRestants ) )
		    {
/*  fprintf(MainFD, */
/*  	"Position OUT a cause de la piece : 0x%02x. Type = %d(%d).\n", */
/*  	piece->caseInitiale, piece->typePiece, type_piece); */
/*  fprintf(MainFD, */
/*  	"current_position = 0x%02x. destination = 0x%02x. ", */
/*  	current_position, */
/*  	piece->destination[destination_index] ); */
/*  fprintf(MainFD, "distance = %d. Nouvelle distance = %d\n", */
/*  	piece->distance[destination_index], */
/*  	new_distance); */
/*  writeAssociation( "" ); */
			distance_pop();
			for (i=0; i<nb_new_distances; i++)
			{
			    ListeSures[new_distances[i].piece_indice].distance[
				new_distances[i].destination_index] =
				new_distances[i].previous_distance;
			}
			NbCoupsBlancsRestants = white_spare_moves;
			NbCoupsNoirsRestants  = black_spare_moves;
			for (i=0; i<nb_new_forbidden_squares; i++)
			{
			    CasesInterdites[new_forbidden_squares[i]] = FALSE;
			}
			return;
		    }
		    if (nb_new_distances < max_nb_of_new_distances)
		    {
			new_distances[nb_new_distances].piece_indice = indice;
			new_distances[nb_new_distances].destination_index =
			    destination_index;
			new_distances[nb_new_distances].previous_distance =
			    piece->distance[destination_index];
			piece->distance[destination_index] = new_distance;
			nb_new_distances++;

			if (piece->camp == BLANC)
			{
			    NbCoupsBlancsRestants -= delta;
			}
			else
			{
			    NbCoupsNoirsRestants  -= delta;
			}
		    }
		    else
		    {
			fprintf(MainFD,
				"Warning : maximum number of new distances reached in update_trajectories_due_to_stationary_pieces\n");
		    }
		}
	    }
	    current_position = piece->destination[destination_index];
	}
    }

    precise_trajectories( 0, 0 );

    distance_pop();

    for (i=0; i<nb_new_distances; i++)
    {
	ListeSures[new_distances[i].piece_indice]
	    .distance[new_distances[i].destination_index] =
	    new_distances[i].previous_distance;
    }
    NbCoupsBlancsRestants = white_spare_moves;
    NbCoupsNoirsRestants  = black_spare_moves;

    for (i=0; i<nb_new_forbidden_squares; i++)
    {
	CasesInterdites[new_forbidden_squares[i]] = FALSE;
    }
}
