/*
 * pcpjcapt.c --- position searching (dealing with captures) package.
 *
 * Copyright (c) 1997-2003 by Pascal Wassong All Rights Reserved.
 *
 * Time-stamp: <2003-03-06 14:37:43 pascal>
 *
 * This file is part of Natch.
 *
 * Natch is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Natch is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include	"common.h"

#include	"pcpjcapt.h"
#include	"types.h"
#include	"pcpj.h"
#include	"captInitSquare.h"
#include	"distimmo.h"
#include	"screen.h"	/* For kbhit() & screen...(...) */
#include	"distance.h"
#include	"piece.h"

static	piece_type_t	TabTypePromotion[] =
{ DAME, TOUR, FOU, CAVALIER };
static const unsigned int Nb_Type_Promotion =
sizeof( TabTypePromotion ) / sizeof( piece_type_t );

extern unsigned long NbPosition ;

static void
captureParQui(int indice)
{
    int			i, j, k;
    piece_t*		p;
    piece_t*		p2;
    distance_t		dist1, dist2, delta;
    square_t		caseInitiale;
    int			indicePromotion;
    piece_type_t	typePiece;

    if (indice == NbSures + NbRestantes)
    {
	capturer_on_initial_square_must_switchback( 0 );
	return;
    }

    p = &ListeSures[indice];
    while ( indice < NbSures + NbRestantes &&
	    p->pieceCapturante != PIECE_CAPTURANTE_INCONNUE )
    {
	indice++;
	p++;
    }

    if (indice == NbSures + NbRestantes)
    {
	capturer_on_initial_square_must_switchback( 0 );
	return;
    }

    if (p->camp == BLANC)
    {
	NbResteACapturerBlanches--;
    }
    else
    {
	NbResteACapturerNoires--;
    }

    /* Cherche une piece qui a pu la capturer */

    for (i=0, p2=ListeSures; i < NbSures + NbRestantes; i++, p2++)
    {
	bool_t	piece_etait_immobile ;

	if (p2->camp == p->camp || p2->pieceCapturante == p->index)
	{
	    continue;
	}

	piece_etait_immobile = distance_desimmobilise_piece_sur_case(
	    p2->caseInitiale );

	indicePromotion = INFINI ;
	typePiece = p2->typePiece;
	j = 0 ;
	if ( typePiece == PION )
	{
	    column_t	column_last_pawn_capture ;
	    column_t	column_pawn_destination  ;
	    int		index_last_pawn_capture  ;

	    if ( p2->casePromotion != CASE_PAS_DE_PROMOTION )
	    {
		indicePromotion = promotion_index( p2 ) + 1;
		column_pawn_destination = column( p2->casePromotion );
	    }
	    else column_pawn_destination = column( p2->caseArrivee );

	    index_last_pawn_capture = 0 ;
	    while ( column( p2->destination[ index_last_pawn_capture ] ) !=
		    column_pawn_destination )
	    {
		index_last_pawn_capture++;
	    }
	    column_last_pawn_capture =
		column( p2->destination[ index_last_pawn_capture ] );

	    if ( column( p2->caseInitiale ) != column_last_pawn_capture )
	    {
		/* If a pawn goes from column `a' (his initial column) to a
		 * column `b' (his final column as a pawn, if it's a promotion,
		 * the promoted piece may of course move), he is not allowed to
		 * capture in the same direction BEFORE the captures leading him
		 * to column `b', to avoid the solution to be found twice.
		 */

		int	delta_X_end_pawn 	;
		int	delta_X_current_capture ;

		delta_X_end_pawn =
		    column( p2->caseInitiale ) < column_last_pawn_capture
		    ? 1 : -1 ;
		if ( column( p2->caseInitiale ) == column( p->caseArrivee ) )
		{
		    delta_X_current_capture = 0 ;
		}
		else
		{
		    delta_X_current_capture =
			column( p2->caseInitiale ) < column( p->caseArrivee )
			? 1 : -1 ;
		}
		if ( delta_X_end_pawn == delta_X_current_capture )
		{
		    /* nb_captures isn't the right value, AND it isn't updated
		     * in this funcion even if the pawn makes a new capture.
		     * These are 2 reasons not to use nb_captures.
		     */
		    j = 0 ;
		    while ( j < index_last_pawn_capture &&
			    column( p2->destination[ j ] ) !=
			    column( p->caseArrivee ) )
		    {
			j++;
		    }
		    j++;
		}
	    }
	}
	else if ( typePiece == ROI && p2->castling != NO_CASTLING )
	{
	    /* We skip the castling move, which cannot capture */
	    j = 1;
	}

	for (; j<=p2->nbDestination; j++)
	{
	    if (j >= indicePromotion)	typePiece = p2->typePromotion;

	    if ( j == 0 )	caseInitiale = p2->caseInitiale ;
	    else		caseInitiale = p2->destination[ j - 1 ];

	    /* Cas general : RNe8-e7 -> RNe8xAaa-e7			+1
	     *
	     * Cas particuliers : (le / signifie case d'arrivee)
	     *		a) RNe8-e8	-> RNe8-??xAe8			+1
	     *		b) RNe8xFe7	-> RNe8xAe7-??xFe7		+2
	     *		c) RNe8-??xFe8	-> RNe8xAaaxFe8			=
	     *		d) RNe8-??xFe8	-> RNe8-??xAe8-??xFe8		+2
	     *		e) RNe8-??-e8	-> RNe8-??xAe8			=
	     *		f) RNe8-e7	-> RNe8xAe7			=
	     *		g) RNe8xFe7/	-> RNe8xFe7xAaa-e7		+2
	     *		   PBh2-h8=X/	-> PBh2-h8=XxAaa-h8		+2
	     *		h) RNe8xFe7/	-> RNe8xFe7-??xAe7		+2
	     *		   PBh2-h8=X/	-> PBh2-h8=X-??xAh8		+2
	     *		i) RNe8-e7	-> RNe8-??xAe8-e7		+2
	     *
	     * Le cas RNe8-e7 -> RNe8-e7xAaa-e7 ne correspond a rien.
	     * L'etape RNe8-e7x... doit etre supprime puisqu'elle n'a
	     * rien d'obligatoire. Cela est donc traite par le cas general.
	     */

	    if (j == p2->nbDestination)
	    {
		if (p2->pieceCapturee[j-1] == PAS_DE_CAPTURE &&
		    j != indicePromotion)
		{
		    continue;
		}

		if (caseInitiale == p->caseArrivee)
		{
		    /* Cas particulier h) : RNe8xFe7 -> RNe8xFe7-??xAe7 +2 */
		    if ((p2->camp == BLANC && NbCoupsBlancsRestants < 2) ||
			(p2->camp == NOIR && NbCoupsNoirsRestants < 2))
		    {
			continue;
		    }

		    p2->destination[j]     = CASE_ARRIVEE_QUELCONQUE;
		    p2->distance[j]        = 1;
		    p2->pieceCapturee[j]   = PAS_DE_CAPTURE;
		    p2->destination[j+1]   = p->caseArrivee;
		    p2->distance[j+1]      = 1;
		    p2->pieceCapturee[j+1] = p->index;
		    p2->nbDestination      += 2;
		    p->pieceCapturante     = p2->index;

		    if (p2->camp == BLANC)	NbCoupsBlancsRestants -= 2;
		    else			NbCoupsNoirsRestants  -= 2;

		    captureParQui(indice + 1);

		    if (p2->camp == BLANC)	NbCoupsBlancsRestants += 2;
		    else			NbCoupsNoirsRestants  += 2;

		    p->pieceCapturante = PIECE_CAPTURANTE_INCONNUE;
		    p2->nbDestination  -= 2;
		}
		else
		{
		    /* Cas particulier g) : RNe8xFe7 -> RNe8xFe7xAaa-e7 +2 */

		    piece_index_t	pieces_ayant_du_bouger[16];
		    int			nombre_pieces_ayant_du_bouger;

		    dist1 = distance_avec_obstacle(
			typePiece,
			p2->camp,
			caseInitiale,
			p->caseArrivee,
			TRUE,
			pieces_ayant_du_bouger,
			&nombre_pieces_ayant_du_bouger);

		    delta = 2 * dist1;
		    if ((p2->camp == BLANC && NbCoupsBlancsRestants < delta) ||
			(p2->camp == NOIR  && NbCoupsNoirsRestants  < delta) )
		    {
			distance_reimmobilise_pieces(
			    pieces_ayant_du_bouger,
			    nombre_pieces_ayant_du_bouger);
			continue;
		    }

		    p2->destination[j]     = p->caseArrivee;
		    p2->distance[j]        = dist1;
		    p2->pieceCapturee[j]   = p->index;
		    p2->destination[j+1]   = caseInitiale;
		    p2->distance[j+1]      = dist1;
		    p2->pieceCapturee[j+1] = PAS_DE_CAPTURE;
		    p2->nbDestination      += 2;
		    p->pieceCapturante     = p2->index;

		    if (p2->camp == BLANC)	NbCoupsBlancsRestants -= delta;
		    else			NbCoupsNoirsRestants  -= delta;

		    captureParQui(indice + 1);

		    if (p2->camp == BLANC)	NbCoupsBlancsRestants += delta;
		    else			NbCoupsNoirsRestants  += delta;

		    p->pieceCapturante = PIECE_CAPTURANTE_INCONNUE;
		    p2->nbDestination  -= 2;

		    distance_reimmobilise_pieces(
			pieces_ayant_du_bouger,
			nombre_pieces_ayant_du_bouger);
		}
	    }
	    else if (p2->destination[j] == CASE_ARRIVEE_QUELCONQUE)
	    {
		if (p2->destination[j+1] != p->caseArrivee)
		{
		    /* Cas particulier c) : RNe8-??xFe8 -> RNe8xAaaxFe8 = */

		    piece_index_t	pieces_ayant_du_bouger[16];
		    int			nombre_pieces_ayant_du_bouger;

		    if ( p2->camp == BLANC)	NbCoupsBlancsRestants += 2;
		    else			NbCoupsNoirsRestants  += 2;

		    dist1 = distance_avec_obstacle(
			typePiece,
			p2->camp,
			caseInitiale,
			p->caseArrivee,
			FALSE,
			pieces_ayant_du_bouger,
			&nombre_pieces_ayant_du_bouger);

		    delta = 2 * dist1 ;
		    if ((p2->camp == BLANC && NbCoupsBlancsRestants < delta) ||
			(p2->camp == NOIR  && NbCoupsNoirsRestants  < delta) )
		    {
			j++;	/* Saute le retour de la piece. */
			distance_reimmobilise_pieces(
			    pieces_ayant_du_bouger,
			    nombre_pieces_ayant_du_bouger);
			if ( p2->camp == BLANC)	NbCoupsBlancsRestants -= 2;
			else			NbCoupsNoirsRestants  -= 2;
			continue;
		    }
		    p2->pieceCapturee[j] = p->index;
		    p2->destination[j]   = p->caseArrivee;
		    p2->distance[j]      = dist1;
		    p2->distance[j+1]    = dist1;
		    p->pieceCapturante   = p2->index;

		    if (p2->camp == BLANC)	NbCoupsBlancsRestants -= delta;
		    else			NbCoupsNoirsRestants  -= delta;

		    captureParQui(indice + 1);

		    if (p2->camp == BLANC)	NbCoupsBlancsRestants += delta;
		    else			NbCoupsNoirsRestants  += delta;

		    p->pieceCapturante   = PIECE_CAPTURANTE_INCONNUE;
		    p2->distance[j+1]    = 1;
		    p2->distance[j]      = 1;
		    p2->destination[j]   = CASE_ARRIVEE_QUELCONQUE;
		    p2->pieceCapturee[j] = PAS_DE_CAPTURE;

		    j++;	/* Saute le retour de la piece. */

		    distance_reimmobilise_pieces(
			pieces_ayant_du_bouger,
			nombre_pieces_ayant_du_bouger);

		    if ( p2->camp == BLANC)	NbCoupsBlancsRestants -= 2;
		    else			NbCoupsNoirsRestants  -= 2;
		}
		else if (p2->pieceCapturee[j+1] == PAS_DE_CAPTURE)
		{
		    /* Cas particulier e) : RNe8-??-e8 -> RNe8-??xAe8 = */
		    p2->pieceCapturee[j+1] = p->index;
		    p->pieceCapturante     = p2->index;

		    captureParQui(indice + 1);

		    p->pieceCapturante     = PIECE_CAPTURANTE_INCONNUE;
		    p2->pieceCapturee[j+1] = PAS_DE_CAPTURE;

		    j++;
		}
		else
		{
		    /* Cas particulier d) : RNe8-??xFe8->RNe8-??xAe8-??xFe8+2*/
		    if ((p2->camp == BLANC && NbCoupsBlancsRestants < 2) ||
			(p2->camp == NOIR  && NbCoupsNoirsRestants  < 2) )
		    {
			j++;
			continue;
		    }

		    /* Insertion avant le courant */
		    for (k=p2->nbDestination+1; k>j+1; k--)
		    {
			p2->destination[k]   = p2->destination[k-2];
			p2->pieceCapturee[k] = p2->pieceCapturee[k-2];
			p2->distance[k]      = p2->distance[k-2];
		    }
		    p2->nbDestination += 2;

		    p2->pieceCapturee[j+1] = p->index;
		    p->pieceCapturante     = p2->index;

		    if (p2->camp == BLANC)	NbCoupsBlancsRestants -= 2;
		    else			NbCoupsNoirsRestants  -= 2;

		    captureParQui(indice + 1);

		    if (p2->camp == BLANC)	NbCoupsBlancsRestants += 2;
		    else			NbCoupsNoirsRestants  += 2;

		    p->pieceCapturante = PIECE_CAPTURANTE_INCONNUE;

		    p2->nbDestination      -= 2;
		    p2->pieceCapturee[j+1] = p2->pieceCapturee[j+3];
		    for (k=j+2;k<p2->nbDestination;k++)
		    {
			p2->destination[k]   = p2->destination[k+2];
			p2->pieceCapturee[k] = p2->pieceCapturee[k+2];
			p2->distance[k]      = p2->distance[k+2];
		    }

		    j++;
		}
	    }
	    else if (p2->destination[j] == p->caseArrivee &&
		     p2->distance[j] == 0)
	    {
		/* Cas particulier a) : RNe8-e8 -> RNe8-??xAe8 +1 */
		if ((p2->camp == BLANC && NbCoupsBlancsRestants < 2) ||
		    (p2->camp == NOIR  && NbCoupsNoirsRestants  < 2) )
		{
		    continue;
		}
		p2->destination[j]     = CASE_ARRIVEE_QUELCONQUE;
		p2->distance[j]        = 1;
		p2->pieceCapturee[j]   = PAS_DE_CAPTURE;
		p2->destination[j+1]   = p->caseArrivee;
		p2->distance[j+1]      = 1;
		p2->pieceCapturee[j+1] = p->index;
		p2->nbDestination++;
		p->pieceCapturante     = p2->index;

		if (p2->camp == BLANC)	NbCoupsBlancsRestants -= 2;
		else			NbCoupsNoirsRestants  -= 2;

		captureParQui(indice + 1);

		if (p2->camp == BLANC)	NbCoupsBlancsRestants += 2;
		else			NbCoupsNoirsRestants  += 2;

		p->pieceCapturante   = PIECE_CAPTURANTE_INCONNUE;
		p2->nbDestination--;
		p2->destination[j]   = p->caseArrivee;
		p2->distance[j]      = 0;
		p2->pieceCapturee[j] = PAS_DE_CAPTURE;
	    }
	    else if (p2->destination[j] == p->caseArrivee)
	    {
		if (p2->pieceCapturee[j] == PAS_DE_CAPTURE)
		{
		    /* Cas particulier f) : RNe8-e7 -> RNe8xAe7 = */
		    p2->pieceCapturee[j] = p->index;
		    p->pieceCapturante   = p2->index;

		    captureParQui(indice + 1);

		    p->pieceCapturante   = PIECE_CAPTURANTE_INCONNUE;
		    p2->pieceCapturee[j] = PAS_DE_CAPTURE;
		}
		else
		{
		    /* Cas particulier b) : RNe8xFe7 -> RNe8xAe7-??xFe7 +2 */
		    /* This case is forbidden in a very special case : when the
		     * current capture is made by a pawn on its promotion
		     * square.  If we allowed that, some solutions could be
		     * found twice.
		     */
		    if ( ( p2->camp == BLANC && NbCoupsBlancsRestants < 2 )
			 || ( p2->camp == NOIR && NbCoupsNoirsRestants < 2 )
			 || j == indicePromotion - 1 )
		    {
			continue;
		    }

		    /* Insertion avant le courant */
		    for (k=p2->nbDestination+1; k>j+1; k--)
		    {
			p2->destination[k]   = p2->destination[k-2];
			p2->pieceCapturee[k] = p2->pieceCapturee[k-2];
			p2->distance[k]      = p2->distance[k-2];
		    }
		    p2->nbDestination += 2;

		    p2->pieceCapturee[j]   = p->index;
		    p2->destination[j+1]   = CASE_ARRIVEE_QUELCONQUE;
		    p2->distance[j+1]      = 1;
		    p2->pieceCapturee[j+1] = PAS_DE_CAPTURE;
		    p2->distance[j+2]      = 1;
		    p->pieceCapturante     = p2->index;

		    if (p2->camp == BLANC)	NbCoupsBlancsRestants -= 2;
		    else			NbCoupsNoirsRestants  -= 2;

		    captureParQui(indice + 1);

		    if (p2->camp == BLANC)	NbCoupsBlancsRestants += 2;
		    else			NbCoupsNoirsRestants  += 2;

		    p->pieceCapturante = PIECE_CAPTURANTE_INCONNUE;

		    p2->nbDestination -= 2;
		    p2->distance[j+2] = p2->distance[j]; /* Astuce ! */
		    for (k=j; k<p2->nbDestination; k++)
		    {
			p2->destination[k]   = p2->destination[k+2];
			p2->pieceCapturee[k] = p2->pieceCapturee[k+2];
			p2->distance[k]      = p2->distance[k+2];
		    }
		}
	    }
	    else if (caseInitiale == p->caseArrivee)
	    {
		/* Cas particulier i) RNe8-e7 -> RNe8-??xAe8-e7 +2 */
		if ((typePiece == PION) ||
		    (p2->camp == BLANC && NbCoupsBlancsRestants < 2) ||
		    (p2->camp == NOIR  && NbCoupsNoirsRestants  < 2) )
		{
		    continue;
		}

		/* Insertion avant le courant */
		for (k=p2->nbDestination+1; k>j+1; k--)
		{
		    p2->destination[k]   = p2->destination[k-2];
		    p2->pieceCapturee[k] = p2->pieceCapturee[k-2];
		    p2->distance[k]      = p2->distance[k-2];
		}
		p2->nbDestination += 2;

		p2->destination[j]     = CASE_ARRIVEE_QUELCONQUE;
		p2->distance[j]        = 1;
		p2->pieceCapturee[j]   = PAS_DE_CAPTURE;
		p2->destination[j+1]   = caseInitiale;
		p2->distance[j+1]      = 1;
		p2->pieceCapturee[j+1] = p->index;
		p->pieceCapturante     = p2->index;

		if (p2->camp == BLANC)	NbCoupsBlancsRestants -= 2;
		else			NbCoupsNoirsRestants  -= 2;

		captureParQui(indice + 1);

		if (p2->camp == BLANC)	NbCoupsBlancsRestants += 2;
		else			NbCoupsNoirsRestants  += 2;

		p->pieceCapturante = PIECE_CAPTURANTE_INCONNUE;

		p2->nbDestination -= 2;
		for (k=j; k<p2->nbDestination; k++)
		{
		    p2->destination[k]   = p2->destination[k+2];
		    p2->pieceCapturee[k] = p2->pieceCapturee[k+2];
		    p2->distance[k]      = p2->distance[k+2];
		}
	    }
	    else
	    {
		/* Cas general : RNe8-e7 -> RNe8xAaa-e7 +1 */

		piece_index_t	pieces_ayant_du_bouger1[16];
		int		nombre_pieces_ayant_du_bouger1;
		piece_index_t	pieces_ayant_du_bouger2[16];
		int		nombre_pieces_ayant_du_bouger2;
		distance_t	ancDist, totalDist ;

		ancDist = p2->distance[j];

		/* Must be done 'cause distance_avec_obstacle makes a
		 * test on the total number of moves available.
		 */
		if ( p2->camp == BLANC ) NbCoupsBlancsRestants += ancDist ;
		else			 NbCoupsNoirsRestants  += ancDist ;

		dist1 = distance_avec_obstacle(typePiece,
					       p2->camp,
					       caseInitiale,
					       p->caseArrivee,
					       FALSE,
					       pieces_ayant_du_bouger1,
					       &nombre_pieces_ayant_du_bouger1);
		if (dist1 == INFINI)
		{
		    if ( p2->camp == BLANC ) NbCoupsBlancsRestants -= ancDist ;
		    else		     NbCoupsNoirsRestants  -= ancDist ;
		    continue;
		}

		dist2 = distance_avec_obstacle(typePiece,
					       p2->camp,
					       p->caseArrivee,
					       p2->destination[j],
					       FALSE,
					       pieces_ayant_du_bouger2,
					       &nombre_pieces_ayant_du_bouger2);

		totalDist = dist1 + dist2 ;
		if (( p2->camp == BLANC && totalDist > NbCoupsBlancsRestants) ||
		    ( p2->camp == NOIR  && totalDist > NbCoupsNoirsRestants ) )
		{
		    distance_reimmobilise_pieces(
			pieces_ayant_du_bouger1,
			nombre_pieces_ayant_du_bouger1 );
		    if ( dist2 != INFINI )
		    {
			distance_reimmobilise_pieces(
			    pieces_ayant_du_bouger2,
			    nombre_pieces_ayant_du_bouger2 );
		    }
		    if ( p2->camp == BLANC ) NbCoupsBlancsRestants -= ancDist ;
		    else		     NbCoupsNoirsRestants  -= ancDist ;
		    continue;
		}

		/* Insertion avant le courant */
		for (k=p2->nbDestination; k>j; k--)
		{
		    p2->destination[k]   = p2->destination[k-1];
		    p2->pieceCapturee[k] = p2->pieceCapturee[k-1];
		    p2->distance[k]      = p2->distance[k-1];
		}
		p2->nbDestination++;

		p2->destination[j]   = p->caseArrivee;
		p2->pieceCapturee[j] = p->index;
		p2->distance[j]      = dist1;
		p2->distance[j+1]    = dist2;

		p->pieceCapturante = p2->index;

		if (p2->camp == BLANC)	NbCoupsBlancsRestants -= totalDist ;
		else			NbCoupsNoirsRestants  -= totalDist ;

		captureParQui(indice + 1);

		if (p2->camp == BLANC)	NbCoupsBlancsRestants += totalDist ;
		else			NbCoupsNoirsRestants  += totalDist ;

		p->pieceCapturante = PIECE_CAPTURANTE_INCONNUE;

		p2->nbDestination--;
		for (k=j; k<p2->nbDestination; k++)
		{
		    p2->destination[k]   = p2->destination[k+1];
		    p2->pieceCapturee[k] = p2->pieceCapturee[k+1];
		    p2->distance[k]      = p2->distance[k+1];
		}
		p2->distance[j] = ancDist;

		distance_reimmobilise_pieces(
		    pieces_ayant_du_bouger1,
		    nombre_pieces_ayant_du_bouger1 );
		distance_reimmobilise_pieces(
		    pieces_ayant_du_bouger2,
		    nombre_pieces_ayant_du_bouger2 );
		if ( p2->camp == BLANC ) NbCoupsBlancsRestants -= ancDist ;
		else			 NbCoupsNoirsRestants  -= ancDist ;
	    }
	}

	if ( piece_etait_immobile )
	{
	    distance_reimmobilise_piece_sur_case( p2->caseInitiale );
	}
    }

    if (p->camp == BLANC)
    {
	NbResteACapturerBlanches++;
    }
    else
    {
	NbResteACapturerNoires++;
    }
}

static void
capturePieceRestante(int indice)
{
    square_t		i;
    piece_t*		p;
    distance_t		dist;
    piece_index_t	pieces_ayant_du_bouger[16];
    int			nombre_pieces_ayant_du_bouger;

    if (indice == NbNonTraites)
    {
	captureParQui(0);
	return;
    }

    p = &ListeNonTraitees[indice];
    while (indice < NbNonTraites && p->caseInitiale == PIECE_DEJA_TRAITEE)
    {
	indice++;
	p++;
    }

    if (indice == NbNonTraites)
    {
	captureParQui(0);
	return;
    }

#ifdef HAVE_LIBNCURSES
    if ( MainVisualFlag && keyboardHit() )
    {
	(void)screenKeyboard( FALSE, NULL, 0, 0 );
    }
#endif /* HAVE_LIBNCURSES */

    /* Teste si la piece a ete prise sur place par manque de coups. */
    if ((p->camp == BLANC && NbCoupsBlancsRestants == 0) ||
	(p->camp == NOIR  && NbCoupsNoirsRestants  == 0) )
    {
	ListeRestantes[NbRestantes] = *p;
	ListeRestantes[NbRestantes].caseArrivee      = p->caseInitiale;
	ListeRestantes[NbRestantes].nbDestination    = 1;
	ListeRestantes[NbRestantes].destination[0]   = p->caseInitiale;
	ListeRestantes[NbRestantes].pieceCapturee[0] = PAS_DE_CAPTURE;
	ListeRestantes[NbRestantes].distance[0]      = 0;
	ListeRestantes[NbRestantes].pieceCapturante  =PIECE_CAPTURANTE_INCONNUE;
	TabIndexPiece[p->index] = &ListeRestantes[NbRestantes];
	p->caseInitiale = PIECE_DEJA_TRAITEE;
	NbRestantes++;

	capturePieceRestante(indice + 1);

	NbRestantes--;
	p->caseInitiale = ListeRestantes[NbRestantes].caseInitiale;
	TabIndexPiece[p->index] = p;

	return;
    }

    for (i=a1; i<=h8; i++)
    {
	if (CasesInterdites[i])		continue;

	dist = distance_avec_obstacle(p->typePiece,
				      p->camp,
				      p->caseInitiale,
				      i,
				      TRUE,
				      pieces_ayant_du_bouger,
				      &nombre_pieces_ayant_du_bouger);

	if (dist == INFINI)
	{
	    continue;
	}

	if (p->camp == BLANC)		NbCoupsBlancsRestants -= dist;
	else if (p->camp == NOIR) 	NbCoupsNoirsRestants  -= dist;

	ListeRestantes[NbRestantes] = *p;
	ListeRestantes[NbRestantes].caseArrivee      = i;
	ListeRestantes[NbRestantes].nbDestination    = 1;
	ListeRestantes[NbRestantes].destination[0]   = i;
	ListeRestantes[NbRestantes].pieceCapturee[0] = PAS_DE_CAPTURE;
	ListeRestantes[NbRestantes].distance[0]      = dist;
	ListeRestantes[NbRestantes].pieceCapturante
	    = PIECE_CAPTURANTE_INCONNUE;
	TabIndexPiece[p->index] = &ListeRestantes[NbRestantes];
	p->caseInitiale = PIECE_DEJA_TRAITEE;
	NbRestantes++;

	capturePieceRestante(indice + 1);

	NbRestantes--;
	p->caseInitiale = ListeRestantes[NbRestantes].caseInitiale;
	TabIndexPiece[p->index] = p;

	if (p->camp == BLANC)
	{
	    NbCoupsBlancsRestants += dist;
	}
	else
	{
	    NbCoupsNoirsRestants  += dist;
	}
	distance_reimmobilise_pieces(
	    pieces_ayant_du_bouger, nombre_pieces_ayant_du_bouger );
    }
}

static void
capturePionRestant(int indice)
{
    square_t		i;
    piece_t*		p;
    distance_t		dist, distPion;
    bool_t		ok;
    row_t		rangeePromotion;
    piece_type_t	typePromotion;
    piece_index_t	pieces_ayant_du_bouger[16];
    int			nombre_pieces_ayant_du_bouger;

    if (indice == NbNonTraites)
    {
	capturePieceRestante(0);
	return;
    }

    p = &ListeNonTraitees[indice];
    while (indice < NbNonTraites &&
	   (p->caseInitiale == PIECE_DEJA_TRAITEE || p->typePiece != PION) )
    {
	indice++;
	p++;
    }

    if (indice == NbNonTraites)
    {
	capturePieceRestante(0);
	return;
    }

    /* Teste si la piece a ete prise sur place par manque de coups. */
    if ((p->camp == BLANC && NbCoupsBlancsRestants == 0) ||
	(p->camp == NOIR  && NbCoupsNoirsRestants  == 0) )
    {
	ListeRestantes[NbRestantes] = *p;
	ListeRestantes[NbRestantes].caseArrivee      = p->caseInitiale;
	ListeRestantes[NbRestantes].nbDestination    = 1;
	ListeRestantes[NbRestantes].destination[0]   = p->caseInitiale;
	ListeRestantes[NbRestantes].pieceCapturee[0] = PAS_DE_CAPTURE;
	ListeRestantes[NbRestantes].distance[0]      = 0;
	ListeRestantes[NbRestantes].pieceCapturante
	    = PIECE_CAPTURANTE_INCONNUE;
	TabIndexPiece[p->index] = &ListeRestantes[NbRestantes];
	p->caseInitiale = PIECE_DEJA_TRAITEE;
	NbRestantes++;

	capturePionRestant(indice + 1);

	NbRestantes--;
	p->caseInitiale = ListeRestantes[NbRestantes].caseInitiale;
	TabIndexPiece[p->index] = p;

	return;
    }

    rangeePromotion = (p->camp == BLANC) ? 0x70 : 0x00;

    for (i=a1; i<=h8; i++)
    {
	int		colProm;
	column_t	depX, arrX;
	square_delta_t	deltaX;

	if (CasesInterdites[i])		continue;

	/* Le PION a ete pris sans promotion.
	 * Attention, ce n'est pas parce qu'un pion ne peut pas atteindre
	 * une case sans se promouvoir qu'il ne peut jamais l'atteindre.
	 * Il ne faut donc pas faire un `continue'.
	 */
	if (row(i) != rangeePromotion)
	{
	    dist = distancePion(p->camp, p->caseInitiale, i);

	    ok = FALSE;

	    if ((p->camp == BLANC && dist <= NbCoupsBlancsRestants) ||
		(p->camp == NOIR  && dist <= NbCoupsNoirsRestants ) )
	    {
		ok = TRUE;
	    }

	    /* Si le PION a change de colonne, verifie qu'il reste
	     * assez de pieces a prendre.
	     */
	    if (ok)
	    {
		depX = column(p->caseInitiale);
		arrX = column(i);
		deltaX = (arrX > depX) ? (arrX - depX) : (depX - arrX);

		if ((p->camp == BLANC && deltaX > NbResteACapturerNoires  ) ||
		    (p->camp == NOIR  && deltaX > NbResteACapturerBlanches) )
		{
		    ok = FALSE;
		}
	    }

	    if (ok)
	    {
		if (p->camp == BLANC)
		{
		    NbCoupsBlancsRestants  -= dist;
		    NbResteACapturerNoires -= deltaX;
		}
		else
		{
		    NbCoupsNoirsRestants     -= dist;
		    NbResteACapturerBlanches -= deltaX;
		}

		ListeRestantes[NbRestantes] = *p;
		ListeRestantes[NbRestantes].caseArrivee      = i;
		ListeRestantes[NbRestantes].nbDestination    = 1;
		ListeRestantes[NbRestantes].destination[0]   = i;
		ListeRestantes[NbRestantes].pieceCapturee[0] = PAS_DE_CAPTURE;
		ListeRestantes[NbRestantes].distance[0]      = dist;
		ListeRestantes[NbRestantes].pieceCapturante
		    = PIECE_CAPTURANTE_INCONNUE;
		ListeRestantes[NbRestantes].nb_captures      = deltaX;
		TabIndexPiece[p->index] = &ListeRestantes[NbRestantes];
		p->caseInitiale = PIECE_DEJA_TRAITEE;
		NbRestantes++;

		if (deltaX == 0) capturePionRestant( indice + 1 );
		else		 captureParPion( NbSures + NbRestantes - 1 );

		NbRestantes--;
		p->caseInitiale = ListeRestantes[NbRestantes].caseInitiale;
		TabIndexPiece[p->index] = p;

		if (p->camp == BLANC)
		{
		    NbCoupsBlancsRestants  += dist;
		    NbResteACapturerNoires += deltaX;
		}
		else
		{
		    NbCoupsNoirsRestants     += dist;
		    NbResteACapturerBlanches += deltaX;
		}
	    }
	}

	/* Le PION a ete pris apres s'etre promu. */
	if (p->camp == BLANC)	rangeePromotion = 0x70;
	else					rangeePromotion = 0x00;

	for (colProm=0; colProm<8; colProm++)
	{
	    int		k;

	    if (CasesInterdites[rangeePromotion + colProm])	continue;

	    distPion = distancePion(p->camp, p->caseInitiale,
				    rangeePromotion + colProm);

	    if ((p->camp == BLANC && distPion > NbCoupsBlancsRestants) ||
		(p->camp == NOIR  && distPion > NbCoupsNoirsRestants ) )
	    {
		continue;
	    }

	    /* Si le pion change de colonne, verifie qu'il
	     * reste assez de piece a prendre.
	     */
	    depX = column(p->caseInitiale);
	    arrX = colProm;
	    deltaX = (arrX > depX) ? (arrX - depX) : (depX - arrX);

	    if ((p->camp == BLANC && deltaX > NbResteACapturerNoires  ) ||
		(p->camp == NOIR  && deltaX > NbResteACapturerBlanches) )
	    {
		continue;
	    }

	    /* Boucle sur tous les types de promotion possible. */
	    for ( k = 0; k < Nb_Type_Promotion ; k++ )
	    {
		typePromotion = TabTypePromotion[k];

		dist = distance_avec_obstacle(typePromotion,
					      p->camp,
					      rangeePromotion + colProm,
					      i,
					      TRUE,
					      pieces_ayant_du_bouger,
					      &nombre_pieces_ayant_du_bouger);
		if (dist == INFINI)
		{
		    continue;
		}

		if ((p->camp == BLANC &&
		     distPion+dist > NbCoupsBlancsRestants) ||
		    (p->camp == NOIR &&
		     distPion+dist > NbCoupsNoirsRestants) )
		{
		    distance_reimmobilise_pieces(pieces_ayant_du_bouger,
						 nombre_pieces_ayant_du_bouger);
		    continue;
		}

		if (p->camp == BLANC)
		{
		    NbCoupsBlancsRestants  -= (distPion + dist);
		    NbResteACapturerNoires -= deltaX;
		}
		else
		{
		    NbCoupsNoirsRestants     -= (distPion + dist);
		    NbResteACapturerBlanches -= deltaX;
		}

		ListeRestantes[NbRestantes] = *p;
		ListeRestantes[NbRestantes].casePromotion    =
		    rangeePromotion + colProm;
		ListeRestantes[NbRestantes].typePromotion    = typePromotion;
		ListeRestantes[NbRestantes].caseArrivee      = i;
		ListeRestantes[NbRestantes].nbDestination    = 1;
		ListeRestantes[NbRestantes].destination[0]   =
		    rangeePromotion + colProm;
		ListeRestantes[NbRestantes].pieceCapturee[0] = PAS_DE_CAPTURE;
		ListeRestantes[NbRestantes].distance[0]      = distPion;
		if (dist != 0)
		{
		    ListeRestantes[NbRestantes].nbDestination++;
		    ListeRestantes[NbRestantes].destination[1]   = i;
		    ListeRestantes[NbRestantes].pieceCapturee[1]
			= PAS_DE_CAPTURE;
		    ListeRestantes[NbRestantes].distance[1]      = dist;
		}
		ListeRestantes[NbRestantes].pieceCapturante
		    = PIECE_CAPTURANTE_INCONNUE;
		ListeRestantes[NbRestantes].nb_captures     = deltaX;
		TabIndexPiece[p->index] = &ListeRestantes[NbRestantes];
		p->caseInitiale = PIECE_DEJA_TRAITEE;
		NbRestantes++;

		if (deltaX == 0)
		{
		    distance_libere_case(i, capturePionRestant, indice + 1);
		}
		else
		{
		    distance_libere_case( i,
					  captureParPion,
					  NbSures + NbRestantes - 1 );
		}

		NbRestantes--;
		p->caseInitiale = ListeRestantes[NbRestantes].caseInitiale;
		TabIndexPiece[p->index] = p;

		if (p->camp == BLANC)
		{
		    NbCoupsBlancsRestants  += distPion + dist;
		    NbResteACapturerNoires += deltaX;
		}
		else
		{
		    NbCoupsNoirsRestants     += distPion + dist;
		    NbResteACapturerBlanches += deltaX;
		}
		distance_reimmobilise_pieces(pieces_ayant_du_bouger,
					     nombre_pieces_ayant_du_bouger);
	    } /* Boucle sur les types de promotion */
	} /* Boucle sur les cases de promotion */
    } /* Boucle sur toutes les cases de captures */
}

static void
capturePionColonne(
    int		indice,
    int		nbCapture,
    int		indiceInsertion,
    square_t	casePrise,
    int		incrementRangee,
    int		incrementColonne,
    row_t	rangeeArrivee,
    row_t	rangeePromotion)
{
    piece_t*		p;	/* Piece qui a pu etre prise */
    piece_t*		p2;	/* Pion en cours de traitement (indice) */
    row_t		rangee;
    int			i, j, k;
    column_t		colProm;
    distance_t		dist, distPion;
    colour_t		camp;
    bool_t		ok;
    column_t		depX, arrX;
    square_delta_t	deltaX, deltaY;
    row_t		rangeePromAdverse;
    piece_type_t	typePromotionAdverse;
    int			indTypeProm;
    bool_t		bonneCase;
    distance_t		ancDist, dist1, dist2, deltaPion;
    square_t		caseInitiale;

    piece_index_t	pieces_ayant_du_bouger[16];
    int			nombre_pieces_ayant_du_bouger;

    rangee = row(casePrise);
    deltaY = (rangeeArrivee > rangee) ? rangeeArrivee - rangee :
	rangee - rangeeArrivee;
    deltaY >>= 4;

    if (nbCapture > deltaY)		return;

    casePrise += incrementRangee + incrementColonne;

    p2 = &ListeSures[indice];
    camp = p2->camp;

    for (i=0; i<deltaY-nbCapture+1; i++, casePrise += incrementRangee)
    {
	/* Verifie si le passage par `casePrise' rallonge le trajet
	 * et si oui, verifie qu'il reste assez de coups de libre.
	 * Rallonger le trajet est possible si on empeche le pion
	 * de faire son double pas.
	 */
	caseInitiale = ( indiceInsertion == 0 )
	    ? p2->caseInitiale : p2->destination[ indiceInsertion - 1 ];

	ancDist = p2->distance[indiceInsertion];
	dist1 = distancePion(camp, caseInitiale, casePrise);
	dist2 = distancePion(camp, casePrise,
			     p2->destination[indiceInsertion]);

	if (dist1 + dist2 > ancDist)
	{
	    deltaPion = dist1 + dist2 - ancDist;
	    if ((camp == BLANC && deltaPion > NbCoupsBlancsRestants) ||
		(camp == NOIR  && deltaPion > NbCoupsNoirsRestants ) )
	    {
		continue;
	    }
	    if (camp == BLANC)	NbCoupsBlancsRestants -= deltaPion;
	    else		NbCoupsNoirsRestants  -= deltaPion;
	}
	else	deltaPion = 0;


	/* Cherche parmi les pieces de l'autre camp deja prises
	 * avec une piece capturante inconnue si l'une d'entre elle
	 * a pu etre prise sur `casePrise'.
	 * Cela ne coute aucun mouvement supplementaire.
	 * C'est forcement un PION (eventuellement promu), puisque
	 * les pieces restantes sont traitees plus tard.
	 *
	 * Cette partie est necessaire. Voici un exemple de position :
	 * Blancs : position initiale et Pb2->a3, -Ph2, Th1->h6
	 * Noirs : position initiale -Pa7, -Pg7.
	 * PCPJ en 9 demi-coups.
	 * Il y a un dual mais ce n'est pas important.
	 */
	for (j=0, p=ListeRestantes; j<NbRestantes; j++, p++)
	{
	    if (p->caseArrivee == casePrise &&
		p->pieceCapturante == PIECE_CAPTURANTE_INCONNUE &&
		p->camp != camp &&
		p->index != p2->pieceCapturante)
	    {
		p->pieceCapturante = p2->index;

		if (casePrise == p2->destination[indiceInsertion])
		{
		    p2->pieceCapturee[indiceInsertion] = p->index;
		    bonneCase = TRUE;
		}
		else
		{
		    for (k=p2->nbDestination; k>indiceInsertion; k--)
		    {
			p2->destination[k]   = p2->destination[k-1];
			p2->pieceCapturee[k] = p2->pieceCapturee[k-1];
			p2->distance[k]      = p2->distance[k-1];
		    }
		    p2->destination[k]            = casePrise;
		    p2->pieceCapturee[k]          = p->index;
		    p2->nbDestination++;
		    p2->distance[indiceInsertion] = dist1;
		    indiceInsertion++;
		    p2->distance[indiceInsertion] = dist2;
		    bonneCase = FALSE;
		}

		if (nbCapture == 1)
		{
		    captureParPion(indice + 1);
		}
		else
		{
		    capturePionColonne(indice,
				       nbCapture - 1,
				       indiceInsertion,
				       casePrise,
				       incrementRangee,
				       incrementColonne,
				       rangeeArrivee,
				       rangeePromotion);
		}

		if (bonneCase)
		{
		    p2->pieceCapturee[indiceInsertion] = PAS_DE_CAPTURE;
		}
		else
		{
		    p2->nbDestination--;
		    indiceInsertion--;
		    for (k=indiceInsertion; k<p2->nbDestination; k++)
		    {
			p2->destination[k]   = p2->destination[k+1];
			p2->pieceCapturee[k] = p2->pieceCapturee[k+1];
			p2->distance[k]      = p2->distance[k+1];
		    }
		    p2->distance[indiceInsertion] = ancDist;
		}

		p->pieceCapturante = PIECE_CAPTURANTE_INCONNUE;
	    }
	}

	/* Cherche parmi les pieces non encore affectees une piece
	 * de l'autre camp qui a pu etre prise sur `casePrise'.
	 */
	for (j=0, p=ListeNonTraitees; j<NbNonTraites; j++, p++)
	{
	    if (p->caseInitiale == PIECE_DEJA_TRAITEE || p->camp == camp)
	    {
		continue;
	    }

	    /* Capture d'une piece ou d'un PION non promu. */
	    if (p->typePiece != PION || row(casePrise) != rangeePromotion)
	    {
		dist = distance_avec_obstacle(p->typePiece,
					      p->camp,
					      p->caseInitiale,
					      casePrise,
					      TRUE,
					      pieces_ayant_du_bouger,
					      &nombre_pieces_ayant_du_bouger);

		if (dist == INFINI)	ok = FALSE;
		else			ok = TRUE;

		/* Si c'est un pion qui a change de colonne, teste
		 * s'il reste assez de pieces a capturer pour le faire.
		 */
		if (ok && p->typePiece == PION)
		{
		    depX = column(p->caseInitiale);
		    arrX = column(casePrise);
		    deltaX = (arrX > depX) ? (arrX - depX) : (depX - arrX);

		    if ((p->camp == BLANC &&
			 deltaX > NbResteACapturerNoires) ||
			(p->camp == NOIR && deltaX > NbResteACapturerBlanches))
		    {
			distance_reimmobilise_pieces(
			    pieces_ayant_du_bouger,
			    nombre_pieces_ayant_du_bouger);
			ok = FALSE;
		    }
		}
		else	deltaX = 0;

		if (ok)
		{
		    if (p->camp == BLANC)
		    {
			NbCoupsBlancsRestants  -= dist;
			NbResteACapturerNoires -= deltaX;
		    }
		    else
		    {
			NbCoupsNoirsRestants     -= dist;
			NbResteACapturerBlanches -= deltaX;
		    }

		    ListeRestantes[NbRestantes] = *p;
		    ListeRestantes[NbRestantes].caseArrivee      = casePrise;
		    ListeRestantes[NbRestantes].nbDestination    = 1;
		    ListeRestantes[NbRestantes].destination[0]   = casePrise;
		    ListeRestantes[NbRestantes].pieceCapturee[0]
			= PAS_DE_CAPTURE;
		    ListeRestantes[NbRestantes].distance[0]      = dist;
		    ListeRestantes[NbRestantes].pieceCapturante  = p2->index;
		    if ( p->typePiece == PION )
			ListeRestantes[ NbRestantes ].nb_captures = deltaX ;
		    TabIndexPiece[p->index] = &ListeRestantes[NbRestantes];
		    p->caseInitiale = PIECE_DEJA_TRAITEE;
		    NbRestantes++;

		    if (casePrise == p2->destination[indiceInsertion])
		    {
			p2->pieceCapturee[indiceInsertion] = p->index;
			bonneCase = TRUE;
		    }
		    else
		    {
			for (k=p2->nbDestination; k>indiceInsertion; k--)
			{
			    p2->destination[k]   = p2->destination[k-1];
			    p2->pieceCapturee[k] = p2->pieceCapturee[k-1];
			    p2->distance[k]      = p2->distance[k-1];
			}
			p2->destination[k]            = casePrise;
			p2->pieceCapturee[k]          = p->index;
			p2->distance[k]               = 0;
			p2->nbDestination++;
			p2->distance[indiceInsertion] = dist1;
			indiceInsertion++;
			p2->distance[indiceInsertion] = dist2;
			bonneCase = FALSE;
		    }

		    if (nbCapture == 1)
		    {
			captureParPion(indice + 1);
		    }
		    else
		    {
			capturePionColonne(indice,
					   nbCapture - 1,
					   indiceInsertion,
					   casePrise,
					   incrementRangee,
					   incrementColonne,
					   rangeeArrivee,
					   rangeePromotion);
		    }

		    if (bonneCase)
		    {
			p2->pieceCapturee[indiceInsertion] = PAS_DE_CAPTURE;
		    }
		    else
		    {
			p2->nbDestination--;
			indiceInsertion--;
			for (k=indiceInsertion; k<p2->nbDestination; k++)
			{
			    p2->destination[k]   = p2->destination[k+1];
			    p2->pieceCapturee[k] = p2->pieceCapturee[k+1];
			    p2->distance[k]      = p2->distance[k+1];
			}
			p2->distance[indiceInsertion] = ancDist;
		    }

		    NbRestantes--;
		    p->caseInitiale = ListeRestantes[NbRestantes].caseInitiale;
		    TabIndexPiece[p->index] = p;

		    if (p->camp == BLANC)
		    {
			NbCoupsBlancsRestants  += dist;
			NbResteACapturerNoires += deltaX;
		    }
		    else
		    {
			NbCoupsNoirsRestants     += dist;
			NbResteACapturerBlanches += deltaX;
		    }

		    distance_reimmobilise_pieces(
			pieces_ayant_du_bouger,
			nombre_pieces_ayant_du_bouger);
		}
	    }

	    /* Capture d'un pion promu. */
	    if (p->typePiece == PION)
	    {
				/* Le PION a ete pris apres s'etre promu. */
		if (p->camp == BLANC)	rangeePromAdverse = 0x70;
		else			rangeePromAdverse = 0x00;

		for (colProm=0; colProm<8; colProm++)
		{
		    if (CasesInterdites[rangeePromAdverse + colProm])
		    {
			continue;
		    }

		    distPion = distancePion(p->camp, p->caseInitiale,
					    rangeePromAdverse + colProm);

		    if ((p->camp == BLANC &&
			 distPion > NbCoupsBlancsRestants) ||
			(p->camp == NOIR && distPion > NbCoupsNoirsRestants) )
		    {
			continue;
		    }

		    /* Si le pion change de colonne, verifie qu'il
		     * reste assez de piece a prendre.
		     */
		    depX = column(p->caseInitiale);
		    arrX = colProm;
		    deltaX = (arrX > depX) ? (arrX - depX) : (depX - arrX);

		    if ((p->camp == BLANC &&
			 deltaX > NbResteACapturerNoires) ||
			(p->camp == NOIR &&
			 deltaX > NbResteACapturerBlanches) )
		    {
			continue;
		    }

		    if (p->camp == BLANC)
		    {
			NbCoupsBlancsRestants  -= distPion;
			NbResteACapturerNoires -= deltaX;
		    }
		    else
		    {
			NbCoupsNoirsRestants     -= distPion;
			NbResteACapturerBlanches -= deltaX;
		    }

		    /* Boucle sur tous les types de promotion possible. */
		    for ( indTypeProm = 0 ;
			  indTypeProm < Nb_Type_Promotion ;
			  indTypeProm++ )
		    {
			typePromotionAdverse = TabTypePromotion[indTypeProm];

			dist = distance_avec_obstacle(
			    typePromotionAdverse,
			    p->camp,
			    rangeePromAdverse + colProm,
			    casePrise,
			    TRUE,
			    pieces_ayant_du_bouger,
			    &nombre_pieces_ayant_du_bouger );
			if (dist == INFINI)
			{
			    continue;
			}

			if (p->camp == BLANC)
			{
			    NbCoupsBlancsRestants -= dist;
			}
			else
			{
			    NbCoupsNoirsRestants  -= dist;
			}

			ListeRestantes[NbRestantes] = *p;
			ListeRestantes[NbRestantes].casePromotion    =
			    rangeePromAdverse + colProm;
			ListeRestantes[NbRestantes].typePromotion    =
			    typePromotionAdverse;
			ListeRestantes[NbRestantes].caseArrivee
			    = casePrise;
			ListeRestantes[NbRestantes].nbDestination    = 1;
			ListeRestantes[NbRestantes].destination[0]   =
			    rangeePromAdverse + colProm;
			ListeRestantes[NbRestantes].pieceCapturee[0]
			    = PAS_DE_CAPTURE;
			ListeRestantes[NbRestantes].distance[0]      = distPion;
			if (dist != 0)
			{
			    ListeRestantes[NbRestantes].nbDestination++;
			    ListeRestantes[NbRestantes].destination[1]   =
				casePrise;
			    ListeRestantes[NbRestantes].pieceCapturee[1]
				= PAS_DE_CAPTURE;
			    ListeRestantes[NbRestantes].distance[1]      = dist;
			}
			ListeRestantes[NbRestantes].pieceCapturante = p2->index;
			ListeRestantes[NbRestantes].nb_captures     = deltaX ;
			TabIndexPiece[p->index] = &ListeRestantes[NbRestantes];
			p->caseInitiale = PIECE_DEJA_TRAITEE;
			NbRestantes++;

			if (casePrise == p2->destination[indiceInsertion])
			{
			    p2->pieceCapturee[indiceInsertion] = p->index;
			    bonneCase = TRUE;
			}
			else
			{
			    for (k=p2->nbDestination; k>indiceInsertion; k--)
			    {
				p2->destination[k]   = p2->destination[k-1];
				p2->pieceCapturee[k] = p2->pieceCapturee[k-1];
				p2->distance[k]      = p2->distance[k-1];
			    }
			    p2->destination[k]            = casePrise;
			    p2->pieceCapturee[k]          = p->index;
			    p2->distance[k]               = 0;
			    p2->nbDestination++;
			    p2->distance[indiceInsertion] = dist1;
			    indiceInsertion++;
			    p2->distance[indiceInsertion] = dist2;
			    bonneCase = FALSE;
			}

			if (nbCapture == 1)
			{
			    captureParPion(indice + 1);
			}
			else
			{
			    capturePionColonne(indice,
					       nbCapture - 1,
					       indiceInsertion,
					       casePrise,
					       incrementRangee,
					       incrementColonne,
					       rangeeArrivee,
					       rangeePromotion);
			}

			if (bonneCase)
			{
			    p2->pieceCapturee[indiceInsertion] = PAS_DE_CAPTURE;
			}
			else
			{
			    p2->nbDestination--;
			    indiceInsertion--;
			    for (k=indiceInsertion; k<p2->nbDestination; k++)
			    {
				p2->destination[k]   = p2->destination[k+1];
				p2->pieceCapturee[k] = p2->pieceCapturee[k+1];
				p2->distance[k]      = p2->distance[k+1];
			    }
			    p2->distance[indiceInsertion] = ancDist;
			}

			NbRestantes--;
			p->caseInitiale =
			    ListeRestantes[NbRestantes].caseInitiale;
			TabIndexPiece[p->index] = p;

			if (p->camp == BLANC)
			{
			    NbCoupsBlancsRestants += dist;
			}
			else
			{
			    NbCoupsNoirsRestants  += dist;
			}

			distance_reimmobilise_pieces(
			    pieces_ayant_du_bouger,
			    nombre_pieces_ayant_du_bouger);
		    } /* Boucle sur les types de promotion */

		    if (p->camp == BLANC)
		    {
			NbCoupsBlancsRestants  += distPion;
			NbResteACapturerNoires += deltaX;
		    }
		    else
		    {
			NbCoupsNoirsRestants     += distPion;
			NbResteACapturerBlanches += deltaX;
		    }

		} /* Boucle sur les cases de promotion */
	    } /* Pion pris apres promotion */
	}

	if (camp == BLANC)	NbCoupsBlancsRestants += deltaPion;
	else			NbCoupsNoirsRestants  += deltaPion;
    }
}

void
captureParPion(int indice)
{
    piece_t*		p;
    int			incrementRangee, incrementColonne;
    row_t		rangeePromotion, rangeeArrivee;

    if (indice == NbSures + NbRestantes)
    {
	capturePionRestant(0);
	return;
    }

    /* Cherche le prochain pion ayant change de colonne */
    p = &ListeSures[indice];
    while ( indice < NbSures + NbRestantes &&
	   ( p->typePiece != PION || p->nb_captures == 0 ) )
    {
	indice++;
	p++;
    }

    if (indice == NbSures + NbRestantes)
    {
	capturePionRestant(0);
	return;
    }

    incrementRangee = (p->camp == BLANC) ? 16   : -16;
    rangeePromotion = (p->camp == BLANC) ? 0x70 : 0x00;

    if (p->casePromotion == CASE_PAS_DE_PROMOTION)
    {
	if (column(p->caseInitiale) > column(p->caseArrivee))
	{
	    incrementColonne = -1;
	}
	else
	{
	    incrementColonne = 1;
	}
	rangeeArrivee = row(p->caseArrivee);
    }
    else
    {
	if (column(p->caseInitiale) > column(p->casePromotion))
	{
	    incrementColonne = -1;
	}
	else
	{
	    incrementColonne = 1;
	}
	rangeeArrivee = rangeePromotion;
    }

    capturePionColonne(indice,
		       p->nb_captures,
		       0,
		       p->caseInitiale,
		       incrementRangee,
		       incrementColonne,
		       rangeeArrivee,
		       rangeePromotion);
}
